<?php
/**
 * Template renderer for Buildfuly generated content
 *
 * This file demonstrates how to render generated content on the frontend.
 * You can include this in your theme or use it as a reference.
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/public/partials
 */

// Prevent direct access
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Buildfuly Template Renderer
 */
class Buildfuly_Template_Renderer {

	/**
	 * The database handler.
	 *
	 * @var Buildfuly_Database
	 */
	private $db;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->db = new Buildfuly_Database();
	}

	/**
	 * Render the complete homepage
	 *
	 * @return string HTML output
	 */
	public function render_homepage() {
		$content = $this->db->get_all_content();

		if ( empty( $content ) ) {
			return '<p>No content generated yet.</p>';
		}

		ob_start();
		?>
		<div class="buildfuly-page">
			<?php
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Methods return escaped content
			echo $this->render_hero( $content );
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Methods return escaped content
			echo $this->render_section( $content, 'section1' );
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Methods return escaped content
			echo $this->render_features( $content );
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Methods return escaped content
			echo $this->render_testimonials( $content );
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Methods return escaped content
			echo $this->render_cta( $content );
			?>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render hero section
	 *
	 * @param array $content Content array
	 * @return string HTML output
	 */
	private function render_hero( $content ) {
		$hero_title = isset( $content['homepage.heroTitle'] ) ? $content['homepage.heroTitle'] : '';
		$hero_content = isset( $content['homepage.heroContent'] ) ? $content['homepage.heroContent'] : '';
		$hero_cta = isset( $content['homepage.heroCTA'] ) ? $content['homepage.heroCTA'] : '';

		if ( empty( $hero_title ) ) {
			return '';
		}

		ob_start();
		?>
		<section class="buildfuly-hero layout-centered">
			<div class="buildfuly-container">
				<h1><?php echo esc_html( $hero_title ); ?></h1>
				<?php if ( ! empty( $hero_content ) ) : ?>
					<p><?php echo esc_html( $hero_content ); ?></p>
				<?php endif; ?>
				<?php if ( ! empty( $hero_cta ) ) : ?>
					<a href="#contact" class="button"><?php echo esc_html( $hero_cta ); ?></a>
				<?php endif; ?>
			</div>
		</section>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render a generic section
	 *
	 * @param array  $content     Content array
	 * @param string $section_key Section key prefix
	 * @return string HTML output
	 */
	private function render_section( $content, $section_key ) {
		$title = isset( $content["homepage.{$section_key}Title"] ) ? $content["homepage.{$section_key}Title"] : '';
		$section_content = isset( $content["homepage.{$section_key}Content"] ) ? $content["homepage.{$section_key}Content"] : '';

		if ( empty( $title ) ) {
			return '';
		}

		ob_start();
		?>
		<section class="buildfuly-section">
			<div class="buildfuly-container">
				<h2><?php echo esc_html( $title ); ?></h2>
				<?php if ( ! empty( $section_content ) ) : ?>
					<p><?php echo esc_html( $section_content ); ?></p>
				<?php endif; ?>
			</div>
		</section>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render features section
	 *
	 * @param array $content Content array
	 * @return string HTML output
	 */
	private function render_features( $content ) {
		$features = array();
		for ( $i = 1; $i <= 3; $i++ ) {
			$title_key = "homepage.feature{$i}Title";
			$desc_key = "homepage.feature{$i}Desc";

			if ( isset( $content[ $title_key ] ) ) {
				$features[] = array(
					'title' => $content[ $title_key ],
					'desc'  => isset( $content[ $desc_key ] ) ? $content[ $desc_key ] : '',
				);
			}
		}

		if ( empty( $features ) ) {
			return '';
		}

		ob_start();
		?>
		<section class="buildfuly-section">
			<div class="buildfuly-container">
				<div class="buildfuly-features layout-grid-3-col">
					<?php foreach ( $features as $feature ) : ?>
						<div class="buildfuly-feature">
							<div class="buildfuly-feature-icon">
								<span class="dashicons dashicons-star-filled"></span>
							</div>
							<h3><?php echo esc_html( $feature['title'] ); ?></h3>
							<p><?php echo esc_html( $feature['desc'] ); ?></p>
						</div>
					<?php endforeach; ?>
				</div>
			</div>
		</section>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render testimonials section
	 *
	 * @param array $content Content array
	 * @return string HTML output
	 */
	private function render_testimonials( $content ) {
		$section_title = isset( $content['homepage.section2Title'] ) ? $content['homepage.section2Title'] : '';
		$testimonial = isset( $content['homepage.testimonial1'] ) ? $content['homepage.testimonial1'] : '';
		$author = isset( $content['homepage.testimonial1Author'] ) ? $content['homepage.testimonial1Author'] : '';

		if ( empty( $testimonial ) ) {
			return '';
		}

		ob_start();
		?>
		<section class="buildfuly-testimonials">
			<div class="buildfuly-container">
				<?php if ( ! empty( $section_title ) ) : ?>
					<h2><?php echo esc_html( $section_title ); ?></h2>
				<?php endif; ?>
				<div class="buildfuly-testimonial">
					<blockquote><?php echo esc_html( $testimonial ); ?></blockquote>
					<?php if ( ! empty( $author ) ) : ?>
						<cite><?php echo esc_html( $author ); ?></cite>
					<?php endif; ?>
				</div>
			</div>
		</section>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render CTA section
	 *
	 * @param array $content Content array
	 * @return string HTML output
	 */
	private function render_cta( $content ) {
		$title = isset( $content['homepage.ctaTitle'] ) ? $content['homepage.ctaTitle'] : '';
		$cta_content = isset( $content['homepage.ctaContent'] ) ? $content['homepage.ctaContent'] : '';
		$button = isset( $content['homepage.ctaButton'] ) ? $content['homepage.ctaButton'] : '';

		if ( empty( $title ) ) {
			return '';
		}

		ob_start();
		?>
		<section class="buildfuly-cta">
			<div class="buildfuly-container">
				<h2><?php echo esc_html( $title ); ?></h2>
				<?php if ( ! empty( $cta_content ) ) : ?>
					<p><?php echo esc_html( $cta_content ); ?></p>
				<?php endif; ?>
				<?php if ( ! empty( $button ) ) : ?>
					<a href="#contact" class="button"><?php echo esc_html( $button ); ?></a>
				<?php endif; ?>
			</div>
		</section>
		<?php
		return ob_get_clean();
	}
}

/**
 * Shortcode to render Buildfuly homepage
 *
 * Usage: [buildfuly_homepage]
 */
function buildfuly_homepage_shortcode() {
	$renderer = new Buildfuly_Template_Renderer();
	return $renderer->render_homepage();
}
add_shortcode( 'buildfuly_homepage', 'buildfuly_homepage_shortcode' );

/**
 * Template tag to render Buildfuly homepage
 *
 * Usage in theme: <?php buildfuly_render_homepage(); ?>
 */
function buildfuly_render_homepage() {
	$renderer = new Buildfuly_Template_Renderer();
	// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- render_homepage returns escaped content
	echo $renderer->render_homepage();
}
