/**
 * Buildfuly Public JavaScript
 * Frontend interactions for generated content
 */

(function($) {
	'use strict';

	const BuildfulyPublic = {
		/**
		 * Initialize
		 */
		init: function() {
			this.bindEvents();
			this.initAnimations();
			this.initComponents(); // New: unified component initialization
			this.initEncodedScripts(); // Legacy: for old components with data-script
			this.updateButtonsFromSettings();
		},

		/**
		 * Initialize all Buildfuly components using data attributes
		 * This is the preferred method - components just need data-buildfuly="type"
		 */
		initComponents: function() {
			this.initFAQs();
			this.initTestimonials();
			this.initPricing();
		},

		/**
		 * FAQ Accordion Component
		 * Usage: Add data-buildfuly="faq" to section element
		 */
		initFAQs: function() {
			document.querySelectorAll('[data-buildfuly="faq"]').forEach(function(section) {
				if (section.dataset.initialized) return;
				section.dataset.initialized = 'true';
				
				var items = section.querySelectorAll('.buildfuly-faq-item');
				items.forEach(function(item, index) {
					var btn = item.querySelector('.buildfuly-faq-question');
					var answer = item.querySelector('.buildfuly-faq-answer');
					var icon = item.querySelector('.buildfuly-faq-icon');
					
					if (!btn || !answer || !icon) return;
					
					btn.addEventListener('click', function() {
						var isOpen = answer.style.maxHeight && answer.style.maxHeight !== '0px';
						if (isOpen) {
							answer.style.maxHeight = '0';
							icon.style.transform = 'rotate(0deg)';
						} else {
							answer.style.maxHeight = answer.scrollHeight + 'px';
							icon.style.transform = 'rotate(180deg)';
						}
					});
					
					// Open first item by default
					if (index === 0) {
						answer.style.maxHeight = answer.scrollHeight + 'px';
						icon.style.transform = 'rotate(180deg)';
					}
				});
			});
		},

		/**
		 * Testimonial Carousel Component
		 * Usage: Add data-buildfuly="testimonials" to section element
		 */
		initTestimonials: function() {
			document.querySelectorAll('[data-buildfuly="testimonials"]').forEach(function(section) {
				if (section.dataset.initialized) return;
				section.dataset.initialized = 'true';
				
				var track = section.querySelector('.testimonial-track');
				var wrapper = section.querySelector('.testimonial-carousel-wrapper');
				var dotsContainer = section.querySelector('.testimonial-dots');
				
				if (!track || !wrapper) return;
				
				var cards = track.querySelectorAll('.testimonial-card:not([aria-hidden])');
				var actualCards = Math.min(cards.length, 7);
				var currentIndex = actualCards;
				var isMobile = window.innerWidth < 768;
				var cardWidth = isMobile ? 280 : 450;
				var cardGap = isMobile ? 16 : 24;
				var isPaused = false;
				
				function getOffset() {
					return Math.floor((window.innerWidth - cardWidth) / 2);
				}
				
				function scrollToIndex() {
					var offset = getOffset();
					var totalCardWidth = cardWidth + cardGap;
					track.style.transform = 'translateX(' + (offset - currentIndex * totalCardWidth) + 'px)';
				}
				
				function updateDots() {
					if (!dotsContainer) return;
					var allDots = dotsContainer.querySelectorAll('button');
					var displayIndex = currentIndex % actualCards;
					allDots.forEach(function(dot, j) {
						dot.className = j === displayIndex ? 'active' : '';
					});
				}
				
				function advance() {
					if (isPaused) return;
					currentIndex++;
					track.style.transition = 'transform 0.8s cubic-bezier(0.4,0,0.2,1)';
					scrollToIndex();
					updateDots();
					
					if (currentIndex >= actualCards * 2) {
						setTimeout(function() {
							track.style.transition = 'none';
							currentIndex = actualCards;
							scrollToIndex();
							setTimeout(updateDots, 50);
						}, 800);
					}
				}
				
				// Initialize
				track.style.transition = 'none';
				scrollToIndex();
				
				// Create dots
				if (dotsContainer) {
					for (var i = 0; i < actualCards; i++) {
						var dot = document.createElement('button');
						dot.setAttribute('aria-label', 'Go to slide ' + (i + 1));
						if (i === 0) dot.className = 'active';
						dot.onclick = (function(idx) {
							return function() {
								isPaused = true;
								currentIndex = actualCards + idx;
								track.style.transition = 'transform 0.8s cubic-bezier(0.4,0,0.2,1)';
								scrollToIndex();
								updateDots();
								setTimeout(function() { isPaused = false; }, 3000);
							};
						})(i);
						dotsContainer.appendChild(dot);
					}
				}
				
				// Pause on hover
				track.addEventListener('mouseenter', function() { isPaused = true; });
				track.addEventListener('mouseleave', function() { isPaused = false; });
				
				// Handle resize
				window.addEventListener('resize', function() {
					isMobile = window.innerWidth < 768;
					cardWidth = isMobile ? 280 : 450;
					cardGap = isMobile ? 16 : 24;
					track.style.transition = 'none';
					scrollToIndex();
					setTimeout(function() {
						track.style.transition = 'transform 0.8s cubic-bezier(0.4,0,0.2,1)';
					}, 50);
				});
				
				// Start autoplay
				setInterval(advance, 4000);
			});
		},

		/**
		 * Pricing Toggle Component
		 * Usage: Add data-buildfuly="pricing" to section element
		 */
		initPricing: function() {
			document.querySelectorAll('[data-buildfuly="pricing"]').forEach(function(section) {
				if (section.dataset.initialized) return;
				section.dataset.initialized = 'true';
				
				var toggle = section.querySelector('.billing-toggle');
				if (!toggle) return;
				
				var monthlyPrices = section.querySelectorAll('.price-monthly');
				var yearlyPrices = section.querySelectorAll('.price-yearly');
				var labels = section.querySelectorAll('.billing-label');
				
				toggle.addEventListener('change', function() {
					var isYearly = toggle.checked;
					monthlyPrices.forEach(function(el) { el.style.display = isYearly ? 'none' : 'block'; });
					yearlyPrices.forEach(function(el) { el.style.display = isYearly ? 'block' : 'none'; });
					labels.forEach(function(label) {
						label.classList.toggle('active', 
							(isYearly && label.classList.contains('yearly')) || 
							(!isYearly && label.classList.contains('monthly'))
						);
					});
				});
			});
		},

		/**
		 * Update all CTA buttons to match current settings
		 * This ensures buttons stay updated even if component was created before settings changed
		 */
		updateButtonsFromSettings: function() {
			if (typeof buildfulyPublic === 'undefined' || !buildfulyPublic.ctaButtons) {
				return;
			}

			const buttons = buildfulyPublic.ctaButtons;
			
			// Update all buttons with data-button-index (linked to global settings)
			// Skip buttons with data-custom-button="true"
			document.querySelectorAll('.buildfuly-btn-primary[data-button-index], .buildfuly-btn-secondary[data-button-index]').forEach(function(btn) {
				// Skip custom buttons
				if (btn.getAttribute('data-custom-button') === 'true') {
					return;
				}
				
				const index = parseInt(btn.getAttribute('data-button-index'));
				
				// If this button index exists in settings and has content
				if (buttons[index] && buttons[index].text && buttons[index].text.trim() !== '') {
					btn.textContent = buttons[index].text;
					if (buttons[index].url) btn.href = buttons[index].url;
					btn.style.display = 'inline-flex';
				} else {
					// Button was removed or has no text - hide it
					btn.style.display = 'none';
				}
			});
			
			// Legacy support: Update old class names too (cta-btn-primary/secondary)
			if (buttons[0]) {
				document.querySelectorAll('.cta-btn-primary').forEach(function(btn) {
					if (buttons[0].text) btn.textContent = buttons[0].text;
					if (buttons[0].url) btn.href = buttons[0].url;
				});
			}
			
			if (buttons[1]) {
				document.querySelectorAll('.cta-btn-secondary').forEach(function(btn) {
					if (buttons[1].text) btn.textContent = buttons[1].text;
					if (buttons[1].url) btn.href = buttons[1].url;
				});
			} else {
				// No secondary button - hide all legacy secondary buttons
				document.querySelectorAll('.cta-btn-secondary').forEach(function(btn) {
					btn.style.display = 'none';
				});
			}
		},

		/**
		 * Initialize encoded scripts from data attributes
		 */
		initEncodedScripts: function() {
			document.querySelectorAll('[data-script]:not([data-script-loaded])').forEach(function(element) {
				const encoded = element.getAttribute('data-script');
				if (!encoded || encoded.trim() === '') {
					return; // Skip empty attributes
				}
				
				try {
					// Decode base64 with UTF-8 support (reverse of btoa(unescape(encodeURIComponent())))
					const decoded = decodeURIComponent(escape(atob(encoded)));
					
					// Execute script safely
					try {
						const script = document.createElement('script');
						script.textContent = decoded;
						document.body.appendChild(script);
						// Mark as loaded to prevent duplicate execution
						element.setAttribute('data-script-loaded', 'true');
					} catch (execError) {
						console.error('Failed to execute decoded script:', execError);
					}
				} catch (e) {
					console.error('Failed to decode script:', e);
					// Don't let script errors break the entire page
					element.setAttribute('data-script-loaded', 'error');
				}
			});
		},

		/**
		 * Bind event handlers
		 */
		bindEvents: function() {
			// Smooth scroll for anchor links
			$('a[href^="#"]').on('click', this.smoothScroll.bind(this));

			// Add intersection observer for scroll animations
			if ('IntersectionObserver' in window) {
				this.observeElements();
			}
		},

		/**
		 * Smooth scroll to anchor
		 */
		smoothScroll: function(e) {
			const target = $(e.currentTarget.getAttribute('href'));
			if (target.length) {
				e.preventDefault();
				$('html, body').animate({
					scrollTop: target.offset().top - 80
				}, 800);
			}
		},

		/**
		 * Initialize animations
		 */
		initAnimations: function() {
			// Add fade-in class to elements
			$('.buildfuly-feature, .buildfuly-testimonial').addClass('buildfuly-fade-element');
		},

		/**
		 * Observe elements for scroll animations
		 */
		observeElements: function() {
			const observer = new IntersectionObserver((entries) => {
				entries.forEach(entry => {
					if (entry.isIntersecting) {
						entry.target.classList.add('buildfuly-visible');
						observer.unobserve(entry.target);
					}
				});
			}, {
				threshold: 0.1,
				rootMargin: '0px 0px -50px 0px'
			});

			document.querySelectorAll('.buildfuly-fade-element').forEach(el => {
				observer.observe(el);
			});
		}
	};

	// Initialize on document ready
	$(document).ready(function() {
		BuildfulyPublic.init();
	});

})(jQuery);
