<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for the public-facing side.
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/public
 * @author     Buildfuly Team
 */
class Buildfuly_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * The style manager.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Buildfuly_Style_Manager    $style_manager    Style manager instance.
	 */
	private $style_manager;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string    $plugin_name    The name of the plugin.
	 * @param    string    $version        The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->style_manager = new Buildfuly_Style_Manager();
		
		// Set favicon with logo fallback
		add_filter( 'get_site_icon_url', array( $this, 'favicon_with_logo_fallback' ), 10, 3 );
		
		// Add body class for Buildfuly pages
		add_filter( 'body_class', array( $this, 'add_buildfuly_body_class' ) );
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
		wp_enqueue_style(
			$this->plugin_name,
			plugin_dir_url( __FILE__ ) . 'css/buildfuly-public.css',
			array(),
			$this->version . '-' . time(), // Add cache busting like admin
			'all'
		);

		// Add inline styles from style manager
		$custom_css = $this->style_manager->get_css_output();
		wp_add_inline_style( $this->plugin_name, $custom_css );
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {
		wp_enqueue_script(
			$this->plugin_name,
			plugin_dir_url( __FILE__ ) . 'js/buildfuly-public.js',
			array( 'jquery' ),
			$this->version . '-' . time(), // Add cache busting like admin
			true
		);
		
		// Localize script with current CTA buttons from settings
		$cta_buttons = json_decode( get_option( 'buildfuly_cta_buttons', '[]' ), true );
		wp_localize_script(
			$this->plugin_name,
			'buildfulyPublic',
			array(
				'ctaButtons' => $cta_buttons ? $cta_buttons : array()
			)
		);
		
		// Add any inline scripts that were stored by components
		global $buildfuly_inline_scripts;
		if (!empty($buildfuly_inline_scripts)) {
			$combined_script = implode("\n", $buildfuly_inline_scripts);
			wp_add_inline_script($this->plugin_name, $combined_script);
		}
	}
	
	/**
	 * Use logo as favicon if no favicon is set
	 *
	 * @param string $url    Site icon URL.
	 * @param int    $size   Size of the site icon.
	 * @param int    $blog_id ID of the blog to get the site icon for.
	 * @return string Modified site icon URL
	 */
	public function favicon_with_logo_fallback( $url, $size, $blog_id ) {
		// If there's already a favicon URL, return it
		if ( ! empty( $url ) ) {
			return $url;
		}
		
		// No favicon set, use logo as fallback
		$logo_id = get_option( 'buildfuly_business_logo_id', '' );
		if ( $logo_id ) {
			// Return full-size image URL instead of scaled down version
			// This ensures high quality on all devices and lets browser scale as needed
			$logo_url = wp_get_attachment_url( $logo_id );
			if ( $logo_url ) {
				return $logo_url;
			}
		}
		
		// No logo either, return empty
		return $url;
	}
	
	/**
	 * Add 'buildfuly-page' class to body on pages with Buildfuly content
	 *
	 * @param array $classes Body classes.
	 * @return array Modified body classes
	 */
	public function add_buildfuly_body_class( $classes ) {
		if ( is_singular() ) {
			global $post;
			// Check if page has Buildfuly content
			$buildfuly_content = get_post_meta( $post->ID, '_buildfuly_sections', true );
			if ( ! empty( $buildfuly_content ) ) {
				$classes[] = 'buildfuly-page';
			}
		}
		return $classes;
	}
}
