<?php
/**
 * Reusable Pricing Card Component
 * Apple-inspired design for use in both WordPress and hosted app
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes/components
 */

// Prevent direct access
if ( ! defined( 'WPINC' ) && ! defined( 'BUILDFULY_API' ) ) {
	die;
}

/**
 * Render pricing cards
 *
 * @param array $args {
 *     Optional. Array of arguments.
 *     @type bool   $show_toggle       Show monthly/yearly toggle. Default true.
 *     @type bool   $show_lifetime     Show lifetime pricing. Default true.
 *     @type string $current_tier      Current user's tier slug. Default empty.
 *     @type string $cta_url           Call-to-action URL template. Use {tier} and {period} placeholders. Default '#'
 *     @type bool   $show_free_tier    Show free starter tier. Default true.
 * }
 */
function buildfuly_render_pricing_cards( $args = array() ) {
	// Load pricing class
	if ( ! class_exists( 'Buildfuly_Pricing' ) ) {
		require_once dirname( dirname( __FILE__ ) ) . '/class-buildfuly-pricing.php';
	}

	// Default args
	$defaults = array(
		'show_toggle'    => true,
		'show_lifetime'  => true,
		'current_tier'   => '',
		'cta_url'        => '#',
		'show_free_tier' => true,
	);
	$args = wp_parse_args( $args, $defaults );

	$plans = Buildfuly_Pricing::get_all_plans();
	
	// Remove starter if not showing free tier
	if ( ! $args['show_free_tier'] && isset( $plans['starter'] ) ) {
		unset( $plans['starter'] );
	}

	?>
	<div class="buildfuly-pricing-container">
		
		<?php if ( $args['show_toggle'] ) : ?>
			<!-- Billing Toggle -->
			<div class="billing-toggle-wrapper">
				<span class="billing-label active" data-period="monthly">Monthly</span>
				<label class="billing-switch">
					<input type="checkbox" id="buildfuly-billing-toggle">
					<span class="billing-slider"></span>
			</label>
			<span class="billing-label" data-period="yearly">
				Yearly
				<span class="discount-badge">
					Save <?php echo absint( Buildfuly_Pricing::get_yearly_discount( 'pro' ) ); ?>%
				</span>
			</span>
		</div>
	<?php endif; ?>		<!-- Pricing Grid -->
		<div class="pricing-grid">
			<?php foreach ( $plans as $tier => $plan ) : 
				$is_current = ( $args['current_tier'] === $tier );
				$is_popular = ( $tier === 'growth' );
				$tier_class = 'tier-' . $tier;
			?>
				<div class="pricing-card <?php echo esc_attr( $tier_class ); ?> <?php echo $is_popular ? 'popular' : ''; ?>" 
					 data-tier="<?php echo esc_attr( $tier ); ?>">
					
					<!-- Popular Badge -->
					<?php if ( $is_popular ) : ?>
						<div class="card-badge popular-badge">Most Popular</div>
					<?php endif; ?>
					
					<!-- Current Plan Badge -->
					<?php if ( $is_current ) : ?>
						<div class="card-badge current-badge">Your Plan</div>
					<?php endif; ?>

					<!-- Card Header -->
					<div class="card-header">
						<h3 class="plan-name"><?php echo esc_html( $plan['name'] ); ?></h3>
						
						<!-- Monthly Price -->
						<div class="price-display price-monthly">
							<div class="price-amount">
								<?php if ( $plan['monthly'] === 0 ) : ?>
									<span class="price-value">Free</span>
								<?php else : ?>
									<span class="price-currency">$</span>
									<span class="price-value"><?php echo esc_html( $plan['monthly'] ); ?></span>
									<span class="price-period">/month</span>
								<?php endif; ?>
							</div>
							<?php if ( $plan['monthly'] > 0 ) : ?>
								<p class="price-note">per month</p>
							<?php endif; ?>
						</div>

						<!-- Yearly Price -->
						<div class="price-display price-yearly" style="display: none;">
							<div class="price-amount">
								<?php if ( $plan['yearly'] === 0 ) : ?>
									<span class="price-value">Free</span>
								<?php else : ?>
									<span class="price-currency">$</span>
									<span class="price-value"><?php echo esc_html( $plan['yearly'] ); ?></span>
									<span class="price-period">/year</span>
								<?php endif; ?>
							</div>
							<?php if ( $plan['yearly'] > 0 ) : ?>
								<p class="price-note">
									$<?php echo number_format( $plan['yearly'] / 12, 2 ); ?> per month
								</p>
							<?php endif; ?>
						</div>

						<!-- Lifetime Price -->
						<?php if ( $args['show_lifetime'] && $plan['lifetime'] > 0 ) : ?>
							<div class="lifetime-option">
								<strong>Lifetime:</strong> $<?php echo esc_html( $plan['lifetime'] ); ?>
								<span class="lifetime-note">(2 years of yearly)</span>
							</div>
						<?php endif; ?>
					</div>

					<!-- Features List -->
					<ul class="features-list">
						<?php foreach ( $plan['features'] as $feature ) : ?>
							<li class="feature-item">
								<svg class="feature-icon" width="20" height="20" viewBox="0 0 20 20" fill="none">
									<circle cx="10" cy="10" r="10" fill="currentColor" opacity="0.1"/>
									<path d="M6 10l3 3 5-6" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
								</svg>
								<?php echo esc_html( $feature ); ?>
							</li>
						<?php endforeach; ?>
					</ul>

					<!-- CTA Button -->
					<div class="card-footer">
						<?php if ( $is_current ) : ?>
							<button class="cta-button current-plan" disabled>
								Current Plan
							</button>
						<?php else : ?>
							<?php
							$cta_url = str_replace(
								array( '{tier}', '{period}' ),
								array( $tier, 'monthly' ),
								$args['cta_url']
							);
							?>
							<a href="<?php echo esc_url( $cta_url ); ?>" class="cta-button <?php echo $tier === 'starter' ? 'secondary' : 'primary'; ?>">
								<?php echo $tier === 'starter' ? 'Get Started Free' : 'Choose ' . esc_html( $plan['name'] ); ?>
							</a>
						<?php endif; ?>
					</div>
				</div>
			<?php endforeach; ?>
		</div>
	</div>

	<!-- Styles -->
	<style>
	.buildfuly-pricing-container {
		max-width: 1200px;
		margin: 0 auto;
		padding: 40px 20px;
	}

	/* Billing Toggle */
	.billing-toggle-wrapper {
		display: flex;
		justify-content: center;
		align-items: center;
		gap: 16px;
		margin-bottom: 48px;
	}

	.billing-label {
		font-size: 18px;
		font-weight: 500;
		color: #86868b;
		cursor: pointer;
		transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
		user-select: none;
	}

	.billing-label.active {
		color: #1d1d1f;
		font-weight: 600;
		transform: scale(1.05);
	}

	.discount-badge {
		display: inline-block;
		background: linear-gradient(135deg, #34c759, #30d158);
		color: white;
		padding: 3px 10px;
		border-radius: 12px;
		font-size: 12px;
		font-weight: 600;
		margin-left: 6px;
		box-shadow: 0 2px 8px rgba(52, 199, 89, 0.3);
	}

	.billing-switch {
		position: relative;
		display: inline-block;
		width: 56px;
		height: 32px;
	}

	.billing-switch input {
		opacity: 0;
		width: 0;
		height: 0;
	}

	.billing-slider {
		position: absolute;
		cursor: pointer;
		top: 0;
		left: 0;
		right: 0;
		bottom: 0;
		background: #e5e5e7;
		transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
		border-radius: 32px;
	}

	.billing-slider:before {
		position: absolute;
		content: "";
		height: 26px;
		width: 26px;
		left: 3px;
		bottom: 3px;
		background: white;
		transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
		border-radius: 50%;
		box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
	}

	.billing-switch input:checked + .billing-slider {
		background: linear-gradient(135deg, #007aff, #0051d5);
	}

	.billing-switch input:checked + .billing-slider:before {
		transform: translateX(24px);
	}

	/* Pricing Grid */
	.pricing-grid {
		display: grid;
		grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
		gap: 24px;
		margin-top: 32px;
	}

	/* Pricing Card */
	.pricing-card {
		position: relative;
		background: white;
		border-radius: 20px;
		padding: 32px 28px;
		text-align: center;
		transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
		border: 2px solid #e5e5e7;
		box-shadow: 0 4px 20px rgba(0, 0, 0, 0.05);
		overflow: hidden;
	}

	.pricing-card:hover {
		transform: translateY(-8px);
		box-shadow: 0 12px 40px rgba(0, 0, 0, 0.12);
	}

	.pricing-card.tier-pro {
		border-color: #007aff;
	}

	.pricing-card.tier-growth {
		border-color: #34c759;
		border-width: 3px;
	}

	.pricing-card.tier-studio {
		border-color: #af52de;
	}

	.pricing-card.popular {
		background: linear-gradient(135deg, rgba(52, 199, 89, 0.03), rgba(48, 209, 88, 0.05));
	}

	.pricing-card.popular::before {
		content: "";
		position: absolute;
		top: 0;
		left: 0;
		right: 0;
		height: 4px;
		background: linear-gradient(90deg, #34c759, #30d158);
	}

	/* Card Badges */
	.card-badge {
		position: absolute;
		top: 16px;
		padding: 6px 14px;
		border-radius: 20px;
		font-size: 11px;
		font-weight: 700;
		letter-spacing: 0.5px;
		text-transform: uppercase;
		box-shadow: 0 2px 12px rgba(0, 0, 0, 0.1);
	}

	.popular-badge {
		right: 16px;
		background: linear-gradient(135deg, #34c759, #30d158);
		color: white;
	}

	.current-badge {
		left: 16px;
		background: linear-gradient(135deg, #007aff, #0051d5);
		color: white;
	}

	/* Card Header */
	.card-header {
		margin-bottom: 28px;
	}

	.plan-name {
		font-size: 28px;
		font-weight: 700;
		margin: 0 0 20px;
		color: #1d1d1f;
		letter-spacing: -0.5px;
	}

	.pricing-card.tier-pro .plan-name {
		color: #007aff;
	}

	.pricing-card.tier-growth .plan-name {
		color: #34c759;
	}

	.pricing-card.tier-studio .plan-name {
		color: #af52de;
	}

	/* Price Display */
	.price-display {
		margin: 24px 0;
	}

	.price-amount {
		display: flex;
		align-items: baseline;
		justify-content: center;
		gap: 4px;
		margin-bottom: 8px;
	}

	.price-currency {
		font-size: 28px;
		font-weight: 600;
		color: #1d1d1f;
	}

	.price-value {
		font-size: 56px;
		font-weight: 800;
		color: #1d1d1f;
		line-height: 1;
		letter-spacing: -2px;
	}

	.price-period {
		font-size: 18px;
		font-weight: 500;
		color: #86868b;
	}

	.price-note {
		font-size: 13px;
		color: #86868b;
		margin: 8px 0 0;
		line-height: 1.4;
	}

	/* Lifetime Option */
	.lifetime-option {
		margin: 16px 0;
		padding: 12px;
		background: #f5f5f7;
		border-radius: 12px;
		font-size: 14px;
		color: #1d1d1f;
	}

	.lifetime-note {
		display: block;
		font-size: 12px;
		color: #86868b;
		margin-top: 4px;
	}

	/* Features List */
	.features-list {
		list-style: none;
		padding: 0;
		margin: 28px 0;
		text-align: left;
	}

	.feature-item {
		display: flex;
		align-items: center;
		gap: 12px;
		padding: 12px 0;
		border-bottom: 1px solid #f5f5f7;
		font-size: 15px;
		color: #1d1d1f;
		line-height: 1.4;
	}

	.feature-item:last-child {
		border-bottom: none;
	}

	.feature-icon {
		flex-shrink: 0;
		color: #34c759;
	}

	.tier-pro .feature-icon {
		color: #007aff;
	}

	.tier-studio .feature-icon {
		color: #af52de;
	}

	/* CTA Button */
	.card-footer {
		margin-top: 28px;
	}

	.cta-button {
		display: block;
		width: 100%;
		padding: 16px 24px;
		border-radius: 12px;
		font-size: 17px;
		font-weight: 600;
		text-align: center;
		text-decoration: none;
		transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
		border: none;
		cursor: pointer;
	}

	.cta-button.primary {
		background: linear-gradient(135deg, #007aff, #0051d5);
		color: white;
		box-shadow: 0 4px 16px rgba(0, 122, 255, 0.3);
	}

	.cta-button.primary:hover {
		transform: scale(1.02);
		box-shadow: 0 6px 24px rgba(0, 122, 255, 0.4);
	}

	.tier-growth .cta-button.primary {
		background: linear-gradient(135deg, #34c759, #30d158);
		box-shadow: 0 4px 16px rgba(52, 199, 89, 0.3);
	}

	.tier-growth .cta-button.primary:hover {
		box-shadow: 0 6px 24px rgba(52, 199, 89, 0.4);
	}

	.tier-studio .cta-button.primary {
		background: linear-gradient(135deg, #af52de, #8e24aa);
		box-shadow: 0 4px 16px rgba(175, 82, 222, 0.3);
	}

	.tier-studio .cta-button.primary:hover {
		box-shadow: 0 6px 24px rgba(175, 82, 222, 0.4);
	}

	.cta-button.secondary {
		background: #f5f5f7;
		color: #1d1d1f;
	}

	.cta-button.secondary:hover {
		background: #e5e5e7;
	}

	.cta-button.current-plan {
		background: #f5f5f7;
		color: #86868b;
		cursor: not-allowed;
		opacity: 0.6;
	}

	/* Responsive */
	@media (max-width: 768px) {
		.pricing-grid {
			grid-template-columns: 1fr;
			gap: 20px;
		}

		.billing-toggle-wrapper {
			margin-bottom: 32px;
		}

		.price-value {
			font-size: 48px;
		}
	}
	</style>

	<!-- Script -->
	<script>
	(function() {
		const toggle = document.getElementById('buildfuly-billing-toggle');
		if (!toggle) return;

		const monthlyPrices = document.querySelectorAll('.price-monthly');
		const yearlyPrices = document.querySelectorAll('.price-yearly');
		const labels = document.querySelectorAll('.billing-label');

		toggle.addEventListener('change', function() {
			const isYearly = this.checked;

			monthlyPrices.forEach(el => el.style.display = isYearly ? 'none' : 'block');
			yearlyPrices.forEach(el => el.style.display = isYearly ? 'block' : 'none');

			labels.forEach(label => label.classList.remove('active'));
			const activeLabel = document.querySelector('.billing-label[data-period="' + (isYearly ? 'yearly' : 'monthly') + '"]');
			if (activeLabel) activeLabel.classList.add('active');
		});

		labels.forEach(label => {
			label.addEventListener('click', function() {
				const period = this.getAttribute('data-period');
				toggle.checked = (period === 'yearly');
				toggle.dispatchEvent(new Event('change'));
			});
		});
	})();
	</script>
	<?php
}
