<?php
/**
 * Component Registry - Dynamic component discovery and metadata
 * 
 * This file provides functions to access component metadata that is
 * dynamically parsed from individual component files.
 * All component data lives in the component files themselves with @tags.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Get all registered components with their metadata (parsed from component files)
 * 
 * @return array Array of component definitions
 */
function buildfuly_get_component_registry() {
	// Load component loader
	require_once plugin_dir_path( dirname( __FILE__ ) ) . 'class-buildfuly-component-loader.php';
	
	$loader = new Buildfuly_Component_Loader();
	return $loader->discover_components();
}

/**
 * Get component names only (for dropdown/selection)
 * 
 * @return array Simple array of component keys
 */
function buildfuly_get_component_names() {
	return array_keys( buildfuly_get_component_registry() );
}

/**
 * Get components by category
 * 
 * @param string $category Category slug
 * @return array Filtered components
 */
function buildfuly_get_components_by_category( $category ) {
	$all = buildfuly_get_component_registry();
	return array_filter( $all, function( $component ) use ( $category ) {
		return $component['category'] === $category;
	});
}

/**
 * Get components that support AI content generation
 * 
 * @return array Components with ai_content_structure defined
 */
function buildfuly_get_ai_supported_components() {
	$all = buildfuly_get_component_registry();
	return array_filter( $all, function( $component ) {
		return ! is_null( $component['ai_content_structure'] );
	});
}

/**
 * Get component metadata for AI prompt generation
 * Used by content generator to understand what content to request from AI
 * 
 * @param string $component_name Component key
 * @return array|null Component metadata or null if not found
 */
function buildfuly_get_component_for_ai( $component_name ) {
	$registry = buildfuly_get_component_registry();
	return isset( $registry[$component_name] ) ? $registry[$component_name] : null;
}

/**
 * Generate randomized page layout
 * Creates a logical flow: Hero → Content Components → CTA
 * 
 * @param int $content_count Number of content sections (default 6)
 * @return array Array of component names in order
 */
function buildfuly_generate_random_page_layout( $content_count = 6 ) {
	$registry = buildfuly_get_component_registry();
	
	// Separate components by category
	$heroes = array();
	$content = array();
	$ctas = array();
	
	foreach ( $registry as $key => $component ) {
		// Skip if requires setting and setting not configured
		if ( isset( $component['requires_setting'] ) ) {
			$setting_value = get_option( $component['requires_setting'], '' );
			if ( empty( $setting_value ) ) {
				continue;
			}
		}
		
		$category = $component['category'];
		
		if ( $category === 'hero' ) {
			$heroes[] = $key;
		} elseif ( $category === 'cta' ) {
			$ctas[] = $key;
		} elseif ( in_array( $category, array( 'content', 'feature', 'testimonial', 'pricing', 'media' ) ) ) {
			$content[] = $key;
		}
	}
	
	// Build layout
	$layout = array();
	
	// 1. Random hero (required)
	if ( ! empty( $heroes ) ) {
		$layout[] = $heroes[ array_rand( $heroes ) ];
	}
	
	// 2. Random content components (no duplicates)
	shuffle( $content );
	$selected_content = array_slice( $content, 0, min( $content_count, count( $content ) ) );
	$layout = array_merge( $layout, $selected_content );
	
	// 3. Random CTA at the end (required)
	if ( ! empty( $ctas ) ) {
		$layout[] = $ctas[ array_rand( $ctas ) ];
	}
	
	Buildfuly_Logger::debug( 'BuildfulyAI: Generated random layout with ' . count( $layout ) . ' components: ' . implode( ', ', $layout ) );
	
	return $layout;
}

/**
 * Get components suitable for specific use case
 * 
 * @param string $use_case Use case slug (e.g., 'landing-page', 'saas')
 * @return array Filtered components
 */
function buildfuly_get_components_for_use_case( $use_case ) {
	$all = buildfuly_get_component_registry();
	return array_filter( $all, function( $component ) use ( $use_case ) {
		return isset( $component['use_cases'] ) && in_array( $use_case, $component['use_cases'] );
	});
}

