<?php
/**
 * Component Helper Functions
 * 
 * Common functions and utilities used across all components
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes/components
 */

/**
 * Get theme colors with fallback defaults
 *
 * @return array Theme colors
 */
function buildfuly_get_theme_colors() {
	$primary = get_option( 'buildfuly_primary_color', '#00bfa5' );
	$accent = get_option( 'buildfuly_accent_color', '#7c4dff' );
	$text = get_option( 'buildfuly_text_color', '#1e293b' );
	$text_light = get_option( 'buildfuly_text_light_color', '#6c757d' );
	$bg = get_option( 'buildfuly_bg_color', '#000000' );
	$star = get_option( 'buildfuly_star_color', '#fbbf24' );
	
	// Auto-calculate secondary from primary (darker shade)
	$secondary = buildfuly_adjust_color_brightness( $primary, -30 );
	
	// Auto-calculate text muted
	$text_muted = buildfuly_adjust_color_brightness( $text, 60 );
	
	// Auto-calculate bg variations from base bg
	$bg_alt = buildfuly_adjust_color_brightness( $bg, -5 );
	
	// Badge colors use primary with opacity
	$badge_bg = $primary . '20'; // 20 = 12% opacity in hex
	$badge_text = $primary;
	
	// Border color - auto-calculate from background
	$border = buildfuly_adjust_color_brightness( $bg, -15 );
	
	return array(
		'primary'       => $primary,
		'secondary'     => $secondary,
		'accent'        => $accent,
		'text'          => $text,
		'text_light'    => $text_light,
		'text_muted'    => $text_muted,
		'heading'       => $text, // Use text color for headings
		'bg'            => $bg,
		'bg_alt'        => $bg_alt,
		// Utility colors
		'badge_bg'      => $badge_bg,
		'badge_text'    => $badge_text,
		'border'        => $border,
		'star'          => $star,
	);
}

/**
 * Adjust color brightness
 */
function buildfuly_adjust_color_brightness( $hex, $steps ) {
	// Remove # if present
	$hex = str_replace( '#', '', $hex );
	
	// Convert to RGB
	$r = hexdec( substr( $hex, 0, 2 ) );
	$g = hexdec( substr( $hex, 2, 2 ) );
	$b = hexdec( substr( $hex, 4, 2 ) );
	
	// Adjust brightness
	$r = max( 0, min( 255, $r + $steps ) );
	$g = max( 0, min( 255, $g + $steps ) );
	$b = max( 0, min( 255, $b + $steps ) );
	
	// Convert back to hex
	return '#' . str_pad( dechex( $r ), 2, '0', STR_PAD_LEFT )
		. str_pad( dechex( $g ), 2, '0', STR_PAD_LEFT )
		. str_pad( dechex( $b ), 2, '0', STR_PAD_LEFT );
}

/**
 * Get font stack - inherits from WordPress theme
 * Components don't specify font-family, letting them inherit the site's typography
 *
 * @return string Empty string (no font-family needed - inherit from theme)
 */
function buildfuly_font_stack() {
	return ''; // Let WordPress theme fonts apply naturally
}

/**
 * Get business information
 *
 * @return array Business details
 */
function buildfuly_get_business_info() {
	return array(
		'name'        => get_option( 'buildfuly_business_name', '' ),
		'city'        => get_option( 'buildfuly_city', '' ),
		'state'       => get_option( 'buildfuly_state', '' ),
		'services'    => get_option( 'buildfuly_services', '' ),
		'description' => get_option( 'buildfuly_business_description', '' ),
		'phone'       => get_option( 'buildfuly_phone', '' ),
	);
}

/**
 * Generate icon box HTML (WordPress-safe with table layout)
 *
 * @param string $icon Icon character (emoji or symbol)
 * @param string $bg_color Background color
 * @param int    $size Size in pixels (default 48)
 * @return string Icon HTML
 */
function buildfuly_icon_box( $icon, $bg_color, $size = 48 ) {
	return sprintf(
		'<div style="width: %1$dpx; height: %1$dpx; background: %2$s; border-radius: 12px; padding: %3$dpx 0; text-align: center; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
			<span style="color: white; font-size: %4$dpx; font-weight: 700; line-height: 1;">%5$s</span>
		</div>',
		$size,
		esc_attr( $bg_color ),
		intval( $size * 0.2 ), // padding = 20% of size
		intval( $size * 0.5 ),  // font = 50% of size
		$icon
	);
}

/**
 * Get random face image URL from Pravatar
 *
 * @param int $size Image size in pixels (default 150)
 * @return string Image URL
 */
function buildfuly_random_face( $size = 150, $gender = null, $name = '' ) {
	// Use UI Faces API for realistic, professional photos
	// Falls back to RoboHash for unique avatars
	if ( $gender === 'male' || $gender === 'female' ) {
		// Use This Person Does Not Exist style API
		// xsgames.co provides realistic AI-generated faces
		$gender_param = $gender === 'female' ? 'women' : 'men';
		$random_id = ! empty( $name ) ? abs( crc32( $name ) ) % 100 : wp_rand( 1, 99 );
		return "https://randomuser.me/api/portraits/{$gender_param}/{$random_id}.jpg";
	}

	// Fallback to random realistic photo
	$random_id = wp_rand( 1, 99 );
	$gender_param = wp_rand( 0, 1 ) ? 'men' : 'women';
	return "https://randomuser.me/api/portraits/{$gender_param}/{$random_id}.jpg";
}

/**
 * Generate gradient text color
 *
 * @param string $color1 First gradient color
 * @param string $color2 Second gradient color
 * @return string CSS style string
 */
function buildfuly_gradient_text( $color1, $color2 ) {
	return sprintf(
		'color: transparent; background: linear-gradient(135deg, %s 0%%, %s 100%%); -webkit-background-clip: text; background-clip: text;',
		esc_attr( $color1 ),
		esc_attr( $color2 )
	);
}
