<?php

/**
 * The core plugin class
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * @since      1.0.0
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @author     Buildfuly Team
 */
class Buildfuly {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Buildfuly_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'BUILDFULY_VERSION' ) ) {
			$this->version = BUILDFULY_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'buildfuly';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->define_api_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
	 * core plugin.
	 */
	require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-loader.php';

	/**
	 * The class responsible for logging debug messages
	 */
	require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-logger.php';

	/**
	 * The class responsible for defining internationalization functionality
	 * of the plugin.
	 */
	require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-i18n.php';		/**
		 * The class responsible for plugin constants and configuration
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-constants.php';

		/**
		 * The class responsible for handling database operations
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-database.php';

		/**
		 * The class responsible for content generation logic
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-content-generator.php';

		/**
		 * The class responsible for styling logic
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-style-manager.php';

		/**
		 * The class responsible for GeneratePress integration
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-theme-integration.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'admin/class-buildfuly-admin.php';

		/**
		 * The class responsible for REST API endpoints
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-api.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'public/class-buildfuly-public.php';

		/**
		 * The class responsible for rendering templates on the frontend
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'public/partials/buildfuly-template-renderer.php';

		/**
		 * The class responsible for auto-generated navigation and footer
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-navigation.php';

		/**
		 * The class responsible for lead capture and forms
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-leads.php';

		/**
		 * SEO functions - sitemap, robots.txt, analytics, schema
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-seo-functions.php';

		/**
		 * Speed optimization - image compression, lazy loading, performance
		 */
		require_once BUILDFULY_PLUGIN_DIR . 'includes/class-buildfuly-speed-optimizer.php';

		$this->loader = new Buildfuly_Loader();
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {
		$plugin_i18n = new Buildfuly_i18n();
		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {
		$plugin_admin = new Buildfuly_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
	$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_admin_menu' );
	$this->loader->add_action( 'admin_init', $plugin_admin, 'register_settings' );
	$this->loader->add_action( 'wp_ajax_buildfuly_install_theme', $plugin_admin, 'ajax_install_theme' );
	$this->loader->add_action( 'wp_ajax_buildfuly_save_theme_colors', $plugin_admin, 'ajax_save_theme_colors' );
	$this->loader->add_action( 'wp_ajax_buildfuly_save_colors', $plugin_admin, 'ajax_save_colors' );
	$this->loader->add_action( 'wp_ajax_buildfuly_publish_page', $plugin_admin, 'ajax_publish_page' );
	$this->loader->add_action( 'wp_ajax_buildfuly_generate_component', $plugin_admin, 'ajax_generate_component' );
	$this->loader->add_action( 'wp_ajax_buildfuly_save_component', $plugin_admin, 'ajax_save_component' );
	$this->loader->add_action( 'wp_ajax_buildfuly_suggest_pages', $plugin_admin, 'ajax_suggest_pages' );
	$this->loader->add_action( 'wp_ajax_buildfuly_complete_onboarding', $plugin_admin, 'ajax_complete_onboarding' );
	$this->loader->add_action( 'wp_ajax_buildfuly_reset_onboarding', $plugin_admin, 'ajax_reset_onboarding' );
	$this->loader->add_action( 'wp_ajax_buildfuly_regenerate_pages', $plugin_admin, 'ajax_regenerate_pages' );
	$this->loader->add_action( 'wp_ajax_buildfuly_regenerate_section_html', $plugin_admin, 'ajax_regenerate_section_html' );
	$this->loader->add_action( 'wp_ajax_buildfuly_get_image_url', $plugin_admin, 'ajax_get_image_url' );
	$this->loader->add_action( 'wp_ajax_buildfuly_get_usage_stats', $plugin_admin, 'ajax_get_usage_stats' );
	$this->loader->add_action( 'wp_ajax_buildfuly_get_components', $plugin_admin, 'ajax_get_components' );
	$this->loader->add_action( 'wp_ajax_buildfuly_get_seo_metadata', $plugin_admin, 'ajax_get_seo_metadata' );
	$this->loader->add_action( 'wp_ajax_buildfuly_update_seo_metadata', $plugin_admin, 'ajax_update_seo_metadata' );
	$this->loader->add_action( 'wp_ajax_buildfuly_activate_license', $plugin_admin, 'ajax_activate_license' );
	$this->loader->add_action( 'wp_ajax_buildfuly_deactivate_license', $plugin_admin, 'ajax_deactivate_license' );
	$this->loader->add_action( 'wp_ajax_buildfuly_update_option', $plugin_admin, 'ajax_update_option' );
	$this->loader->add_action( 'wp_ajax_buildfuly_set_homepage', $plugin_admin, 'ajax_set_homepage' );
	$this->loader->add_action( 'wp_ajax_buildfuly_sync_page_ids', $plugin_admin, 'ajax_sync_page_ids' );
	$this->loader->add_action( 'wp_ajax_buildfuly_sync_page', $plugin_admin, 'ajax_sync_page' );
	$this->loader->add_action( 'wp_ajax_buildfuly_get_page_url', $plugin_admin, 'ajax_get_page_url' );
	
	// Image generation AJAX handlers
	$this->loader->add_action( 'wp_ajax_buildfuly_upload_image_from_url', $plugin_admin, 'ajax_upload_image_from_url' );
	$this->loader->add_action( 'wp_ajax_buildfuly_get_gallery_images', $plugin_admin, 'ajax_get_gallery_images' );
	$this->loader->add_action( 'wp_ajax_buildfuly_delete_gallery_image', $plugin_admin, 'ajax_delete_gallery_image' );
	$this->loader->add_action( 'wp_ajax_buildfuly_save_auto_generate_images', $plugin_admin, 'ajax_save_auto_generate_images' );
	
	// Logo & Favicon AJAX handlers
	$this->loader->add_action( 'wp_ajax_buildfuly_save_business_logo', $plugin_admin, 'ajax_save_business_logo' );
	$this->loader->add_action( 'wp_ajax_buildfuly_save_favicon', $plugin_admin, 'ajax_save_favicon' );
	$this->loader->add_action( 'wp_ajax_buildfuly_create_favicon_from_logo', $plugin_admin, 'ajax_create_favicon_from_logo' );
	$this->loader->add_action( 'wp_ajax_buildfuly_proxy_image', $plugin_admin, 'ajax_proxy_image' );
	
	// Page Builder Data Storage AJAX handlers (store in wp_postmeta instead of localStorage)
	$this->loader->add_action( 'wp_ajax_buildfuly_save_page_data', $plugin_admin, 'ajax_save_page_data' );
	$this->loader->add_action( 'wp_ajax_buildfuly_load_page_data', $plugin_admin, 'ajax_load_page_data' );
	$this->loader->add_action( 'wp_ajax_buildfuly_load_all_pages', $plugin_admin, 'ajax_load_all_pages' );
	$this->loader->add_action( 'wp_ajax_buildfuly_delete_page_data', $plugin_admin, 'ajax_delete_page_data' );
	
	// WordPress Admin Pages List customization
	$this->loader->add_filter( 'page_row_actions', $plugin_admin, 'add_buildfuly_row_actions', 10, 2 );
	$this->loader->add_filter( 'post_row_actions', $plugin_admin, 'add_buildfuly_row_actions', 10, 2 );
	
	// Auto-update site title to business name
	$this->loader->add_action( 'admin_init', $plugin_admin, 'auto_update_site_title' );
	
	// Auto-build hook for provisioned sites - just sets flag, JS handles generation
	add_action( 'buildfuly_auto_build_site', array( $this, 'handle_auto_build_site' ) );
}

	/**
	 * Handle auto-build for provisioned sites
	 * Sets up the site for auto-generation when admin first visits builder
	 * The actual AI generation happens via existing JS autoGenerateHomepage()
	 *
	 * @since    1.0.0
	 */
	public function handle_auto_build_site() {
		Buildfuly_Logger::debug( 'Buildfuly: Auto-build site hook triggered' );
		
		// Get business settings
		$business_name = get_option( 'buildfuly_business_name', '' );
		
		if ( empty( $business_name ) ) {
			Buildfuly_Logger::debug( 'Buildfuly: Auto-build aborted - no business name' );
			return;
		}
		
		Buildfuly_Logger::debug( 'Buildfuly: Setting up auto-build for: ' . $business_name );
		
		// Create homepage if it doesn't exist
		$homepage_id = get_option( 'page_on_front' );
		
		if ( ! $homepage_id ) {
			// Create blank homepage - content will be generated when admin visits
			$homepage_id = wp_insert_post( array(
				'post_title' => 'Home',
				'post_content' => '', // Empty - will be filled by autoGenerateHomepage()
				'post_status' => 'publish',
				'post_type' => 'page',
				'meta_input' => array(
					'_buildfuly_generated' => '1',
					'_buildfuly_page_type' => 'homepage',
				),
			) );
			
			if ( $homepage_id && ! is_wp_error( $homepage_id ) ) {
				// Set as static homepage
				update_option( 'show_on_front', 'page' );
				update_option( 'page_on_front', $homepage_id );
				Buildfuly_Logger::debug( 'Buildfuly: Created homepage with ID: ' . $homepage_id );
			}
		}
		
		// Set flag for JS to trigger autoGenerateHomepage() on first builder visit
		update_option( 'buildfuly_needs_auto_build', '1' );
		
		// Clear the auto_build trigger flag
		delete_option( 'buildfuly_auto_build' );
		
		Buildfuly_Logger::debug( 'Buildfuly: Auto-build setup complete - waiting for admin visit' );
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {
		$plugin_public = new Buildfuly_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

		// Disable wpautop for Buildfuly-generated pages (must run before content filters)
		$this->loader->add_action( 'template_redirect', $this, 'disable_wpautop_for_buildfuly_pages' );
		
		// Hide theme's default page header/featured image on Buildfuly pages
		$this->loader->add_action( 'wp_head', $this, 'hide_theme_page_header', 1 );

		// Allow display CSS properties in inline styles
		$this->loader->add_filter( 'safe_style_css', $this, 'allow_display_in_inline_styles' );

		// Disable content filtering when saving Buildfuly pages
		$this->loader->add_filter( 'content_save_pre', $this, 'disable_content_filtering_on_save', 9 );

		// Add SEO meta tags
		require_once plugin_dir_path( __FILE__ ) . 'class-buildfuly-seo.php';
		require_once plugin_dir_path( __FILE__ ) . 'class-buildfuly-legal-pages.php';
		$seo = new Buildfuly_SEO();
		$this->loader->add_action( 'wp_head', $seo, 'output_seo_meta_tags', 1 );
		
		// Override WordPress document title with Buildfuly SEO title
		$this->loader->add_filter( 'pre_get_document_title', $seo, 'filter_document_title', 10 );
		$this->loader->add_filter( 'document_title_parts', $seo, 'filter_document_title_parts', 10 );

		// Add global LocalBusiness schema from SEO settings
		$this->loader->add_action( 'wp_head', 'Buildfuly_SEO', 'output_local_business_schema', 5 );
		
		// Add Breadcrumb schema for better SERP display
		$this->loader->add_action( 'wp_head', 'Buildfuly_SEO', 'output_breadcrumb_schema', 6 );
		
		// Add FAQ schema for rich snippets
		$this->loader->add_action( 'wp_head', 'Buildfuly_SEO', 'output_faq_schema_from_content', 7 );
		
		// Add Service schema for service pages
		$this->loader->add_action( 'wp_head', 'Buildfuly_SEO', 'output_service_schema', 8 );

		// Add Google Maps to footer if enabled
		$this->loader->add_action( 'wp_footer', 'Buildfuly_SEO', 'output_footer_map', 10 );

		// Register map shortcode
		add_shortcode( 'buildfuly_map', array( 'Buildfuly_SEO', 'map_shortcode' ) );

		// Initialize auto-generated navigation and footer
		new Buildfuly_Navigation();
	}

	/**
	 * Register all of the hooks related to the REST API
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_api_hooks() {
		$plugin_api = new Buildfuly_API( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'rest_api_init', $plugin_api, 'register_routes' );
	}

	/**
	 * Disable wpautop for Buildfuly-generated pages
	 *
	 * @since    1.0.0
	 */
	/**
	 * Disable wpautop for Buildfuly-generated pages
	 *
	 * @since    1.0.0
	 */
	public function disable_wpautop_for_buildfuly_pages() {
		// Check if this is a Buildfuly-generated page
		if ( is_page() && get_post_meta( get_the_ID(), 'buildfuly_generated', true ) ) {
			// Remove wpautop and wptexturize filters before content is processed
			remove_filter( 'the_content', 'wpautop' );
			remove_filter( 'the_content', 'wptexturize' );
			
			// Remove GeneratePress content filters that might interfere
			remove_filter( 'the_content', 'convert_chars' );
			remove_filter( 'the_content', 'wpautop', 10 );
			
			// Add filter to return content as-is
			add_filter( 'the_content', function( $content ) {
				if ( is_page() && get_post_meta( get_the_ID(), 'buildfuly_generated', true ) ) {
					// Process shortcodes in the content (for contact forms, etc.)
					return do_shortcode( $content );
				}
				return $content;
			}, 99 );
		}
	}
	
	/**
	 * Hide theme's default page header/featured image on Buildfuly pages
	 * Injects CSS directly into head to ensure it loads before theme styles
	 *
	 * @since    1.0.0
	 */
	public function hide_theme_page_header() {
		// Only on Buildfuly-generated pages
		if ( ! is_page() ) {
			return;
		}
		
		$post_id = get_the_ID();
		if ( ! $post_id ) {
			return;
		}
		
		// Check if this is a Buildfuly page
		$is_buildfuly = get_post_meta( $post_id, 'buildfuly_generated', true ) || 
		                get_post_meta( $post_id, '_buildfuly_sections', true );
		
		if ( ! $is_buildfuly ) {
			return;
		}
		
		// Output CSS to hide theme's page header elements
		?>
		<style id="buildfuly-hide-theme-header">
			/* Hide theme's default page title and featured image */
			.wp-block-post-title,
			.wp-block-post-featured-image,
			figure.wp-block-post-featured-image,
			h1.wp-block-post-title,
			.entry-header,
			.page-header,
			.entry-title,
			.page-title,
			article > header,
			.post-thumbnail,
			.entry-hero,
			.page-hero,
			/* Twenty Twenty-One/Two/Three/Four specific */
			.singular .entry-header,
			.singular .post-thumbnail,
			/* Astra specific */
			.ast-single-post-order .entry-header,
			/* GeneratePress specific */
			.inside-article > header,
			/* Kadence specific */
			.entry-header.single-entry-header,
			/* Flavor theme / FSE themes */
			.wp-block-template-part figure.wp-block-post-featured-image,
			.wp-block-template-part .wp-block-post-title {
				display: none !important;
			}
			
			/* Remove theme's wrapper padding that adds space above content */
			.wp-block-group.has-global-padding {
				padding-top: 0 !important;
				padding-bottom: 0 !important;
			}
			
			/* Remove margin from featured image figure */
			figure.wp-block-post-featured-image {
				margin: 0 !important;
			}
			
			/* Ensure entry-content has no top padding/margin */
			.entry-content,
			.wp-block-post-content {
				padding-top: 0 !important;
				margin-top: 0 !important;
			}
			
			/* CRITICAL: Allow full-width breakout from WordPress constrained layout */
			.is-layout-constrained,
			.wp-block-post-content-is-layout-constrained,
			.entry-content,
			.wp-block-post-content {
				overflow: visible !important;
			}
			
			/* Remove max-width constraint for full-width Buildfuly components - but keep their internal positioning */
			.is-layout-constrained > .buildfuly-hero-wide,
			.is-layout-constrained > .buildfuly-hero-centered-wide,
			.is-layout-constrained > .buildfuly-testimonial,
			.is-layout-constrained > .buildfuly-testimonial-v2,
			.is-layout-constrained > .buildfuly-cta-banner,
			.is-layout-constrained > [class*="buildfuly-hero"],
			.is-layout-constrained > [class*="buildfuly-testimonial"],
			.is-layout-constrained > [class*="buildfuly-cta-banner"] {
				max-width: none !important;
			}
		</style>
		<?php
	}

	/**
	 * Allow display CSS properties in inline styles
	 *
	 * @since    1.0.0
	 */
	public function allow_display_in_inline_styles( $styles ) {
		$styles[] = 'display';
		$styles[] = 'flex';
		$styles[] = 'flex-direction';
		$styles[] = 'flex-wrap';
		$styles[] = 'gap';
		$styles[] = 'align-items';
		$styles[] = 'justify-content';
		$styles[] = 'background-image';
		$styles[] = 'background-size';
		$styles[] = 'background-position';
		$styles[] = 'filter';
		$styles[] = 'transform';
		$styles[] = 'opacity';
		$styles[] = 'z-index';
		$styles[] = 'position';
		$styles[] = 'top';
		$styles[] = 'left';
		$styles[] = 'right';
		$styles[] = 'bottom';
		$styles[] = 'object-fit';
		$styles[] = 'overflow';
		$styles[] = 'overflow-x';
		$styles[] = 'overflow-y';
		$styles[] = 'flex-shrink';
		$styles[] = '-webkit-overflow-scrolling';
		$styles[] = 'scrollbar-width';
		$styles[] = 'scroll-behavior';
		$styles[] = 'min-width';
		$styles[] = 'max-width';
		$styles[] = 'box-shadow';
		return $styles;
	}

	/**

	/**
	 * Disable content filtering when saving Buildfuly pages
	 *
	 * @since    1.0.0
	 */
	public function disable_content_filtering_on_save( $content ) {
		// Check if we're saving a Buildfuly page
		// phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- WordPress core save_post context, nonce verified by WordPress
		if ( isset( $_POST['buildfuly_generated'] ) || ( isset( $_POST['post_ID'] ) && get_post_meta( absint( $_POST['post_ID'] ), 'buildfuly_generated', true ) ) ) {
			// Remove kses filtering temporarily
			kses_remove_filters();
		}
		return $content;
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Buildfuly_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}
}
