<?php

/**
 * Theme integration handler
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

/**
 * Theme integration handler.
 *
 * Handles integration with GeneratePress and Kadence themes.
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @author     Buildfuly Team
 */
class Buildfuly_Theme_Integration {

	/**
	 * The style manager.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Buildfuly_Style_Manager    $style_manager    Style manager instance.
	 */
	private $style_manager;

	/**
	 * Initialize the class.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		$this->style_manager = new Buildfuly_Style_Manager();
		$this->init_hooks();
	}

	/**
	 * Initialize WordPress hooks.
	 *
	 * @since    1.0.0
	 */
	private function init_hooks() {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_theme_styles' ), 999 );
		
		// GeneratePress specific hooks
		if ( $this->is_generatepress_active() ) {
			add_filter( 'generate_typography_default_fonts', array( $this, 'modify_generatepress_fonts' ) );
			add_filter( 'generate_colors_defaults', array( $this, 'modify_generatepress_colors' ) );
		}

		// Kadence specific hooks
		if ( $this->is_kadence_active() ) {
			add_filter( 'kadence_theme_options', array( $this, 'modify_kadence_options' ) );
		}
	}

	/**
	 * Check if GeneratePress is active.
	 *
	 * @since    1.0.0
	 * @return   bool    True if GeneratePress is active
	 */
	private function is_generatepress_active() {
		return function_exists( 'generate_get_color_defaults' );
	}

	/**
	 * Check if Kadence is active.
	 *
	 * @since    1.0.0
	 * @return   bool    True if Kadence is active
	 */
	private function is_kadence_active() {
		return class_exists( 'Kadence\Theme' );
	}

	/**
	 * Enqueue custom theme styles.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_theme_styles() {
		// Generate dynamic CSS
		$custom_css = $this->style_manager->get_css_output();
		
		// Add inline styles
		wp_add_inline_style( 'buildfuly-public', $custom_css );
	}

	/**
	 * Modify GeneratePress fonts.
	 *
	 * @since    1.0.0
	 * @param    array    $fonts    Default fonts
	 * @return   array              Modified fonts
	 */
	public function modify_generatepress_fonts( $fonts ) {
		$custom_fonts = $this->style_manager->get_component_styles( 'fonts', array() );

		if ( ! empty( $custom_fonts['heading'] ) ) {
			$fonts['font_body'] = $custom_fonts['body'];
		}

		if ( ! empty( $custom_fonts['body'] ) ) {
			$fonts['font_heading_1'] = $custom_fonts['heading'];
			$fonts['font_heading_2'] = $custom_fonts['heading'];
			$fonts['font_heading_3'] = $custom_fonts['heading'];
		}

		return $fonts;
	}

	/**
	 * Modify GeneratePress colors.
	 *
	 * @since    1.0.0
	 * @param    array    $colors    Default colors
	 * @return   array               Modified colors
	 */
	public function modify_generatepress_colors( $colors ) {
		$custom_colors = $this->style_manager->get_component_styles( 'colors', array() );

		if ( ! empty( $custom_colors['primary'] ) ) {
			$colors['link_color'] = $custom_colors['primary'];
		}

		if ( ! empty( $custom_colors['text'] ) ) {
			$colors['text_color'] = $custom_colors['text'];
		}

		if ( ! empty( $custom_colors['background'] ) ) {
			$colors['background_color'] = $custom_colors['background'];
		}

		return $colors;
	}

	/**
	 * Modify Kadence theme options.
	 *
	 * @since    1.0.0
	 * @param    array    $options    Theme options
	 * @return   array                Modified options
	 */
	public function modify_kadence_options( $options ) {
		$custom_colors = $this->style_manager->get_component_styles( 'colors', array() );
		$custom_fonts = $this->style_manager->get_component_styles( 'fonts', array() );

		// Apply color modifications
		if ( ! empty( $custom_colors['primary'] ) ) {
			$options['palette']['palette'][0]['color'] = $custom_colors['primary'];
		}

		// Apply font modifications
		if ( ! empty( $custom_fonts['body'] ) ) {
			$options['base_font']['family'] = $custom_fonts['body'];
		}

		if ( ! empty( $custom_fonts['heading'] ) ) {
			$options['heading_font']['family'] = $custom_fonts['heading'];
		}

		return $options;
	}

	/**
	 * Get active theme name.
	 *
	 * @since    1.0.0
	 * @return   string    Theme name
	 */
	public function get_active_theme() {
		if ( $this->is_generatepress_active() ) {
			return 'generatepress';
		}

		if ( $this->is_kadence_active() ) {
			return 'kadence';
		}

		return 'unknown';
	}

	/**
	 * Check if integration is supported.
	 *
	 * @since    1.0.0
	 * @return   bool    True if theme integration is supported
	 */
	public function is_supported() {
		return $this->is_generatepress_active() || $this->is_kadence_active();
	}

	/**
	 * Get theme customizer URL.
	 *
	 * @since    1.0.0
	 * @return   string    Customizer URL
	 */
	public function get_customizer_url() {
		return admin_url( 'customize.php?autofocus[panel]=buildfuly_panel' );
	}
}
