<?php

/**
 * Style management handler
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

/**
 * Style management handler.
 *
 * Handles styling options, AI style generation, and theme integration.
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @author     Buildfuly Team
 */
class Buildfuly_Style_Manager {

	/**
	 * The database handler.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Buildfuly_Database    $db    Database handler instance.
	 */
	private $db;

	/**
	 * Initialize the class.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		$this->db = new Buildfuly_Database();
	}

	/**
	 * Get default styles based on business type.
	 *
	 * @since    1.0.0
	 * @param    string    $business_type    The business type
	 * @return   array                       Default styles
	 */
	public function get_default_styles( $business_type = 'saas' ) {
		$defaults = array(
			'saas' => array(
				'colors' => array(
					'primary'   => '#007bff',
					'secondary' => '#6c757d',
					'accent'    => '#28a745',
					'text'      => '#212529',
					'background' => '#ffffff',
				),
				'fonts' => array(
					'heading' => 'Inter',
					'body'    => 'Inter',
				),
				'layout' => array(
					'hero_layout'     => 'centered',
					'features_layout' => 'grid-3-col',
					'cta_position'    => 'bottom',
				),
			),
			'professional_service' => array(
				'colors' => array(
					'primary'   => '#2c3e50',
					'secondary' => '#34495e',
					'accent'    => '#e74c3c',
					'text'      => '#2c3e50',
					'background' => '#ffffff',
				),
				'fonts' => array(
					'heading' => 'Playfair Display',
					'body'    => 'Source Sans Pro',
				),
				'layout' => array(
					'hero_layout'     => 'split',
					'features_layout' => 'list',
					'cta_position'    => 'sidebar',
				),
			),
			'consulting' => array(
				'colors' => array(
					'primary'   => '#1e3a8a',
					'secondary' => '#475569',
					'accent'    => '#f59e0b',
					'text'      => '#1e293b',
					'background' => '#f8fafc',
				),
				'fonts' => array(
					'heading' => 'Montserrat',
					'body'    => 'Open Sans',
				),
				'layout' => array(
					'hero_layout'     => 'centered',
					'features_layout' => 'grid-2-col',
					'cta_position'    => 'inline',
				),
			),
		);

		return isset( $defaults[ $business_type ] ) ? $defaults[ $business_type ] : $defaults['saas'];
	}

	/**
	 * Generate AI styles for a component.
	 *
	 * @since    1.0.0
	 * @param    string    $component    The component to style
	 * @param    array     $params       Generation parameters
	 * @return   array                   Generated styles
	 */
	public function generate_component_styles( $component, $params = array() ) {
		// TODO: Replace with actual AI API call
		// For now, using template-based styles
		
		$business_type = isset( $params['business_type'] ) ? $params['business_type'] : 'saas';
		$defaults = $this->get_default_styles( $business_type );

		$styles = array();

		switch ( $component ) {
			case 'colors':
				$styles = $defaults['colors'];
				break;
			case 'fonts':
				$styles = $defaults['fonts'];
				break;
			case 'layout':
				$styles = $defaults['layout'];
				break;
			case 'hero':
				$styles = array(
					'background_color' => $defaults['colors']['primary'],
					'text_color'       => '#ffffff',
					'layout'           => $defaults['layout']['hero_layout'],
					'padding'          => '80px 20px',
				);
				break;
			case 'features':
				$styles = array(
					'layout'           => $defaults['layout']['features_layout'],
					'card_background'  => '#ffffff',
					'card_border'      => $defaults['colors']['secondary'],
					'icon_color'       => $defaults['colors']['accent'],
				);
				break;
			case 'testimonials':
				$styles = array(
					'background_color' => $defaults['colors']['background'],
					'text_color'       => $defaults['colors']['text'],
					'accent_color'     => $defaults['colors']['accent'],
					'layout'           => 'carousel',
				);
				break;
			case 'cta':
				$styles = array(
					'background_color' => $defaults['colors']['accent'],
					'text_color'       => '#ffffff',
					'button_style'     => 'solid',
					'position'         => $defaults['layout']['cta_position'],
				);
				break;
		}

		// Save generated styles
		$this->save_component_styles( $component, $styles );

		return $styles;
	}

	/**
	 * Save component styles.
	 *
	 * @since    1.0.0
	 * @param    string    $component    The component name
	 * @param    array     $styles       The styles to save
	 * @return   bool                    Success status
	 */
	public function save_component_styles( $component, $styles ) {
		$result = $this->db->save_style( $component, $styles, $component );

		if ( $result ) {
			$this->db->log_history( 'styles_updated', array(
				'component' => $component,
				'styles'    => $styles
			) );
		}

		return $result;
	}

	/**
	 * Get all styles.
	 *
	 * @since    1.0.0
	 * @return   array    All styles
	 */
	public function get_all_styles() {
		return $this->db->get_all_styles();
	}

	/**
	 * Get component styles.
	 *
	 * @since    1.0.0
	 * @param    string    $component    The component name
	 * @param    mixed     $default      Default value
	 * @return   mixed                   Component styles
	 */
	public function get_component_styles( $component, $default = null ) {
		return $this->db->get_style( $component, $default );
	}

	/**
	 * Update single style property.
	 *
	 * @since    1.0.0
	 * @param    string    $component    The component name
	 * @param    string    $property     The property name
	 * @param    mixed     $value        The value
	 * @return   bool                    Success status
	 */
	public function update_style_property( $component, $property, $value ) {
		$current_styles = $this->get_component_styles( $component, array() );
		
		if ( ! is_array( $current_styles ) ) {
			$current_styles = array();
		}

		$current_styles[ $property ] = $value;

		return $this->save_component_styles( $component, $current_styles );
	}

	/**
	 * Generate complete theme styles.
	 *
	 * @since    1.0.0
	 * @param    string    $business_type    The business type
	 * @return   array                       Complete theme styles
	 */
	public function generate_complete_theme( $business_type = 'saas' ) {
		$defaults = $this->get_default_styles( $business_type );

		// Save all default styles
		$this->db->save_style( 'colors', $defaults['colors'], 'global' );
		$this->db->save_style( 'fonts', $defaults['fonts'], 'global' );
		$this->db->save_style( 'layout', $defaults['layout'], 'global' );

		// Generate component-specific styles
		$components = array( 'hero', 'features', 'testimonials', 'cta' );
		foreach ( $components as $component ) {
			$this->generate_component_styles( $component, array( 'business_type' => $business_type ) );
		}

		$this->db->log_history( 'theme_generated', array(
			'business_type' => $business_type
		) );

		return $this->get_all_styles();
	}

	/**
	 * Get CSS output for current styles.
	 *
	 * @since    1.0.0
	 * @return   string    CSS code
	 */
	public function get_css_output() {
		// Get colors from settings (not database)
		$primary = get_option( 'buildfuly_primary_color', '#00bfa5' );
		$secondary = get_option( 'buildfuly_secondary_color', '#00897b' );
		$accent = get_option( 'buildfuly_accent_color', '#7c4dff' );
		$text = get_option( 'buildfuly_text_color', '#1e293b' );
		$text_light = get_option( 'buildfuly_text_light_color', '#6c757d' );
		$bg = get_option( 'buildfuly_bg_color', '#000000' );
		$star = get_option( 'buildfuly_star_color', '#fbbf24' );
		
		$fonts = $this->get_component_styles( 'fonts', array() );

		$css = ":root {\n";

		// Add color variables with variations
		$css .= "  --buildfuly-color-primary: {$primary};\n";
		$css .= "  --buildfuly-color-primary-light: " . $this->hex_to_rgba( $primary, 0.1 ) . ";\n";
		$css .= "  --buildfuly-color-primary-dark: " . $this->darken_color( $primary, 15 ) . ";\n";
		$css .= "  --buildfuly-color-primary-text: " . $this->get_contrast_color( $primary ) . ";\n";
		$css .= "  --buildfuly-color-secondary: {$secondary};\n";
		$css .= "  --buildfuly-color-accent: {$accent};\n";
		$css .= "  --buildfuly-color-text: {$text};\n";
		$css .= "  --buildfuly-color-text-light: {$text_light};\n";
		$css .= "  --buildfuly-color-background: {$bg};\n";
		$css .= "  --buildfuly-color-star: {$star};\n";

		// Add font variables
		if ( ! empty( $fonts ) ) {
			foreach ( $fonts as $key => $value ) {
				$css .= "  --buildfuly-font-{$key}: '{$value}', sans-serif;\n";
			}
		}

		$css .= "}\n\n";

		// Add base styles
		$css .= ".buildfuly-page {\n";
		if ( isset( $fonts['body'] ) ) {
			$css .= "  font-family: var(--buildfuly-font-body);\n";
		}
		$css .= "  color: var(--buildfuly-color-text);\n";
		$css .= "}\n\n";
		
		// Make sure body and all WordPress wrappers use background color
		$css .= "body, html, .site, .site-content, .entry-content, #primary, #content, #page, .hfeed, article, .post, .page {\n";
		$css .= "  background-color: var(--buildfuly-color-background) !important;\n";
		$css .= "}\n\n";
		
		// Remove any padding/margin that creates white space
		$css .= ".entry-content, .site-content, #primary, #content {\n";
		$css .= "  padding: 0 !important;\n";
		$css .= "  margin: 0 !important;\n";
		$css .= "}\n\n";
		
		// Override GeneratePress theme containers
		$css .= ".separate-containers .inside-article, .separate-containers .comments-area, .separate-containers .page-header, .one-container .container, .separate-containers .paging-navigation, .inside-page-header {\n";
		$css .= "  background-color: var(--buildfuly-color-background) !important;\n";
		$css .= "}\n\n";

		$css .= ".buildfuly-page h1, .buildfuly-page h2, .buildfuly-page h3 {\n";
		if ( isset( $fonts['heading'] ) ) {
			$css .= "  font-family: var(--buildfuly-font-heading);\n";
		}
		$css .= "}\n";

		return $css;
	}
	
	/**
	 * Convert hex color to rgba
	 */
	private function hex_to_rgba( $hex, $alpha = 1 ) {
		$hex = str_replace( '#', '', $hex );
		if ( strlen( $hex ) == 3 ) {
			$r = hexdec( substr( $hex, 0, 1 ) . substr( $hex, 0, 1 ) );
			$g = hexdec( substr( $hex, 1, 1 ) . substr( $hex, 1, 1 ) );
			$b = hexdec( substr( $hex, 2, 1 ) . substr( $hex, 2, 1 ) );
		} else {
			$r = hexdec( substr( $hex, 0, 2 ) );
			$g = hexdec( substr( $hex, 2, 2 ) );
			$b = hexdec( substr( $hex, 4, 2 ) );
		}
		return "rgba({$r}, {$g}, {$b}, {$alpha})";
	}
	
	/**
	 * Darken a hex color
	 */
	private function darken_color( $hex, $percent ) {
		$hex = str_replace( '#', '', $hex );
		$r = hexdec( substr( $hex, 0, 2 ) );
		$g = hexdec( substr( $hex, 2, 2 ) );
		$b = hexdec( substr( $hex, 4, 2 ) );
		
		$r = max( 0, $r - ( $r * $percent / 100 ) );
		$g = max( 0, $g - ( $g * $percent / 100 ) );
		$b = max( 0, $b - ( $b * $percent / 100 ) );
		
		return sprintf( "#%02x%02x%02x", $r, $g, $b );
	}
	
	/**
	 * Get contrasting color (white or black) based on background
	 */
	private function get_contrast_color( $hex ) {
		$hex = str_replace( '#', '', $hex );
		$r = hexdec( substr( $hex, 0, 2 ) );
		$g = hexdec( substr( $hex, 2, 2 ) );
		$b = hexdec( substr( $hex, 4, 2 ) );
		
		// Calculate relative luminance
		$luminance = ( 0.299 * $r + 0.587 * $g + 0.114 * $b ) / 255;
		
		// Return white for dark backgrounds, black for light backgrounds
		return $luminance > 0.5 ? '#000000' : '#ffffff';
	}

	/**
	 * Clear all styles.
	 *
	 * @since    1.0.0
	 * @return   bool    Success status
	 */
	public function clear_all_styles() {
		$result = $this->db->delete_all_styles();
		
		if ( $result ) {
			$this->db->log_history( 'styles_cleared' );
		}

		return $result;
	}
}
