<?php
/**
 * Speed Optimization & Image Compression
 *
 * Handles automatic image compression, lazy loading, caching, minification,
 * and all performance optimizations for better Google PageSpeed scores.
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Buildfuly_Speed_Optimizer {

	/**
	 * Initialize the speed optimizer
	 */
	public function __construct() {
		// Image compression on upload
		add_filter( 'wp_handle_upload', array( $this, 'compress_on_upload' ) );
		add_filter( 'wp_generate_attachment_metadata', array( $this, 'compress_thumbnails' ), 10, 2 );
		
		// Lazy loading
		add_filter( 'the_content', array( $this, 'add_lazy_loading' ), 99 );
		add_filter( 'post_thumbnail_html', array( $this, 'add_lazy_loading_to_thumbnails' ), 99, 5 );
		
		// WebP conversion
		add_filter( 'wp_generate_attachment_metadata', array( $this, 'generate_webp_versions' ), 10, 2 );
		
		// Defer JavaScript
		add_filter( 'script_loader_tag', array( $this, 'defer_scripts' ), 10, 3 );
		
		// Preload critical resources
		add_action( 'wp_head', array( $this, 'add_resource_hints' ), 1 );
		
		// Remove query strings from static resources
		add_filter( 'style_loader_src', array( $this, 'remove_query_strings' ), 10, 2 );
		add_filter( 'script_loader_src', array( $this, 'remove_query_strings' ), 10, 2 );
		
		// Optimize CSS delivery
		add_action( 'wp_enqueue_scripts', array( $this, 'optimize_css_delivery' ), 999 );
		
		// Add AJAX handlers
		add_action( 'wp_ajax_buildfuly_compress_all_images', array( $this, 'ajax_compress_all_images' ) );
		add_action( 'wp_ajax_buildfuly_generate_webp_all', array( $this, 'ajax_generate_webp_all' ) );
		add_action( 'wp_ajax_buildfuly_clear_cache', array( $this, 'ajax_clear_cache' ) );
		add_action( 'wp_ajax_buildfuly_count_backups', array( $this, 'ajax_count_backups' ) );
		add_action( 'wp_ajax_buildfuly_restore_originals', array( $this, 'ajax_restore_originals' ) );
	}

	/**
	 * Compress image on upload
	 */
	public function compress_on_upload( $upload ) {
		// Check if automatic compression is enabled
		if ( ! get_option( 'buildfuly_auto_compress', true ) ) {
			return $upload;
		}

		if ( ! isset( $upload['file'] ) || ! isset( $upload['type'] ) ) {
			return $upload;
		}

		// Only process images
		if ( strpos( $upload['type'], 'image/' ) !== 0 ) {
			return $upload;
		}

		$file_path = $upload['file'];
		$image_type = $upload['type'];

		// Get quality setting (default 82 for good balance)
		$quality = get_option( 'buildfuly_image_quality', 82 );

		// Compress based on image type
		if ( $image_type === 'image/jpeg' || $image_type === 'image/jpg' ) {
			$this->compress_jpeg( $file_path, $quality );
		} elseif ( $image_type === 'image/png' ) {
			$this->compress_png( $file_path, $quality );
		}

		return $upload;
	}

	/**
	 * Compress JPEG image
	 */
	private function compress_jpeg( $file_path, $quality = 82, $force = false ) {
		if ( ! function_exists( 'imagecreatefromjpeg' ) ) {
			return false;
		}

		// Get original file size
		$original_size = filesize( $file_path );

		// Create backup of original file
		$backup_file = $file_path . '.original';
		if ( ! file_exists( $backup_file ) ) {
			copy( $file_path, $backup_file );
		}

		// NUCLEAR OPTION: If file is bloated (>500KB) and force mode is on,
		// re-encode from the backup if it exists and is smaller
		if ( $force && $original_size > 512000 && file_exists( $backup_file ) ) {
			$backup_size = filesize( $backup_file );
			if ( $backup_size < $original_size ) {
				error_log( sprintf( '[Buildfuly] FORCE MODE: Using backup for %s (%s vs %s)', 
					basename( $file_path ), size_format( $backup_size ), size_format( $original_size ) ) );
				$image = imagecreatefromjpeg( $backup_file );
			} else {
				$image = imagecreatefromjpeg( $file_path );
			}
		} else {
			$image = imagecreatefromjpeg( $file_path );
		}
		
		if ( ! $image ) {
			return false;
		}

		// Create temporary file for compressed version
		$temp_file = $file_path . '.tmp';

		// Save compressed version to temp file
		$result = imagejpeg( $image, $temp_file, $quality );
		imagedestroy( $image );

		if ( ! $result || ! file_exists( $temp_file ) ) {
			return false;
		}

		// Check if compression actually made it smaller
		$compressed_size = filesize( $temp_file );
		
		// Debug logging
		error_log( sprintf( 
			'[Buildfuly] JPEG compression: %s | Original: %s | Compressed: %s | Quality: %d | Force: %s',
			basename( $file_path ),
			size_format( $original_size ),
			size_format( $compressed_size ),
			$quality,
			$force ? 'YES' : 'NO'
		) );
		
		// Force mode: always replace, or safe mode: only if smaller
		if ( $force || $compressed_size < $original_size ) {
			// Compression worked! Replace original with compressed version
			rename( $temp_file, $file_path );
			error_log( sprintf( '[Buildfuly] ✓ Compressed %s (saved %s)', basename( $file_path ), size_format( $original_size - $compressed_size ) ) );
			return true;
		} else {
			// Compression made it bigger! Keep original and delete temp
			unlink( $temp_file );
			error_log( sprintf( '[Buildfuly] ✗ Skipped %s (would increase by %s)', basename( $file_path ), size_format( $compressed_size - $original_size ) ) );
			return false;
		}
	}

	/**
	 * Compress PNG image
	 */
	private function compress_png( $file_path, $quality = 82, $force = false ) {
		if ( ! function_exists( 'imagecreatefrompng' ) ) {
			return false;
		}

		// Get original file size
		$original_size = filesize( $file_path );

		// Create backup of original file
		$backup_file = $file_path . '.original';
		if ( ! file_exists( $backup_file ) ) {
			copy( $file_path, $backup_file );
		}

		// NUCLEAR OPTION: If file is bloated (>500KB) and force mode is on,
		// re-encode from the backup if it exists and is smaller
		if ( $force && $original_size > 512000 && file_exists( $backup_file ) ) {
			$backup_size = filesize( $backup_file );
			if ( $backup_size < $original_size ) {
				error_log( sprintf( '[Buildfuly] FORCE MODE: Using backup for %s (%s vs %s)', 
					basename( $file_path ), size_format( $backup_size ), size_format( $original_size ) ) );
				$image = imagecreatefrompng( $backup_file );
			} else {
				$image = imagecreatefrompng( $file_path );
			}
		} else {
			$image = imagecreatefrompng( $file_path );
		}
		
		if ( ! $image ) {
			return false;
		}

		// Enable alpha transparency
		imagealphablending( $image, false );
		imagesavealpha( $image, true );

		// PNG compression: 0 = no compression, 9 = max compression
		// For PNG, always use maximum compression (9) since it's lossless
		// The quality slider doesn't affect PNG visual quality, only compression effort
		$compression_level = 9;
		
		error_log( sprintf( '[Buildfuly] PNG compression: %s | Original size: %s | Using compression level: %d', 
			basename( $file_path ), size_format( $original_size ), $compression_level ) );

		// Create temporary file for compressed version
		$temp_file = $file_path . '.tmp';

		$result = imagepng( $image, $temp_file, $compression_level );
		imagedestroy( $image );

		if ( ! $result || ! file_exists( $temp_file ) ) {
			return false;
		}

		// Check if compression actually made it smaller
		$compressed_size = filesize( $temp_file );
		
		// Force mode: always replace, or safe mode: only if smaller
		if ( $force || $compressed_size < $original_size ) {
			// Compression worked! Replace original with compressed version
			rename( $temp_file, $file_path );
			return true;
		} else {
			// Compression made it bigger! Keep original and delete temp
			unlink( $temp_file );
			return false;
		}
	}

	/**
	 * Compress all generated thumbnails
	 */
	public function compress_thumbnails( $metadata, $attachment_id, $force = false ) {
		if ( ! isset( $metadata['sizes'] ) || ! is_array( $metadata['sizes'] ) ) {
			return $metadata;
		}

		$upload_dir = wp_get_upload_dir();
		$file_path = get_attached_file( $attachment_id );
		$base_dir = dirname( $file_path );
		
		$quality = get_option( 'buildfuly_image_quality', 82 );
		$mime_type = get_post_mime_type( $attachment_id );

		foreach ( $metadata['sizes'] as $size => $size_data ) {
			$thumbnail_path = $base_dir . '/' . $size_data['file'];
			
			if ( file_exists( $thumbnail_path ) ) {
				if ( $mime_type === 'image/jpeg' || $mime_type === 'image/jpg' ) {
					$this->compress_jpeg( $thumbnail_path, $quality, $force );
				} elseif ( $mime_type === 'image/png' ) {
					$this->compress_png( $thumbnail_path, $quality, $force );
				}
			}
		}

		return $metadata;
	}

	/**
	 * Generate WebP versions of images
	 */
	public function generate_webp_versions( $metadata, $attachment_id ) {
		// Check if WebP is enabled
		if ( ! get_option( 'buildfuly_enable_webp', true ) ) {
			return $metadata;
		}

		if ( ! function_exists( 'imagewebp' ) ) {
			return $metadata;
		}

		$file_path = get_attached_file( $attachment_id );
		$mime_type = get_post_mime_type( $attachment_id );

		// Generate WebP for main image
		$this->create_webp_version( $file_path, $mime_type );

		// Generate WebP for thumbnails
		if ( isset( $metadata['sizes'] ) && is_array( $metadata['sizes'] ) ) {
			$base_dir = dirname( $file_path );
			
			foreach ( $metadata['sizes'] as $size => $size_data ) {
				$thumbnail_path = $base_dir . '/' . $size_data['file'];
				if ( file_exists( $thumbnail_path ) ) {
					$this->create_webp_version( $thumbnail_path, $mime_type );
				}
			}
		}

		return $metadata;
	}

	/**
	 * Create WebP version of an image
	 */
	private function create_webp_version( $file_path, $mime_type ) {
		$webp_path = preg_replace( '/\.(jpe?g|png)$/i', '.webp', $file_path );
		
		// Skip if WebP already exists
		if ( file_exists( $webp_path ) ) {
			return;
		}

		$quality = get_option( 'buildfuly_webp_quality', 80 );

		if ( $mime_type === 'image/jpeg' || $mime_type === 'image/jpg' ) {
			$image = imagecreatefromjpeg( $file_path );
		} elseif ( $mime_type === 'image/png' ) {
			$image = imagecreatefrompng( $file_path );
			imagealphablending( $image, false );
			imagesavealpha( $image, true );
		} else {
			return;
		}

		if ( $image ) {
			imagewebp( $image, $webp_path, $quality );
			imagedestroy( $image );
		}
	}

	/**
	 * Add lazy loading to images in content
	 */
	public function add_lazy_loading( $content ) {
		if ( ! get_option( 'buildfuly_enable_lazy_load', true ) ) {
			return $content;
		}

		// Skip if lazy loading already applied
		if ( strpos( $content, 'loading="lazy"' ) !== false ) {
			return $content;
		}

		// Add loading="lazy" to all images
		$content = preg_replace(
			'/<img((?![^>]*loading=)[^>]*)>/i',
			'<img$1 loading="lazy">',
			$content
		);

		// Add width and height attributes if missing (CLS optimization)
		$content = $this->add_image_dimensions( $content );

		return $content;
	}

	/**
	 * Add lazy loading to featured images
	 */
	public function add_lazy_loading_to_thumbnails( $html, $post_id, $post_thumbnail_id, $size, $attr ) {
		if ( ! get_option( 'buildfuly_enable_lazy_load', true ) ) {
			return $html;
		}

		if ( strpos( $html, 'loading=' ) === false ) {
			$html = str_replace( '<img', '<img loading="lazy"', $html );
		}

		return $html;
	}

	/**
	 * Add width/height to images to prevent CLS
	 */
	private function add_image_dimensions( $content ) {
		// Find all img tags without width/height
		preg_match_all( '/<img[^>]+>/i', $content, $images );
		
		if ( empty( $images[0] ) ) {
			return $content;
		}

		foreach ( $images[0] as $image_tag ) {
			// Skip if already has dimensions
			if ( strpos( $image_tag, 'width=' ) !== false && strpos( $image_tag, 'height=' ) !== false ) {
				continue;
			}

			// Extract src
			if ( preg_match( '/src=["\']([^"\']+)["\']/', $image_tag, $src_match ) ) {
				$src = $src_match[1];
				
				// Get image dimensions
				$attachment_id = attachment_url_to_postid( $src );
				if ( $attachment_id ) {
					$image_meta = wp_get_attachment_metadata( $attachment_id );
					if ( isset( $image_meta['width'] ) && isset( $image_meta['height'] ) ) {
						$new_image_tag = str_replace(
							'<img',
							'<img width="' . $image_meta['width'] . '" height="' . $image_meta['height'] . '"',
							$image_tag
						);
						$content = str_replace( $image_tag, $new_image_tag, $content );
					}
				}
			}
		}

		return $content;
	}

	/**
	 * Defer non-critical JavaScript
	 */
	public function defer_scripts( $tag, $handle, $src ) {
		if ( ! get_option( 'buildfuly_defer_js', true ) ) {
			return $tag;
		}

		// Skip if already deferred or async
		if ( strpos( $tag, 'defer' ) !== false || strpos( $tag, 'async' ) !== false ) {
			return $tag;
		}

		// Don't defer jQuery or critical scripts
		$skip_handles = array( 'jquery', 'jquery-core', 'jquery-migrate' );
		if ( in_array( $handle, $skip_handles ) ) {
			return $tag;
		}

		// Add defer attribute
		return str_replace( ' src=', ' defer src=', $tag );
	}

	/**
	 * Add resource hints for faster loading
	 */
	public function add_resource_hints() {
		if ( ! get_option( 'buildfuly_enable_resource_hints', true ) ) {
			return;
		}

		// DNS prefetch for external domains
		$external_domains = array(
			'fonts.googleapis.com',
			'fonts.gstatic.com',
		);

		foreach ( $external_domains as $domain ) {
			echo '<link rel="dns-prefetch" href="//' . esc_attr( $domain ) . '">' . "\n";
		}

		// Preconnect to critical origins
		echo '<link rel="preconnect" href="https://fonts.googleapis.com">' . "\n";
		echo '<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>' . "\n";

		// Preload critical assets
		$logo_id = get_option( 'buildfuly_business_logo' );
		if ( $logo_id ) {
			$logo_url = wp_get_attachment_url( $logo_id );
			if ( $logo_url ) {
				echo '<link rel="preload" as="image" href="' . esc_url( $logo_url ) . '">' . "\n";
			}
		}
	}

	/**
	 * Remove query strings from static resources
	 */
	public function remove_query_strings( $src, $handle ) {
		if ( ! get_option( 'buildfuly_remove_query_strings', true ) ) {
			return $src;
		}

		if ( strpos( $src, '?ver=' ) !== false ) {
			$src = remove_query_arg( 'ver', $src );
		}

		return $src;
	}

	/**
	 * Optimize CSS delivery
	 */
	public function optimize_css_delivery() {
		if ( ! get_option( 'buildfuly_optimize_css', true ) ) {
			return;
		}

		// Move non-critical CSS to footer
		global $wp_styles;
		
		// List of critical CSS handles to keep in head
		$critical_handles = array( 'wp-block-library', 'global-styles' );
		
		foreach ( $wp_styles->queue as $handle ) {
			if ( ! in_array( $handle, $critical_handles ) ) {
				wp_styles()->add_data( $handle, 'group', 1 );
			}
		}
	}

	/**
	 * AJAX: Compress all existing images
	 */
	public function ajax_compress_all_images() {
		check_ajax_referer( 'buildfuly_speed_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Permission denied' );
		}

		$offset = intval( $_POST['offset'] ?? 0 );
		$batch_size = 10; // Process 10 images at a time
		$force = !empty( $_POST['force'] ) && $_POST['force'] == 1;
		$quality = isset( $_POST['quality'] ) ? intval( $_POST['quality'] ) : get_option( 'buildfuly_image_quality', 82 );
		
		// Ensure quality is within valid range
		$quality = max( 1, min( 100, $quality ) );

		error_log( sprintf( '[Buildfuly] Starting compression batch. Offset: %d, Quality: %d, Force: %s', $offset, $quality, $force ? 'YES' : 'NO' ) );

		$args = array(
			'post_type'      => 'attachment',
			'post_mime_type' => array( 'image/jpeg', 'image/jpg', 'image/png' ),
			'post_status'    => 'inherit',
			'posts_per_page' => $batch_size,
			'offset'         => $offset,
			'orderby'        => 'ID',
			'order'          => 'ASC',
		);

		$attachments = get_posts( $args );
		$total_query = new WP_Query( array_merge( $args, array( 'posts_per_page' => -1, 'fields' => 'ids' ) ) );
		$total = $total_query->found_posts;

		error_log( sprintf( '[Buildfuly] Found %d total images, processing batch of %d at quality %d', $total, count( $attachments ), $quality ) );

		$compressed = 0;

		foreach ( $attachments as $attachment ) {
			$file_path = get_attached_file( $attachment->ID );
			$mime_type = get_post_mime_type( $attachment->ID );

			error_log( sprintf( '[Buildfuly] Processing: %s (ID: %d, Type: %s, Size: %s)', 
				basename( $file_path ), 
				$attachment->ID, 
				$mime_type,
				file_exists( $file_path ) ? size_format( filesize( $file_path ) ) : 'NOT FOUND'
			) );

			if ( file_exists( $file_path ) ) {
				$result = false;
				if ( $mime_type === 'image/jpeg' || $mime_type === 'image/jpg' ) {
					$result = $this->compress_jpeg( $file_path, $quality, $force );
				} elseif ( $mime_type === 'image/png' ) {
					$result = $this->compress_png( $file_path, $quality, $force );
				}

				if ( $result ) {
					$compressed++;
				}

				// Compress thumbnails
				$metadata = wp_get_attachment_metadata( $attachment->ID );
				$this->compress_thumbnails( $metadata, $attachment->ID, $force );
			}
		}

		$new_offset = $offset + $batch_size;
		$complete = $new_offset >= $total;

		error_log( sprintf( '[Buildfuly] Batch complete. Compressed: %d, Progress: %d%%', $compressed, round( ( $new_offset / $total ) * 100 ) ) );

		wp_send_json_success( array(
			'compressed' => $compressed,
			'total'      => $total,
			'offset'     => $new_offset,
			'complete'   => $complete,
			'progress'   => $total > 0 ? round( ( $new_offset / $total ) * 100 ) : 100,
			'quality'    => $quality,
			'force'      => $force
		) );
	}

	/**
	 * AJAX: Generate WebP for all images
	 */
	public function ajax_generate_webp_all() {
		check_ajax_referer( 'buildfuly_speed_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Permission denied' );
		}

		if ( ! function_exists( 'imagewebp' ) ) {
			wp_send_json_error( 'WebP support not available on this server' );
		}

		$offset = intval( $_POST['offset'] ?? 0 );
		$batch_size = 10;

		$args = array(
			'post_type'      => 'attachment',
			'post_mime_type' => array( 'image/jpeg', 'image/jpg', 'image/png' ),
			'post_status'    => 'inherit',
			'posts_per_page' => $batch_size,
			'offset'         => $offset,
			'orderby'        => 'ID',
			'order'          => 'ASC',
		);

		$attachments = get_posts( $args );
		$total_query = new WP_Query( array_merge( $args, array( 'posts_per_page' => -1, 'fields' => 'ids' ) ) );
		$total = $total_query->found_posts;

		$generated = 0;

		foreach ( $attachments as $attachment ) {
			$metadata = wp_get_attachment_metadata( $attachment->ID );
			$this->generate_webp_versions( $metadata, $attachment->ID );
			$generated++;
		}

		$new_offset = $offset + $batch_size;
		$complete = $new_offset >= $total;

		wp_send_json_success( array(
			'generated' => $generated,
			'total'     => $total,
			'offset'    => $new_offset,
			'complete'  => $complete,
			'progress'  => $total > 0 ? round( ( $new_offset / $total ) * 100 ) : 100,
		) );
	}

	/**
	 * AJAX: Clear all caches
	 */
	public function ajax_clear_cache() {
		check_ajax_referer( 'buildfuly_speed_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Permission denied' );
		}

		// Clear WordPress object cache
		wp_cache_flush();

		// Clear transients
		global $wpdb;
		$wpdb->query( "DELETE FROM $wpdb->options WHERE option_name LIKE '_transient_%'" );

		// Clear common cache plugins
		if ( function_exists( 'wp_cache_clear_cache' ) ) {
			wp_cache_clear_cache();
		}

		// W3 Total Cache
		if ( function_exists( 'w3tc_flush_all' ) ) {
			w3tc_flush_all();
		}

		// WP Super Cache
		if ( function_exists( 'wp_cache_clean_cache' ) ) {
			wp_cache_clean_cache( 'all' );
		}

		// WP Fastest Cache
		if ( class_exists( 'WpFastestCache' ) ) {
			$wpfc = new WpFastestCache();
			$wpfc->deleteCache();
		}

		wp_send_json_success( 'Cache cleared successfully' );
	}

	/**
	 * AJAX: Count backup files
	 */
	public function ajax_count_backups() {
		check_ajax_referer( 'buildfuly_speed_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Permission denied' );
		}

		$upload_dir = wp_get_upload_dir();
		$base_dir = $upload_dir['basedir'];

		// Find all .original files
		$backup_files = $this->find_backup_files( $base_dir );

		wp_send_json_success( array(
			'count' => count( $backup_files )
		) );
	}

	/**
	 * AJAX: Restore original images
	 */
	public function ajax_restore_originals() {
		check_ajax_referer( 'buildfuly_speed_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Permission denied' );
		}

		$offset = isset( $_POST['offset'] ) ? intval( $_POST['offset'] ) : 0;
		$batch_size = 10;

		$upload_dir = wp_get_upload_dir();
		$base_dir = $upload_dir['basedir'];

		// Find all .original files
		$backup_files = $this->find_backup_files( $base_dir );
		$total = count( $backup_files );

		if ( $total === 0 ) {
			wp_send_json_success( array(
				'complete' => true,
				'restored' => 0,
				'progress' => 100,
				'message' => 'No backup files found'
			) );
		}

		// Process batch
		$batch = array_slice( $backup_files, $offset, $batch_size );
		$restored = 0;

		foreach ( $batch as $backup_file ) {
			$original_file = str_replace( '.original', '', $backup_file );
			
			// Restore original (replace compressed with backup)
			if ( file_exists( $backup_file ) ) {
				copy( $backup_file, $original_file );
				// Delete backup file after restoration
				unlink( $backup_file );
				$restored++;
			}
		}

		$new_offset = $offset + $batch_size;
		$progress = min( 100, round( ( $new_offset / $total ) * 100 ) );
		$complete = $new_offset >= $total;

		wp_send_json_success( array(
			'complete' => $complete,
			'offset' => $new_offset,
			'progress' => $progress,
			'restored' => $restored,
			'total' => $total
		) );
	}

	/**
	 * Find all .original backup files recursively
	 */
	private function find_backup_files( $dir ) {
		$backup_files = array();
		
		if ( ! is_dir( $dir ) ) {
			return $backup_files;
		}

		$iterator = new RecursiveIteratorIterator(
			new RecursiveDirectoryIterator( $dir, RecursiveDirectoryIterator::SKIP_DOTS ),
			RecursiveIteratorIterator::SELF_FIRST
		);

		foreach ( $iterator as $file ) {
			if ( $file->isFile() && substr( $file->getFilename(), -9 ) === '.original' ) {
				$backup_files[] = $file->getPathname();
			}
		}

		return $backup_files;
	}
}

// Initialize
new Buildfuly_Speed_Optimizer();
