<?php
/**
 * SEO Metadata Generation Class
 *
 * @link       https://build.fuly.io
 * @since      1.0.4
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

class Buildfuly_SEO {

	/**
	 * Generate comprehensive SEO metadata for a page.
	 *
	 * @since    1.0.4
	 * @param    string    $primary_keyword    Main keyword/topic
	 * @param    string    $page_content       Generated page content (HTML)
	 * @return   array                         SEO metadata array
	 */
	public function generate_seo_metadata( $primary_keyword, $page_content = '' ) {
		$business_name = get_option( 'buildfuly_business_name', '' );
		$city = get_option( 'buildfuly_city', '' );
		$state = get_option( 'buildfuly_state', '' );

		$location = '';
		if ( ! empty( $city ) && ! empty( $state ) ) {
			$location = "{$city}, {$state}";
		}

		// DELETE: Removed broken AI call for SEO metadata generation
		// Using basic SEO metadata instead - works reliably without API calls
		$seo_data = $this->generate_basic_seo_metadata( $primary_keyword, $business_name, $location );

	// Add computed fields
	// Remove content between seo-ignore comments before counting words
	$content_for_seo = preg_replace('/<!--\s*seo-ignore-start\s*-->.*?<!--\s*seo-ignore-end\s*-->/s', '', $page_content);
	$seo_data['canonical_url'] = ''; // Will be set when page is created
	$seo_data['word_count'] = str_word_count( wp_strip_all_tags( $content_for_seo ) );
	$seo_data['keyword_density'] = $this->calculate_keyword_density( $primary_keyword, $content_for_seo );
	
	// Semantic keywords come from the page suggestion (already generated by API)
	// The primary_keyword passed in is typically the first keyword from a comma-separated list
	// Store as array for consistency with expected format
	$seo_data['semantic_keywords'] = array( $primary_keyword );

	Buildfuly_Logger::debug( 'BuildfulyAI: Generated SEO metadata - Title: ' . $seo_data['page_title'] );		return $seo_data;
	}

	/**
	 * Generate basic SEO metadata when AI is unavailable.
	 *
	 * @since    1.0.4
	 */
	private function generate_basic_seo_metadata( $keyword, $business_name, $location ) {
		// Clean and validate keyword
		$keyword = trim( $keyword );
		if ( empty( $keyword ) ) {
			$keyword = 'Home'; // Default fallback
		}
		
		$keyword_clean = ucwords( $keyword );
		$slug = sanitize_title( $keyword );

		// Build title with keyword first (don't add brand - WordPress/Yoast adds it automatically)
		$title = $keyword_clean;
		if ( ! empty( $location ) ) {
			$title .= " in {$location}";
		}

		$description = "Professional {$keyword} services";
		if ( ! empty( $location ) ) {
			$description .= " in {$location}";
		}
		$description .= ". Expert solutions with proven results. Contact us today!";

		return array(
			'page_title' => substr( $title, 0, 60 ),
			'meta_description' => substr( $description, 0, 155 ),
			'slug' => $slug,
			'focus_keyphrase' => $keyword,
			'og_title' => substr( $keyword_clean, 0, 60 ),
			'og_description' => substr( $description, 0, 200 ),
			'schema_type' => 'LocalBusiness',
			'h1_tag' => $keyword_clean,
			'breadcrumb_title' => $keyword_clean,
			'canonical_url' => '',
			'word_count' => 0,
			'keyword_density' => 0,
			'semantic_keywords' => array(),
		);
	}

	/**
	 * Calculate keyword density in content.
	 *
	 * @since    1.0.4
	 */
	private function calculate_keyword_density( $keyword, $content ) {
		// Return 0 if keyword is empty
		if ( empty( $keyword ) || empty( trim( $keyword ) ) ) {
			return 0;
		}

		$text = strtolower( wp_strip_all_tags( $content ) );
		$keyword_lower = strtolower( $keyword );

		$total_words = str_word_count( $text );
		if ( $total_words === 0 ) {
			return 0;
		}

		$keyword_count = substr_count( $text, $keyword_lower );
		
		// Also count semantic matches (partial words)
		$keyword_parts = explode( ' ', $keyword_lower );
		foreach ( $keyword_parts as $part ) {
			if ( strlen( $part ) > 3 ) { // Only count significant words
				$keyword_count += substr_count( $text, $part ) * 0.5; // Partial credit
			}
		}

		$density = ( $keyword_count / $total_words ) * 100;
		return round( $density, 2 );
	}

	/**
	 * Generate Schema.org structured data.
	 *
	 * @since    1.0.4
	 */
	public function generate_schema_markup( $page_id, $seo_metadata ) {
		$business_name = get_option( 'buildfuly_business_name', '' );
		$phone = get_option( 'buildfuly_phone', '' );
		$address = get_option( 'buildfuly_address', '' );
		$city = get_option( 'buildfuly_city', '' );
		$state = get_option( 'buildfuly_state', '' );
		$zip = get_option( 'buildfuly_zip', '' );
		
		$schema_type = isset( $seo_metadata['schema_type'] ) ? $seo_metadata['schema_type'] : 'LocalBusiness';
		
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => $schema_type,
			'name' => $business_name,
			'url' => get_permalink( $page_id ),
			'description' => isset( $seo_metadata['meta_description'] ) ? $seo_metadata['meta_description'] : '',
		);

		// Add address if available
		if ( ! empty( $address ) && ! empty( $city ) && ! empty( $state ) ) {
			$schema['address'] = array(
				'@type' => 'PostalAddress',
				'streetAddress' => $address,
				'addressLocality' => $city,
				'addressRegion' => $state,
				'postalCode' => $zip,
				'addressCountry' => 'US',
			);
		}

		// Add contact info
		if ( ! empty( $phone ) ) {
			$schema['telephone'] = $phone;
		}

		// Add aggregate rating (if you collect reviews)
		// $schema['aggregateRating'] = array(...);

		return $schema;
	}

	/**
	 * Apply SEO metadata to WordPress page.
	 *
	 * @since    1.0.4
	 */
	public function apply_seo_to_page( $page_id, $seo_metadata ) {
		// Update page slug
		if ( isset( $seo_metadata['slug'] ) ) {
			wp_update_post( array(
				'ID' => $page_id,
				'post_name' => $seo_metadata['slug'],
			) );
		}

		// Save as post meta
		update_post_meta( $page_id, '_buildfuly_seo_title', $seo_metadata['page_title'] );
		update_post_meta( $page_id, '_buildfuly_seo_description', $seo_metadata['meta_description'] );
		update_post_meta( $page_id, '_buildfuly_focus_keyphrase', $seo_metadata['focus_keyphrase'] );
		update_post_meta( $page_id, '_buildfuly_og_title', $seo_metadata['og_title'] );
		update_post_meta( $page_id, '_buildfuly_og_description', $seo_metadata['og_description'] );
		update_post_meta( $page_id, '_buildfuly_schema_type', $seo_metadata['schema_type'] );
		update_post_meta( $page_id, '_buildfuly_h1_tag', $seo_metadata['h1_tag'] );
		update_post_meta( $page_id, '_buildfuly_word_count', $seo_metadata['word_count'] );
		update_post_meta( $page_id, '_buildfuly_keyword_density', $seo_metadata['keyword_density'] );
		
		if ( isset( $seo_metadata['semantic_keywords'] ) ) {
			update_post_meta( $page_id, '_buildfuly_semantic_keywords', $seo_metadata['semantic_keywords'] );
		}

		// Generate and save schema markup
		$schema = $this->generate_schema_markup( $page_id, $seo_metadata );
		update_post_meta( $page_id, '_buildfuly_schema_markup', wp_json_encode( $schema ) );

		Buildfuly_Logger::debug( 'BuildfulyAI: Applied SEO to page ' . $page_id . ' - Density: ' . $seo_metadata['keyword_density'] . '%' );
	}

	/**
	 * Output SEO meta tags in page head.
	 *
	 * @since    1.0.4
	 */
	public function output_seo_meta_tags() {
		if ( ! is_page() ) {
			return;
		}

		$page_id = get_the_ID();
		
		$seo_title = get_post_meta( $page_id, '_buildfuly_seo_title', true );
		$seo_description = get_post_meta( $page_id, '_buildfuly_seo_description', true );
		$og_title = get_post_meta( $page_id, '_buildfuly_og_title', true );
		$og_description = get_post_meta( $page_id, '_buildfuly_og_description', true );
		$schema_markup = get_post_meta( $page_id, '_buildfuly_schema_markup', true );

		// Note: Title is now handled by filter_document_title() and filter_document_title_parts()
		// Output OG title meta tag
		if ( $seo_title ) {
			echo '<meta property="og:title" content="' . esc_attr( $og_title ?: $seo_title ) . '">' . "\n";
		}

		if ( $seo_description ) {
			echo '<meta name="description" content="' . esc_attr( $seo_description ) . '">' . "\n";
			echo '<meta property="og:description" content="' . esc_attr( $og_description ?: $seo_description ) . '">' . "\n";
		}

		echo '<meta property="og:url" content="' . esc_url( get_permalink() ) . '">' . "\n";
		echo '<meta property="og:type" content="website">' . "\n";
		echo '<link rel="canonical" href="' . esc_url( get_permalink() ) . '">' . "\n";

		// Output Schema.org markup
		if ( $schema_markup ) {
			echo '<script type="application/ld+json">' . "\n";
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Schema markup is JSON-LD, must not be escaped
			echo $schema_markup . "\n";
			echo '</script>' . "\n";
		}
	}

	/**
	 * Filter the document title for Buildfuly pages.
	 * This replaces WordPress's default title entirely.
	 *
	 * @since    1.1.2
	 * @param    string    $title    The document title.
	 * @return   string              Modified title or original.
	 */
	public function filter_document_title( $title ) {
		if ( ! is_page() ) {
			return $title;
		}

		$page_id = get_the_ID();
		$seo_title = get_post_meta( $page_id, '_buildfuly_seo_title', true );
		
		if ( $seo_title ) {
			// Add site name to SEO title
			$site_name = get_bloginfo( 'name' );
			return $seo_title . ' | ' . $site_name;
		}

		return $title;
	}

	/**
	 * Filter document title parts for Buildfuly pages.
	 * This modifies the title and tagline parts.
	 *
	 * @since    1.1.2
	 * @param    array    $title_parts    The document title parts.
	 * @return   array                    Modified title parts.
	 */
	public function filter_document_title_parts( $title_parts ) {
		if ( ! is_page() ) {
			return $title_parts;
		}

		$page_id = get_the_ID();
		$seo_title = get_post_meta( $page_id, '_buildfuly_seo_title', true );
		
		if ( $seo_title ) {
			$title_parts['title'] = $seo_title;
			// Remove tagline for cleaner title
			if ( isset( $title_parts['tagline'] ) ) {
				unset( $title_parts['tagline'] );
			}
		}

		return $title_parts;
	}

	/**
	 * Output global LocalBusiness schema from SEO settings.
	 * This runs on all pages to provide consistent business information.
	 *
	 * @since    1.0.6
	 */
	public static function output_local_business_schema() {
		$seo_settings = get_option( 'buildfuly_seo_settings', array() );
		
		// Check if schema is enabled
		if ( empty( $seo_settings['schema_enabled'] ) ) {
			return;
		}
		
		// Build the schema
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => ! empty( $seo_settings['schema_type'] ) ? $seo_settings['schema_type'] : 'LocalBusiness',
			'name' => ! empty( $seo_settings['business_name'] ) ? $seo_settings['business_name'] : get_bloginfo( 'name' ),
			'url' => home_url(),
		);
		
		// Add telephone
		if ( ! empty( $seo_settings['business_phone'] ) ) {
			$schema['telephone'] = $seo_settings['business_phone'];
		}
		
		// Add email
		if ( ! empty( $seo_settings['business_email'] ) ) {
			$schema['email'] = $seo_settings['business_email'];
		}
		
		// Add address
		if ( ! empty( $seo_settings['business_address'] ) || ! empty( $seo_settings['business_city'] ) ) {
			$schema['address'] = array(
				'@type' => 'PostalAddress',
				'streetAddress' => $seo_settings['business_address'] ?? '',
				'addressLocality' => $seo_settings['business_city'] ?? '',
				'addressRegion' => $seo_settings['business_state'] ?? '',
				'postalCode' => $seo_settings['business_zip'] ?? '',
				'addressCountry' => $seo_settings['business_country'] ?? 'US',
			);
		}
		
		// Add geo coordinates
		if ( ! empty( $seo_settings['business_latitude'] ) && ! empty( $seo_settings['business_longitude'] ) ) {
			$schema['geo'] = array(
				'@type' => 'GeoCoordinates',
				'latitude' => floatval( $seo_settings['business_latitude'] ),
				'longitude' => floatval( $seo_settings['business_longitude'] ),
			);
		}
		
		// Add logo
		if ( ! empty( $seo_settings['business_logo'] ) ) {
			$schema['logo'] = $seo_settings['business_logo'];
		}
		
		// Add image (default OG image)
		if ( ! empty( $seo_settings['default_og_image'] ) ) {
			$schema['image'] = $seo_settings['default_og_image'];
		}
		
		// Add price range
		if ( ! empty( $seo_settings['business_price_range'] ) ) {
			$schema['priceRange'] = $seo_settings['business_price_range'];
		}
		
		// Add opening hours
		if ( ! empty( $seo_settings['business_hours'] ) ) {
			$hours_lines = explode( "\n", $seo_settings['business_hours'] );
			$opening_hours = array();
			foreach ( $hours_lines as $line ) {
				$line = trim( $line );
				if ( ! empty( $line ) ) {
					$opening_hours[] = $line;
				}
			}
			if ( ! empty( $opening_hours ) ) {
				$schema['openingHours'] = $opening_hours;
			}
		}
		
		// Add social profiles
		$same_as = array();
		$social_fields = array( 'social_facebook', 'social_instagram', 'social_twitter', 'social_linkedin', 'social_youtube', 'social_yelp', 'social_google_business' );
		foreach ( $social_fields as $field ) {
			if ( ! empty( $seo_settings[ $field ] ) ) {
				$same_as[] = $seo_settings[ $field ];
			}
		}
		if ( ! empty( $same_as ) ) {
			$schema['sameAs'] = $same_as;
		}
		
		// Add service area if set
		if ( ! empty( $seo_settings['service_area'] ) ) {
			$areas = array_map( 'trim', explode( "\n", $seo_settings['service_area'] ) );
			$area_served = array();
			foreach ( $areas as $area ) {
				if ( ! empty( $area ) ) {
					$area_served[] = array(
						'@type' => 'City',
						'name' => $area,
					);
				}
			}
			if ( ! empty( $area_served ) ) {
				$schema['areaServed'] = $area_served;
			}
		}
		
		// Output the schema
		echo '<script type="application/ld+json">' . "\n";
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Schema markup is JSON-LD
		echo wp_json_encode( $schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
		echo "\n</script>\n";
	}

	/**
	 * Output Google Maps in footer if enabled.
	 *
	 * @since    1.0.6
	 */
	public static function output_footer_map() {
		$seo_settings = get_option( 'buildfuly_seo_settings', array() );
		
		// Check if maps footer is enabled
		if ( empty( $seo_settings['google_maps_enabled'] ) || empty( $seo_settings['google_maps_footer'] ) ) {
			return;
		}
		
		if ( empty( $seo_settings['google_maps_embed_url'] ) ) {
			return;
		}
		
		$embed_url = $seo_settings['google_maps_embed_url'];
		
		// Extract src from iframe if needed
		if ( strpos( $embed_url, '<iframe' ) !== false ) {
			preg_match( '/src=["\']([^"\']+)["\']/', $embed_url, $matches );
			$embed_url = $matches[1] ?? '';
		}
		
		if ( empty( $embed_url ) ) {
			return;
		}
		
		$height = ! empty( $seo_settings['google_maps_height'] ) ? intval( $seo_settings['google_maps_height'] ) : 300;
		$business_name = $seo_settings['business_name'] ?? get_bloginfo( 'name' );
		?>
		<section class="buildfuly-footer-map" style="width: 100%; margin: 0; padding: 0;">
			<div class="buildfuly-map-container" style="position: relative; width: 100%;">
				<?php if ( ! empty( $business_name ) ) : ?>
				<div class="buildfuly-map-overlay" style="position: absolute; top: 20px; left: 20px; background: white; padding: 15px 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.15); z-index: 10; max-width: 300px;">
					<h3 style="margin: 0 0 5px; font-size: 16px; color: #1f2937;"><?php echo esc_html( $business_name ); ?></h3>
					<?php if ( ! empty( $seo_settings['business_address'] ) ) : ?>
					<p style="margin: 0; font-size: 14px; color: #6b7280;">
						<?php echo esc_html( $seo_settings['business_address'] ); ?><br>
						<?php echo esc_html( ( $seo_settings['business_city'] ?? '' ) . ', ' . ( $seo_settings['business_state'] ?? '' ) . ' ' . ( $seo_settings['business_zip'] ?? '' ) ); ?>
					</p>
					<?php endif; ?>
					<?php if ( ! empty( $seo_settings['business_phone'] ) ) : ?>
					<p style="margin: 8px 0 0; font-size: 14px;">
						<a href="tel:<?php echo esc_attr( preg_replace( '/[^0-9+]/', '', $seo_settings['business_phone'] ) ); ?>" style="color: #3b82f6; text-decoration: none;">
							<?php echo esc_html( $seo_settings['business_phone'] ); ?>
						</a>
					</p>
					<?php endif; ?>
				</div>
				<?php endif; ?>
				<iframe 
					src="<?php echo esc_url( $embed_url ); ?>" 
					width="100%" 
					height="<?php echo esc_attr( $height ); ?>" 
					style="border:0; display: block;" 
					allowfullscreen="" 
					loading="lazy" 
					referrerpolicy="no-referrer-when-downgrade"
					title="<?php echo esc_attr( $business_name . ' Location Map' ); ?>"
				></iframe>
			</div>
		</section>
		<?php
	}

	/**
	 * Shortcode to embed Google Map anywhere.
	 *
	 * @since    1.0.6
	 * @param    array    $atts    Shortcode attributes
	 * @return   string            Map HTML
	 */
	public static function map_shortcode( $atts ) {
		$atts = shortcode_atts( array(
			'height' => '',
		), $atts, 'buildfuly_map' );
		
		$seo_settings = get_option( 'buildfuly_seo_settings', array() );
		
		if ( empty( $seo_settings['google_maps_enabled'] ) || empty( $seo_settings['google_maps_embed_url'] ) ) {
			return '';
		}
		
		$embed_url = $seo_settings['google_maps_embed_url'];
		
		// Extract src from iframe if needed
		if ( strpos( $embed_url, '<iframe' ) !== false ) {
			preg_match( '/src=["\']([^"\']+)["\']/', $embed_url, $matches );
			$embed_url = $matches[1] ?? '';
		}
		
		if ( empty( $embed_url ) ) {
			return '';
		}
		
		$height = ! empty( $atts['height'] ) ? intval( $atts['height'] ) : ( ! empty( $seo_settings['google_maps_height'] ) ? intval( $seo_settings['google_maps_height'] ) : 300 );
		$business_name = $seo_settings['business_name'] ?? get_bloginfo( 'name' );
		
		return '<div class="buildfuly-map-embed" style="width: 100%; border-radius: 8px; overflow: hidden;">
			<iframe 
				src="' . esc_url( $embed_url ) . '" 
				width="100%" 
				height="' . esc_attr( $height ) . '" 
				style="border:0; display: block;" 
				allowfullscreen="" 
				loading="lazy" 
				referrerpolicy="no-referrer-when-downgrade"
				title="' . esc_attr( $business_name . ' Location Map' ) . '"
			></iframe>
		</div>';
	}

	/**
	 * Generate BreadcrumbList schema for a page.
	 *
	 * @since    1.1.2
	 * @param    int    $page_id    WordPress page ID
	 * @return   array              BreadcrumbList schema array
	 */
	public static function generate_breadcrumb_schema( $page_id ) {
		$breadcrumbs = array();
		$position = 1;
		
		// Home
		$breadcrumbs[] = array(
			'@type' => 'ListItem',
			'position' => $position++,
			'name' => 'Home',
			'item' => home_url( '/' ),
		);
		
		// Get page ancestors
		$ancestors = get_post_ancestors( $page_id );
		$ancestors = array_reverse( $ancestors );
		
		foreach ( $ancestors as $ancestor_id ) {
			$breadcrumbs[] = array(
				'@type' => 'ListItem',
				'position' => $position++,
				'name' => get_the_title( $ancestor_id ),
				'item' => get_permalink( $ancestor_id ),
			);
		}
		
		// Current page
		$breadcrumbs[] = array(
			'@type' => 'ListItem',
			'position' => $position,
			'name' => get_the_title( $page_id ),
			'item' => get_permalink( $page_id ),
		);
		
		return array(
			'@context' => 'https://schema.org',
			'@type' => 'BreadcrumbList',
			'itemListElement' => $breadcrumbs,
		);
	}

	/**
	 * Generate FAQPage schema from FAQ content.
	 *
	 * @since    1.1.2
	 * @param    array    $faq_items    Array of FAQ items with 'question' and 'answer' keys
	 * @return   array                  FAQPage schema array
	 */
	public static function generate_faq_schema( $faq_items ) {
		if ( empty( $faq_items ) || ! is_array( $faq_items ) ) {
			return array();
		}
		
		$main_entity = array();
		
		foreach ( $faq_items as $item ) {
			if ( ! empty( $item['question'] ) && ! empty( $item['answer'] ) ) {
				$main_entity[] = array(
					'@type' => 'Question',
					'name' => wp_strip_all_tags( $item['question'] ),
					'acceptedAnswer' => array(
						'@type' => 'Answer',
						'text' => wp_strip_all_tags( $item['answer'] ),
					),
				);
			}
		}
		
		if ( empty( $main_entity ) ) {
			return array();
		}
		
		return array(
			'@context' => 'https://schema.org',
			'@type' => 'FAQPage',
			'mainEntity' => $main_entity,
		);
	}

	/**
	 * Generate Service schema for service pages.
	 *
	 * @since    1.1.2
	 * @param    int      $page_id         WordPress page ID
	 * @param    string   $service_name    Name of the service
	 * @param    string   $description     Service description
	 * @param    array    $options         Additional options (price, area_served, etc.)
	 * @return   array                     Service schema array
	 */
	public static function generate_service_schema( $page_id, $service_name, $description = '', $options = array() ) {
		$seo_settings = get_option( 'buildfuly_seo_settings', array() );
		$business_name = ! empty( $seo_settings['business_name'] ) ? $seo_settings['business_name'] : get_bloginfo( 'name' );
		
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => 'Service',
			'name' => $service_name,
			'description' => $description ?: get_post_meta( $page_id, '_buildfuly_seo_description', true ),
			'url' => get_permalink( $page_id ),
			'provider' => array(
				'@type' => 'LocalBusiness',
				'name' => $business_name,
				'url' => home_url(),
			),
		);
		
		// Add service area
		if ( ! empty( $seo_settings['service_area'] ) ) {
			$areas = array_map( 'trim', explode( ',', $seo_settings['service_area'] ) );
			$schema['areaServed'] = array();
			foreach ( $areas as $area ) {
				$schema['areaServed'][] = array(
					'@type' => 'City',
					'name' => $area,
				);
			}
		} elseif ( ! empty( $seo_settings['business_city'] ) ) {
			$schema['areaServed'] = array(
				'@type' => 'City',
				'name' => $seo_settings['business_city'],
			);
		}
		
		// Add price range if provided
		if ( ! empty( $options['price_range'] ) ) {
			$schema['priceRange'] = $options['price_range'];
		} elseif ( ! empty( $seo_settings['business_price_range'] ) ) {
			$schema['priceRange'] = $seo_settings['business_price_range'];
		}
		
		// Add image if available
		$thumbnail = get_the_post_thumbnail_url( $page_id, 'large' );
		if ( $thumbnail ) {
			$schema['image'] = $thumbnail;
		}
		
		return $schema;
	}

	/**
	 * Generate Article schema for blog/article pages.
	 *
	 * @since    1.1.2
	 * @param    int      $page_id    WordPress page/post ID
	 * @param    string   $type       Article type: 'Article', 'BlogPosting', 'NewsArticle'
	 * @return   array                Article schema array
	 */
	public static function generate_article_schema( $page_id, $type = 'Article' ) {
		$post = get_post( $page_id );
		if ( ! $post ) {
			return array();
		}
		
		$seo_settings = get_option( 'buildfuly_seo_settings', array() );
		$business_name = ! empty( $seo_settings['business_name'] ) ? $seo_settings['business_name'] : get_bloginfo( 'name' );
		$author_name = get_the_author_meta( 'display_name', $post->post_author ) ?: $business_name;
		
		$schema = array(
			'@context' => 'https://schema.org',
			'@type' => $type,
			'headline' => get_the_title( $page_id ),
			'description' => get_post_meta( $page_id, '_buildfuly_seo_description', true ) ?: wp_trim_words( $post->post_content, 30 ),
			'url' => get_permalink( $page_id ),
			'datePublished' => get_the_date( 'c', $page_id ),
			'dateModified' => get_the_modified_date( 'c', $page_id ),
			'author' => array(
				'@type' => 'Person',
				'name' => $author_name,
			),
			'publisher' => array(
				'@type' => 'Organization',
				'name' => $business_name,
				'url' => home_url(),
			),
		);
		
		// Add logo to publisher
		if ( ! empty( $seo_settings['business_logo'] ) ) {
			$schema['publisher']['logo'] = array(
				'@type' => 'ImageObject',
				'url' => $seo_settings['business_logo'],
			);
		}
		
		// Add main image
		$thumbnail = get_the_post_thumbnail_url( $page_id, 'large' );
		if ( $thumbnail ) {
			$schema['image'] = $thumbnail;
		}
		
		// Add word count
		$schema['wordCount'] = str_word_count( wp_strip_all_tags( $post->post_content ) );
		
		return $schema;
	}

	/**
	 * Output breadcrumb schema in page head.
	 * Called automatically for Buildfuly pages.
	 *
	 * @since    1.1.2
	 */
	public static function output_breadcrumb_schema() {
		if ( ! is_page() && ! is_single() ) {
			return;
		}
		
		$page_id = get_the_ID();
		
		// Check if this is a Buildfuly page
		if ( ! get_post_meta( $page_id, 'buildfuly_generated', true ) ) {
			return;
		}
		
		$schema = self::generate_breadcrumb_schema( $page_id );
		
		echo '<script type="application/ld+json">' . "\n";
		echo wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
		echo "\n</script>\n";
	}

	/**
	 * Extract FAQ items from page content and output FAQ schema.
	 *
	 * @since    1.1.2
	 */
	public static function output_faq_schema_from_content() {
		if ( ! is_page() ) {
			return;
		}
		
		$page_id = get_the_ID();
		
		// Check if this is a Buildfuly page
		if ( ! get_post_meta( $page_id, 'buildfuly_generated', true ) ) {
			return;
		}
		
		$content = get_post_field( 'post_content', $page_id );
		
		// Look for FAQ sections in content
		// Pattern matches common FAQ structures with questions (h3/h4/strong) and answers
		$faq_items = array();
		
		// Match accordion-style FAQs
		if ( preg_match_all( '/<(?:h[34]|strong)[^>]*>([^<]+)<\/(?:h[34]|strong)>\s*(?:<[^>]+>)*\s*<p>([^<]+)<\/p>/i', $content, $matches, PREG_SET_ORDER ) ) {
			foreach ( $matches as $match ) {
				$question = trim( $match[1] );
				$answer = trim( $match[2] );
				
				// Only include if it looks like a question
				if ( strpos( $question, '?' ) !== false || preg_match( '/^(what|how|why|when|where|who|is|can|do|does|will|should)/i', $question ) ) {
					$faq_items[] = array(
						'question' => $question,
						'answer' => $answer,
					);
				}
			}
		}
		
		// Also check for stored FAQ data
		$stored_faq = get_post_meta( $page_id, '_buildfuly_faq_items', true );
		if ( ! empty( $stored_faq ) && is_array( $stored_faq ) ) {
			$faq_items = array_merge( $faq_items, $stored_faq );
		}
		
		if ( empty( $faq_items ) ) {
			return;
		}
		
		$schema = self::generate_faq_schema( $faq_items );
		
		if ( ! empty( $schema ) ) {
			echo '<script type="application/ld+json">' . "\n";
			echo wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
			echo "\n</script>\n";
		}
	}

	/**
	 * Output service schema for service pages.
	 *
	 * @since    1.1.2
	 */
	public static function output_service_schema() {
		if ( ! is_page() ) {
			return;
		}
		
		$page_id = get_the_ID();
		
		// Check if this is a Buildfuly page
		if ( ! get_post_meta( $page_id, 'buildfuly_generated', true ) ) {
			return;
		}
		
		// Check if page is marked as a service page or has service-related keywords
		$seo_title = get_post_meta( $page_id, '_buildfuly_seo_title', true );
		$focus_keyword = get_post_meta( $page_id, '_buildfuly_focus_keyphrase', true );
		
		// Use the SEO title or focus keyword as the service name
		$service_name = $focus_keyword ?: $seo_title ?: get_the_title( $page_id );
		
		// Don't output service schema for homepage
		$homepage_id = get_option( 'page_on_front' );
		if ( $page_id == $homepage_id ) {
			return;
		}
		
		$schema = self::generate_service_schema( $page_id, $service_name );
		
		echo '<script type="application/ld+json">' . "\n";
		echo wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT );
		echo "\n</script>\n";
	}
}

