<?php
/**
 * SEO Functions - Sitemap, Robots.txt, Analytics, Schema
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @since      1.0.7
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * AJAX handler for saving SEO settings
 */
function buildfuly_ajax_save_seo_settings() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_seo_settings' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	// Parse the form data
	parse_str( $_POST['formData'] ?? '', $form_data );
	
	// Build settings array
	$settings = array(
		// Sitemap
		'sitemap_enabled' => ! empty( $form_data['sitemap_enabled'] ),
		'sitemap_posts' => ! empty( $form_data['sitemap_posts'] ),
		'sitemap_pages' => ! empty( $form_data['sitemap_pages'] ),
		'sitemap_images' => ! empty( $form_data['sitemap_images'] ),
		
		// Robots.txt
		'robots_txt' => sanitize_textarea_field( $form_data['robots_txt'] ?? '' ),
		
		// Analytics
		'google_analytics_id' => sanitize_text_field( $form_data['google_analytics_id'] ?? '' ),
		'google_tag_manager_id' => sanitize_text_field( $form_data['google_tag_manager_id'] ?? '' ),
		
		// Search Console
		'google_verification' => sanitize_text_field( $form_data['google_verification'] ?? '' ),
		'bing_verification' => sanitize_text_field( $form_data['bing_verification'] ?? '' ),
		
		// Schema/Local SEO
		'schema_type' => sanitize_text_field( $form_data['schema_type'] ?? 'LocalBusiness' ),
		'business_name' => sanitize_text_field( $form_data['business_name'] ?? '' ),
		'business_phone' => sanitize_text_field( $form_data['business_phone'] ?? '' ),
		'business_email' => sanitize_email( $form_data['business_email'] ?? '' ),
		'business_address' => sanitize_text_field( $form_data['business_address'] ?? '' ),
		'business_city' => sanitize_text_field( $form_data['business_city'] ?? '' ),
		'business_state' => sanitize_text_field( $form_data['business_state'] ?? '' ),
		'business_zip' => sanitize_text_field( $form_data['business_zip'] ?? '' ),
		'business_country' => sanitize_text_field( $form_data['business_country'] ?? 'US' ),
		'business_hours' => sanitize_textarea_field( $form_data['business_hours'] ?? '' ),
		'business_price_range' => sanitize_text_field( $form_data['business_price_range'] ?? '$$' ),
		'business_logo' => esc_url_raw( $form_data['business_logo'] ?? '' ),
		'service_area' => sanitize_textarea_field( $form_data['service_area'] ?? '' ),
		
		// Social Profiles
		'social_facebook' => esc_url_raw( $form_data['social_facebook'] ?? '' ),
		'social_twitter' => esc_url_raw( $form_data['social_twitter'] ?? '' ),
		'social_instagram' => esc_url_raw( $form_data['social_instagram'] ?? '' ),
		'social_linkedin' => esc_url_raw( $form_data['social_linkedin'] ?? '' ),
		'social_youtube' => esc_url_raw( $form_data['social_youtube'] ?? '' ),
		
		// Default Meta
		'default_og_image' => esc_url_raw( $form_data['default_og_image'] ?? '' ),
		'twitter_card_type' => sanitize_text_field( $form_data['twitter_card_type'] ?? 'summary_large_image' ),
		
		// Redirects
		'redirects' => array(),
	);
	
	// Process redirects
	if ( ! empty( $form_data['redirects'] ) && is_array( $form_data['redirects'] ) ) {
		foreach ( $form_data['redirects'] as $redirect ) {
			$from = sanitize_text_field( $redirect['from'] ?? '' );
			$to = sanitize_text_field( $redirect['to'] ?? '' );
			if ( ! empty( $from ) && ! empty( $to ) ) {
				$settings['redirects'][] = array( 'from' => $from, 'to' => $to );
			}
		}
	}
	
	// Also update business settings for backward compatibility
	update_option( 'buildfuly_business_name', $settings['business_name'] );
	update_option( 'buildfuly_phone', $settings['business_phone'] );
	update_option( 'buildfuly_address', $settings['business_address'] );
	update_option( 'buildfuly_city', $settings['business_city'] );
	update_option( 'buildfuly_state', $settings['business_state'] );
	update_option( 'buildfuly_zip', $settings['business_zip'] );
	
	update_option( 'buildfuly_seo_settings', $settings );
	
	// Sync legal pages (Contact, Privacy Policy, Terms of Service) with NAP info
	require_once plugin_dir_path( __FILE__ ) . 'class-buildfuly-legal-pages.php';
	Buildfuly_Legal_Pages::sync_legal_pages( $settings );
	
	// Flush rewrite rules if sitemap setting changed
	flush_rewrite_rules();
	
	wp_send_json_success();
}
add_action( 'wp_ajax_buildfuly_save_seo_settings', 'buildfuly_ajax_save_seo_settings' );

/**
 * Generate XML Sitemap
 */
function buildfuly_generate_sitemap() {
	$settings = get_option( 'buildfuly_seo_settings', array() );
	
	if ( empty( $settings['sitemap_enabled'] ) ) {
		return '';
	}
	
	$xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
	$xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"';
	if ( ! empty( $settings['sitemap_images'] ) ) {
		$xml .= ' xmlns:image="http://www.google.com/schemas/sitemap-image/1.1"';
	}
	$xml .= '>' . "\n";
	
	// Add homepage
	$xml .= '  <url>' . "\n";
	$xml .= '    <loc>' . esc_url( home_url( '/' ) ) . '</loc>' . "\n";
	$xml .= '    <changefreq>daily</changefreq>' . "\n";
	$xml .= '    <priority>1.0</priority>' . "\n";
	$xml .= '  </url>' . "\n";
	
	// Add pages
	if ( ! empty( $settings['sitemap_pages'] ) ) {
		$pages = get_posts( array(
			'post_type' => 'page',
			'post_status' => 'publish',
			'posts_per_page' => -1,
			'orderby' => 'modified',
			'order' => 'DESC',
		) );
		
		foreach ( $pages as $page ) {
			$xml .= '  <url>' . "\n";
			$xml .= '    <loc>' . esc_url( get_permalink( $page->ID ) ) . '</loc>' . "\n";
			$xml .= '    <lastmod>' . get_the_modified_date( 'c', $page->ID ) . '</lastmod>' . "\n";
			$xml .= '    <changefreq>weekly</changefreq>' . "\n";
			$xml .= '    <priority>0.8</priority>' . "\n";
			
			// Add images
			if ( ! empty( $settings['sitemap_images'] ) ) {
				$thumbnail = get_the_post_thumbnail_url( $page->ID, 'full' );
				if ( $thumbnail ) {
					$xml .= '    <image:image>' . "\n";
					$xml .= '      <image:loc>' . esc_url( $thumbnail ) . '</image:loc>' . "\n";
					$xml .= '    </image:image>' . "\n";
				}
			}
			
			$xml .= '  </url>' . "\n";
		}
	}
	
	// Add posts
	if ( ! empty( $settings['sitemap_posts'] ) ) {
		$posts = get_posts( array(
			'post_type' => 'post',
			'post_status' => 'publish',
			'posts_per_page' => -1,
			'orderby' => 'modified',
			'order' => 'DESC',
		) );
		
		foreach ( $posts as $post ) {
			$xml .= '  <url>' . "\n";
			$xml .= '    <loc>' . esc_url( get_permalink( $post->ID ) ) . '</loc>' . "\n";
			$xml .= '    <lastmod>' . get_the_modified_date( 'c', $post->ID ) . '</lastmod>' . "\n";
			$xml .= '    <changefreq>monthly</changefreq>' . "\n";
			$xml .= '    <priority>0.6</priority>' . "\n";
			$xml .= '  </url>' . "\n";
		}
	}
	
	$xml .= '</urlset>';
	
	return $xml;
}

/**
 * Handle sitemap.xml request
 */
function buildfuly_sitemap_request() {
	global $wp;
	
	if ( isset( $wp->query_vars['buildfuly_sitemap'] ) ) {
		header( 'Content-Type: application/xml; charset=utf-8' );
		echo buildfuly_generate_sitemap();
		exit;
	}
}
add_action( 'template_redirect', 'buildfuly_sitemap_request' );

/**
 * Add sitemap rewrite rule
 */
function buildfuly_sitemap_rewrite() {
	add_rewrite_rule( '^sitemap\.xml$', 'index.php?buildfuly_sitemap=1', 'top' );
}
add_action( 'init', 'buildfuly_sitemap_rewrite' );

/**
 * Add sitemap query var
 */
function buildfuly_sitemap_query_vars( $vars ) {
	$vars[] = 'buildfuly_sitemap';
	$vars[] = 'buildfuly_robots';
	return $vars;
}
add_filter( 'query_vars', 'buildfuly_sitemap_query_vars' );

/**
 * Handle robots.txt - override WordPress default
 */
function buildfuly_robots_txt( $output, $public ) {
	$settings = get_option( 'buildfuly_seo_settings', array() );
	
	if ( ! empty( $settings['robots_txt'] ) ) {
		return $settings['robots_txt'];
	}
	
	return $output;
}
add_filter( 'robots_txt', 'buildfuly_robots_txt', 10, 2 );

/**
 * Add verification meta tags to head
 */
function buildfuly_verification_tags() {
	$settings = get_option( 'buildfuly_seo_settings', array() );
	
	// Google Search Console
	if ( ! empty( $settings['google_verification'] ) ) {
		echo '<meta name="google-site-verification" content="' . esc_attr( $settings['google_verification'] ) . '">' . "\n";
	}
	
	// Bing
	if ( ! empty( $settings['bing_verification'] ) ) {
		echo '<meta name="msvalidate.01" content="' . esc_attr( $settings['bing_verification'] ) . '">' . "\n";
	}
}
add_action( 'wp_head', 'buildfuly_verification_tags', 1 );

/**
 * Add Google Analytics / Tag Manager code
 */
function buildfuly_analytics_code() {
	$settings = get_option( 'buildfuly_seo_settings', array() );
	
	// Google Tag Manager (in head)
	if ( ! empty( $settings['google_tag_manager_id'] ) ) {
		$gtm_id = esc_attr( $settings['google_tag_manager_id'] );
		echo "<!-- Google Tag Manager -->
<script>(function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':
new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],
j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=
'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);
})(window,document,'script','dataLayer','{$gtm_id}');</script>
<!-- End Google Tag Manager -->\n";
	}
	// Google Analytics (GA4)
	elseif ( ! empty( $settings['google_analytics_id'] ) ) {
		$ga_id = esc_attr( $settings['google_analytics_id'] );
		echo "<!-- Google Analytics -->
<script async src=\"https://www.googletagmanager.com/gtag/js?id={$ga_id}\"></script>
<script>
  window.dataLayer = window.dataLayer || [];
  function gtag(){dataLayer.push(arguments);}
  gtag('js', new Date());
  gtag('config', '{$ga_id}');
</script>
<!-- End Google Analytics -->\n";
	}
}
add_action( 'wp_head', 'buildfuly_analytics_code', 1 );

/**
 * Add GTM noscript fallback after body tag
 */
function buildfuly_gtm_body() {
	$settings = get_option( 'buildfuly_seo_settings', array() );
	
	if ( ! empty( $settings['google_tag_manager_id'] ) ) {
		$gtm_id = esc_attr( $settings['google_tag_manager_id'] );
		echo "<!-- Google Tag Manager (noscript) -->
<noscript><iframe src=\"https://www.googletagmanager.com/ns.html?id={$gtm_id}\"
height=\"0\" width=\"0\" style=\"display:none;visibility:hidden\"></iframe></noscript>
<!-- End Google Tag Manager (noscript) -->\n";
	}
}
add_action( 'wp_body_open', 'buildfuly_gtm_body' );

/**
 * Generate Organization/LocalBusiness schema for site
 */
function buildfuly_output_organization_schema() {
	if ( ! is_front_page() && ! is_home() ) {
		return;
	}
	
	$settings = get_option( 'buildfuly_seo_settings', array() );
	
	if ( empty( $settings['business_name'] ) ) {
		return;
	}
	
	$schema = array(
		'@context' => 'https://schema.org',
		'@type' => $settings['schema_type'] ?? 'LocalBusiness',
		'name' => $settings['business_name'],
		'url' => home_url( '/' ),
	);
	
	// Add contact info
	if ( ! empty( $settings['business_phone'] ) ) {
		$schema['telephone'] = $settings['business_phone'];
	}
	if ( ! empty( $settings['business_email'] ) ) {
		$schema['email'] = $settings['business_email'];
	}
	
	// Add address
	if ( ! empty( $settings['business_address'] ) || ! empty( $settings['business_city'] ) ) {
		$schema['address'] = array(
			'@type' => 'PostalAddress',
		);
		if ( ! empty( $settings['business_address'] ) ) {
			$schema['address']['streetAddress'] = $settings['business_address'];
		}
		if ( ! empty( $settings['business_city'] ) ) {
			$schema['address']['addressLocality'] = $settings['business_city'];
		}
		if ( ! empty( $settings['business_state'] ) ) {
			$schema['address']['addressRegion'] = $settings['business_state'];
		}
		if ( ! empty( $settings['business_zip'] ) ) {
			$schema['address']['postalCode'] = $settings['business_zip'];
		}
		$schema['address']['addressCountry'] = $settings['business_country'] ?? 'US';
	}
	
	// Add price range
	if ( ! empty( $settings['business_price_range'] ) ) {
		$schema['priceRange'] = $settings['business_price_range'];
	}
	
	// Add social profiles
	$social_profiles = array();
	$social_keys = array( 'social_facebook', 'social_twitter', 'social_instagram', 'social_linkedin', 'social_youtube' );
	foreach ( $social_keys as $key ) {
		if ( ! empty( $settings[ $key ] ) ) {
			$social_profiles[] = $settings[ $key ];
		}
	}
	if ( ! empty( $social_profiles ) ) {
		$schema['sameAs'] = $social_profiles;
	}
	
	// Add opening hours
	if ( ! empty( $settings['business_hours'] ) ) {
		$hours = array_filter( array_map( 'trim', explode( "\n", $settings['business_hours'] ) ) );
		if ( ! empty( $hours ) ) {
			$schema['openingHours'] = $hours;
		}
	}
	
	// Add default image
	if ( ! empty( $settings['default_og_image'] ) ) {
		$schema['image'] = $settings['default_og_image'];
	}
	
	echo '<script type="application/ld+json">' . "\n";
	echo wp_json_encode( $schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
	echo "\n</script>\n";
}
add_action( 'wp_head', 'buildfuly_output_organization_schema', 5 );

/**
 * Add default OG image if page doesn't have one
 */
function buildfuly_default_og_image() {
	$settings = get_option( 'buildfuly_seo_settings', array() );
	
	if ( ! empty( $settings['default_og_image'] ) ) {
		// Only output if the page doesn't already have an OG image
		$page_id = get_the_ID();
		$page_og_image = get_post_meta( $page_id, '_buildfuly_og_image', true );
		
		if ( empty( $page_og_image ) ) {
			echo '<meta property="og:image" content="' . esc_url( $settings['default_og_image'] ) . '">' . "\n";
		}
	}
	
	// Twitter card type
	if ( ! empty( $settings['twitter_card_type'] ) ) {
		echo '<meta name="twitter:card" content="' . esc_attr( $settings['twitter_card_type'] ) . '">' . "\n";
	}
}
add_action( 'wp_head', 'buildfuly_default_og_image', 10 );

/**
 * Handle 301 redirects
 */
function buildfuly_handle_redirects() {
	if ( is_admin() ) {
		return;
	}
	
	$settings = get_option( 'buildfuly_seo_settings', array() );
	
	if ( empty( $settings['redirects'] ) ) {
		return;
	}
	
	$current_path = $_SERVER['REQUEST_URI'] ?? '';
	$current_path = parse_url( $current_path, PHP_URL_PATH );
	
	foreach ( $settings['redirects'] as $redirect ) {
		$from = $redirect['from'] ?? '';
		$to = $redirect['to'] ?? '';
		
		if ( empty( $from ) || empty( $to ) ) {
			continue;
		}
		
		// Normalize paths
		$from = '/' . ltrim( $from, '/' );
		
		if ( $current_path === $from || $current_path === rtrim( $from, '/' ) ) {
			// If 'to' is relative, make it absolute
			if ( strpos( $to, 'http' ) !== 0 ) {
				$to = home_url( $to );
			}
			
			wp_redirect( $to, 301 );
			exit;
		}
	}
}
add_action( 'template_redirect', 'buildfuly_handle_redirects', 1 );

/**
 * Ping search engines when sitemap is updated
 */
function buildfuly_ping_search_engines() {
	$settings = get_option( 'buildfuly_seo_settings', array() );
	
	if ( empty( $settings['sitemap_enabled'] ) ) {
		return;
	}
	
	$sitemap_url = home_url( '/sitemap.xml' );
	
	// Ping Google
	wp_remote_get( 'https://www.google.com/ping?sitemap=' . urlencode( $sitemap_url ), array( 'blocking' => false ) );
	
	// Ping Bing
	wp_remote_get( 'https://www.bing.com/ping?sitemap=' . urlencode( $sitemap_url ), array( 'blocking' => false ) );
}
// Ping when a post is published
add_action( 'publish_post', 'buildfuly_ping_search_engines' );
add_action( 'publish_page', 'buildfuly_ping_search_engines' );
