<?php
/**
 * Pricing Configuration - Single Source of Truth
 * 
 * All pricing is defined here. Update these constants to change prices everywhere.
 * Run sync-stripe-prices.js after changing to update Stripe.
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

class Buildfuly_Pricing {

	// ============================================================
	// AI COSTS (for margin calculations - commented for reference)
	// ============================================================
	// Using GPT-4o-mini pricing:
	// - Input: $0.150 per 1M tokens
	// - Output: $0.600 per 1M tokens
	// Average page generation:
	// - ~2,000 input tokens = $0.0003
	// - ~1,500 output tokens = $0.0009
	// Total: ~$0.0012 per page
	//
	// Monthly costs by tier:
	// - Starter (1/day × 30): 30 pages = $0.036/month
	// - Pro (5/day × 30): 150 pages = $0.18/month
	// - Growth (10/day × 30): 300 pages = $0.36/month
	// - Studio (50/day × 30): 1,500 pages = $1.80/month
	const COST_PER_PAGE = 0.0012;
	
	// ============================================================
	// HOSTING COSTS (for margin calculations - commented for reference)
	// ============================================================
	// Actual server cost: ~$5-10/month per site
	// We charge: $20/month (sold separately from plugin license)
	// This covers: WordPress hosting, CDN, SSL, backups, support
	const HOSTING_COST_ACTUAL = 10; // Our actual cost
	const HOSTING_PRICE_TO_CUSTOMER = 20; // What we charge

	/**
	 * Pricing Plans
	 * 
	 * NOTE: Hosting ($20/month) is sold separately through a different funnel.
	 * These prices are ONLY for the plugin license/tier access.
	 * 
	 * Format: tier => [ monthly, yearly, lifetime, pages limit, sites_limit ]
	 * Starter uses total_pages (lifetime), paid tiers use pages_per_day
	 * Lifetime = 2 years of yearly price
	 */
	const PLANS = array(
		'starter' => array(
			'name'          => 'Starter',
			'monthly'       => 0,      // Free tier
			'yearly'        => 0,
			'lifetime'      => 0,
			'total_pages'   => 3,
			'sites_limit'   => 1,
			'features'      => array(
				'3 pages total',
				'1 website',
				'AI-powered generation',
			),
		),
		'pro' => array(
			'name'          => 'Pro',
			'monthly'       => 19,     // Tier fee only (hosting separate)
			'yearly'        => 182,    // ~$15.16/month (20% discount)
			'lifetime'      => 364,    // 2 years of yearly
			'pages_per_day' => 5,
			'sites_limit'   => 1,
			'features'      => array(
				'5 pages per day',
				'1 website',
				'Email support',
			),
		),
		'growth' => array(
			'name'          => 'Growth',
			'monthly'       => 39,     // Tier fee only
			'yearly'        => 374,    // ~$31.16/month (20% discount)
			'lifetime'      => 748,    // 2 years of yearly
			'pages_per_day' => 10,
			'sites_limit'   => 3,
			'features'      => array(
				'10 pages per day',
				'3 websites',
				'Priority support',
			),
		),
		'studio' => array(
			'name'          => 'Studio',
			'monthly'       => 79,     // Tier fee only
			'yearly'        => 758,    // ~$63.16/month (20% discount)
			'lifetime'      => 1516,   // 2 years of yearly
			'pages_per_day' => 50,
			'sites_limit'   => -1,     // Unlimited
			'features'      => array(
				'50 pages per day',
				'Unlimited websites',
				'Premium support',
			),
		),
	);

	/**
	 * Get plan details
	 *
	 * @param string $tier Plan tier slug
	 * @return array|null Plan details or null if not found
	 */
	public static function get_plan( $tier ) {
		return isset( self::PLANS[ $tier ] ) ? self::PLANS[ $tier ] : null;
	}

	/**
	 * Get all plans
	 *
	 * @return array All plans
	 */
	public static function get_all_plans() {
		return self::PLANS;
	}

	/**
	 * Calculate profit margin for a tier
	 *
	 * @param string $tier Plan tier
	 * @param string $billing monthly|yearly
	 * @return array [ revenue, hosting_cost, api_cost, profit, margin_percent ]
	 */
	public static function calculate_margin( $tier, $billing = 'monthly' ) {
		$plan = self::get_plan( $tier );
		if ( ! $plan ) {
			return null;
		}

		$revenue = $plan[ $billing ];
		$hosting_cost = 0; // Hosting is sold separately, not included in tier revenue
		
		// API cost calculation:
		// - Starter: total_pages × cost_per_page (one-time)
		// - Paid tiers: pages_per_day × 30 days × cost_per_page
		if ( isset( $plan['total_pages'] ) ) {
			// Starter tier - one-time cost for total pages
			$api_cost = $plan['total_pages'] * self::COST_PER_PAGE;
		} else {
			// Paid tiers - monthly cost based on daily limit
			$api_cost = $plan['pages_per_day'] * 30 * self::COST_PER_PAGE;
		}
		
		// For yearly, divide by 12 for monthly average
		if ( $billing === 'yearly' ) {
			$revenue = $revenue / 12;
		}
		
		$profit = $revenue - $hosting_cost - $api_cost;
		$margin_percent = $revenue > 0 ? ( $profit / $revenue ) * 100 : 0;

		return array(
			'revenue'        => round( $revenue, 2 ),
			'hosting_cost'   => $hosting_cost, // 0 because sold separately
			'api_cost'       => round( $api_cost, 2 ),
			'profit'         => round( $profit, 2 ),
			'margin_percent' => round( $margin_percent, 1 ),
		);
	}

	/**
	 * Get yearly discount percentage
	 *
	 * @param string $tier Plan tier
	 * @return float Discount percentage
	 */
	public static function get_yearly_discount( $tier ) {
		$plan = self::get_plan( $tier );
		if ( ! $plan || $plan['monthly'] === 0 ) {
			return 0;
		}

		$monthly_annual_cost = $plan['monthly'] * 12;
		$yearly_cost = $plan['yearly'];
		$savings = $monthly_annual_cost - $yearly_cost;
		$discount_percent = ( $savings / $monthly_annual_cost ) * 100;

		return round( $discount_percent );
	}

	/**
	 * Format price for display
	 *
	 * @param float $price Price amount
	 * @param string $period Billing period (monthly, yearly, lifetime)
	 * @return string Formatted price string
	 */
	public static function format_price( $price, $period = 'monthly' ) {
		if ( $price === 0 ) {
			return 'Free';
		}

		$formatted = '$' . number_format( $price, 0 );
		
		switch ( $period ) {
			case 'monthly':
				return $formatted . '/mo';
			case 'yearly':
				return $formatted . '/yr';
			case 'lifetime':
				return $formatted . ' lifetime';
			default:
				return $formatted;
		}
	}

	/**
	 * Get Stripe price configuration for sync script
	 * Used by sync-stripe-prices.js
	 *
	 * @return array Stripe price config
	 */
	public static function get_stripe_config() {
		$config = array();

		foreach ( self::PLANS as $tier => $plan ) {
			if ( $plan['monthly'] > 0 ) {
				$config[ $tier ] = array(
					'name'     => $plan['name'],
					'monthly'  => $plan['monthly'] * 100, // Convert to cents
					'yearly'   => $plan['yearly'] * 100,
					'lifetime' => $plan['lifetime'] * 100,
				);
			}
		}

		return $config;
	}
}
