<?php
/**
 * Navigation and Footer Handler
 *
 * Manages auto-generated navigation menus and footer with NAP info
 * Optionally takes over full page layout (hides theme header/footer)
 *
 * @link       https://build.fuly.io
 * @since      1.1.0
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

class Buildfuly_Navigation {

	/**
	 * Page types that should appear in Services dropdown
	 */
	const SERVICE_PAGE_TYPES = array( 'service' );

	/**
	 * Initialize hooks
	 */
	public function __construct() {
		// Register nav menu locations
		add_action( 'after_setup_theme', array( $this, 'register_nav_menus' ) );
		
		// Check if we should take over the page layout
		add_action( 'template_redirect', array( $this, 'maybe_takeover_layout' ) );
		
		// Auto-generate menus
		add_filter( 'wp_nav_menu_args', array( $this, 'maybe_use_buildfuly_menu' ) );
	}

	/**
	 * Check if we should take over the page layout
	 */
	public function maybe_takeover_layout() {
		// Only on frontend, single pages
		if ( is_admin() || ! is_singular( 'page' ) ) {
			return;
		}

		// Check if this is a Buildfuly-generated page
		$post_id = get_the_ID();
		$is_buildfuly_page = get_post_meta( $post_id, 'buildfuly_generated', true );
		
		if ( ! $is_buildfuly_page ) {
			return;
		}

		// Check if layout takeover is enabled
		$takeover_enabled = get_option( 'buildfuly_takeover_layout', '1' );
		if ( $takeover_enabled !== '1' ) {
			return;
		}

		// Add CSS to hide theme header/footer
		add_action( 'wp_head', array( $this, 'hide_theme_header_footer_css' ), 999 );
		
		// Add our header at the start of body
		add_action( 'wp_body_open', array( $this, 'render_header_nav' ), 1 );
		
		// Add our footer at the end
		add_action( 'wp_footer', array( $this, 'render_footer' ), 1 );
	}

	/**
	 * CSS to hide common theme header/footer elements
	 */
	public function hide_theme_header_footer_css() {
		?>
		<style id="buildfuly-layout-takeover">
			/* Hide common theme headers */
			body > header:not(.buildfuly-site-header),
			#masthead,
			.site-header:not(.buildfuly-site-header),
			#site-header,
			.header-wrap,
			.main-header,
			#header,
			.elementor-location-header,
			.ast-header-html-1, /* Astra */
			.site-branding-container,
			header.wp-block-template-part,
			/* GeneratePress specific */
			.site-header,
			.main-navigation {
				display: none !important;
			}
			
			/* Hide common theme footers */
			body > footer:not(.buildfuly-site-footer),
			#colophon,
			.site-footer:not(.buildfuly-site-footer),
			#site-footer,
			.footer-wrap,
			.main-footer,
			#footer,
			.elementor-location-footer,
			footer.wp-block-template-part,
			/* GeneratePress specific */
			.site-info {
				display: none !important;
			}
			
			/* Hide theme featured image / page header image */
			.featured-image.page-header-image,
			.post-image,
			.entry-image,
			.wp-post-image-wrapper,
			.page-hero,
			.page-header-image-single {
				display: none !important;
			}
			
			/* Ensure our elements show */
			.buildfuly-site-header,
			.buildfuly-site-footer {
				display: block !important;
			}
			
			/* Remove any top margin/padding from main content area */
			main, #main, .site-main, .site-content, #content, #primary {
				margin-top: 0 !important;
				padding-top: 0 !important;
			}
		</style>
		<?php
	}

	/**
	 * Register custom nav menu locations
	 */
	public function register_nav_menus() {
		register_nav_menus( array(
			'buildfuly-primary' => __( 'Buildfuly Primary Menu', 'buildfuly' ),
			'buildfuly-footer'  => __( 'Buildfuly Footer Menu', 'buildfuly' ),
		) );
	}

	/**
	 * Get all service pages (pages tagged as 'service')
	 *
	 * @return array Array of WP_Post objects
	 */
	public static function get_service_pages() {
		$args = array(
			'post_type'      => 'page',
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'meta_query'     => array(
				array(
					'key'     => 'buildfuly_page_type',
					'value'   => self::SERVICE_PAGE_TYPES,
					'compare' => 'IN',
				),
			),
			'orderby'        => 'title',
			'order'          => 'ASC',
		);

		return get_posts( $args );
	}

	/**
	 * Get all Buildfuly-generated pages
	 *
	 * @return array Array of WP_Post objects
	 */
	public static function get_all_buildfuly_pages() {
		$args = array(
			'post_type'      => 'page',
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'meta_query'     => array(
				array(
					'key'     => 'buildfuly_generated',
					'value'   => '1',
					'compare' => '=',
				),
			),
			'orderby'        => 'menu_order',
			'order'          => 'ASC',
		);

		return get_posts( $args );
	}

	/**
	 * Get NAP (Name, Address, Phone) data
	 *
	 * @return array NAP data
	 */
	public static function get_nap_data() {
		return array(
			'business_name' => get_option( 'buildfuly_business_name', '' ),
			'phone'         => get_option( 'buildfuly_phone', '' ),
			'address'       => get_option( 'buildfuly_address', '' ),
			'city'          => get_option( 'buildfuly_city', '' ),
			'state'         => get_option( 'buildfuly_state', '' ),
			'zip'           => get_option( 'buildfuly_zip', '' ),
			'hours'         => get_option( 'buildfuly_hours', '' ),
		);
	}

	/**
	 * Get formatted full address
	 *
	 * @return string Full address or empty string
	 */
	public static function get_full_address() {
		$nap = self::get_nap_data();
		
		$parts = array_filter( array(
			$nap['address'],
			$nap['city'],
			$nap['state'] . ( $nap['zip'] ? ' ' . $nap['zip'] : '' ),
		) );

		return implode( ', ', $parts );
	}

	/**
	 * Clean page title for navigation display
	 * Removes "| Brand Name" suffix and trims
	 *
	 * @param string $title Page title
	 * @return string Cleaned title
	 */
	public static function clean_nav_title( $title ) {
		// Remove " | Brand Name" suffix if present
		if ( strpos( $title, ' | ' ) !== false ) {
			$parts = explode( ' | ', $title );
			$title = $parts[0]; // Keep only the first part
		}
		
		// Also remove " - Brand Name" format
		if ( strpos( $title, ' - ' ) !== false && substr_count( $title, ' - ' ) === 1 ) {
			$parts = explode( ' - ', $title );
			// Only strip if second part looks like a brand name (short, capitalized)
			if ( strlen( $parts[1] ) < 30 ) {
				$title = $parts[0];
			}
		}
		
		return trim( $title );
	}

	/**
	 * Maybe use Buildfuly auto-generated menu
	 *
	 * @param array $args Menu arguments
	 * @return array Modified arguments
	 */
	public function maybe_use_buildfuly_menu( $args ) {
		// Only intercept if using our menu locations and no custom menu is set
		if ( isset( $args['theme_location'] ) && 
			 in_array( $args['theme_location'], array( 'buildfuly-primary', 'buildfuly-footer' ), true ) &&
			 ! has_nav_menu( $args['theme_location'] ) ) {
			
			// Override with our custom menu
			$args['items_wrap'] = '<ul id="%1$s" class="%2$s">%3$s</ul>';
			$args['fallback_cb'] = array( $this, 'fallback_menu' );
		}

		return $args;
	}

	/**
	 * Fallback menu when no custom menu is set
	 *
	 * @param array $args Menu arguments
	 */
	public function fallback_menu( $args ) {
		$service_pages = self::get_service_pages();
		$all_pages = self::get_all_buildfuly_pages();
		
		// Legal page slugs to exclude from header
		$legal_slugs = array( 'privacy-policy', 'terms-of-service' );
		
		// Use Buildfuly homepage if set, otherwise site root
		$homepage_id = get_option( 'buildfuly_homepage_id', 0 );
		$home_url = $homepage_id ? get_permalink( $homepage_id ) : home_url( '/' );
		
		$menu_html = '<ul class="buildfuly-nav-menu">';
		
		// Home link
		$menu_html .= sprintf(
			'<li class="menu-item menu-item-home"><a href="%s">%s</a></li>',
			esc_url( $home_url ),
			esc_html__( 'Home', 'buildfuly' )
		);

		// Services dropdown (if there are service pages)
		if ( ! empty( $service_pages ) ) {
			$menu_html .= '<li class="menu-item menu-item-has-children buildfuly-services-dropdown">';
			$menu_html .= '<a href="#">' . esc_html__( 'Services', 'buildfuly' ) . ' <span class="dropdown-arrow">▼</span></a>';
			$menu_html .= '<ul class="sub-menu">';
			
			foreach ( $service_pages as $page ) {
				$menu_html .= sprintf(
					'<li class="menu-item"><a href="%s">%s</a></li>',
					esc_url( get_permalink( $page ) ),
					esc_html( self::clean_nav_title( $page->post_title ) )
				);
			}
			
			$menu_html .= '</ul></li>';
		}

		// Other pages (non-service, non-home, non-legal)
		foreach ( $all_pages as $page ) {
			$page_type = get_post_meta( $page->ID, 'buildfuly_page_type', true );
			$is_home = ( get_option( 'page_on_front' ) == $page->ID );
			
			// Skip home, service pages, and legal pages
			if ( $is_home || in_array( $page_type, self::SERVICE_PAGE_TYPES, true ) ) {
				continue;
			}
			
			// Skip legal pages (privacy, terms) - they go in footer only
			if ( in_array( $page->post_name, $legal_slugs, true ) ) {
				continue;
			}
			
			$menu_html .= sprintf(
				'<li class="menu-item"><a href="%s">%s</a></li>',
				esc_url( get_permalink( $page ) ),
				esc_html( self::clean_nav_title( $page->post_title ) )
			);
		}

		// Contact link (if exists)
		$contact_page = get_page_by_path( 'contact' );
		if ( ! $contact_page ) {
			$contact_page = get_page_by_title( 'Contact' );
		}
		if ( $contact_page && $contact_page->post_status === 'publish' ) {
			// Check if not already in the list
			$already_listed = false;
			foreach ( $all_pages as $page ) {
				if ( $page->ID === $contact_page->ID ) {
					$already_listed = true;
					break;
				}
			}
			if ( ! $already_listed ) {
				$menu_html .= sprintf(
					'<li class="menu-item"><a href="%s">%s</a></li>',
					esc_url( get_permalink( $contact_page ) ),
					esc_html__( 'Contact', 'buildfuly' )
				);
			}
		}

		$menu_html .= '</ul>';

		echo $menu_html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- HTML is escaped above
	}

	/**
	 * Render header navigation bar
	 */
	public function render_header_nav() {
		// Only render on frontend, not in admin
		if ( is_admin() ) {
			return;
		}

		// Check if Buildfuly header is enabled
		$header_enabled = get_option( 'buildfuly_enable_header', '1' );
		if ( $header_enabled !== '1' ) {
			return;
		}

		$nap = self::get_nap_data();
		$logo_id = get_theme_mod( 'custom_logo' );
		$logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'medium' ) : '';
		$site_name = $nap['business_name'] ?: get_bloginfo( 'name' );
		$phone = $nap['phone'];
		
		// Get theme colors - use same options as components
		$primary_color = get_option( 'buildfuly_primary_color', '#00bfa5' );
		$text_color = get_option( 'buildfuly_text_color', '#1e293b' );
		$bg_color = get_option( 'buildfuly_bg_color', '#ffffff' );
		
		// Get homepage URL - use Buildfuly homepage if set, otherwise site root
		$homepage_id = get_option( 'buildfuly_homepage_id', 0 );
		$home_link = $homepage_id ? get_permalink( $homepage_id ) : home_url( '/' );
		?>
		<header class="buildfuly-site-header">
			<div class="buildfuly-header-inner">
				<!-- Logo/Brand - Apple Style -->
				<div class="buildfuly-brand">
					<a href="<?php echo esc_url( $home_link ); ?>" class="buildfuly-brand-link">
						<?php if ( $logo_url ) : ?>
							<img src="<?php echo esc_url( $logo_url ); ?>" alt="<?php echo esc_attr( $site_name ); ?>" class="buildfuly-logo">
						<?php else : ?>
							<span class="buildfuly-brand-text"><?php echo esc_html( $site_name ); ?></span>
						<?php endif; ?>
					</a>
				</div>

				<!-- Navigation -->
				<nav class="buildfuly-main-nav">
					<?php
					wp_nav_menu( array(
						'theme_location'  => 'buildfuly-primary',
						'container'       => false,
						'menu_class'      => 'buildfuly-nav-menu',
						'fallback_cb'     => array( $this, 'fallback_menu' ),
						'depth'           => 2,
					) );
					?>

					<?php if ( $phone ) : ?>
						<a href="tel:<?php echo esc_attr( preg_replace( '/[^0-9+]/', '', $phone ) ); ?>" class="buildfuly-phone-cta">
							<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
								<path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"></path>
							</svg>
							<span><?php echo esc_html( $phone ); ?></span>
						</a>
					<?php endif; ?>
				</nav>

				<!-- Mobile Menu Toggle -->
				<button class="buildfuly-mobile-toggle" aria-label="Toggle menu">
					<svg class="menu-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
						<line x1="3" y1="6" x2="21" y2="6"></line>
						<line x1="3" y1="12" x2="21" y2="12"></line>
						<line x1="3" y1="18" x2="21" y2="18"></line>
					</svg>
					<svg class="close-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display:none;">
						<line x1="18" y1="6" x2="6" y2="18"></line>
						<line x1="6" y1="6" x2="18" y2="18"></line>
					</svg>
				</button>
			</div>
		</header>

		<style>
			/* Apple-style Header */
			.buildfuly-site-header {
				position: sticky;
				top: 0;
				z-index: 9999;
				background: rgba(255, 255, 255, 0.8);
				backdrop-filter: saturate(180%) blur(20px);
				-webkit-backdrop-filter: saturate(180%) blur(20px);
				border-bottom: 1px solid rgba(0, 0, 0, 0.1);
			}
			.buildfuly-header-inner {
				max-width: 1200px;
				margin: 0 auto;
				padding: 12px 22px;
				display: flex;
				justify-content: space-between;
				align-items: center;
			}
			
			/* Apple-style Brand - single line */
			.buildfuly-brand {
				flex-shrink: 0;
			}
			.buildfuly-brand-link {
				display: flex;
				align-items: center;
				text-decoration: none;
			}
			.buildfuly-logo {
				max-height: 44px;
				width: auto;
			}
			.buildfuly-brand-text {
				font-size: 21px;
				font-weight: 600;
				color: <?php echo esc_attr( $text_color ); ?>;
				letter-spacing: -0.02em;
				font-family: -apple-system, BlinkMacSystemFont, "SF Pro Display", "Segoe UI", Roboto, sans-serif;
				white-space: nowrap;
			}
			.buildfuly-brand-link:hover .buildfuly-brand-text {
				color: <?php echo esc_attr( $primary_color ); ?>;
			}
			
			/* Hide theme's page title/H1 on Buildfuly pages */
			.entry-title,
			.page-title,
			article.page > header .entry-title,
			.hentry .entry-title,
			.ast-archive-description .ast-archive-title,
			.single .entry-header .entry-title,
			.page .entry-header .entry-title {
				display: none !important;
			}
			
			/* Navigation */
			.buildfuly-main-nav {
				display: flex;
				align-items: center;
				gap: 8px;
			}
			.buildfuly-nav-menu {
				display: flex;
				list-style: none;
				margin: 0;
				padding: 0;
				gap: 4px;
			}
			.buildfuly-nav-menu li {
				position: relative;
			}
			.buildfuly-nav-menu > li > a {
				color: <?php echo esc_attr( $text_color ); ?>;
				text-decoration: none;
				font-weight: 400;
				font-size: 14px;
				padding: 8px 12px;
				display: inline-block;
				transition: all 0.2s ease;
				border-radius: 6px;
				font-family: -apple-system, BlinkMacSystemFont, "SF Pro Text", "Segoe UI", Roboto, sans-serif;
			}
			.buildfuly-nav-menu > li > a:hover {
				background: rgba(0, 0, 0, 0.04);
				color: <?php echo esc_attr( $primary_color ); ?>;
			}
			.buildfuly-nav-menu .dropdown-arrow {
				font-size: 8px;
				margin-left: 4px;
				opacity: 0.5;
			}
			
			/* Dropdown */
			.buildfuly-nav-menu .sub-menu {
				position: absolute;
				top: calc(100% + 8px);
				left: 50%;
				transform: translateX(-50%) translateY(10px);
				background: rgba(255, 255, 255, 0.98);
				backdrop-filter: saturate(180%) blur(20px);
				-webkit-backdrop-filter: saturate(180%) blur(20px);
				box-shadow: 0 4px 24px rgba(0, 0, 0, 0.12), 0 0 0 1px rgba(0, 0, 0, 0.05);
				border-radius: 12px;
				padding: 8px;
				min-width: 180px;
				list-style: none;
				margin: 0;
				opacity: 0;
				visibility: hidden;
				transition: all 0.2s ease;
				z-index: 100;
			}
			.buildfuly-nav-menu li:hover > .sub-menu {
				opacity: 1;
				visibility: visible;
				transform: translateX(-50%) translateY(0);
			}
			.buildfuly-nav-menu .sub-menu li a {
				display: block;
				padding: 10px 14px;
				color: <?php echo esc_attr( $text_color ); ?>;
				text-decoration: none;
				font-size: 13px;
				border-radius: 8px;
				transition: background 0.15s ease;
				font-family: -apple-system, BlinkMacSystemFont, "SF Pro Text", "Segoe UI", Roboto, sans-serif;
			}
			.buildfuly-nav-menu .sub-menu li a:hover {
				background: <?php echo esc_attr( $primary_color ); ?>;
				color: #fff;
			}
			
			/* Phone CTA Button */
			.buildfuly-phone-cta {
				display: inline-flex;
				align-items: center;
				gap: 8px;
				padding: 10px 18px;
				background: <?php echo esc_attr( $primary_color ); ?>;
				color: #fff;
				text-decoration: none;
				border-radius: 980px;
				font-weight: 500;
				font-size: 14px;
				transition: all 0.2s ease;
				font-family: -apple-system, BlinkMacSystemFont, "SF Pro Text", "Segoe UI", Roboto, sans-serif;
				margin-left: 16px;
			}
			.buildfuly-phone-cta:hover {
				transform: scale(1.02);
				filter: brightness(1.1);
			}
			.buildfuly-phone-cta svg {
				width: 16px;
				height: 16px;
			}
			
			/* Mobile Toggle */
			.buildfuly-mobile-toggle {
				display: none;
				background: none;
				border: none;
				padding: 8px;
				cursor: pointer;
				color: <?php echo esc_attr( $text_color ); ?>;
				border-radius: 8px;
				transition: background 0.2s ease;
			}
			.buildfuly-mobile-toggle:hover {
				background: rgba(0, 0, 0, 0.04);
			}

			@media (max-width: 768px) {
				.buildfuly-main-nav {
					display: none !important;
				}
				.buildfuly-mobile-toggle {
					display: flex !important;
				}
				.buildfuly-main-nav.mobile-open {
					display: flex !important;
					flex-direction: column;
					position: absolute;
					top: 100%;
					left: 0;
					right: 0;
					background: rgba(255, 255, 255, 0.98);
					backdrop-filter: saturate(180%) blur(20px);
					-webkit-backdrop-filter: saturate(180%) blur(20px);
					padding: 16px;
					border-bottom: 1px solid rgba(0, 0, 0, 0.1);
				}
				.buildfuly-main-nav.mobile-open .buildfuly-nav-menu {
					flex-direction: column;
					gap: 0;
				}
				.buildfuly-main-nav.mobile-open .buildfuly-nav-menu li a {
					padding: 15px 0;
					border-bottom: 1px solid #eee;
				}
				.buildfuly-main-nav.mobile-open .sub-menu {
					position: static;
					opacity: 1;
					visibility: visible;
					transform: none;
					box-shadow: none;
					padding-left: 20px;
				}
			}
		</style>

		<script>
			document.addEventListener('DOMContentLoaded', function() {
				var toggle = document.querySelector('.buildfuly-mobile-toggle');
				var nav = document.querySelector('.buildfuly-main-nav');
				var menuIcon = toggle ? toggle.querySelector('.menu-icon') : null;
				var closeIcon = toggle ? toggle.querySelector('.close-icon') : null;
				
				if (toggle && nav) {
					toggle.addEventListener('click', function() {
						nav.classList.toggle('mobile-open');
						var isOpen = nav.classList.contains('mobile-open');
						if (menuIcon) menuIcon.style.display = isOpen ? 'none' : 'block';
						if (closeIcon) closeIcon.style.display = isOpen ? 'block' : 'none';
					});
				}
			});
		</script>
		<?php
	}

	/**
	 * Render footer with NAP and service links
	 */
	public function render_footer() {
		// Only render on frontend, not in admin
		if ( is_admin() ) {
			return;
		}

		// Check if Buildfuly footer is enabled
		$footer_enabled = get_option( 'buildfuly_enable_footer', '1' );
		if ( $footer_enabled !== '1' ) {
			return;
		}

		$nap = self::get_nap_data();
		$service_pages = self::get_service_pages();
		$all_pages = self::get_all_buildfuly_pages();
		$full_address = self::get_full_address();
		$site_name = $nap['business_name'] ?: get_bloginfo( 'name' );
		$current_year = gmdate( 'Y' );
		
		// Get theme colors - use same options as components
		$primary_color = get_option( 'buildfuly_primary_color', '#00bfa5' );
		$text_color = get_option( 'buildfuly_text_color', '#1e293b' );
		$bg_color = get_option( 'buildfuly_bg_color', '#ffffff' );
		?>
		<footer class="buildfuly-site-footer">
			<div class="buildfuly-footer-inner">
				<div class="buildfuly-footer-grid">
					
					<!-- NAP Column -->
					<div class="buildfuly-footer-col buildfuly-footer-nap">
						<h3 class="buildfuly-footer-brand"><?php echo esc_html( $site_name ); ?></h3>
						
						<?php if ( $full_address ) : ?>
							<p class="buildfuly-footer-address">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
									<path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"></path>
									<circle cx="12" cy="10" r="3"></circle>
								</svg>
								<span itemscope itemtype="http://schema.org/PostalAddress">
									<?php if ( $nap['address'] ) : ?>
										<span itemprop="streetAddress"><?php echo esc_html( $nap['address'] ); ?></span><br>
									<?php endif; ?>
									<?php if ( $nap['city'] ) : ?>
										<span itemprop="addressLocality"><?php echo esc_html( $nap['city'] ); ?></span><?php echo $nap['state'] ? ', ' : ''; ?>
									<?php endif; ?>
									<?php if ( $nap['state'] ) : ?>
										<span itemprop="addressRegion"><?php echo esc_html( $nap['state'] ); ?></span>
									<?php endif; ?>
									<?php if ( $nap['zip'] ) : ?>
										<span itemprop="postalCode"><?php echo esc_html( $nap['zip'] ); ?></span>
									<?php endif; ?>
								</span>
							</p>
						<?php endif; ?>
						
						<?php if ( $nap['phone'] ) : ?>
							<p class="buildfuly-footer-phone">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
									<path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"></path>
								</svg>
								<a href="tel:<?php echo esc_attr( preg_replace( '/[^0-9+]/', '', $nap['phone'] ) ); ?>" itemprop="telephone">
									<?php echo esc_html( $nap['phone'] ); ?>
								</a>
							</p>
						<?php endif; ?>
						
						<?php if ( $nap['hours'] ) : ?>
							<p class="buildfuly-footer-hours">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
									<circle cx="12" cy="12" r="10"></circle>
									<polyline points="12 6 12 12 16 14"></polyline>
								</svg>
								<span><?php echo nl2br( esc_html( $nap['hours'] ) ); ?></span>
							</p>
						<?php endif; ?>
					</div>

					<!-- Services Column -->
					<?php if ( ! empty( $service_pages ) ) : ?>
						<div class="buildfuly-footer-col buildfuly-footer-services">
							<h4 class="buildfuly-footer-heading"><?php esc_html_e( 'Services', 'buildfuly' ); ?></h4>
							<ul class="buildfuly-footer-list">
								<?php foreach ( $service_pages as $page ) : ?>
									<li>
										<a href="<?php echo esc_url( get_permalink( $page ) ); ?>">
											<?php echo esc_html( $page->post_title ); ?>
										</a>
									</li>
								<?php endforeach; ?>
							</ul>
						</div>
					<?php endif; ?>

					<!-- Quick Links Column -->
					<?php
					// Legal page slugs to separate
					$legal_slugs = array( 'privacy-policy', 'terms-of-service' );
					?>
					<div class="buildfuly-footer-col buildfuly-footer-links">
						<h4 class="buildfuly-footer-heading"><?php esc_html_e( 'Company', 'buildfuly' ); ?></h4>
						<ul class="buildfuly-footer-list">
							<li>
								<a href="<?php echo esc_url( home_url( '/' ) ); ?>">
									<?php esc_html_e( 'Home', 'buildfuly' ); ?>
								</a>
							</li>
							<?php
							// Add non-service, non-legal pages
							foreach ( $all_pages as $page ) :
								$page_type = get_post_meta( $page->ID, 'buildfuly_page_type', true );
								$is_home = ( get_option( 'page_on_front' ) == $page->ID );
								
								// Skip home, service pages, and legal pages
								if ( $is_home || in_array( $page_type, self::SERVICE_PAGE_TYPES, true ) ) {
									continue;
								}
								
								// Skip legal pages - they go in separate section
								if ( in_array( $page->post_name, $legal_slugs, true ) ) {
									continue;
								}
								?>
								<li>
									<a href="<?php echo esc_url( get_permalink( $page ) ); ?>">
										<?php echo esc_html( $page->post_title ); ?>
									</a>
								</li>
							<?php endforeach; ?>
						</ul>
					</div>

					<!-- Legal Links Column -->
					<?php
					$privacy_page = get_page_by_path( 'privacy-policy' );
					$terms_page = get_page_by_path( 'terms-of-service' );
					if ( ( $privacy_page && $privacy_page->post_status === 'publish' ) || ( $terms_page && $terms_page->post_status === 'publish' ) ) :
					?>
					<div class="buildfuly-footer-col buildfuly-footer-links">
						<h4 class="buildfuly-footer-heading"><?php esc_html_e( 'Legal', 'buildfuly' ); ?></h4>
						<ul class="buildfuly-footer-list">
							<?php if ( $privacy_page && $privacy_page->post_status === 'publish' ) : ?>
								<li>
									<a href="<?php echo esc_url( get_permalink( $privacy_page ) ); ?>">
										<?php esc_html_e( 'Privacy Policy', 'buildfuly' ); ?>
									</a>
								</li>
							<?php endif; ?>
							<?php if ( $terms_page && $terms_page->post_status === 'publish' ) : ?>
								<li>
									<a href="<?php echo esc_url( get_permalink( $terms_page ) ); ?>">
										<?php esc_html_e( 'Terms of Service', 'buildfuly' ); ?>
									</a>
								</li>
							<?php endif; ?>
						</ul>
					</div>
					<?php endif; ?>

				</div>

				<!-- Copyright -->
				<div class="buildfuly-footer-bottom">
					<p>© <?php echo esc_html( $current_year ); ?> <?php echo esc_html( $site_name ); ?>. <?php esc_html_e( 'All rights reserved.', 'buildfuly' ); ?></p>
				</div>
			</div>
		</footer>

		<style>
			/* Apple-style Footer */
			.buildfuly-site-footer {
				background: #f5f5f7;
				color: #1d1d1f;
				padding: 48px 22px 24px;
				font-family: -apple-system, BlinkMacSystemFont, "SF Pro Text", "Segoe UI", Roboto, sans-serif;
			}
			.buildfuly-footer-inner {
				max-width: 980px;
				margin: 0 auto;
			}
			.buildfuly-footer-grid {
				display: grid;
				grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
				gap: 32px;
				padding-bottom: 24px;
				border-bottom: 1px solid #d2d2d7;
			}
			.buildfuly-footer-col {
				min-width: 0;
			}
			
			/* Brand in Footer */
			.buildfuly-footer-brand {
				font-size: 17px;
				font-weight: 600;
				color: #1d1d1f;
				margin: 0 0 16px 0;
				letter-spacing: -0.01em;
			}
			
			/* NAP Info */
			.buildfuly-footer-address,
			.buildfuly-footer-phone,
			.buildfuly-footer-hours {
				display: flex;
				gap: 10px;
				align-items: flex-start;
				margin: 0 0 12px 0;
				color: #6e6e73;
				font-size: 13px;
				line-height: 1.5;
			}
			.buildfuly-footer-address svg,
			.buildfuly-footer-phone svg,
			.buildfuly-footer-hours svg {
				flex-shrink: 0;
				margin-top: 2px;
				opacity: 0.6;
			}
			.buildfuly-footer-phone a {
				color: <?php echo esc_attr( $primary_color ); ?>;
				text-decoration: none;
				transition: opacity 0.2s ease;
			}
			.buildfuly-footer-phone a:hover {
				opacity: 0.7;
			}
			
			/* Section Headings */
			.buildfuly-footer-heading {
				font-size: 12px;
				font-weight: 600;
				color: #1d1d1f;
				margin: 0 0 12px 0;
				text-transform: uppercase;
				letter-spacing: 0.03em;
			}
			
			/* Links List */
			.buildfuly-footer-list {
				list-style: none;
				margin: 0;
				padding: 0;
			}
			.buildfuly-footer-list li {
				margin-bottom: 8px;
			}
			.buildfuly-footer-list a {
				color: #6e6e73;
				text-decoration: none;
				font-size: 13px;
				transition: color 0.2s ease;
			}
			.buildfuly-footer-list a:hover {
				color: <?php echo esc_attr( $primary_color ); ?>;
			}
			
			/* Copyright */
			.buildfuly-footer-bottom {
				padding-top: 20px;
				text-align: center;
			}
			.buildfuly-footer-bottom p {
				margin: 0;
				color: #86868b;
				font-size: 12px;
			}
			
			@media (max-width: 768px) {
				.buildfuly-footer-grid {
					grid-template-columns: 1fr 1fr;
				}
				.buildfuly-footer-nap {
					grid-column: 1 / -1;
				}
			}
			@media (max-width: 480px) {
				.buildfuly-footer-grid {
					grid-template-columns: 1fr;
				}
			}
		</style>

		<?php
		// Output LocalBusiness Schema.org JSON-LD
		$this->output_local_business_schema( $nap, $service_pages );
	}

	/**
	 * Output LocalBusiness Schema.org structured data
	 *
	 * @param array $nap NAP data
	 * @param array $service_pages Array of service page posts
	 */
	private function output_local_business_schema( $nap, $service_pages ) {
		if ( empty( $nap['business_name'] ) && empty( get_bloginfo( 'name' ) ) ) {
			return;
		}

		$site_name = $nap['business_name'] ?: get_bloginfo( 'name' );
		$logo_id = get_theme_mod( 'custom_logo' );
		$logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'full' ) : '';

		$schema = array(
			'@context'  => 'https://schema.org',
			'@type'     => 'LocalBusiness',
			'name'      => $site_name,
			'url'       => home_url( '/' ),
		);

		if ( $logo_url ) {
			$schema['logo'] = $logo_url;
			$schema['image'] = $logo_url;
		}

		// Add address
		if ( $nap['address'] || $nap['city'] || $nap['state'] || $nap['zip'] ) {
			$schema['address'] = array(
				'@type'           => 'PostalAddress',
				'streetAddress'   => $nap['address'] ?: '',
				'addressLocality' => $nap['city'] ?: '',
				'addressRegion'   => $nap['state'] ?: '',
				'postalCode'      => $nap['zip'] ?: '',
				'addressCountry'  => 'US',
			);
		}

		// Add phone
		if ( $nap['phone'] ) {
			$schema['telephone'] = $nap['phone'];
		}

		// Add services offered
		if ( ! empty( $service_pages ) ) {
			$schema['hasOfferCatalog'] = array(
				'@type'            => 'OfferCatalog',
				'name'             => __( 'Services', 'buildfuly' ),
				'itemListElement'  => array(),
			);

			foreach ( $service_pages as $index => $page ) {
				$schema['hasOfferCatalog']['itemListElement'][] = array(
					'@type'    => 'Offer',
					'position' => $index + 1,
					'itemOffered' => array(
						'@type' => 'Service',
						'name'  => $page->post_title,
						'url'   => get_permalink( $page ),
					),
				);
			}
		}

		// Add opening hours if available
		if ( $nap['hours'] ) {
			// Simple text representation - proper parsing would need more structure
			$schema['openingHours'] = $nap['hours'];
		}

		?>
		<script type="application/ld+json">
		<?php echo wp_json_encode( $schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES ); ?>
		</script>
		<?php
	}
}
