<?php
/**
 * Logger utility for Buildfuly plugin
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

/**
 * Logger utility class.
 *
 * Provides WordPress-compliant debug logging that respects WP_DEBUG settings.
 *
 * @since      1.0.0
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @author     Buildfuly
 */
class Buildfuly_Logger {

	/**
	 * Log debug messages only when WP_DEBUG is enabled.
	 *
	 * @since    1.0.0
	 * @param    string    $message    The message to log.
	 */
	public static function debug( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log, WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging only when WP_DEBUG is enabled, intentional for debugging
			error_log( '[Buildfuly] ' . $message );
		}
	}

	/**
	 * Log error messages (always logged, even without WP_DEBUG).
	 *
	 * @since    1.0.0
	 * @param    string    $message    The error message to log.
	 */
	public static function error( $message ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log, WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Critical errors must be logged for debugging
		error_log( '[Buildfuly ERROR] ' . $message );
	}

	/**
	 * Log complex data structures (arrays, objects) for debugging.
	 *
	 * @since    1.0.0
	 * @param    string $prefix  Prefix message to add context.
	 * @param    mixed  $data    The data to log (array, object, etc).
	 */
	public static function debug_data( $prefix, $data ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log, WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging only when WP_DEBUG is enabled, intentional for debugging
			error_log( '[Buildfuly] ' . $prefix . ': ' . wp_json_encode( $data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES ) );
		}
	}
}
