<?php
/**
 * License Management for Buildfuly
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

class Buildfuly_License {

	/**
	 * API client instance
	 *
	 * @var Buildfuly_API_Client
	 */
	private $api_client;

	/**
	 * Initialize the license manager
	 */
	public function __construct() {
		$this->api_client = new Buildfuly_API_Client();
		
		// Show payment failure notices
		add_action( 'admin_notices', array( $this, 'show_payment_notices' ) );
	}

	/**
	 * Show admin notices for payment failures
	 */
	public function show_payment_notices() {
		$license_data = get_option( 'buildfuly_license_data' );
		
		if ( ! $license_data || ! isset( $license_data['subscription_status'] ) ) {
			return;
		}

		$subscription_status = $license_data['subscription_status'];
		
		// Payment failed - show warning
		if ( $subscription_status === 'past_due' ) {
			$tier = $license_data['tier'] ?? 'Starter';
			?>
			<div class="notice notice-error is-dismissible">
				<p>
					<strong>Payment Failed:</strong> Your payment could not be processed. 
					You've been temporarily downgraded to <strong><?php echo esc_html( ucfirst( $tier ) ); ?> tier</strong>. 
					Please update your payment method in your 
					<a href="https://billing.build.fuly.io" target="_blank">billing portal</a> 
					to restore your plan. We'll automatically retry the payment.
				</p>
			</div>
			<?php
		}
		
		// Subscription canceled
		if ( $subscription_status === 'canceled' ) {
			?>
			<div class="notice notice-warning is-dismissible">
				<p>
					<strong>Subscription Canceled:</strong> Your subscription has been canceled. 
					You're now on the <strong>Starter tier</strong> (1 page/day). 
					<a href="https://billing.build.fuly.io" target="_blank">Reactivate your subscription</a> 
					to restore full access.
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Activate a license key
	 *
	 * @param string $license_key License key to activate
	 * @return true|WP_Error True on success, WP_Error on failure
	 */
	public function activate( $license_key ) {
		// Validate format
		$license_key = sanitize_text_field( $license_key );
		
		if ( empty( $license_key ) ) {
			return new WP_Error( 'invalid_key', 'License key cannot be empty.' );
		}

		// Get old values before updating (to detect upgrades/changes)
		$old_license_key = get_option( 'buildfuly_license_key', '' );
		$old_tier = get_option( 'buildfuly_user_tier', 'starter' );

		// Authenticate with server
		$result = $this->api_client->authenticate( $license_key );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		// Store license information
		$new_tier = $result['license']['tier'] ?? 'starter';
		$status = $result['license']['subscription_status'] ?? 'active';
		update_option( 'buildfuly_license_key', $license_key );
		update_option( 'buildfuly_license_status', $status );
		update_option( 'buildfuly_user_tier', $new_tier );
		update_option( 'buildfuly_license_data', $result );
		update_option( 'buildfuly_license_activated_at', time() );

		// Clear any cached data
		delete_transient( 'buildfuly_license_check' );
		delete_transient( 'buildfuly_component_word_estimates' ); // Clear component estimates cache
		
		// Note: We do NOT reset usage here - usage is managed entirely by the API
		// and is tied to the license key in the database, resetting at midnight UTC
		Buildfuly_Logger::debug( "BuildfulyAI: License activated - tier: '{$new_tier}', status: '{$status}' (usage managed by API)" );

		do_action( 'buildfuly_license_activated', $license_key, $result );

		return true;
	}

	/**
	 * Deactivate current license
	 *
	 * @return true|WP_Error
	 */
	public function deactivate() {
		$license_key = get_option( 'buildfuly_license_key' );

		if ( ! $license_key ) {
			return new WP_Error( 'no_license', 'No active license to deactivate.' );
		}

		// Clear license data
		delete_option( 'buildfuly_license_key' );
		delete_option( 'buildfuly_license_status' );
		delete_option( 'buildfuly_license_data' );
		delete_option( 'buildfuly_free_license' );
		delete_option( 'buildfuly_api_token' );
		delete_transient( 'buildfuly_license_check' );
		
		// Revert to starter tier
		update_option( 'buildfuly_user_tier', 'starter' );

		// Always re-register free license after deactivation
		// This ensures users always have at least the free tier available
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-activator.php';
		Buildfuly_Activator::register_free_license();

		do_action( 'buildfuly_license_deactivated', $license_key );

		return true;
	}

	/**
	 * Initiate a domain transfer
	 * Sets a temporary flag that preserves usage when reactivating on a new domain
	 * 
	 * SECURITY LIMITS:
	 * - Max 2 transfers per month to prevent abuse
	 * - Transfers reset on the 1st of each month
	 * - Flag expires in 72 hours (not 24h to allow for migration time)
	 *
	 * @return true|WP_Error
	 */
	public function initiate_domain_transfer() {
		$license_key = get_option( 'buildfuly_license_key' );

		if ( ! $license_key ) {
			return new WP_Error( 'no_license', 'No active license to transfer.' );
		}

		// Check current transfer count from license data
		$license_data = get_option( 'buildfuly_license_data' );
		$transfers_this_month = isset( $license_data['license']['site_transfers_this_month'] ) 
			? intval( $license_data['license']['site_transfers_this_month'] ) 
			: 0;
		
		// Enforce monthly transfer limit (generous for legitimate use, restrictive for abuse)
		$max_monthly_transfers = 2;
		if ( $transfers_this_month >= $max_monthly_transfers ) {
			return new WP_Error( 
				'transfer_limit_exceeded', 
				sprintf( 
					'Monthly site transfer limit reached (%d/%d). Limit resets on the 1st of each month. If you need additional transfers, please contact support.', 
					$transfers_this_month, 
					$max_monthly_transfers 
				)
			);
		}

		// Set transfer flag for 72 hours (allows time for DNS propagation, migration, etc.)
		set_transient( 'buildfuly_domain_transfer_' . $license_key, true, 3 * DAY_IN_SECONDS );
		
		Buildfuly_Logger::debug( sprintf(
			"BuildfulyAI: Domain transfer initiated for license %s (transfers this month: %d/%d) - usage will be preserved on next activation within 72 hours",
			$license_key,
			$transfers_this_month,
			$max_monthly_transfers
		) );

		// Now deactivate normally
		$result = $this->deactivate();
		
		if ( ! is_wp_error( $result ) ) {
			// Show helpful message
			add_settings_error(
				'buildfuly_license',
				'domain_transfer_initiated',
				sprintf(
					'Domain transfer initiated! You have 72 hours to activate this license on your new domain. Your usage stats will be preserved. Transfers remaining this month: %d',
					$max_monthly_transfers - $transfers_this_month - 1
				),
				'success'
			);
		}
		
		return $result;
	}

	/**
	 * Check if license is active and valid
	 *
	 * @param bool $force_check Force fresh check, bypass cache
	 * @return bool
	 */
	public function is_active( $force_check = false ) {
		$license_key = get_option( 'buildfuly_license_key' );
		
		if ( ! $license_key ) {
			return false;
		}

		// Check cache first (24 hour)
		if ( ! $force_check ) {
			$cached = get_transient( 'buildfuly_license_check' );
			if ( $cached !== false ) {
				// Accept 'active', 'lifetime', or 'trialing' as valid
				return in_array( $cached, array( 'active', 'lifetime', 'trialing' ), true );
			}
		}

		// Validate with server
		$result = $this->api_client->validate_license( $license_key );

		if ( is_wp_error( $result ) ) {
			update_option( 'buildfuly_license_status', 'invalid' );
			set_transient( 'buildfuly_license_check', 'invalid', HOUR_IN_SECONDS );
			return false;
		}

		// Get status from API response
		$status = $result['license']['subscription_status'] ?? 'active';
		
		// Update local data with actual status from API
		update_option( 'buildfuly_license_status', $status );
		update_option( 'buildfuly_user_tier', $result['license']['tier'] ?? 'starter' );
		update_option( 'buildfuly_license_data', $result );
		set_transient( 'buildfuly_license_check', $status, DAY_IN_SECONDS );

		// Accept 'active', 'lifetime', or 'trialing' as valid
		return in_array( $status, array( 'active', 'lifetime', 'trialing' ), true );
	}

	/**
	 * Get current tier
	 *
	 * @return string Tier slug (starter, pro, growth, studio)
	 */
	public function get_tier() {
		return Buildfuly_Constants::get_user_tier();
	}

	/**
	 * Get tier display name
	 *
	 * @return string
	 */
	public function get_tier_name() {
		$tier = $this->get_tier();
		$config = Buildfuly_Constants::get_tier( $tier );
		return $config ? $config['name'] : 'Starter';
	}

	/**
	 * Get pages limit (total for starter, daily for paid tiers)
	 *
	 * @return int
	 */
	public function get_pages_limit() {
		return Buildfuly_Constants::get_pages_limit();
	}

	/**
	 * Get remaining pages
	 *
	 * @return int
	 */
	public function get_remaining_pages() {
		$limit = $this->get_pages_limit();
		$tier = Buildfuly_Constants::get_user_tier();
		
		// For starter tier, check total pages used
		if ( $tier === 'starter' ) {
			$used = $this->get_total_pages_used();
		} else {
			// For paid tiers, check daily usage
			$used = $this->get_pages_used_today();
		}
		
		return max( 0, $limit - $used );
	}

	/**
	 * Get total pages used (for starter tier - lifetime limit)
	 *
	 * @return int
	 */
	public function get_total_pages_used() {
		$usage = get_option( 'buildfuly_total_pages_used', 0 );
		return intval( $usage );
	}

	/**
	 * Get pages used today (for paid tiers - daily limit)
	 *
	 * @return int
	 */
	public function get_pages_used_today() {
		$usage = get_transient( 'buildfuly_daily_usage' );
		return $usage !== false ? intval( $usage ) : 0;
	}

	/**
	 * Check if user has access to a feature
	 *
	 * @param string $feature Feature identifier
	 * @return bool
	 */
	public function has_feature( $feature ) {
		return Buildfuly_Constants::has_feature( $feature );
	}

	/**
	 * Get time until next generation is allowed
	 *
	 * @return int Seconds until next generation, 0 if allowed now
	 */
	public function get_cooldown_remaining() {
		$last_generation = get_option( 'buildfuly_last_generation_time', 0 );
		
		if ( ! $last_generation ) {
			return 0;
		}

		$tier = $this->get_tier();
		$cooldown = Buildfuly_Constants::RATE_LIMIT_COOLDOWN[ $tier ] ?? 86400;
		
		$time_passed = time() - $last_generation;
		$remaining = $cooldown - $time_passed;

		return max( 0, $remaining );
	}

	/**
	 * Check if user can generate a page now
	 *
	 * @return true|WP_Error True if allowed, WP_Error with reason if not
	 */
	public function can_generate() {
		// Check if license is active
		if ( ! $this->is_active() ) {
			$tier = $this->get_tier();
			if ( $tier !== 'starter' ) {
				return new WP_Error(
					'license_inactive',
					'Your license is inactive. Please activate a valid license or switch to Starter tier.'
				);
			}
		}

		// Check pages limit
		$remaining = $this->get_remaining_pages();
		if ( $remaining <= 0 ) {
			$tier = Buildfuly_Constants::get_user_tier();
			$tier_name = $this->get_tier_name();
			$limit = $this->get_pages_limit();
			
			if ( $tier === 'starter' ) {
				$message = sprintf(
					'You have reached your limit of %d total page(s) for the %s tier. Upgrade for more pages!',
					$limit,
					$tier_name
				);
			} else {
				$message = sprintf(
					'You have reached your daily limit of %d page(s) for the %s tier. Please wait until tomorrow or upgrade!',
					$limit,
					$tier_name
				);
			}
			
			return new WP_Error(
				'pages_limit_reached',
				$message
			);
		}

		// Check cooldown
		$cooldown = $this->get_cooldown_remaining();
		if ( $cooldown > 0 ) {
			$hours = floor( $cooldown / 3600 );
			$minutes = floor( ( $cooldown % 3600 ) / 60 );
			
			return new WP_Error(
				'cooldown_active',
				sprintf(
					'Please wait %s before generating another page.',
					$hours > 0 ? "{$hours}h {$minutes}m" : "{$minutes}m"
				)
			);
		}

		return true;
	}

	/**
	 * Get license data
	 *
	 * @return array
	 */
	public function get_license_data() {
		$data = get_option( 'buildfuly_license_data', array() );
		
		return array_merge( array(
			'tier' => $this->get_tier(),
			'tier_name' => $this->get_tier_name(),
			'status' => get_option( 'buildfuly_license_status', 'inactive' ),
			'pages_limit' => $this->get_pages_limit(),
			'pages_used' => $this->get_pages_used_today(),
			'pages_remaining' => $this->get_remaining_pages(),
			'cooldown_remaining' => $this->get_cooldown_remaining(),
		), $data );
	}

	/**
	 * Get upgrade URL for user's tier
	 *
	 * @return string
	 */
	public function get_upgrade_url() {
		$tier = $this->get_tier();
		
		// Base upgrade URL - replace with your actual URL
		$base_url = 'https://build.fuly.io/pricing';
		
		return add_query_arg( array(
			'current_tier' => $tier,
			'site' => urlencode( get_site_url() ),
		), $base_url );
	}
}
