<?php
/**
 * Auto-generate and update legal/essential pages.
 *
 * Generates Contact, Privacy Policy, and Terms of Service pages
 * that automatically sync with business NAP (Name, Address, Phone) settings.
 * Uses Buildfuly styling with header/footer support.
 *
 * @link       https://buildfuly.com
 * @since      1.1.2
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

/**
 * Class Buildfuly_Legal_Pages
 *
 * Handles automatic generation and updating of legal pages.
 *
 * @since      1.1.2
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @author     Buildfuly
 */
class Buildfuly_Legal_Pages {

	/**
	 * Page slugs for legal pages.
	 *
	 * @since    1.1.2
	 * @access   private
	 * @var      array
	 */
	private static $page_slugs = array(
		'contact'         => 'contact',
		'privacy-policy'  => 'privacy-policy',
		'terms-of-service' => 'terms-of-service',
	);

	/**
	 * Initialize hooks for legal pages.
	 *
	 * @since    1.1.2
	 */
	public function __construct() {
		// Hook will be called from seo-functions when settings are saved
	}

	/**
	 * Generate or update all legal pages.
	 *
	 * @since    1.1.2
	 * @param    array $settings The SEO settings array.
	 */
	public static function sync_legal_pages( $settings = null ) {
		if ( $settings === null ) {
			$settings = get_option( 'buildfuly_seo_settings', array() );
		}

		self::sync_contact_page( $settings );
		self::sync_privacy_policy_page( $settings );
		self::sync_terms_of_service_page( $settings );
	}

	/**
	 * Get business info from settings.
	 *
	 * @since    1.1.2
	 * @param    array $settings The SEO settings array.
	 * @return   array Business info array.
	 */
	private static function get_business_info( $settings ) {
		$business_name = ! empty( $settings['business_name'] ) 
			? $settings['business_name'] 
			: get_option( 'buildfuly_business_name', get_bloginfo( 'name' ) );

		return array(
			'name'    => $business_name,
			'phone'   => $settings['business_phone'] ?? '',
			'email'   => $settings['business_email'] ?? '',
			'address' => $settings['business_address'] ?? '',
			'city'    => $settings['business_city'] ?? '',
			'state'   => $settings['business_state'] ?? '',
			'zip'     => $settings['business_zip'] ?? '',
			'country' => $settings['business_country'] ?? 'US',
			'website' => home_url(),
		);
	}

	/**
	 * Get theme colors.
	 *
	 * @since    1.1.2
	 * @return   array Colors array.
	 */
	private static function get_colors() {
		// Try to get colors from settings
		$primary = get_option( 'buildfuly_primary_color', '#0066cc' );
		$bg = get_option( 'buildfuly_background_color', '#ffffff' );
		$text = get_option( 'buildfuly_text_color', '#1a1a1a' );
		$text_light = get_option( 'buildfuly_text_light_color', '#666666' );
		$bg_alt = get_option( 'buildfuly_bg_alt_color', '#f8f9fa' );
		
		return array(
			'primary' => $primary,
			'bg' => $bg,
			'text' => $text,
			'text_light' => $text_light,
			'bg_alt' => $bg_alt,
		);
	}

	/**
	 * Build full address HTML.
	 *
	 * @since    1.1.2
	 * @param    array $info Business info array.
	 * @return   string Full address HTML.
	 */
	private static function build_full_address_html( $info ) {
		$parts = array();
		
		if ( ! empty( $info['address'] ) ) {
			$parts[] = esc_html( $info['address'] );
		}
		
		$city_state_zip = array();
		if ( ! empty( $info['city'] ) ) {
			$city_state_zip[] = esc_html( $info['city'] );
		}
		if ( ! empty( $info['state'] ) ) {
			$city_state_zip[] = esc_html( $info['state'] );
		}
		if ( ! empty( $info['zip'] ) ) {
			$city_state_zip[] = esc_html( $info['zip'] );
		}
		
		if ( ! empty( $city_state_zip ) ) {
			$parts[] = implode( ', ', $city_state_zip );
		}
		
		return implode( '<br>', $parts );
	}

	/**
	 * Find existing page by slug or meta.
	 *
	 * @since    1.1.2
	 * @param    string $page_type Page type (contact, privacy-policy, terms-of-service).
	 * @return   int|null Post ID or null if not found.
	 */
	private static function find_existing_page( $page_type ) {
		// First check by meta key
		$pages = get_posts( array(
			'post_type'      => 'page',
			'post_status'    => array( 'publish', 'draft' ),
			'meta_key'       => '_buildfuly_legal_page_type',
			'meta_value'     => $page_type,
			'posts_per_page' => 1,
		) );

		if ( ! empty( $pages ) ) {
			return $pages[0]->ID;
		}

		// Fallback: check by slug
		$slug = self::$page_slugs[ $page_type ] ?? $page_type;
		$page = get_page_by_path( $slug );
		
		if ( $page ) {
			// Tag it for future reference
			update_post_meta( $page->ID, '_buildfuly_legal_page_type', $page_type );
			return $page->ID;
		}

		return null;
	}

	/**
	 * Create or update a legal page.
	 *
	 * @since    1.1.2
	 * @param    string $page_type  Page type.
	 * @param    string $title      Page title.
	 * @param    string $content    Page content.
	 * @param    string $seo_desc   SEO description.
	 * @return   int Post ID.
	 */
	private static function create_or_update_page( $page_type, $title, $content, $seo_desc = '' ) {
		$existing_id = self::find_existing_page( $page_type );
		$slug = self::$page_slugs[ $page_type ] ?? $page_type;

		$page_data = array(
			'post_title'   => $title,
			'post_content' => $content,
			'post_status'  => 'publish',
			'post_type'    => 'page',
			'post_name'    => $slug,
		);

		if ( $existing_id ) {
			$page_data['ID'] = $existing_id;
			$post_id = wp_update_post( $page_data );
		} else {
			$post_id = wp_insert_post( $page_data );
		}

		if ( ! is_wp_error( $post_id ) && $post_id ) {
			// Mark as Buildfuly page for header/footer
			update_post_meta( $post_id, 'buildfuly_generated', '1' );
			update_post_meta( $post_id, '_buildfuly_legal_page_type', $page_type );
			update_post_meta( $post_id, '_buildfuly_auto_generated', '1' );
			update_post_meta( $post_id, '_buildfuly_last_synced', current_time( 'mysql' ) );
			
			// Set SEO meta
			if ( ! empty( $seo_desc ) ) {
				update_post_meta( $post_id, '_buildfuly_seo_description', $seo_desc );
			}
			update_post_meta( $post_id, '_buildfuly_seo_title', $title );
		}

		return $post_id;
	}

	/**
	 * Sync Contact page.
	 *
	 * @since    1.1.2
	 * @param    array $settings SEO settings.
	 * @return   int Post ID.
	 */
	public static function sync_contact_page( $settings ) {
		$info = self::get_business_info( $settings );
		$content = self::generate_contact_content( $info );
		$seo_desc = sprintf( 'Contact %s today. Call us at %s or visit our location.', $info['name'], $info['phone'] ?: 'our number' );
		
		return self::create_or_update_page( 'contact', 'Contact Us', $content, $seo_desc );
	}

	/**
	 * Sync Privacy Policy page.
	 *
	 * @since    1.1.2
	 * @param    array $settings SEO settings.
	 * @return   int Post ID.
	 */
	public static function sync_privacy_policy_page( $settings ) {
		$info = self::get_business_info( $settings );
		$content = self::generate_privacy_policy_content( $info );
		$seo_desc = sprintf( 'Privacy Policy for %s. Learn how we collect, use, and protect your personal information.', $info['name'] );
		
		return self::create_or_update_page( 'privacy-policy', 'Privacy Policy', $content, $seo_desc );
	}

	/**
	 * Sync Terms of Service page.
	 *
	 * @since    1.1.2
	 * @param    array $settings SEO settings.
	 * @return   int Post ID.
	 */
	public static function sync_terms_of_service_page( $settings ) {
		$info = self::get_business_info( $settings );
		$content = self::generate_terms_of_service_content( $info );
		$seo_desc = sprintf( 'Terms of Service for %s. Read our terms and conditions for using our services.', $info['name'] );
		
		return self::create_or_update_page( 'terms-of-service', 'Terms of Service', $content, $seo_desc );
	}

	/**
	 * Generate Contact page content with Buildfuly styling.
	 *
	 * @since    1.1.2
	 * @param    array $info Business info.
	 * @return   string HTML content.
	 */
	private static function generate_contact_content( $info ) {
		$colors = self::get_colors();
		$business_name = esc_html( $info['name'] );
		$full_address = self::build_full_address_html( $info );
		
		// Hero Section
		$content = sprintf(
			'<section class="buildfuly-hero-centered" style="padding: 100px 20px 80px; background: var(--buildfuly-color-background, %s); position: relative; overflow: hidden;">
				<div style="max-width: 900px; margin: 0 auto; text-align: center; position: relative; z-index: 1;">
					<div class="buildfuly-badge" style="display: inline-block; padding: 8px 16px; border-radius: 20px; margin-bottom: 24px; background: rgba(0, 102, 204, 0.1); border: 1px solid var(--buildfuly-color-primary, %s); color: var(--buildfuly-color-primary, %s);">
						<span style="font-size: 12px; font-weight: 700; text-transform: uppercase; letter-spacing: 0.08em;">Get In Touch</span>
					</div>
					<h1 class="buildfuly-heading" style="font-size: 48px; margin: 0 0 24px 0; font-weight: 800; line-height: 1.1; letter-spacing: -0.02em; color: var(--buildfuly-color-text, %s);">We\'d Love to Hear From You</h1>
					<p class="buildfuly-text" style="font-size: 20px; line-height: 1.6; margin: 0 auto; max-width: 600px; color: var(--buildfuly-color-text-light, %s);">Have questions about our services? Ready to start your project? Reach out and let\'s talk.</p>
				</div>
			</section>',
			esc_attr( $colors['bg'] ),
			esc_attr( $colors['primary'] ),
			esc_attr( $colors['primary'] ),
			esc_attr( $colors['text'] ),
			esc_attr( $colors['text_light'] )
		);

		// Contact Info Cards Section
		$content .= sprintf(
			'<section style="padding: 80px 20px; background: var(--buildfuly-color-bg-alt, %s);">
				<div style="max-width: 1100px; margin: 0 auto;">
					<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 24px;">',
			esc_attr( $colors['bg_alt'] )
		);

		// Phone Card
		if ( ! empty( $info['phone'] ) ) {
			$phone_clean = preg_replace( '/[^0-9+]/', '', $info['phone'] );
			$content .= sprintf(
				'<div style="background: var(--buildfuly-color-background, %s); padding: 40px 32px; border-radius: 16px; text-align: center; box-shadow: 0 4px 20px rgba(0,0,0,0.06);">
					<div style="width: 64px; height: 64px; background: rgba(0, 102, 204, 0.1); border-radius: 16px; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
						<svg width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="%s" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"></path></svg>
					</div>
					<h3 style="font-size: 20px; font-weight: 700; margin: 0 0 8px 0; color: var(--buildfuly-color-text, %s);">Phone</h3>
					<a href="tel:%s" style="font-size: 18px; color: var(--buildfuly-color-primary, %s); text-decoration: none; font-weight: 600;">%s</a>
				</div>',
				esc_attr( $colors['bg'] ),
				esc_attr( $colors['primary'] ),
				esc_attr( $colors['text'] ),
				esc_attr( $phone_clean ),
				esc_attr( $colors['primary'] ),
				esc_html( $info['phone'] )
			);
		}

		// Email Card
		if ( ! empty( $info['email'] ) ) {
			$content .= sprintf(
				'<div style="background: var(--buildfuly-color-background, %s); padding: 40px 32px; border-radius: 16px; text-align: center; box-shadow: 0 4px 20px rgba(0,0,0,0.06);">
					<div style="width: 64px; height: 64px; background: rgba(0, 102, 204, 0.1); border-radius: 16px; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
						<svg width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="%s" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"></path><polyline points="22,6 12,13 2,6"></polyline></svg>
					</div>
					<h3 style="font-size: 20px; font-weight: 700; margin: 0 0 8px 0; color: var(--buildfuly-color-text, %s);">Email</h3>
					<a href="mailto:%s" style="font-size: 18px; color: var(--buildfuly-color-primary, %s); text-decoration: none; font-weight: 600;">%s</a>
				</div>',
				esc_attr( $colors['bg'] ),
				esc_attr( $colors['primary'] ),
				esc_attr( $colors['text'] ),
				esc_attr( $info['email'] ),
				esc_attr( $colors['primary'] ),
				esc_html( $info['email'] )
			);
		}

		// Address Card
		if ( ! empty( $full_address ) ) {
			$maps_url = 'https://maps.google.com/?q=' . urlencode( strip_tags( str_replace( '<br>', ', ', $full_address ) ) );
			$content .= sprintf(
				'<div style="background: var(--buildfuly-color-background, %s); padding: 40px 32px; border-radius: 16px; text-align: center; box-shadow: 0 4px 20px rgba(0,0,0,0.06);">
					<div style="width: 64px; height: 64px; background: rgba(0, 102, 204, 0.1); border-radius: 16px; display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
						<svg width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="%s" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"></path><circle cx="12" cy="10" r="3"></circle></svg>
					</div>
					<h3 style="font-size: 20px; font-weight: 700; margin: 0 0 8px 0; color: var(--buildfuly-color-text, %s);">Location</h3>
					<a href="%s" target="_blank" style="font-size: 16px; color: var(--buildfuly-color-text-light, %s); text-decoration: none; line-height: 1.6;">%s</a>
				</div>',
				esc_attr( $colors['bg'] ),
				esc_attr( $colors['primary'] ),
				esc_attr( $colors['text'] ),
				esc_url( $maps_url ),
				esc_attr( $colors['text_light'] ),
				$full_address
			);
		}

		$content .= '</div></div></section>';

		// Business Hours Section
		$content .= sprintf(
			'<section style="padding: 80px 20px; background: var(--buildfuly-color-background, %s);">
				<div style="max-width: 600px; margin: 0 auto; text-align: center;">
					<h2 class="buildfuly-heading" style="font-size: 36px; margin: 0 0 32px 0; font-weight: 700; color: var(--buildfuly-color-text, %s);">Business Hours</h2>
					<div style="background: var(--buildfuly-color-bg-alt, %s); padding: 32px; border-radius: 16px;">
						<div style="display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid rgba(0,0,0,0.08);">
							<span style="color: var(--buildfuly-color-text, %s); font-weight: 600;">Monday - Friday</span>
							<span style="color: var(--buildfuly-color-text-light, %s);">9:00 AM - 5:00 PM</span>
						</div>
						<div style="display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid rgba(0,0,0,0.08);">
							<span style="color: var(--buildfuly-color-text, %s); font-weight: 600;">Saturday</span>
							<span style="color: var(--buildfuly-color-text-light, %s);">By Appointment</span>
						</div>
						<div style="display: flex; justify-content: space-between; padding: 12px 0;">
							<span style="color: var(--buildfuly-color-text, %s); font-weight: 600;">Sunday</span>
							<span style="color: var(--buildfuly-color-text-light, %s);">Closed</span>
						</div>
					</div>
				</div>
			</section>',
			esc_attr( $colors['bg'] ),
			esc_attr( $colors['text'] ),
			esc_attr( $colors['bg_alt'] ),
			esc_attr( $colors['text'] ),
			esc_attr( $colors['text_light'] ),
			esc_attr( $colors['text'] ),
			esc_attr( $colors['text_light'] ),
			esc_attr( $colors['text'] ),
			esc_attr( $colors['text_light'] )
		);

		// CTA Section
		$content .= sprintf(
			'<section class="buildfuly-cta-banner" style="padding: 0; background: transparent; position: relative; overflow: visible;">
				<div style="width: 100%%; padding: 80px 20px; background: var(--buildfuly-color-primary, %s); overflow: hidden; position: relative;">
					<div style="position: absolute; top: -100px; right: -100px; width: 300px; height: 300px; background: rgba(255, 255, 255, 0.08); border-radius: 50%%; pointer-events: none;"></div>
					<div style="position: absolute; bottom: -80px; left: -80px; width: 250px; height: 250px; background: rgba(255, 255, 255, 0.08); border-radius: 50%%; pointer-events: none;"></div>
					<div style="max-width: 900px; margin: 0 auto; text-align: center; position: relative; z-index: 1;">
						<h2 class="buildfuly-heading" style="font-size: 36px; margin: 0 0 20px 0; font-weight: 800; line-height: 1.2; color: %s;">Ready to Get Started?</h2>
						<p class="buildfuly-text" style="font-size: 18px; line-height: 1.6; margin: 0 auto 32px auto; color: %s; opacity: 0.95; max-width: 600px;">Contact us today and let\'s discuss how we can help you achieve your goals.</p>
						<div style="display: flex; flex-wrap: wrap; gap: 16px; justify-content: center;">',
			esc_attr( $colors['primary'] ),
			esc_attr( $colors['bg'] ),
			esc_attr( $colors['bg'] )
		);

		if ( ! empty( $info['phone'] ) ) {
			$phone_clean = preg_replace( '/[^0-9+]/', '', $info['phone'] );
			$content .= sprintf(
				'<a href="tel:%s" class="buildfuly-btn-primary" style="display: inline-flex; align-items: center; padding: 18px 40px; background: var(--buildfuly-color-background, %s); color: var(--buildfuly-color-primary, %s); text-decoration: none; border-radius: 10px; font-weight: 700; font-size: 17px; transition: all 0.3s ease; box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);">
					<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right: 10px;"><path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"></path></svg>
					Call Now
				</a>',
				esc_attr( $phone_clean ),
				esc_attr( $colors['bg'] ),
				esc_attr( $colors['primary'] )
			);
		}

		if ( ! empty( $info['email'] ) ) {
			$content .= sprintf(
				'<a href="mailto:%s" class="buildfuly-btn-secondary" style="display: inline-flex; align-items: center; padding: 18px 40px; background: rgba(255, 255, 255, 0.15); color: var(--buildfuly-color-background, %s); text-decoration: none; border-radius: 10px; font-weight: 700; font-size: 17px; transition: all 0.3s ease; border: 2px solid rgba(255, 255, 255, 0.6); backdrop-filter: blur(10px);">
					<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right: 10px;"><path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"></path><polyline points="22,6 12,13 2,6"></polyline></svg>
					Send Email
				</a>',
				esc_attr( $info['email'] ),
				esc_attr( $colors['bg'] )
			);
		}

		$content .= '</div></div></div></section>';

		return $content;
	}

	/**
	 * Generate Privacy Policy content with Buildfuly styling.
	 *
	 * @since    1.1.2
	 * @param    array $info Business info.
	 * @return   string HTML content.
	 */
	private static function generate_privacy_policy_content( $info ) {
		$colors = self::get_colors();
		$business_name = esc_html( $info['name'] );
		$website = esc_url( $info['website'] );
		$email = esc_html( $info['email'] );
		$date = date( 'F j, Y' );

		// Hero Section
		$content = sprintf(
			'<section class="buildfuly-hero-centered" style="padding: 100px 20px 60px; background: var(--buildfuly-color-background, %s); position: relative;">
				<div style="max-width: 800px; margin: 0 auto; text-align: center;">
					<h1 class="buildfuly-heading" style="font-size: 48px; margin: 0 0 16px 0; font-weight: 800; line-height: 1.1; color: var(--buildfuly-color-text, %s);">Privacy Policy</h1>
					<p class="buildfuly-text" style="font-size: 16px; color: var(--buildfuly-color-text-light, %s);">Last updated: %s</p>
				</div>
			</section>',
			esc_attr( $colors['bg'] ),
			esc_attr( $colors['text'] ),
			esc_attr( $colors['text_light'] ),
			esc_html( $date )
		);

		// Content Section
		$content .= sprintf(
			'<section style="padding: 60px 20px 100px; background: var(--buildfuly-color-background, %s);">
				<div style="max-width: 800px; margin: 0 auto;">
					<div style="background: var(--buildfuly-color-bg-alt, %s); padding: 48px; border-radius: 24px; box-shadow: 0 4px 20px rgba(0,0,0,0.04);">',
			esc_attr( $colors['bg'] ),
			esc_attr( $colors['bg_alt'] )
		);

		$content .= self::generate_legal_section( 'Introduction', sprintf(
			'%s ("we", "us", or "our") operates %s (the "Site"). This page informs you of our policies regarding the collection, use, and disclosure of Personal Information we receive from users of the Site.',
			$business_name, $website
		), $colors );

		$content .= self::generate_legal_section( 'Information Collection and Use', 
			'While using our Site, we may ask you to provide us with certain personally identifiable information that can be used to contact or identify you. Personally identifiable information may include, but is not limited to:<br><br>
			• Name<br>
			• Email address<br>
			• Phone number<br>
			• Address',
			$colors
		);

		$content .= self::generate_legal_section( 'Log Data',
			'Like many site operators, we collect information that your browser sends whenever you visit our Site ("Log Data"). This Log Data may include information such as your computer\'s Internet Protocol ("IP") address, browser type, browser version, the pages of our Site that you visit, the time and date of your visit, the time spent on those pages, and other statistics.',
			$colors
		);

		$content .= self::generate_legal_section( 'Cookies',
			'Cookies are files with a small amount of data, which may include an anonymous unique identifier. Cookies are sent to your browser from a website and stored on your computer\'s hard drive. Like many sites, we use "cookies" to collect information. You can instruct your browser to refuse all cookies or to indicate when a cookie is being sent.',
			$colors
		);

		$content .= self::generate_legal_section( 'Security',
			'The security of your Personal Information is important to us, but remember that no method of transmission over the Internet, or method of electronic storage, is 100% secure. While we strive to use commercially acceptable means to protect your Personal Information, we cannot guarantee its absolute security.',
			$colors
		);

		$content .= self::generate_legal_section( 'Third-Party Services',
			'We may employ third-party companies and individuals to facilitate our Service, provide the Service on our behalf, perform Service-related tasks, or assist us in analyzing how our Service is used. These third parties have access to your Personal Information only to perform these tasks on our behalf and are obligated not to disclose or use it for any other purpose.',
			$colors
		);

		$content .= self::generate_legal_section( 'Children\'s Privacy',
			'Our Site does not address anyone under the age of 13. We do not knowingly collect personally identifiable information from children under 13. If you are a parent or guardian and you are aware that your child has provided us with Personal Information, please contact us.',
			$colors
		);

		$content .= self::generate_legal_section( 'Changes to This Privacy Policy',
			'We may update our Privacy Policy from time to time. We will notify you of any changes by posting the new Privacy Policy on this page. You are advised to review this Privacy Policy periodically for any changes.',
			$colors
		);

		// Contact Section
		$contact_text = sprintf( 'If you have any questions about this Privacy Policy, please contact us at <strong>%s</strong>', $business_name );
		if ( ! empty( $email ) ) {
			$contact_text .= sprintf( ' via email at <a href="mailto:%s" style="color: var(--buildfuly-color-primary, %s);">%s</a>', $email, $colors['primary'], $email );
		}
		if ( ! empty( $info['phone'] ) ) {
			$contact_text .= sprintf( ' or by phone at <a href="tel:%s" style="color: var(--buildfuly-color-primary, %s);">%s</a>', preg_replace( '/[^0-9+]/', '', $info['phone'] ), $colors['primary'], esc_html( $info['phone'] ) );
		}
		$contact_text .= '.';

		$content .= self::generate_legal_section( 'Contact Us', $contact_text, $colors, true );

		$content .= '</div></div></section>';

		return $content;
	}

	/**
	 * Generate Terms of Service content with Buildfuly styling.
	 *
	 * @since    1.1.2
	 * @param    array $info Business info.
	 * @return   string HTML content.
	 */
	private static function generate_terms_of_service_content( $info ) {
		$colors = self::get_colors();
		$business_name = esc_html( $info['name'] );
		$website = esc_url( $info['website'] );
		$email = esc_html( $info['email'] );
		$date = date( 'F j, Y' );

		// Hero Section
		$content = sprintf(
			'<section class="buildfuly-hero-centered" style="padding: 100px 20px 60px; background: var(--buildfuly-color-background, %s); position: relative;">
				<div style="max-width: 800px; margin: 0 auto; text-align: center;">
					<h1 class="buildfuly-heading" style="font-size: 48px; margin: 0 0 16px 0; font-weight: 800; line-height: 1.1; color: var(--buildfuly-color-text, %s);">Terms of Service</h1>
					<p class="buildfuly-text" style="font-size: 16px; color: var(--buildfuly-color-text-light, %s);">Last updated: %s</p>
				</div>
			</section>',
			esc_attr( $colors['bg'] ),
			esc_attr( $colors['text'] ),
			esc_attr( $colors['text_light'] ),
			esc_html( $date )
		);

		// Content Section
		$content .= sprintf(
			'<section style="padding: 60px 20px 100px; background: var(--buildfuly-color-background, %s);">
				<div style="max-width: 800px; margin: 0 auto;">
					<div style="background: var(--buildfuly-color-bg-alt, %s); padding: 48px; border-radius: 24px; box-shadow: 0 4px 20px rgba(0,0,0,0.04);">',
			esc_attr( $colors['bg'] ),
			esc_attr( $colors['bg_alt'] )
		);

		$content .= self::generate_legal_section( 'Agreement to Terms', sprintf(
			'Please read these Terms of Service ("Terms") carefully before using %s (the "Service") operated by %s ("us", "we", or "our"). By accessing or using our Service, you agree to be bound by these Terms. If you disagree with any part of the terms, then you may not access the Service.',
			$website, $business_name
		), $colors );

		$content .= self::generate_legal_section( 'Intellectual Property', sprintf(
			'The Service and its original content, features, and functionality are and will remain the exclusive property of %s and its licensors. The Service is protected by copyright, trademark, and other laws of both the United States and foreign countries.',
			$business_name
		), $colors );

		$content .= self::generate_legal_section( 'User Responsibilities',
			'When using our Service, you agree to:<br><br>
			• Provide accurate and complete information<br>
			• Maintain the security of your account<br>
			• Not use the Service for any illegal or unauthorized purpose<br>
			• Not violate any laws in your jurisdiction<br>
			• Not transmit any harmful code or interfere with the Service',
			$colors
		);

		$content .= self::generate_legal_section( 'Limitation of Liability', sprintf(
			'In no event shall %s, nor its directors, employees, partners, agents, suppliers, or affiliates, be liable for any indirect, incidental, special, consequential, or punitive damages, including without limitation, loss of profits, data, use, goodwill, or other intangible losses, resulting from your access to or use of or inability to access or use the Service.',
			$business_name
		), $colors );

		$content .= self::generate_legal_section( 'Disclaimer',
			'Your use of the Service is at your sole risk. The Service is provided on an "AS IS" and "AS AVAILABLE" basis. The Service is provided without warranties of any kind, whether express or implied, including, but not limited to, implied warranties of merchantability, fitness for a particular purpose, non-infringement, or course of performance.',
			$colors
		);

		$content .= self::generate_legal_section( 'Governing Law',
			'These Terms shall be governed and construed in accordance with the laws of the United States, without regard to its conflict of law provisions. Our failure to enforce any right or provision of these Terms will not be considered a waiver of those rights.',
			$colors
		);

		$content .= self::generate_legal_section( 'Changes to Terms',
			'We reserve the right, at our sole discretion, to modify or replace these Terms at any time. If a revision is material, we will try to provide at least 30 days\' notice prior to any new terms taking effect.',
			$colors
		);

		// Contact Section
		$contact_text = sprintf( 'If you have any questions about these Terms, please contact us at <strong>%s</strong>', $business_name );
		if ( ! empty( $email ) ) {
			$contact_text .= sprintf( ' via email at <a href="mailto:%s" style="color: var(--buildfuly-color-primary, %s);">%s</a>', $email, $colors['primary'], $email );
		}
		if ( ! empty( $info['phone'] ) ) {
			$contact_text .= sprintf( ' or by phone at <a href="tel:%s" style="color: var(--buildfuly-color-primary, %s);">%s</a>', preg_replace( '/[^0-9+]/', '', $info['phone'] ), $colors['primary'], esc_html( $info['phone'] ) );
		}
		$contact_text .= '.';

		$content .= self::generate_legal_section( 'Contact Us', $contact_text, $colors, true );

		$content .= '</div></div></section>';

		return $content;
	}

	/**
	 * Generate a styled legal section.
	 *
	 * @since    1.1.2
	 * @param    string $title Section title.
	 * @param    string $content Section content.
	 * @param    array  $colors Colors array.
	 * @param    bool   $is_last Whether this is the last section.
	 * @return   string HTML content.
	 */
	private static function generate_legal_section( $title, $content, $colors, $is_last = false ) {
		$border = $is_last ? '' : 'border-bottom: 1px solid rgba(0,0,0,0.08);';
		$margin = $is_last ? '' : 'margin-bottom: 32px;';
		$padding = $is_last ? '' : 'padding-bottom: 32px;';
		
		return sprintf(
			'<div style="%s %s %s">
				<h2 style="font-size: 24px; font-weight: 700; margin: 0 0 16px 0; color: var(--buildfuly-color-text, %s);">%s</h2>
				<p style="font-size: 16px; line-height: 1.8; margin: 0; color: var(--buildfuly-color-text-light, %s);">%s</p>
			</div>',
			$border,
			$margin,
			$padding,
			esc_attr( $colors['text'] ),
			esc_html( $title ),
			esc_attr( $colors['text_light'] ),
			$content
		);
	}

	/**
	 * Get all legal page IDs.
	 *
	 * @since    1.1.2
	 * @return   array Array of page type => page ID.
	 */
	public static function get_legal_page_ids() {
		$ids = array();
		
		foreach ( array_keys( self::$page_slugs ) as $page_type ) {
			$id = self::find_existing_page( $page_type );
			if ( $id ) {
				$ids[ $page_type ] = $id;
			}
		}
		
		return $ids;
	}

	/**
	 * Check if a page is auto-generated legal page.
	 *
	 * @since    1.1.2
	 * @param    int $page_id Page ID.
	 * @return   bool
	 */
	public static function is_legal_page( $page_id ) {
		return (bool) get_post_meta( $page_id, '_buildfuly_auto_generated', true );
	}
}
