<?php
/**
 * Lead Capture Shortcode
 *
 * Usage: [buildfuly_capture calculator="Driveway Calculator" button="Get My Estimate"]
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Configure WordPress to use SMTP for sending emails
 * Settings are stored in the database under 'buildfuly_smtp_settings'
 */
function buildfuly_configure_smtp( $phpmailer ) {
	$smtp = get_option( 'buildfuly_smtp_settings', array() );
	
	// Only configure if SMTP is enabled and has settings
	if ( empty( $smtp['enabled'] ) || empty( $smtp['host'] ) ) {
		return;
	}
	
	$phpmailer->isSMTP();
	$phpmailer->Host       = $smtp['host'];
	$phpmailer->Port       = $smtp['port'] ?? 587;
	$phpmailer->SMTPSecure = $smtp['encryption'] ?? 'tls'; // 'tls' or 'ssl'
	$phpmailer->SMTPAuth   = true;
	$phpmailer->Username   = $smtp['username'];
	$phpmailer->Password   = $smtp['password'];
	
	// ALWAYS use the configured from_email - override whatever WordPress set
	$from_email = $smtp['from_email'] ?? '';
	if ( ! empty( $from_email ) && is_email( $from_email ) ) {
		$phpmailer->From = $from_email;
		// Also clear and reset the Sender to match
		$phpmailer->Sender = $from_email;
	}
	
	$from_name = $smtp['from_name'] ?? '';
	if ( ! empty( $from_name ) ) {
		$phpmailer->FromName = $from_name;
	}
	
	// Debug mode - capture output for display
	if ( ! empty( $smtp['debug'] ) ) {
		global $buildfuly_smtp_debug_log;
		$buildfuly_smtp_debug_log = array();
		
		$phpmailer->SMTPDebug = 2;
		$phpmailer->Debugoutput = function( $str, $level ) {
			global $buildfuly_smtp_debug_log;
			$buildfuly_smtp_debug_log[] = $str;
			error_log( "SMTP Debug: $str" );
		};
	}
}
add_action( 'phpmailer_init', 'buildfuly_configure_smtp' );

/**
 * Send webhook notification
 * 
 * @param string $url Webhook URL
 * @param array $data Data to send
 * @param array $settings Webhook settings (headers, secret, etc.)
 */
function buildfuly_send_webhook( $url, $data, $settings = array() ) {
	// Prepare headers
	$headers = array(
		'Content-Type' => 'application/json',
		'User-Agent'   => 'Buildfuly-Webhook/1.0',
	);
	
	// Add custom headers if configured
	if ( ! empty( $settings['headers'] ) ) {
		$custom_headers = explode( "\n", $settings['headers'] );
		foreach ( $custom_headers as $header ) {
			if ( strpos( $header, ':' ) !== false ) {
				list( $key, $value ) = explode( ':', $header, 2 );
				$headers[ trim( $key ) ] = trim( $value );
			}
		}
	}
	
	// Add secret/signature if configured
	if ( ! empty( $settings['secret'] ) ) {
		$signature = hash_hmac( 'sha256', wp_json_encode( $data ), $settings['secret'] );
		$headers['X-Buildfuly-Signature'] = $signature;
	}
	
	// Add timestamp
	$data['webhook_timestamp'] = time();
	
	// Send webhook asynchronously (non-blocking)
	$args = array(
		'body'        => wp_json_encode( $data ),
		'headers'     => $headers,
		'timeout'     => 5, // Quick timeout so it doesn't block form submission
		'blocking'    => false, // Non-blocking request
		'sslverify'   => true,
	);
	
	$response = wp_remote_post( $url, $args );
	
	// Log errors for debugging
	if ( is_wp_error( $response ) && ! empty( $settings['debug'] ) ) {
		error_log( 'Buildfuly Webhook Error: ' . $response->get_error_message() );
	}
	
	return $response;
}

/**
 * Get email settings with sensible defaults (emails enabled by default)
 */
function buildfuly_get_email_settings() {
	$defaults = array(
		'send_to_admin' => true,  // Enabled by default
		'admin_email'   => get_option( 'admin_email' ),
		'send_to_lead'  => true,  // Enabled by default
		'email_subject' => 'Your Estimate from ' . get_bloginfo( 'name' ),
		'email_template' => "Hi {name},\n\nThank you for your interest! Here's your estimate:\n\n{estimate_details}\n\nTotal: {estimate_total}\n\nWe'll be in touch soon!\n\nBest regards,\n" . get_bloginfo( 'name' ),
	);
	
	$settings = get_option( 'buildfuly_lead_email_settings', array() );
	
	// Merge with defaults (settings override defaults)
	return array_merge( $defaults, $settings );
}

/**
 * Run database migration on admin init (once per version)
 */
function buildfuly_maybe_migrate_leads_table() {
	$migrated_version = get_option( 'buildfuly_leads_table_version', '0' );
	$current_version = '1.1'; // Increment this when schema changes
	
	if ( version_compare( $migrated_version, $current_version, '<' ) ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_leads';
		
		// Check if table exists
		$table_exists = $wpdb->get_var( "SHOW TABLES LIKE '$table_name'" );
		if ( $table_exists ) {
			// Get current columns
			$columns = $wpdb->get_results( "SHOW COLUMNS FROM $table_name" );
			$column_names = array_map( function( $col ) { return $col->Field; }, $columns );
			
			// Migrate calculator_name to source
			if ( in_array( 'calculator_name', $column_names ) && ! in_array( 'source', $column_names ) ) {
				$wpdb->query( "ALTER TABLE $table_name CHANGE `calculator_name` `source` varchar(255)" );
			}
			
			// Migrate estimate_breakdown to estimate_details  
			if ( in_array( 'estimate_breakdown', $column_names ) && ! in_array( 'estimate_details', $column_names ) ) {
				$wpdb->query( "ALTER TABLE $table_name CHANGE `estimate_breakdown` `estimate_details` text" );
			}
			
			// Add source column if missing
			if ( ! in_array( 'source', $column_names ) && ! in_array( 'calculator_name', $column_names ) ) {
				$wpdb->query( "ALTER TABLE $table_name ADD `source` varchar(255) AFTER `phone`" );
			}
			
			// Add estimate_details column if missing
			if ( ! in_array( 'estimate_details', $column_names ) && ! in_array( 'estimate_breakdown', $column_names ) ) {
				$wpdb->query( "ALTER TABLE $table_name ADD `estimate_details` text AFTER `estimate_total`" );
			}
		}
		
		update_option( 'buildfuly_leads_table_version', $current_version );
	}
}
add_action( 'admin_init', 'buildfuly_maybe_migrate_leads_table' );

/**
 * Register the lead capture shortcode
 */
function buildfuly_capture_shortcode( $atts ) {
	$atts = shortcode_atts( array(
		'calculator'  => 'Calculator',
		'button'      => 'Get My Estimate',
		'show_name'   => 'true',
		'show_phone'  => 'false',
		'success'     => 'Check your email for your estimate!',
	), $atts, 'buildfuly_capture' );
	
	$show_name = filter_var( $atts['show_name'], FILTER_VALIDATE_BOOLEAN );
	$show_phone = filter_var( $atts['show_phone'], FILTER_VALIDATE_BOOLEAN );
	
	// Generate unique ID for this form instance
	$form_id = 'buildfuly-capture-' . wp_rand( 1000, 9999 );
	
	ob_start();
	?>
	<div class="buildfuly-capture-form" id="<?php echo esc_attr( $form_id ); ?>">
		<form class="buildfuly-lead-form" data-calculator="<?php echo esc_attr( $atts['calculator'] ); ?>">
			<?php wp_nonce_field( 'buildfuly_capture_lead', 'buildfuly_lead_nonce' ); ?>
			
			<!-- Hidden fields for estimate data -->
			<input type="hidden" id="buildfuly-estimate-total" name="estimate_total" value="0">
			<input type="hidden" id="buildfuly-estimate-breakdown" name="estimate_breakdown" value="">
			<input type="hidden" name="calculator_name" value="<?php echo esc_attr( $atts['calculator'] ); ?>">
			
			<div class="capture-fields">
				<?php if ( $show_name ) : ?>
				<div class="capture-field">
					<input type="text" name="name" placeholder="Your Name" class="buildfuly-input">
				</div>
				<?php endif; ?>
				
				<div class="capture-field">
					<input type="email" name="email" placeholder="Your Email" required class="buildfuly-input">
				</div>
				
				<?php if ( $show_phone ) : ?>
				<div class="capture-field">
					<input type="tel" name="phone" placeholder="Phone (optional)" class="buildfuly-input">
				</div>
				<?php endif; ?>
				
				<div class="capture-field">
					<button type="submit" class="buildfuly-capture-btn">
						<span class="btn-text"><?php echo esc_html( $atts['button'] ); ?></span>
						<span class="btn-loading" style="display:none;">
							<svg width="20" height="20" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
								<circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="3" fill="none" stroke-dasharray="31.4 31.4" transform="rotate(-90 12 12)">
									<animateTransform attributeName="transform" type="rotate" from="0 12 12" to="360 12 12" dur="1s" repeatCount="indefinite"/>
								</circle>
							</svg>
						</span>
					</button>
				</div>
			</div>
			
			<div class="capture-success" style="display:none;">
				<div class="success-icon">✓</div>
				<div class="success-message"><?php echo esc_html( $atts['success'] ); ?></div>
			</div>
			
			<div class="capture-error" style="display:none;"></div>
		</form>
	</div>
	
	<style>
	.buildfuly-capture-form {
		margin: 20px 0;
	}
	
	.buildfuly-lead-form .capture-fields {
		display: flex;
		flex-wrap: wrap;
		gap: 10px;
		align-items: stretch;
	}
	
	.buildfuly-lead-form .capture-field {
		flex: 1;
		min-width: 150px;
	}
	
	.buildfuly-lead-form .buildfuly-input {
		width: 100%;
		padding: 14px 16px;
		border: 2px solid #e5e7eb;
		border-radius: 10px;
		font-size: 16px;
		transition: border-color 0.2s, box-shadow 0.2s;
		box-sizing: border-box;
	}
	
	.buildfuly-lead-form .buildfuly-input:focus {
		outline: none;
		border-color: #3b82f6;
		box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
	}
	
	.buildfuly-capture-btn {
		width: 100%;
		padding: 14px 24px;
		background: linear-gradient(135deg, #3b82f6, #2563eb);
		color: #fff;
		border: none;
		border-radius: 10px;
		font-size: 16px;
		font-weight: 600;
		cursor: pointer;
		transition: transform 0.2s, box-shadow 0.2s;
		display: flex;
		align-items: center;
		justify-content: center;
		gap: 8px;
	}
	
	.buildfuly-capture-btn:hover {
		transform: translateY(-2px);
		box-shadow: 0 4px 12px rgba(59, 130, 246, 0.4);
	}
	
	.buildfuly-capture-btn:disabled {
		opacity: 0.7;
		cursor: not-allowed;
		transform: none;
	}
	
	.capture-success {
		text-align: center;
		padding: 30px;
		background: #ecfdf5;
		border-radius: 12px;
		animation: fadeIn 0.3s ease;
	}
	
	.capture-success .success-icon {
		width: 50px;
		height: 50px;
		background: #10b981;
		color: #fff;
		border-radius: 50%;
		display: inline-flex;
		align-items: center;
		justify-content: center;
		font-size: 24px;
		margin-bottom: 12px;
	}
	
	.capture-success .success-message {
		color: #065f46;
		font-size: 16px;
		font-weight: 500;
	}
	
	.capture-error {
		padding: 12px 16px;
		background: #fef2f2;
		color: #dc2626;
		border-radius: 8px;
		margin-top: 10px;
		font-size: 14px;
	}
	
	@keyframes fadeIn {
		from { opacity: 0; transform: translateY(-10px); }
		to { opacity: 1; transform: translateY(0); }
	}
	
	@media (max-width: 600px) {
		.buildfuly-lead-form .capture-fields {
			flex-direction: column;
		}
		
		.buildfuly-lead-form .capture-field {
			min-width: 100%;
		}
	}
	</style>
	
	<script>
	(function() {
		const form = document.querySelector('#<?php echo esc_js( $form_id ); ?> form');
		if (!form) return;
		
		form.addEventListener('submit', function(e) {
			e.preventDefault();
			
			const btn = form.querySelector('.buildfuly-capture-btn');
			const btnText = btn.querySelector('.btn-text');
			const btnLoading = btn.querySelector('.btn-loading');
			const successDiv = form.querySelector('.capture-success');
			const errorDiv = form.querySelector('.capture-error');
			const fieldsDiv = form.querySelector('.capture-fields');
			
			// Get form data
			const formData = new FormData(form);
			formData.append('action', 'buildfuly_capture_lead');
			
			// Show loading
			btn.disabled = true;
			btnText.style.display = 'none';
			btnLoading.style.display = 'block';
			errorDiv.style.display = 'none';
			
			// Submit
			fetch('<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>', {
				method: 'POST',
				body: formData
			})
			.then(response => response.json())
			.then(data => {
				btn.disabled = false;
				btnText.style.display = 'block';
				btnLoading.style.display = 'none';
				
				if (data.success) {
					fieldsDiv.style.display = 'none';
					successDiv.style.display = 'block';
				} else {
					errorDiv.textContent = data.data?.message || 'Something went wrong. Please try again.';
					errorDiv.style.display = 'block';
				}
			})
			.catch(error => {
				btn.disabled = false;
				btnText.style.display = 'block';
				btnLoading.style.display = 'none';
				errorDiv.textContent = 'Connection error. Please try again.';
				errorDiv.style.display = 'block';
			});
		});
	})();
	</script>
	<?php
	return ob_get_clean();
}
add_shortcode( 'buildfuly_capture', 'buildfuly_capture_shortcode' );

/**
 * AJAX handler for lead capture
 */
function buildfuly_ajax_capture_lead() {
	// Verify nonce
	if ( ! wp_verify_nonce( $_POST['buildfuly_lead_nonce'] ?? '', 'buildfuly_capture_lead' ) ) {
		wp_send_json_error( array( 'message' => 'Security check failed.' ) );
	}
	
	// Validate email
	$email = sanitize_email( $_POST['email'] ?? '' );
	if ( ! is_email( $email ) ) {
		wp_send_json_error( array( 'message' => 'Please enter a valid email address.' ) );
	}
	
	// Sanitize inputs
	$name = sanitize_text_field( $_POST['name'] ?? '' );
	$phone = sanitize_text_field( $_POST['phone'] ?? '' );
	$calculator_name = sanitize_text_field( $_POST['calculator_name'] ?? 'Calculator' );
	$estimate_total = floatval( $_POST['estimate_total'] ?? 0 );
	$estimate_breakdown = sanitize_text_field( $_POST['estimate_breakdown'] ?? '' );
	
	// Save to database
	global $wpdb;
	$table_name = $wpdb->prefix . 'buildfuly_leads';
	
	// Auto-create leads table if it doesn't exist
	$table_exists = $wpdb->get_var( "SHOW TABLES LIKE '$table_name'" ) === $table_name;
	if ( ! $table_exists ) {
		$charset_collate = $wpdb->get_charset_collate();
		$sql = "CREATE TABLE $table_name (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			email varchar(255),
			name varchar(255),
			phone varchar(50),
			source varchar(255),
			estimate_total decimal(10,2) DEFAULT 0,
			estimate_details text,
			form_data longtext,
			status varchar(50) DEFAULT 'new',
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id)
		) $charset_collate;";
		
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql );
	}
	
	$result = $wpdb->insert(
		$table_name,
		array(
			'email'            => $email,
			'name'             => $name,
			'phone'            => $phone,
			'source'           => $calculator_name,
			'estimate_total'   => $estimate_total,
			'estimate_details' => $estimate_breakdown,
			'form_data'        => wp_json_encode( $_POST ),
			'status'           => 'new',
			'created_at'       => current_time( 'mysql' ),
		),
		array( '%s', '%s', '%s', '%s', '%f', '%s', '%s', '%s', '%s' )
	);
	
	if ( $result === false ) {
		error_log( 'Buildfuly lead insert error: ' . $wpdb->last_error );
		wp_send_json_error( array( 'message' => 'Database error: ' . $wpdb->last_error ) );
	}
	
	// Get email settings (with defaults - emails enabled by default)
	$settings = buildfuly_get_email_settings();
	$site_name = get_bloginfo( 'name' );
	
	// Send admin notification
	if ( ! empty( $settings['send_to_admin'] ) ) {
		$admin_email = $settings['admin_email'] ?? get_option( 'admin_email' );
		$admin_subject = "New Lead: $calculator_name - \$" . number_format( $estimate_total, 0 );
		$admin_message = "
			<h2>New Lead from $calculator_name</h2>
			<p><strong>Email:</strong> $email</p>
			<p><strong>Name:</strong> " . ( $name ?: 'Not provided' ) . "</p>
			<p><strong>Phone:</strong> " . ( $phone ?: 'Not provided' ) . "</p>
			<p><strong>Estimate:</strong> \$" . number_format( $estimate_total, 0 ) . "</p>
			<p><strong>Breakdown:</strong> $estimate_breakdown</p>
			<p><a href='" . admin_url( 'admin.php?page=buildfuly-leads' ) . "'>View in Dashboard</a></p>
		";
		
		$headers = array( 'Content-Type: text/html; charset=UTF-8' );
		$mail_sent = wp_mail( $admin_email, $admin_subject, $admin_message, $headers );
		if ( ! $mail_sent ) {
			error_log( 'Buildfuly: Failed to send admin notification email to ' . $admin_email );
		}
	}
	
	// Send auto-reply email to lead
	if ( ! empty( $settings['send_to_lead'] ) && $email ) {
		// Replace variables in subject
		$subject = $settings['email_subject'] ?? 'Your Estimate from {site_name}';
		$subject = str_replace(
			array( '{site_name}', '{name}', '{estimate_total}' ),
			array( $site_name, $name ?: 'there', '$' . number_format( $estimate_total, 0 ) ),
			$subject
		);
		
		// Replace variables in template
		$template = $settings['email_template'] ?? "Hi {name},\n\nThank you for your interest! Here's your estimate:\n\nTotal: {estimate_total}\n\nWe'll be in touch soon.\n\nBest regards,\n{site_name}";
		$message = str_replace(
			array( '{name}', '{email}', '{estimate_total}', '{estimate_breakdown}', '{site_name}', '{calculator_name}' ),
			array( $name ?: 'there', $email, '$' . number_format( $estimate_total, 0 ), $estimate_breakdown, $site_name, $calculator_name ),
			$template
		);
		
		// Set headers
		$from_name = $settings['from_name'] ?? $site_name;
		$from_email = $settings['from_email'] ?? get_option( 'admin_email' );
		$headers = array(
			'Content-Type: text/html; charset=UTF-8',
			"From: $from_name <$from_email>",
		);
		
		// Convert newlines to HTML and send
		$html_message = nl2br( $message );
		$mail_sent = wp_mail( $email, $subject, $html_message, $headers );
		if ( ! $mail_sent ) {
			error_log( 'Buildfuly: Failed to send auto-reply email to ' . $email );
		}
	}
	
	// Send webhook notification if configured
	$webhook_settings = get_option( 'buildfuly_webhook_settings', array() );
	if ( ! empty( $webhook_settings['enabled'] ) && ! empty( $webhook_settings['url'] ) ) {
		buildfuly_send_webhook( $webhook_settings['url'], array(
			'event'           => 'lead.created',
			'lead_id'         => $wpdb->insert_id,
			'email'           => $email,
			'name'            => $name,
			'phone'           => $phone,
			'source'          => $calculator_name,
			'estimate_total'  => $estimate_total,
			'estimate_details'=> $estimate_breakdown,
			'created_at'      => current_time( 'mysql' ),
			'site_name'       => $site_name,
			'site_url'        => home_url(),
		), $webhook_settings );
	}
	
	wp_send_json_success( array( 'message' => 'Lead captured successfully!' ) );
}
add_action( 'wp_ajax_buildfuly_capture_lead', 'buildfuly_ajax_capture_lead' );
add_action( 'wp_ajax_nopriv_buildfuly_capture_lead', 'buildfuly_ajax_capture_lead' );

/**
 * AJAX handler for saving email settings
 */
function buildfuly_ajax_save_lead_settings() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => 'Permission denied.' ) );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_lead_settings' ) ) {
		wp_send_json_error( array( 'message' => 'Security check failed.' ) );
	}
	
	$settings = array(
		'enabled'      => ! empty( $_POST['email_enabled'] ),
		'subject'      => sanitize_text_field( $_POST['email_subject'] ?? '' ),
		'from_name'    => sanitize_text_field( $_POST['from_name'] ?? '' ),
		'template'     => wp_kses_post( $_POST['email_template'] ?? '' ),
		'notify_admin' => ! empty( $_POST['notify_admin'] ),
		'admin_email'  => sanitize_email( $_POST['admin_email'] ?? '' ),
	);
	
	update_option( 'buildfuly_lead_email_settings', $settings );
	
	wp_send_json_success();
}
add_action( 'wp_ajax_buildfuly_save_lead_settings', 'buildfuly_ajax_save_lead_settings' );

/**
 * AJAX handler for saving email settings (updated)
 */
function buildfuly_ajax_save_email_settings() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	$settings = array(
		'send_to_admin' => ! empty( $_POST['send_to_admin'] ),
		'admin_email'   => sanitize_email( $_POST['admin_email'] ?? get_option( 'admin_email' ) ),
		'send_to_lead'  => ! empty( $_POST['send_to_lead'] ),
		'email_subject' => sanitize_text_field( $_POST['email_subject'] ?? '' ),
		'email_template'=> wp_kses_post( $_POST['email_template'] ?? '' ),
	);
	
	update_option( 'buildfuly_lead_email_settings', $settings );
	
	wp_send_json_success();
}
add_action( 'wp_ajax_buildfuly_save_email_settings', 'buildfuly_ajax_save_email_settings' );

/**
 * AJAX handler for saving SMTP settings
 */
function buildfuly_ajax_save_smtp_settings() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	$settings = array(
		'enabled'    => ! empty( $_POST['smtp_enabled'] ),
		'host'       => sanitize_text_field( $_POST['smtp_host'] ?? '' ),
		'port'       => intval( $_POST['smtp_port'] ?? 587 ),
		'encryption' => sanitize_text_field( $_POST['smtp_encryption'] ?? 'tls' ),
		'username'   => sanitize_text_field( $_POST['smtp_username'] ?? '' ),
		'password'   => $_POST['smtp_password'] ?? '', // Don't sanitize password
		'from_email' => sanitize_email( $_POST['smtp_from_email'] ?? '' ),
		'from_name'  => sanitize_text_field( $_POST['smtp_from_name'] ?? '' ),
		'debug'      => ! empty( $_POST['smtp_debug'] ),
	);
	
	update_option( 'buildfuly_smtp_settings', $settings );
	
	wp_send_json_success();
}
add_action( 'wp_ajax_buildfuly_save_smtp_settings', 'buildfuly_ajax_save_smtp_settings' );

/**
 * AJAX handler for testing SMTP
 */
function buildfuly_ajax_test_smtp() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	// Get SMTP settings
	$smtp = get_option( 'buildfuly_smtp_settings', array() );
	
	// Check if SMTP is properly configured
	if ( empty( $smtp['enabled'] ) ) {
		wp_send_json_error( 'SMTP is not enabled. Please enable SMTP and fill in all required fields.' );
	}
	
	if ( empty( $smtp['host'] ) || empty( $smtp['username'] ) || empty( $smtp['password'] ) ) {
		wp_send_json_error( 'SMTP settings incomplete. Please fill in Host, Username, and Password.' );
	}
	
	// Validate from email
	$from_email = $smtp['from_email'] ?? '';
	if ( empty( $from_email ) || ! is_email( $from_email ) || strpos( $from_email, 'localhost' ) !== false ) {
		// Try to use SMTP username as from email
		if ( is_email( $smtp['username'] ) ) {
			$from_email = $smtp['username'];
		} else {
			wp_send_json_error( 'Invalid "From Email" address. Please enter a valid email address in the SMTP settings (not localhost).' );
		}
	}
	
	$admin_email = get_option( 'admin_email' );
	
	// Validate admin email (recipient)
	if ( empty( $admin_email ) || ! is_email( $admin_email ) || strpos( $admin_email, 'localhost' ) !== false ) {
		wp_send_json_error( 'Invalid admin email address. Please update your WordPress admin email in Settings > General.' );
	}
	
	$site_name = get_bloginfo( 'name' );
	$from_name = $smtp['from_name'] ?? $site_name;
	
	$subject = "Test Email from $site_name - Buildfuly";
	$message = "
		<h2>SMTP Test Successful!</h2>
		<p>If you're reading this, your SMTP settings are working correctly.</p>
		<p>Sent at: " . current_time( 'mysql' ) . "</p>
		<p>From: $from_name &lt;$from_email&gt;</p>
		<p>To: $admin_email</p>
	";
	
	$headers = array(
		'Content-Type: text/html; charset=UTF-8',
		"From: $from_name <$from_email>",
	);
	
	$sent = wp_mail( $admin_email, $subject, $message, $headers );
	
	// Get debug log if available
	global $buildfuly_smtp_debug_log;
	$debug_log = ! empty( $buildfuly_smtp_debug_log ) ? implode( "\n", $buildfuly_smtp_debug_log ) : '';
	
	if ( $sent ) {
		$response = array( 'email' => $admin_email );
		if ( ! empty( $debug_log ) ) {
			$response['debug_log'] = $debug_log;
		}
		wp_send_json_success( $response );
	} else {
		global $phpmailer;
		$error = 'Unknown error';
		if ( isset( $phpmailer ) && is_object( $phpmailer ) && ! empty( $phpmailer->ErrorInfo ) ) {
			$error = $phpmailer->ErrorInfo;
		}
		if ( ! empty( $debug_log ) ) {
			$error .= "\n\nDebug Log:\n" . $debug_log;
		}
		wp_send_json_error( $error );
	}
}
add_action( 'wp_ajax_buildfuly_test_smtp', 'buildfuly_ajax_test_smtp' );

/**
 * AJAX handler for saving webhook settings
 */
function buildfuly_ajax_save_webhook_settings() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	$settings = array(
		'enabled' => ! empty( $_POST['webhook_enabled'] ),
		'url'     => esc_url_raw( $_POST['webhook_url'] ?? '' ),
		'secret'  => sanitize_text_field( $_POST['webhook_secret'] ?? '' ),
		'headers' => sanitize_textarea_field( $_POST['webhook_headers'] ?? '' ),
		'debug'   => ! empty( $_POST['webhook_debug'] ),
	);
	
	update_option( 'buildfuly_webhook_settings', $settings );
	
	wp_send_json_success();
}
add_action( 'wp_ajax_buildfuly_save_webhook_settings', 'buildfuly_ajax_save_webhook_settings' );

/**
 * AJAX handler for testing webhook
 */
function buildfuly_ajax_test_webhook() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	$webhook_url = esc_url_raw( $_POST['webhook_url'] ?? '' );
	if ( empty( $webhook_url ) ) {
		wp_send_json_error( 'Please enter a webhook URL.' );
	}
	
	// Prepare test data
	$test_data = array(
		'event'           => 'webhook.test',
		'test'            => true,
		'email'           => 'test@example.com',
		'name'            => 'Test User',
		'phone'           => '555-0123',
		'source'          => 'Test Calculator',
		'estimate_total'  => 1500.00,
		'estimate_details'=> 'Service A: $500, Service B: $1000',
		'created_at'      => current_time( 'mysql' ),
		'site_name'       => get_bloginfo( 'name' ),
		'site_url'        => home_url(),
	);
	
	// Get webhook settings (or use test values)
	$settings = array(
		'secret'  => sanitize_text_field( $_POST['webhook_secret'] ?? '' ),
		'headers' => sanitize_textarea_field( $_POST['webhook_headers'] ?? '' ),
		'debug'   => true,
	);
	
	// Send test webhook (blocking this time to get response)
	$headers = array(
		'Content-Type' => 'application/json',
		'User-Agent'   => 'Buildfuly-Webhook/1.0',
	);
	
	// Add custom headers
	if ( ! empty( $settings['headers'] ) ) {
		$custom_headers = explode( "\n", $settings['headers'] );
		foreach ( $custom_headers as $header ) {
			if ( strpos( $header, ':' ) !== false ) {
				list( $key, $value ) = explode( ':', $header, 2 );
				$headers[ trim( $key ) ] = trim( $value );
			}
		}
	}
	
	// Add signature
	if ( ! empty( $settings['secret'] ) ) {
		$signature = hash_hmac( 'sha256', wp_json_encode( $test_data ), $settings['secret'] );
		$headers['X-Buildfuly-Signature'] = $signature;
	}
	
	$test_data['webhook_timestamp'] = time();
	
	$response = wp_remote_post( $webhook_url, array(
		'body'      => wp_json_encode( $test_data ),
		'headers'   => $headers,
		'timeout'   => 10,
		'blocking'  => true, // Wait for response for test
		'sslverify' => true,
	) );
	
	if ( is_wp_error( $response ) ) {
		wp_send_json_error( 'Webhook error: ' . $response->get_error_message() );
	}
	
	$status_code = wp_remote_retrieve_response_code( $response );
	$body = wp_remote_retrieve_body( $response );
	
	if ( $status_code >= 200 && $status_code < 300 ) {
		wp_send_json_success( array(
			'message'     => 'Webhook sent successfully!',
			'status_code' => $status_code,
			'response'    => $body,
		) );
	} else {
		wp_send_json_error( "Webhook returned status $status_code: $body" );
	}
}
add_action( 'wp_ajax_buildfuly_test_webhook', 'buildfuly_ajax_test_webhook' );

/**
 * AJAX handler for updating lead status
 */
function buildfuly_ajax_update_lead_status() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	global $wpdb;
	$table_name = $wpdb->prefix . 'buildfuly_leads';
	
	$lead_id = intval( $_POST['lead_id'] ?? 0 );
	$status = sanitize_text_field( $_POST['status'] ?? 'new' );
	
	$wpdb->update(
		$table_name,
		array( 'status' => $status ),
		array( 'id' => $lead_id ),
		array( '%s' ),
		array( '%d' )
	);
	
	wp_send_json_success();
}
add_action( 'wp_ajax_buildfuly_update_lead_status', 'buildfuly_ajax_update_lead_status' );

/**
 * AJAX handler for deleting a lead
 */
function buildfuly_ajax_delete_lead() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	global $wpdb;
	$table_name = $wpdb->prefix . 'buildfuly_leads';
	
	$lead_id = intval( $_POST['lead_id'] ?? 0 );
	
	$wpdb->delete( $table_name, array( 'id' => $lead_id ), array( '%d' ) );
	
	wp_send_json_success();
}
add_action( 'wp_ajax_buildfuly_delete_lead', 'buildfuly_ajax_delete_lead' );

// ==========================================
// FORM BUILDER AJAX HANDLERS
// ==========================================

/**
 * AJAX handler for saving a form
 */
function buildfuly_ajax_save_form() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	global $wpdb;
	$table_name = $wpdb->prefix . 'buildfuly_forms';
	
	$form_id = intval( $_POST['form_id'] ?? 0 );
	$name = sanitize_text_field( $_POST['name'] ?? '' );
	$fields = wp_unslash( $_POST['fields'] ?? '[]' );
	$settings = wp_unslash( $_POST['settings'] ?? '{}' );
	
	if ( empty( $name ) ) {
		wp_send_json_error( 'Form name is required.' );
	}
	
	$data = array(
		'name'     => $name,
		'fields'   => $fields,
		'settings' => $settings,
	);
	
	if ( $form_id > 0 ) {
		// Update existing form
		$wpdb->update(
			$table_name,
			$data,
			array( 'id' => $form_id ),
			array( '%s', '%s', '%s' ),
			array( '%d' )
		);
	} else {
		// Create new form
		$wpdb->insert(
			$table_name,
			$data,
			array( '%s', '%s', '%s' )
		);
		$form_id = $wpdb->insert_id;
	}
	
	wp_send_json_success( array( 'id' => $form_id ) );
}
add_action( 'wp_ajax_buildfuly_save_form', 'buildfuly_ajax_save_form' );

/**
 * AJAX handler for getting a form
 */
function buildfuly_ajax_get_form() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	global $wpdb;
	$table_name = $wpdb->prefix . 'buildfuly_forms';
	
	$form_id = intval( $_POST['form_id'] ?? 0 );
	
	$form = $wpdb->get_row( $wpdb->prepare(
		"SELECT * FROM $table_name WHERE id = %d",
		$form_id
	) );
	
	if ( ! $form ) {
		wp_send_json_error( 'Form not found.' );
	}
	
	wp_send_json_success( array(
		'id'       => $form->id,
		'name'     => $form->name,
		'fields'   => json_decode( $form->fields, true ),
		'settings' => json_decode( $form->settings, true ),
	) );
}
add_action( 'wp_ajax_buildfuly_get_form', 'buildfuly_ajax_get_form' );

/**
 * AJAX handler for deleting a form
 */
function buildfuly_ajax_delete_form() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	global $wpdb;
	$table_name = $wpdb->prefix . 'buildfuly_forms';
	
	$form_id = intval( $_POST['form_id'] ?? 0 );
	
	$wpdb->delete( $table_name, array( 'id' => $form_id ), array( '%d' ) );
	
	wp_send_json_success();
}
add_action( 'wp_ajax_buildfuly_delete_form', 'buildfuly_ajax_delete_form' );

/**
 * AJAX handler for previewing a form
 */
function buildfuly_ajax_preview_form() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	if ( ! wp_verify_nonce( $_POST['nonce'] ?? '', 'buildfuly_leads_nonce' ) ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	$form_id = intval( $_POST['form_id'] ?? 0 );
	$html = buildfuly_render_form( $form_id, true );
	
	wp_send_json_success( array( 'html' => $html ) );
}
add_action( 'wp_ajax_buildfuly_preview_form', 'buildfuly_ajax_preview_form' );

/**
 * AJAX handler for rendering shortcode preview
 * Used by both Leads Form Builder and Page Builder
 */
function buildfuly_ajax_render_shortcode_preview() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	// Accept nonce from either leads page or page builder
	$nonce = $_POST['nonce'] ?? '';
	$valid_nonce = wp_verify_nonce( $nonce, 'buildfuly_leads_nonce' ) || 
	               wp_verify_nonce( $nonce, 'buildfuly_admin_nonce' );
	
	if ( ! $valid_nonce ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	$html_content = wp_unslash( $_POST['html_content'] ?? '' );
	
	// Process shortcodes
	$rendered = do_shortcode( $html_content );
	
	wp_send_json_success( array( 'html' => $rendered ) );
}
add_action( 'wp_ajax_buildfuly_render_shortcode_preview', 'buildfuly_ajax_render_shortcode_preview' );

/**
 * AJAX handler for getting list of forms (for page builder dropdown)
 */
function buildfuly_ajax_get_forms_list() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
	}
	
	// Accept nonce from page builder
	$nonce = $_POST['nonce'] ?? '';
	$valid_nonce = wp_verify_nonce( $nonce, 'buildfuly_admin_nonce' ) || 
	               wp_verify_nonce( $nonce, 'buildfuly_leads_nonce' );
	
	if ( ! $valid_nonce ) {
		wp_send_json_error( 'Security check failed.' );
	}
	
	global $wpdb;
	$table_name = $wpdb->prefix . 'buildfuly_forms';
	
	// Check if table exists
	$table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) );
	
	if ( ! $table_exists ) {
		wp_send_json_success( array( 'forms' => array() ) );
	}
	
	$forms = $wpdb->get_results( "SELECT id, name FROM $table_name ORDER BY name ASC", ARRAY_A );
	
	wp_send_json_success( array( 'forms' => $forms ? $forms : array() ) );
}
add_action( 'wp_ajax_buildfuly_get_forms_list', 'buildfuly_ajax_get_forms_list' );

/**
 * Render a form by ID
 */
function buildfuly_render_form( $form_id, $is_preview = false ) {
	global $wpdb;
	$table_name = $wpdb->prefix . 'buildfuly_forms';
	
	$form = $wpdb->get_row( $wpdb->prepare(
		"SELECT * FROM $table_name WHERE id = %d",
		$form_id
	) );
	
	if ( ! $form ) {
		return '<p>Form not found.</p>';
	}
	
	$fields = json_decode( $form->fields, true );
	$settings = json_decode( $form->settings, true );
	$button_text = $settings['button_text'] ?? 'Submit';
	$success_message = $settings['success_message'] ?? 'Thank you! We\'ll be in touch soon.';
	
	// Check if form has custom HTML (calculator) - if so, use wider layout
	$has_calculator = false;
	foreach ( $fields as $field ) {
		if ( $field['type'] === 'html' ) {
			$has_calculator = true;
			break;
		}
	}
	
	if ( empty( $fields ) || ! is_array( $fields ) ) {
		return '<p>Form has no fields.</p>';
	}
	
	$unique_id = 'buildfuly-form-' . $form_id . '-' . wp_rand( 1000, 9999 );
	$wrapper_class = $has_calculator ? 'buildfuly-form-wrapper has-calculator' : 'buildfuly-form-wrapper';
	
	ob_start();
	?>
	<div class="<?php echo esc_attr( $wrapper_class ); ?>" id="<?php echo esc_attr( $unique_id ); ?>">
		<form class="buildfuly-custom-form" data-form-id="<?php echo esc_attr( $form_id ); ?>">
			<?php if ( ! $is_preview ) : ?>
				<?php wp_nonce_field( 'buildfuly_submit_form', 'buildfuly_form_nonce' ); ?>
			<?php endif; ?>
			<input type="hidden" name="form_id" value="<?php echo esc_attr( $form_id ); ?>">
			<input type="hidden" name="form_name" value="<?php echo esc_attr( $form->name ); ?>">
			
			<!-- Hidden fields for estimate data (if used with calculator) -->
			<input type="hidden" class="buildfuly-estimate-total" name="estimate_total" value="">
			<input type="hidden" class="buildfuly-estimate-breakdown" name="estimate_breakdown" value="">
			
			<?php foreach ( $fields as $field ) : ?>
				<?php 
				// Skip label for HTML, divider and hidden fields
				$skip_label = in_array( $field['type'], array( 'hidden', 'html', 'divider' ) );
				?>
				<div class="buildfuly-form-field <?php echo 'field-type-' . esc_attr( $field['type'] ); ?>">
					<?php if ( ! $skip_label ) : ?>
						<label class="buildfuly-form-label">
							<?php echo esc_html( $field['label'] ); ?>
							<?php if ( ! empty( $field['required'] ) ) : ?>
								<span class="required">*</span>
							<?php endif; ?>
						</label>
					<?php endif; ?>
					
					<?php echo buildfuly_render_field( $field, $is_preview ); ?>
				</div>
			<?php endforeach; ?>
			
			<div class="buildfuly-form-submit">
				<button type="submit" class="buildfuly-submit-btn" <?php echo $is_preview ? 'disabled' : ''; ?>>
					<span class="btn-text"><?php echo esc_html( $button_text ); ?></span>
					<span class="btn-loading" style="display:none;">
						<svg width="20" height="20" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
							<circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="3" fill="none" stroke-dasharray="31.4 31.4" transform="rotate(-90 12 12)">
								<animateTransform attributeName="transform" type="rotate" from="0 12 12" to="360 12 12" dur="1s" repeatCount="indefinite"/>
							</circle>
						</svg>
					</span>
				</button>
			</div>
			
			<div class="buildfuly-form-success" style="display:none;">
				<div class="success-icon">✓</div>
				<div class="success-message"><?php echo esc_html( $success_message ); ?></div>
			</div>
			
			<div class="buildfuly-form-error" style="display:none;"></div>
		</form>
	</div>
	
	<style>
	.buildfuly-form-wrapper {
		max-width: 600px;
		margin: 20px auto;
	}
	
	.buildfuly-form-wrapper.has-calculator {
		max-width: 100%;
	}
	
	.buildfuly-custom-form .buildfuly-form-field {
		margin-bottom: 18px;
	}
	
	.buildfuly-custom-form .buildfuly-form-field.field-type-html {
		margin-bottom: 25px;
	}
	
	.buildfuly-custom-form .buildfuly-form-field.field-type-divider {
		margin-bottom: 5px;
	}
	
	.buildfuly-custom-html {
		/* Reset text alignment to prevent theme/page styles from centering form elements */
		text-align: left !important;
	}
	
	.buildfuly-custom-html input[type="checkbox"],
	.buildfuly-custom-html input[type="radio"] {
		margin-right: 8px;
	}
	
	.buildfuly-custom-html label {
		text-align: left !important;
		display: inline-flex !important;
		align-items: center;
	}
	
	.buildfuly-form-label {
		display: block;
		font-weight: 500;
		margin-bottom: 6px;
		color: #374151;
	}
	
	.buildfuly-form-label .required {
		color: #dc2626;
	}
	
	.buildfuly-custom-form input[type="text"],
	.buildfuly-custom-form input[type="email"],
	.buildfuly-custom-form input[type="tel"],
	.buildfuly-custom-form input[type="number"],
	.buildfuly-custom-form input[type="date"],
	.buildfuly-custom-form textarea,
	.buildfuly-custom-form select {
		width: 100%;
		padding: 12px 14px;
		border: 2px solid #e5e7eb;
		border-radius: 8px;
		font-size: 15px;
		transition: border-color 0.2s, box-shadow 0.2s;
		box-sizing: border-box;
		font-family: inherit;
	}
	
	.buildfuly-custom-form input:focus,
	.buildfuly-custom-form textarea:focus,
	.buildfuly-custom-form select:focus {
		outline: none;
		border-color: #3b82f6;
		box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
	}
	
	.buildfuly-custom-form textarea {
		min-height: 100px;
		resize: vertical;
	}
	
	.buildfuly-custom-form .checkbox-group,
	.buildfuly-custom-form .radio-group {
		display: flex;
		flex-direction: column;
		gap: 8px;
	}
	
	.buildfuly-custom-form .checkbox-group label,
	.buildfuly-custom-form .radio-group label {
		display: flex;
		align-items: center;
		gap: 8px;
		cursor: pointer;
		font-weight: normal;
	}
	
	.buildfuly-custom-form .checkbox-group input,
	.buildfuly-custom-form .radio-group input {
		width: auto;
	}
	
	.buildfuly-submit-btn {
		width: 100%;
		padding: 14px 24px;
		background: linear-gradient(135deg, #3b82f6, #2563eb);
		color: #fff;
		border: none;
		border-radius: 8px;
		font-size: 16px;
		font-weight: 600;
		cursor: pointer;
		transition: transform 0.2s, box-shadow 0.2s;
		display: flex;
		align-items: center;
		justify-content: center;
		gap: 8px;
	}
	
	.buildfuly-submit-btn:hover:not(:disabled) {
		transform: translateY(-2px);
		box-shadow: 0 4px 12px rgba(59, 130, 246, 0.4);
	}
	
	.buildfuly-submit-btn:disabled {
		opacity: 0.7;
		cursor: not-allowed;
	}
	
	.buildfuly-form-success {
		text-align: center;
		padding: 30px;
		background: #ecfdf5;
		border-radius: 12px;
	}
	
	.buildfuly-form-success .success-icon {
		width: 50px;
		height: 50px;
		background: #10b981;
		color: #fff;
		border-radius: 50%;
		display: inline-flex;
		align-items: center;
		justify-content: center;
		font-size: 24px;
		margin-bottom: 12px;
	}
	
	.buildfuly-form-success .success-message {
		color: #065f46;
		font-size: 16px;
		font-weight: 500;
	}
	
	.buildfuly-form-error {
		padding: 12px 16px;
		background: #fef2f2;
		color: #dc2626;
		border-radius: 8px;
		margin-top: 10px;
		font-size: 14px;
	}
	</style>
	
	<?php if ( ! $is_preview ) : ?>
	<script>
	(function() {
		const wrapper = document.getElementById('<?php echo esc_js( $unique_id ); ?>');
		const form = wrapper.querySelector('form');
		if (!form) return;
		
		form.addEventListener('submit', function(e) {
			e.preventDefault();
			
			const btn = form.querySelector('.buildfuly-submit-btn');
			const btnText = btn.querySelector('.btn-text');
			const btnLoading = btn.querySelector('.btn-loading');
			const successDiv = form.querySelector('.buildfuly-form-success');
			const errorDiv = form.querySelector('.buildfuly-form-error');
			
			// Get form data
			const formData = new FormData(form);
			formData.append('action', 'buildfuly_submit_form');
			
			// Show loading
			btn.disabled = true;
			btnText.style.display = 'none';
			btnLoading.style.display = 'block';
			errorDiv.style.display = 'none';
			
			// Submit
			fetch('<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>', {
				method: 'POST',
				body: formData
			})
			.then(response => response.json())
			.then(data => {
				btn.disabled = false;
				btnText.style.display = 'block';
				btnLoading.style.display = 'none';
				
				if (data.success) {
					// Hide form fields, show success
					Array.from(form.querySelectorAll('.buildfuly-form-field, .buildfuly-form-submit')).forEach(el => {
						el.style.display = 'none';
					});
					successDiv.style.display = 'block';
				} else {
					errorDiv.textContent = data.data?.message || data.data || 'Something went wrong. Please try again.';
					errorDiv.style.display = 'block';
				}
			})
			.catch(error => {
				btn.disabled = false;
				btnText.style.display = 'block';
				btnLoading.style.display = 'none';
				errorDiv.textContent = 'Connection error. Please try again.';
				errorDiv.style.display = 'block';
			});
		});
	})();
	</script>
	<?php endif; ?>
	<?php
	return ob_get_clean();
}

/**
 * Render a single form field
 */
function buildfuly_render_field( $field, $is_preview = false ) {
	$name = esc_attr( $field['name'] ?? '' );
	$placeholder = esc_attr( $field['placeholder'] ?? '' );
	$required = ! empty( $field['required'] ) ? 'required' : '';
	$disabled = $is_preview ? 'disabled' : '';
	
	switch ( $field['type'] ) {
		case 'textarea':
			return "<textarea name=\"$name\" placeholder=\"$placeholder\" $required $disabled></textarea>";
			
		case 'select':
			$options = explode( "\n", $field['options'] ?? '' );
			$html = "<select name=\"$name\" $required $disabled><option value=\"\">Select...</option>";
			foreach ( $options as $opt ) {
				$opt = trim( $opt );
				if ( $opt ) {
					$html .= '<option value="' . esc_attr( $opt ) . '">' . esc_html( $opt ) . '</option>';
				}
			}
			$html .= '</select>';
			return $html;
			
		case 'checkbox':
			return "<div class=\"checkbox-group\"><label><input type=\"checkbox\" name=\"$name\" value=\"1\" $disabled> " . esc_html( $field['label'] ) . "</label></div>";
			
		case 'radio':
			$options = explode( "\n", $field['options'] ?? '' );
			$html = '<div class="radio-group">';
			foreach ( $options as $i => $opt ) {
				$opt = trim( $opt );
				if ( $opt ) {
					$html .= '<label><input type="radio" name="' . $name . '" value="' . esc_attr( $opt ) . '" ' . $disabled . '> ' . esc_html( $opt ) . '</label>';
				}
			}
			$html .= '</div>';
			return $html;
			
		case 'hidden':
			return "<input type=\"hidden\" name=\"$name\" value=\"$placeholder\">";
			
		case 'date':
			return "<input type=\"date\" name=\"$name\" $required $disabled>";
			
		case 'number':
			return "<input type=\"number\" name=\"$name\" placeholder=\"$placeholder\" $required $disabled>";
			
		case 'phone':
			return "<input type=\"tel\" name=\"$name\" placeholder=\"$placeholder\" $required $disabled>";
			
		case 'email':
			return "<input type=\"email\" name=\"$name\" placeholder=\"$placeholder\" $required $disabled>";
		
		case 'html':
			// Custom HTML/Calculator - render the raw HTML content
			$html_content = $field['html_content'] ?? '';
			if ( $is_preview ) {
				return '<div class="buildfuly-html-preview" style="background:#f6f7f7;padding:20px;border-radius:8px;text-align:center;color:#646970;"><span class="dashicons dashicons-editor-code" style="font-size:24px;"></span><br>Custom HTML/Calculator will render here</div>';
			}
			// Process any WordPress shortcodes in the content
			$html_content = do_shortcode( $html_content );
			// Render the actual HTML content with scripts, centered by default
			return '<div class="buildfuly-custom-html">' . $html_content . '</div>';
			
		case 'divider':
			$divider_style = $field['divider_style'] ?? 'heading';
			if ( $divider_style === 'line' ) {
				return '<hr class="buildfuly-divider-line" style="border:none;border-top:1px solid #e5e7eb;margin:20px 0;">';
			} else {
				return '<h3 class="buildfuly-divider-heading" style="font-size:18px;font-weight:600;color:#1f2937;margin:25px 0 15px;padding-bottom:8px;border-bottom:2px solid #3b82f6;">' . esc_html( $field['label'] ) . '</h3>';
			}
			
		default: // text
			return "<input type=\"text\" name=\"$name\" placeholder=\"$placeholder\" $required $disabled>";
	}
}

/**
 * Form shortcode
 */
function buildfuly_form_shortcode( $atts ) {
	$atts = shortcode_atts( array(
		'id' => 0,
	), $atts, 'buildfuly_form' );
	
	$form_id = intval( $atts['id'] );
	
	if ( ! $form_id ) {
		return '<p>Invalid form ID.</p>';
	}
	
	return buildfuly_render_form( $form_id );
}
add_shortcode( 'buildfuly_form', 'buildfuly_form_shortcode' );

/**
 * AJAX handler for form submission
 */
function buildfuly_ajax_submit_form() {
	// Verify nonce
	if ( ! wp_verify_nonce( $_POST['buildfuly_form_nonce'] ?? '', 'buildfuly_submit_form' ) ) {
		error_log( 'Buildfuly form: Nonce verification failed' );
		wp_send_json_error( 'Security check failed. Please refresh the page and try again.' );
	}
	
	$form_id = intval( $_POST['form_id'] ?? 0 );
	$form_name = sanitize_text_field( $_POST['form_name'] ?? 'Form' );
	
	// Get form to know which fields to expect
	global $wpdb;
	$forms_table = $wpdb->prefix . 'buildfuly_forms';
	$leads_table = $wpdb->prefix . 'buildfuly_leads';
	
	// Check if tables exist
	$forms_exists = $wpdb->get_var( "SHOW TABLES LIKE '$forms_table'" ) === $forms_table;
	$leads_exists = $wpdb->get_var( "SHOW TABLES LIKE '$leads_table'" ) === $leads_table;
	
	// Auto-create leads table if it doesn't exist
	if ( ! $leads_exists ) {
		$charset_collate = $wpdb->get_charset_collate();
		$sql = "CREATE TABLE $leads_table (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			email varchar(255),
			name varchar(255),
			phone varchar(50),
			source varchar(255),
			estimate_total decimal(10,2) DEFAULT 0,
			estimate_details text,
			form_data longtext,
			status varchar(50) DEFAULT 'new',
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id)
		) $charset_collate;";
		
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql );
		
		// Re-check
		$leads_exists = $wpdb->get_var( "SHOW TABLES LIKE '$leads_table'" ) === $leads_table;
	}
	
	if ( ! $leads_exists ) {
		error_log( 'Buildfuly: Leads table does not exist. Please deactivate and reactivate the plugin.' );
		wp_send_json_error( 'Database not configured. Please contact the site administrator.' );
	}
	
	// If no form_id provided or forms table doesn't exist, save directly without form validation
	if ( ! $form_id || ! $forms_exists ) {
		// Direct save mode - collect common fields
		$email = sanitize_email( $_POST['email'] ?? '' );
		$name = sanitize_text_field( $_POST['name'] ?? '' );
		$phone = sanitize_text_field( $_POST['phone'] ?? '' );
		$estimate_total = floatval( $_POST['estimate_total'] ?? 0 );
		$estimate_breakdown = sanitize_text_field( $_POST['estimate_breakdown'] ?? '' );
		
		$result = $wpdb->insert(
			$leads_table,
			array(
				'email'            => $email,
				'name'             => $name,
				'phone'            => $phone,
				'source'           => $form_name,
				'estimate_total'   => $estimate_total,
				'estimate_details' => $estimate_breakdown,
				'form_data'        => wp_json_encode( $_POST ),
				'status'           => 'new',
				'created_at'       => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%s', '%s', '%f', '%s', '%s', '%s', '%s' )
		);
		
		if ( $result === false ) {
			error_log( 'Buildfuly direct save error: ' . $wpdb->last_error );
			wp_send_json_error( 'Database error: ' . $wpdb->last_error );
		}
		
		// Send emails even for direct save mode (with defaults - emails enabled by default)
		$settings = buildfuly_get_email_settings();
		$site_name = get_bloginfo( 'name' );
		
		// Send to admin
		if ( ! empty( $settings['send_to_admin'] ) ) {
			$admin_email = $settings['admin_email'] ?? get_option( 'admin_email' );
			$subject = "New Lead from $form_name - $site_name";
			$message = "<h2>New Lead from $form_name</h2>";
			$message .= "<p><strong>Name:</strong> " . ( $name ?: 'Not provided' ) . "</p>";
			$message .= "<p><strong>Email:</strong> " . ( $email ?: 'Not provided' ) . "</p>";
			$message .= "<p><strong>Phone:</strong> " . ( $phone ?: 'Not provided' ) . "</p>";
			if ( $estimate_total > 0 ) {
				$message .= "<p><strong>Estimate:</strong> \$" . number_format( $estimate_total, 2 ) . "</p>";
			}
			if ( $estimate_breakdown ) {
				$message .= "<p><strong>Details:</strong> $estimate_breakdown</p>";
			}
			$message .= "<p><a href='" . admin_url( 'admin.php?page=buildfuly-leads' ) . "'>View in Dashboard</a></p>";
			
			$headers = array( 'Content-Type: text/html; charset=UTF-8' );
			$sent = wp_mail( $admin_email, $subject, $message, $headers );
			if ( ! $sent ) {
				error_log( 'Buildfuly: Failed to send admin email to ' . $admin_email );
			}
		}
		
		// Send to lead
		if ( ! empty( $settings['send_to_lead'] ) && $email ) {
			$subject = str_replace(
				array( '{site_name}', '{name}' ),
				array( $site_name, $name ?: 'there' ),
				$settings['email_subject'] ?? 'Thank you for contacting us'
			);
			
			$template = $settings['email_template'] ?? "Hi {name},\n\nThank you for contacting us. We'll be in touch soon!\n\nBest regards,\n{site_name}";
			$message = str_replace(
				array( '{name}', '{email}', '{phone}', '{estimate_total}', '{estimate_details}', '{site_name}' ),
				array(
					$name ?: 'there',
					$email,
					$phone ?: 'Not provided',
					$estimate_total > 0 ? '$' . number_format( $estimate_total, 2 ) : 'N/A',
					$estimate_breakdown ?: 'N/A',
					$site_name
				),
				$template
			);
			
			$headers = array( 'Content-Type: text/html; charset=UTF-8' );
			$sent = wp_mail( $email, $subject, nl2br( $message ), $headers );
			if ( ! $sent ) {
				error_log( 'Buildfuly: Failed to send lead email to ' . $email );
			}
		}
		
		// Send webhook if configured
		$webhook_settings = get_option( 'buildfuly_webhook_settings', array() );
		if ( ! empty( $webhook_settings['enabled'] ) && ! empty( $webhook_settings['url'] ) ) {
			buildfuly_send_webhook( $webhook_settings['url'], array(
				'event'            => 'lead.created',
				'lead_id'          => $wpdb->insert_id,
				'email'            => $email,
				'name'             => $name,
				'phone'            => $phone,
				'source'           => $form_name,
				'estimate_total'   => $estimate_total,
				'estimate_details' => $estimate_breakdown,
				'created_at'       => current_time( 'mysql' ),
				'site_name'        => $site_name,
				'site_url'         => home_url(),
			), $webhook_settings );
		}
		
		wp_send_json_success( array( 'message' => 'Form submitted successfully!' ) );
		return;
	}
	
	$form = $wpdb->get_row( $wpdb->prepare(
		"SELECT * FROM $forms_table WHERE id = %d",
		$form_id
	) );
	
	if ( ! $form ) {
		error_log( "Buildfuly: Form ID $form_id not found in database" );
		wp_send_json_error( 'Form configuration not found.' );
	}
	
	$fields = json_decode( $form->fields, true );
	
	// Collect form data
	$form_data = array();
	$email = '';
	$name = '';
	$phone = '';
	
	foreach ( $fields as $field ) {
		// Skip non-input fields
		if ( in_array( $field['type'], array( 'html', 'divider' ) ) ) {
			continue;
		}
		
		$field_name = $field['name'];
		$value = sanitize_text_field( $_POST[ $field_name ] ?? '' );
		
		// Check required fields
		if ( ! empty( $field['required'] ) && empty( $value ) ) {
			wp_send_json_error( $field['label'] . ' is required.' );
		}
		
		$form_data[ $field_name ] = $value;
		
		// Capture special fields
		if ( $field['type'] === 'email' || $field_name === 'email' ) {
			$email = sanitize_email( $value );
		}
		if ( $field_name === 'name' || strpos( $field_name, 'name' ) !== false ) {
			$name = $value;
		}
		if ( $field['type'] === 'phone' || $field_name === 'phone' ) {
			$phone = $value;
		}
	}
	
	// Get estimate data if provided
	$estimate_total = floatval( $_POST['estimate_total'] ?? 0 );
	$estimate_breakdown = sanitize_text_field( $_POST['estimate_breakdown'] ?? '' );
	
	// Save to leads table
	$result = $wpdb->insert(
		$leads_table,
		array(
			'email'            => $email,
			'name'             => $name,
			'phone'            => $phone,
			'source'           => $form_name,
			'estimate_total'   => $estimate_total,
			'estimate_details' => $estimate_breakdown,
			'form_data'        => wp_json_encode( $form_data ),
			'status'           => 'new',
			'created_at'       => current_time( 'mysql' ),
		),
		array( '%s', '%s', '%s', '%s', '%f', '%s', '%s', '%s', '%s' )
	);
	
	if ( $result === false ) {
		error_log( 'Buildfuly form submit error: ' . $wpdb->last_error );
		wp_send_json_error( 'Database error: ' . $wpdb->last_error );
	}
	
	// Send emails (with defaults - emails enabled by default)
	$settings = buildfuly_get_email_settings();
	
	// Send to admin
	if ( ! empty( $settings['send_to_admin'] ) ) {
		$admin_email = $settings['admin_email'] ?? get_option( 'admin_email' );
		$site_name = get_bloginfo( 'name' );
		
		$subject = "New Lead from $form_name - $site_name";
		$message = "<h2>New Lead from $form_name</h2>";
		$message .= "<p><strong>Name:</strong> " . ( $name ?: 'Not provided' ) . "</p>";
		$message .= "<p><strong>Email:</strong> " . ( $email ?: 'Not provided' ) . "</p>";
		$message .= "<p><strong>Phone:</strong> " . ( $phone ?: 'Not provided' ) . "</p>";
		
		if ( $estimate_total > 0 ) {
			$message .= "<p><strong>Estimate:</strong> \$" . number_format( $estimate_total, 2 ) . "</p>";
		}
		
		$message .= "<h3>Form Data:</h3><ul>";
		foreach ( $form_data as $key => $val ) {
			$message .= "<li><strong>$key:</strong> $val</li>";
		}
		$message .= "</ul>";
		$message .= "<p><a href='" . admin_url( 'admin.php?page=buildfuly-leads' ) . "'>View in Dashboard</a></p>";
		
		$headers = array( 'Content-Type: text/html; charset=UTF-8' );
		wp_mail( $admin_email, $subject, $message, $headers );
	}
	
	// Send to lead
	if ( ! empty( $settings['send_to_lead'] ) && $email ) {
		$site_name = get_bloginfo( 'name' );
		$subject = str_replace(
			array( '{site_name}', '{name}' ),
			array( $site_name, $name ?: 'there' ),
			$settings['email_subject'] ?? 'Thank you for contacting us'
		);
		
		$template = $settings['email_template'] ?? "Hi {name},\n\nThank you for contacting us. We'll be in touch soon!\n\nBest regards,\n{site_name}";
		
		// Build estimate details string
		$details_str = '';
		if ( ! empty( $estimate_breakdown ) ) {
			$details_str = $estimate_breakdown;
		}
		
		$message = str_replace(
			array( '{name}', '{email}', '{phone}', '{estimate_total}', '{estimate_details}', '{site_name}' ),
			array(
				$name ?: 'there',
				$email,
				$phone ?: 'Not provided',
				$estimate_total > 0 ? '$' . number_format( $estimate_total, 2 ) : 'N/A',
				$details_str ?: 'N/A',
				$site_name
			),
			$template
		);
		
		$headers = array( 'Content-Type: text/html; charset=UTF-8' );
		wp_mail( $email, $subject, nl2br( $message ), $headers );
	}
	
	// Send webhook if configured
	$webhook_settings = get_option( 'buildfuly_webhook_settings', array() );
	if ( ! empty( $webhook_settings['enabled'] ) && ! empty( $webhook_settings['url'] ) ) {
		$site_name = get_bloginfo( 'name' );
		buildfuly_send_webhook( $webhook_settings['url'], array(
			'event'            => 'lead.created',
			'lead_id'          => $wpdb->insert_id,
			'email'            => $email,
			'name'             => $name,
			'phone'            => $phone,
			'source'           => $form_name,
			'estimate_total'   => $estimate_total,
			'estimate_details' => $estimate_breakdown,
			'form_data'        => $form_data,
			'created_at'       => current_time( 'mysql' ),
			'site_name'        => $site_name,
			'site_url'         => home_url(),
		), $webhook_settings );
	}
	
	wp_send_json_success( array( 'message' => 'Form submitted successfully!' ) );
}
add_action( 'wp_ajax_buildfuly_submit_form', 'buildfuly_ajax_submit_form' );
add_action( 'wp_ajax_nopriv_buildfuly_submit_form', 'buildfuly_ajax_submit_form' );

