<?php

/**
 * Database operations handler
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

/**
 * Database operations handler.
 *
 * Handles all database operations for content and styling.
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @author     Buildfuly Team
 */
class Buildfuly_Database {

	/**
	 * Perform a cached database query
	 *
	 * @since    1.0.0
	 * @param    string    $cache_key    The cache key
	 * @param    callable  $query_func   The database query function
	 * @param    string    $cache_group  The cache group (default 'buildfuly')
	 * @param    int       $expiration   Cache expiration in seconds (default 3600)
	 * @return   mixed                   Query result
	 */
	private function cached_query( $cache_key, $query_func, $cache_group = 'buildfuly', $expiration = 3600 ) {
		$cached = wp_cache_get( $cache_key, $cache_group );
		
		if ( false !== $cached ) {
			return $cached;
		}
		
		$result = $query_func();
		
		if ( false !== $result ) {
			wp_cache_set( $cache_key, $result, $cache_group, $expiration );
		}
		
		return $result;
	}

	/**
	 * Save content to database
	 *
	 * @since    1.0.0
	 * @param    string    $key      The content key
	 * @param    mixed     $value    The content value
	 * @param    string    $type     The content type (text, json, html)
	 * @return   bool                Success status
	 */
	public function save_content( $key, $value, $type = 'text' ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_content';

		if ( is_array( $value ) || is_object( $value ) ) {
			$value = json_encode( $value );
			$type = 'json';
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cache invalidated below
		$result = $wpdb->replace(
			$table_name,
			array(
				'content_key'   => $key,
				'content_value' => $value,
				'content_type'  => $type,
			),
			array( '%s', '%s', '%s' )
		);

		// Invalidate cache after update
		wp_cache_delete( 'content_' . $key, 'buildfuly' );
		wp_cache_delete( 'all_content', 'buildfuly' );

		return $result !== false;
	}

	/**
	 * Get content from database
	 *
	 * @since    1.0.0
	 * @param    string    $key      The content key
	 * @param    mixed     $default  Default value if not found
	 * @return   mixed               The content value
	 */
	public function get_content( $key, $default = null ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_content';

		return $this->cached_query(
			'content_' . $key,
			function() use ( $wpdb, $table_name, $key, $default ) {
				$row = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Using wp_cache via cached_query wrapper, table name is prefixed and safe
					$wpdb->prepare( "SELECT content_value, content_type FROM $table_name WHERE content_key = %s", $key )
				);

				if ( ! $row ) {
					return $default;
				}

				if ( $row->content_type === 'json' ) {
					return json_decode( $row->content_value, true );
				}

				return $row->content_value;
			}
		);
	}

	/**
	 * Get all content
	 *
	 * @since    1.0.0
	 * @return   array    All content as associative array
	 */
	public function get_all_content() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_content';

		return $this->cached_query(
			'all_content',
			function() use ( $wpdb, $table_name ) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Using wp_cache via cached_query wrapper, table name is prefixed and safe
				$rows = $wpdb->get_results( "SELECT content_key, content_value, content_type FROM $table_name" );

				$content = array();
				foreach ( $rows as $row ) {
					if ( $row->content_type === 'json' ) {
						$content[ $row->content_key ] = json_decode( $row->content_value, true );
					} else {
						$content[ $row->content_key ] = $row->content_value;
					}
				}

				return $content;
			}
		);
	}

	/**
	 * Save style to database
	 *
	 * @since    1.0.0
	 * @param    string    $key        The style key
	 * @param    mixed     $value      The style value
	 * @param    string    $component  The component name
	 * @return   bool                  Success status
	 */
	public function save_style( $key, $value, $component = 'global' ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_styles';

		if ( is_array( $value ) || is_object( $value ) ) {
			$value = json_encode( $value );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cache invalidated below
		$result = $wpdb->replace(
			$table_name,
			array(
				'style_key'   => $key,
				'style_value' => $value,
				'component'   => $component,
			),
			array( '%s', '%s', '%s' )
		);

		// Invalidate cache after update
		wp_cache_delete( 'style_' . $key, 'buildfuly' );
		wp_cache_delete( 'all_styles', 'buildfuly' );
		wp_cache_delete( 'all_styles_' . $component, 'buildfuly' );

		return $result !== false;
	}

	/**
	 * Get style from database
	 *
	 * @since    1.0.0
	 * @param    string    $key      The style key
	 * @param    mixed     $default  Default value if not found
	 * @return   mixed               The style value
	 */
	public function get_style( $key, $default = null ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_styles';

		return $this->cached_query(
			'style_' . $key,
			function() use ( $wpdb, $table_name, $key, $default ) {
				$row = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Using wp_cache via cached_query wrapper, table name is prefixed and safe
					$wpdb->prepare( "SELECT style_value FROM $table_name WHERE style_key = %s", $key )
				);

				if ( ! $row ) {
					return $default;
				}

				// Try to decode JSON
				$decoded = json_decode( $row->style_value, true );
				return $decoded !== null ? $decoded : $row->style_value;
			}
		);
	}

	/**
	 * Get all styles
	 *
	 * @since    1.0.0
	 * @param    string    $component  Filter by component (optional)
	 * @return   array                 All styles as associative array
	 */
	public function get_all_styles( $component = null ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_styles';

		$cache_key = $component ? 'all_styles_' . $component : 'all_styles';

		return $this->cached_query(
			$cache_key,
			function() use ( $wpdb, $table_name, $component ) {
				if ( $component ) {
					$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Using wp_cache via cached_query wrapper, table name is prefixed and safe
						$wpdb->prepare( "SELECT style_key, style_value, component FROM $table_name WHERE component = %s", $component )
					);
				} else {
					$rows = $wpdb->get_results( "SELECT style_key, style_value, component FROM $table_name" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Using wp_cache via cached_query wrapper, table name is prefixed and safe
				}

				$styles = array();
				foreach ( $rows as $row ) {
					$decoded = json_decode( $row->style_value, true );
					$styles[ $row->style_key ] = array(
						'value'     => $decoded !== null ? $decoded : $row->style_value,
						'component' => $row->component
					);
				}

				return $styles;
			}
		);
	}

	/**
	 * Log action to history
	 *
	 * @since    1.0.0
	 * @param    string    $action_type  The action type
	 * @param    mixed     $action_data  The action data
	 * @return   bool                    Success status
	 */
	public function log_history( $action_type, $action_data = null ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_history';

		if ( is_array( $action_data ) || is_object( $action_data ) ) {
			$action_data = json_encode( $action_data );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Insert operation, caching not applicable
		$result = $wpdb->insert(
			$table_name,
			array(
				'action_type' => $action_type,
				'action_data' => $action_data,
				'user_id'     => get_current_user_id(),
			),
			array( '%s', '%s', '%d' )
		);

		return $result !== false;
	}

	/**
	 * Get history
	 *
	 * @since    1.0.0
	 * @param    int      $limit  Number of records to retrieve
	 * @return   array            History records
	 */
	public function get_history( $limit = 50 ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_history';

		return $this->cached_query(
			'history_' . $limit,
			function() use ( $wpdb, $table_name, $limit ) {
				$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Using wp_cache via cached_query wrapper, table name is prefixed and safe
					$wpdb->prepare( "SELECT * FROM $table_name ORDER BY created_at DESC LIMIT %d", $limit )
				);

				$history = array();
				foreach ( $rows as $row ) {
					$row->action_data = json_decode( $row->action_data, true );
					$history[] = $row;
				}

				return $history;
			},
			'buildfuly',
			300 // Shorter cache for history (5 minutes)
		);
	}

	/**
	 * Delete all content
	 *
	 * @since    1.0.0
	 * @return   bool    Success status
	 */
	public function delete_all_content() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_content';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Delete operation, cache flushed below, table name is prefixed and safe
		$result = $wpdb->query( "TRUNCATE TABLE $table_name" ) !== false;
		
		// Flush all content cache
		wp_cache_flush();
		
		return $result;
	}

	/**
	 * Delete all styles
	 *
	 * @since    1.0.0
	 * @return   bool    Success status
	 */
	public function delete_all_styles() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_styles';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Delete operation, cache flushed below, table name is prefixed and safe
		$result = $wpdb->query( "TRUNCATE TABLE $table_name" ) !== false;
		
		// Flush all styles cache
		wp_cache_flush();
		
		return $result;
	}
}
