<?php

/**
 * Content generation handler
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

/**
 * Content generation handler.
 *
 * Handles AI content generation and preparation for future API integration.
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @author     Buildfuly Team
 */
class Buildfuly_Content_Generator {

	/**
	 * The database handler.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Buildfuly_Database    $db    Database handler instance.
	 */
	private $db;

	/**
	 * Initialize the class.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		$this->db = new Buildfuly_Database();
		
		// Component registry no longer loaded locally - fetched from API when needed
	}
	
	/**
	 * Get all available components for AI selection.
	 * Used when AI needs to choose which components to use for a page.
	 * Now fetches from API instead of local registry.
	 *
	 * @since    1.0.3
	 * @return   array    Array of component metadata
	 */
	public function get_available_components() {
		require_once plugin_dir_path( __FILE__ ) . 'class-buildfuly-api-client.php';
		$api_client = new Buildfuly_API_Client();
		$components = $api_client->get_components();
		
		if ( is_wp_error( $components ) ) {
			Buildfuly_Logger::debug( 'BuildfulyAI: Failed to load components from API - ' . $components->get_error_message() );
			return array();
		}
		
		return $components;
	}
	
	/**
	 * Get list of AI-supported components.
	 * Now uses API instead of local registry.
	 *
	 * @since    1.0.3
	 * @return   array    Array of AI-supported components
	 */
	public function get_ai_supported_components() {
		$all_components = $this->get_available_components();
		$ai_supported = array();
		
		// Filter components that support AI
		foreach ( $all_components as $type => $components ) {
			foreach ( $components as $slug => $component ) {
				if ( isset( $component['ai_support'] ) && $component['ai_support'] ) {
					$ai_supported[ $slug ] = $component;
				}
			}
		}
		
		return $ai_supported;
	}



	/**
	 * Generate homepage content structure.
	 *
	 * This method prepares the structure for AI-generated content.
	 * In production, this would call an AI API endpoint.
	 *
	 * @since    1.0.0
	 * @param    array    $params    Generation parameters (business type, industry, etc.)
	 * @return   array               Generated content structure
	 */
	public function generate_homepage_content( $params = array() ) {
		$business_type = isset( $params['business_type'] ) ? $params['business_type'] : 'saas';
		$industry = isset( $params['industry'] ) ? $params['industry'] : 'technology';

		// TODO: Replace with actual AI API call
		// For now, using template-based content
		$content = $this->get_template_content( $business_type, $industry );

		// Save generated content to database
		foreach ( $content as $key => $value ) {
			$this->db->save_content( $key, $value );
		}

		// Log generation action
		$this->db->log_history( 'content_generated', array(
			'business_type' => $business_type,
			'industry'      => $industry,
			'timestamp'     => current_time( 'mysql' )
		) );

		return $content;
	}

	/**
	 * Get template content based on business type.
	 *
	 * @since    1.0.0
	 * @param    string    $business_type    The business type
	 * @param    string    $industry         The industry
	 * @return   array                       Template content
	 */
	private function get_template_content( $business_type, $industry ) {
		$templates = array(
			'saas' => array(
				'homepage.title'           => 'Transform Your Business with Modern Software',
				'homepage.heroTitle'       => 'Build Better. Ship Faster. Scale Smarter.',
				'homepage.heroContent'     => 'Our platform empowers teams to create exceptional products with cutting-edge tools and seamless collaboration.',
				'homepage.heroCTA'         => 'Start Free Trial',
				'homepage.section1Title'   => 'Why Choose Us',
				'homepage.section1Content' => 'We provide enterprise-grade solutions that scale with your business, backed by world-class support and security.',
				'homepage.feature1Title'   => 'Lightning Fast',
				'homepage.feature1Desc'    => 'Deploy in seconds with our optimized infrastructure.',
				'homepage.feature2Title'   => 'Secure by Default',
				'homepage.feature2Desc'    => 'Enterprise-grade security built into every feature.',
				'homepage.feature3Title'   => 'Scale Effortlessly',
				'homepage.feature3Desc'    => 'From startup to enterprise, we grow with you.',
				'homepage.section2Title'   => 'Testimonials',
				'homepage.testimonial1'    => 'This platform transformed how we work. Highly recommended!',
				'homepage.testimonial1Author' => 'Jane Doe, CEO at TechCorp',
				'homepage.ctaTitle'        => 'Ready to Get Started?',
				'homepage.ctaContent'      => 'Join thousands of satisfied customers today.',
				'homepage.ctaButton'       => 'Get Started Now',
			),
			'professional_service' => array(
				'homepage.title'           => 'Professional Services You Can Trust',
				'homepage.heroTitle'       => 'Expertise That Delivers Results',
				'homepage.heroContent'     => 'Partner with industry experts who understand your challenges and deliver tailored solutions.',
				'homepage.heroCTA'         => 'Schedule Consultation',
				'homepage.section1Title'   => 'Our Approach',
				'homepage.section1Content' => 'We combine deep industry knowledge with proven methodologies to drive measurable outcomes.',
				'homepage.feature1Title'   => 'Proven Track Record',
				'homepage.feature1Desc'    => '20+ years of excellence in delivering results.',
				'homepage.feature2Title'   => 'Expert Team',
				'homepage.feature2Desc'    => 'Industry-certified professionals at your service.',
				'homepage.feature3Title'   => 'Custom Solutions',
				'homepage.feature3Desc'    => 'Tailored approaches for your unique needs.',
				'homepage.section2Title'   => 'Client Success Stories',
				'homepage.testimonial1'    => 'Their expertise helped us achieve goals we thought impossible.',
				'homepage.testimonial1Author' => 'John Smith, Director at FinanceCo',
				'homepage.ctaTitle'        => 'Let\'s Discuss Your Project',
				'homepage.ctaContent'      => 'Contact us for a free consultation.',
				'homepage.ctaButton'       => 'Contact Us',
			),
			'consulting' => array(
				'homepage.title'           => 'Strategic Consulting for Growth',
				'homepage.heroTitle'       => 'Transform Strategy into Success',
				'homepage.heroContent'     => 'Data-driven insights and strategic guidance to accelerate your business growth.',
				'homepage.heroCTA'         => 'Book a Call',
				'homepage.section1Title'   => 'Our Methodology',
				'homepage.section1Content' => 'We use a systematic approach to understand your business and deliver actionable strategies.',
				'homepage.feature1Title'   => 'Data-Driven',
				'homepage.feature1Desc'    => 'Decisions backed by comprehensive analysis.',
				'homepage.feature2Title'   => 'Strategic Focus',
				'homepage.feature2Desc'    => 'Long-term growth, not quick fixes.',
				'homepage.feature3Title'   => 'Measurable Results',
				'homepage.feature3Desc'    => 'Clear metrics and KPIs for success.',
				'homepage.section2Title'   => 'What Our Clients Say',
				'homepage.testimonial1'    => 'Best consulting partnership we\'ve ever had. Exceptional value.',
				'homepage.testimonial1Author' => 'Sarah Johnson, Founder at StartupX',
				'homepage.ctaTitle'        => 'Ready to Accelerate Growth?',
				'homepage.ctaContent'      => 'Schedule your strategy session today.',
				'homepage.ctaButton'       => 'Book Strategy Call',
			),
		);

		return isset( $templates[ $business_type ] ) ? $templates[ $business_type ] : $templates['saas'];
	}

	/**
	 * Get all generated content.
	 *
	 * @since    1.0.0
	 * @return   array    All content
	 */
	public function get_all_content() {
		return $this->db->get_all_content();
	}

	/**
	 * Update specific content item.
	 *
	 * @since    1.0.0
	 * @param    string    $key      The content key
	 * @param    mixed     $value    The content value
	 * @return   bool                Success status
	 */
	public function update_content( $key, $value ) {
		$result = $this->db->save_content( $key, $value );

		if ( $result ) {
			$this->db->log_history( 'content_updated', array(
				'key'   => $key,
				'value' => $value
			) );
		}

		return $result;
	}

	/**
	 * Regenerate specific content component.
	 *
	 * @since    1.0.0
	 * @param    string    $component    The component to regenerate
	 * @param    array     $params       Generation parameters
	 * @return   mixed                   Generated content
	 */
	public function regenerate_component( $component, $params = array() ) {
		// TODO: Replace with actual AI API call for single component
		// This saves tokens by only regenerating one component
		
		$business_type = isset( $params['business_type'] ) ? $params['business_type'] : 'saas';
		$template = $this->get_template_content( $business_type, '' );

		if ( isset( $template[ $component ] ) ) {
			$value = $template[ $component ];
			$this->db->save_content( $component, $value );
			
			$this->db->log_history( 'component_regenerated', array(
				'component' => $component
			) );

			return $value;
		}

		return null;
	}

	/**
	 * Clear all generated content.
	 *
	 * @since    1.0.0
	 * @return   bool    Success status
	 */
	public function clear_all_content() {
		$result = $this->db->delete_all_content();
		
		if ( $result ) {
			$this->db->log_history( 'content_cleared' );
		}

		return $result;
	}

	/**
	 * Generate a single SEO-optimized content section.
	 *
	 * @since    1.0.1
	 * @param    array    $params    Parameters (keywords, section_type)
	 * @return   array               Section data with HTML
	 */
	public function generate_seo_section( $params = array() ) {
		$keywords = isset( $params['keywords'] ) ? $params['keywords'] : '';
		$section_type = isset( $params['section_type'] ) ? $params['section_type'] : 'random';
		$image_id = isset( $params['image_id'] ) ? intval( $params['image_id'] ) : 0;
		$business_context = isset( $params['business_context'] ) ? $params['business_context'] : '';

		// If random, pick a section type from API components
		if ( $section_type === 'random' ) {
			// Get components from API
			$all_components = $this->get_available_components();
			
			// Build list of all available components
			$available_components = array();
			foreach ( $all_components as $type => $components ) {
				// Exclude contact forms from random selection
				foreach ( $components as $slug => $component ) {
					if ( $slug !== 'contact-form' && $slug !== 'signup-form' ) {
						$available_components[] = $slug;
					}
				}
			}
			
			if ( ! empty( $available_components ) ) {
				$section_type = $available_components[ array_rand( $available_components ) ];
				Buildfuly_Logger::debug( 'BuildfulyAI: Random component selected: ' . $section_type );
			} else {
				// Fallback to features if no components available
				$section_type = 'features';
			}
		}
		
		$keyword_array = array_map( 'trim', explode( ',', $keywords ) );
		$primary_keyword = $keyword_array[0];
		
		// DELETE: These settings no longer exist in the UI
		$generation_mode = get_option( 'buildfuly_generation_mode', 'keyword' );
		$generation_instructions = get_option( 'buildfuly_generation_instructions', '' );
		
		// DELETE: Semantic keyword generation no longer needed - API handles this internally
		// The Azure API already generates keyword variations when we pass it the main keywords
		// $semantic_keywords = $this->generate_semantic_keywords( $primary_keyword );
		
		// DELETE: This entire section is unnecessary - we just send ALL keywords to API
		// The API is smart enough to distribute them across the content naturally
		// No need to pre-select which keyword to use for which component type
		
		// Generate HTML via API (100% API-driven, no local generation)
		Buildfuly_Logger::debug( 'BuildfulyAI: Generating component via API: ' . $section_type );
		
		$business_name = get_option( 'buildfuly_business_name', '' );
		$location = get_option( 'buildfuly_city', '' );
		
		// Call API client to generate component
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-api-client.php';
		$api_client = new Buildfuly_API_Client();
		
		$response = $api_client->generate_component( 
			$section_type, 
			implode( ', ', $keyword_array ), 
			$image_id, 
			$business_name, 
			$location,
			$business_context
		);
		
		if ( is_wp_error( $response ) ) {
			Buildfuly_Logger::debug( 'BuildfulyAI: API error - ' . $response->get_error_message() );
			return false;
		}
		
		// API must return HTML - no local fallback
		if ( empty( $response['html'] ) ) {
			Buildfuly_Logger::debug( 'BuildfulyAI: API returned empty HTML for component: ' . $section_type );
			return false;
		}
		
		$html = $response['html'];
		$ai_content = isset( $response['ai_content'] ) ? $response['ai_content'] : null;
		$ai_word_count = isset( $response['word_count'] ) ? intval( $response['word_count'] ) : 0;
		$usage = isset( $response['usage'] ) ? $response['usage'] : null;
		
		Buildfuly_Logger::debug( 'BuildfulyAI: Successfully generated component with ' . $ai_word_count . ' words' );
		
		if ( $usage ) {
			Buildfuly_Logger::debug( 'BuildfulyAI: Usage stats - Remaining: ' . $usage['remaining'] . ' pages, ' . $usage['remaining_words'] . ' words' );
		}

		$result = array(
			'type'       => $section_type,
			'keywords'   => $keywords,
			'html'       => $html,
			'content'    => $html, // Alias for backward compatibility
			'ai_content' => $ai_content, // Include AI content for reference
			'word_count' => $ai_word_count, // Actual words generated by AI
			'image_id'   => $image_id, // Include image_id so it's stored in the section
		);
		
		// Include usage stats if available from API
		if ( $usage ) {
			$result['usage'] = $usage;
		}
		
		return $result;
	}

	/**
	 * Generate HTML for a specific section type with keyword optimization.
	 *
	 * @since    1.0.1
	 * @param    string    $type          Section type
	 * @param    array     $keywords      Keywords array
	 * @param    int       $image_id      Image ID (optional)
	 * @param    array     $ai_content    AI-generated content fields (optional)
	 * @return   string|false             HTML content or false on error
	 */
	/**
	 * Get API base URL.
	 *
	 * @since    1.0.7
	 * @return   string    API base URL
	 */
	private function get_api_url() {
		// Check if custom API URL is defined
		if ( defined( 'BUILDFULY_API_URL' ) ) {
			return trailingslashit( BUILDFULY_API_URL );
		}
		
		// Default to production API
		return 'https://build.fuly.io/';
	}

	/**
	 * Generate section HTML by calling the API.
	 * Replaces the old local component loading with API-based generation.
	 *
	 * @since    1.0.7
	 * @param    string    $type         Component type
	 * @param    array     $keywords     Keywords array
	 * @param    int       $image_id     Image ID (optional)
	 * @param    array     $ai_content   AI-generated content (optional)
	 * @return   string|false             Generated HTML or false on failure
	 */
	public function generate_section_html( $type, $keywords, $image_id = 0, $ai_content = null ) {
		// ALL generation MUST go through API - no local rendering
		require_once plugin_dir_path( __FILE__ ) . 'class-buildfuly-api-client.php';
		$api_client = new Buildfuly_API_Client();
		
		// Get business details for better AI context
		$business_name = get_option( 'buildfuly_business_name', '' );
		$city = get_option( 'buildfuly_city', '' );
		$state = get_option( 'buildfuly_state', '' );
		$location = trim( "{$city}, {$state}" );
		
		Buildfuly_Logger::debug( "BuildfulyAI: Generating component '{$type}' via API..." );
		
		// Call API to generate component (AI content + HTML rendering)
		$response = $api_client->generate_component(
			$type,
			$keywords,
			$image_id,
			$business_name,
			$location
		);
		
		// If API call failed, return false
		if ( is_wp_error( $response ) ) {
			$error_message = $response->get_error_message();
			Buildfuly_Logger::debug( 'BuildfulyAI: API error: ' . $error_message );
			return false;
		}
		
		// API must return HTML - no local fallback
		if ( empty( $response['html'] ) ) {
			Buildfuly_Logger::debug( "BuildfulyAI: API returned empty HTML for component: {$type}" );
			return false;
		}
		
		Buildfuly_Logger::debug( "BuildfulyAI: Component '{$type}' generated successfully via API" );
		return $response['html'];
	}

	/**
	 * Generate section HTML locally (fallback method).
	 * Used when API is unavailable or errors occur.
	 *
	 * @since    1.0.7
	 * @param    string    $type         Component type
	 * @param    array     $keywords     Keywords array
	 * @param    int       $image_id     Image ID (optional)
	 * @param    array     $ai_content   AI-generated content (optional)
	 * @return   string|false             Generated HTML or false on failure
	 */
	private function generate_section_html_local( $type, $keywords, $image_id = 0, $ai_content = null ) {
		try {
			// Load component helper functions
			require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/components/component-helpers.php';
			
			// Use component loader to find component file
			require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-component-loader.php';
			$component_loader = new Buildfuly_Component_Loader();
			$component = $component_loader->get_component( $type );
			
			if ( ! $component ) {
				Buildfuly_Logger::debug( "BuildfulyAI: Component '{$type}' not found in generate_section_html" );
				return false;
			}
			
			$component_file = $component['file'];
			
			if ( ! file_exists( $component_file ) ) {
				Buildfuly_Logger::debug( "BuildfulyAI: Component file does not exist: {$component_file}" );
				return false;
			}
			
			// Include and execute the component file with error handling
			// Component files have access to $keywords, $image_id, and $ai_content variables
			ob_start();
			$html = @include $component_file;
			$output = ob_get_clean();
			
			// Check for include errors
			if ( $html === false ) {
				Buildfuly_Logger::debug( "BuildfulyAI: Error including component file: {$component_file}" );
				return false;
			}
			
			// Return either the returned value or the output buffer
			$result = ! empty( $html ) ? $html : $output;
			
			// Validate result is not empty
			if ( empty( trim( $result ) ) ) {
				Buildfuly_Logger::debug( "BuildfulyAI: Component '{$type}' generated empty content" );
				return false;
			}
			
			// Log if style tag exists
			if ( strpos( $result, '<style>' ) !== false ) {
				Buildfuly_Logger::debug( 'BuildfulyAI: Component ' . $type . ' has <style> tag' );
			} else {
				Buildfuly_Logger::debug( 'BuildfulyAI: WARNING - Component ' . $type . ' missing <style> tag!' );
			}
			
			return $result;
			
		} catch ( Exception $e ) {
			Buildfuly_Logger::debug( "BuildfulyAI: Exception in generate_section_html for '{$type}': " . $e->getMessage() );
			return false;
		} catch ( Error $e ) {
			Buildfuly_Logger::debug( "BuildfulyAI: Fatal error in generate_section_html for '{$type}': " . $e->getMessage() );
			return false;
		}
	}
}

