<?php
/**
 * Define plugin constants and configuration
 *
 * @link       https://build.fuly.io
 * @since      1.0.5
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

class Buildfuly_Constants {

	// ============================================================
	// PRICING & TIER CONFIGURATION
	// ============================================================
	
	/**
	 * Words to page conversion ratio
	 * 600 words = 1 page for rate limiting purposes
	 */
	const WORDS_PER_PAGE = 600;

	/**
	 * Pricing tier definitions
	 * Starter tier uses total_pages (lifetime limit), paid tiers use pages_per_day
	 */
	const TIERS = array(
		'starter' => array(
			'name'           => 'Starter',
			'price_monthly'  => 0,
			'price_yearly'   => 0,
			'price_lifetime' => 0,
			'total_pages'    => 3,
			'sites_limit'    => 1,
			'features'       => array(
				'basic_templates',
				'wp_plugin',
				'community_support',
			),
			'cta'            => 'Try inside WordPress — free',
		),
		'pro' => array(
			'name'           => 'Pro',
			'price_monthly'  => 19,
			'price_yearly'   => 190,  // ~17/mo (2 months free)
			'price_lifetime' => 0,
			'pages_per_day'  => 5,
			'sites_limit'    => 1,
			'features'       => array(
				'extra_templates',
				'saas_templates',
				'priority_queue',
				'component_ai_edits',
				'email_support_48h',
			),
			'cta'            => 'For small businesses',
		),
		'growth' => array(
			'name'           => 'Growth',
			'price_monthly'  => 49,
			'price_yearly'   => 490,  // ~41/mo (2 months free)
			'price_lifetime' => 0,
			'pages_per_day'  => 10,
			'sites_limit'    => 1,
			'features'       => array(
				'premium_templates',
				'seo_auto_generation',
				'scheduling',
				'analytics_integration',
				'live_chat_support',
			),
			'cta'            => 'Scale your content',
		),
		'studio' => array(
			'name'           => 'Studio',
			'price_monthly'  => 149,
			'price_yearly'   => 1490, // ~124/mo (2 months free)
			'price_lifetime' => 499,
			'pages_per_day'  => 50,
			'sites_limit'    => -1,   // -1 = unlimited
			'features'       => array(
				'unlimited_sites',
				'team_seats',
				'white_label_export',
				'priority_support',
				'multi_domain',
				'advanced_api_access',
			),
			'cta'            => 'For agencies & teams',
		),
	);

	/**
	 * Rate limiting - how long to wait between page generations (seconds)
	 */
	const RATE_LIMIT_COOLDOWN = array(
		'starter' => 86400,  // 24 hours (1 per day)
		'pro'     => 17280,  // ~4.8 hours (5 per day)
		'growth'  => 8640,   // ~2.4 hours (10 per day)
		'studio'  => 1728,   // ~28 minutes (50 per day)
	);

	/**
	 * Template access by tier
	 */
	const TEMPLATE_ACCESS = array(
		'starter' => array('basic'),
		'pro'     => array('basic', 'saas', 'professional'),
		'growth'  => array('basic', 'saas', 'professional', 'premium'),
		'studio'  => array('basic', 'saas', 'professional', 'premium', 'agency'),
	);

	// ============================================================
	// API CONFIGURATION
	// ============================================================
	
	/**
	 * Azure API endpoint (production)
	 * Set in wp-config.php: define('BUILDFULY_API_URL', 'https://api.build.fuly.io');
	 */
	const API_ENDPOINT_PRODUCTION = 'https://build.fuly.io/api';
	
	/**
	 * Local API endpoint (development)
	 */
	const API_ENDPOINT_LOCAL = 'http://localhost:8000/api';

	/**
	 * API timeout in seconds
	 */
	const API_TIMEOUT = 30;

	/**
	 * API retry attempts
	 */
	const API_RETRY_ATTEMPTS = 3;

	// ============================================================
	// ONBOARDING CONFIGURATION
	// ============================================================

	/**
	 * Number of suggested pages to generate during onboarding
	 * 
	 * @var int
	 */
	const ONBOARDING_PAGE_SUGGESTIONS = 10;

	/**
	 * Maximum pages a user can have in onboarding
	 * 
	 * @var int
	 */
	const ONBOARDING_MAX_PAGES = 20;

	/**
	 * Default page that's always auto-selected (none - AI generates all including Home)
	 * 
	 * @var array
	 */
	const DEFAULT_PAGES_WITH_KEYWORDS = array();

	/**
	 * Number of AI-suggested pages to auto-select (none - user picks)
	 * 
	 * @var int
	 */
	const AUTO_SELECT_COUNT = 0;

	/**
	 * Common service-based page types for keyword generation
	 * 
	 * @var array
	 */
	const COMMON_PAGE_TYPES = array(
		'About Us',
		'Services',
		'Contact',
		'Testimonials',
		'FAQ',
		'Blog',
		'Portfolio',
		'Pricing',
		'Book Now',
		'Get Quote',
	);

	/**
	 * Available review platforms with their display names and URLs
	 * 
	 * @var array
	 */
	const REVIEW_PLATFORMS = array(
		'google' => array(
			'name' => 'Google',
			'url'  => 'https://www.google.com/search?q=%s+reviews',
		),
		'yelp' => array(
			'name' => 'Yelp',
			'url'  => 'https://www.yelp.com/biz/%s',
		),
		'facebook' => array(
			'name' => 'Facebook',
			'url'  => 'https://www.facebook.com/%s/reviews',
		),
		'trustpilot' => array(
			'name' => 'Trustpilot',
			'url'  => 'https://www.trustpilot.com/review/%s',
		),
	);

	/**
	 * Get number of page suggestions for onboarding
	 * 
	 * @return int
	 */
	public static function get_onboarding_page_count() {
		return apply_filters( 'buildfuly_onboarding_page_count', self::ONBOARDING_PAGE_SUGGESTIONS );
	}

	/**
	 * Get default page suggestions
	 * 
	 * @return array
	 */
	public static function get_default_pages() {
		return apply_filters( 'buildfuly_default_pages', self::DEFAULT_PAGES_WITH_KEYWORDS );
	}
	
	/**
	 * Get auto-select count for AI-suggested pages
	 * 
	 * @return int
	 */
	public static function get_auto_select_count() {
		return apply_filters( 'buildfuly_auto_select_count', self::AUTO_SELECT_COUNT );
	}

	/**
	 * Get common page types
	 * 
	 * @return array
	 */
	public static function get_common_page_types() {
		return apply_filters( 'buildfuly_common_page_types', self::COMMON_PAGE_TYPES );
	}

	/**
	 * Get review platform settings from WordPress options
	 * Returns array with 'name' and 'url' for current review platform
	 * 
	 * @return array
	 */
	public static function get_review_settings() {
		$platform_key = get_option( 'buildfuly_review_platform', 'yelp' );
		$platform_key = strtolower( $platform_key );
		
		// If platform doesn't exist in our list, default to Yelp
		if ( ! isset( self::REVIEW_PLATFORMS[ $platform_key ] ) ) {
			$platform_key = 'yelp';
		}
		
		$platform = self::REVIEW_PLATFORMS[ $platform_key ];
		$business_name = get_option( 'buildfuly_business_name', '' );
		$review_url = get_option( 'buildfuly_review_url', '' );
		
		// If custom URL is set, use it; otherwise generate from platform
		if ( empty( $review_url ) && ! empty( $business_name ) ) {
			// Generate URL based on platform and business name
			$slug = sanitize_title( $business_name );
			$review_url = sprintf( $platform['url'], $slug );
		}
		
		return array(
			'platform' => $platform['name'],
			'url'      => $review_url,
		);
	}

	/**
	 * Get available review platforms
	 * 
	 * @return array
	 */
	public static function get_review_platforms() {
		return self::REVIEW_PLATFORMS;
	}

	// ============================================================
	// TIER & PRICING METHODS
	// ============================================================

	/**
	 * Get all pricing tiers
	 * 
	 * @return array
	 */
	public static function get_tiers() {
		return apply_filters( 'buildfuly_pricing_tiers', self::TIERS );
	}

	/**
	 * Get specific tier configuration
	 * 
	 * @param string $tier_slug Tier identifier (starter, pro, growth, studio)
	 * @return array|null
	 */
	public static function get_tier( $tier_slug ) {
		$tiers = self::get_tiers();
		return isset( $tiers[ $tier_slug ] ) ? $tiers[ $tier_slug ] : null;
	}

	/**
	 * Get current user's tier
	 * 
	 * @return string Default 'starter'
	 */
	public static function get_user_tier() {
		$tier = get_option( 'buildfuly_user_tier', 'starter' );
		return apply_filters( 'buildfuly_user_tier', $tier );
	}

	/**
	 * Get pages limit for current user
	 * Returns total_pages for starter tier, pages_per_day for paid tiers
	 * 
	 * @return int
	 */
	public static function get_pages_limit() {
		$tier = self::get_user_tier();
		$tier_config = self::get_tier( $tier );
		
		if ( ! $tier_config ) {
			return 3; // Default to starter limit
		}
		
		// Starter tier uses total_pages (lifetime limit)
		if ( $tier === 'starter' && isset( $tier_config['total_pages'] ) ) {
			return $tier_config['total_pages'];
		}
		
		// Paid tiers use pages_per_day
		if ( isset( $tier_config['pages_per_day'] ) ) {
			return $tier_config['pages_per_day'];
		}
		
		// Fallback
		return isset( $tier_config['total_pages'] ) ? $tier_config['total_pages'] : 3;
	}

	/**
	 * @deprecated Use get_pages_limit() instead
	 */
	public static function get_total_pages_limit() {
		return self::get_pages_limit();
	}

	/**
	 * @deprecated Use get_pages_limit() instead
	 */
	public static function get_pages_per_day_limit() {
		return self::get_pages_limit();
	}

	/**
	 * Get words per page conversion ratio
	 * 
	 * @return int
	 */
	public static function get_words_per_page() {
		return apply_filters( 'buildfuly_words_per_page', self::WORDS_PER_PAGE );
	}

	/**
	 * Get total words per day limit for current user
	 * 
	 * @return int
	 */
	public static function get_words_per_day_limit() {
		return self::get_pages_per_day_limit() * self::get_words_per_page();
	}

	/**
	 * Get sites limit for current user
	 * 
	 * @return int -1 for unlimited
	 */
	public static function get_sites_limit() {
		$tier = self::get_user_tier();
		$tier_config = self::get_tier( $tier );
		return $tier_config ? $tier_config['sites_limit'] : 1;
	}

	/**
	 * Check if user has access to specific feature
	 * 
	 * @param string $feature Feature identifier
	 * @return bool
	 */
	public static function has_feature( $feature ) {
		$tier = self::get_user_tier();
		$tier_config = self::get_tier( $tier );
		
		if ( ! $tier_config ) {
			return false;
		}
		
		return in_array( $feature, $tier_config['features'], true );
	}

	/**
	 * Get available templates for current user
	 * 
	 * @return array
	 */
	public static function get_available_templates() {
		$tier = self::get_user_tier();
		return isset( self::TEMPLATE_ACCESS[ $tier ] ) ? self::TEMPLATE_ACCESS[ $tier ] : array( 'basic' );
	}

	/**
	 * Get API endpoint based on environment
	 * 
	 * @return string
	 */
	public static function get_api_endpoint() {
		// Check if custom endpoint is defined in wp-config.php
		if ( defined( 'BUILDFULY_API_URL' ) ) {
			return BUILDFULY_API_URL;
		}
		
		// Check if we're in development mode
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			return self::API_ENDPOINT_LOCAL;
		}
		
		return self::API_ENDPOINT_PRODUCTION;
	}

	/**
	 * Get API timeout
	 * 
	 * @return int Seconds
	 */
	public static function get_api_timeout() {
		return apply_filters( 'buildfuly_api_timeout', self::API_TIMEOUT );
	}

	// ============================================================
	// OPENAI CONFIGURATION
	// ============================================================

	/**
	 * OpenAI model to use for generation
	 */
	const OPENAI_MODEL = 'gpt-4-turbo-preview';

	/**
	 * Maximum tokens for OpenAI responses
	 */
	const OPENAI_MAX_TOKENS = 4000;

	/**
	 * Temperature for OpenAI generation (0.0 - 2.0)
	 * Lower = more focused, Higher = more creative
	 */
	const OPENAI_TEMPERATURE = 0.7;

	/**
	 * Get OpenAI model name
	 * 
	 * @return string
	 */
	public static function get_openai_model() {
		return apply_filters( 'buildfuly_openai_model', self::OPENAI_MODEL );
	}

	/**
	 * Get OpenAI max tokens
	 * 
	 * @return int
	 */
	public static function get_openai_max_tokens() {
		return apply_filters( 'buildfuly_openai_max_tokens', self::OPENAI_MAX_TOKENS );
	}

	/**
	 * Get OpenAI temperature
	 * 
	 * @return float
	 */
	public static function get_openai_temperature() {
		return apply_filters( 'buildfuly_openai_temperature', self::OPENAI_TEMPERATURE );
	}
}

