<?php
/**
 * Component Auto-Loader and Manager
 *
 * Automatically discovers components from filesystem and organizes by type
 *
 * @link       https://build.fuly.io
 * @since      1.0.5
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

class Buildfuly_Component_Loader {

	/**
	 * Components directory path
	 */
	private $components_dir;

	/**
	 * Cached components list
	 */
	private $components_cache = null;

	/**
	 * Initialize the loader
	 */
	public function __construct() {
		$this->components_dir = plugin_dir_path( __FILE__ ) . 'components/';
		Buildfuly_Logger::debug(( 'BuildfulyAI: Component Loader initialized. Base dir: ' . $this->components_dir );
		Buildfuly_Logger::debug(( 'BuildfulyAI: Components dir exists: ' . ( is_dir( $this->components_dir ) ? 'YES' : 'NO' ) );
	}

	/**
	 * Auto-discover all components from filesystem
	 *
	 * Reads component files and extracts metadata from headers
	 *
	 * @return array Components organized by type
	 */
	public function discover_components() {
		if ( $this->components_cache !== null ) {
			return $this->components_cache;
		}

		$components = array(
			'hero' => array(),
			'content' => array(),
			'form' => array(),
			'cta' => array(),
		);

		// Scan components directory and subdirectories (hero, content, cta)
		$subdirs = array( '', 'hero/', 'content/', 'cta/' );
		$total_files = 0;
		
		foreach ( $subdirs as $subdir ) {
			$files = glob( $this->components_dir . $subdir . '*.php' );
			
			Buildfuly_Logger::debug(( 'BuildfulyAI: Scanning ' . $this->components_dir . $subdir . '*.php' );
			Buildfuly_Logger::debug(( 'BuildfulyAI: Found ' . ( $files ? count( $files ) : 0 ) . ' files in ' . $subdir );
			
			if ( ! $files ) {
				continue;
			}
			
			$total_files += count( $files );

			foreach ( $files as $file ) {
				$filename = basename( $file, '.php' );
				
				// Skip helper files and function-based components
				if ( in_array( $filename, array( 'component-helpers', 'component-registry', 'pricing-card' ) ) ) {
					continue;
				}

				// Read file headers
				$metadata = $this->parse_component_headers( $file );
				
				$component_data = array(
					'slug' => $filename,
					'name' => $metadata['name'],
					'file' => $file,
					'type' => $metadata['type'],
					'ai_support' => $metadata['ai_support'],
					'ai_prompt' => $metadata['ai_prompt'],
					'requires_image' => $metadata['requires_image'],
					'uses_buttons' => $metadata['uses_buttons'],
					'description' => $metadata['description'],
				);

				// Add to appropriate category
				$type = $metadata['type'];
				if ( isset( $components[ $type ] ) ) {
					$components[ $type ][ $filename ] = $component_data;
					Buildfuly_Logger::debug(( "BuildfulyAI: Registered '{$filename}' as type '{$type}'" );
				} else {
					// Default to content if unknown type
					$components['content'][ $filename ] = $component_data;
					Buildfuly_Logger::debug(( "BuildfulyAI: Registered '{$filename}' as content (unknown type: {$type})" );
				}
			}
		}

		$this->components_cache = $components;
		
		Buildfuly_Logger::debug(( 'BuildfulyAI: Discovered ' . $total_files . ' components - Hero: ' . count( $components['hero'] ) . ', Content: ' . count( $components['content'] ) . ', CTA: ' . count( $components['cta'] ) . ', Form: ' . count( $components['form'] ) );

		return $components;
	}

	/**
	 * Parse component file headers
	 *
	 * Expected format at top of component file:
	 * 
	 * @component-type: hero|content|form
	 * @component-name: Hero Modern
	 * @ai-support: true
	 * @requires-image: false
	 * @uses-buttons: 1 (or 2 for primary + secondary)
	 * @description: Modern hero section with split layout
	 * @ai-prompt: Write a modern hero section for a {business} business. Provide JSON: {...}
	 *
	 * @param string $file Component file path
	 * @return array Metadata array
	 */
	private function parse_component_headers( $file ) {
		$content = file_get_contents( $file );
		$filename = basename( $file, '.php' );

		// Default metadata
		$metadata = array(
			'name' => ucwords( str_replace( '-', ' ', $filename ) ),
			'type' => $this->guess_component_type( $filename ),
			'ai_support' => true,
			'ai_prompt' => '',
			'requires_image' => false,
			'uses_buttons' => 0,
			'description' => '',
			'estimated_words' => 350, // Default estimate
		);

		// Parse @component-type
		if ( preg_match( '/@component-type:\s*(\w+)/i', $content, $matches ) ) {
			$metadata['type'] = strtolower( trim( $matches[1] ) );
		}

		// Parse @component-name
		if ( preg_match( '/@component-name:\s*(.+)/i', $content, $matches ) ) {
			$metadata['name'] = trim( $matches[1] );
		}

		// Parse @ai-support
		if ( preg_match( '/@ai-support:\s*(true|false)/i', $content, $matches ) ) {
			$metadata['ai_support'] = strtolower( $matches[1] ) === 'true';
		}

		// Parse @requires-image
		if ( preg_match( '/@requires-image:\s*(true|false)/i', $content, $matches ) ) {
			$metadata['requires_image'] = strtolower( $matches[1] ) === 'true';
		}

		// Parse @uses-buttons
		if ( preg_match( '/@uses-buttons:\s*(\d+)/i', $content, $matches ) ) {
			$metadata['uses_buttons'] = (int) $matches[1];
		}

		// Parse @description
		if ( preg_match( '/@description:\s*(.+)/i', $content, $matches ) ) {
			$metadata['description'] = trim( $matches[1] );
		}

		// Parse @estimated-words
		if ( preg_match( '/@estimated-words:\s*(\d+)/i', $content, $matches ) ) {
			$metadata['estimated_words'] = (int) $matches[1];
		}

		// Parse @ai-prompt (can be multi-line, may contain special chars like {}, (), etc)
		if ( preg_match( '/@ai-prompt:\s*(.+?)(?=\n\s*\*\s*@|\n\s*\*\/)/s', $content, $matches ) ) {
			$metadata['ai_prompt'] = trim( preg_replace( '/\n\s*\*\s*/', ' ', $matches[1] ) );
		}

		return $metadata;
	}

	/**
	 * Guess component type from filename
	 */
	private function guess_component_type( $filename ) {
		// Hero patterns
		if ( preg_match( '/^hero/i', $filename ) ) {
			return 'hero';
		}

		// CTA patterns
		if ( preg_match( '/^cta/i', $filename ) ) {
			return 'hero'; // CTAs act like heroes at the end
		}

		// Form patterns
		if ( preg_match( '/form|contact|signup/i', $filename ) ) {
			return 'form';
		}

		// Default to content
		return 'content';
	}

	/**
	 * Get random components for page generation
	 *
	 * @param int $hero_count Number of hero components (default 1)
	 * @param int $content_count Number of content components (default 6)
	 * @param int $cta_count Number of CTA components (default 1)
	 * @return array Array of component slugs in order
	 */
	public function get_random_components( $hero_count = 1, $content_count = 6, $cta_count = 1 ) {
		$all_components = $this->discover_components();
		$selected = array();

		// Select random hero
		$heroes = array_keys( $all_components['hero'] );
		$hero_only = array_filter( $heroes, function( $h ) {
			return ! preg_match( '/^cta/i', $h ); // Exclude CTAs from hero pool
		});
		
		if ( ! empty( $hero_only ) ) {
			$random_heroes = array_rand( array_flip( $hero_only ), min( $hero_count, count( $hero_only ) ) );
			$selected = array_merge( $selected, (array) $random_heroes );
		}

		// Select random content components
		$content_components = array_keys( $all_components['content'] );
		if ( ! empty( $content_components ) ) {
			$count = min( $content_count, count( $content_components ) );
			if ( $count > 0 ) {
				$random_content = (array) array_rand( array_flip( $content_components ), $count );
				$selected = array_merge( $selected, $random_content );
			}
		}

		// Select random CTA
		$ctas = array_filter( $heroes, function( $h ) {
			return preg_match( '/^cta/i', $h );
		});
		
		if ( ! empty( $ctas ) ) {
			$random_ctas = array_rand( array_flip( $ctas ), min( $cta_count, count( $ctas ) ) );
			$selected = array_merge( $selected, (array) $random_ctas );
		}

		Buildfuly_Logger::debug(( 'BuildfulyAI: Generated random layout - Hero: ' . $hero_count . ', Content: ' . count( $random_content ?? array() ) . ', CTA: ' . $cta_count );

		return $selected;
	}

	/**
	 * Get all components organized by type (for admin dropdown)
	 */
	public function get_components_by_type() {
		return $this->discover_components();
	}

	/**
	 * Get component metadata by slug
	 */
	public function get_component( $slug ) {
		$all = $this->discover_components();
		
		foreach ( $all as $type => $components ) {
			if ( isset( $components[ $slug ] ) ) {
				return $components[ $slug ];
			}
		}

		Buildfuly_Logger::debug( "BuildfulyAI: Component '{$slug}' not found." );
		Buildfuly_Logger::debug_data( 'BuildfulyAI: Available components', array_keys( array_merge( ...array_values( $all ) ) ) );
		return null;
	}

	/**
	 * Get AI prompt for component
	 */
	public function get_component_prompt( $slug ) {
		$component = $this->get_component( $slug );
		
		if ( ! $component || empty( $component['ai_prompt'] ) ) {
			return null;
		}

		return $component['ai_prompt'];
	}

	/**
	 * Load and render component
	 */
	public function render_component( $slug, $keywords, $ai_content = null, $image_id = 0 ) {
		$component = $this->get_component( $slug );
		
		if ( ! $component ) {
			return '<!-- Component not found: ' . esc_html( $slug ) . ' -->';
		}

		// Include component file
		ob_start();
		include $component['file'];
		$output = ob_get_clean();

		return $output;
	}

	/**
	 * Get CTA buttons from settings with smart handling
	 *
	 * Returns array with primary button always, secondary only if configured
	 *
	 * @return array Array with 'primary' and optionally 'secondary' keys
	 */
	public static function get_cta_buttons() {
		$cta_buttons = json_decode( get_option( 'buildfuly_cta_buttons', '[]' ), true );
		
		$result = array();

		if ( ! empty( $cta_buttons ) && is_array( $cta_buttons ) ) {
			// Primary button
			if ( isset( $cta_buttons[0] ) ) {
				$result['primary'] = array(
					'text' => isset( $cta_buttons[0]['text'] ) ? $cta_buttons[0]['text'] : 'Get Started',
					'url' => isset( $cta_buttons[0]['url'] ) ? $cta_buttons[0]['url'] : '#contact',
					'style' => isset( $cta_buttons[0]['style'] ) ? $cta_buttons[0]['style'] : 'primary',
				);
			}

			// Secondary button (only if exists)
			if ( isset( $cta_buttons[1] ) ) {
				$result['secondary'] = array(
					'text' => isset( $cta_buttons[1]['text'] ) ? $cta_buttons[1]['text'] : 'Learn More',
					'url' => isset( $cta_buttons[1]['url'] ) ? $cta_buttons[1]['url'] : '#learn-more',
					'style' => isset( $cta_buttons[1]['style'] ) ? $cta_buttons[1]['style'] : 'secondary',
				);
			}
		} else {
			// Fallback
			$result['primary'] = array(
				'text' => 'Get Started',
				'url' => '#contact',
				'style' => 'primary',
			);
		}

		return $result;
	}
}
