<?php

/**
 * REST API endpoints handler
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

/**
 * REST API endpoints handler.
 *
 * Defines REST API endpoints for content generation, styling, and data management.
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @author     Buildfuly Team
 */
class Buildfuly_API {

	/**
	 * The plugin name.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The plugin name.
	 */
	private $plugin_name;

	/**
	 * The plugin version.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The plugin version.
	 */
	private $version;

	/**
	 * The content generator.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Buildfuly_Content_Generator    $content_generator    Content generator instance.
	 */
	private $content_generator;

	/**
	 * The style manager.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Buildfuly_Style_Manager    $style_manager    Style manager instance.
	 */
	private $style_manager;

	/**
	 * The database handler.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Buildfuly_Database    $db    Database handler instance.
	 */
	private $db;

	/**
	 * Initialize the class.
	 *
	 * @since    1.0.0
	 * @param    string    $plugin_name    The plugin name.
	 * @param    string    $version        The plugin version.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->content_generator = new Buildfuly_Content_Generator();
		$this->style_manager = new Buildfuly_Style_Manager();
		$this->db = new Buildfuly_Database();
	}

	/**
	 * Register REST API routes.
	 *
	 * @since    1.0.0
	 */
	public function register_routes() {
		$namespace = 'buildfuly/v1';

		// Content section generation - NEW ENDPOINT
		register_rest_route( $namespace, '/content/generate-section', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'generate_section' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		// Generate keywords from page description - NEW ENDPOINT
		register_rest_route( $namespace, '/content/generate-keywords', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'generate_keywords' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		// AI page type analysis - NEW ENDPOINT
		register_rest_route( $namespace, '/content/analyze-page-type', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'analyze_page_type' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		// SEO metadata generation - NEW ENDPOINT
		register_rest_route( $namespace, '/content/generate-seo', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'generate_seo_metadata' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		// Content generation routes
		register_rest_route( $namespace, '/content/generate', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'generate_content' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		register_rest_route( $namespace, '/content', array(
			'methods'             => 'GET',
			'callback'            => array( $this, 'get_content' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		register_rest_route( $namespace, '/content/(?P<key>[a-zA-Z0-9._-]+)', array(
			'methods'             => 'PUT',
			'callback'            => array( $this, 'update_content_item' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		register_rest_route( $namespace, '/content/component/(?P<component>[a-zA-Z0-9._-]+)', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'regenerate_component' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		// Styling routes
		register_rest_route( $namespace, '/styles/generate', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'generate_styles' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		register_rest_route( $namespace, '/styles', array(
			'methods'             => 'GET',
			'callback'            => array( $this, 'get_styles' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		register_rest_route( $namespace, '/styles/(?P<component>[a-zA-Z0-9._-]+)', array(
			'methods'             => 'PUT',
			'callback'            => array( $this, 'update_component_styles' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		register_rest_route( $namespace, '/styles/property', array(
			'methods'             => 'PUT',
			'callback'            => array( $this, 'update_style_property' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		// History route
		register_rest_route( $namespace, '/history', array(
			'methods'             => 'GET',
			'callback'            => array( $this, 'get_history' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		// Theme integration route
		register_rest_route( $namespace, '/theme/info', array(
			'methods'             => 'GET',
			'callback'            => array( $this, 'get_theme_info' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );

		// Component word estimates route (for pre-budget checking)
		register_rest_route( $namespace, '/components/word-estimates', array(
			'methods'             => 'GET',
			'callback'            => array( $this, 'get_component_word_estimates' ),
			'permission_callback' => array( $this, 'check_admin_permissions' ),
		) );
	}

	/**
	 * Check admin permissions.
	 *
	 * @since    1.0.0
	 * @return   bool    True if user has admin permissions
	 */
	public function check_admin_permissions() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * Generate a single content section with SEO keywords.
	 *
	 * @since    1.0.1
	 * @param    WP_REST_Request    $request    Request object
	 * @return   WP_REST_Response               Response object
	 */
	public function generate_section( $request ) {
		$params = $request->get_json_params();
		$keywords = isset( $params['keywords'] ) ? sanitize_text_field( $params['keywords'] ) : '';
		$section_type = isset( $params['section_type'] ) ? sanitize_text_field( $params['section_type'] ) : 'hero-split';
		$image_id = isset( $params['image_id'] ) ? intval( $params['image_id'] ) : 0;

		// Load API client class once at the beginning
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-api-client.php';
		$api_client = new Buildfuly_API_Client();

		// Handle "random" component types - pick from available list using cached components
		if ( $section_type === 'random' || strpos( $section_type, 'random-' ) === 0 ) {
			Buildfuly_Logger::debug( 'BuildfulyAI: Random component requested: ' . $section_type );
			
			$components = $api_client->get_components(); // Uses cached list
			
			if ( ! is_wp_error( $components ) && ! empty( $components ) ) {
				$all_component_slugs = array();
				
				// Handle specific random types
				if ( $section_type === 'random-hero' ) {
					// Only hero components (excluding CTA)
					if ( isset( $components['hero'] ) && is_array( $components['hero'] ) ) {
						foreach ( $components['hero'] as $slug => $component_data ) {
							if ( strpos( $slug, 'cta' ) === false ) {
								$all_component_slugs[] = $slug;
							}
						}
					}
					Buildfuly_Logger::debug( 'BuildfulyAI: Random hero - available: ' . implode( ', ', $all_component_slugs ) );
				} elseif ( $section_type === 'random-content' ) {
					// Only content components
					if ( isset( $components['content'] ) && is_array( $components['content'] ) ) {
						$all_component_slugs = array_keys( $components['content'] );
					}
					Buildfuly_Logger::debug( 'BuildfulyAI: Random content - available: ' . implode( ', ', $all_component_slugs ) );
				} elseif ( $section_type === 'random-cta' ) {
					// Only CTA components
					if ( isset( $components['cta'] ) && is_array( $components['cta'] ) ) {
						$all_component_slugs = array_keys( $components['cta'] );
					}
					Buildfuly_Logger::debug( 'BuildfulyAI: Random CTA - available: ' . implode( ', ', $all_component_slugs ) );
				} else {
					// Original "random" - any component except contact-form
					foreach ( $components as $category => $category_components ) {
						if ( is_array( $category_components ) ) {
							foreach ( $category_components as $slug => $component_data ) {
								if ( $slug !== 'contact-form' ) {
									$all_component_slugs[] = $slug;
								}
							}
						}
					}
					Buildfuly_Logger::debug( 'BuildfulyAI: Random (any) - available: ' . implode( ', ', $all_component_slugs ) );
				}
				
				if ( ! empty( $all_component_slugs ) ) {
					$section_type = $all_component_slugs[ array_rand( $all_component_slugs ) ];
					Buildfuly_Logger::debug( 'BuildfulyAI: Random component selected: ' . $section_type );
				} else {
					Buildfuly_Logger::debug( 'BuildfulyAI: No components after filtering, using fallback' );
					$section_type = 'hero-split'; // Fallback
				}
			} else {
				Buildfuly_Logger::debug( 'BuildfulyAI: get_components failed or empty, using fallback' );
				if ( is_wp_error( $components ) ) {
					Buildfuly_Logger::debug( 'BuildfulyAI: Error: ' . $components->get_error_message() );
				}
				$section_type = 'hero-split'; // Fallback
			}
		}

		// Get business context from settings
		$business_name = get_option( 'buildfuly_business_name', '' );
		$business_description = get_option( 'buildfuly_business_description', '' );
		$location = get_option( 'buildfuly_business_location', '' );
		$business_context = '';
		if ( ! empty( $business_description ) ) {
			$business_context = $business_description;
		} elseif ( ! empty( $business_name ) ) {
			$business_context = $business_name;
		}

		Buildfuly_Logger::debug( 'BuildfulyAI: API received - Type: ' . $section_type . ', Keywords: ' . $keywords . ', Context: ' . substr( $business_context, 0, 50 ) );

		if ( empty( $keywords ) ) {
			return new WP_REST_Response( array(
				'success' => false,
				'message' => __( 'Keywords are required', 'buildfuly' ),
			), 400 );
		}

		try {
			// Generate component through Azure API (using the $api_client created above)
			$result = $api_client->generate_component(
				$section_type,
				$keywords,
				$image_id,
				$business_name,
				$location,
				$business_context
			);
			
			// Check if result is a WP_Error
			if ( is_wp_error( $result ) ) {
				$error_message = $result->get_error_message();
				
				// If it's a rate limit error, try to use cached component
				if ( strpos( $error_message, 'limit' ) !== false || strpos( $error_message, 'credits' ) !== false ) {
					$cached_component = $this->get_cached_component( $section_type );
					
					if ( $cached_component ) {
						Buildfuly_Logger::debug( 'BuildfulyAI: Rate limit hit - serving cached component for ' . $section_type );
						
						return new WP_REST_Response( array(
							'success' => true,
							'data'    => array(
								'html'       => $cached_component['html'],
								'type'       => $section_type,
								'ai_content' => $cached_component['ai_content'],
								'word_count' => 0,
								'usage'      => null,
								'from_cache' => true,
							),
							'message' => __( 'You\'ve reached your daily limit. Here\'s a sample component you can customize.', 'buildfuly' ),
						), 200 );
					}
				}
				
				throw new Exception( $error_message );
			}
			
			if ( ! $result || ! isset( $result['html'] ) ) {
				throw new Exception( 'Failed to generate section content' );
			}
			
			// Cache this successful generation for future use
			$this->cache_component( $section_type, $result );
			
			return new WP_REST_Response( array(
				'success' => true,
				'data'    => array(
					'html'       => $result['html'],
					'type'       => $section_type,
					'ai_content' => isset( $result['ai_content'] ) ? $result['ai_content'] : null,
					'word_count' => isset( $result['word_count'] ) ? $result['word_count'] : 0,
					'usage'      => isset( $result['usage'] ) ? $result['usage'] : null,
				),
				'message' => __( 'Section generated successfully', 'buildfuly' ),
			), 200 );
			
		} catch ( Exception $e ) {
			Buildfuly_Logger::debug( 'BuildfulyAI: Section generation error: ' . $e->getMessage() );
			
			return new WP_REST_Response( array(
				'success' => false,
				'message' => $e->getMessage(),
			), 500 );
		}
	}
	
	/**
	 * Cache a generated component for future use when out of credits
	 *
	 * @param string $component_type Component type
	 * @param array  $result Generation result
	 */
	private function cache_component( $component_type, $result ) {
		$cache_key = 'buildfuly_cached_component_' . $component_type;
		
		$cached_data = array(
			'html'       => $result['html'],
			'ai_content' => isset( $result['ai_content'] ) ? $result['ai_content'] : null,
			'cached_at'  => time(),
		);
		
		// Cache for 30 days
		set_transient( $cache_key, $cached_data, 30 * DAY_IN_SECONDS );
	}
	
	/**
	 * Get a cached component
	 *
	 * @param string $component_type Component type
	 * @return array|null Cached component or null
	 */
	private function get_cached_component( $component_type ) {
		$cache_key = 'buildfuly_cached_component_' . $component_type;
		$cached = get_transient( $cache_key );
		
		if ( $cached !== false ) {
			return $cached;
		}
		
		return null;
	}

	/**
	 * Generate SEO keywords from page description.
	 *
	 * @since    1.0.3
	 * @param    WP_REST_Request    $request    The REST request object
	 * @return   WP_REST_Response               The response
	 */
	public function generate_keywords( $request ) {
		$params = $request->get_json_params();
		$description = isset( $params['description'] ) ? sanitize_text_field( $params['description'] ) : '';
		$page_title = isset( $params['page_title'] ) ? sanitize_text_field( $params['page_title'] ) : '';
		$generate_seo = isset( $params['generate_seo'] ) ? (bool) $params['generate_seo'] : false;

		if ( empty( $description ) ) {
			return new WP_REST_Response( array(
				'success' => false,
				'message' => __( 'Description is required', 'buildfuly' ),
			), 400 );
		}

		try {
			// Get business info from settings
			$business_name = get_option( 'buildfuly_business_name', '' );
			$business_description = get_option( 'buildfuly_business_description', '' );
			$location = get_option( 'buildfuly_business_location', '' );
			
			// Call Azure API to generate keywords
			$api_url = defined( 'BUILDFULY_API_URL' ) ? trailingslashit( BUILDFULY_API_URL ) : 'https://build.fuly.io/';
			$api_url .= 'api/keywords/generate';
			
			$request_body = array(
				'description' => $description,
			);
			
			// Add business context if available
			if ( ! empty( $business_name ) ) {
				$request_body['business_name'] = $business_name;
			}
			if ( ! empty( $business_description ) ) {
				$request_body['business_description'] = $business_description;
			}
			if ( ! empty( $location ) ) {
				$request_body['location'] = $location;
			}
			
			// Request SEO metadata generation if needed
			if ( $generate_seo && ! empty( $page_title ) ) {
				$request_body['generate_seo'] = true;
				$request_body['page_title'] = $page_title;
			}
			
			$response = wp_remote_post( $api_url, array(
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'body' => wp_json_encode( $request_body ),
				'timeout' => 15,
			) );

			if ( is_wp_error( $response ) ) {
				throw new Exception( $response->get_error_message() );
			}

			$body = wp_remote_retrieve_body( $response );
			$data = json_decode( $body, true );

			if ( empty( $data['success'] ) || empty( $data['keywords'] ) ) {
				// Fallback: extract keywords from description
				$keywords = $this->extract_keywords_from_description( $description );
				$seo = null;
			} else {
				$keywords = is_array( $data['keywords'] ) ? $data['keywords'] : explode( ',', $data['keywords'] );
				$seo = isset( $data['seo'] ) ? $data['seo'] : null;
			}

			$response_data = array(
				'success' => true,
				'data'    => array(
					'keywords' => $keywords,
				),
				'message' => __( 'Keywords generated successfully', 'buildfuly' ),
			);
			
			// Include SEO data if generated
			if ( $seo ) {
				$response_data['data']['seo'] = $seo;
			}

			return new WP_REST_Response( $response_data, 200 );
		} catch ( Exception $e ) {
			// Fallback to simple keyword extraction
			$keywords = $this->extract_keywords_from_description( $description );
			
			return new WP_REST_Response( array(
				'success' => true,
				'data'    => array(
					'keywords' => $keywords,
				),
				'message' => __( 'Keywords extracted from description', 'buildfuly' ),
			), 200 );
		}
	}

	/**
	 * Extract keywords from description (fallback method).
	 *
	 * @since    1.0.3
	 * @param    string    $description    The page description
	 * @return   array                     Array of keywords
	 */
	private function extract_keywords_from_description( $description ) {
		// Keep the original description as the primary keyword
		$keywords = array( $description );
		
		// Add common variations
		$keywords[] = 'professional ' . $description;
		$keywords[] = $description . ' services';
		$keywords[] = 'best ' . $description;
		$keywords[] = $description . ' near me';
		$keywords[] = 'hire ' . $description;
		
		// Limit to 6 keywords
		return array_slice( $keywords, 0, 6 );
	}

	/**
	 * AI-powered page type analysis and section recommendation.
	 *
	 * @since    1.0.2
	 * @param    WP_REST_Request    $request    The REST request object
	 * @return   WP_REST_Response               The response
	 */
	public function analyze_page_type( $request ) {
		$params = $request->get_json_params();
		$page_type = isset( $params['page_type'] ) ? sanitize_text_field( $params['page_type'] ) : '';
		$keywords = isset( $params['keywords'] ) ? sanitize_text_field( $params['keywords'] ) : '';

		if ( empty( $page_type ) ) {
			return new WP_REST_Response( array(
				'success' => false,
				'message' => __( 'Page type is required', 'buildfuly' ),
			), 400 );
		}

		try {
			$sections = $this->determine_sections_for_page_type( $page_type, $keywords );

			return new WP_REST_Response( array(
				'success' => true,
				'data'    => array(
					'sections' => $sections,
					'reasoning' => $this->get_page_type_reasoning( $page_type ),
				),
				'message' => __( 'Page type analyzed successfully', 'buildfuly' ),
			), 200 );
		} catch ( Exception $e ) {
			return new WP_REST_Response( array(
				'success' => false,
				'message' => $e->getMessage(),
			), 500 );
		}
	}

	/**
	 * Determine optimal sections for a given page type.
	 *
	 * @since    1.0.2
	 * @param    string    $page_type    The type of page being created
	 * @param    string    $keywords     Optional keywords for context
	 * @return   array                   Array of section types
	 */
	private function determine_sections_for_page_type( $page_type, $keywords = '' ) {
		$page_type_lower = strtolower( $page_type );
		
		// Load components from API
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-api-client.php';
		$api_client = new Buildfuly_API_Client();
		$all_components = $api_client->get_components();
		
		if ( is_wp_error( $all_components ) ) {
			Buildfuly_Logger::debug( 'BuildfulyAI: Failed to load components from API - ' . $all_components->get_error_message() );
			// Fallback to basic structure
			$all_components = array(
				'hero' => array( 'hero-centered' => array( 'name' => 'Hero Centered' ) ),
				'content' => array( 'features' => array( 'name' => 'Features' ) ),
				'cta' => array( 'cta-banner' => array( 'name' => 'CTA Banner' ) ),
			);
		}
		
		// Get available components by category
		$heroes = isset( $all_components['hero'] ) ? array_keys( $all_components['hero'] ) : array();
		$content_components = isset( $all_components['content'] ) ? array_keys( $all_components['content'] ) : array();
		$cta_components = isset( $all_components['cta'] ) ? array_keys( $all_components['cta'] ) : array();
		
		// Filter out forms from content
		$content_only = array_filter( $content_components, function( $c ) {
			return ! in_array( $c, array( 'contact-form', 'signup-form' ) );
		});
		
		// Pick 1 random hero
		$selected_hero = ! empty( $heroes ) ? $heroes[ array_rand( $heroes ) ] : 'hero-centered';
		
		// Pick 1 random CTA from actual CTA components
		$selected_cta = ! empty( $cta_components ) ? $cta_components[ array_rand( $cta_components ) ] : 'cta-banner';
		
		// Determine content sections based on page type
		$content_count = 6; // Default
		$selected_content = array();
		
		if ( strpos( $page_type_lower, 'home' ) !== false || strpos( $page_type_lower, 'landing' ) !== false ) {
			// Homepage: Features, benefits, testimonials, stats
			$content_count = 6;
			$preferred = array( 'features', 'benefits', 'testimonial', 'stats' );
		} elseif ( strpos( $page_type_lower, 'about' ) !== false ) {
			// About: Stats, process, testimonials
			$content_count = 5;
			$preferred = array( 'stats', 'process', 'testimonial', 'benefits' );
		} elseif ( strpos( $page_type_lower, 'service' ) !== false || strpos( $page_type_lower, 'product' ) !== false ) {
			// Services: Features, benefits, process
			$content_count = 6;
			$preferred = array( 'features', 'benefits', 'process', 'bento-grid' );
		} elseif ( strpos( $page_type_lower, 'contact' ) !== false ) {
			// Contact: Benefits, FAQ
			$content_count = 3;
			$preferred = array( 'benefits', 'faq' );
		} elseif ( strpos( $page_type_lower, 'pricing' ) !== false || strpos( $page_type_lower, 'price' ) !== false ) {
			// Pricing: Features, benefits, FAQ
			$content_count = 5;
			$preferred = array( 'features', 'benefits', 'faq', 'testimonial' );
		} elseif ( strpos( $page_type_lower, 'testimonial' ) !== false || strpos( $page_type_lower, 'review' ) !== false ) {
			// Testimonials: Testimonials, stats
			$content_count = 4;
			$preferred = array( 'testimonial', 'stats', 'benefits' );
		} elseif ( strpos( $page_type_lower, 'faq' ) !== false ) {
			// FAQ: FAQ sections
			$content_count = 3;
			$preferred = array( 'faq', 'benefits' );
		} else {
			// Default varied layout
			$content_count = 6;
			$preferred = array();
		}
		
		// Build content section list with smart randomization
		$available_content = ! empty( $content_only ) ? $content_only : array( 'features', 'benefits', 'testimonial' );
		
		// First, add preferred components if they exist
		if ( ! empty( $preferred ) ) {
			foreach ( $preferred as $pref ) {
				if ( in_array( $pref, $available_content ) ) {
					$selected_content[] = $pref;
					// Remove from available to avoid duplicates
					$available_content = array_values( array_diff( $available_content, array( $pref ) ) );
					
					if ( count( $selected_content ) >= $content_count ) {
						break;
					}
				}
			}
		}
		
		// Fill remaining slots with random components
		if ( count( $selected_content ) < $content_count && ! empty( $available_content ) ) {
			$remaining = $content_count - count( $selected_content );
			shuffle( $available_content );
			$selected_content = array_merge( $selected_content, array_slice( $available_content, 0, $remaining ) );
		}
		
		// Build final layout: Hero + Content + CTA
		$layout = array_merge(
			array( $selected_hero ),
			$selected_content,
			array( $selected_cta )
		);
		
		Buildfuly_Logger::debug( 'BuildfulyAI: Generated layout for "' . $page_type . '": ' . implode( ', ', $layout ) );
		
		return $layout;
	}

	/**
	 * Get reasoning explanation for why certain sections were chosen.
	 *
	 * @since    1.0.2
	 * @param    string    $page_type    The type of page
	 * @return   string                  Human-readable reasoning
	 */
	private function get_page_type_reasoning( $page_type ) {
		$page_type_lower = strtolower( $page_type );
		
		if ( strpos( $page_type_lower, 'home' ) !== false || strpos( $page_type_lower, 'landing' ) !== false ) {
			return 'Homepage/Landing pages need: Hero to capture attention, Features to showcase offerings, Benefits to show value, Testimonials for trust, and CTA for conversion.';
		} elseif ( strpos( $page_type_lower, 'about' ) !== false ) {
			return 'About pages need: Hero to introduce the company, Stats to show credibility, Process to explain how you work, Testimonials for social proof, and CTA for next steps.';
		} elseif ( strpos( $page_type_lower, 'contact' ) !== false ) {
			return 'Contact pages need: Hero with clear contact info, Benefits of reaching out, FAQ for common questions, and CTA to encourage action.';
		} else {
			return 'Using a balanced layout with hero, features, benefits, and conversion elements.';
		}
	}

	/**
	 * Generate content endpoint.
	 *
	 * @since    1.0.0
	 * @param    WP_REST_Request    $request    Request object
	 * @return   WP_REST_Response               Response object
	 */
	public function generate_content( $request ) {
		// This endpoint is used by generateSectionFromKeyword()
		// It should work the same as generate_section()
		$params = $request->get_json_params();
		$keywords = isset( $params['keywords'] ) ? sanitize_text_field( $params['keywords'] ) : '';
		$component = isset( $params['component'] ) ? sanitize_text_field( $params['component'] ) : 'random';
		$image_id = isset( $params['image_id'] ) ? intval( $params['image_id'] ) : 0;
		$page_type = isset( $params['page_type'] ) ? sanitize_text_field( $params['page_type'] ) : '';
		$colors = isset( $params['colors'] ) ? $params['colors'] : array();

		// Get business context from settings
		$business_name = get_option( 'buildfuly_business_name', '' );
		$business_description = get_option( 'buildfuly_business_description', '' );
		$business_context = '';
		if ( ! empty( $business_description ) ) {
			$business_context = $business_description;
		} elseif ( ! empty( $business_name ) ) {
			$business_context = $business_name;
		}

		Buildfuly_Logger::debug( 'BuildfulyAI: /content/generate - Component: ' . $component . ', Keywords: ' . $keywords );

		if ( empty( $keywords ) ) {
			return new WP_REST_Response( array(
				'success' => false,
				'message' => __( 'Keywords are required', 'buildfuly' ),
			), 400 );
		}

		// DELETE: Old pre-check code removed - API now handles rate limiting automatically
		// The /api/components/generate endpoint checks limits before generating
		// and returns 429 error if limit exceeded

		try {
			$section_data = $this->content_generator->generate_seo_section( array(
				'keywords'         => $keywords,
				'section_type'     => $component,
				'image_id'         => $image_id,
				'business_context' => $business_context,
				'page_type'        => $page_type,
				'colors'           => $colors,
			) );

			// Check if section generation failed
			if ( $section_data === false ) {
				Buildfuly_Logger::debug( "BuildfulyAI: Component '{$component}' failed to generate" );
				return new WP_REST_Response( array(
						'success' => false,
						/* translators: %s: Component name (e.g., hero, testimonials, pricing) */
					'message' => sprintf( __( 'Failed to generate component "%s". Component may have errors.', 'buildfuly' ), $component ),
					'skip'    => true,
				), 500 );
			}
		} catch ( Exception $e ) {
			return new WP_REST_Response( array(
				'success' => false,
				'message' => $e->getMessage(),
			), 500 );
		}
	}
	
	/**
	 * Get history endpoint.
	 *
	 * @since    1.0.0
	 * @param    WP_REST_Request    $request    Request object
	 * @return   WP_REST_Response               Response object
	 */
	public function get_history( $request ) {
		$limit = $request->get_param( 'limit' );
		$limit = $limit ? intval( $limit ) : 50;

		$history = $this->db->get_history( $limit );

		return new WP_REST_Response( array(
			'success' => true,
			'data'    => $history,
		), 200 );
	}

	/**
	 * Get theme info endpoint.
	 *
	 * @since    1.0.0
	 * @param    WP_REST_Request    $request    Request object
	 * @return   WP_REST_Response               Response object
	 */
	public function get_theme_info( $request ) {
		$theme_integration = new Buildfuly_Theme_Integration();

		return new WP_REST_Response( array(
			'success'   => true,
			'data'      => array(
				'active_theme' => $theme_integration->get_active_theme(),
				'supported'    => $theme_integration->is_supported(),
			),
		), 200 );
	}

	/**
	 * Get component word estimates endpoint.
	 * Returns estimated word counts for each component type to enable
	 * pre-budget validation before generation.
	 *
	 * @since    1.0.2
	 * @param    WP_REST_Request    $request    Request object
	 * @return   WP_REST_Response               Response object with component estimates
	 */
	public function get_component_word_estimates( $request ) {
		// Check transient cache first
		$cached = get_transient( 'buildfuly_component_word_estimates' );
		if ( $cached !== false ) {
			return new WP_REST_Response( array(
				'success' => true,
				'data'    => $cached,
				'cached'  => true,
			), 200 );
		}
		
		// Load components from API
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-api-client.php';
		$api_client = new Buildfuly_API_Client();
		$all_components = $api_client->get_components();
		
		if ( is_wp_error( $all_components ) ) {
			return new WP_REST_Response( array(
				'success' => false,
				'error'   => $all_components->get_error_message(),
			), 500 );
		}
		
		$estimates = array();
		
		// Extract word estimates from each component's metadata
		foreach ( $all_components as $type => $components ) {
			foreach ( $components as $component_slug => $component_data ) {
				if ( isset( $component_data['estimated_words'] ) ) {
					$estimates[ $component_slug ] = intval( $component_data['estimated_words'] );
				}
			}
		}
		
		// Add default for 'random' type (average of all components)
		if ( ! empty( $estimates ) ) {
			$estimates['random'] = intval( array_sum( $estimates ) / count( $estimates ) );
		}
		
		// Cache for 1 hour
		set_transient( 'buildfuly_component_word_estimates', $estimates, HOUR_IN_SECONDS );
		
		return new WP_REST_Response( array(
			'success' => true,
			'data'    => $estimates,
			'cached'  => false,
		), 200 );
	}
}
