<?php
/**
 * API Client for Buildfuly Azure Backend
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

class Buildfuly_API_Client {

	/**
	 * API endpoint base URL
	 *
	 * @var string
	 */
	private $api_url;

	/**
	 * API timeout in seconds
	 *
	 * @var int
	 */
	private $timeout;

	/**
	 * Number of retry attempts
	 *
	 * @var int
	 */
	private $retry_attempts;

	/**
	 * JWT token for authentication
	 *
	 * @var string|null
	 */
	private $token;

	/**
	 * Initialize the API client
	 */
	public function __construct() {
		$this->api_url = Buildfuly_Constants::get_api_endpoint();
		$this->timeout = Buildfuly_Constants::get_api_timeout();
		$this->retry_attempts = Buildfuly_Constants::API_RETRY_ATTEMPTS;
		$this->token = get_option( 'buildfuly_api_token', null );
	}

	/**
	 * Authenticate with license key
	 *
	 * @param string $license_key User's license key
	 * @return array|WP_Error
	 */
	public function authenticate( $license_key ) {
		$endpoint = $this->api_url . '/api/auth';
		$site_url = get_site_url();
		
		Buildfuly_Logger::debug( 'BuildfulyAI: Authenticating with license: ' . $license_key );
		Buildfuly_Logger::debug( 'BuildfulyAI: Site URL: ' . $site_url );
		Buildfuly_Logger::debug( 'BuildfulyAI: API endpoint: ' . $endpoint );
		
		$response = wp_remote_post( $endpoint, array(
			'timeout' => $this->timeout,
			'headers' => array(
				'Content-Type' => 'application/json',
			),
			'body' => wp_json_encode( array(
				'license_key' => $license_key,
				'site_url' => $site_url,
			) ),
			'sslverify' => false,
			'reject_unsafe_urls' => false,
		) );

		if ( is_wp_error( $response ) ) {
			Buildfuly_Logger::debug( 'BuildfulyAI: Authentication WP_Error: ' . $response->get_error_message() );
			return $response;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );
		
		Buildfuly_Logger::debug( 'BuildfulyAI: Auth response - Status: ' . $status_code );
		Buildfuly_Logger::debug_data( 'BuildfulyAI: Auth response body', $body );

		if ( $status_code !== 200 ) {
			$error_message = 'Authentication failed';
			
			// Check for error message in different formats
			if ( isset( $body['error'] ) ) {
				$error_message = $body['error'];
			} elseif ( isset( $body['message'] ) ) {
				$error_message = $body['message'];
			}
			
			Buildfuly_Logger::debug( 'BuildfulyAI: Auth failed - ' . $error_message );
			
			return new WP_Error(
				'auth_failed',
				$error_message,
				array( 'status' => $status_code )
			);
		}

		// Store the JWT token
		if ( isset( $body['token'] ) ) {
			$this->token = $body['token'];
			update_option( 'buildfuly_api_token', $this->token );
		}

		return $body;
	}

	/**
	 * Generate a component via API
	 *
	 * @param string $component_type Component type (e.g., 'hero-section')
	 * @param array $keywords Keywords for the component
	 * @param int|null $image_id Optional image ID
	 * @param string|null $business_name Optional business name
	 * @param string|null $location Optional location
	 * @param string|null $business_context Optional business description for context (not mentioned in output)
	 * @return array|WP_Error
	 */
public function generate_component( $component_type, $keywords, $image_id = null, $business_name = null, $location = null, $business_context = null ) {
	if ( ! $this->token ) {
		return new WP_Error( 'no_token', 'Not authenticated. Please activate your license.' );
	}

	$endpoint = $this->api_url . '/api/components/generate';
	
	$params = array(
		'component_type' => $component_type,
		'keywords' => $keywords,
	);
	
	if ( $image_id ) {
		Buildfuly_Logger::debug( 'BuildfulyAI: Image ID provided: ' . $image_id );
		$params['image_id'] = $image_id;
		// Also send image URL since Azure API can't access WordPress database
		$image_url = wp_get_attachment_image_url( $image_id, 'large' );
		Buildfuly_Logger::debug( 'BuildfulyAI: Image URL retrieved: ' . ( $image_url ? $image_url : 'EMPTY' ) );
		if ( $image_url ) {
			$params['image_url'] = $image_url;
		}
	} else {
		Buildfuly_Logger::debug( 'BuildfulyAI: No image ID provided (value: ' . ( $image_id === null ? 'null' : $image_id ) . ')' );
	}
	
	if ( $business_name ) {
		$params['business_name'] = $business_name;
	}
	if ( $location ) {
		$params['location'] = $location;
	}
	if ( $business_context ) {
		$params['business_context'] = $business_context;
	}
		
		// Pass WordPress settings to API for components that need them
		$params['wordpress_settings'] = array(
			'form_embed' => get_option( 'buildfuly_contact_form_embed', '' ),
			'review_platform' => get_option( 'buildfuly_review_platform', 'Google' ),
			'review_url' => get_option( 'buildfuly_review_url', '' ),
			'cta_buttons' => json_decode( get_option( 'buildfuly_cta_buttons', '[]' ), true ),
		);
		
		return $this->make_authenticated_request( $endpoint, $params );
	}

	/**
	 * Get available components for user's tier
	 * Results are cached for 1 hour
	 *
	 * @param bool $force_refresh Force refresh from API
	 * @return array|WP_Error
	 */
	public function get_components( $force_refresh = false ) {
		if ( ! $this->token ) {
			return new WP_Error( 'no_token', 'Not authenticated. Please activate your license.' );
		}

		// Check cache first
		$cache_key = 'buildfuly_api_components';
		if ( ! $force_refresh ) {
			$cached = get_transient( $cache_key );
			if ( $cached !== false ) {
				return $cached;
			}
		}

		$endpoint = $this->api_url . '/api/components';
		$response = $this->make_authenticated_request( $endpoint, array(), 'GET' );
		
		if ( ! is_wp_error( $response ) && isset( $response['components'] ) ) {
			// Cache for 1 hour
			set_transient( $cache_key, $response['components'], HOUR_IN_SECONDS );
			return $response['components'];
		}
		
		return $response;
	}

	/**
	 * Validate license with server
	 *
	 * @param string $license_key License key to validate
	 * @return array|WP_Error
	 */
	public function validate_license( $license_key ) {
		$endpoint = $this->api_url . '/api/license/validate';
		
		$response = wp_remote_post( $endpoint, array(
			'timeout' => $this->timeout,
			'headers' => array(
				'Content-Type' => 'application/json',
			),
			'body' => wp_json_encode( array(
				'license_key' => $license_key,
				'site_url' => get_site_url(),
			) ),
		) );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( $status_code !== 200 ) {
			return new WP_Error(
				'validation_failed',
				$body['message'] ?? 'License validation failed',
				array( 'status' => $status_code )
			);
		}

		return $body;
	}

	/**
	 * Get usage statistics for current user
	 *
	 * @return array|WP_Error
	 */
	public function get_usage_stats() {
		Buildfuly_Logger::debug( 'BuildfulyAI: get_usage_stats called. Token exists: ' . ( $this->token ? 'yes' : 'no' ) );
		
		if ( ! $this->token ) {
			// No token - try to authenticate
			$license_key = get_option( 'buildfuly_license_key' );
			if ( $license_key ) {
				$auth_result = $this->authenticate( $license_key );
				if ( is_wp_error( $auth_result ) ) {
					Buildfuly_Logger::debug( 'BuildfulyAI: Authentication failed - returning local usage data as fallback' );
					return $this->get_local_usage_stats();
				}
			} else {
				Buildfuly_Logger::debug( 'BuildfulyAI: No license key - returning local usage stats' );
				return $this->get_local_usage_stats();
			}
	}

	$endpoint = $this->api_url . '/api/usage';
	Buildfuly_Logger::debug( 'BuildfulyAI: Making authenticated request to: ' . $endpoint );
	
	$response = $this->make_authenticated_request( $endpoint, array(), 'GET' );		// If API call fails, fall back to local data
		if ( is_wp_error( $response ) ) {
			Buildfuly_Logger::debug( 'BuildfulyAI: API usage check failed - falling back to local data: ' . $response->get_error_message() );
			return $this->get_local_usage_stats();
		}
		
		return $response;
	}

	/**
	 * Get usage stats from local WordPress data (fallback only)
	 * This should NOT be used for enforcement - API is source of truth
	 *
	 * @return array
	 */
	private function get_local_usage_stats() {
		// Get license instance
		require_once plugin_dir_path( __FILE__ ) . 'class-buildfuly-license.php';
		$license = new Buildfuly_License();
		
		// Get tier and limits
		$tier = Buildfuly_Constants::get_user_tier();
		$limit = Buildfuly_Constants::get_pages_limit();
		$remaining = $license->get_remaining_pages();
		
		// Calculate remaining words (600 words per page)
		$remaining_words = $remaining * Buildfuly_Constants::get_words_per_page();
		
		Buildfuly_Logger::debug( 'BuildfulyAI: Local usage stats - Tier: ' . $tier . ', Limit: ' . $limit . ', Remaining: ' . $remaining );
		
		return array(
			'tier' => $tier,
			'limit' => $limit,
			'remaining' => $remaining,
			'remaining_words' => $remaining_words,
			'is_starter' => ( $tier === 'starter' ),
			'local_fallback' => true, // Flag to indicate this is not authoritative
		);
	}

	/**
	 * Make authenticated API request with retry logic
	 *
	 * @param string $endpoint API endpoint URL
	 * @param array  $params Request parameters
	 * @param string $method HTTP method (POST or GET)
	 * @return array|WP_Error
	 */
	private function make_authenticated_request( $endpoint, $params = array(), $method = 'POST' ) {
		$attempt = 0;
		$last_error = null;

		while ( $attempt < $this->retry_attempts ) {
			$attempt++;

			$args = array(
				'timeout' => $this->timeout,
				'headers' => array(
					'Content-Type' => 'application/json',
					'Authorization' => 'Bearer ' . $this->token,
				),
				'sslverify' => false, // Disable SSL verification for localhost
				'reject_unsafe_urls' => false, // Allow localhost connections
			);

			if ( $method === 'POST' ) {
				$args['body'] = wp_json_encode( $params );
				$response = wp_remote_post( $endpoint, $args );
			} else {
				$response = wp_remote_get( $endpoint, $args );
			}

		if ( is_wp_error( $response ) ) {
			$last_error = $response;
			Buildfuly_Logger::debug( 'BuildfulyAI API Error (attempt ' . $attempt . '): ' . $response->get_error_message() );
			
			// Wait before retry (exponential backoff)
			if ( $attempt < $this->retry_attempts ) {
				sleep( pow( 2, $attempt ) );
			}
			continue;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );
		
		Buildfuly_Logger::debug( 'BuildfulyAI API Response: Status=' . $status_code . ', Body=' . wp_remote_retrieve_body( $response ) );

		// Success
		if ( $status_code === 200 ) {
			return $body;
		}			// Rate limit - don't retry
			if ( $status_code === 429 ) {
				return new WP_Error(
					'rate_limit_exceeded',
					$body['message'] ?? 'Rate limit exceeded',
					array( 'status' => $status_code )
				);
			}

			// Unauthorized - token expired, try to refresh
			if ( $status_code === 401 ) {
				$license_key = get_option( 'buildfuly_license_key' );
				if ( $license_key ) {
					$auth_result = $this->authenticate( $license_key );
					if ( ! is_wp_error( $auth_result ) ) {
						// Retry with new token
						continue;
					}
				}
				
				return new WP_Error(
					'unauthorized',
					'Session expired. Please re-activate your license.',
					array( 'status' => $status_code )
				);
			}

			// Other error
			$last_error = new WP_Error(
				'api_error',
				$body['message'] ?? 'API request failed',
				array( 'status' => $status_code )
			);

			// Wait before retry
			if ( $attempt < $this->retry_attempts ) {
				sleep( pow( 2, $attempt ) );
			}
		}

		return $last_error ?? new WP_Error( 'api_error', 'Unknown API error' );
	}

	/**
	 * Check if user is within rate limit
	 * Starter tier: checks total pages used (lifetime)
	 * Paid tiers: checks daily usage
	 *
	 * @return bool
	 */
	private function check_rate_limit() {
		$tier = Buildfuly_Constants::get_user_tier();
		$limit = Buildfuly_Constants::get_pages_limit();
		
		if ( $tier === 'starter' ) {
			// Starter: check total pages used (lifetime limit)
			$used = get_option( 'buildfuly_total_pages_used', 0 );
		} else {
			// Paid tiers: check daily usage
			$used = get_transient( 'buildfuly_daily_usage' );
			if ( $used === false ) {
				$used = 0;
			}
		}
		
		return $used < $limit;
	}

	/**
	 * Record a successful page generation
	 * Starter tier: increments total pages counter
	 * Paid tiers: increments daily usage with midnight reset
	 */
	private function record_generation() {
		$tier = Buildfuly_Constants::get_user_tier();
		
		if ( $tier === 'starter' ) {
			// Starter: increment total pages used (permanent)
			$total_used = get_option( 'buildfuly_total_pages_used', 0 );
			$total_used++;
			update_option( 'buildfuly_total_pages_used', $total_used );
		} else {
			// Paid tiers: increment daily usage with midnight expiration
			$usage_today = get_transient( 'buildfuly_daily_usage' );
			if ( $usage_today === false ) {
				$usage_today = 0;
			}
			$usage_today++;
			$seconds_until_midnight = strtotime( 'tomorrow' ) - time();
			set_transient( 'buildfuly_daily_usage', $usage_today, $seconds_until_midnight );
		}
	}

	/**
	 * Clear authentication token
	 */
	public function logout() {
		delete_option( 'buildfuly_api_token' );
		delete_transient( 'buildfuly_daily_usage' );
		delete_option( 'buildfuly_total_pages_used' );
		$this->token = null;
	}
}
