<?php

/**
 * Fired during plugin activation
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 */

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    Buildfuly
 * @subpackage Buildfuly/includes
 * @author     Buildfuly Team
 */
class Buildfuly_Activator {

	/**
	 * Activate the plugin.
	 *
	 * Creates necessary database tables and sets default options.
	 *
	 * @since    1.0.0
	 */
	public static function activate() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();

		// Table for storing generated content
		$table_content = $wpdb->prefix . 'buildfuly_content';
		$sql_content = "CREATE TABLE $table_content (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			content_key varchar(100) NOT NULL,
			content_value longtext NOT NULL,
			content_type varchar(50) DEFAULT 'text',
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			UNIQUE KEY content_key (content_key)
		) $charset_collate;";

		// Table for storing styling options
		$table_styles = $wpdb->prefix . 'buildfuly_styles';
		$sql_styles = "CREATE TABLE $table_styles (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			style_key varchar(100) NOT NULL,
			style_value text NOT NULL,
			component varchar(50) DEFAULT 'global',
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			UNIQUE KEY style_key (style_key)
		) $charset_collate;";

		// Table for storing generation history
		$table_history = $wpdb->prefix . 'buildfuly_history';
		$sql_history = "CREATE TABLE $table_history (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			action_type varchar(50) NOT NULL,
			action_data longtext,
			user_id bigint(20),
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id)
		) $charset_collate;";

		// Table for storing SEO metadata for published pages
		$table_seo = $wpdb->prefix . 'buildfuly_seo_pages';
		$sql_seo = "CREATE TABLE $table_seo (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			page_id bigint(20) NOT NULL,
			keywords text,
			slug varchar(200) NOT NULL,
			meta_title varchar(255),
			meta_description text,
			focus_keyphrase varchar(255),
			og_title varchar(255),
			og_description text,
			twitter_title varchar(255),
			twitter_description text,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			UNIQUE KEY page_id (page_id),
			UNIQUE KEY slug (slug)
		) $charset_collate;";

		// Table for storing leads
		$table_leads = $wpdb->prefix . 'buildfuly_leads';
		$sql_leads = "CREATE TABLE $table_leads (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			email varchar(255),
			name varchar(255),
			phone varchar(50),
			source varchar(255),
			estimate_total decimal(10,2) DEFAULT 0,
			estimate_details text,
			form_data longtext,
			status varchar(50) DEFAULT 'new',
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id)
		) $charset_collate;";

		// Table for storing custom forms
		$table_forms = $wpdb->prefix . 'buildfuly_forms';
		$sql_forms = "CREATE TABLE $table_forms (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			name varchar(255) NOT NULL,
			fields longtext NOT NULL,
			settings longtext,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY  (id)
		) $charset_collate;";

		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql_content );
		dbDelta( $sql_styles );
		dbDelta( $sql_history );
		dbDelta( $sql_seo );
		dbDelta( $sql_leads );
		dbDelta( $sql_forms );

		// Migrate old leads table columns if needed
		self::migrate_leads_table();

		// Set default options
		add_option( 'buildfuly_version', BUILDFULY_VERSION );
		add_option( 'buildfuly_activated', time() );
		
		// Set default styling options
		$default_styles = array(
			'primary_color' => '#007bff',
			'secondary_color' => '#6c757d',
			'accent_color' => '#28a745',
			'heading_font' => 'Inter',
			'body_font' => 'Inter',
			'business_type' => 'saas'
		);
		
		add_option( 'buildfuly_default_styles', $default_styles );
		
		// Auto-register free license for tracking (silent, no UI shown)
		self::register_free_license();
		
		// Check if this is a provisioned site that needs auto-build
		self::check_auto_build();
		
		// Flush rewrite rules
		flush_rewrite_rules();
	}

	/**
	 * Check if site was provisioned and needs auto-build
	 * Triggers homepage generation if buildfuly_auto_build flag is set
	 *
	 * @since    1.0.0
	 */
	public static function check_auto_build() {
		$auto_build = get_option( 'buildfuly_auto_build', false );
		$is_provisioned = get_option( 'buildfuly_is_provisioned', false );
		
		if ( ! $auto_build || ! $is_provisioned ) {
			return;
		}
		
		Buildfuly_Logger::debug( 'Buildfuly: Auto-build triggered for provisioned site' );
		
		// Get business settings
		$business_name = get_option( 'buildfuly_business_name', '' );
		$business_type = get_option( 'buildfuly_business_type', 'service' );
		$description = get_option( 'buildfuly_business_description', '' );
		$location = get_option( 'buildfuly_location', '' );
		$phone = get_option( 'buildfuly_phone', '' );
		
		if ( empty( $business_name ) ) {
			Buildfuly_Logger::debug( 'Buildfuly: Auto-build skipped - no business name set' );
			return;
		}
		
		// Schedule the auto-build to run after activation completes
		// This ensures all WordPress hooks are available
		if ( ! wp_next_scheduled( 'buildfuly_auto_build_site' ) ) {
			wp_schedule_single_event( time() + 5, 'buildfuly_auto_build_site' );
		}
		
		// Clear the auto-build flag so it doesn't run again
		delete_option( 'buildfuly_auto_build' );
		
		Buildfuly_Logger::debug( 'Buildfuly: Auto-build scheduled for ' . $business_name );
	}

	/**
	 * Register a free license automatically for usage tracking
	 * This happens silently in the background - users don't see the license key
	 *
	 * @since    1.0.0
	 */
	public static function register_free_license() {
		// Skip if already registered
		$existing_license = get_option( 'buildfuly_license_key' );
		if ( ! empty( $existing_license ) ) {
			Buildfuly_Logger::debug( 'Buildfuly: Free license registration skipped - license already exists' );
			return;
		}

		$api_url = defined( 'BUILDFULY_API_URL' ) ? trailingslashit( BUILDFULY_API_URL ) : 'https://build.fuly.io/';
		$api_url .= 'api/auth/register-free';
		
		$site_url = get_site_url();
		$admin_email = get_option( 'admin_email' );
		
		Buildfuly_Logger::debug( 'Buildfuly: Registering free license for site: ' . $site_url );
		
		$response = wp_remote_post( $api_url, array(
			'timeout' => 15,
			'headers' => array(
				'Content-Type' => 'application/json',
			),
			'body' => wp_json_encode( array(
				'site_url' => $site_url,
				'email' => $admin_email,
			) ),
			'sslverify' => false,
		) );

		if ( is_wp_error( $response ) ) {
			Buildfuly_Logger::debug( 'Buildfuly: Free license registration failed - ' . $response->get_error_message() );
			return;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );
		
		if ( $status_code === 200 && isset( $body['success'] ) && $body['success'] ) {
			// Store license data (silently, not shown in UI)
			update_option( 'buildfuly_api_token', $body['token'] );
			update_option( 'buildfuly_license_key', $body['license_key'] );
			update_option( 'buildfuly_license_status', 'active' );
			update_option( 'buildfuly_user_tier', 'starter' );
			update_option( 'buildfuly_license_data', $body );
			update_option( 'buildfuly_free_license', true ); // Mark as free license
			
			Buildfuly_Logger::debug( 'Buildfuly: Free license registered successfully - ' . $body['license_key'] );
		} else {
			Buildfuly_Logger::debug( 'Buildfuly: Free license registration failed - Status: ' . $status_code );
			Buildfuly_Logger::debug_data( 'Buildfuly: Free license registration response', $body );
		}
	}

	/**
	 * Migrate leads table from old schema to new schema
	 * Renames calculator_name -> source and estimate_breakdown -> estimate_details
	 *
	 * @since    1.0.0
	 */
	public static function migrate_leads_table() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_leads';
		
		// Check if table exists
		$table_exists = $wpdb->get_var( "SHOW TABLES LIKE '$table_name'" );
		if ( ! $table_exists ) {
			return; // Table will be created by dbDelta
		}
		
		// Get current columns
		$columns = $wpdb->get_results( "SHOW COLUMNS FROM $table_name" );
		$column_names = array_map( function( $col ) { return $col->Field; }, $columns );
		
		// Migrate calculator_name to source
		if ( in_array( 'calculator_name', $column_names ) && ! in_array( 'source', $column_names ) ) {
			$wpdb->query( "ALTER TABLE $table_name CHANGE `calculator_name` `source` varchar(255)" );
		}
		
		// Migrate estimate_breakdown to estimate_details
		if ( in_array( 'estimate_breakdown', $column_names ) && ! in_array( 'estimate_details', $column_names ) ) {
			$wpdb->query( "ALTER TABLE $table_name CHANGE `estimate_breakdown` `estimate_details` text" );
		}
		
		// If neither old nor new columns exist, add the new columns
		if ( ! in_array( 'source', $column_names ) && ! in_array( 'calculator_name', $column_names ) ) {
			$wpdb->query( "ALTER TABLE $table_name ADD `source` varchar(255) AFTER `phone`" );
		}
		if ( ! in_array( 'estimate_details', $column_names ) && ! in_array( 'estimate_breakdown', $column_names ) ) {
			$wpdb->query( "ALTER TABLE $table_name ADD `estimate_details` text AFTER `estimate_total`" );
		}
	}
}
