<?php
/**
 * Speed Optimizer Admin Display
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/admin/partials
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get current settings
$image_quality = get_option( 'buildfuly_image_quality', 82 );
$webp_quality = get_option( 'buildfuly_webp_quality', 80 );
$enable_webp = get_option( 'buildfuly_enable_webp', true );
$enable_lazy_load = get_option( 'buildfuly_enable_lazy_load', true );
$defer_js = get_option( 'buildfuly_defer_js', true );
$optimize_css = get_option( 'buildfuly_optimize_css', true );
$remove_query_strings = get_option( 'buildfuly_remove_query_strings', true );
$enable_resource_hints = get_option( 'buildfuly_enable_resource_hints', true );

// Get stats
$total_images = wp_count_posts( 'attachment' )->inherit;
?>

<div class="wrap buildfuly-speed-wrap">
	<h1>
		<span class="dashicons dashicons-performance" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px;"></span>
		Speed Optimizer
	</h1>
	<p class="description">Optimize your site for blazing-fast page loads and better Google PageSpeed scores.</p>

	<!-- Stats Cards -->
	<div class="speed-stats-cards">
		<div class="stat-card">
			<div class="stat-icon">📊</div>
			<div class="stat-content">
				<div class="stat-number"><?php echo esc_html( $total_images ); ?></div>
				<div class="stat-label">Total Images</div>
			</div>
		</div>
		<div class="stat-card stat-quality">
			<div class="stat-icon">🎯</div>
			<div class="stat-content">
				<div class="stat-number"><?php echo esc_html( $image_quality ); ?>%</div>
				<div class="stat-label">Image Quality</div>
			</div>
		</div>
		<div class="stat-card stat-webp">
			<div class="stat-icon">⚡</div>
			<div class="stat-content">
				<div class="stat-number"><?php echo $enable_webp ? 'ON' : 'OFF'; ?></div>
				<div class="stat-label">WebP Enabled</div>
			</div>
		</div>
		<div class="stat-card stat-lazy">
			<div class="stat-icon">🚀</div>
			<div class="stat-content">
				<div class="stat-number"><?php echo $enable_lazy_load ? 'ON' : 'OFF'; ?></div>
				<div class="stat-label">Lazy Loading</div>
			</div>
		</div>
	</div>

	<!-- Tabs -->
	<div class="speed-tabs">
		<a href="#image-optimization" class="tab-link active" data-tab="image-optimization">
			<span class="dashicons dashicons-format-image"></span> Image Optimization
		</a>
		<a href="#performance" class="tab-link" data-tab="performance">
			<span class="dashicons dashicons-dashboard"></span> Performance
		</a>
		<a href="#bulk-actions" class="tab-link" data-tab="bulk-actions">
			<span class="dashicons dashicons-update"></span> Bulk Actions
		</a>
		<a href="#guide" class="tab-link" data-tab="guide">
			<span class="dashicons dashicons-info"></span> SEO Guide
		</a>
	</div>

	<form method="post" action="options.php" id="speed-settings-form">
		<?php settings_fields( 'buildfuly_speed_settings' ); ?>

	<!-- Image Optimization Tab -->
	<div class="tab-content" id="image-optimization-tab" style="display: block;">
			
			<div class="speed-section">
				<h2>⚡ Automatic Compression</h2>
				<p><strong style="color: #d63638;">IMPORTANT:</strong> Only enable if your images need compression!</p>
				
				<table class="form-table">
					<tr>
						<th scope="row">Auto-Compress on Upload</th>
						<td>
							<label>
								<input type="checkbox" name="buildfuly_auto_compress" value="1" 
									   <?php checked( get_option( 'buildfuly_auto_compress', true ) ); ?>>
								Automatically compress images when uploaded
							</label>
							<p class="description">
								<strong>⚠️ READ THIS:</strong><br>
								✅ Only compresses if it makes the file SMALLER (safe mode enabled)<br>
								❌ If your images are already optimized (e.g., from export software), <strong>DISABLE THIS</strong><br>
								💡 You can still use the "Compress All Images" button below for manual compression
							</p>
						</td>
					</tr>
				</table>
			</div>

			<div class="speed-section">
				<h2>🖼️ Image Compression</h2>
				<p>Control the balance between image quality and file size for JPEG/PNG compression.</p>
				
				<table class="form-table">
					<tr>
						<th scope="row">JPEG/PNG Quality</th>
						<td>
							<input type="range" name="buildfuly_image_quality" 
								   value="<?php echo esc_attr( $image_quality ); ?>" 
								   min="1" max="100" step="1" 
								   id="image-quality-slider">
							<span id="image-quality-value"><?php echo esc_html( $image_quality ); ?>%</span>
							<p class="description">
								<strong>For bloated images: Try 30-50%</strong> to force aggressive compression<br>
								<strong>Normal use: 82%</strong> - Perfect balance between quality and file size<br>
								• 1-50%: Very aggressive compression (use to fix 1MB files)<br>
								• 60-70%: Small files, some quality loss<br>
								• 80-85%: Great quality, good compression (recommended)<br>
								• 90-100%: Maximum quality, larger files
							</p>
						</td>
					</tr>
				</table>
			</div>

			<div class="speed-section">
				<h2>⚡ WebP Conversion</h2>
				<p>Convert images to WebP format for 25-35% smaller file sizes with same visual quality.</p>
				
				<table class="form-table">
					<tr>
						<th scope="row">Enable WebP</th>
						<td>
							<label>
								<input type="checkbox" name="buildfuly_enable_webp" value="1" 
									   <?php checked( $enable_webp ); ?>>
								Automatically generate WebP versions of all images
							</label>
							<p class="description">
								WebP images are 25-35% smaller than JPEG/PNG with same quality. 
								Modern browsers will use WebP, older browsers fall back to original format.
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row">WebP Quality</th>
						<td>
							<input type="range" name="buildfuly_webp_quality" 
								   value="<?php echo esc_attr( $webp_quality ); ?>" 
								   min="60" max="100" step="1" 
								   id="webp-quality-slider">
							<span id="webp-quality-value"><?php echo esc_html( $webp_quality ); ?>%</span>
							<p class="description">
								<strong>Recommended: 80%</strong> - WebP provides better compression than JPEG at same quality level.
							</p>
						</td>
					</tr>
				</table>
			</div>

			<div class="speed-section">
				<h2>📱 Lazy Loading</h2>
				<p>Only load images when they're about to enter the viewport. Dramatically improves initial page load time.</p>
				
				<table class="form-table">
					<tr>
						<th scope="row">Enable Lazy Loading</th>
						<td>
							<label>
								<input type="checkbox" name="buildfuly_enable_lazy_load" value="1" 
									   <?php checked( $enable_lazy_load ); ?>>
								Add <code>loading="lazy"</code> to all images
							</label>
							<p class="description">
								✅ <strong>Google recommends this</strong> - Images load only when needed, improving page speed scores.
							</p>
						</td>
					</tr>
				</table>
			</div>

	</div>

	<!-- Performance Tab -->
	<div class="tab-content" id="performance-tab" style="display: none;">			<div class="speed-section">
				<h2>⚡ JavaScript Optimization</h2>
				
				<table class="form-table">
					<tr>
						<th scope="row">Defer JavaScript</th>
						<td>
							<label>
								<input type="checkbox" name="buildfuly_defer_js" value="1" 
									   <?php checked( $defer_js ); ?>>
								Defer non-critical JavaScript loading
							</label>
							<p class="description">
								Prevents JavaScript from blocking page render. Improves First Contentful Paint (FCP).
							</p>
						</td>
					</tr>
				</table>
			</div>

			<div class="speed-section">
				<h2>🎨 CSS Optimization</h2>
				
				<table class="form-table">
					<tr>
						<th scope="row">Optimize CSS Delivery</th>
						<td>
							<label>
								<input type="checkbox" name="buildfuly_optimize_css" value="1" 
									   <?php checked( $optimize_css ); ?>>
								Move non-critical CSS to footer
							</label>
							<p class="description">
								Loads critical CSS first, deferring non-essential styles. Improves render time.
							</p>
						</td>
					</tr>
				</table>
			</div>

			<div class="speed-section">
				<h2>🔗 Resource Hints</h2>
				
				<table class="form-table">
					<tr>
						<th scope="row">Enable Resource Hints</th>
						<td>
							<label>
								<input type="checkbox" name="buildfuly_enable_resource_hints" value="1" 
									   <?php checked( $enable_resource_hints ); ?>>
								Add DNS prefetch and preconnect hints
							</label>
							<p class="description">
								Tells browser to establish early connections to external resources (fonts, APIs, etc).
							</p>
						</td>
					</tr>
				</table>
			</div>

			<div class="speed-section">
				<h2>🔧 Advanced</h2>
				
				<table class="form-table">
					<tr>
						<th scope="row">Remove Query Strings</th>
						<td>
							<label>
								<input type="checkbox" name="buildfuly_remove_query_strings" value="1" 
									   <?php checked( $remove_query_strings ); ?>>
								Remove version query strings from static resources
							</label>
							<p class="description">
								Improves caching on some CDNs and proxies. Changes <code>style.css?ver=1.0</code> to <code>style.css</code>
							</p>
						</td>
					</tr>
				</table>
			</div>

		<p class="submit">
			<button type="submit" class="button button-primary button-large">💾 Save All Settings</button>
		</p>
	</div>

	<!-- Bulk Actions Tab -->
	<div class="tab-content" id="bulk-actions-tab" style="display: none;">
		<div class="speed-section" style="background: #fff3cd; border-left: 4px solid #ffc107;">
			<h2>🔥 Convert PNG to JPEG (Fixes 1MB Images!)</h2>
			<p><strong>Your images are PNG format.</strong> PNG is lossless and cannot be compressed smaller. Converting to JPEG will shrink 1MB PNGs down to ~50-100KB.</p>
			
			<div class="bulk-action-card">
				<div class="action-info">
					<h3>PNG → JPEG Conversion</h3>
					<p>Found <strong><?php 
						$png_count = count( get_posts( array(
							'post_type' => 'attachment',
							'post_mime_type' => 'image/png',
							'posts_per_page' => -1,
							'fields' => 'ids',
						) ) );
						echo esc_html( $png_count );
					?></strong> PNG images that can be converted to JPEG.</p>
					<p><strong>⚠️ Note:</strong> This will replace PNG files with JPEG. Transparent backgrounds will become white.</p>
				</div>
				<div class="action-button">
					<button type="button" class="button button-primary button-hero" id="convert-png-to-jpeg-btn" style="background: #ffc107; border-color: #ffc107; color: #000;">
						<span class="dashicons dashicons-image-rotate"></span>
						Convert PNG → JPEG
					</button>
				</div>
			</div>
			
			<div id="convert-progress" class="progress-container" style="display: none;">
				<div class="progress-bar">
					<div class="progress-fill" style="width: 0%"></div>
				</div>
				<div class="progress-text">Converting images... 0%</div>
			</div>
		</div>

		<div class="speed-section">
			<h2>🔄 Compress All Existing Images</h2>
			<p>Compress all images already in your media library. This may take several minutes depending on how many images you have.</p>
			
			<div class="bulk-action-card">
				<div class="action-info">
					<h3>Image Compression</h3>
					<p>Will compress <?php echo esc_html( $total_images ); ?> images using your quality settings.</p>
					<p><strong>Time estimate:</strong> ~<?php echo ceil( $total_images / 10 ); ?> minutes</p>
					<p>
						<label style="margin-top: 10px; display: block;">
							<input type="checkbox" id="force-compress" value="1">
							<strong>⚠️ Force Compression</strong> - Re-compress even if files are already compressed
						</label>
						<small style="color: #d63638;">Use this to fix bloated 1MB images by forcing them to compress again at lower quality.</small>
					</p>
				</div>
				<div class="action-button">
					<button type="button" class="button button-primary button-hero" id="compress-all-images-btn">
						<span class="dashicons dashicons-image-crop"></span>
						Compress All Images
					</button>
				</div>
			</div>
			
			<div id="compress-progress" class="progress-container" style="display: none;">
				<div class="progress-bar">
					<div class="progress-fill" style="width: 0%"></div>
				</div>
				<div class="progress-text">Compressing images... 0%</div>
			</div>
		</div>

		<div class="speed-section">
			<h2>⚡ Generate WebP Versions</h2>
			<p>Create WebP versions of all existing images for better performance.</p>
			
			<?php if ( ! function_exists( 'imagewebp' ) ) : ?>
				<div class="notice notice-error">
					<p><strong>WebP Not Supported</strong> - Your server doesn't have WebP support enabled. Contact your hosting provider.</p>
				</div>
			<?php else : ?>
				<div class="bulk-action-card">
					<div class="action-info">
						<h3>WebP Generation</h3>
						<p>Will generate WebP versions for <?php echo esc_html( $total_images ); ?> images.</p>
						<p><strong>Time estimate:</strong> ~<?php echo ceil( $total_images / 10 ); ?> minutes</p>
					</div>
					<div class="action-button">
						<button type="button" class="button button-primary button-hero" id="generate-webp-btn">
							<span class="dashicons dashicons-format-image"></span>
							Generate WebP Versions
						</button>
					</div>
				</div>
				
				<div id="webp-progress" class="progress-container" style="display: none;">
					<div class="progress-bar">
						<div class="progress-fill" style="width: 0%"></div>
					</div>
					<div class="progress-text">Generating WebP... 0%</div>
				</div>
			<?php endif; ?>
		</div>

		<div class="speed-section">
			<h2>🗑️ Clear All Caches</h2>
			<p>Clear WordPress cache and transients. Also clears popular cache plugin caches.</p>
			
			<div class="bulk-action-card">
				<div class="action-info">
					<h3>Cache Management</h3>
					<p>Clears: WordPress object cache, transients, W3TC, WP Super Cache, WP Fastest Cache</p>
				</div>
				<div class="action-button">
					<button type="button" class="button button-secondary button-hero" id="clear-cache-btn">
						<span class="dashicons dashicons-trash"></span>
						Clear All Caches
					</button>
				</div>
			</div>
		</div>

		<div class="speed-section">
			<h2>⏮️ Restore Original Images</h2>
			<p><strong style="color: #d63638;">⚠️ UNDO Compression:</strong> Restore all images to their original versions before compression.</p>
			
			<div class="bulk-action-card" style="border-left: 4px solid #d63638;">
				<div class="action-info">
					<h3>Restore Backups</h3>
					<p>This will restore all compressed images to their original versions from the <code>.original</code> backup files.</p>
					<p><strong>Note:</strong> Only works if images were compressed AFTER this backup feature was added.</p>
					<p><strong>Backup files found:</strong> <span id="backup-count">Checking...</span></p>
				</div>
				<div class="action-button">
					<button type="button" class="button button-secondary button-hero" id="restore-originals-btn">
						<span class="dashicons dashicons-backup"></span>
						Restore Original Images
					</button>
				</div>
			</div>
			
			<div id="restore-progress" class="progress-container" style="display: none;">
				<div class="progress-bar">
					<div class="progress-fill" style="width: 0%"></div>
				</div>
				<div class="progress-text">Restoring images... 0%</div>
			</div>
		</div>
	</div>
					</button>
				</div>
			</div>
		</div>
	</div>

	<!-- SEO Guide Tab -->
	<div class="tab-content" id="guide-tab" style="display: none;">
		<div class="speed-guide">
			<h2>🎯 PageSpeed Optimization Guide</h2>
			
			<div class="guide-section">
				<h3>✅ What This Plugin Optimizes</h3>
				<ul class="checklist">
					<li><strong>Image Compression:</strong> Reduces file sizes automatically</li>
					<li><strong>WebP Format:</strong> Modern image format, 25-35% smaller</li>
					<li><strong>Lazy Loading:</strong> Loads images only when needed</li>
					<li><strong>JavaScript Deferring:</strong> Non-blocking script loading</li>
					<li><strong>CSS Optimization:</strong> Critical CSS first, rest later</li>
					<li><strong>Resource Hints:</strong> Early connection to external resources</li>
					<li><strong>Image Dimensions:</strong> Prevents CLS (layout shift)</li>
					<li><strong>Query String Removal:</strong> Better CDN caching</li>
				</ul>
			</div>

			<div class="guide-section">
				<h3>📊 Google PageSpeed Metrics</h3>
				<table class="wp-list-table widefat fixed striped">
					<thead>
						<tr>
							<th>Metric</th>
							<th>What It Measures</th>
							<th>How We Optimize</th>
						</tr>
					</thead>
					<tbody>
						<tr>
							<td><strong>LCP</strong><br><small>Largest Contentful Paint</small></td>
							<td>Time to load main content</td>
							<td>Image compression, WebP, lazy loading</td>
						</tr>
						<tr>
							<td><strong>FID</strong><br><small>First Input Delay</small></td>
							<td>Time until page is interactive</td>
							<td>Defer JavaScript, optimize CSS</td>
						</tr>
						<tr>
							<td><strong>CLS</strong><br><small>Cumulative Layout Shift</small></td>
							<td>Visual stability</td>
							<td>Add width/height to images</td>
						</tr>
						<tr>
							<td><strong>FCP</strong><br><small>First Contentful Paint</small></td>
							<td>Time to first visible content</td>
							<td>Resource hints, defer JS, optimize CSS</td>
						</tr>
					</tbody>
				</table>
			</div>

			<div class="guide-section">
				<h3>🚀 Recommended Settings for Best Speed</h3>
				<div class="settings-recommendation">
					<h4>For Most Sites:</h4>
					<ul>
						<li>Image Quality: <strong>82%</strong></li>
						<li>WebP Quality: <strong>80%</strong></li>
						<li>Enable WebP: <strong>✅ Yes</strong></li>
						<li>Lazy Loading: <strong>✅ Yes</strong></li>
						<li>Defer JavaScript: <strong>✅ Yes</strong></li>
						<li>Optimize CSS: <strong>✅ Yes</strong></li>
					</ul>

					<h4>For Photography/Portfolio Sites:</h4>
					<ul>
						<li>Image Quality: <strong>85-90%</strong> (higher quality)</li>
						<li>WebP Quality: <strong>85%</strong></li>
						<li>Everything else: <strong>✅ Enabled</strong></li>
					</ul>

					<h4>For Fast E-commerce:</h4>
					<ul>
						<li>Image Quality: <strong>78-82%</strong> (smaller files)</li>
						<li>WebP: <strong>✅ Essential</strong></li>
						<li>Lazy Loading: <strong>✅ Essential</strong></li>
					</ul>
				</div>
			</div>

			<div class="guide-section">
				<h3>💡 Additional Tips</h3>
				<ul class="tips-list">
					<li><strong>Use a CDN:</strong> Cloudflare (free), BunnyCDN, or KeyCDN</li>
					<li><strong>Enable Browser Caching:</strong> Add expires headers in .htaccess</li>
					<li><strong>Minify HTML/CSS/JS:</strong> Use plugins like Autoptimize</li>
					<li><strong>Use Fast Hosting:</strong> Quality hosting makes huge difference</li>
					<li><strong>Remove Unused Plugins:</strong> Each plugin adds overhead</li>
					<li><strong>Test Regularly:</strong> Use Google PageSpeed Insights</li>
				</ul>
			</div>

			<div class="guide-section">
				<h3>🔗 Testing Tools</h3>
				<ul>
					<li><a href="https://pagespeed.web.dev/" target="_blank">Google PageSpeed Insights</a> - Official Google tool</li>
					<li><a href="https://gtmetrix.com/" target="_blank">GTmetrix</a> - Detailed performance analysis</li>
					<li><a href="https://tools.pingdom.com/" target="_blank">Pingdom</a> - Load time testing</li>
					<li><a href="https://webpagetest.org/" target="_blank">WebPageTest</a> - Advanced testing</li>
				</ul>
			</div>
		</div>
	</div>
	</form>
</div>

<style>
/* Stats Cards */
.speed-stats-cards {
	display: grid;
	grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
	gap: 20px;
	margin: 20px 0;
}

.stat-card {
	background: #fff;
	border: 1px solid #c3c4c7;
	border-radius: 8px;
	padding: 20px;
	display: flex;
	align-items: center;
	gap: 15px;
}

.stat-icon {
	font-size: 32px;
	width: 50px;
	height: 50px;
	display: flex;
	align-items: center;
	justify-content: center;
	background: #f0f0f1;
	border-radius: 8px;
}

.stat-content {
	flex: 1;
}

.stat-number {
	font-size: 24px;
	font-weight: bold;
	color: #1d2327;
}

.stat-label {
	color: #646970;
	margin-top: 2px;
	font-size: 13px;
}

/* Tabs */
.speed-tabs {
	display: flex;
	gap: 0;
	border-bottom: 1px solid #c3c4c7;
	margin-bottom: 20px;
}

.speed-tabs .tab-link {
	padding: 12px 20px;
	text-decoration: none;
	color: #646970;
	border: 1px solid transparent;
	border-bottom: none;
	margin-bottom: -1px;
	background: transparent;
	display: flex;
	align-items: center;
	gap: 5px;
}

.speed-tabs .tab-link:hover {
	color: #2271b1;
}

.speed-tabs .tab-link.active {
	background: #fff;
	border-color: #c3c4c7;
	color: #1d2327;
	font-weight: 500;
}

/* Sections */
.speed-section {
	background: #fff;
	border: 1px solid #c3c4c7;
	border-radius: 8px;
	padding: 20px 30px;
	margin-bottom: 20px;
}

.speed-section h2 {
	margin-top: 0;
	font-size: 18px;
}

/* Sliders */
#image-quality-slider,
#webp-quality-slider {
	width: 300px;
	margin-right: 10px;
}

#image-quality-value,
#webp-quality-value {
	font-weight: bold;
	color: #2271b1;
	font-size: 16px;
}

/* Bulk Actions */
.bulk-action-card {
	display: flex;
	align-items: center;
	justify-content: space-between;
	background: #f6f7f7;
	padding: 20px;
	border-radius: 8px;
	margin-bottom: 15px;
}

.action-info h3 {
	margin: 0 0 8px 0;
}

.action-info p {
	margin: 4px 0;
	color: #646970;
}

/* Progress */
.progress-container {
	margin: 20px 0;
}

.progress-bar {
	width: 100%;
	height: 30px;
	background: #f0f0f1;
	border-radius: 15px;
	overflow: hidden;
}

.progress-fill {
	height: 100%;
	background: linear-gradient(90deg, #667eea 0%, #764ba2 100%);
	transition: width 0.3s ease;
	display: flex;
	align-items: center;
	justify-content: center;
	color: white;
	font-weight: bold;
}

.progress-text {
	text-align: center;
	margin-top: 10px;
	font-weight: 600;
	color: #1d2327;
}

/* Guide */
.speed-guide {
	max-width: 900px;
}

.guide-section {
	background: #fff;
	border: 1px solid #c3c4c7;
	border-radius: 8px;
	padding: 20px 30px;
	margin-bottom: 20px;
}

.guide-section h3 {
	margin-top: 0;
}

.checklist {
	list-style: none;
	padding: 0;
}

.checklist li {
	padding: 8px 0;
	padding-left: 30px;
	position: relative;
}

.checklist li:before {
	content: "✓";
	position: absolute;
	left: 0;
	color: #10b981;
	font-weight: bold;
	font-size: 18px;
}

.tips-list {
	list-style: none;
	padding: 0;
}

.tips-list li {
	padding: 8px 0;
	padding-left: 30px;
	position: relative;
}

.tips-list li:before {
	content: "💡";
	position: absolute;
	left: 0;
	font-size: 18px;
}

.settings-recommendation {
	background: #f0f7fc;
	padding: 15px 20px;
	border-radius: 8px;
	border-left: 4px solid #2271b1;
}

.settings-recommendation h4 {
	margin: 15px 0 8px 0;
	color: #1d2327;
}

.settings-recommendation h4:first-child {
	margin-top: 0;
}
</style>

<script>
jQuery(document).ready(function($) {
	// Tab switching
	$('.tab-link').on('click', function(e) {
		e.preventDefault();
		var tab = $(this).data('tab');
		
		console.log('[Buildfuly] Switching to tab:', tab);
		
		$('.tab-link').removeClass('active');
		$(this).addClass('active');
		
		$('.tab-content').hide();
		$('#' + tab + '-tab').show();
		
		// Show save button only on Image/Performance tabs
		if (tab === 'image-optimization' || tab === 'performance') {
			$('.submit').show();
		} else {
			$('.submit').hide();
		}
		
		console.log('[Buildfuly] Tab visible:', $('#' + tab + '-tab').is(':visible'));
	});

	// Quality sliders
	$('#image-quality-slider').on('input', function() {
		$('#image-quality-value').text($(this).val() + '%');
	});

	$('#webp-quality-slider').on('input', function() {
		$('#webp-quality-value').text($(this).val() + '%');
	});

	// Compress all images
	$('#compress-all-images-btn').on('click', function() {
		var forceCompress = $('#force-compress').is(':checked');
		var qualityValue = parseInt($('#image-quality-slider').val());
		var message = forceCompress 
			? '⚠️ FORCE MODE: This will re-compress ALL images at ' + qualityValue + '% quality.\n\nThis will fix bloated 1MB images. Continue?' 
			: 'This will compress all existing images at ' + qualityValue + '% quality. Continue?';
			
		if (!confirm(message)) return;

		var $btn = $(this);
		var $progress = $('#compress-progress');
		var $progressBar = $progress.find('.progress-fill');
		var $progressText = $progress.find('.progress-text');

		$btn.prop('disabled', true);
		$progress.show();
		
		console.log('[Buildfuly] Starting compression with quality=' + qualityValue + ', force=' + forceCompress);

		function processNextBatch(offset = 0) {
			$.post(ajaxurl, {
				action: 'buildfuly_compress_all_images',
				nonce: '<?php echo wp_create_nonce( 'buildfuly_speed_nonce' ); ?>',
				offset: offset,
				force: forceCompress ? 1 : 0,
				quality: qualityValue
			}, function(response) {
				console.log('[Buildfuly] Response:', response);
				if (response.success) {
					console.log('[Buildfuly] Quality: ' + response.data.quality + '%, Force: ' + (response.data.force ? 'YES' : 'NO'));
					$progressBar.css('width', response.data.progress + '%');
					$progressText.text('Compressing images... ' + response.data.progress + '%');

					if (!response.data.complete) {
						processNextBatch(response.data.offset);
					} else {
						$progressText.text('✓ Compression complete! ' + response.data.total + ' images processed.');
						console.log('[Buildfuly] Done! Check file sizes in Media Library.');
						$btn.prop('disabled', false);
					}
				} else {
					console.error('[Buildfuly] Error:', response);
					alert('Error: ' + (response.data || 'Unknown error'));
					$btn.prop('disabled', false);
				}
			}).fail(function(xhr, status, error) {
				console.error('[Buildfuly] AJAX failed:', status, error);
				alert('Compression failed! Check browser console (F12) for details.');
				$btn.prop('disabled', false);
			});
		}

		processNextBatch();
	});

	// Generate WebP
	$('#generate-webp-btn').on('click', function() {
		if (!confirm('This will generate WebP versions of all images. Continue?')) return;

		var $btn = $(this);
		var $progress = $('#webp-progress');
		var $progressBar = $progress.find('.progress-fill');
		var $progressText = $progress.find('.progress-text');

		$btn.prop('disabled', true);
		$progress.show();

		function processNextBatch(offset = 0) {
			$.post(ajaxurl, {
				action: 'buildfuly_generate_webp_all',
				nonce: '<?php echo wp_create_nonce( 'buildfuly_speed_nonce' ); ?>',
				offset: offset
			}, function(response) {
				if (response.success) {
					$progressBar.css('width', response.data.progress + '%');
					$progressText.text('Generating WebP... ' + response.data.progress + '%');

					if (!response.data.complete) {
						processNextBatch(response.data.offset);
					} else {
						$progressText.text('✓ WebP generation complete! ' + response.data.total + ' images processed.');
						$btn.prop('disabled', false);
					}
				}
			});
		}

		processNextBatch();
	});

	// Clear cache
	$('#clear-cache-btn').on('click', function() {
		if (!confirm('Clear all caches?')) return;

		var $btn = $(this);
		$btn.prop('disabled', true).text('Clearing...');

		$.post(ajaxurl, {
			action: 'buildfuly_clear_cache',
			nonce: '<?php echo wp_create_nonce( 'buildfuly_speed_nonce' ); ?>'
		}, function(response) {
			$btn.prop('disabled', false).text('Clear All Caches');
			
			if (response.success) {
				alert('✓ Cache cleared successfully!');
			} else {
				alert('Error: ' + response.data);
			}
		});
	});

	// Check backup count on load
	$.post(ajaxurl, {
		action: 'buildfuly_count_backups',
		nonce: '<?php echo wp_create_nonce( 'buildfuly_speed_nonce' ); ?>'
	}, function(response) {
		if (response.success) {
			$('#backup-count').text(response.data.count + ' backup files');
		}
	});

	// Restore original images
	$('#restore-originals-btn').on('click', function() {
		if (!confirm('⚠️ WARNING: This will restore all images to their ORIGINAL versions (before compression).\n\nYour compressed images will be replaced with the originals.\n\nAre you sure?')) return;

		var $btn = $(this);
		var $progress = $('#restore-progress');
		var $progressBar = $progress.find('.progress-fill');
		var $progressText = $progress.find('.progress-text');

		$btn.prop('disabled', true);
		$progress.show();

		function processNextBatch(offset = 0) {
			$.post(ajaxurl, {
				action: 'buildfuly_restore_originals',
				nonce: '<?php echo wp_create_nonce( 'buildfuly_speed_nonce' ); ?>',
				offset: offset
			}, function(response) {
				if (response.success) {
					$progressBar.css('width', response.data.progress + '%');
					$progressText.text('Restoring images... ' + response.data.progress + '%');

					if (!response.data.complete) {
						processNextBatch(response.data.offset);
					} else {
						$progressText.text('✓ Restoration complete! ' + response.data.restored + ' images restored to original.');
						$btn.prop('disabled', false);
						// Refresh backup count
						setTimeout(function() {
							location.reload();
						}, 2000);
					}
				}
			});
		}

		processNextBatch();
	});
});
</script>
