<?php
/**
 * Unified Settings Page - All settings in ONE form
 *
 * @link       https://build.fuly.io
 * @since      1.0.4
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/admin/partials
 */

// Prevent direct access
if ( ! defined( 'WPINC' ) ) {
	die;
}

// Get all settings
// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Template variables, not globals
$generation_mode = get_option( 'buildfuly_generation_mode', 'keyword' );
$generation_instructions = get_option( 'buildfuly_generation_instructions', '' );
$cta_buttons = json_decode( get_option( 'buildfuly_cta_buttons', '[]' ), true );
$contact_form_embed = get_option( 'buildfuly_contact_form_embed', '' );

// Ensure at least one button exists
if ( empty( $cta_buttons ) ) {
	$cta_buttons = array(
		array( 'text' => 'Get Started', 'url' => '#contact', 'style' => 'primary' )
	);
}
// phpcs:enable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound

// Get homepage setting
$buildfuly_homepage_id = get_option( 'buildfuly_homepage_id', 0 );
$all_buildfuly_pages = get_posts( array(
	'post_type'      => 'page',
	'post_status'    => 'publish',
	'posts_per_page' => -1,
	'meta_query'     => array(
		array(
			'key'     => 'buildfuly_generated',
			'value'   => '1',
			'compare' => '=',
		),
	),
	'orderby'        => 'title',
	'order'          => 'ASC',
) );
?>

<div class="wrap buildfuly-admin-wrap">
	<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
	
	<!-- ONE UNIFIED FORM FOR ALL SETTINGS -->
	<form method="post" action="options.php" id="buildfuly-unified-settings-form">
		<?php settings_fields( 'buildfuly_business_settings' ); ?>
		
		<div class="buildfuly-container">
			
			<!-- Homepage Settings -->
			<div class="buildfuly-card">
				<h3><?php esc_html_e( '🏠 Homepage Settings', 'buildfuly' ); ?></h3>
				<p class="description">
					<?php esc_html_e( 'Select which page should be your homepage. This page will be set as the front page and linked from the brand logo.', 'buildfuly' ); ?>
				</p>
				
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="buildfuly_homepage_id"><?php esc_html_e( 'Homepage', 'buildfuly' ); ?></label>
						</th>
						<td>
							<select id="buildfuly_homepage_id" name="buildfuly_homepage_id" class="regular-text">
								<option value="0"><?php esc_html_e( '— Select Homepage —', 'buildfuly' ); ?></option>
								<?php foreach ( $all_buildfuly_pages as $page ) : ?>
									<option value="<?php echo esc_attr( $page->ID ); ?>" <?php selected( $buildfuly_homepage_id, $page->ID ); ?>>
										<?php echo esc_html( $page->post_title ); ?>
									</option>
								<?php endforeach; ?>
							</select>
							<?php if ( $buildfuly_homepage_id ) : ?>
								<a href="<?php echo esc_url( get_permalink( $buildfuly_homepage_id ) ); ?>" target="_blank" class="button button-secondary" style="margin-left: 8px;">
									<?php esc_html_e( 'View Homepage', 'buildfuly' ); ?>
								</a>
							<?php endif; ?>
							<p class="description">
								<?php esc_html_e( 'The selected page will become the front page of your site and be accessible at /.', 'buildfuly' ); ?>
							</p>
						</td>
					</tr>
				</table>
			</div>
			
			<!-- CTA Buttons -->
			<div class="buildfuly-card">
				<h3><?php esc_html_e( '🔘 Call-to-Action Buttons', 'buildfuly' ); ?></h3>
				<p class="description">
					<?php esc_html_e( 'Configure buttons that appear across all components.', 'buildfuly' ); ?>
				</p>
				
				<div id="button-list">
					<?php 
					// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Foreach loop variables
					foreach ( $cta_buttons as $index => $button ) : 
					?>
				<div class="button-row" data-index="<?php echo esc_attr( $index ); ?>" style="background: #f9fafb; padding: 20px; border-radius: 8px; margin-bottom: 16px; border: 2px solid #e5e7eb;">
					<div style="display: flex; align-items: center; margin-bottom: 12px;">
						<h4 style="margin: 0;">
							<?php 
							/* translators: %d: Button number (e.g., 1, 2, 3) */
							printf( esc_html__( 'Button %d', 'buildfuly' ), esc_html( $index + 1 ) ); 
							?>
							<?php if ( $index === 0 ) : ?>
								<span style="background: #10b981; color: white; padding: 2px 8px; border-radius: 4px; font-size: 11px; font-weight: 600; margin-left: 8px;">PRIMARY</span>
							<?php endif; ?>
							</h4>
							<?php if ( $index > 0 ) : ?>
							<button type="button" class="button remove-button-btn" data-index="<?php echo esc_attr( $index ); ?>">
								<?php esc_html_e( 'Remove', 'buildfuly' ); ?>
							</button>
							<?php endif; ?>
						</div>
						
						<table class="form-table" style="margin: 0;">
							<tr>
								<th style="width: 120px; padding-left: 0;">
									<label><?php esc_html_e( 'Button Text', 'buildfuly' ); ?></label>
								</th>
								<td>
									<input type="text" 
										   name="button_text_<?php echo esc_attr( $index ); ?>" 
										   value="<?php echo esc_attr( $button['text'] ); ?>" 
										   class="regular-text" required>
								</td>
							</tr>
							<tr>
								<th style="padding-left: 0;">
									<label><?php esc_html_e( 'Link URL', 'buildfuly' ); ?></label>
								</th>
								<td>
									<input type="text" 
										   name="button_url_<?php echo esc_attr( $index ); ?>" 
										   value="<?php echo esc_attr( $button['url'] ); ?>" 
										   class="large-text" 
										   placeholder="https://example.com or tel:+15551234567 or #contact"
										   required>
									<p class="description">
										<?php esc_html_e( 'Enter a URL (https://...), phone number (tel:+1...), email (mailto:...), or anchor link (#section)', 'buildfuly' ); ?>
									</p>
								</td>
							</tr>
							<tr>
								<th style="padding-left: 0;">
									<label><?php esc_html_e( 'Style', 'buildfuly' ); ?></label>
								</th>
								<td>
									<select name="button_style_<?php echo esc_attr( $index ); ?>" class="regular-text">
										<option value="primary" <?php selected( $button['style'], 'primary' ); ?>>Primary</option>
										<option value="secondary" <?php selected( $button['style'], 'secondary' ); ?>>Secondary</option>
									</select>
								</td>
							</tr>
						</table>
					</div>
					<?php endforeach; ?>
				</div>

				<button type="button" id="add-button-btn" class="button button-secondary">
					<span class="dashicons dashicons-plus-alt"></span>
					<?php esc_html_e( 'Add Another Button', 'buildfuly' ); ?>
				</button>

				<input type="hidden" name="buildfuly_cta_buttons" id="buildfuly_cta_buttons" value="">
			</div>

			<!-- Contact Form -->
			<div class="buildfuly-card">
				<h3><?php esc_html_e( '📧 Contact Form Integration', 'buildfuly' ); ?></h3>
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="buildfuly_contact_form_embed"><?php esc_html_e( 'Form Embed Code', 'buildfuly' ); ?></label>
						</th>
						<td>
							<textarea id="buildfuly_contact_form_embed" 
									  name="buildfuly_contact_form_embed" 
									  rows="4" 
									  class="large-text code"
									  placeholder='[contact-form-7 id="123"]'><?php echo esc_textarea( $contact_form_embed ); ?></textarea>
						</td>
					</tr>
				</table>
			</div>
			
			<!-- Business Information -->
			<div class="buildfuly-card">
				<h3><?php esc_html_e( 'Business Information (NAP + SEO)', 'buildfuly' ); ?></h3>
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="buildfuly_business_name"><?php esc_html_e( 'Business Name', 'buildfuly' ); ?> *</label>
						</th>
						<td>
							<input type="text" 
								   id="buildfuly_business_name" 
								   name="buildfuly_business_name" 
								   value="<?php echo esc_attr( get_option( 'buildfuly_business_name', '' ) ); ?>" 
								   class="regular-text" required>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="buildfuly_business_description"><?php esc_html_e( 'Business Description', 'buildfuly' ); ?> *</label>
						</th>
						<td>
							<textarea id="buildfuly_business_description" 
									  name="buildfuly_business_description" 
									  rows="4" 
									  class="large-text" required><?php echo esc_textarea( get_option( 'buildfuly_business_description', '' ) ); ?></textarea>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="buildfuly_business_logo"><?php esc_html_e( 'Business Logo', 'buildfuly' ); ?></label>
						</th>
						<td>
							<?php 
							// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Template variable
							$logo_id = get_option( 'buildfuly_business_logo_id', '' );
							$logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'medium' ) : '';
							?>
							<div id="buildfuly-logo-preview" style="margin-bottom: 10px;">
								<?php if ( $logo_url ) : ?>
									<img src="<?php echo esc_url( $logo_url ); ?>" style="max-width: 200px; max-height: 100px; border: 1px solid #ddd; border-radius: 4px; padding: 4px;">
								<?php endif; ?>
							</div>
							<input type="hidden" id="buildfuly_business_logo_id" name="buildfuly_business_logo_id" value="<?php echo esc_attr( $logo_id ); ?>">
							<button type="button" class="button" id="buildfuly-upload-logo">
								<span class="dashicons dashicons-upload" style="vertical-align: middle;"></span>
								<?php esc_html_e( 'Upload Logo', 'buildfuly' ); ?>
							</button>
							<?php if ( $logo_url ) : ?>
							<button type="button" class="button" id="buildfuly-remove-logo" style="color: #dc2626;">
								<span class="dashicons dashicons-trash" style="vertical-align: middle;"></span>
								<?php esc_html_e( 'Remove', 'buildfuly' ); ?>
							</button>
							<?php endif; ?>
							<p class="description"><?php esc_html_e( 'Your logo is used for AI image generation branding and Local SEO schema.', 'buildfuly' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="buildfuly_phone"><?php esc_html_e( 'Phone Number', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="tel" 
								   id="buildfuly_phone" 
								   name="buildfuly_phone" 
								   value="<?php echo esc_attr( get_option( 'buildfuly_phone', '' ) ); ?>" 
								   class="regular-text">
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="buildfuly_address"><?php esc_html_e( 'Address', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="text" 
								   id="buildfuly_address" 
								   name="buildfuly_address" 
								   value="<?php echo esc_attr( get_option( 'buildfuly_address', '' ) ); ?>" 
								   class="large-text">
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="buildfuly_location_relevant">
								<?php esc_html_e( 'Location Relevant', 'buildfuly' ); ?>
							</label>
						</th>
						<td>
							<label>
								<input type="checkbox" 
									   id="buildfuly_location_relevant" 
									   name="buildfuly_location_relevant" 
									   value="1"
									   <?php checked( get_option( 'buildfuly_location_relevant', false ) ); ?>>
								<?php esc_html_e( 'Location is relevant to my business (use in keywords)', 'buildfuly' ); ?>
							</label>
						</td>
					</tr>
					<tr id="buildfuly_location_row" style="<?php echo get_option( 'buildfuly_location_relevant', false ) ? '' : 'display: none;'; ?>">
						<th scope="row">
							<label for="buildfuly_location"><?php esc_html_e( 'Location (City)', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="text" 
								   id="buildfuly_location" 
								   name="buildfuly_location" 
								   value="<?php echo esc_attr( get_option( 'buildfuly_location', '' ) ); ?>" 
								   class="regular-text"
								   placeholder="e.g., Miami">
							<p class="description"><?php esc_html_e( 'Used in SEO keywords and content generation for local businesses', 'buildfuly' ); ?></p>
						</td>
					</tr>
					<tr>
					<th scope="row">
						<label for="buildfuly_review_platform"><?php esc_html_e( 'Review Platform', 'buildfuly' ); ?></label>
					</th>
					<td>
						<?php 
						// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Template variable
						$current_platform = strtolower( get_option( 'buildfuly_review_platform', 'yelp' ) ); 
						?>
						<select id="buildfuly_review_platform" name="buildfuly_review_platform" class="regular-text">
								<option value="google" <?php selected( $current_platform, 'google' ); ?>>Google</option>
								<option value="yelp" <?php selected( $current_platform, 'yelp' ); ?>>Yelp</option>
								<option value="facebook" <?php selected( $current_platform, 'facebook' ); ?>>Facebook</option>
								<option value="trustpilot" <?php selected( $current_platform, 'trustpilot' ); ?>>Trustpilot</option>
							</select>
							<p class="description">
								<?php esc_html_e( 'Select the review platform. The URL will be auto-generated based on your business name, or you can provide a custom URL below.', 'buildfuly' ); ?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="buildfuly_review_url"><?php esc_html_e( 'Review URL (Optional)', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="url" 
								   id="buildfuly_review_url" 
								   name="buildfuly_review_url" 
								   value="<?php echo esc_attr( get_option( 'buildfuly_review_url', '' ) ); ?>" 
								   class="regular-text"
								   placeholder="https://www.yelp.com/biz/your-business">
							<p class="description">
								<?php esc_html_e( 'Optional: Provide your direct review page URL. If empty, will auto-generate based on your business name.', 'buildfuly' ); ?>
							</p>
						</td>
					</tr>
				</table>
			</div>

			<!-- Navigation & Footer Settings -->
			<div class="buildfuly-card">
				<h3><?php esc_html_e( '🧭 Navigation & Footer', 'buildfuly' ); ?></h3>
				<p class="description">
					<?php esc_html_e( 'Control the auto-generated header navigation and footer on your Buildfuly pages.', 'buildfuly' ); ?>
				</p>
				
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="buildfuly_takeover_layout"><?php esc_html_e( 'Use Buildfuly Layout', 'buildfuly' ); ?></label>
						</th>
						<td>
							<label class="buildfuly-toggle">
								<input type="checkbox" 
									   id="buildfuly_takeover_layout" 
									   name="buildfuly_takeover_layout" 
									   value="1"
									   <?php checked( get_option( 'buildfuly_takeover_layout', '1' ), '1' ); ?>>
								<span class="buildfuly-toggle-slider"></span>
							</label>
							<p class="description">
								<?php esc_html_e( 'Replace your theme\'s header and footer with Buildfuly\'s optimized navigation on generated pages. This hides your theme\'s default header/footer and shows our SEO-friendly version.', 'buildfuly' ); ?>
							</p>
						</td>
					</tr>
					<tr class="buildfuly-layout-option">
						<th scope="row">
							<label for="buildfuly_enable_header"><?php esc_html_e( 'Show Header Navigation', 'buildfuly' ); ?></label>
						</th>
						<td>
							<label class="buildfuly-toggle">
								<input type="checkbox" 
									   id="buildfuly_enable_header" 
									   name="buildfuly_enable_header" 
									   value="1"
									   <?php checked( get_option( 'buildfuly_enable_header', '1' ), '1' ); ?>>
								<span class="buildfuly-toggle-slider"></span>
							</label>
							<p class="description">
								<?php esc_html_e( 'Display a sticky header with your logo, navigation menu with Services dropdown, and phone CTA button.', 'buildfuly' ); ?>
							</p>
						</td>
					</tr>
					<tr class="buildfuly-layout-option">
						<th scope="row">
							<label for="buildfuly_enable_footer"><?php esc_html_e( 'Show Footer with NAP', 'buildfuly' ); ?></label>
						</th>
						<td>
							<label class="buildfuly-toggle">
								<input type="checkbox" 
									   id="buildfuly_enable_footer" 
									   name="buildfuly_enable_footer" 
									   value="1"
									   <?php checked( get_option( 'buildfuly_enable_footer', '1' ), '1' ); ?>>
								<span class="buildfuly-toggle-slider"></span>
							</label>
							<p class="description">
								<?php esc_html_e( 'Display a professional footer with your business NAP (Name, Address, Phone), service links, and Schema.org LocalBusiness structured data.', 'buildfuly' ); ?>
							</p>
						</td>
					</tr>
				</table>
				
				<div class="buildfuly-info-box" style="background: #eef6ff; border-left: 4px solid #2271b1; padding: 12px 16px; border-radius: 4px; margin-top: 16px;">
					<p style="margin: 0; color: #1e3a5f;">
						<strong>💡 Tip:</strong> <?php esc_html_e( 'The Services dropdown automatically populates from pages tagged as "service". Pages are auto-tagged when generated based on their content.', 'buildfuly' ); ?>
					</p>
				</div>
			</div>

			<!-- License Activation -->
			<div class="buildfuly-card">
				<h3><?php esc_html_e( '🔑 License Activation', 'buildfuly' ); ?></h3>
				
				<?php
				// Get current license information
				// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Template variables
				$license_key = get_option( 'buildfuly_license_key', '' );
				$license_status = get_option( 'buildfuly_license_status', 'inactive' );
				$license_data = get_option( 'buildfuly_license_data', array() );
				$user_tier = get_option( 'buildfuly_user_tier', '' );
				
				// Auto-register free license if user has no license at all (failsafe)
				if ( empty( $license_key ) && $license_status === 'inactive' ) {
					require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-buildfuly-activator.php';
					Buildfuly_Activator::register_free_license();
					// Refresh data after registration
					$license_key = get_option( 'buildfuly_license_key', '' );
					$license_status = get_option( 'buildfuly_license_status', 'inactive' );
					$license_data = get_option( 'buildfuly_license_data', array() );
					$user_tier = get_option( 'buildfuly_user_tier', '' );
				}
				
				// Handle activation/deactivation via AJAX or form submission
				$license_message = '';
				$license_message_type = '';
				
				if ( isset( $_POST['buildfuly_activate_license'] ) && check_admin_referer( 'buildfuly_license_action', 'buildfuly_license_nonce' ) ) {
					$new_license_key = isset( $_POST['buildfuly_license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['buildfuly_license_key'] ) ) : '';
					
					require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-buildfuly-api-client.php';
					require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-buildfuly-license.php';
					$license_manager = new Buildfuly_License();
					
					$result = $license_manager->activate( $new_license_key );
					
					if ( is_wp_error( $result ) ) {
						$license_message = $result->get_error_message();
						$license_message_type = 'error';
					} else {
						$license_message = 'License activated successfully!';
						$license_message_type = 'success';
						// Refresh data
						$license_key = get_option( 'buildfuly_license_key', '' );
						$license_status = get_option( 'buildfuly_license_status', 'inactive' );
						$license_data = get_option( 'buildfuly_license_data', array() );
						$user_tier = get_option( 'buildfuly_user_tier', '' );
					}
				}
				
				if ( isset( $_POST['buildfuly_deactivate_license'] ) && check_admin_referer( 'buildfuly_license_action', 'buildfuly_license_nonce' ) ) {
					require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-buildfuly-api-client.php';
					require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-buildfuly-license.php';
					$license_manager = new Buildfuly_License();
					
					$result = $license_manager->deactivate();
					
					if ( is_wp_error( $result ) ) {
						$license_message = $result->get_error_message();
						$license_message_type = 'error';
					} else {
						$license_message = 'License deactivated successfully.';
						$license_message_type = 'success';
						// Refresh data
						$license_key = '';
						$license_status = 'inactive';
						$license_data = array();
						$user_tier = '';
					}
				}
				?>
				
				<?php if ( $license_message ) : ?>
					<div class="notice notice-<?php echo esc_attr( $license_message_type ); ?> inline">
						<p><?php echo esc_html( $license_message ); ?></p>
					</div>
				<?php endif; ?>
				
				<?php if ( $license_status === 'active' && ! empty( $license_key ) ) : ?>
					<!-- Active License -->
					<div class="license-status-active" style="padding: 20px; background: #d4edda; border: 2px solid #34c759; border-radius: 12px; margin-bottom: 20px;">
						<div style="display: flex; align-items: center; gap: 12px; margin-bottom: 16px;">
							<span class="dashicons dashicons-yes-alt" style="color: #155724; font-size: 32px; width: 32px; height: 32px;"></span>
							<div>
								<h4 style="margin: 0; color: #155724;">✅ License Active</h4>
								<p style="margin: 4px 0 0; color: #155724;">Your Buildfuly license is active and working.</p>
							</div>
						</div>
						
						<table class="form-table">
							<tr>
								<th scope="row"><?php esc_html_e( 'License Key', 'buildfuly' ); ?></th>
								<td>
									<code style="background: #fff; padding: 8px 12px; border-radius: 4px; display: inline-block;">
										<?php echo esc_html( $license_key ); ?>
									</code>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php esc_html_e( 'Plan', 'buildfuly' ); ?></th>
								<td>
									<strong><?php echo esc_html( ucfirst( $user_tier ) ); ?></strong>
									<?php if ( isset( $license_data['license']['pages_per_day'] ) ) : ?>
										<br>
										<span class="description">
											<?php echo esc_html( $license_data['license']['pages_per_day'] ); ?> pages per day
										</span>
									<?php endif; ?>
								</td>
							</tr>
							<tr>
								<th scope="row"><?php esc_html_e( 'Usage Today', 'buildfuly' ); ?></th>
								<td>
									<?php
									// Get current usage stats from API
									require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-buildfuly-api-client.php';
									$api_client = new Buildfuly_API_Client();
									$usage_stats = $api_client->get_usage_stats();
									
									if ( ! is_wp_error( $usage_stats ) && isset( $usage_stats['remaining'] ) ) :
										$remaining = $usage_stats['remaining'];
										$remaining_words = isset( $usage_stats['remaining_words'] ) ? $usage_stats['remaining_words'] : floor( $remaining * 600 );
										$limit = isset( $usage_stats['limit'] ) ? $usage_stats['limit'] : 5;
										$used = $limit - $remaining;
									?>
										<strong style="color: #10b981;"><?php echo esc_html( number_format( $remaining, 1 ) ); ?> pages remaining</strong>
										<br>
										<span class="description">
											<?php echo esc_html( number_format( $remaining_words ) ); ?> words left today
											(<?php echo esc_html( number_format( $used, 1 ) ); ?> / <?php echo esc_html( $limit ); ?> pages used)
										</span>
									<?php else : ?>
										<span class="description" style="color: #666;">
											Go to Page Builder to see current usage
										</span>
									<?php endif; ?>
								</td>
							</tr>
							<?php if ( isset( $license_data['user']['email'] ) ) : ?>
							<tr>
								<th scope="row"><?php esc_html_e( 'Email', 'buildfuly' ); ?></th>
								<td><?php echo esc_html( $license_data['user']['email'] ); ?></td>
							</tr>
							<?php endif; ?>
						</table>
					</div>
					
					<!-- Deactivate Button (AJAX) -->
					<button type="button" 
							id="buildfuly-deactivate-license-btn" 
							class="button button-secondary"
							style="margin-top: 16px;">
						<?php esc_html_e( 'Deactivate License', 'buildfuly' ); ?>
					</button>
					<div id="license-deactivate-status" style="margin-top: 12px; display: none;"></div>
					
				<?php else : ?>
					<!-- No Active License -->
					<?php 
					$is_free_license = get_option( 'buildfuly_free_license', false );
					?>
					<div class="license-status-inactive" style="padding: 20px; background: #fff3cd; border: 2px solid #fbbf24; border-radius: 12px; margin-bottom: 20px;">
						<div style="display: flex; align-items: center; gap: 12px; margin-bottom: 16px;">
							<span class="dashicons dashicons-warning" style="color: #856404; font-size: 32px; width: 32px; height: 32px;"></span>
							<div>
								<?php if ( $is_free_license ) : ?>
									<h4 style="margin: 0; color: #856404;">🎁 Free Starter License Active</h4>
									<p style="margin: 4px 0 0; color: #856404;">
										You're on the free tier (3 pages). Enter a paid license key below to upgrade.
										<a href="https://build.fuly.io/pricing" target="_blank" style="color: #007aff; font-weight: 600; text-decoration: none;">
											View Pricing Plans →
										</a>
									</p>
								<?php else : ?>
									<h4 style="margin: 0; color: #856404;">⚠️ No Active License</h4>
									<p style="margin: 4px 0 0; color: #856404;">
										Activate your license to unlock all features. 
										<a href="https://build.fuly.io/pricing" target="_blank" style="color: #007aff; font-weight: 600; text-decoration: none;">
											View Pricing Plans →
										</a>
									</p>
								<?php endif; ?>
							</div>
						</div>
					</div>
					
					<!-- Activate License (AJAX) - Always show for upgrades -->
					<table class="form-table">
						<tr>
							<th scope="row">
								<label for="buildfuly_license_key_input">
									<?php echo $is_free_license ? esc_html__( 'Upgrade License Key', 'buildfuly' ) : esc_html__( 'License Key', 'buildfuly' ); ?>
								</label>
							</th>
							<td>
								<input type="text" 
									   id="buildfuly_license_key_input" 
									   value="<?php echo esc_attr( $license_key ); ?>" 
									   class="regular-text" 
									   placeholder="XXXX-XXXX-XXXX-XXXX">
								<p class="description">
									<?php 
									if ( $is_free_license ) {
										esc_html_e( 'Enter your paid license key to upgrade from the free tier.', 'buildfuly' ); 
									} else {
										esc_html_e( 'Enter your license key from your purchase email.', 'buildfuly' );
									}
									?>
								</p>
							</td>
						</tr>
					</table>
					
					<p class="submit" style="margin: 16px 0 0; padding: 0;">
						<button type="button" 
								id="buildfuly-activate-license-btn" 
								class="button button-primary">
							<?php echo $is_free_license ? esc_html__( 'Upgrade License', 'buildfuly' ) : esc_html__( 'Activate License', 'buildfuly' ); ?>
						</button>
					</p>
					<div id="license-activate-status" style="margin-top: 12px; display: none;"></div>
				<?php endif; ?>
			</div>

		</div>

		<!-- Floating Save Button -->
		<div id="buildfuly-floating-save" class="buildfuly-floating-save">
			<?php submit_button( __( 'Save All Settings', 'buildfuly' ), 'primary large', 'submit', false ); ?>
		</div>
	</form>
	
	<!-- Reset Onboarding Button -->
	<div class="buildfuly-reset-section" style="margin-top: 40px; padding: 24px; background: #fff; border-radius: 12px; border: 2px solid #fef3c7;">
		<h3 style="margin-top: 0; color: #92400e;">⚠️ Reset Onboarding</h3>
		<p style="color: #78350f; margin-bottom: 16px;">
			This will clear all your settings and pages, allowing you to go through the onboarding process again. 
			<strong>This action cannot be undone.</strong>
		</p>
		<button type="button" id="buildfuly-reset-onboarding" class="button button-secondary">
			<span class="dashicons dashicons-update"></span> Reset & Start Over
		</button>
	</div>
</div>

<style>
.buildfuly-container {
	max-width: 1000px;
	margin: 0 auto;
}

.buildfuly-card {
	background: #fff;
	border-radius: 16px;
	padding: 36px;
	margin-bottom: 24px;
	box-shadow: 0 1px 3px rgba(0,0,0,0.1);
	border: 1px solid #e5e7eb;
	transition: all 0.2s;
}

.buildfuly-card:hover {
	box-shadow: 0 4px 12px rgba(0,0,0,0.08);
	transform: translateY(-2px);
}

.buildfuly-card h3 {
	margin-top: 0;
	font-size: 22px;
	font-weight: 700;
	color: #111827;
	margin-bottom: 8px;
}

.buildfuly-card > .description {
	color: #6b7280;
	font-size: 15px;
	margin-bottom: 24px;
}

.form-table th {
	font-weight: 600;
	color: #374151;
}

.form-table input[type="text"],
.form-table input[type="url"],
.form-table input[type="tel"],
.form-table input[type="password"],
.form-table textarea,
.form-table select {
	border: 2px solid #e5e7eb;
	border-radius: 8px;
	padding: 10px 14px;
	transition: all 0.2s;
}

.form-table input:focus,
.form-table textarea:focus,
.form-table select:focus {
	border-color: #667eea;
	box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
	outline: none;
}

.button-row {
	transition: all 0.2s;
}

.button-row:hover {
	transform: translateX(4px);
}

#add-button-btn {
	background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
	color: white;
	border: none;
	padding: 12px 24px;
	border-radius: 8px;
	font-weight: 600;
	transition: all 0.2s;
}

#add-button-btn:hover {
	transform: translateY(-2px);
	box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
}

/* Floating Save Button - Always visible */
.buildfuly-floating-save {
	position: fixed;
	bottom: 30px;
	right: 30px;
	z-index: 9999;
	background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
	padding: 16px 24px;
	border-radius: 12px;
	box-shadow: 0 10px 40px rgba(102, 126, 234, 0.4);
	opacity: 0;
	transform: translateY(20px);
	transition: all 0.3s ease;
	pointer-events: none;
}
.buildfuly-floating-save.visible {
	opacity: 1;
	transform: translateY(0);
	pointer-events: all;
}
.buildfuly-floating-save .button {
	margin: 0;
	font-size: 16px;
	padding: 12px 32px;
	height: auto;
	background: white;
	color: #667eea;
	border: none;
	font-weight: 600;
	box-shadow: none;
}
.buildfuly-floating-save .button:hover {
	background: #f9fafb;
	transform: translateY(-2px);
}

/* Toggle Switch Styles */
.buildfuly-toggle {
	position: relative;
	display: inline-block;
	width: 52px;
	height: 28px;
}
.buildfuly-toggle input {
	opacity: 0;
	width: 0;
	height: 0;
}
.buildfuly-toggle-slider {
	position: absolute;
	cursor: pointer;
	top: 0;
	left: 0;
	right: 0;
	bottom: 0;
	background-color: #ccc;
	transition: 0.3s;
	border-radius: 28px;
}
.buildfuly-toggle-slider:before {
	position: absolute;
	content: "";
	height: 20px;
	width: 20px;
	left: 4px;
	bottom: 4px;
	background-color: white;
	transition: 0.3s;
	border-radius: 50%;
	box-shadow: 0 2px 4px rgba(0,0,0,0.2);
}
.buildfuly-toggle input:checked + .buildfuly-toggle-slider {
	background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
}
.buildfuly-toggle input:checked + .buildfuly-toggle-slider:before {
	transform: translateX(24px);
}
.buildfuly-layout-option {
	transition: opacity 0.3s, max-height 0.3s;
}
.buildfuly-layout-option.hidden {
	opacity: 0.5;
	pointer-events: none;
}
</style>

<script>
jQuery(document).ready(function($) {
	// Navigation Layout Toggle
	function updateLayoutOptions() {
		const takeoverEnabled = $('#buildfuly_takeover_layout').is(':checked');
		$('.buildfuly-layout-option').toggleClass('hidden', !takeoverEnabled);
	}
	
	$('#buildfuly_takeover_layout').on('change', updateLayoutOptions);
	updateLayoutOptions(); // Run on page load
	
	// Toggle descriptive mode fields
	$('input[name="buildfuly_generation_mode"]').on('change', function() {
		if ($(this).val() === 'descriptive') {
			$('.descriptive-mode-field').slideDown();
		} else {
			$('.descriptive-mode-field').slideUp();
		}
	});

	// Button management
	let buttonIndex = <?php echo count( $cta_buttons ); ?>;

	$('#add-button-btn').on('click', function() {
		const newButton = `
			<div class="button-row" data-index="${buttonIndex}" style="background: #f9fafb; padding: 20px; border-radius: 8px; margin-bottom: 16px; border: 2px solid #e5e7eb;">
				<div style="display: flex; align-items: center; margin-bottom: 12px;">
					<h4 style="margin: 0;">Button ${buttonIndex + 1}</h4>
					<button type="button" class="button remove-button-btn" data-index="${buttonIndex}">Remove</button>
				</div>
				<table class="form-table" style="margin: 0;">
					<tr>
						<th style="width: 120px; padding-left: 0;"><label>Button Text</label></th>
						<td><input type="text" name="button_text_${buttonIndex}" class="regular-text" required></td>
					</tr>
					<tr>
						<th style="padding-left: 0;"><label>Link URL</label></th>
						<td>
							<input type="text" name="button_url_${buttonIndex}" class="large-text" placeholder="https://example.com or tel:+15551234567 or #contact" required>
							<p class="description">Enter a URL (https://...), phone number (tel:+1...), email (mailto:...), or anchor link (#section)</p>
						</td>
					</tr>
					<tr>
						<th style="padding-left: 0;"><label>Style</label></th>
						<td>
							<select name="button_style_${buttonIndex}" class="regular-text">
								<option value="primary">Primary</option>
								<option value="secondary">Secondary</option>
							</select>
						</td>
					</tr>
				</table>
			</div>
		`;
		$('#button-list').append(newButton);
		buttonIndex++;
	});

	$(document).on('click', '.remove-button-btn', function() {
		$(this).closest('.button-row').slideUp(300, function() {
			$(this).remove();
		});
	});

	// Collect button data before submit
	$('#buildfuly-unified-settings-form').on('submit', function() {
		const buttons = [];
		$('.button-row').each(function() {
			const index = $(this).data('index');
			const text = $(`input[name="button_text_${index}"]`).val();
			let url = $(`input[name="button_url_${index}"]`).val();
			const style = $(`select[name="button_style_${index}"]`).val();
			
			if (text && url) {
				// Auto-add https:// if no protocol specified (unless it's tel:, mailto:, or #)
				if (url && !url.match(/^(https?:\/\/|tel:|mailto:|#)/)) {
					url = 'https://' + url;
				}
				
				buttons.push({ text: text, url: url, style: style });
			}
		});
		
		$('#buildfuly_cta_buttons').val(JSON.stringify(buttons));
	});

	// Floating save button - always visible
	const $floatingSave = $('#buildfuly-floating-save');
	
	// Show immediately on page load
	setTimeout(function() {
		$floatingSave.addClass('visible');
	}, 100);
	
	// Keep visible when scrolling down
	$(window).on('scroll', function() {
		const scrollTop = $(window).scrollTop();
		
		if (scrollTop > 50) {
			$floatingSave.addClass('visible');
		} else {
			$floatingSave.removeClass('visible');
		}
	});
	
	// Regenerate all published pages
	$('#buildfuly-regenerate-pages').on('click', function() {
		if (!confirm('This will update all published pages with your current settings. This may take a moment. Continue?')) {
			return;
		}
		
		const $btn = $(this);
		const $status = $('#regenerate-status');
		
		$btn.prop('disabled', true).html('<span class="spinner is-active" style="float: none; margin: 0 8px 0 0;"></span> Updating...');
		$status.hide();
		
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_regenerate_pages',
				nonce: buildfulyAdmin.nonce
			},
			success: function(response) {
				if (response.success) {
					$status.html('<span style="color: #10b981;">✓ ' + response.data.message + '</span>').show();
					$btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Update All Published Pages');
				} else {
					$status.html('<span style="color: #ef4444;">✗ Error: ' + (response.data || 'Unknown error') + '</span>').show();
					$btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Update All Published Pages');
				}
			},
			error: function() {
				$status.html('<span style="color: #ef4444;">✗ Error connecting to server. Please try again.</span>').show();
				$btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Update All Published Pages');
			}
		});
	});
	
	// Reset onboarding
	$('#buildfuly-reset-onboarding').on('click', function() {
		if (!confirm('Are you sure you want to reset everything and start over? This will:\n\n• Clear all your business settings\n• Delete all pages\n• Reset all configurations\n\nThis action cannot be undone!')) {
			return;
		}
		
		const $btn = $(this);
		$btn.prop('disabled', true).html('<span class="spinner is-active" style="float: none; margin: 0 8px 0 0;"></span> Resetting...');
		
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_reset_onboarding',
				nonce: buildfulyAdmin.nonce
			},
			success: function(response) {
				if (response.success) {
					// Clear localStorage
					localStorage.removeItem('buildfuly_pages');
					
					// Show success message and reload
					alert('Settings reset successfully! Redirecting to onboarding...');
					window.location.reload();
				} else {
					alert('Error resetting settings: ' + (response.data || 'Unknown error'));
					$btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Reset & Start Over');
				}
			},
			error: function() {
				alert('Error connecting to server. Please try again.');
				$btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Reset & Start Over');
			}
		});
	});
	
	// License activation via AJAX
	$('#buildfuly-activate-license-btn').on('click', function() {
		const $btn = $(this);
		const $status = $('#license-activate-status');
		const licenseKey = $('#buildfuly_license_key_input').val().trim();
		
		if (!licenseKey) {
			$status.html('<span style="color: #ef4444;">Please enter a license key.</span>').show();
			return;
		}
		
		$btn.prop('disabled', true).html('<span class="spinner is-active" style="float: none; margin: 0 8px 0 0;"></span> Activating...');
		$status.hide();
		
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_activate_license',
				nonce: buildfulyAdmin.adminNonce,
				license_key: licenseKey
			},
			success: function(response) {
				if (response.success) {
					$status.html('<span style="color: #10b981;">✓ License activated successfully! Reloading...</span>').show();
					setTimeout(function() {
						window.location.reload();
					}, 1500);
				} else {
					$status.html('<span style="color: #ef4444;">✗ ' + (response.data || 'Activation failed') + '</span>').show();
					$btn.prop('disabled', false).html('<?php esc_html_e( 'Activate License', 'buildfuly' ); ?>');
				}
			},
			error: function() {
				$status.html('<span style="color: #ef4444;">✗ Error connecting to server. Please try again.</span>').show();
				$btn.prop('disabled', false).html('<?php esc_html_e( 'Activate License', 'buildfuly' ); ?>');
			}
		});
	});
	
	// Toggle location field visibility
	$('#buildfuly_location_relevant').on('change', function() {
		if ($(this).is(':checked')) {
			$('#buildfuly_location_row').slideDown(300);
		} else {
			$('#buildfuly_location_row').slideUp(300);
		}
	});
	
	// License deactivation via AJAX
	$('#buildfuly-deactivate-license-btn').on('click', function() {
		if (!confirm('Are you sure you want to deactivate this license?')) {
			return;
		}
		
		const $btn = $(this);
		const $status = $('#license-deactivate-status');
		
		$btn.prop('disabled', true).html('<span class="spinner is-active" style="float: none; margin: 0 8px 0 0;"></span> Deactivating...');
		$status.hide();
		
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_deactivate_license',
				nonce: buildfulyAdmin.adminNonce
			},
			success: function(response) {
				if (response.success) {
					$status.html('<span style="color: #10b981;">✓ License deactivated successfully! Reloading...</span>').show();
					setTimeout(function() {
						window.location.reload();
					}, 1500);
				} else {
					$status.html('<span style="color: #ef4444;">✗ ' + (response.data || 'Deactivation failed') + '</span>').show();
					$btn.prop('disabled', false).html('<?php esc_html_e( 'Deactivate License', 'buildfuly' ); ?>');
				}
			},
			error: function() {
				$status.html('<span style="color: #ef4444;">✗ Error connecting to server. Please try again.</span>').show();
				$btn.prop('disabled', false).html('<?php esc_html_e( 'Deactivate License', 'buildfuly' ); ?>');
			}
		});
	});
	
	// Logo upload via WordPress Media Library
	$('#buildfuly-upload-logo').on('click', function(e) {
		e.preventDefault();
		
		const frame = wp.media({
			title: '<?php esc_html_e( 'Select Business Logo', 'buildfuly' ); ?>',
			button: { text: '<?php esc_html_e( 'Use as Logo', 'buildfuly' ); ?>' },
			multiple: false,
			library: { type: 'image' }
		});
		
		frame.on('select', function() {
			const attachment = frame.state().get('selection').first().toJSON();
			const imageUrl = attachment.sizes?.medium?.url || attachment.url;
			
			$('#buildfuly_business_logo_id').val(attachment.id);
			$('#buildfuly-logo-preview').html(
				'<img src="' + imageUrl + '" style="max-width: 200px; max-height: 100px; border: 1px solid #ddd; border-radius: 4px; padding: 4px;">'
			);
			
			// Add remove button if not present
			if ($('#buildfuly-remove-logo').length === 0) {
				$('#buildfuly-upload-logo').after(
					'<button type="button" class="button" id="buildfuly-remove-logo" style="color: #dc2626; margin-left: 8px;">' +
					'<span class="dashicons dashicons-trash" style="vertical-align: middle;"></span> <?php esc_html_e( 'Remove', 'buildfuly' ); ?></button>'
				);
			}
		});
		
		frame.open();
	});
	
	// Remove logo
	$(document).on('click', '#buildfuly-remove-logo', function() {
		$('#buildfuly_business_logo_id').val('');
		$('#buildfuly-logo-preview').empty();
		$(this).remove();
	});
});
</script>
