<?php
/**
 * SEO Settings Page - Complete SEO Dashboard
 *
 * @link       https://build.fuly.io
 * @since      1.0.5
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/admin/partials
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

// Get all SEO settings
$seo_settings = get_option( 'buildfuly_seo_settings', array() );

// Default values
$defaults = array(
	// Sitemap
	'sitemap_enabled' => true,
	'sitemap_posts' => true,
	'sitemap_pages' => true,
	'sitemap_images' => true,
	
	// Robots.txt
	'robots_txt' => "User-agent: *\nAllow: /\nDisallow: /wp-admin/\nDisallow: /wp-includes/\n\nSitemap: " . home_url( '/sitemap.xml' ),
	
	// Analytics
	'google_analytics_id' => '',
	'google_tag_manager_id' => '',
	
	// Search Console
	'google_verification' => '',
	'bing_verification' => '',
	
	// Schema/Local SEO
	'schema_enabled' => true,
	'schema_type' => 'LocalBusiness',
	'business_name' => get_option( 'buildfuly_business_name', '' ),
	'business_phone' => get_option( 'buildfuly_phone', '' ),
	'business_email' => get_option( 'admin_email', '' ),
	'business_address' => get_option( 'buildfuly_address', '' ),
	'business_city' => get_option( 'buildfuly_city', '' ),
	'business_state' => get_option( 'buildfuly_state', '' ),
	'business_zip' => get_option( 'buildfuly_zip', '' ),
	'business_country' => 'US',
	'business_hours' => '',
	'business_price_range' => '$$',
	'business_latitude' => '',
	'business_longitude' => '',
	'business_logo' => '',
	'business_image' => '',
	'service_area' => '',
	
	// Google Maps
	'google_maps_enabled' => false,
	'google_maps_api_key' => '',
	'google_maps_embed_url' => '',
	'google_maps_footer' => false,
	'google_maps_height' => '300',
	'google_maps_zoom' => '15',
	
	// Social Profiles
	'social_facebook' => '',
	'social_twitter' => '',
	'social_instagram' => '',
	'social_linkedin' => '',
	'social_youtube' => '',
	'social_yelp' => '',
	'social_google_business' => '',
	
	// Default Meta
	'default_og_image' => '',
	'twitter_card_type' => 'summary_large_image',
	
	// Redirects
	'redirects' => array(),
);

$settings = wp_parse_args( $seo_settings, $defaults );

// Calculate SEO completion score
function buildfuly_calculate_seo_score( $settings ) {
	$score = 0;
	$total_items = 0;
	$checklist = array();
	
	// Core Business Info (30 points)
	$checklist['business_name'] = array(
		'label' => 'Business Name',
		'complete' => ! empty( $settings['business_name'] ),
		'weight' => 5,
		'category' => 'Business Info'
	);
	$checklist['business_phone'] = array(
		'label' => 'Business Phone',
		'complete' => ! empty( $settings['business_phone'] ),
		'weight' => 5,
		'category' => 'Business Info'
	);
	$checklist['business_address'] = array(
		'label' => 'Business Address',
		'complete' => ! empty( $settings['business_address'] ) && ! empty( $settings['business_city'] ),
		'weight' => 5,
		'category' => 'Business Info'
	);
	$checklist['business_email'] = array(
		'label' => 'Business Email',
		'complete' => ! empty( $settings['business_email'] ),
		'weight' => 3,
		'category' => 'Business Info'
	);
	
	// Technical SEO (25 points)
	$checklist['sitemap'] = array(
		'label' => 'XML Sitemap Enabled',
		'complete' => ! empty( $settings['sitemap_enabled'] ),
		'weight' => 8,
		'category' => 'Technical SEO'
	);
	$checklist['google_verification'] = array(
		'label' => 'Google Search Console Verified',
		'complete' => ! empty( $settings['google_verification'] ),
		'weight' => 10,
		'category' => 'Technical SEO'
	);
	$checklist['robots_txt'] = array(
		'label' => 'Robots.txt Configured',
		'complete' => ! empty( $settings['robots_txt'] ),
		'weight' => 5,
		'category' => 'Technical SEO'
	);
	
	// Analytics (15 points)
	$checklist['analytics'] = array(
		'label' => 'Google Analytics Connected',
		'complete' => ! empty( $settings['google_analytics_id'] ) || ! empty( $settings['google_tag_manager_id'] ),
		'weight' => 15,
		'category' => 'Analytics'
	);
	
	// Social (15 points)
	$has_social = ! empty( $settings['social_facebook'] ) || ! empty( $settings['social_instagram'] ) || ! empty( $settings['social_linkedin'] );
	$checklist['social_profiles'] = array(
		'label' => 'Social Profiles Added',
		'complete' => $has_social,
		'weight' => 8,
		'category' => 'Social'
	);
	$checklist['og_image'] = array(
		'label' => 'Default Social Share Image',
		'complete' => ! empty( $settings['default_og_image'] ),
		'weight' => 7,
		'category' => 'Social'
	);
	
	// Schema (15 points)
	$checklist['schema_type'] = array(
		'label' => 'Schema Type Selected',
		'complete' => ! empty( $settings['schema_type'] ) && ! empty( $settings['schema_enabled'] ),
		'weight' => 8,
		'category' => 'Schema'
	);
	$checklist['business_hours'] = array(
		'label' => 'Business Hours Set',
		'complete' => ! empty( $settings['business_hours'] ),
		'weight' => 4,
		'category' => 'Schema'
	);
	$checklist['geo_coordinates'] = array(
		'label' => 'Location Coordinates Set',
		'complete' => ! empty( $settings['business_latitude'] ) && ! empty( $settings['business_longitude'] ),
		'weight' => 3,
		'category' => 'Schema'
	);
	
	// Google Maps (bonus)
	$checklist['google_maps'] = array(
		'label' => 'Google Maps Embedded',
		'complete' => ! empty( $settings['google_maps_enabled'] ) && ( ! empty( $settings['google_maps_embed_url'] ) || ! empty( $settings['google_maps_api_key'] ) ),
		'weight' => 5,
		'category' => 'Local SEO'
	);
	
	// Calculate score
	$max_score = 0;
	foreach ( $checklist as $item ) {
		$max_score += $item['weight'];
		if ( $item['complete'] ) {
			$score += $item['weight'];
		}
	}
	
	return array(
		'score' => $score,
		'max_score' => $max_score,
		'percentage' => $max_score > 0 ? round( ( $score / $max_score ) * 100 ) : 0,
		'checklist' => $checklist
	);
}

$seo_score = buildfuly_calculate_seo_score( $settings );

// Schema types for dropdown
$schema_types = array(
	'LocalBusiness' => 'Local Business (General)',
	'HomeAndConstructionBusiness' => 'Home & Construction',
	'Plumber' => 'Plumber',
	'Electrician' => 'Electrician',
	'RoofingContractor' => 'Roofing Contractor',
	'HVACBusiness' => 'HVAC Business',
	'AutoRepair' => 'Auto Repair',
	'LegalService' => 'Legal Service / Attorney',
	'Dentist' => 'Dentist',
	'Physician' => 'Physician / Doctor',
	'RealEstateAgent' => 'Real Estate Agent',
	'Restaurant' => 'Restaurant',
	'Store' => 'Store / Retail',
	'ProfessionalService' => 'Professional Service',
	'FinancialService' => 'Financial Service',
	'InsuranceAgency' => 'Insurance Agency',
);
?>

<div class="wrap buildfuly-admin-wrap">
	<h1>🔍 <?php esc_html_e( 'SEO Settings', 'buildfuly' ); ?></h1>
	
	<!-- SEO Score Dashboard -->
	<div class="buildfuly-seo-dashboard">
		<div class="buildfuly-card buildfuly-seo-score-card">
			<div class="seo-score-circle" data-score="<?php echo esc_attr( $seo_score['percentage'] ); ?>">
				<svg viewBox="0 0 36 36" class="circular-chart">
					<path class="circle-bg" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831"/>
					<path class="circle" stroke-dasharray="<?php echo esc_attr( $seo_score['percentage'] ); ?>, 100" d="M18 2.0845 a 15.9155 15.9155 0 0 1 0 31.831 a 15.9155 15.9155 0 0 1 0 -31.831"/>
					<text x="18" y="20.35" class="percentage"><?php echo esc_html( $seo_score['percentage'] ); ?>%</text>
				</svg>
			</div>
			<div class="seo-score-info">
				<h2><?php esc_html_e( 'SEO Score', 'buildfuly' ); ?></h2>
				<p class="seo-score-label">
					<?php
					if ( $seo_score['percentage'] >= 80 ) {
						echo '🎉 ' . esc_html__( 'Excellent! Your SEO is well optimized.', 'buildfuly' );
					} elseif ( $seo_score['percentage'] >= 60 ) {
						echo '👍 ' . esc_html__( 'Good progress! A few more items to complete.', 'buildfuly' );
					} elseif ( $seo_score['percentage'] >= 40 ) {
						echo '📈 ' . esc_html__( 'Getting there! Complete the checklist below.', 'buildfuly' );
					} else {
						echo '🚀 ' . esc_html__( 'Let\'s boost your SEO! Start with the basics below.', 'buildfuly' );
					}
					?>
				</p>
			</div>
		</div>
		
		<!-- Checklist -->
		<div class="buildfuly-card buildfuly-seo-checklist">
			<h3><?php esc_html_e( 'SEO Checklist', 'buildfuly' ); ?></h3>
			<div class="checklist-items">
				<?php 
				$categories = array();
				foreach ( $seo_score['checklist'] as $key => $item ) {
					$categories[ $item['category'] ][ $key ] = $item;
				}
				foreach ( $categories as $cat_name => $items ) : ?>
					<div class="checklist-category">
						<h4><?php echo esc_html( $cat_name ); ?></h4>
						<?php foreach ( $items as $key => $item ) : ?>
							<div class="checklist-item <?php echo $item['complete'] ? 'complete' : 'incomplete'; ?>">
								<span class="checklist-icon"><?php echo $item['complete'] ? '✅' : '⬜'; ?></span>
								<span class="checklist-label"><?php echo esc_html( $item['label'] ); ?></span>
							</div>
						<?php endforeach; ?>
					</div>
				<?php endforeach; ?>
			</div>
		</div>
	</div>
	
	<!-- Settings Form -->
	<form method="post" action="" id="buildfuly-seo-settings-form">
		<?php wp_nonce_field( 'buildfuly_seo_settings', 'buildfuly_seo_nonce' ); ?>
		
		<div class="buildfuly-container">
			
			<!-- XML Sitemap -->
			<div class="buildfuly-card">
				<h3>🗺️ <?php esc_html_e( 'XML Sitemap', 'buildfuly' ); ?></h3>
				<p class="description"><?php esc_html_e( 'Automatically generate a sitemap for search engines.', 'buildfuly' ); ?></p>
				
				<table class="form-table">
					<tr>
						<th scope="row"><?php esc_html_e( 'Enable Sitemap', 'buildfuly' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="sitemap_enabled" value="1" <?php checked( $settings['sitemap_enabled'] ); ?>>
								<?php esc_html_e( 'Generate XML sitemap automatically', 'buildfuly' ); ?>
							</label>
							<?php if ( $settings['sitemap_enabled'] ) : ?>
								<p class="description">
									<a href="<?php echo esc_url( home_url( '/sitemap.xml' ) ); ?>" target="_blank">
										<?php echo esc_url( home_url( '/sitemap.xml' ) ); ?>
									</a>
								</p>
							<?php endif; ?>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php esc_html_e( 'Include', 'buildfuly' ); ?></th>
						<td>
							<label style="display: block; margin-bottom: 8px;">
								<input type="checkbox" name="sitemap_pages" value="1" <?php checked( $settings['sitemap_pages'] ); ?>>
								<?php esc_html_e( 'Pages', 'buildfuly' ); ?>
							</label>
							<label style="display: block; margin-bottom: 8px;">
								<input type="checkbox" name="sitemap_posts" value="1" <?php checked( $settings['sitemap_posts'] ); ?>>
								<?php esc_html_e( 'Blog Posts', 'buildfuly' ); ?>
							</label>
							<label style="display: block;">
								<input type="checkbox" name="sitemap_images" value="1" <?php checked( $settings['sitemap_images'] ); ?>>
								<?php esc_html_e( 'Images in sitemap', 'buildfuly' ); ?>
							</label>
						</td>
					</tr>
				</table>
			</div>
			
			<!-- Robots.txt -->
			<div class="buildfuly-card">
				<h3>🤖 <?php esc_html_e( 'Robots.txt', 'buildfuly' ); ?></h3>
				<p class="description"><?php esc_html_e( 'Control how search engines crawl your site.', 'buildfuly' ); ?></p>
				
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="robots_txt"><?php esc_html_e( 'Robots.txt Content', 'buildfuly' ); ?></label>
						</th>
						<td>
							<textarea id="robots_txt" name="robots_txt" rows="8" class="large-text code"><?php echo esc_textarea( $settings['robots_txt'] ); ?></textarea>
							<p class="description">
								<a href="<?php echo esc_url( home_url( '/robots.txt' ) ); ?>" target="_blank"><?php esc_html_e( 'View robots.txt', 'buildfuly' ); ?></a>
							</p>
						</td>
					</tr>
				</table>
			</div>
			
			<!-- Google Search Console & Analytics -->
			<div class="buildfuly-card">
				<h3>📊 <?php esc_html_e( 'Google Integration', 'buildfuly' ); ?></h3>
				<p class="description"><?php esc_html_e( 'Connect Google Search Console and Analytics.', 'buildfuly' ); ?></p>
				
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="google_verification"><?php esc_html_e( 'Search Console Verification', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="text" id="google_verification" name="google_verification" value="<?php echo esc_attr( $settings['google_verification'] ); ?>" class="regular-text" placeholder="google-site-verification code">
							<p class="description">
								<?php esc_html_e( 'Enter the verification code from', 'buildfuly' ); ?>
								<a href="https://search.google.com/search-console" target="_blank">Google Search Console</a>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="google_analytics_id"><?php esc_html_e( 'Google Analytics ID', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="text" id="google_analytics_id" name="google_analytics_id" value="<?php echo esc_attr( $settings['google_analytics_id'] ); ?>" class="regular-text" placeholder="G-XXXXXXXXXX or UA-XXXXXXXX-X">
							<p class="description"><?php esc_html_e( 'Your GA4 Measurement ID or Universal Analytics ID', 'buildfuly' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="google_tag_manager_id"><?php esc_html_e( 'Google Tag Manager', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="text" id="google_tag_manager_id" name="google_tag_manager_id" value="<?php echo esc_attr( $settings['google_tag_manager_id'] ); ?>" class="regular-text" placeholder="GTM-XXXXXXX">
							<p class="description"><?php esc_html_e( 'Optional: If you use GTM instead of direct GA installation', 'buildfuly' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="bing_verification"><?php esc_html_e( 'Bing Verification', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="text" id="bing_verification" name="bing_verification" value="<?php echo esc_attr( $settings['bing_verification'] ); ?>" class="regular-text" placeholder="Bing verification code">
							<p class="description">
								<?php esc_html_e( 'Optional: Verify with', 'buildfuly' ); ?>
								<a href="https://www.bing.com/webmasters" target="_blank">Bing Webmaster Tools</a>
							</p>
						</td>
					</tr>
				</table>
			</div>
			
			<!-- Local Business Schema -->
			<div class="buildfuly-card">
				<h3>🏢 <?php esc_html_e( 'Local Business Information (Schema)', 'buildfuly' ); ?></h3>
				<p class="description"><?php esc_html_e( 'This information is used to generate rich snippets in search results.', 'buildfuly' ); ?></p>
				
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="schema_type"><?php esc_html_e( 'Business Type', 'buildfuly' ); ?></label>
						</th>
						<td>
							<select id="schema_type" name="schema_type" class="regular-text">
								<?php foreach ( $schema_types as $value => $label ) : ?>
									<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $settings['schema_type'], $value ); ?>>
										<?php echo esc_html( $label ); ?>
									</option>
								<?php endforeach; ?>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="business_name"><?php esc_html_e( 'Business Name', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="text" id="business_name" name="business_name" value="<?php echo esc_attr( $settings['business_name'] ); ?>" class="regular-text">
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="business_phone"><?php esc_html_e( 'Phone Number', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="text" id="business_phone" name="business_phone" value="<?php echo esc_attr( $settings['business_phone'] ); ?>" class="regular-text" placeholder="+1-555-123-4567">
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="business_email"><?php esc_html_e( 'Email', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="email" id="business_email" name="business_email" value="<?php echo esc_attr( $settings['business_email'] ); ?>" class="regular-text">
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="business_address"><?php esc_html_e( 'Street Address', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="text" id="business_address" name="business_address" value="<?php echo esc_attr( $settings['business_address'] ); ?>" class="regular-text">
						</td>
					</tr>
					<tr>
						<th scope="row"><?php esc_html_e( 'City, State, ZIP', 'buildfuly' ); ?></th>
						<td>
							<input type="text" name="business_city" value="<?php echo esc_attr( $settings['business_city'] ); ?>" placeholder="City" style="width: 150px;">
							<input type="text" name="business_state" value="<?php echo esc_attr( $settings['business_state'] ); ?>" placeholder="State" style="width: 80px;">
							<input type="text" name="business_zip" value="<?php echo esc_attr( $settings['business_zip'] ); ?>" placeholder="ZIP" style="width: 80px;">
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="business_hours"><?php esc_html_e( 'Business Hours', 'buildfuly' ); ?></label>
						</th>
						<td>
							<textarea id="business_hours" name="business_hours" rows="4" class="regular-text" placeholder="Mon-Fri: 9am-5pm&#10;Sat: 10am-2pm&#10;Sun: Closed"><?php echo esc_textarea( $settings['business_hours'] ); ?></textarea>
							<p class="description"><?php esc_html_e( 'One line per day or range', 'buildfuly' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="business_price_range"><?php esc_html_e( 'Price Range', 'buildfuly' ); ?></label>
						</th>
						<td>
							<select id="business_price_range" name="business_price_range">
								<option value="$" <?php selected( $settings['business_price_range'], '$' ); ?>>$ - Budget</option>
								<option value="$$" <?php selected( $settings['business_price_range'], '$$' ); ?>>$$ - Moderate</option>
								<option value="$$$" <?php selected( $settings['business_price_range'], '$$$' ); ?>>$$$ - Premium</option>
								<option value="$$$$" <?php selected( $settings['business_price_range'], '$$$$' ); ?>>$$$$ - Luxury</option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php esc_html_e( 'Geo Coordinates', 'buildfuly' ); ?></th>
						<td>
							<input type="text" name="business_latitude" id="business_latitude" value="<?php echo esc_attr( $settings['business_latitude'] ); ?>" placeholder="Latitude (e.g., 40.7128)" style="width: 150px;">
							<input type="text" name="business_longitude" id="business_longitude" value="<?php echo esc_attr( $settings['business_longitude'] ); ?>" placeholder="Longitude (e.g., -74.0060)" style="width: 150px;">
							<button type="button" class="button" id="get-coordinates"><?php esc_html_e( 'Get from Address', 'buildfuly' ); ?></button>
							<p class="description"><?php esc_html_e( 'Used for local schema and map placement. Get coordinates from', 'buildfuly' ); ?> <a href="https://www.google.com/maps" target="_blank">Google Maps</a></p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="service_area"><?php esc_html_e( 'Service Area', 'buildfuly' ); ?></label>
						</th>
						<td>
							<textarea id="service_area" name="service_area" rows="3" class="regular-text" placeholder="City 1, City 2, County Name&#10;Or: 25 mile radius from location"><?php echo esc_textarea( $settings['service_area'] ); ?></textarea>
							<p class="description"><?php esc_html_e( 'Cities or areas you serve (for service-area businesses)', 'buildfuly' ); ?></p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="business_logo"><?php esc_html_e( 'Business Logo', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="url" id="business_logo" name="business_logo" value="<?php echo esc_url( $settings['business_logo'] ); ?>" class="regular-text">
							<button type="button" class="button" id="upload-business-logo"><?php esc_html_e( 'Upload Logo', 'buildfuly' ); ?></button>
							<p class="description"><?php esc_html_e( 'Square logo, minimum 112x112px', 'buildfuly' ); ?></p>
							<?php if ( ! empty( $settings['business_logo'] ) ) : ?>
								<div style="margin-top: 10px;">
									<img src="<?php echo esc_url( $settings['business_logo'] ); ?>" style="max-width: 100px; height: auto; border-radius: 8px;">
								</div>
							<?php endif; ?>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="schema_enabled"><?php esc_html_e( 'Enable Schema', 'buildfuly' ); ?></label>
						</th>
						<td>
							<label>
								<input type="checkbox" id="schema_enabled" name="schema_enabled" value="1" <?php checked( $settings['schema_enabled'], true ); ?>>
								<?php esc_html_e( 'Output LocalBusiness schema markup in page header', 'buildfuly' ); ?>
							</label>
							<p class="description"><?php esc_html_e( 'Helps Google understand your business for rich search results', 'buildfuly' ); ?></p>
						</td>
					</tr>
				</table>
				
				<!-- Schema Preview -->
				<div style="margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 8px; border: 1px solid #e5e7eb;">
					<h4 style="margin: 0 0 10px; display: flex; align-items: center; gap: 8px;">
						<span class="dashicons dashicons-code-standards"></span>
						<?php esc_html_e( 'Schema Preview', 'buildfuly' ); ?>
						<button type="button" class="button button-small" id="copy-schema" style="margin-left: auto;">
							<?php esc_html_e( 'Copy Schema', 'buildfuly' ); ?>
						</button>
					</h4>
					<pre id="schema-preview" style="background: #1e1e1e; color: #d4d4d4; padding: 15px; border-radius: 6px; overflow-x: auto; font-size: 12px; max-height: 300px; margin: 0;"><code></code></pre>
				</div>
			</div>
			
			<!-- Google Maps Integration -->
			<div class="buildfuly-card">
				<h3>🗺️ <?php esc_html_e( 'Google Maps Integration', 'buildfuly' ); ?></h3>
				<p class="description"><?php esc_html_e( 'Embed a Google Map on your website to help customers find you.', 'buildfuly' ); ?></p>
				
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="google_maps_enabled"><?php esc_html_e( 'Enable Google Maps', 'buildfuly' ); ?></label>
						</th>
						<td>
							<label>
								<input type="checkbox" id="google_maps_enabled" name="google_maps_enabled" value="1" <?php checked( $settings['google_maps_enabled'], true ); ?>>
								<?php esc_html_e( 'Enable Google Maps embedding', 'buildfuly' ); ?>
							</label>
						</td>
					</tr>
					<tr class="maps-option" style="<?php echo empty( $settings['google_maps_enabled'] ) ? 'display:none;' : ''; ?>">
						<th scope="row">
							<label for="google_maps_embed_url"><?php esc_html_e( 'Map Embed URL', 'buildfuly' ); ?></label>
						</th>
						<td>
							<textarea id="google_maps_embed_url" name="google_maps_embed_url" rows="3" class="large-text code" placeholder="Paste your Google Maps embed URL or iframe code here..."><?php echo esc_textarea( $settings['google_maps_embed_url'] ); ?></textarea>
							<p class="description">
								<?php esc_html_e( 'Get this from Google Maps:', 'buildfuly' ); ?>
								<ol style="margin: 8px 0 0 20px; font-size: 12px;">
									<li><?php esc_html_e( 'Search your business on Google Maps', 'buildfuly' ); ?></li>
									<li><?php esc_html_e( 'Click "Share" → "Embed a map"', 'buildfuly' ); ?></li>
									<li><?php esc_html_e( 'Copy the embed code or URL', 'buildfuly' ); ?></li>
								</ol>
							</p>
						</td>
					</tr>
					<tr class="maps-option" style="<?php echo empty( $settings['google_maps_enabled'] ) ? 'display:none;' : ''; ?>">
						<th scope="row">
							<label for="google_maps_footer"><?php esc_html_e( 'Show in Footer', 'buildfuly' ); ?></label>
						</th>
						<td>
							<label>
								<input type="checkbox" id="google_maps_footer" name="google_maps_footer" value="1" <?php checked( $settings['google_maps_footer'], true ); ?>>
								<?php esc_html_e( 'Automatically display map in website footer', 'buildfuly' ); ?>
							</label>
							<p class="description"><?php esc_html_e( 'Map will appear above footer on all pages', 'buildfuly' ); ?></p>
						</td>
					</tr>
					<tr class="maps-option" style="<?php echo empty( $settings['google_maps_enabled'] ) ? 'display:none;' : ''; ?>">
						<th scope="row">
							<label for="google_maps_height"><?php esc_html_e( 'Map Height', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="number" id="google_maps_height" name="google_maps_height" value="<?php echo esc_attr( $settings['google_maps_height'] ); ?>" style="width: 100px;" min="150" max="600"> px
							<p class="description"><?php esc_html_e( 'Height of the embedded map (150-600px)', 'buildfuly' ); ?></p>
						</td>
					</tr>
				</table>
				
				<!-- Map Preview -->
				<div class="maps-option" id="map-preview-container" style="margin-top: 20px; <?php echo empty( $settings['google_maps_enabled'] ) || empty( $settings['google_maps_embed_url'] ) ? 'display:none;' : ''; ?>">
					<h4 style="margin: 0 0 10px;">
						<span class="dashicons dashicons-location"></span>
						<?php esc_html_e( 'Map Preview', 'buildfuly' ); ?>
					</h4>
					<div id="map-preview" style="border-radius: 8px; overflow: hidden; border: 1px solid #e5e7eb;">
						<?php 
						if ( ! empty( $settings['google_maps_embed_url'] ) ) {
							$embed_url = $settings['google_maps_embed_url'];
							// If it's an iframe, extract the src
							if ( strpos( $embed_url, '<iframe' ) !== false ) {
								preg_match( '/src=["\']([^"\']+)["\']/', $embed_url, $matches );
								$embed_url = $matches[1] ?? '';
							}
							if ( ! empty( $embed_url ) ) {
								echo '<iframe src="' . esc_url( $embed_url ) . '" width="100%" height="' . esc_attr( $settings['google_maps_height'] ) . '" style="border:0;" allowfullscreen="" loading="lazy" referrerpolicy="no-referrer-when-downgrade"></iframe>';
							}
						}
						?>
					</div>
				</div>
				
				<!-- Shortcode Info -->
				<div class="maps-option" style="margin-top: 20px; padding: 15px; background: #f0f9ff; border-radius: 8px; border: 1px solid #bfdbfe; <?php echo empty( $settings['google_maps_enabled'] ) ? 'display:none;' : ''; ?>">
					<h4 style="margin: 0 0 8px; color: #1e40af;">
						<span class="dashicons dashicons-shortcode"></span>
						<?php esc_html_e( 'Shortcode', 'buildfuly' ); ?>
					</h4>
					<code style="background: white; padding: 8px 12px; border-radius: 4px; display: inline-block;">[buildfuly_map]</code>
					<p class="description" style="margin-top: 8px; color: #1e40af;">
						<?php esc_html_e( 'Use this shortcode to embed the map anywhere on your site. Options: height="400"', 'buildfuly' ); ?>
					</p>
				</div>
			</div>
			
			<!-- Social Profiles -->
			<div class="buildfuly-card">
				<h3>📱 <?php esc_html_e( 'Social Profiles', 'buildfuly' ); ?></h3>
				<p class="description"><?php esc_html_e( 'Link your social profiles for knowledge graph and rich snippets.', 'buildfuly' ); ?></p>
				
				<table class="form-table">
					<tr>
						<th scope="row"><label for="social_facebook">Facebook</label></th>
						<td>
							<input type="url" id="social_facebook" name="social_facebook" value="<?php echo esc_url( $settings['social_facebook'] ); ?>" class="regular-text" placeholder="https://facebook.com/yourbusiness">
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="social_instagram">Instagram</label></th>
						<td>
							<input type="url" id="social_instagram" name="social_instagram" value="<?php echo esc_url( $settings['social_instagram'] ); ?>" class="regular-text" placeholder="https://instagram.com/yourbusiness">
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="social_twitter">Twitter/X</label></th>
						<td>
							<input type="url" id="social_twitter" name="social_twitter" value="<?php echo esc_url( $settings['social_twitter'] ); ?>" class="regular-text" placeholder="https://twitter.com/yourbusiness">
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="social_linkedin">LinkedIn</label></th>
						<td>
							<input type="url" id="social_linkedin" name="social_linkedin" value="<?php echo esc_url( $settings['social_linkedin'] ); ?>" class="regular-text" placeholder="https://linkedin.com/company/yourbusiness">
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="social_youtube">YouTube</label></th>
						<td>
							<input type="url" id="social_youtube" name="social_youtube" value="<?php echo esc_url( $settings['social_youtube'] ); ?>" class="regular-text" placeholder="https://youtube.com/@yourbusiness">
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="social_yelp">Yelp</label></th>
						<td>
							<input type="url" id="social_yelp" name="social_yelp" value="<?php echo esc_url( $settings['social_yelp'] ); ?>" class="regular-text" placeholder="https://yelp.com/biz/yourbusiness">
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="social_google_business">Google Business Profile</label></th>
						<td>
							<input type="url" id="social_google_business" name="social_google_business" value="<?php echo esc_url( $settings['social_google_business'] ); ?>" class="regular-text" placeholder="https://g.page/yourbusiness">
							<p class="description"><?php esc_html_e( 'Your Google Business Profile URL (important for local SEO)', 'buildfuly' ); ?></p>
						</td>
					</tr>
				</table>
			</div>
			
			<!-- Default Social Sharing -->
			<div class="buildfuly-card">
				<h3>🖼️ <?php esc_html_e( 'Social Sharing Defaults', 'buildfuly' ); ?></h3>
				<p class="description"><?php esc_html_e( 'Default settings for when pages are shared on social media.', 'buildfuly' ); ?></p>
				
				<table class="form-table">
					<tr>
						<th scope="row">
							<label for="default_og_image"><?php esc_html_e( 'Default Share Image', 'buildfuly' ); ?></label>
						</th>
						<td>
							<input type="url" id="default_og_image" name="default_og_image" value="<?php echo esc_url( $settings['default_og_image'] ); ?>" class="regular-text">
							<button type="button" class="button" id="upload-og-image"><?php esc_html_e( 'Upload Image', 'buildfuly' ); ?></button>
							<p class="description"><?php esc_html_e( 'Recommended: 1200x630 pixels', 'buildfuly' ); ?></p>
							<?php if ( ! empty( $settings['default_og_image'] ) ) : ?>
								<div style="margin-top: 10px;">
									<img src="<?php echo esc_url( $settings['default_og_image'] ); ?>" style="max-width: 300px; height: auto; border-radius: 8px;">
								</div>
							<?php endif; ?>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="twitter_card_type"><?php esc_html_e( 'Twitter Card Type', 'buildfuly' ); ?></label>
						</th>
						<td>
							<select id="twitter_card_type" name="twitter_card_type">
								<option value="summary" <?php selected( $settings['twitter_card_type'], 'summary' ); ?>><?php esc_html_e( 'Summary (small image)', 'buildfuly' ); ?></option>
								<option value="summary_large_image" <?php selected( $settings['twitter_card_type'], 'summary_large_image' ); ?>><?php esc_html_e( 'Summary with Large Image', 'buildfuly' ); ?></option>
							</select>
						</td>
					</tr>
				</table>
			</div>
			
			<!-- 301 Redirects -->
			<div class="buildfuly-card">
				<h3>↪️ <?php esc_html_e( '301 Redirects', 'buildfuly' ); ?></h3>
				<p class="description"><?php esc_html_e( 'Redirect old URLs to new ones to preserve SEO value.', 'buildfuly' ); ?></p>
				
				<div id="redirects-list">
					<?php 
					$redirects = ! empty( $settings['redirects'] ) ? $settings['redirects'] : array();
					if ( empty( $redirects ) ) {
						$redirects = array( array( 'from' => '', 'to' => '' ) );
					}
					foreach ( $redirects as $index => $redirect ) : ?>
						<div class="redirect-row" style="display: flex; gap: 10px; margin-bottom: 10px; align-items: center;">
							<input type="text" name="redirects[<?php echo esc_attr( $index ); ?>][from]" value="<?php echo esc_attr( $redirect['from'] ?? '' ); ?>" placeholder="/old-page" style="flex: 1;">
							<span>→</span>
							<input type="text" name="redirects[<?php echo esc_attr( $index ); ?>][to]" value="<?php echo esc_attr( $redirect['to'] ?? '' ); ?>" placeholder="/new-page" style="flex: 1;">
							<button type="button" class="button remove-redirect" style="color: #dc3545;">✕</button>
						</div>
					<?php endforeach; ?>
				</div>
				<button type="button" class="button" id="add-redirect">+ <?php esc_html_e( 'Add Redirect', 'buildfuly' ); ?></button>
			</div>
			
		</div>
		
		<!-- Save Button -->
		<p class="submit">
			<?php submit_button( __( 'Save SEO Settings', 'buildfuly' ), 'primary large', 'submit', false ); ?>
		</p>
	</form>
</div>

<style>
/* SEO Dashboard Styles */
.buildfuly-seo-dashboard {
	display: grid;
	grid-template-columns: 300px 1fr;
	gap: 24px;
	margin-bottom: 24px;
}

.buildfuly-seo-score-card {
	display: flex;
	flex-direction: column;
	align-items: center;
	text-align: center;
	padding: 30px !important;
}

.seo-score-circle {
	width: 150px;
	height: 150px;
	margin-bottom: 20px;
}

.circular-chart {
	display: block;
	margin: 0 auto;
	max-width: 100%;
	max-height: 150px;
}

.circle-bg {
	fill: none;
	stroke: #eee;
	stroke-width: 3;
}

.circle {
	fill: none;
	stroke-width: 3;
	stroke-linecap: round;
	animation: progress 1s ease-out forwards;
	stroke: #10b981;
}

.seo-score-circle[data-score*="0"] .circle,
.seo-score-circle[data-score*="1"] .circle,
.seo-score-circle[data-score*="2"] .circle,
.seo-score-circle[data-score*="3"] .circle {
	stroke: #ef4444;
}

.seo-score-circle[data-score*="4"] .circle,
.seo-score-circle[data-score*="5"] .circle {
	stroke: #f59e0b;
}

.seo-score-circle[data-score*="6"] .circle,
.seo-score-circle[data-score*="7"] .circle {
	stroke: #3b82f6;
}

@keyframes progress {
	0% { stroke-dasharray: 0 100; }
}

.percentage {
	fill: #374151;
	font-size: 0.5em;
	text-anchor: middle;
	font-weight: bold;
}

.seo-score-info h2 {
	margin: 0 0 8px;
	color: #1f2937;
}

.seo-score-label {
	color: #6b7280;
	margin: 0;
}

.buildfuly-seo-checklist {
	padding: 24px !important;
}

.buildfuly-seo-checklist h3 {
	margin-top: 0;
}

.checklist-items {
	display: grid;
	grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
	gap: 20px;
}

.checklist-category h4 {
	margin: 0 0 12px;
	color: #374151;
	font-size: 14px;
	text-transform: uppercase;
	letter-spacing: 0.5px;
}

.checklist-item {
	display: flex;
	align-items: center;
	gap: 8px;
	padding: 8px 0;
	border-bottom: 1px solid #f3f4f6;
}

.checklist-item.complete {
	color: #059669;
}

.checklist-item.incomplete {
	color: #6b7280;
}

.checklist-icon {
	font-size: 16px;
}

/* Cards */
.buildfuly-card {
	background: #fff;
	border-radius: 12px;
	padding: 24px;
	margin-bottom: 24px;
	box-shadow: 0 1px 3px rgba(0,0,0,0.1);
}

.buildfuly-card h3 {
	margin-top: 0;
	margin-bottom: 8px;
	font-size: 18px;
	color: #1f2937;
}

.buildfuly-card > .description {
	color: #6b7280;
	margin-bottom: 20px;
}

.buildfuly-container {
	max-width: 900px;
}

/* Responsive */
@media (max-width: 782px) {
	.buildfuly-seo-dashboard {
		grid-template-columns: 1fr;
	}
}
</style>

<script>
jQuery(document).ready(function($) {
	// Add redirect row
	var redirectIndex = <?php echo count( $redirects ); ?>;
	$('#add-redirect').on('click', function() {
		var html = '<div class="redirect-row" style="display: flex; gap: 10px; margin-bottom: 10px; align-items: center;">' +
			'<input type="text" name="redirects[' + redirectIndex + '][from]" placeholder="/old-page" style="flex: 1;">' +
			'<span>→</span>' +
			'<input type="text" name="redirects[' + redirectIndex + '][to]" placeholder="/new-page" style="flex: 1;">' +
			'<button type="button" class="button remove-redirect" style="color: #dc3545;">✕</button>' +
			'</div>';
		$('#redirects-list').append(html);
		redirectIndex++;
	});
	
	// Remove redirect row
	$(document).on('click', '.remove-redirect', function() {
		$(this).closest('.redirect-row').remove();
	});
	
	// Media uploader for OG image
	$('#upload-og-image').on('click', function(e) {
		e.preventDefault();
		var mediaUploader = wp.media({
			title: 'Select Default Share Image',
			button: { text: 'Use this image' },
			multiple: false
		});
		
		mediaUploader.on('select', function() {
			var attachment = mediaUploader.state().get('selection').first().toJSON();
			$('#default_og_image').val(attachment.url);
		});
		
		mediaUploader.open();
	});
	
	// Media uploader for Business Logo
	$('#upload-business-logo').on('click', function(e) {
		e.preventDefault();
		var mediaUploader = wp.media({
			title: 'Select Business Logo',
			button: { text: 'Use this logo' },
			multiple: false
		});
		
		mediaUploader.on('select', function() {
			var attachment = mediaUploader.state().get('selection').first().toJSON();
			$('#business_logo').val(attachment.url);
		});
		
		mediaUploader.open();
	});
	
	// Toggle Google Maps options visibility
	$('#google_maps_enabled').on('change', function() {
		if ($(this).is(':checked')) {
			$('.maps-option').show();
		} else {
			$('.maps-option').hide();
		}
	});
	
	// Update map preview when URL changes
	$('#google_maps_embed_url').on('change blur', function() {
		var embedCode = $(this).val();
		if (!embedCode) {
			$('#map-preview-container').hide();
			return;
		}
		
		var embedUrl = embedCode;
		// Extract src from iframe if needed
		if (embedCode.indexOf('<iframe') !== -1) {
			var match = embedCode.match(/src=["']([^"']+)["']/);
			if (match) embedUrl = match[1];
		}
		
		if (embedUrl) {
			var height = $('#google_maps_height').val() || 300;
			$('#map-preview').html('<iframe src="' + embedUrl + '" width="100%" height="' + height + '" style="border:0;" allowfullscreen="" loading="lazy"></iframe>');
			$('#map-preview-container').show();
		}
	});
	
	// Update map height preview
	$('#google_maps_height').on('change', function() {
		var height = $(this).val();
		$('#map-preview iframe').attr('height', height);
	});
	
	// Generate and update schema preview
	function updateSchemaPreview() {
		var schema = {
			"@context": "https://schema.org",
			"@type": $('#schema_type').val() || "LocalBusiness",
			"name": $('#business_name').val() || "",
			"telephone": $('#business_phone').val() || "",
			"email": $('#business_email').val() || ""
		};
		
		// Add address
		var address = $('input[name="business_address"]').val();
		var city = $('input[name="business_city"]').val();
		var state = $('input[name="business_state"]').val();
		var zip = $('input[name="business_zip"]').val();
		
		if (address || city || state || zip) {
			schema.address = {
				"@type": "PostalAddress",
				"streetAddress": address || "",
				"addressLocality": city || "",
				"addressRegion": state || "",
				"postalCode": zip || "",
				"addressCountry": "US"
			};
		}
		
		// Add geo coordinates
		var lat = $('#business_latitude').val();
		var lng = $('#business_longitude').val();
		if (lat && lng) {
			schema.geo = {
				"@type": "GeoCoordinates",
				"latitude": parseFloat(lat),
				"longitude": parseFloat(lng)
			};
		}
		
		// Add logo
		var logo = $('#business_logo').val();
		if (logo) {
			schema.logo = logo;
		}
		
		// Add price range
		var priceRange = $('#business_price_range').val();
		if (priceRange) {
			schema.priceRange = priceRange;
		}
		
		// Add social profiles
		var sameAs = [];
		$('#social_facebook, #social_instagram, #social_twitter, #social_linkedin, #social_youtube, #social_yelp, #social_google_business').each(function() {
			var url = $(this).val();
			if (url) sameAs.push(url);
		});
		if (sameAs.length > 0) {
			schema.sameAs = sameAs;
		}
		
		// Add opening hours
		var hours = $('#business_hours').val();
		if (hours) {
			var hoursSpec = [];
			var lines = hours.split('\n');
			lines.forEach(function(line) {
				if (line.trim()) {
					hoursSpec.push({
						"@type": "OpeningHoursSpecification",
						"description": line.trim()
					});
				}
			});
			if (hoursSpec.length > 0) {
				schema.openingHoursSpecification = hoursSpec;
			}
		}
		
		// Add URL
		schema.url = '<?php echo esc_js( home_url() ); ?>';
		
		$('#schema-preview code').text(JSON.stringify(schema, null, 2));
	}
	
	// Update schema on any field change
	$('#schema_type, #business_name, #business_phone, #business_email, #business_latitude, #business_longitude, #business_logo, #business_price_range, #business_hours').on('change keyup', updateSchemaPreview);
	$('input[name="business_address"], input[name="business_city"], input[name="business_state"], input[name="business_zip"]').on('change keyup', updateSchemaPreview);
	$('#social_facebook, #social_instagram, #social_twitter, #social_linkedin, #social_youtube, #social_yelp, #social_google_business').on('change', updateSchemaPreview);
	
	// Initial schema preview
	updateSchemaPreview();
	
	// Copy schema to clipboard
	$('#copy-schema').on('click', function() {
		var schemaText = $('#schema-preview code').text();
		navigator.clipboard.writeText(schemaText).then(function() {
			var $btn = $('#copy-schema');
			$btn.text('Copied!');
			setTimeout(function() {
				$btn.text('Copy Schema');
			}, 2000);
		});
	});
	
	// Save via AJAX
	$('#buildfuly-seo-settings-form').on('submit', function(e) {
		e.preventDefault();
		
		var $form = $(this);
		var $submit = $form.find('input[type="submit"]');
		var originalText = $submit.val();
		
		$submit.val('Saving...').prop('disabled', true);
		
		$.ajax({
			url: ajaxurl,
			type: 'POST',
			data: {
				action: 'buildfuly_save_seo_settings',
				nonce: $('#buildfuly_seo_nonce').val(),
				formData: $form.serialize()
			},
			success: function(response) {
				if (response.success) {
					$submit.val('✓ Saved!');
					setTimeout(function() {
						$submit.val(originalText);
						location.reload(); // Reload to update score
					}, 1000);
				} else {
					alert('Error saving settings: ' + (response.data || 'Unknown error'));
					$submit.val(originalText);
				}
			},
			error: function() {
				alert('Error saving settings');
				$submit.val(originalText);
			},
			complete: function() {
				$submit.prop('disabled', false);
			}
		});
	});
});
</script>
