<?php
/**
 * Onboarding flow for first-time setup
 *
 * @link       https://build.fuly.io
 * @since      1.0.5
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/admin/partials
 */

// Prevent direct access
if ( ! defined( 'WPINC' ) ) {
	die;
}

require_once plugin_dir_path( dirname( dirname( __FILE__ ) ) ) . 'includes/class-buildfuly-constants.php';

// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound -- Template variables, not globals
$page_count = Buildfuly_Constants::get_onboarding_page_count();
$default_pages = Buildfuly_Constants::get_default_pages();
$auto_select_count = Buildfuly_Constants::get_auto_select_count();
// phpcs:enable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
?>

<div class="buildfuly-onboarding-wrap">
	<div class="buildfuly-onboarding-container">
		
		<!-- Header -->
		<div class="buildfuly-onboarding-header">
			<h1>🚀 Welcome to Buildfuly!</h1>
			<p>Let's get your AI-powered landing page builder set up in under 60 seconds.</p>
		</div>

		<!-- Step 1: Business Basics -->
		<div class="buildfuly-onboarding-step active" data-step="1">
			<div class="buildfuly-onboarding-card">
				<div class="step-indicator">Step 1 of 3</div>
				<h2>Tell us about your business</h2>
				<p class="subtitle">This helps AI create better, more relevant content for you.</p>
				
				<form id="buildfuly-onboarding-form">
					<div class="form-group">
						<label for="onboarding_business_name">
							Business Name <span class="required">*</span>
						</label>
						<input type="text" 
							   id="onboarding_business_name" 
							   name="business_name" 
							   class="large-input" 
							   placeholder="Acme Plumbing Services"
							   required>
					</div>

					<div class="form-group">
						<label for="onboarding_business_description">
							What does your business do? <span class="required">*</span>
						</label>
						<textarea id="onboarding_business_description" 
								  name="business_description" 
								  class="large-textarea" 
								  rows="4"
								  placeholder="We provide 24/7 emergency plumbing services for residential and commercial clients in the Greater NYC area..."
								  required></textarea>
						<p class="helper-text">💡 Be specific! This helps AI generate better content and SEO keywords.</p>
					</div>

					<div class="form-group">
						<label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
							<input type="checkbox" id="location_relevant" name="location_relevant" style="width: auto; margin: 0;">
							<span style="font-size: 14px; font-weight: 500;">My business location is relevant (use in keywords)</span>
						</label>
						<p class="helper-text" style="margin-top: 4px;">Check this if customers need to be near you (local services, restaurants, stores).</p>
						
						<div id="location-extracted-container" style="display: none; margin-top: 16px;">
							<label for="onboarding_location">City/Location</label>
							<input type="text" 
								   id="onboarding_location" 
								   name="location" 
								   class="large-input" 
								   placeholder="Miami">
							<p class="helper-text">This will be used in SEO keywords and content generation</p>
						</div>
					</div>

					<div class="form-actions">
						<button type="button" class="buildfuly-btn buildfuly-btn-primary buildfuly-btn-large" id="next-to-buttons">
							Next: Setup Your Buttons
							<span class="dashicons dashicons-arrow-right-alt2"></span>
						</button>
					</div>
				</form>
			</div>
		</div>

		<!-- Step 2: Call-to-Action Buttons -->
		<div class="buildfuly-onboarding-step" data-step="2">
			<div class="buildfuly-onboarding-card">
				<div class="step-indicator">Step 2 of 3</div>
				<h2>Where should we send your visitors?</h2>
				<p class="subtitle">Add buttons that will appear on all your pages. You can always change these later.</p>
				
				<div id="onboarding-buttons-list">
					<!-- Buttons will be added here -->
				</div>

				<button type="button" class="buildfuly-btn buildfuly-btn-secondary" id="add-onboarding-button">
					<span class="dashicons dashicons-plus-alt2"></span>
					Add Button
				</button>

				<div class="form-actions">
					<button type="button" class="buildfuly-btn buildfuly-btn-secondary" id="back-to-basics">
						<span class="dashicons dashicons-arrow-left-alt2"></span>
						Back
					</button>
					<button type="button" class="buildfuly-btn buildfuly-btn-primary buildfuly-btn-large" id="next-to-pages">
						Next: Choose Your Pages
						<span class="dashicons dashicons-arrow-right-alt2"></span>
					</button>
				</div>
			</div>
		</div>

	<!-- Step 3: Page Selection -->
	<div class="buildfuly-onboarding-step" data-step="3">
		<div class="buildfuly-onboarding-card">
			<div class="step-indicator">Step 3 of 3</div>
			<h2>Select the pages you want to create</h2>
			<p class="subtitle">AI is generating personalized page suggestions with SEO keywords...</p>
			
			<div id="suggested-pages-container">
				<div class="page-suggestions-loading">
					<div class="spinner"></div>
					<p><strong id="generation-status">Analyzing your business...</strong></p>
					<p style="font-size: 14px; color: #86868b; margin-top: 8px;" id="generation-substatus">Preparing AI generation...</p>
					
					<!-- Live page suggestions appear here as they're generated -->
					<div id="live-suggestions" style="margin-top: 32px; display: none;">
						<div id="live-suggestions-header" style="font-size: 14px; font-weight: 600; color: #1d1d1f; margin-bottom: 16px; text-align: left; display: none;">
							<span style="color: #0071e3;">✨</span> AI Recommended Pages:
						</div>
						<div id="live-pages-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 12px; text-align: left;">
							<!-- Pages appear here dynamically -->
						</div>
					</div>
				</div>
			</div>

			<div id="page-selection-container" style="display: none;">
				<div class="page-selection-grid" id="all-pages-grid">
					<!-- AI-suggested pages (including Home) will appear here -->
				</div>
				
				<div class="page-count-info">
					<span id="page-count">0</span> pages selected
				</div>
			</div>

			<div class="form-actions">
				<button type="button" class="buildfuly-btn buildfuly-btn-secondary" id="back-to-buttons" disabled>
					<span class="dashicons dashicons-arrow-left-alt2"></span>
					Back
				</button>
				<button type="button" class="buildfuly-btn buildfuly-btn-primary buildfuly-btn-large" id="complete-onboarding" disabled>
					Continue to Page Builder
					<span class="dashicons dashicons-arrow-right-alt2"></span>
				</button>
			</div>
		</div>
	</div>	</div>
</div>

<style>
/* Apple-inspired Design System */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap');

.buildfuly-onboarding-wrap {
	background: #f5f5f7;
	min-height: 100vh;
	padding: 60px 20px;
	margin-left: -20px;
	font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', 'Helvetica Neue', Arial, sans-serif;
	-webkit-font-smoothing: antialiased;
	-moz-osx-font-smoothing: grayscale;
}

.buildfuly-onboarding-container {
	max-width: 680px;
	margin: 0 auto;
}

.buildfuly-onboarding-header {
	text-align: center;
	color: #1d1d1f;
	margin-bottom: 48px;
}

.buildfuly-onboarding-header h1 {
	font-size: 56px;
	font-weight: 700;
	margin: 0 0 12px 0;
	letter-spacing: -0.015em;
	line-height: 1.07;
	background: linear-gradient(90deg, #000000 0%, #434343 100%);
	-webkit-background-clip: text;
	-webkit-text-fill-color: transparent;
	background-clip: text;
}

.buildfuly-onboarding-header p {
	font-size: 21px;
	font-weight: 400;
	line-height: 1.38;
	color: #6e6e73;
	margin: 0;
	letter-spacing: 0.011em;
}

.buildfuly-onboarding-step {
	display: none;
}

.buildfuly-onboarding-step.active {
	display: block;
	animation: fadeSlideIn 0.5s cubic-bezier(0.28, 0.11, 0.32, 1);
}

@keyframes fadeSlideIn {
	from {
		opacity: 0;
		transform: translateY(30px);
	}
	to {
		opacity: 1;
		transform: translateY(0);
	}
}

.buildfuly-onboarding-card {
	background: #ffffff;
	border-radius: 18px;
	padding: 48px 56px;
	box-shadow: 0 4px 6px rgba(0, 0, 0, 0.07),
				0 10px 15px rgba(0, 0, 0, 0.04);
	border: 1px solid rgba(0, 0, 0, 0.04);
}

.step-indicator {
	display: inline-block;
	background: #f5f5f7;
	color: #86868b;
	padding: 8px 18px;
	border-radius: 100px;
	font-size: 12px;
	font-weight: 600;
	margin-bottom: 24px;
	letter-spacing: 0.03em;
	text-transform: uppercase;
}

.buildfuly-onboarding-card h2 {
	font-size: 32px;
	font-weight: 700;
	margin: 0 0 8px 0;
	color: #1d1d1f;
	letter-spacing: -0.01em;
	line-height: 1.2;
}

.subtitle {
	font-size: 17px;
	color: #86868b;
	margin: 0 0 40px 0;
	line-height: 1.47;
	font-weight: 400;
}

.form-group {
	margin-bottom: 24px;
}

.form-group label {
	display: block;
	font-size: 14px;
	font-weight: 600;
	color: #1d1d1f;
	margin-bottom: 8px;
	letter-spacing: -0.01em;
}

.required {
	color: #ff3b30;
}

.large-input,
.large-textarea {
	width: 100%;
	padding: 12px 16px;
	font-size: 17px;
	border: 1px solid #d2d2d7;
	border-radius: 12px;
	transition: all 0.2s cubic-bezier(0.28, 0.11, 0.32, 1);
	font-family: inherit;
	background: #ffffff;
	color: #1d1d1f;
	font-weight: 400;
}

.large-input:focus,
.large-textarea:focus {
	outline: none;
	border-color: #0071e3;
	box-shadow: 0 0 0 4px rgba(0, 113, 227, 0.1);
	background: #ffffff;
}

.large-input::placeholder,
.large-textarea::placeholder {
	color: #86868b;
}

.helper-text {
	font-size: 12px;
	color: #86868b;
	margin: 6px 0 0 0;
	line-height: 1.4;
	font-weight: 400;
}

.form-actions {
	margin-top: 40px;
	display: flex;
	gap: 12px;
	justify-content: flex-end;
}

.buildfuly-btn {
	padding: 12px 24px;
	font-size: 17px;
	font-weight: 500;
	border-radius: 980px;
	cursor: pointer;
	transition: all 0.2s cubic-bezier(0.28, 0.11, 0.32, 1);
	border: none;
	display: inline-flex;
	align-items: center;
	gap: 6px;
	letter-spacing: -0.01em;
	line-height: 1.2;
}

.buildfuly-btn-large {
	padding: 14px 28px;
	font-size: 17px;
	font-weight: 500;
}

.buildfuly-btn-primary {
	background: #0071e3;
	color: #ffffff;
	box-shadow: 0 1px 3px rgba(0, 0, 0, 0.12);
}

.buildfuly-btn-primary:hover {
	background: #0077ed;
	transform: translateY(-1px);
	box-shadow: 0 4px 12px rgba(0, 113, 227, 0.25);
}

.buildfuly-btn-primary:active {
	transform: translateY(0);
	box-shadow: 0 1px 3px rgba(0, 0, 0, 0.12);
}

.buildfuly-btn-secondary {
	background: #f5f5f7;
	color: #1d1d1f;
	border: 1px solid #d2d2d7;
}

.buildfuly-btn-secondary:hover {
	background: #e8e8ed;
	border-color: #b8b8bd;
}

.buildfuly-btn-secondary:active {
	background: #d2d2d7;
}

.buildfuly-btn:disabled,
.buildfuly-btn.disabled {
	opacity: 0.4;
	cursor: not-allowed;
	pointer-events: none;
}

/* Page Suggestions Loading */
.page-suggestions-loading {
	text-align: center;
	padding: 80px 20px;
	background: linear-gradient(135deg, #f5f7fa 0%, #f9fafb 100%);
	border-radius: 18px;
	border: 1px solid #d2d2d7;
}

.page-suggestions-loading p {
	font-size: 17px;
	color: #1d1d1f;
	margin: 0;
	font-weight: 500;
	line-height: 1.5;
}

.page-suggestions-loading p strong {
	color: #0071e3;
	font-weight: 600;
}

.spinner {
	width: 48px;
	height: 48px;
	border: 3px solid #e5e7eb;
	border-top-color: #0071e3;
	border-radius: 50%;
	animation: spin 0.8s cubic-bezier(0.4, 0.0, 0.2, 1) infinite;
	margin: 0 auto 24px;
	box-shadow: 0 2px 8px rgba(0, 113, 227, 0.1);
}

@keyframes spin {
	to { transform: rotate(360deg); }
}

/* Button Builder */
.onboarding-button-item {
	background: #f5f5f7;
	border: 1px solid #d2d2d7;
	border-radius: 16px;
	padding: 24px;
	margin-bottom: 16px;
	transition: all 0.2s cubic-bezier(0.28, 0.11, 0.32, 1);
}

.onboarding-button-item:hover {
	border-color: #0071e3;
	transform: translateY(-2px);
	box-shadow: 0 4px 12px rgba(0, 113, 227, 0.12);
}

.button-item-header {
	display: flex;
	justify-content: space-between;
	align-items: center;
	margin-bottom: 20px;
}

.button-item-header h4 {
	margin: 0;
	color: #1d1d1f;
	font-size: 17px;
	font-weight: 600;
	letter-spacing: -0.01em;
}

.remove-button-item {
	background: #ff3b30;
	color: white;
	border: none;
	padding: 6px 14px;
	border-radius: 980px;
	font-size: 12px;
	font-weight: 500;
	cursor: pointer;
	transition: all 0.2s cubic-bezier(0.28, 0.11, 0.32, 1);
}

.remove-button-item:hover {
	background: #ff453a;
	transform: scale(1.05);
}

.button-type-selector {
	display: flex;
	gap: 12px;
	margin-bottom: 20px;
}

.button-type-option {
	flex: 1;
	padding: 20px;
	border: 1px solid #d2d2d7;
	border-radius: 12px;
	cursor: pointer;
	transition: all 0.2s cubic-bezier(0.28, 0.11, 0.32, 1);
	text-align: center;
	background: #ffffff;
}

.button-type-option input[type="radio"] {
	display: none;
}

.button-type-option .dashicons {
	font-size: 28px;
	width: 28px;
	height: 28px;
	margin-bottom: 8px;
	color: #86868b;
}

.button-type-option label {
	cursor: pointer;
	display: block;
	font-weight: 500;
	color: #86868b;
	font-size: 14px;
	letter-spacing: -0.01em;
}

.button-type-option:hover {
	border-color: #b8b8bd;
	background: #fafafa;
}

.button-type-option.selected {
	border-color: #0071e3;
	background: #f0f7ff;
}

.button-type-option.selected .dashicons {
	color: #0071e3;
}

.button-type-option.selected label {
	color: #0071e3;
}

.button-fields input {
	margin-bottom: 12px;
}

.button-fields input:last-child {
	margin-bottom: 0;
}

/* Page Selection Grid */
.page-selection-grid {
	display: grid;
	grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
	gap: 16px;
	margin-bottom: 32px;
}

.page-selection-item {
	background: #ffffff;
	border: 1px solid #d2d2d7;
	border-radius: 16px;
	padding: 24px;
	transition: all 0.2s cubic-bezier(0.28, 0.11, 0.32, 1);
	position: relative;
	cursor: pointer;
}

.page-selection-item:hover {
	border-color: #0071e3;
	transform: translateY(-4px);
	box-shadow: 0 8px 20px rgba(0, 113, 227, 0.15);
}

.page-selection-item.selected {
	border-color: #0071e3;
	background: #f0f7ff;
}

.page-selection-item input[type="checkbox"] {
	position: absolute;
	top: 16px;
	right: 16px;
	width: 20px;
	height: 20px;
	cursor: pointer;
	accent-color: #0071e3;
}

.page-selection-item h4 {
	margin: 0 0 8px 0;
	color: #1d1d1f;
	font-size: 17px;
	font-weight: 600;
	padding-right: 30px;
	letter-spacing: -0.01em;
}

.page-selection-item .page-keyword {
	margin: 0;
	font-size: 12px;
	color: #86868b;
	line-height: 1.5;
	font-weight: 400;
}

.page-selection-item.default-page {
	border-color: #34c759;
	background: #f0fdf4;
	cursor: default;
}

.page-selection-item.default-page input[type="checkbox"] {
	pointer-events: none;
	accent-color: #34c759;
}

.page-count-info {
	text-align: center;
	font-size: 14px;
	color: #86868b;
	padding: 16px;
	background: #f5f5f7;
	border-radius: 12px;
	font-weight: 500;
	letter-spacing: -0.01em;
}

#page-count {
	font-weight: 700;
	color: #0071e3;
	font-size: 20px;
}
</style>

<script>
jQuery(document).ready(function($) {
	let selectedPages = <?php echo json_encode( $default_pages ); ?>;
	let businessData = {};
	let buttons = [];
	let buttonCounter = 0;
	
	// Debug helper function
	function debugLog(message, ...args) {
		if (typeof buildfulyAdmin !== 'undefined' && buildfulyAdmin.debug) {
			console.log('[Buildfuly Onboarding]', message, ...args);
		}
	}

	// ============================================
	// STEP 1: Business Basics
	// ============================================

	// Auto-extract location when address changes or checkbox is toggled
	$('#onboarding_address, #location_relevant').on('change input', function() {
		const isLocationRelevant = $('#location_relevant').is(':checked');
		const address = $('#onboarding_address').val();
		
		if (isLocationRelevant && address) {
			const extractedCity = extractCityFromAddress(address);
			$('#onboarding_location').val(extractedCity);
			$('#location-extracted-container').slideDown(300);
		} else {
			$('#location-extracted-container').slideUp(300);
			$('#onboarding_location').val('');
		}
	});

	// Show/hide location input when checkbox is toggled
	$('#location_relevant').on('change', function() {
		if ($(this).is(':checked')) {
			$('#location-extracted-container').slideDown();
		} else {
			$('#location-extracted-container').slideUp();
			$('#onboarding_location').val('');
		}
	});

	// Next to buttons step
	$('#next-to-buttons').on('click', function() {
		const name = $('#onboarding_business_name').val().trim();
		const description = $('#onboarding_business_description').val().trim();

		if (!name || !description) {
			alert('Please fill in your business name and description.');
			return;
		}

		// Save business data
		businessData = {
			name: name,
			description: description,
			location: $('#onboarding_location').val() ? $('#onboarding_location').val().trim() : '',
			location_relevant: $('#location_relevant').is(':checked')
		};

		// Move to step 2
		$('.buildfuly-onboarding-step').removeClass('active');
		$('[data-step="2"]').addClass('active');
	});

	// ============================================
	// STEP 2: Button Builder
	// ============================================

	// Add button
	$('#add-onboarding-button').on('click', function() {
		const index = buttonCounter++;
		const buttonHtml = `
			<div class="onboarding-button-item" data-index="${index}">
				<div class="button-item-header">
					<h4>Button ${index + 1}</h4>
					<button type="button" class="remove-button-item">
						<span class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				
				<div class="button-type-selector">
					<div class="button-type-option" data-type="call">
						<input type="radio" name="button_type_${index}" value="call" id="button_type_call_${index}">
						<label for="button_type_call_${index}">
							<span class="dashicons dashicons-phone"></span>
							<span>Call</span>
						</label>
					</div>
					<div class="button-type-option" data-type="link">
						<input type="radio" name="button_type_${index}" value="link" id="button_type_link_${index}">
						<label for="button_type_link_${index}">
							<span class="dashicons dashicons-admin-links"></span>
							<span>Link</span>
						</label>
					</div>
				</div>
				
				<div class="button-fields">
					<input type="text" class="button-name-input" placeholder="Button Name (e.g., Call Us Now)" />
					<input type="tel" class="button-phone-input" placeholder="Phone Number (e.g., 555-1234)" style="display:none;" />
					<input type="url" class="button-url-input" placeholder="Link URL (e.g., https://example.com)" style="display:none;" />
				</div>
			</div>
		`;
		$('#onboarding-buttons-list').append(buttonHtml);
	});

	// Type selection
	$(document).on('click', '.button-type-option', function() {
		const $option = $(this);
		const $item = $option.closest('.onboarding-button-item');
		const type = $option.data('type');
		
		// Update radio button
		$option.find('input').prop('checked', true);
		
		// Update selected styling
		$option.addClass('selected').siblings().removeClass('selected');
		
		// Show/hide fields based on type
		if (type === 'call') {
			$item.find('.button-phone-input').show();
			$item.find('.button-url-input').hide();
		} else {
			$item.find('.button-phone-input').hide();
			$item.find('.button-url-input').show();
		}
	});

	// Remove button
	$(document).on('click', '.remove-button-item', function() {
		$(this).closest('.onboarding-button-item').remove();
	});

	// Back to basics
	$('#back-to-basics').on('click', function() {
		$('.buildfuly-onboarding-step').removeClass('active');
		$('[data-step="1"]').addClass('active');
	});

	// Next to pages
	$('#next-to-pages').on('click', function() {
		// Collect button data
		buttons = [];
		$('.onboarding-button-item').each(function() {
			const $item = $(this);
			const type = $item.find('.button-type-selector input:checked').val();
			const name = $item.find('.button-name-input').val().trim();
			let url = '';
			
			if (type === 'call') {
				const phone = $item.find('.button-phone-input').val().trim();
				// Add tel: prefix and strip non-digits
				url = phone.startsWith('tel:') ? phone : 'tel:' + phone.replace(/\D/g, '');
			} else {
				url = $item.find('.button-url-input').val().trim();
				// Validate URL (allow http://, https://, #, tel:, mailto:)
				if (url && !url.match(/^(https?:\/\/|tel:|mailto:|#)/)) {
					url = 'https://' + url; // Default to https if no protocol
				}
			}
			
			if (name && url) {
				buttons.push({ 
					text: name,
					url: url,
					style: 'primary'
				});
			}
		});

		// Move to step 3
		$('.buildfuly-onboarding-step').removeClass('active');
		$('[data-step="3"]').addClass('active');

		// Generate page suggestions with full business data
		generatePageSuggestions();
	});
	
	// Auto-suggest phone button when entering Step 2
	function suggestPhoneButton() {
		const phone = businessData.phone;
		if (!phone) return;
		
		// Check if a phone button already exists
		const existingPhoneButton = buttons.find(b => b.type === 'call');
		if (existingPhoneButton) return;
		
		// Add suggested phone button
		const index = buttonCounter++;
		const buttonHtml = `
			<div class="onboarding-button-item suggested-button" data-index="${index}">
				<div class="button-item-header">
					<h4>Button ${index + 1} <span style="background: #10b981; color: white; font-size: 11px; padding: 2px 8px; border-radius: 4px; margin-left: 8px;">Suggested</span></h4>
					<button type="button" class="remove-button-item">
						<span class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				
				<div class="button-type-selector">
					<div class="button-type-option selected" data-type="call">
						<input type="radio" name="button_type_${index}" value="call" id="button_type_call_${index}" checked>
						<label for="button_type_call_${index}">
							<span class="dashicons dashicons-phone"></span>
							<span>Call</span>
						</label>
					</div>
					<div class="button-type-option" data-type="link">
						<input type="radio" name="button_type_${index}" value="link" id="button_type_link_${index}">
						<label for="button_type_link_${index}">
							<span class="dashicons dashicons-admin-links"></span>
							<span>Link</span>
						</label>
					</div>
				</div>
				
				<div class="button-fields">
					<input type="text" class="button-name-input" placeholder="Button Name (e.g., Call Us Now)" value="Call Now" />
					<input type="tel" class="button-phone-input" placeholder="Phone Number (e.g., 555-1234)" value="${phone}" style="display:block;" />
					<input type="url" class="button-url-input" placeholder="Link URL (e.g., https://example.com)" style="display:none;" />
				</div>
			</div>
		`;
		$('#onboarding-buttons-list').prepend(buttonHtml);
		
		// Animate it
		$('.suggested-button').hide().slideDown(300);
	}

	// ============================================
	// STEP 3: Page Selection
	// ============================================
	
	// Extract city from address string
	function extractCityFromAddress(address) {
		if (!address || address.trim().length === 0) return '';
		
		// Common address formats:
		// "123 Main St, New York, NY 10001"
		// "456 Oak Ave, Los Angeles, California"
		// "789 Pine Rd, Chicago IL"
		
		const parts = address.split(',').map(p => p.trim());
		
		if (parts.length >= 2) {
			// City is usually the second-to-last part (before state/zip)
			const cityPart = parts[parts.length - 2];
			
			// Remove any numbers/zip codes from the city part
			const city = cityPart.replace(/\d+/g, '').trim();
			
			return city;
		}
		
		// Fallback: try to find a capitalized word that's not a number
		const words = address.split(/[\s,]+/);
		for (const word of words) {
			if (word.length > 2 && /^[A-Z][a-z]+$/.test(word) && !/^\d+$/.test(word)) {
				return word;
			}
		}
		
		return '';
	}

	// Generate page suggestions using AI
	function generatePageSuggestions() {
		// Disable navigation buttons during generation
		$('#back-to-buttons, #complete-onboarding').prop('disabled', true).addClass('disabled');
		
		$('#suggested-pages-container').show();
		$('#page-selection-container').hide();
		
		// Clear previous live pages
		$('#live-pages-grid').empty();
		$('#live-suggestions-header').hide();
		
		// Use the location from the extracted field if checkbox is checked
		let locationToSend = '';
		if ($('#location_relevant').is(':checked')) {
			locationToSend = $('#onboarding_location').val() || '';
		}
		
		// Use shared animation helper with detailed context
		if (window.BuildfulyAnimations) {
			window.BuildfulyAnimations.animateGeneration({
			statusId: 'generation-status',
			substatusId: 'generation-substatus',
			liveContainerId: 'live-suggestions',
			context: {
				description: businessData.description,
				location: locationToSend,
				count: <?php echo absint( $page_count ); ?>,
				existingPages: [] // No existing pages in onboarding
			}
		});
	}
	
	const requestData = {
		action: 'buildfuly_suggest_pages',
		nonce: buildfulyAdmin.nonce,
		description: businessData.description,
		location: locationToSend,
		count: <?php echo absint( $page_count ); ?>
	};		debugLog('Requesting AI page suggestions with data:', requestData);

		// Call AI to suggest pages with full business context
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: requestData,
			success: function(response) {
				debugLog('AI Response:', response);
				
				let suggestedPages = [];
				if (response.success && response.data.pages) {
					suggestedPages = response.data.pages;
					debugLog('Pages received from AI:', suggestedPages.length, suggestedPages);
					
					// Show pages one by one with animation
					suggestedPages.forEach((pageObj, index) => {
						setTimeout(() => {
							const pageName = typeof pageObj === 'string' ? pageObj : pageObj.name;
							const keyword = typeof pageObj === 'object' ? pageObj.keyword : '';
							
							// Use shared animation helper
							if (window.BuildfulyAnimations) {
								window.BuildfulyAnimations.showLivePage({
									gridId: 'live-pages-grid',
									headerId: 'live-suggestions-header'
								}, pageName, keyword);
							}
						}, index * 200);
					});
				} else {
					debugLog('AI did not return pages, using fallback. Response:', response);
					// Fallback suggestions with keywords
					suggestedPages = [
						{ name: 'Home', keyword: 'main landing page showcasing our business' },
						{ name: 'About', keyword: 'our story, mission, and values' },
						{ name: 'Contact', keyword: 'get in touch with us' },
						{ name: 'Portfolio', keyword: 'showcase our best work' },
						{ name: 'Pricing', keyword: 'pricing plans and packages' },
						{ name: 'Testimonials', keyword: 'customer reviews and success stories' }
					];
				}
				
				debugLog('Rendering pages:', suggestedPages);
				renderPageCheckboxes(suggestedPages);
				
				// Delay showing the grid to let live animations finish
				const animationDelay = suggestedPages.length * 200 + 800;
				setTimeout(() => {
					$('#suggested-pages-container').fadeOut(300, function() {
						$('#page-selection-container').fadeIn(300);
						// Re-enable navigation buttons
						$('#back-to-buttons').prop('disabled', false).removeClass('disabled');
						// complete-onboarding stays disabled until pages are selected
					});
				}, animationDelay);
			},
			error: function(xhr, status, error) {
				debugLog('AJAX Error:', status, error);
				debugLog('Response:', xhr.responseText);
				
				// Fallback with keywords
				const suggestedPages = [
					{ name: 'Home', keyword: 'main landing page showcasing our business' },
					{ name: 'About', keyword: 'our story, mission, and values' },
					{ name: 'Contact', keyword: 'get in touch with us' },
					{ name: 'Portfolio', keyword: 'showcase our best work' },
					{ name: 'Pricing', keyword: 'pricing plans and packages' },
					{ name: 'Testimonials', keyword: 'customer reviews and success stories' }
				];
				renderPageCheckboxes(suggestedPages);
				$('#suggested-pages-container').hide();
				$('#page-selection-container').fadeIn();
				
				// Re-enable navigation buttons
				$('#back-to-buttons').prop('disabled', false).removeClass('disabled');
				// complete-onboarding stays disabled until pages are selected
			}
		});
	}

	// Render pages as checkboxes with keywords - Home auto-selected from AI
	function renderPageCheckboxes(suggestedPages) {
		const $grid = $('#all-pages-grid');
		$grid.empty();
		selectedPages = [];
		
		// All pages are from AI now (including Home)
		suggestedPages.forEach((pageObj) => {
			// Check if it's an object or just a string
			const pageName = typeof pageObj === 'string' ? pageObj : pageObj.name;
			const pageKeyword = typeof pageObj === 'string' ? `landing page for ${pageObj}` : pageObj.keyword;
			
			const pageData = { name: pageName, keyword: pageKeyword };
			
			// Home is auto-selected
			const isHome = pageName === 'Home';
			
			const $item = $(`
				<div class="page-selection-item ${isHome ? 'default-page selected' : ''}" data-page='${JSON.stringify(pageData)}'>
					<input type="checkbox" ${isHome ? 'checked disabled' : ''} />
					<h4>${pageName}</h4>
					<p class="page-keyword">${pageKeyword}</p>
				</div>
			`);
			
			$grid.append($item);
			
			// Auto-select Home
			if (isHome) {
				selectedPages.push(pageData);
			}
		});
		
		updatePageCount();
		
		// Enable complete button since Home is auto-selected
		if (selectedPages.length > 0) {
			$('#complete-onboarding').prop('disabled', false).removeClass('disabled');
		}
	}

	// Checkbox change
	$(document).on('change', '.page-selection-item input[type="checkbox"]', function() {
		const $item = $(this).closest('.page-selection-item');
		const pageData = JSON.parse($item.attr('data-page'));
		
		if ($(this).is(':checked')) {
			$item.addClass('selected');
			if (!selectedPages.some(p => p.name === pageData.name)) {
				selectedPages.push(pageData);
			}
		} else {
			$item.removeClass('selected');
			selectedPages = selectedPages.filter(p => p.name !== pageData.name);
		}
		
		updatePageCount();
		
		// Enable complete button if pages selected
		if (selectedPages.length > 0) {
			$('#complete-onboarding').prop('disabled', false).removeClass('disabled');
		} else {
			$('#complete-onboarding').prop('disabled', true).addClass('disabled');
		}
	});

	// Click on card = toggle checkbox (but not for Home)
	$(document).on('click', '.page-selection-item', function(e) {
		// Skip if clicking checkbox or if it's the Home page
		if ($(e.target).is('input') || $(this).hasClass('default-page')) return;
		
		const $checkbox = $(this).find('input[type="checkbox"]');
		if (!$checkbox.prop('disabled')) {
			$checkbox.prop('checked', !$checkbox.prop('checked')).trigger('change');
		}
	});

	// Update page count
	function updatePageCount() {
		$('#page-count').text(selectedPages.length);
	}

	// Back to buttons
	$('#back-to-buttons').on('click', function() {
		$('.buildfuly-onboarding-step').removeClass('active');
		$('[data-step="2"]').addClass('active');
	});

	// Complete onboarding
	$('#complete-onboarding').on('click', function() {
		const $btn = $(this);
		$btn.prop('disabled', true).html('<span class="spinner" style="width: 20px; height: 20px; border-width: 2px;"></span> Creating your pages...');

		// Save everything
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_complete_onboarding',
				nonce: buildfulyAdmin.nonce,
				business_data: businessData,
				buttons: buttons,
				pages: selectedPages // Now includes page objects with name and keyword
			},
			success: function(response) {
				if (response.success) {
					// Create pages in localStorage with keywords
					const pages = selectedPages.map(pageData => {
						return {
							id: 'page-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9),
							title: pageData.name,
							keyword: pageData.keyword, // Store keyword for later use
							sections: [], // Empty sections - user will generate content
							colors: {
								primary: '#00bfa5',
								secondary: '#00897b',
								accent: '#7c4dff',
								text: '#1e293b',
								bg: '#ffffff'
							},
							settings: {
								metaTitle: pageData.name,
								metaDescription: '',
								slug: pageData.name.toLowerCase().replace(/\s+/g, '-')
							}
						};
					});
					
					// Save to localStorage
					localStorage.setItem('buildfuly_pages', JSON.stringify(pages));
					
					// Redirect to page builder
					window.location.href = '<?php echo esc_url( admin_url( 'admin.php?page=buildfuly' ) ); ?>';
				} else {
					alert('Error saving settings. Please try again.');
					$btn.prop('disabled', false).html('Create My Pages & Start Building <span class="dashicons dashicons-yes-alt"></span>');
				}
			},
			error: function() {
				alert('Error saving settings. Please try again.');
				$btn.prop('disabled', false).html('Create My Pages & Start Building <span class="dashicons dashicons-yes-alt"></span>');
			}
		});
	});
});
</script>
