<?php
/**
 * Logo Generator Admin Page
 * AI-powered logo generation with 3 options, background removal, and favicon creation
 *
 * @link       https://build.fuly.io
 * @since      1.1.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/admin/partials
 */

// Prevent direct access
if ( ! defined( 'WPINC' ) ) {
	die;
}

// Get business settings
$business_name = get_option( 'buildfuly_business_name', '' );
$business_description = get_option( 'buildfuly_business_description', '' );

// Get brand colors
$primary_color = get_option( 'buildfuly_primary_color', '#00bfa5' );
$secondary_color = get_option( 'buildfuly_secondary_color', '#00897b' );

// Current logo & favicon
$logo_id = get_option( 'buildfuly_business_logo_id', '' );
$logo_url = $logo_id ? wp_get_attachment_url( $logo_id ) : '';
$favicon_id = get_option( 'buildfuly_favicon_id', '' );
$favicon_url = $favicon_id ? wp_get_attachment_url( $favicon_id ) : '';

// If no favicon, use logo as fallback for display
if ( empty( $favicon_url ) && ! empty( $logo_url ) ) {
	$favicon_url = $logo_url;
	$favicon_is_logo = true;
} else {
	$favicon_is_logo = false;
}

// Example logos
$example_logos = array(
	'style1' => array(
		'url' => 'https://durable.co/_next/image?url=https%3A%2F%2Frjdavx8ozyznxeyh.public.blob.vercel-storage.com%2Fproduction%2Fpages%2Flogo-wall%2Flogos%2F563.png&w=828&q=75',
		'name' => 'Icon + Text',
		'description' => 'Simple icon above business name'
	),
	'style2' => array(
		'url' => 'https://durable.co/_next/image?url=https%3A%2F%2Frjdavx8ozyznxeyh.public.blob.vercel-storage.com%2Fproduction%2Fpages%2Flogo-wall%2Flogos%2F662.png&w=828&q=75',
		'name' => 'Minimal Mark',
		'description' => 'Clean geometric shape with text'
	),
	'style3' => array(
		'url' => 'https://durable.co/_next/image?url=https%3A%2F%2Frjdavx8ozyznxeyh.public.blob.vercel-storage.com%2Fproduction%2Fpages%2Flogo-wall%2Flogos%2F2410.png&w=828&q=75',
		'name' => 'Bold Wordmark',
		'description' => 'Typography-focused design'
	),
	'style4' => array(
		'url' => 'https://durable.co/_next/image?url=https%3A%2F%2Frjdavx8ozyznxeyh.public.blob.vercel-storage.com%2Fproduction%2Fpages%2Flogo-wall%2Flogos%2F452.png&w=828&q=75',
		'name' => 'Modern Badge',
		'description' => 'Contained design with icon'
	),
);
?>

<div class="wrap buildfuly-admin-wrap">
	<input type="hidden" id="buildfuly-nonce" value="<?php echo esc_attr( wp_create_nonce('buildfuly-admin') ); ?>">
	
	<div class="buildfuly-page-header" style="margin-bottom: 24px;">
		<h1 style="display: flex; align-items: center; gap: 12px; margin: 0;">
			<span style="font-size: 32px;">✨</span>
			<?php esc_html_e( 'Logo Generator', 'buildfuly' ); ?>
		</h1>
		<p style="color: #6b7280; margin: 8px 0 0 0;">
			<?php esc_html_e( 'Generate 3 logo options, pick your favorite, and we\'ll auto-create a transparent version + favicon.', 'buildfuly' ); ?>
		</p>
	</div>

	<!-- Current Logo & Favicon -->
	<div class="buildfuly-card" style="margin-bottom: 24px; display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 24px;">
		<div>
			<h3 style="margin-top: 0;"><?php esc_html_e( '📌 Current Logo', 'buildfuly' ); ?></h3>
			<div id="current-logo-preview" style="background: repeating-conic-gradient(#f3f4f6 0% 25%, white 0% 50%) 50% / 16px 16px; border-radius: 12px; padding: 24px; text-align: center; min-height: 100px; display: flex; align-items: center; justify-content: center;">
				<?php if ( $logo_url ) : ?>
					<img src="<?php echo esc_url( $logo_url ); ?>" alt="Current Logo" style="max-height: 80px; max-width: 180px;" data-logo-url="<?php echo esc_url( $logo_url ); ?>">
				<?php else : ?>
					<span style="color: #9ca3af;"><?php esc_html_e( 'No logo yet', 'buildfuly' ); ?></span>
				<?php endif; ?>
			</div>
			<?php if ( $logo_url ) : ?>
			<div style="margin-top: 12px; text-align: center;">
				<button type="button" id="create-favicon-from-logo-btn" class="button" style="font-size: 12px;">
					<span class="dashicons dashicons-format-image" style="vertical-align: middle; font-size: 14px;"></span>
					<?php esc_html_e( 'Create Favicon from Logo', 'buildfuly' ); ?>
				</button>
				<button type="button" id="crop-favicon-btn" class="button" style="font-size: 12px; margin-left: 8px;">
					<span class="dashicons dashicons-image-crop" style="vertical-align: middle; font-size: 14px;"></span>
					<?php esc_html_e( 'Crop Favicon', 'buildfuly' ); ?>
				</button>
			</div>
			<?php endif; ?>
		</div>
		<div>
			<h3 style="margin-top: 0;"><?php esc_html_e( '🔲 Current Favicon', 'buildfuly' ); ?></h3>
			<div id="current-favicon-preview" style="background: #f9fafb; border-radius: 12px; padding: 24px; text-align: center; min-height: 100px; display: flex; align-items: center; justify-content: center; gap: 16px; flex-direction: column;">
				<?php if ( $favicon_url ) : ?>
					<div style="display: flex; align-items: center; gap: 16px;">
						<div style="text-align: center;">
							<img src="<?php echo esc_url( $favicon_url ); ?>" alt="Favicon" style="width: 32px; height: 32px; border-radius: 4px;">
							<div style="font-size: 10px; color: #6b7280; margin-top: 4px;">32px</div>
						</div>
						<div style="text-align: center;">
							<img src="<?php echo esc_url( $favicon_url ); ?>" alt="Favicon" style="width: 16px; height: 16px;">
							<div style="font-size: 10px; color: #6b7280; margin-top: 4px;">16px</div>
						</div>
					</div>
					<?php if ( $favicon_is_logo ) : ?>
						<div style="font-size: 12px; color: #f59e0b; background: #fef3c7; padding: 6px 12px; border-radius: 6px;">
							<span class="dashicons dashicons-info" style="font-size: 14px; vertical-align: middle;"></span>
							<?php esc_html_e( 'Using logo as favicon (no custom favicon set)', 'buildfuly' ); ?>
						</div>
					<?php endif; ?>
				<?php else : ?>
					<span style="color: #9ca3af;"><?php esc_html_e( 'No favicon yet', 'buildfuly' ); ?></span>
				<?php endif; ?>
			</div>
		</div>
		<div>
			<h3 style="margin-top: 0;"><?php esc_html_e( '🎨 Brand Colors', 'buildfuly' ); ?></h3>
			<div style="display: flex; gap: 12px; padding: 16px 0;">
				<div style="text-align: center;">
					<div style="width: 50px; height: 50px; border-radius: 10px; background: <?php echo esc_attr( $primary_color ); ?>; box-shadow: 0 2px 6px rgba(0,0,0,0.1);"></div>
					<span style="font-size: 11px; color: #6b7280;">Primary</span>
				</div>
				<div style="text-align: center;">
					<div style="width: 50px; height: 50px; border-radius: 10px; background: <?php echo esc_attr( $secondary_color ); ?>; box-shadow: 0 2px 6px rgba(0,0,0,0.1);"></div>
					<span style="font-size: 11px; color: #6b7280;">Secondary</span>
				</div>
			</div>
		</div>
	</div>

	<!-- Step 1: Style Selection -->
	<div id="step-1" class="buildfuly-card" style="margin-bottom: 24px;">
		<h3 style="margin-top: 0;">
			<span style="background: <?php echo esc_attr( $primary_color ); ?>; color: white; width: 28px; height: 28px; border-radius: 50%; display: inline-flex; align-items: center; justify-content: center; font-size: 14px; margin-right: 8px;">1</span>
			<?php esc_html_e( 'Pick a Style', 'buildfuly' ); ?>
		</h3>
		
		<div class="logo-styles-grid" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 16px; margin-bottom: 20px;">
			<?php foreach ( $example_logos as $style_id => $style ) : ?>
				<label class="logo-style-option" style="cursor: pointer;">
					<input type="radio" name="logo_style" value="<?php echo esc_attr( $style_id ); ?>" style="display: none;" <?php checked( $style_id, 'style1' ); ?>>
					<div class="style-card" style="background: #fff; border: 3px solid #e5e7eb; border-radius: 12px; padding: 12px; text-align: center; transition: all 0.2s;">
						<div style="background: #f9fafb; border-radius: 8px; padding: 16px; height: 90px; display: flex; align-items: center; justify-content: center;">
							<img src="<?php echo esc_url( $style['url'] ); ?>" alt="<?php echo esc_attr( $style['name'] ); ?>" style="max-width: 100%; max-height: 100%; object-fit: contain;">
						</div>
						<div style="margin-top: 8px; font-size: 13px; font-weight: 600;"><?php echo esc_html( $style['name'] ); ?></div>
					</div>
				</label>
			<?php endforeach; ?>
		</div>
		
		<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 16px;">
			<div>
				<label style="display: block; font-weight: 500; margin-bottom: 6px; font-size: 14px;">
					<?php esc_html_e( 'Business Name', 'buildfuly' ); ?> <span style="color: #ef4444;">*</span>
				</label>
				<input type="text" id="logo-business-name" value="<?php echo esc_attr( $business_name ); ?>" 
					placeholder="<?php esc_attr_e( 'Your Business Name', 'buildfuly' ); ?>"
					style="width: 100%; padding: 10px 12px; border: 1px solid #d1d5db; border-radius: 8px; font-size: 14px;">
			</div>
			<div>
				<label style="display: block; font-weight: 500; margin-bottom: 6px; font-size: 14px;">
					<?php esc_html_e( 'Business Type', 'buildfuly' ); ?>
				</label>
				<input type="text" id="logo-business-type" value="" 
					placeholder="<?php esc_attr_e( 'e.g., Plumbing, Coffee Shop', 'buildfuly' ); ?>"
					style="width: 100%; padding: 10px 12px; border: 1px solid #d1d5db; border-radius: 8px; font-size: 14px;">
			</div>
		</div>
		
		<div style="margin-top: 20px; text-align: center;">
			<button type="button" id="generate-logos-btn" class="button button-primary" style="padding: 12px 32px; font-size: 15px; height: auto; border-radius: 8px;">
				<span class="dashicons dashicons-images-alt2" style="vertical-align: middle; margin-right: 6px;"></span>
				<?php esc_html_e( 'Generate 3 Options', 'buildfuly' ); ?>
			</button>
		</div>
	</div>

	<!-- Generation Progress -->
	<div id="generation-progress" class="buildfuly-card" style="margin-bottom: 24px; display: none;">
		<div style="text-align: center; padding: 40px;">
			<div style="width: 60px; height: 60px; border: 4px solid #e5e7eb; border-top-color: <?php echo esc_attr( $primary_color ); ?>; border-radius: 50%; animation: spin 1s linear infinite; margin: 0 auto 20px;"></div>
			<h3 id="progress-title" style="margin: 0 0 8px 0;"><?php esc_html_e( 'Generating 3 logo options...', 'buildfuly' ); ?></h3>
			<p id="progress-subtitle" style="color: #6b7280; margin: 0;"><?php esc_html_e( 'This takes about 30-60 seconds', 'buildfuly' ); ?></p>
		</div>
	</div>

	<!-- Step 2: Pick Your Logo -->
	<div id="step-2" class="buildfuly-card" style="margin-bottom: 24px; display: none;">
		<h3 style="margin-top: 0;">
			<span style="background: <?php echo esc_attr( $primary_color ); ?>; color: white; width: 28px; height: 28px; border-radius: 50%; display: inline-flex; align-items: center; justify-content: center; font-size: 14px; margin-right: 8px;">2</span>
			<?php esc_html_e( 'Pick Your Favorite', 'buildfuly' ); ?>
		</h3>
		
		<div id="logo-options" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px;">
			<!-- Logo options will be inserted here -->
		</div>
		
		<div style="margin-top: 20px; text-align: center;">
			<button type="button" id="regenerate-logos-btn" class="button" style="padding: 10px 20px;">
				<span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
				<?php esc_html_e( 'Generate New Options', 'buildfuly' ); ?>
			</button>
		</div>
	</div>

	<!-- Step 3: Processing & Final Result -->
	<div id="step-3" class="buildfuly-card" style="margin-bottom: 24px; display: none;">
		<h3 style="margin-top: 0;">
			<span style="background: <?php echo esc_attr( $primary_color ); ?>; color: white; width: 28px; height: 28px; border-radius: 50%; display: inline-flex; align-items: center; justify-content: center; font-size: 14px; margin-right: 8px;">3</span>
			<?php esc_html_e( 'Your Logo & Favicon', 'buildfuly' ); ?>
		</h3>
		
		<div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 24px;">
			<!-- Original -->
			<div style="text-align: center;">
				<div style="font-size: 13px; font-weight: 600; margin-bottom: 8px; color: #6b7280;"><?php esc_html_e( 'Original', 'buildfuly' ); ?></div>
				<div id="final-original" style="background: #f9fafb; border-radius: 12px; padding: 20px; min-height: 150px; display: flex; align-items: center; justify-content: center;"></div>
			</div>
			<!-- Transparent -->
			<div style="text-align: center;">
				<div style="font-size: 13px; font-weight: 600; margin-bottom: 8px; color: #6b7280;"><?php esc_html_e( 'Transparent', 'buildfuly' ); ?></div>
				<div id="final-transparent" style="background: repeating-conic-gradient(#e5e7eb 0% 25%, white 0% 50%) 50% / 16px 16px; border-radius: 12px; padding: 20px; min-height: 150px; display: flex; align-items: center; justify-content: center;"></div>
			</div>
			<!-- Favicon -->
			<div style="text-align: center;">
				<div style="font-size: 13px; font-weight: 600; margin-bottom: 8px; color: #6b7280;"><?php esc_html_e( 'Favicon', 'buildfuly' ); ?></div>
				<div id="final-favicon" style="background: #f9fafb; border-radius: 12px; padding: 20px; min-height: 150px; display: flex; align-items: center; justify-content: center; gap: 20px;"></div>
			</div>
		</div>
		
		<div style="margin-top: 24px; text-align: center; display: flex; gap: 12px; justify-content: center;">
			<button type="button" id="save-all-btn" class="button button-primary" style="padding: 12px 28px; font-size: 15px; height: auto;">
				<span class="dashicons dashicons-yes-alt" style="vertical-align: middle; margin-right: 6px;"></span>
				<?php esc_html_e( 'Save Logo & Favicon', 'buildfuly' ); ?>
			</button>
			<button type="button" id="back-to-options-btn" class="button" style="padding: 12px 20px;">
				<?php esc_html_e( '← Back to Options', 'buildfuly' ); ?>
			</button>
		</div>
	</div>
</div>

<!-- Crop Favicon Modal -->
<div id="crop-favicon-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.8); z-index: 100000; align-items: center; justify-content: center;">
	<div style="background: white; border-radius: 12px; max-width: 700px; width: 90%; max-height: 90vh; overflow: auto; position: relative;">
		<div style="padding: 24px; border-bottom: 1px solid #e5e7eb;">
			<h2 style="margin: 0; font-size: 20px;">
				<span class="dashicons dashicons-image-crop" style="vertical-align: middle;"></span>
				<?php esc_html_e( 'Crop Favicon', 'buildfuly' ); ?>
			</h2>
			<button id="close-crop-modal" style="position: absolute; top: 20px; right: 20px; background: none; border: none; font-size: 24px; cursor: pointer; color: #6b7280;">&times;</button>
		</div>
		<div style="padding: 24px;">
			<p style="color: #6b7280; margin-top: 0;">
				<?php esc_html_e( 'Select the area of your logo to use as favicon. The selection will be resized to 512x512px.', 'buildfuly' ); ?>
			</p>
			<div style="max-height: 500px; overflow: hidden;">
				<img id="crop-favicon-image" src="" style="max-width: 100%; display: block;">
			</div>
		</div>
		<div style="padding: 24px; border-top: 1px solid #e5e7eb; display: flex; gap: 12px; justify-content: flex-end;">
			<button id="cancel-crop-btn" class="button">
				<?php esc_html_e( 'Cancel', 'buildfuly' ); ?>
			</button>
			<button id="save-crop-btn" class="button button-primary">
				<span class="dashicons dashicons-yes" style="vertical-align: middle;"></span>
				<?php esc_html_e( 'Save Favicon', 'buildfuly' ); ?>
			</button>
		</div>
	</div>
</div>

<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/cropperjs/1.6.1/cropper.min.css">
<script src="https://cdnjs.cloudflare.com/ajax/libs/cropperjs/1.6.1/cropper.min.js"></script>

<style>
@keyframes spin { to { transform: rotate(360deg); } }
.spin { animation: spin 1s linear infinite; }

.logo-style-option input:checked + .style-card {
	border-color: <?php echo esc_attr( $primary_color ); ?>;
	box-shadow: 0 0 0 3px <?php echo esc_attr( $primary_color ); ?>25;
}
.logo-style-option:hover .style-card { border-color: <?php echo esc_attr( $primary_color ); ?>80; }

.logo-option-card { 
	border: 3px solid #e5e7eb; 
	border-radius: 12px; 
	padding: 16px; 
	cursor: pointer; 
	transition: all 0.2s;
	background: #fff;
}
.logo-option-card:hover { border-color: <?php echo esc_attr( $primary_color ); ?>80; transform: translateY(-2px); }
.logo-option-card.selected { 
	border-color: <?php echo esc_attr( $primary_color ); ?>; 
	box-shadow: 0 0 0 3px <?php echo esc_attr( $primary_color ); ?>25;
}

#logo-options img, #final-original img, #final-transparent img { max-width: 100%; max-height: 140px; object-fit: contain; }
#final-favicon img { border-radius: 4px; }
</style>

<script type="text/javascript">
(function() {
	function initLogoGenerator() {
		if (typeof jQuery === 'undefined') {
			setTimeout(initLogoGenerator, 100);
			return;
		}
		
		jQuery(function($) {
			// Fallback if buildfulyAdmin not defined
			if (typeof buildfulyAdmin === 'undefined') {
				window.buildfulyAdmin = {
					ajaxUrl: '<?php echo esc_js( admin_url( 'admin-ajax.php' ) ); ?>',
					apiUrl: '<?php echo esc_js( defined( 'BUILDFULY_API_URL' ) ? BUILDFULY_API_URL : 'https://build.fuly.io' ); ?>',
					licenseKey: '<?php echo esc_js( get_option( 'buildfuly_license_key', '' ) ); ?>',
					adminNonce: '<?php echo esc_js( wp_create_nonce( 'buildfuly_admin_nonce' ) ); ?>'
				};
			}
			
			const primaryColor = '<?php echo esc_js( $primary_color ); ?>';
			const secondaryColor = '<?php echo esc_js( $secondary_color ); ?>';
			
			let generatedLogos = [];
			let selectedLogoIndex = -1;
			let finalLogoData = { original: null, transparent: null, favicon: null };
	
	const styleImageUrls = {
		'style1': 'https://rjdavx8ozyznxeyh.public.blob.vercel-storage.com/production/pages/logo-wall/logos/563.png',
		'style2': 'https://rjdavx8ozyznxeyh.public.blob.vercel-storage.com/production/pages/logo-wall/logos/662.png',
		'style3': 'https://rjdavx8ozyznxeyh.public.blob.vercel-storage.com/production/pages/logo-wall/logos/2410.png',
		'style4': 'https://rjdavx8ozyznxeyh.public.blob.vercel-storage.com/production/pages/logo-wall/logos/452.png'
	};
	
	function buildPrompt(businessName, businessType, variation) {
		// Different approaches for each variation
		const approaches = [
			'Focus on a symbolic icon that represents what this business does or offers.',
			'Create an abstract or geometric mark that feels modern and professional.',
			'Use a creative interpretation of the first letter or initials as the icon.'
		];
		
		// Icon suggestions based on common business types
		const iconExamples = {
			'plumber': 'wrench, pipe, water drop, faucet',
			'plumbing': 'wrench, pipe, water drop, faucet',
			'electrician': 'lightning bolt, light bulb, power plug',
			'electrical': 'lightning bolt, light bulb, power plug',
			'restaurant': 'fork and knife, chef hat, plate',
			'cafe': 'coffee cup, steam, bean',
			'coffee': 'coffee cup, steam, bean',
			'bakery': 'bread, wheat, cupcake',
			'salon': 'scissors, comb, mirror',
			'barber': 'scissors, razor, pole',
			'gym': 'dumbbell, muscle, weights',
			'fitness': 'dumbbell, running figure, heart',
			'law': 'scales, gavel, pillar',
			'lawyer': 'scales, gavel, pillar',
			'dental': 'tooth, smile',
			'dentist': 'tooth, smile',
			'medical': 'cross, heart, stethoscope',
			'doctor': 'cross, heart, stethoscope',
			'construction': 'hammer, hard hat, building',
			'roofing': 'house roof, shingles',
			'landscaping': 'tree, leaf',
			'cleaning': 'sparkle, broom',
			'photography': 'camera, lens',
			'tech': 'circuit, code brackets',
			'software': 'code brackets, cursor'
		};
		
		// Find icon hints if business type matches
		let iconHint = '';
		if (businessType) {
			const lowerType = businessType.toLowerCase();
			for (const [key, icons] of Object.entries(iconExamples)) {
				if (lowerType.includes(key)) {
					iconHint = `Icon ideas for ${businessType}: ${icons}`;
					break;
				}
			}
		}
		
		// Build clean, focused prompt
		const iconDescription = businessType 
			? `Design an icon representing ${businessType} (like a ${iconHint || 'relevant symbol'})`
			: `Design an icon that fits a business named "${businessName}"`;
		
		return `Create a logo for "${businessName}":

TEXT RULES (VERY IMPORTANT):
- The ONLY text allowed is: "${businessName}"
- DO NOT add any extra words like "${businessType}", "services", "company", taglines, etc.
- Just the business name, nothing else

ICON RULES:
${iconDescription}
${approaches[variation] || approaches[0]}
- The icon must be DIFFERENT from the reference image's icon
- Create something NEW that represents ${businessType || 'this business'}

COLORS:
- Primary: ${primaryColor}
- Secondary: ${secondaryColor}
- Background: white

Generate a clean, professional logo with ONLY "${businessName}" as text and a new relevant icon.`;
	}
	
	async function generateSingleLogo(prompt, referenceImageUrl) {
		const response = await fetch(buildfulyAdmin.apiUrl + '/api/logo/generate', {
			method: 'POST',
			headers: {
				'Content-Type': 'application/json',
				'Authorization': 'Bearer ' + buildfulyAdmin.licenseKey
			},
			body: JSON.stringify({ prompt, businessName: $('#logo-business-name').val(), referenceImageUrl })
		});
		return response.json();
	}
	
	$('#generate-logos-btn, #regenerate-logos-btn').on('click', async function() {
		const businessName = $('#logo-business-name').val().trim();
		if (!businessName) {
			alert('<?php esc_html_e( 'Please enter your business name.', 'buildfuly' ); ?>');
			return;
		}
		
		const selectedStyle = $('input[name="logo_style"]:checked').val();
		const businessType = $('#logo-business-type').val().trim();
		const referenceImageUrl = styleImageUrls[selectedStyle] || styleImageUrls['style1'];
		
		$('#step-2').hide();
		$('#step-3').hide();
		$('#generation-progress').show();
		$('#progress-title').text('<?php esc_html_e( 'Generating 3 logo options...', 'buildfuly' ); ?>');
		
		generatedLogos = [];
		
		try {
			// Generate 3 logos sequentially (parallel might hit rate limits)
			for (let i = 0; i < 3; i++) {
				$('#progress-subtitle').text(`Creating option ${i + 1} of 3...`);
				const prompt = buildPrompt(businessName, businessType, i);
				const result = await generateSingleLogo(prompt, referenceImageUrl);
				if (result.success && (result.logoUrl || result.imageUrl)) {
					generatedLogos.push(result.logoUrl || result.imageUrl);
				}
			}
			
			if (generatedLogos.length === 0) {
				throw new Error('Failed to generate any logos');
			}
			
			// Show options
			$('#generation-progress').hide();
			displayLogoOptions();
			$('#step-2').show();
			
		} catch (error) {
			console.error('Generation error:', error);
			alert('<?php esc_html_e( 'Failed to generate logos: ', 'buildfuly' ); ?>' + error.message);
			$('#generation-progress').hide();
		}
	});
	
	function displayLogoOptions() {
		const container = $('#logo-options');
		container.empty();
		
		generatedLogos.forEach((url, index) => {
			container.append(`
				<div class="logo-option-card" data-index="${index}">
					<div style="background: #f9fafb; border-radius: 8px; padding: 20px; min-height: 140px; display: flex; align-items: center; justify-content: center;">
						<img src="${url}" alt="Option ${index + 1}">
					</div>
					<div style="text-align: center; margin-top: 12px;">
						<button class="button select-logo-btn" data-index="${index}" style="width: 100%;">
							<?php esc_html_e( 'Select This One', 'buildfuly' ); ?>
						</button>
					</div>
				</div>
			`);
		});
	}
	
	$(document).on('click', '.logo-option-card', function() {
		$('.logo-option-card').removeClass('selected');
		$(this).addClass('selected');
		selectedLogoIndex = $(this).data('index');
	});
	
	$(document).on('click', '.select-logo-btn', async function(e) {
		e.stopPropagation();
		selectedLogoIndex = $(this).data('index');
		await processSelectedLogo();
	});
	
	// Fetch image as base64 through WordPress proxy (avoids CORS)
	async function fetchImageAsBase64(imageUrl) {
		if (imageUrl.startsWith('data:')) {
			return imageUrl;
		}
		
		const response = await $.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_proxy_image',
				nonce: buildfulyAdmin.adminNonce,
				image_url: imageUrl
			}
		});
		
		if (response.success && response.data && response.data.base64) {
			return response.data.base64;
		}
		throw new Error(response.data?.message || 'Failed to fetch image');
	}
	
	// Client-side white background removal using canvas
	function removeWhiteBackground(base64Image, threshold = 245) {
		return new Promise((resolve, reject) => {
			const img = new Image();
			
			img.onload = function() {
				const canvas = document.createElement('canvas');
				const ctx = canvas.getContext('2d', { willReadFrequently: true });
				
				canvas.width = img.width;
				canvas.height = img.height;
				ctx.drawImage(img, 0, 0);
				
				try {
					const imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);
					const data = imageData.data;
					
					for (let i = 0; i < data.length; i += 4) {
						const r = data[i];
						const g = data[i + 1];
						const b = data[i + 2];
						const brightness = (r + g + b) / 3;
						const maxDiff = Math.max(Math.abs(r - g), Math.abs(g - b), Math.abs(r - b));
						const isGrayish = maxDiff < 20;
						
						if (brightness >= threshold && isGrayish) {
							data[i + 3] = 0;
						} else if (brightness >= (threshold - 30) && isGrayish) {
							const alpha = Math.round(255 * (1 - (brightness - (threshold - 30)) / 30));
							data[i + 3] = Math.min(data[i + 3], alpha);
						}
					}
					
					ctx.putImageData(imageData, 0, 0);
					resolve(canvas.toDataURL('image/png'));
				} catch (canvasError) {
					reject(canvasError);
				}
			};
			
			img.onerror = () => reject(new Error('Failed to load image'));
			img.src = base64Image;
		});
	}
	
	// Create favicon from image (resize to square)
	async function createFaviconFromImage(imageDataUrl, size = 512) {
		return new Promise((resolve, reject) => {
			const img = new Image();
			
			img.onload = function() {
				const canvas = document.createElement('canvas');
				const ctx = canvas.getContext('2d');
				
				canvas.width = size;
				canvas.height = size;
				
				// Calculate scaling to fit image in square while maintaining aspect ratio
				const scale = Math.min(size / img.width, size / img.height);
				const scaledWidth = img.width * scale;
				const scaledHeight = img.height * scale;
				
				// Center the image
				const x = (size - scaledWidth) / 2;
				const y = (size - scaledHeight) / 2;
				
				// Clear with transparency
				ctx.clearRect(0, 0, size, size);
				
				// Draw scaled image centered
				ctx.drawImage(img, x, y, scaledWidth, scaledHeight);
				
				resolve(canvas.toDataURL('image/png'));
			};
			
			img.onerror = function() {
				reject(new Error('Failed to load image for favicon creation'));
			};
			
			img.src = imageDataUrl;
		});
	}
	
	async function processSelectedLogo() {
		if (selectedLogoIndex < 0 || !generatedLogos[selectedLogoIndex]) {
			alert('<?php esc_html_e( 'Please select a logo first.', 'buildfuly' ); ?>');
			return;
		}
		
		const originalUrl = generatedLogos[selectedLogoIndex];
		
		$('#step-2').hide();
		$('#generation-progress').show();
		$('#progress-title').text('<?php esc_html_e( 'Processing your logo...', 'buildfuly' ); ?>');
		$('#progress-subtitle').text('<?php esc_html_e( 'Fetching image...', 'buildfuly' ); ?>');
		
		try {
			// Step 1: Fetch the image as base64 through our proxy
			let base64Image = await fetchImageAsBase64(originalUrl);
			
			// Step 2: Remove white background client-side
			$('#progress-subtitle').text('<?php esc_html_e( 'Removing background...', 'buildfuly' ); ?>');
			let transparentImage;
			try {
				transparentImage = await removeWhiteBackground(base64Image);
			} catch (bgError) {
				transparentImage = base64Image;
			}
			
			// Step 3: Create favicon using Gemini API (extracts just the icon)
			$('#progress-subtitle').text('<?php esc_html_e( 'Creating favicon...', 'buildfuly' ); ?>');
			const businessName = $('#logo-business-name').val().trim();
			const businessType = $('#logo-business-type').val().trim();
			
			let faviconImage;
			try {
				const faviconResponse = await fetch(buildfulyAdmin.apiUrl + '/api/logo/create-favicon', {
					method: 'POST',
					headers: {
						'Content-Type': 'application/json',
						'Authorization': 'Bearer ' + buildfulyAdmin.licenseKey
					},
					body: JSON.stringify({ 
						imageUrl: transparentImage,
						businessName: businessName,
						businessType: businessType
					})
				});
				const faviconResult = await faviconResponse.json();
				
				if (faviconResult.success && faviconResult.faviconUrl) {
					faviconImage = faviconResult.faviconUrl;
				} else {
					// Fallback: resize transparent logo
					faviconImage = await createFaviconFromImage(transparentImage, 512);
				}
			} catch (faviconError) {
				// Fallback: resize transparent logo
				faviconImage = await createFaviconFromImage(transparentImage, 512);
			}
			
			// Store final data
			finalLogoData = {
				original: base64Image,
				transparent: transparentImage,
				favicon: faviconImage
			};
			
			// Display results
			$('#generation-progress').hide();
			$('#final-original').html(`<img src="${finalLogoData.original}" alt="Original">`);
			$('#final-transparent').html(`<img src="${finalLogoData.transparent}" alt="Transparent">`);
			$('#final-favicon').html(`
				<div style="text-align: center;">
					<img src="${finalLogoData.favicon}" style="width: 64px; height: 64px;" alt="Favicon 64px">
					<div style="font-size: 10px; color: #6b7280; margin-top: 4px;">64px</div>
				</div>
				<div style="text-align: center;">
					<img src="${finalLogoData.favicon}" style="width: 32px; height: 32px;" alt="Favicon 32px">
					<div style="font-size: 10px; color: #6b7280; margin-top: 4px;">32px</div>
				</div>
				<div style="text-align: center;">
					<img src="${finalLogoData.favicon}" style="width: 16px; height: 16px;" alt="Favicon 16px">
					<div style="font-size: 10px; color: #6b7280; margin-top: 4px;">16px</div>
				</div>
			`);
			$('#step-3').show();
			
		} catch (error) {
			console.error('Processing error:', error);
			alert('<?php esc_html_e( 'Error: ', 'buildfuly' ); ?>' + error.message);
			$('#generation-progress').hide();
			$('#step-2').show();
		}
	}
	
	$('#back-to-options-btn').on('click', function() {
		$('#step-3').hide();
		$('#step-2').show();
	});
	
	$('#save-all-btn').on('click', async function() {
		const $btn = $(this);
		$btn.prop('disabled', true).text('<?php esc_html_e( 'Saving...', 'buildfuly' ); ?>');
		
		const uploadNonce = $('#buildfuly-nonce').val();
		const saveNonce = buildfulyAdmin.adminNonce;
		
		if (!finalLogoData.transparent || !finalLogoData.favicon) {
			alert('<?php esc_html_e( 'No logo data to save. Please process a logo first.', 'buildfuly' ); ?>');
			$btn.prop('disabled', false).html('<span class="dashicons dashicons-yes-alt" style="vertical-align: middle; margin-right: 6px;"></span> <?php esc_html_e( 'Save Logo & Favicon', 'buildfuly' ); ?>');
			return;
		}
		
		try {
			// Upload and save logo
			const logoResponse = await $.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_upload_image_from_url',
					nonce: uploadNonce,
					image_url: finalLogoData.transparent,
					title: ($('#logo-business-name').val() || 'Logo') + ' Logo',
					alt_text: ($('#logo-business-name').val() || 'Logo') + ' Logo',
					is_ai_generated: 'true'
				}
			});
			
			if (logoResponse.success) {
				await $.ajax({
					url: buildfulyAdmin.ajaxUrl,
					type: 'POST',
					data: {
						action: 'buildfuly_save_business_logo',
						nonce: saveNonce,
						logo_id: logoResponse.data.attachment_id
					}
				});
				$('#current-logo-preview').html(`<img src="${logoResponse.data.url}" alt="Current Logo" style="max-height: 80px; max-width: 180px;">`);
			} else {
				throw new Error(logoResponse.data?.message || 'Failed to upload logo');
			}
			
			// Upload and save favicon
			const faviconResponse = await $.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_upload_image_from_url',
					nonce: uploadNonce,
					image_url: finalLogoData.favicon,
					title: ($('#logo-business-name').val() || 'Logo') + ' Favicon',
					alt_text: 'Favicon',
					is_ai_generated: 'true'
				}
			});
			
			if (faviconResponse.success) {
				await $.ajax({
					url: buildfulyAdmin.ajaxUrl,
					type: 'POST',
					data: {
						action: 'buildfuly_save_favicon',
						nonce: saveNonce,
						favicon_id: faviconResponse.data.attachment_id
					}
				});
				$('#current-favicon-preview').html(`
					<div style="display: flex; align-items: center; gap: 16px;">
						<div style="text-align: center;"><img src="${faviconResponse.data.url}" style="width: 32px; height: 32px; border-radius: 4px;"><div style="font-size: 10px; color: #6b7280; margin-top: 4px;">32px</div></div>
						<div style="text-align: center;"><img src="${faviconResponse.data.url}" style="width: 16px; height: 16px;"><div style="font-size: 10px; color: #6b7280; margin-top: 4px;">16px</div></div>
					</div>
				`);
			} else {
				throw new Error(faviconResponse.data?.message || 'Failed to upload favicon');
			}
			
			alert('<?php esc_html_e( 'Logo and favicon saved successfully!', 'buildfuly' ); ?>');
			
		} catch (error) {
			alert('<?php esc_html_e( 'Failed to save: ', 'buildfuly' ); ?>' + error.message);
		}
		
		$btn.prop('disabled', false).html('<span class="dashicons dashicons-yes-alt" style="vertical-align: middle; margin-right: 6px;"></span> <?php esc_html_e( 'Save Logo & Favicon', 'buildfuly' ); ?>');
	});
	
	// Create favicon from existing logo
	$('#create-favicon-from-logo-btn').on('click', async function() {
		const $btn = $(this);
		const logoImg = $('#current-logo-preview img');
		
		if (!logoImg.length) {
			alert('<?php esc_html_e( 'No logo found. Please upload or generate a logo first.', 'buildfuly' ); ?>');
			return;
		}
		
		let logoUrl = logoImg.attr('src') || logoImg.data('logo-url');
		if (!logoUrl) {
			alert('<?php esc_html_e( 'Could not get logo URL.', 'buildfuly' ); ?>');
			return;
		}
		
		const businessName = $('#logo-business-name').val().trim() || '<?php echo esc_js( $business_name ); ?>';
		const businessType = $('#logo-business-type').val().trim();
		
		$btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin" style="vertical-align: middle;"></span> <?php esc_html_e( 'Creating...', 'buildfuly' ); ?>');
		
		try {
			// Call WordPress AJAX endpoint (handles base64 conversion server-side)
			const faviconResponse = await $.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_create_favicon_from_logo',
					nonce: buildfulyAdmin.adminNonce,
					logo_url: logoUrl,
					business_name: businessName,
					business_type: businessType
				}
			});
			
			if (!faviconResponse.success) {
				throw new Error(faviconResponse.data?.message || 'Failed to create favicon');
			}
			
			const faviconResult = faviconResponse.data;
			
			// Upload to WordPress media library
			const uploadResponse = await $.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_upload_image_from_url',
					nonce: $('#buildfuly-nonce').val(),
					image_url: faviconResult.faviconUrl,
					title: businessName + ' Favicon',
					alt_text: 'Favicon',
					is_ai_generated: 'true'
				}
			});
			
			if (uploadResponse.success) {
				// Save as site favicon
				await $.ajax({
					url: buildfulyAdmin.ajaxUrl,
					type: 'POST',
					data: {
						action: 'buildfuly_save_favicon',
						nonce: buildfulyAdmin.adminNonce,
						favicon_id: uploadResponse.data.attachment_id
					}
				});
				
				// Update preview
				$('#current-favicon-preview').html(`
					<div style="text-align: center;"><img src="${uploadResponse.data.url}" style="width: 32px; height: 32px; border-radius: 4px;"><div style="font-size: 10px; color: #6b7280; margin-top: 4px;">32px</div></div>
					<div style="text-align: center;"><img src="${uploadResponse.data.url}" style="width: 16px; height: 16px;"><div style="font-size: 10px; color: #6b7280; margin-top: 4px;">16px</div></div>
				`);
				
				alert('<?php esc_html_e( 'Favicon created and saved successfully!', 'buildfuly' ); ?>');
			} else {
				throw new Error(uploadResponse.data?.message || 'Failed to upload favicon');
			}
			
		} catch (error) {
			console.error('Favicon creation error:', error);
			alert('<?php esc_html_e( 'Failed to create favicon: ', 'buildfuly' ); ?>' + error.message);
		}
		
		$btn.prop('disabled', false).html('<span class="dashicons dashicons-format-image" style="vertical-align: middle; font-size: 14px;"></span> <?php esc_html_e( 'Create Favicon from Logo', 'buildfuly' ); ?>');
	});

	// Crop Favicon button
	$('#crop-favicon-btn').on('click', function() {
		const logoUrl = $('#current-logo-preview img').attr('src');
		if (!logoUrl) {
			alert('<?php esc_html_e( 'Please upload a logo first', 'buildfuly' ); ?>');
			return;
		}
		
		// Show crop modal
		$('#crop-favicon-modal').show();
		$('#crop-favicon-image').attr('src', logoUrl);
		
		// Initialize cropper
		if (window.faviconCropper) {
			window.faviconCropper.destroy();
		}
		
		setTimeout(() => {
			window.faviconCropper = new Cropper(document.getElementById('crop-favicon-image'), {
				aspectRatio: 1,
				viewMode: 1,
				autoCropArea: 0.8,
				guides: true,
				center: true,
				highlight: true,
				cropBoxResizable: true,
				cropBoxMovable: true,
				toggleDragModeOnDblclick: false
			});
		}, 100);
	});

	// Close crop modal
	$('#close-crop-modal, #cancel-crop-btn').on('click', function() {
		$('#crop-favicon-modal').hide();
		if (window.faviconCropper) {
			window.faviconCropper.destroy();
			window.faviconCropper = null;
		}
	});

	// Save cropped favicon
	$('#save-crop-btn').on('click', async function() {
		if (!window.faviconCropper) return;
		
		const $btn = $(this);
		$btn.prop('disabled', true).text('<?php esc_html_e( 'Creating...', 'buildfuly' ); ?>');
		
		try {
			// Get cropped canvas
			const canvas = window.faviconCropper.getCroppedCanvas({
				width: 512,
				height: 512,
				imageSmoothingQuality: 'high'
			});
			
			// Convert to blob
			const blob = await new Promise(resolve => canvas.toBlob(resolve, 'image/png'));
			
			// Convert to base64
			const reader = new FileReader();
			reader.onloadend = async function() {
				const base64data = reader.result;
				
				try {
					// Upload to WordPress media library
					const uploadResponse = await $.ajax({
						url: buildfulyAdmin.ajaxUrl,
						type: 'POST',
						data: {
							action: 'buildfuly_upload_image_from_url',
							nonce: $('#buildfuly-nonce').val(),
							image_url: base64data,
							title: '<?php echo esc_js( $business_name ); ?> Favicon (Cropped)',
							alt_text: 'Favicon',
							is_ai_generated: 'false'
						}
					});
					
					if (uploadResponse.success) {
						// Save as site favicon
						await $.ajax({
							url: buildfulyAdmin.ajaxUrl,
							type: 'POST',
							data: {
								action: 'buildfuly_save_favicon',
								nonce: buildfulyAdmin.adminNonce,
								favicon_id: uploadResponse.data.attachment_id
							}
						});
						
						// Update preview
						$('#current-favicon-preview').html(`
							<div style="display: flex; align-items: center; gap: 16px;">
								<div style="text-align: center;">
									<img src="${uploadResponse.data.url}" style="width: 32px; height: 32px; border-radius: 4px;">
									<div style="font-size: 10px; color: #6b7280; margin-top: 4px;">32px</div>
								</div>
								<div style="text-align: center;">
									<img src="${uploadResponse.data.url}" style="width: 16px; height: 16px;">
									<div style="font-size: 10px; color: #6b7280; margin-top: 4px;">16px</div>
								</div>
							</div>
						`);
						
						// Close modal
						$('#crop-favicon-modal').hide();
						if (window.faviconCropper) {
							window.faviconCropper.destroy();
							window.faviconCropper = null;
						}
						
						alert('<?php esc_html_e( 'Favicon cropped and saved successfully!', 'buildfuly' ); ?>');
					} else {
						throw new Error(uploadResponse.data?.message || 'Failed to upload favicon');
					}
				} catch (error) {
					console.error('Favicon upload error:', error);
					alert('<?php esc_html_e( 'Failed to save favicon: ', 'buildfuly' ); ?>' + error.message);
				}
				
				$btn.prop('disabled', false).text('<?php esc_html_e( 'Save Favicon', 'buildfuly' ); ?>');
			};
			reader.readAsDataURL(blob);
			
		} catch (error) {
			console.error('Crop error:', error);
			alert('<?php esc_html_e( 'Failed to crop favicon: ', 'buildfuly' ); ?>' + error.message);
			$btn.prop('disabled', false).text('<?php esc_html_e( 'Save Favicon', 'buildfuly' ); ?>');
		}
	});
		}); // End jQuery function
	} // End initLogoGenerator
	
	// Start initialization
	initLogoGenerator();
})();
</script>
