<?php
/**
 * Leads & Forms Management Display
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/admin/partials
 */

if (!defined('ABSPATH')) {
    exit;
}

// Get leads data
global $wpdb;
$leads_table = $wpdb->prefix . 'buildfuly_leads';
$forms_table = $wpdb->prefix . 'buildfuly_forms';

// Create forms table if not exists
$charset_collate = $wpdb->get_charset_collate();
$sql = "CREATE TABLE IF NOT EXISTS $forms_table (
    id mediumint(9) NOT NULL AUTO_INCREMENT,
    name varchar(255) NOT NULL,
    fields longtext NOT NULL,
    settings longtext,
    created_at datetime DEFAULT CURRENT_TIMESTAMP,
    updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (id)
) $charset_collate;";
require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
dbDelta($sql);

// Pagination
$per_page = 20;
$current_page = isset($_GET['lead_page']) ? max(1, intval($_GET['lead_page'])) : 1;
$offset = ($current_page - 1) * $per_page;

// Get total leads
$total_leads = $wpdb->get_var("SELECT COUNT(*) FROM $leads_table");
$total_pages = ceil($total_leads / $per_page);

// Get leads for current page
$leads = $wpdb->get_results($wpdb->prepare(
    "SELECT * FROM $leads_table ORDER BY created_at DESC LIMIT %d OFFSET %d",
    $per_page,
    $offset
));

// Get stats
$new_leads = $wpdb->get_var("SELECT COUNT(*) FROM $leads_table WHERE status = 'new'");
$contacted_leads = $wpdb->get_var("SELECT COUNT(*) FROM $leads_table WHERE status = 'contacted'");

// Get all forms
$forms = $wpdb->get_results("SELECT * FROM $forms_table ORDER BY created_at DESC");

// Get email settings with defaults (emails enabled by default)
$email_defaults = array(
    'send_to_admin' => true,
    'admin_email' => get_option('admin_email'),
    'send_to_lead' => true,
    'email_subject' => 'Your Estimate from ' . get_bloginfo('name'),
    'email_template' => "Hi {name},\n\nThank you for your interest! Here's your estimate:\n\n{estimate_details}\n\nTotal: {estimate_total}\n\nWe'll be in touch soon!\n\nBest regards,\n" . get_bloginfo('name')
);
$saved_settings = get_option('buildfuly_lead_email_settings', array());
$email_settings = array_merge($email_defaults, $saved_settings);

// Get SMTP settings
$smtp_settings = get_option('buildfuly_smtp_settings', array(
    'enabled' => false,
    'host' => '',
    'port' => 587,
    'encryption' => 'tls',
    'username' => '',
    'password' => '',
    'from_email' => get_option('admin_email'),
    'from_name' => get_bloginfo('name'),
    'debug' => false
));

// Get webhook settings
$webhook_settings = get_option('buildfuly_webhook_settings', array(
    'enabled' => false,
    'url' => '',
    'secret' => '',
    'headers' => '',
    'debug' => false
));

// Check if we're on localhost or if SMTP might be needed
$is_localhost = in_array($_SERVER['SERVER_NAME'] ?? '', ['localhost', '127.0.0.1']) || 
                strpos($_SERVER['HTTP_HOST'] ?? '', 'localhost') !== false;
$smtp_configured = !empty($smtp_settings['enabled']) && !empty($smtp_settings['host']);
$show_smtp_warning = !$smtp_configured;
?>

<div class="wrap buildfuly-leads-wrap">
    <h1>
        <span class="dashicons dashicons-email-alt" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px;"></span>
        Leads & Forms
    </h1>

    <?php if ($show_smtp_warning) : ?>
    <div class="notice notice-warning" style="padding: 12px 15px; margin: 15px 0;">
        <strong>⚠️ Email Setup Required:</strong> 
        <?php if ($is_localhost) : ?>
            You're on localhost - emails won't send without SMTP configuration.
        <?php else : ?>
            Configure SMTP for reliable email delivery (takes 2 minutes).
        <?php endif; ?>
        <a href="#settings" class="tab-link-inline" onclick="document.querySelector('[data-tab=settings]').click(); return false;">
            <strong>Quick Setup →</strong>
        </a>
        <br><small style="color: #666; margin-top: 5px; display: inline-block;">
            Click "Email Settings" tab → Choose your provider (Hostinger, Gmail, SendGrid) → Enter credentials → Done!
        </small>
    </div>
    <?php endif; ?>

    <!-- Stats Cards -->
    <div class="buildfuly-stats-cards">
        <div class="stat-card">
            <div class="stat-number"><?php echo esc_html($total_leads); ?></div>
            <div class="stat-label">Total Leads</div>
        </div>
        <div class="stat-card stat-new">
            <div class="stat-number"><?php echo esc_html($new_leads); ?></div>
            <div class="stat-label">New</div>
        </div>
        <div class="stat-card stat-contacted">
            <div class="stat-number"><?php echo esc_html($contacted_leads); ?></div>
            <div class="stat-label">Contacted</div>
        </div>
        <div class="stat-card stat-forms">
            <div class="stat-number"><?php echo count($forms); ?></div>
            <div class="stat-label">Forms</div>
        </div>
    </div>

    <!-- Tabs -->
    <div class="buildfuly-leads-tabs">
        <a href="#leads" class="tab-link active" data-tab="leads">
            <span class="dashicons dashicons-groups"></span> Leads
        </a>
        <a href="#forms" class="tab-link" data-tab="forms">
            <span class="dashicons dashicons-feedback"></span> Form Builder
        </a>
        <a href="#settings" class="tab-link" data-tab="settings">
            <span class="dashicons dashicons-admin-generic"></span> Email Settings
        </a>
        <a href="#webhook" class="tab-link" data-tab="webhook">
            <span class="dashicons dashicons-rest-api"></span> Webhooks
        </a>
        <a href="#shortcode" class="tab-link" data-tab="shortcode">
            <span class="dashicons dashicons-info"></span> Setup Guide
        </a>
    </div>

    <!-- Leads Tab -->
    <div class="tab-content" id="leads-tab" style="display: block;">
        <?php if (empty($leads)) : ?>
            <div class="buildfuly-empty-state">
                <span class="dashicons dashicons-email-alt" style="font-size: 48px; width: 48px; height: 48px; color: #ccc;"></span>
                <h3>No leads yet</h3>
                <p>Leads will appear here when visitors submit forms on your calculators.</p>
            </div>
        <?php else : ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 50px;">ID</th>
                        <th>Name</th>
                        <th>Email</th>
                        <th>Phone</th>
                        <th>Source</th>
                        <th>Estimate</th>
                        <th>Status</th>
                        <th>Date</th>
                        <th style="width: 100px;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($leads as $lead) : 
                        $data = json_decode($lead->form_data, true);
                    ?>
                        <tr>
                            <td><?php echo esc_html($lead->id); ?></td>
                            <td><strong><?php echo esc_html($lead->name ?: '-'); ?></strong></td>
                            <td>
                                <?php if ($lead->email) : ?>
                                    <a href="mailto:<?php echo esc_attr($lead->email); ?>">
                                        <?php echo esc_html($lead->email); ?>
                                    </a>
                                <?php else : ?>
                                    -
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($lead->phone) : ?>
                                    <a href="tel:<?php echo esc_attr($lead->phone); ?>">
                                        <?php echo esc_html($lead->phone); ?>
                                    </a>
                                <?php else : ?>
                                    -
                                <?php endif; ?>
                            </td>
                            <td><?php echo esc_html($lead->source ?: 'Direct'); ?></td>
                            <td>
                                <?php if (!empty($lead->estimate_total)) : ?>
                                    <strong style="color: #2271b1;">
                                        $<?php echo number_format((float)$lead->estimate_total, 2); ?>
                                    </strong>
                                <?php else : ?>
                                    -
                                <?php endif; ?>
                            </td>
                            <td>
                                <select class="lead-status-select" data-lead-id="<?php echo esc_attr($lead->id); ?>">
                                    <option value="new" <?php selected($lead->status, 'new'); ?>>🆕 New</option>
                                    <option value="contacted" <?php selected($lead->status, 'contacted'); ?>>📞 Contacted</option>
                                    <option value="qualified" <?php selected($lead->status, 'qualified'); ?>>✅ Qualified</option>
                                    <option value="converted" <?php selected($lead->status, 'converted'); ?>>🎉 Converted</option>
                                    <option value="lost" <?php selected($lead->status, 'lost'); ?>>❌ Lost</option>
                                </select>
                            </td>
                            <td><?php echo esc_html(date('M j, Y g:i a', strtotime($lead->created_at))); ?></td>
                            <td>
                                <button class="button button-small view-lead-details" 
                                        data-lead='<?php echo esc_attr(json_encode($lead)); ?>'>
                                    <span class="dashicons dashicons-visibility" style="vertical-align: middle;"></span>
                                </button>
                                <button class="button button-small delete-lead" 
                                        data-lead-id="<?php echo esc_attr($lead->id); ?>"
                                        style="color: #b32d2e;">
                                    <span class="dashicons dashicons-trash" style="vertical-align: middle;"></span>
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <?php if ($total_pages > 1) : ?>
                <div class="tablenav">
                    <div class="tablenav-pages">
                        <span class="displaying-num"><?php echo esc_html($total_leads); ?> items</span>
                        <span class="pagination-links">
                            <?php if ($current_page > 1) : ?>
                                <a class="prev-page button" href="<?php echo esc_url(add_query_arg('lead_page', $current_page - 1)); ?>">
                                    <span aria-hidden="true">‹</span>
                                </a>
                            <?php endif; ?>
                            <span class="paging-input">
                                <?php echo esc_html($current_page); ?> of <?php echo esc_html($total_pages); ?>
                            </span>
                            <?php if ($current_page < $total_pages) : ?>
                                <a class="next-page button" href="<?php echo esc_url(add_query_arg('lead_page', $current_page + 1)); ?>">
                                    <span aria-hidden="true">›</span>
                                </a>
                            <?php endif; ?>
                        </span>
                    </div>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>

    <!-- Form Builder Tab -->
    <div class="tab-content" id="forms-tab" style="display: none;">
        <div class="forms-header">
            <h2>Your Forms</h2>
            <button class="button button-primary" id="create-new-form">
                <span class="dashicons dashicons-plus-alt" style="vertical-align: middle;"></span>
                Create New Form
            </button>
        </div>

        <?php if (empty($forms)) : ?>
            <div class="buildfuly-empty-state">
                <span class="dashicons dashicons-feedback" style="font-size: 48px; width: 48px; height: 48px; color: #ccc;"></span>
                <h3>No forms yet</h3>
                <p>Create your first form to start capturing leads!</p>
                <button class="button button-primary button-hero" id="create-first-form">
                    <span class="dashicons dashicons-plus-alt" style="vertical-align: middle;"></span>
                    Create Your First Form
                </button>
            </div>
        <?php else : ?>
            <div class="forms-grid">
                <?php foreach ($forms as $form) : 
                    $fields = json_decode($form->fields, true);
                    $field_count = is_array($fields) ? count($fields) : 0;
                ?>
                    <div class="form-card" data-form-id="<?php echo esc_attr($form->id); ?>">
                        <div class="form-card-header">
                            <h3><?php echo esc_html($form->name); ?></h3>
                            <span class="field-count"><?php echo $field_count; ?> fields</span>
                        </div>
                        <div class="form-card-body">
                            <div class="shortcode-display">
                                <code>[buildfuly_form id="<?php echo esc_attr($form->id); ?>"]</code>
                                <button class="button button-small copy-shortcode" 
                                        data-shortcode='[buildfuly_form id="<?php echo esc_attr($form->id); ?>"]'>
                                    <span class="dashicons dashicons-clipboard" style="vertical-align: middle;"></span>
                                </button>
                            </div>
                        </div>
                        <div class="form-card-footer">
                            <button class="button edit-form" data-form-id="<?php echo esc_attr($form->id); ?>">
                                <span class="dashicons dashicons-edit" style="vertical-align: middle;"></span> Edit
                            </button>
                            <button class="button preview-form" data-form-id="<?php echo esc_attr($form->id); ?>">
                                <span class="dashicons dashicons-visibility" style="vertical-align: middle;"></span> Preview
                            </button>
                            <button class="button delete-form" data-form-id="<?php echo esc_attr($form->id); ?>" style="color: #b32d2e;">
                                <span class="dashicons dashicons-trash" style="vertical-align: middle;"></span>
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <!-- Form Builder Modal -->
        <div id="form-builder-modal" class="buildfuly-modal" style="display: none;">
            <div class="buildfuly-modal-content form-builder-modal">
                <div class="buildfuly-modal-header">
                    <h2 id="form-builder-title">Create New Form</h2>
                    <button class="buildfuly-modal-close">&times;</button>
                </div>
                <div class="buildfuly-modal-body">
                    <div class="form-builder-container">
                        <!-- Left: Field Types -->
                        <div class="field-types-panel">
                            <h3>Add Fields</h3>
                            <p class="description">Click to add fields to your form</p>
                            <div class="field-types">
                                <button class="add-field-btn" data-type="text">
                                    <span class="dashicons dashicons-editor-textcolor"></span>
                                    Text Input
                                </button>
                                <button class="add-field-btn" data-type="email">
                                    <span class="dashicons dashicons-email"></span>
                                    Email
                                </button>
                                <button class="add-field-btn" data-type="phone">
                                    <span class="dashicons dashicons-phone"></span>
                                    Phone
                                </button>
                                <button class="add-field-btn" data-type="number">
                                    <span class="dashicons dashicons-calculator"></span>
                                    Number
                                </button>
                                <button class="add-field-btn" data-type="textarea">
                                    <span class="dashicons dashicons-text"></span>
                                    Text Area
                                </button>
                                <button class="add-field-btn" data-type="select">
                                    <span class="dashicons dashicons-arrow-down-alt2"></span>
                                    Dropdown
                                </button>
                                <button class="add-field-btn" data-type="checkbox">
                                    <span class="dashicons dashicons-yes"></span>
                                    Checkbox
                                </button>
                                <button class="add-field-btn" data-type="radio">
                                    <span class="dashicons dashicons-marker"></span>
                                    Radio Buttons
                                </button>
                                <button class="add-field-btn" data-type="date">
                                    <span class="dashicons dashicons-calendar-alt"></span>
                                    Date
                                </button>
                                <button class="add-field-btn" data-type="hidden">
                                    <span class="dashicons dashicons-hidden"></span>
                                    Hidden Field
                                </button>
                                
                                <hr style="margin: 15px 0; border: none; border-top: 1px solid #ddd;">
                                <p class="description" style="margin-bottom: 8px;"><strong>Advanced</strong></p>
                                
                                <button class="add-field-btn" data-type="html">
                                    <span class="dashicons dashicons-editor-code"></span>
                                    Custom HTML/Calculator
                                </button>
                                <button class="add-field-btn" data-type="divider">
                                    <span class="dashicons dashicons-minus"></span>
                                    Divider/Heading
                                </button>
                            </div>
                        </div>

                        <!-- Center: Form Preview -->
                        <div class="form-preview-panel">
                            <div class="form-name-input">
                                <label>Form Name</label>
                                <input type="text" id="form-name" placeholder="My Contact Form" value="">
                            </div>
                            
                            <h3>Form Preview</h3>
                            <div id="form-fields-preview" class="form-fields-preview">
                                <div class="empty-form-message">
                                    <span class="dashicons dashicons-plus-alt2"></span>
                                    <p>Click field types on the left to add them here</p>
                                </div>
                            </div>

                            <div class="form-settings">
                                <h4>Submit Button</h4>
                                <input type="text" id="submit-button-text" placeholder="Submit" value="Get My Estimate">
                                
                                <h4 style="margin-top: 15px;">Success Message</h4>
                                <input type="text" id="success-message" placeholder="Thank you! We'll be in touch." value="Thank you! Check your email for your estimate.">
                                
                                <div class="calculator-integration-info" style="margin-top: 15px; padding: 12px; background: #f0f7fc; border-radius: 6px; border-left: 3px solid #2271b1;">
                                    <strong style="display: flex; align-items: center; gap: 5px;">
                                        <span class="dashicons dashicons-info"></span> Calculator Integration
                                    </strong>
                                    <p style="margin: 8px 0 0; font-size: 12px; color: #50575e;">
                                        Add a <strong>Custom HTML/Calculator</strong> field to embed your calculator. 
                                        Your calculator can set estimate values that will be captured with the lead.
                                    </p>
                                </div>
                            </div>
                        </div>

                        <!-- Right: Field Settings -->
                        <div class="field-settings-panel">
                            <h3>Field Settings</h3>
                            <div id="field-settings-content">
                                <p class="description">Select a field to edit its settings</p>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="buildfuly-modal-footer">
                    <button class="button" id="cancel-form-builder">Cancel</button>
                    <button class="button button-primary" id="save-form">
                        <span class="dashicons dashicons-saved" style="vertical-align: middle;"></span>
                        Save Form
                    </button>
                </div>
            </div>
        </div>

        <!-- Form Preview Modal -->
        <div id="form-preview-modal" class="buildfuly-modal" style="display: none;">
            <div class="buildfuly-modal-content">
                <div class="buildfuly-modal-header">
                    <h2>Form Preview</h2>
                    <button class="buildfuly-modal-close">&times;</button>
                </div>
                <div class="buildfuly-modal-body">
                    <div id="form-preview-content"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Email Settings Tab -->
    <div class="tab-content" id="settings-tab" style="display: none;">
        <form id="buildfuly-email-settings-form">
            <table class="form-table">
                <tr>
                    <th scope="row">Send to Admin</th>
                    <td>
                        <label>
                            <input type="checkbox" name="send_to_admin" value="1" 
                                   <?php checked($email_settings['send_to_admin'], true); ?>>
                            Send notification emails to admin when new leads are captured
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Admin Email</th>
                    <td>
                        <input type="email" name="admin_email" class="regular-text" 
                               value="<?php echo esc_attr($email_settings['admin_email']); ?>">
                        <p class="description">Where to send admin notifications</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Send to Lead</th>
                    <td>
                        <label>
                            <input type="checkbox" name="send_to_lead" value="1" 
                                   <?php checked($email_settings['send_to_lead'], true); ?>>
                            Send estimate emails to leads (requires email field in form)
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Email Subject</th>
                    <td>
                        <input type="text" name="email_subject" class="large-text" 
                               value="<?php echo esc_attr($email_settings['email_subject']); ?>">
                    </td>
                </tr>
                <tr>
                    <th scope="row">Email Template</th>
                    <td>
                        <textarea name="email_template" rows="10" class="large-text"><?php 
                            echo esc_textarea($email_settings['email_template']); 
                        ?></textarea>
                        <p class="description">
                            Available variables: <code>{name}</code>, <code>{email}</code>, <code>{phone}</code>, 
                            <code>{estimate_total}</code>, <code>{estimate_details}</code>, <code>{site_name}</code>
                        </p>
                    </td>
                </tr>
            </table>
            <p class="submit">
                <button type="submit" class="button button-primary">Save Email Settings</button>
            </p>
        </form>

        <hr style="margin: 30px 0;">

        <!-- SMTP Settings -->
        <h2>SMTP Configuration</h2>
        
        <!-- Quick Setup Presets -->
        <div class="smtp-presets" style="margin-bottom: 25px; padding: 20px; background: #f0f7fc; border-radius: 8px; border: 1px solid #c3d9ed;">
            <h3 style="margin-top: 0; display: flex; align-items: center; gap: 8px;">
                <span class="dashicons dashicons-superhero" style="color: #2271b1;"></span>
                Quick Setup (Choose Your Provider)
            </h3>
            <p class="description" style="margin-bottom: 15px;">Click a button to auto-fill settings. You just need to enter your credentials.</p>
            
            <div style="display: flex; flex-wrap: wrap; gap: 10px;">
                <button type="button" class="button smtp-preset-btn" data-preset="hostinger" style="display: flex; align-items: center; gap: 5px;">
                    🏠 Hostinger
                </button>
                <button type="button" class="button smtp-preset-btn" data-preset="gmail" style="display: flex; align-items: center; gap: 5px;">
                    📧 Gmail
                </button>
                <button type="button" class="button smtp-preset-btn" data-preset="sendgrid" style="display: flex; align-items: center; gap: 5px;">
                    📨 SendGrid <small style="color: #666;">(Free)</small>
                </button>
                <button type="button" class="button smtp-preset-btn" data-preset="brevo" style="display: flex; align-items: center; gap: 5px;">
                    💌 Brevo <small style="color: #666;">(Free)</small>
                </button>
                <button type="button" class="button smtp-preset-btn" data-preset="mailgun" style="display: flex; align-items: center; gap: 5px;">
                    ✉️ Mailgun
                </button>
                <button type="button" class="button smtp-preset-btn" data-preset="outlook" style="display: flex; align-items: center; gap: 5px;">
                    📬 Outlook/Office365
                </button>
            </div>
            
            <!-- Preset Instructions (shown when preset selected) -->
            <div id="smtp-preset-instructions" style="display: none; margin-top: 15px; padding: 15px; background: #fff; border-radius: 6px; border-left: 4px solid #2271b1;">
            </div>
        </div>
        
        <p class="description" style="margin-bottom: 20px;">
            Configure SMTP to reliably send emails. Required for localhost development and recommended for production.
        </p>
        
        <form id="buildfuly-smtp-settings-form">
            <table class="form-table">
                <tr>
                    <th scope="row">Enable SMTP</th>
                    <td>
                        <label>
                            <input type="checkbox" name="smtp_enabled" value="1" 
                                   <?php checked(!empty($smtp_settings['enabled'])); ?>>
                            Use SMTP for sending emails
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row">SMTP Host</th>
                    <td>
                        <input type="text" name="smtp_host" class="regular-text" 
                               value="<?php echo esc_attr($smtp_settings['host']); ?>"
                               placeholder="smtp.gmail.com">
                        <p class="description">
                            Gmail: <code>smtp.gmail.com</code> | 
                            SendGrid: <code>smtp.sendgrid.net</code> | 
                            Mailgun: <code>smtp.mailgun.org</code>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">SMTP Port</th>
                    <td>
                        <input type="number" name="smtp_port" class="small-text" 
                               value="<?php echo esc_attr($smtp_settings['port'] ?? 587); ?>">
                        <p class="description">Usually 587 (TLS) or 465 (SSL)</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Encryption</th>
                    <td>
                        <select name="smtp_encryption">
                            <option value="tls" <?php selected($smtp_settings['encryption'] ?? 'tls', 'tls'); ?>>TLS</option>
                            <option value="ssl" <?php selected($smtp_settings['encryption'] ?? 'tls', 'ssl'); ?>>SSL</option>
                            <option value="" <?php selected($smtp_settings['encryption'] ?? 'tls', ''); ?>>None</option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Username</th>
                    <td>
                        <input type="text" name="smtp_username" class="regular-text" 
                               value="<?php echo esc_attr($smtp_settings['username']); ?>"
                               placeholder="your-email@gmail.com">
                        <p class="description">Your email address or API key (for SendGrid use "apikey")</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Password</th>
                    <td>
                        <input type="password" name="smtp_password" class="regular-text" 
                               value="<?php echo esc_attr($smtp_settings['password']); ?>"
                               placeholder="••••••••">
                        <p class="description">
                            For Gmail: Use an <a href="https://myaccount.google.com/apppasswords" target="_blank">App Password</a> (not your regular password)
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">From Email</th>
                    <td>
                        <input type="email" name="smtp_from_email" class="regular-text" 
                               value="<?php echo esc_attr($smtp_settings['from_email']); ?>">
                    </td>
                </tr>
                <tr>
                    <th scope="row">From Name</th>
                    <td>
                        <input type="text" name="smtp_from_name" class="regular-text" 
                               value="<?php echo esc_attr($smtp_settings['from_name']); ?>">
                    </td>
                </tr>
                <tr>
                    <th scope="row">Debug Mode</th>
                    <td>
                        <label>
                            <input type="checkbox" name="smtp_debug" value="1" 
                                   <?php checked(!empty($smtp_settings['debug'])); ?>>
                            Log SMTP debug info (check error log)
                        </label>
                    </td>
                </tr>
            </table>
            <p class="submit">
                <button type="submit" class="button button-primary">Save SMTP Settings</button>
                <button type="button" class="button" id="test-smtp-btn" style="margin-left: 10px;">
                    Send Test Email
                </button>
            </p>
        </form>
    </div>

    <!-- Webhook Settings Tab -->
    <div class="tab-content" id="webhook-tab" style="display: none;">
        <h2>Webhook Configuration</h2>
        <p class="description" style="margin-bottom: 20px;">
            Send lead data to external services like Zapier, Make (Integromat), n8n, or your own API endpoint. 
            Webhooks are sent asynchronously and won't slow down form submissions.
        </p>
        
        <form id="buildfuly-webhook-settings-form">
            <table class="form-table">
                <tr>
                    <th scope="row">Enable Webhooks</th>
                    <td>
                        <label>
                            <input type="checkbox" name="webhook_enabled" value="1" 
                                   <?php checked($webhook_settings['enabled'] ?? false); ?>>
                            Send webhook notifications when leads are captured
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Webhook URL</th>
                    <td>
                        <input type="url" name="webhook_url" class="large-text code" 
                               value="<?php echo esc_attr($webhook_settings['url'] ?? ''); ?>"
                               placeholder="https://hooks.zapier.com/hooks/catch/...">
                        <p class="description">
                            <strong>Popular webhook services:</strong><br>
                            • <a href="https://zapier.com/apps/webhook/integrations" target="_blank">Zapier</a> - Connect to 5,000+ apps<br>
                            • <a href="https://www.make.com/en/integrations/webhook" target="_blank">Make (Integromat)</a> - Advanced automation<br>
                            • <a href="https://n8n.io/" target="_blank">n8n</a> - Self-hosted automation<br>
                            • Or use your own API endpoint
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Webhook Secret (Optional)</th>
                    <td>
                        <input type="text" name="webhook_secret" class="regular-text" 
                               value="<?php echo esc_attr($webhook_settings['secret'] ?? ''); ?>"
                               placeholder="your-secret-key">
                        <p class="description">
                            If provided, webhooks will include an <code>X-Buildfuly-Signature</code> header with HMAC SHA256 signature for verification.
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Custom Headers (Optional)</th>
                    <td>
                        <textarea name="webhook_headers" rows="4" class="large-text code" 
                                  placeholder="Authorization: Bearer your-token&#10;X-Custom-Header: value"><?php 
                            echo esc_textarea($webhook_settings['headers'] ?? ''); 
                        ?></textarea>
                        <p class="description">
                            One header per line in format: <code>Header-Name: value</code>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Debug Mode</th>
                    <td>
                        <label>
                            <input type="checkbox" name="webhook_debug" value="1" 
                                   <?php checked($webhook_settings['debug'] ?? false); ?>>
                            Log webhook errors to WordPress debug log
                        </label>
                    </td>
                </tr>
            </table>
            
            <p class="submit">
                <button type="submit" class="button button-primary">Save Webhook Settings</button>
                <button type="button" class="button" id="test-webhook-btn" style="margin-left: 10px;">
                    Send Test Webhook
                </button>
            </p>
        </form>

        <hr style="margin: 30px 0;">

        <h3>Webhook Payload Example</h3>
        <p class="description">Your webhook endpoint will receive this JSON data:</p>
        <div class="code-example">
<pre>{
  "event": "lead.created",
  "lead_id": 123,
  "email": "customer@example.com",
  "name": "John Doe",
  "phone": "555-0123",
  "source": "Driveway Calculator",
  "estimate_total": 1500.00,
  "estimate_details": "Service A: $500, Service B: $1000",
  "created_at": "2024-01-15 14:30:00",
  "site_name": "Your Site Name",
  "site_url": "https://yoursite.com",
  "webhook_timestamp": 1705329000
}</pre>
        </div>

        <h3 style="margin-top: 30px;">Quick Setup Guides</h3>
        <div class="setup-section">
            <h4>Zapier Setup</h4>
            <ol>
                <li>Create a new Zap at <a href="https://zapier.com" target="_blank">zapier.com</a></li>
                <li>Choose "Webhooks by Zapier" as the trigger</li>
                <li>Select "Catch Hook"</li>
                <li>Copy the webhook URL and paste it above</li>
                <li>Click "Send Test Webhook" to test the connection</li>
                <li>Choose your action (Google Sheets, Slack, Email, etc.)</li>
            </ol>
        </div>

        <div class="setup-section">
            <h4>Make (Integromat) Setup</h4>
            <ol>
                <li>Create a new scenario at <a href="https://www.make.com" target="_blank">make.com</a></li>
                <li>Add "Webhooks" module as the first step</li>
                <li>Select "Custom webhook"</li>
                <li>Copy the webhook URL and paste it above</li>
                <li>Click "Send Test Webhook" to send data</li>
                <li>Add your automation actions</li>
            </ol>
        </div>
    </div>

    <!-- Setup Guide Tab -->
    <div class="tab-content" id="shortcode-tab" style="display: none;">
        <div class="buildfuly-setup-guide">
            <h2>How to Capture Leads</h2>
            
            <div class="setup-section">
                <h3>Option 1: Use Form Builder (Recommended)</h3>
                <ol>
                    <li>Go to the <strong>Form Builder</strong> tab</li>
                    <li>Click <strong>Create New Form</strong></li>
                    <li>Add fields by clicking the field type buttons</li>
                    <li>Customize each field's settings (label, placeholder, required)</li>
                    <li>Save your form</li>
                    <li>Copy the shortcode and paste it in any page or post</li>
                </ol>
                <div class="code-example">
                    <code>[buildfuly_form id="1"]</code>
                </div>
            </div>

            <div class="setup-section">
                <h3>Option 2: Quick Capture Shortcode</h3>
                <p>Add a simple lead capture form to any calculator:</p>
                <div class="code-example">
                    <code>[buildfuly_capture calculator="My Calculator" button="Get My Estimate"]</code>
                </div>
                <h4>Available Options:</h4>
                <table class="widefat">
                    <thead>
                        <tr>
                            <th>Option</th>
                            <th>Default</th>
                            <th>Description</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>calculator</code></td>
                            <td>"Calculator"</td>
                            <td>Name to identify the source in leads list</td>
                        </tr>
                        <tr>
                            <td><code>button</code></td>
                            <td>"Get My Estimate"</td>
                            <td>Submit button text</td>
                        </tr>
                        <tr>
                            <td><code>show_name</code></td>
                            <td>"true"</td>
                            <td>Show name field</td>
                        </tr>
                        <tr>
                            <td><code>show_phone</code></td>
                            <td>"true"</td>
                            <td>Show phone field</td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <div class="setup-section">
                <h3>Option 3: Advanced Integration</h3>
                <p>For custom calculator integrations, you can submit leads via JavaScript:</p>
                <div class="code-example">
<pre>// Submit lead data programmatically
jQuery.post(ajaxurl, {
    action: 'buildfuly_submit_lead',
    nonce: buildfuly_leads.nonce,
    name: 'John Doe',
    email: 'john@example.com',
    phone: '555-0123',
    source: 'Custom Calculator',
    estimate_total: 1500.00,
    estimate_details: 'Service A: $500, Service B: $1000'
});</pre>
                </div>
            </div>
        </div>
    </div>

    <!-- Lead Details Modal -->
    <div id="lead-details-modal" class="buildfuly-modal" style="display: none;">
        <div class="buildfuly-modal-content">
            <div class="buildfuly-modal-header">
                <h2>Lead Details</h2>
                <button class="buildfuly-modal-close">&times;</button>
            </div>
            <div class="buildfuly-modal-body" id="lead-details-content">
            </div>
        </div>
    </div>
</div>

<style>
/* Stats Cards */
.buildfuly-stats-cards {
    display: flex;
    gap: 20px;
    margin: 20px 0;
}
.stat-card {
    background: #fff;
    border: 1px solid #c3c4c7;
    border-radius: 8px;
    padding: 20px 30px;
    text-align: center;
    min-width: 120px;
}
.stat-card .stat-number {
    font-size: 32px;
    font-weight: bold;
    color: #1d2327;
}
.stat-card .stat-label {
    color: #646970;
    margin-top: 5px;
}
.stat-card.stat-new .stat-number { color: #2271b1; }
.stat-card.stat-contacted .stat-number { color: #00a32a; }
.stat-card.stat-forms .stat-number { color: #8c5eb6; }

/* Tabs */
.buildfuly-leads-tabs {
    display: flex;
    gap: 0;
    border-bottom: 1px solid #c3c4c7;
    margin-bottom: 20px;
}
.buildfuly-leads-tabs .tab-link {
    padding: 12px 20px;
    text-decoration: none;
    color: #646970;
    border: 1px solid transparent;
    border-bottom: none;
    margin-bottom: -1px;
    background: transparent;
    display: flex;
    align-items: center;
    gap: 5px;
}
.buildfuly-leads-tabs .tab-link:hover {
    color: #2271b1;
}
.buildfuly-leads-tabs .tab-link.active {
    background: #fff;
    border-color: #c3c4c7;
    color: #1d2327;
    font-weight: 500;
}
.buildfuly-leads-tabs .tab-link .dashicons {
    font-size: 16px;
    width: 16px;
    height: 16px;
}

/* Empty State */
.buildfuly-empty-state {
    text-align: center;
    padding: 60px 20px;
    background: #fff;
    border: 1px solid #c3c4c7;
    border-radius: 8px;
}
.buildfuly-empty-state h3 {
    margin: 20px 0 10px;
}
.buildfuly-empty-state p {
    color: #646970;
}

/* Lead Status Select */
.lead-status-select {
    padding: 4px 8px;
    border-radius: 4px;
    border: 1px solid #c3c4c7;
}

/* Forms Grid */
.forms-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}
.forms-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
    gap: 20px;
}
.form-card {
    background: #fff;
    border: 1px solid #c3c4c7;
    border-radius: 8px;
    overflow: hidden;
}
.form-card-header {
    padding: 15px 20px;
    border-bottom: 1px solid #f0f0f1;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.form-card-header h3 {
    margin: 0;
    font-size: 16px;
}
.form-card-header .field-count {
    background: #f0f0f1;
    padding: 2px 8px;
    border-radius: 10px;
    font-size: 12px;
    color: #646970;
}
.form-card-body {
    padding: 15px 20px;
}
.shortcode-display {
    display: flex;
    align-items: center;
    gap: 10px;
    background: #f6f7f7;
    padding: 10px;
    border-radius: 4px;
}
.shortcode-display code {
    flex: 1;
    background: transparent;
    font-size: 12px;
}
.form-card-footer {
    padding: 15px 20px;
    border-top: 1px solid #f0f0f1;
    display: flex;
    gap: 10px;
}

/* Modal */
.buildfuly-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.7);
    z-index: 100000;
    display: flex;
    align-items: center;
    justify-content: center;
}
.buildfuly-modal-content {
    background: #fff;
    border-radius: 8px;
    max-width: 600px;
    width: 90%;
    max-height: 90vh;
    overflow: auto;
}
.buildfuly-modal-content.form-builder-modal {
    max-width: 1200px;
}
.buildfuly-modal-header {
    padding: 15px 20px;
    border-bottom: 1px solid #c3c4c7;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.buildfuly-modal-header h2 {
    margin: 0;
}
.buildfuly-modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #646970;
}
.buildfuly-modal-body {
    padding: 20px;
}
.buildfuly-modal-footer {
    padding: 15px 20px;
    border-top: 1px solid #c3c4c7;
    display: flex;
    justify-content: flex-end;
    gap: 10px;
}

/* Form Builder */
.form-builder-container {
    display: grid;
    grid-template-columns: 200px 1fr 250px;
    gap: 20px;
    min-height: 400px;
}
.field-types-panel {
    background: #f6f7f7;
    padding: 15px;
    border-radius: 8px;
}
.field-types-panel h3 {
    margin-top: 0;
    font-size: 14px;
}
.field-types {
    display: flex;
    flex-direction: column;
    gap: 8px;
}
.add-field-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 12px;
    background: #fff;
    border: 1px solid #c3c4c7;
    border-radius: 4px;
    cursor: pointer;
    text-align: left;
    font-size: 13px;
    transition: all 0.2s;
}
.add-field-btn:hover {
    border-color: #2271b1;
    background: #f0f7fc;
}
.add-field-btn .dashicons {
    color: #2271b1;
}

.form-preview-panel {
    background: #fff;
    border: 1px solid #c3c4c7;
    border-radius: 8px;
    padding: 20px;
}
.form-name-input {
    margin-bottom: 20px;
}
.form-name-input label {
    display: block;
    font-weight: 500;
    margin-bottom: 5px;
}
.form-name-input input {
    width: 100%;
    padding: 8px 12px;
    font-size: 16px;
}
.form-fields-preview {
    min-height: 200px;
    border: 2px dashed #c3c4c7;
    border-radius: 8px;
    padding: 20px;
}
.empty-form-message {
    text-align: center;
    color: #646970;
    padding: 40px 20px;
}
.empty-form-message .dashicons {
    font-size: 32px;
    width: 32px;
    height: 32px;
}
.form-settings {
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid #f0f0f1;
}
.form-settings h4 {
    margin: 0 0 10px;
}
.form-settings input {
    width: 100%;
}

.field-settings-panel {
    background: #f6f7f7;
    padding: 15px;
    border-radius: 8px;
}
.field-settings-panel h3 {
    margin-top: 0;
    font-size: 14px;
}

/* Form Field Preview Items */
.form-field-item {
    background: #fff;
    border: 1px solid #c3c4c7;
    border-radius: 4px;
    padding: 12px;
    margin-bottom: 10px;
    cursor: move;
    position: relative;
}
.form-field-item:hover {
    border-color: #2271b1;
}
.form-field-item.selected {
    border-color: #2271b1;
    box-shadow: 0 0 0 1px #2271b1;
}
.form-field-item .field-label {
    font-weight: 500;
    margin-bottom: 5px;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.form-field-item .field-label .required-marker {
    color: #d63638;
}
.form-field-item .field-preview input,
.form-field-item .field-preview textarea,
.form-field-item .field-preview select {
    width: 100%;
    padding: 6px 10px;
    border: 1px solid #c3c4c7;
    border-radius: 4px;
    background: #f6f7f7;
}
.form-field-item .field-actions {
    position: absolute;
    top: 8px;
    right: 8px;
    display: none;
}
.form-field-item:hover .field-actions {
    display: flex;
    gap: 5px;
}
.form-field-item .field-actions button {
    padding: 2px 6px;
    background: #fff;
    border: 1px solid #c3c4c7;
    border-radius: 3px;
    cursor: pointer;
}
.form-field-item .field-actions button:hover {
    background: #f0f0f1;
}

/* HTML Preview Box */
.html-preview-box {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: #fff;
    padding: 20px;
    border-radius: 6px;
    text-align: center;
    font-weight: 500;
}
.html-preview-box .dashicons {
    font-size: 24px;
    width: 24px;
    height: 24px;
    margin-bottom: 5px;
    display: block;
}
.html-preview-box small {
    opacity: 0.8;
    font-weight: normal;
}

/* HTML Preview Container in Form Builder */
.html-preview-container {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 6px;
    overflow: hidden;
}
.html-preview-header {
    color: #fff;
    padding: 12px 15px;
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 8px;
}
.html-preview-header .dashicons {
    font-size: 18px;
    width: 18px;
    height: 18px;
}
.html-preview-header .toggle-html-preview {
    margin-left: auto;
    font-size: 11px;
    padding: 2px 8px;
}
.html-live-preview {
    background: #fff;
    margin: 0 10px 10px;
    border-radius: 4px;
    min-height: 50px;
}

/* Field Settings Panel */
#field-html-content {
    font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
    font-size: 12px;
    line-height: 1.5;
    tab-size: 2;
    background: #1e1e1e;
    color: #d4d4d4;
    border: none;
    border-radius: 4px;
    padding: 12px;
}

#html-code-preview {
    background: #fff;
    border: 1px solid #c3c4c7;
    border-radius: 4px;
}
#html-code-preview iframe {
    display: block;
}

/* Field Settings Form */
.field-setting {
    margin-bottom: 15px;
}
.field-setting label {
    display: block;
    font-weight: 500;
    margin-bottom: 5px;
    font-size: 12px;
}
.field-setting input[type="text"],
.field-setting textarea {
    width: 100%;
    padding: 6px 10px;
    border: 1px solid #c3c4c7;
    border-radius: 4px;
}
.field-setting input[type="checkbox"] {
    margin-right: 5px;
}

/* Setup Guide */
.buildfuly-setup-guide {
    max-width: 800px;
}
.setup-section {
    background: #fff;
    border: 1px solid #c3c4c7;
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 20px;
}
.setup-section h3 {
    margin-top: 0;
}
.code-example {
    background: #f6f7f7;
    padding: 15px;
    border-radius: 4px;
    margin: 15px 0;
    overflow-x: auto;
}
.code-example code {
    background: transparent;
}
.code-example pre {
    margin: 0;
    white-space: pre-wrap;
}
</style>

<script>
jQuery(document).ready(function($) {
    
    // ==========================================
    // SMTP PRESETS - Quick Setup
    // ==========================================
    var smtpPresets = {
        hostinger: {
            name: 'Hostinger',
            host: 'smtp.hostinger.com',
            port: 465,
            encryption: 'ssl',
            username_placeholder: 'your-email@yourdomain.com',
            instructions: `
                <strong>Hostinger Setup:</strong>
                <ol style="margin: 10px 0; padding-left: 20px;">
                    <li>Go to <a href="https://hpanel.hostinger.com/emails" target="_blank">Hostinger hPanel → Emails</a></li>
                    <li>Create an email account (e.g., noreply@yourdomain.com)</li>
                    <li>Enter that email as Username below</li>
                    <li>Enter the email password as Password</li>
                </ol>
            `
        },
        gmail: {
            name: 'Gmail',
            host: 'smtp.gmail.com',
            port: 587,
            encryption: 'tls',
            username_placeholder: 'your-email@gmail.com',
            instructions: `
                <strong>Gmail Setup:</strong>
                <ol style="margin: 10px 0; padding-left: 20px;">
                    <li>Go to <a href="https://myaccount.google.com/apppasswords" target="_blank">Google App Passwords</a></li>
                    <li>Sign in and select "Mail" as the app</li>
                    <li>Click "Generate" and copy the 16-character password</li>
                    <li>Enter your Gmail address as Username</li>
                    <li>Paste the App Password as Password (NOT your Gmail password)</li>
                </ol>
                <p style="margin: 0; color: #666;"><strong>Requires:</strong> 2-Step Verification enabled on your Google account</p>
            `
        },
        sendgrid: {
            name: 'SendGrid',
            host: 'smtp.sendgrid.net',
            port: 587,
            encryption: 'tls',
            username_placeholder: 'apikey',
            instructions: `
                <strong>SendGrid Setup (100 free emails/day):</strong>
                <ol style="margin: 10px 0; padding-left: 20px;">
                    <li>Sign up at <a href="https://signup.sendgrid.com/" target="_blank">SendGrid.com</a> (free)</li>
                    <li>Go to Settings → API Keys → Create API Key</li>
                    <li>Give it "Full Access" and copy the key</li>
                    <li>Username is literally: <code>apikey</code></li>
                    <li>Password is your API key (starts with SG.)</li>
                </ol>
                <p style="margin: 0; color: #00a32a;"><strong>✓ Recommended:</strong> Best deliverability, owned by Twilio</p>
            `
        },
        brevo: {
            name: 'Brevo (Sendinblue)',
            host: 'smtp-relay.brevo.com',
            port: 587,
            encryption: 'tls',
            username_placeholder: 'your-email@example.com',
            instructions: `
                <strong>Brevo Setup (300 free emails/day):</strong>
                <ol style="margin: 10px 0; padding-left: 20px;">
                    <li>Sign up at <a href="https://www.brevo.com/free-smtp-server/" target="_blank">Brevo.com</a> (free)</li>
                    <li>Go to Settings → SMTP & API → SMTP</li>
                    <li>Copy your SMTP key</li>
                    <li>Username is your Brevo login email</li>
                    <li>Password is your SMTP key</li>
                </ol>
                <p style="margin: 0; color: #00a32a;"><strong>✓ Generous free tier:</strong> 300 emails/day forever</p>
            `
        },
        mailgun: {
            name: 'Mailgun',
            host: 'smtp.mailgun.org',
            port: 587,
            encryption: 'tls',
            username_placeholder: 'postmaster@mg.yourdomain.com',
            instructions: `
                <strong>Mailgun Setup:</strong>
                <ol style="margin: 10px 0; padding-left: 20px;">
                    <li>Sign up at <a href="https://www.mailgun.com/" target="_blank">Mailgun.com</a></li>
                    <li>Add and verify your domain</li>
                    <li>Go to Sending → Domain settings → SMTP credentials</li>
                    <li>Copy your SMTP login and password</li>
                </ol>
                <p style="margin: 0; color: #666;"><strong>Note:</strong> 5,000 free emails for first 3 months</p>
            `
        },
        outlook: {
            name: 'Outlook/Office365',
            host: 'smtp.office365.com',
            port: 587,
            encryption: 'tls',
            username_placeholder: 'your-email@outlook.com',
            instructions: `
                <strong>Outlook/Office365 Setup:</strong>
                <ol style="margin: 10px 0; padding-left: 20px;">
                    <li>Username is your full Outlook email address</li>
                    <li>Password is your Outlook password</li>
                    <li>If you have 2FA, create an App Password in Microsoft Account settings</li>
                </ol>
            `
        }
    };
    
    // Handle preset button clicks
    $('.smtp-preset-btn').on('click', function() {
        var preset = $(this).data('preset');
        var config = smtpPresets[preset];
        
        if (!config) return;
        
        // Fill in the form
        $('input[name="smtp_enabled"]').prop('checked', true);
        $('input[name="smtp_host"]').val(config.host);
        $('input[name="smtp_port"]').val(config.port);
        $('select[name="smtp_encryption"]').val(config.encryption);
        $('input[name="smtp_username"]').attr('placeholder', config.username_placeholder);
        
        // For SendGrid, auto-fill username
        if (preset === 'sendgrid') {
            $('input[name="smtp_username"]').val('apikey');
        }
        
        // Show instructions
        $('#smtp-preset-instructions').html(config.instructions).slideDown();
        
        // Highlight the selected button
        $('.smtp-preset-btn').removeClass('button-primary');
        $(this).addClass('button-primary');
        
        // Scroll to the form
        $('html, body').animate({
            scrollTop: $('input[name="smtp_host"]').offset().top - 100
        }, 500);
        
        // Focus on the username field (or password for sendgrid)
        setTimeout(function() {
            if (preset === 'sendgrid') {
                $('input[name="smtp_password"]').focus();
            } else {
                $('input[name="smtp_username"]').focus();
            }
        }, 600);
    });

    // Tab switching
    $('.tab-link').on('click', function(e) {
        e.preventDefault();
        var tab = $(this).data('tab');
        
        $('.tab-link').removeClass('active');
        $(this).addClass('active');
        
        $('.tab-content').hide();
        $('#' + tab + '-tab').show();
    });

    // Copy shortcode
    $(document).on('click', '.copy-shortcode', function() {
        var shortcode = $(this).data('shortcode');
        navigator.clipboard.writeText(shortcode).then(function() {
            alert('Shortcode copied to clipboard!');
        });
    });

    // Lead status change
    $('.lead-status-select').on('change', function() {
        var leadId = $(this).data('lead-id');
        var status = $(this).val();
        
        $.post(ajaxurl, {
            action: 'buildfuly_update_lead_status',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            lead_id: leadId,
            status: status
        });
    });

    // Delete lead
    $('.delete-lead').on('click', function() {
        if (!confirm('Are you sure you want to delete this lead?')) return;
        
        var leadId = $(this).data('lead-id');
        var $row = $(this).closest('tr');
        
        $.post(ajaxurl, {
            action: 'buildfuly_delete_lead',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            lead_id: leadId
        }, function() {
            $row.fadeOut(function() { $(this).remove(); });
        });
    });

    // View lead details
    $('.view-lead-details').on('click', function() {
        var lead = $(this).data('lead');
        var formData = {};
        try {
            formData = JSON.parse(lead.form_data || '{}');
        } catch(e) {}
        
        var html = '<table class="widefat">';
        html += '<tr><th>Name</th><td>' + (lead.name || '-') + '</td></tr>';
        html += '<tr><th>Email</th><td>' + (lead.email || '-') + '</td></tr>';
        html += '<tr><th>Phone</th><td>' + (lead.phone || '-') + '</td></tr>';
        html += '<tr><th>Source</th><td>' + (lead.source || '-') + '</td></tr>';
        html += '<tr><th>Estimate Total</th><td>' + (lead.estimate_total ? '$' + parseFloat(lead.estimate_total).toFixed(2) : '-') + '</td></tr>';
        html += '<tr><th>Estimate Details</th><td><pre>' + (lead.estimate_details || '-') + '</pre></td></tr>';
        html += '<tr><th>Status</th><td>' + lead.status + '</td></tr>';
        html += '<tr><th>Date</th><td>' + lead.created_at + '</td></tr>';
        
        if (Object.keys(formData).length > 0) {
            html += '<tr><th colspan="2"><strong>Additional Form Data</strong></th></tr>';
            for (var key in formData) {
                html += '<tr><th>' + key + '</th><td>' + formData[key] + '</td></tr>';
            }
        }
        
        html += '</table>';
        
        $('#lead-details-content').html(html);
        $('#lead-details-modal').show();
    });

    // Close modals
    $('.buildfuly-modal-close').on('click', function() {
        $(this).closest('.buildfuly-modal').hide();
    });
    
    $('.buildfuly-modal').on('click', function(e) {
        if ($(e.target).hasClass('buildfuly-modal')) {
            $(this).hide();
        }
    });

    // Email settings form
    $('#buildfuly-email-settings-form').on('submit', function(e) {
        e.preventDefault();
        
        var formData = {
            action: 'buildfuly_save_email_settings',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            send_to_admin: $('input[name="send_to_admin"]').is(':checked') ? 1 : 0,
            admin_email: $('input[name="admin_email"]').val(),
            send_to_lead: $('input[name="send_to_lead"]').is(':checked') ? 1 : 0,
            email_subject: $('input[name="email_subject"]').val(),
            email_template: $('textarea[name="email_template"]').val()
        };
        
        $.post(ajaxurl, formData, function(response) {
            alert('Email settings saved!');
        });
    });

    // SMTP settings form
    $('#buildfuly-smtp-settings-form').on('submit', function(e) {
        e.preventDefault();
        
        var formData = {
            action: 'buildfuly_save_smtp_settings',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            smtp_enabled: $('input[name="smtp_enabled"]').is(':checked') ? 1 : 0,
            smtp_host: $('input[name="smtp_host"]').val(),
            smtp_port: $('input[name="smtp_port"]').val(),
            smtp_encryption: $('select[name="smtp_encryption"]').val(),
            smtp_username: $('input[name="smtp_username"]').val(),
            smtp_password: $('input[name="smtp_password"]').val(),
            smtp_from_email: $('input[name="smtp_from_email"]').val(),
            smtp_from_name: $('input[name="smtp_from_name"]').val(),
            smtp_debug: $('input[name="smtp_debug"]').is(':checked') ? 1 : 0
        };
        
        $.post(ajaxurl, formData, function(response) {
            if (response.success) {
                alert('SMTP settings saved!');
            } else {
                alert('Error saving settings: ' + (response.data || 'Unknown error'));
            }
        });
    });

    // Test SMTP email
    $('#test-smtp-btn').on('click', function() {
        var btn = $(this);
        var originalText = btn.text();
        btn.text('Sending...').prop('disabled', true);
        
        $.post(ajaxurl, {
            action: 'buildfuly_test_smtp',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>'
        }, function(response) {
            btn.text(originalText).prop('disabled', false);
            if (response.success) {
                var msg = 'Test email sent successfully! Check your inbox at: ' + response.data.email;
                if (response.data.debug_log) {
                    msg += '\n\n--- SMTP Debug Log ---\n' + response.data.debug_log;
                    console.log('SMTP Debug Log:', response.data.debug_log);
                }
                alert(msg);
            } else {
                var errorMsg = response.data || 'Unknown error';
                console.log('SMTP Error:', errorMsg);
                alert('Failed to send test email: ' + errorMsg);
            }
        }).fail(function() {
            btn.text(originalText).prop('disabled', false);
            alert('Error sending test email');
        });
    });

    // Webhook settings form
    $('#buildfuly-webhook-settings-form').on('submit', function(e) {
        e.preventDefault();
        
        var formData = {
            action: 'buildfuly_save_webhook_settings',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            webhook_enabled: $('input[name="webhook_enabled"]').is(':checked') ? 1 : 0,
            webhook_url: $('input[name="webhook_url"]').val(),
            webhook_secret: $('input[name="webhook_secret"]').val(),
            webhook_headers: $('textarea[name="webhook_headers"]').val(),
            webhook_debug: $('input[name="webhook_debug"]').is(':checked') ? 1 : 0
        };
        
        $.post(ajaxurl, formData, function(response) {
            if (response.success) {
                alert('Webhook settings saved!');
            } else {
                alert('Error saving settings: ' + (response.data || 'Unknown error'));
            }
        });
    });

    // Test webhook
    $('#test-webhook-btn').on('click', function() {
        var btn = $(this);
        var originalText = btn.text();
        var webhookUrl = $('input[name="webhook_url"]').val();
        
        if (!webhookUrl) {
            alert('Please enter a webhook URL first');
            return;
        }
        
        btn.text('Sending...').prop('disabled', true);
        
        $.post(ajaxurl, {
            action: 'buildfuly_test_webhook',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            webhook_url: webhookUrl,
            webhook_secret: $('input[name="webhook_secret"]').val(),
            webhook_headers: $('textarea[name="webhook_headers"]').val()
        }, function(response) {
            btn.text(originalText).prop('disabled', false);
            if (response.success) {
                var msg = 'Test webhook sent successfully!\n\n';
                msg += 'Status: ' + response.data.status_code + '\n';
                if (response.data.response) {
                    msg += 'Response: ' + response.data.response;
                }
                alert(msg);
            } else {
                alert('Failed to send webhook: ' + (response.data || 'Unknown error'));
            }
        }).fail(function(xhr, status, error) {
            btn.text(originalText).prop('disabled', false);
            alert('Error sending webhook: ' + error);
        });
    });

    // ==========================================
    // FORM BUILDER
    // ==========================================
    
    var currentFormId = null;
    var formFields = [];
    var selectedFieldIndex = null;
    
    // Open form builder modal
    $('#create-new-form, #create-first-form').on('click', function() {
        currentFormId = null;
        formFields = [];
        selectedFieldIndex = null;
        $('#form-name').val('');
        $('#submit-button-text').val('Get My Estimate');
        $('#form-builder-title').text('Create New Form');
        renderFormPreview();
        renderFieldSettings();
        $('#form-builder-modal').show();
    });
    
    // Edit form
    $(document).on('click', '.edit-form', function() {
        var formId = $(this).data('form-id');
        
        // Fetch form data
        $.post(ajaxurl, {
            action: 'buildfuly_get_form',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            form_id: formId
        }, function(response) {
            if (response.success) {
                currentFormId = response.data.id;
                formFields = response.data.fields || [];
                var settings = response.data.settings || {};
                
                $('#form-name').val(response.data.name);
                $('#submit-button-text').val(settings.button_text || 'Get My Estimate');
                $('#success-message').val(settings.success_message || 'Thank you! Check your email for your estimate.');
                $('#form-builder-title').text('Edit Form');
                selectedFieldIndex = null;
                
                renderFormPreview();
                renderFieldSettings();
                $('#form-builder-modal').show();
            }
        });
    });
    
    // Cancel form builder
    $('#cancel-form-builder').on('click', function() {
        $('#form-builder-modal').hide();
    });
    
    // Add field
    $('.add-field-btn').on('click', function() {
        var type = $(this).data('type');
        var field = createFieldObject(type);
        formFields.push(field);
        selectedFieldIndex = formFields.length - 1;
        renderFormPreview();
        renderFieldSettings();
    });
    
    function createFieldObject(type) {
        var labels = {
            'text': 'Text Field',
            'email': 'Email Address',
            'phone': 'Phone Number',
            'number': 'Number',
            'textarea': 'Message',
            'select': 'Select Option',
            'checkbox': 'Checkbox',
            'radio': 'Select One',
            'date': 'Date',
            'hidden': 'Hidden Field',
            'html': 'Custom Calculator',
            'divider': 'Section'
        };
        
        var field = {
            type: type,
            label: labels[type] || 'Field',
            name: type + '_' + Date.now(),
            placeholder: '',
            required: false,
            options: (type === 'select' || type === 'radio') ? 'Option 1\nOption 2\nOption 3' : '',
            html_content: '',
            divider_style: 'heading'
        };
        
        // Special defaults
        if (type === 'email') {
            field.label = 'Email';
            field.name = 'email';
            field.placeholder = 'your@email.com';
            field.required = true;
        } else if (type === 'phone') {
            field.label = 'Phone';
            field.name = 'phone';
            field.placeholder = '(555) 123-4567';
        } else if (type === 'text') {
            field.label = 'Your Name';
            field.name = 'name';
            field.placeholder = 'John Smith';
        } else if (type === 'html') {
            field.label = 'Calculator';
            field.name = 'calculator_' + Date.now();
            field.html_content = '<!-- Paste your calculator HTML/CSS/JS here -->\n<div class="my-calculator">\n  <p>Your calculator will appear here</p>\n</div>';
        } else if (type === 'divider') {
            field.label = 'Your Information';
            field.divider_style = 'heading';
        }
        
        return field;
    }
    
    function renderFormPreview() {
        var $preview = $('#form-fields-preview');
        
        if (formFields.length === 0) {
            $preview.html('<div class="empty-form-message"><span class="dashicons dashicons-plus-alt2"></span><p>Click field types on the left to add them here</p></div>');
            return;
        }
        
        var html = '';
        formFields.forEach(function(field, index) {
            var selected = (index === selectedFieldIndex) ? ' selected' : '';
            html += '<div class="form-field-item' + selected + '" data-index="' + index + '">';
            html += '<div class="field-label">' + field.label;
            if (field.required) html += ' <span class="required-marker">*</span>';
            html += '</div>';
            html += '<div class="field-preview">' + getFieldPreviewHTML(field) + '</div>';
            html += '<div class="field-actions">';
            html += '<button type="button" class="move-field-up" title="Move Up"><span class="dashicons dashicons-arrow-up-alt2"></span></button>';
            html += '<button type="button" class="move-field-down" title="Move Down"><span class="dashicons dashicons-arrow-down-alt2"></span></button>';
            html += '<button type="button" class="delete-field" title="Delete"><span class="dashicons dashicons-trash"></span></button>';
            html += '</div>';
            html += '</div>';
        });
        
        $preview.html(html);
    }
    
    function getFieldPreviewHTML(field) {
        switch(field.type) {
            case 'textarea':
                return '<textarea placeholder="' + escapeAttr(field.placeholder) + '" disabled></textarea>';
            case 'select':
                var options = (field.options || '').split('\n');
                var opts = '<option value="">Select...</option>';
                options.forEach(function(opt) {
                    if (opt.trim()) opts += '<option>' + escapeHtml(opt.trim()) + '</option>';
                });
                return '<select disabled>' + opts + '</select>';
            case 'checkbox':
                return '<label><input type="checkbox" disabled> ' + escapeHtml(field.label) + '</label>';
            case 'radio':
                var options = (field.options || '').split('\n');
                var radios = '';
                options.forEach(function(opt, i) {
                    if (opt.trim()) {
                        radios += '<label style="display:block;margin:3px 0;"><input type="radio" name="preview_' + field.name + '" disabled> ' + escapeHtml(opt.trim()) + '</label>';
                    }
                });
                return radios;
            case 'date':
                return '<input type="date" disabled>';
            case 'hidden':
                return '<input type="text" value="' + escapeAttr(field.placeholder) + '" disabled style="background:#f0f0f1;font-style:italic;">';
            case 'html':
                // Show a collapsible live preview of the HTML
                var previewId = 'html-live-preview-' + field.name;
                return '<div class="html-preview-container">' +
                    '<div class="html-preview-header"><span class="dashicons dashicons-editor-code"></span> Custom HTML/Calculator <button type="button" class="toggle-html-preview button button-small">Preview</button></div>' +
                    '<div class="html-live-preview" id="' + previewId + '" style="display:none;"></div>' +
                    '</div>';
            case 'divider':
                if (field.divider_style === 'line') {
                    return '<hr style="border:none;border-top:1px solid #ddd;margin:10px 0;">';
                } else {
                    return '<div style="font-weight:600;font-size:14px;color:#1d2327;padding:5px 0;border-bottom:2px solid #2271b1;">' + escapeHtml(field.label) + '</div>';
                }
            default:
                return '<input type="' + field.type + '" placeholder="' + escapeAttr(field.placeholder) + '" disabled>';
        }
    }
    
    function renderFieldSettings() {
        var $settings = $('#field-settings-content');
        
        if (selectedFieldIndex === null || !formFields[selectedFieldIndex]) {
            $settings.html('<p class="description">Select a field to edit its settings</p>');
            return;
        }
        
        var field = formFields[selectedFieldIndex];
        
        var html = '';
        
        // HTML/Calculator field - special settings
        if (field.type === 'html') {
            html += '<div class="field-setting">';
            html += '<label>Section Label (optional)</label>';
            html += '<input type="text" id="field-label" value="' + escapeAttr(field.label) + '">';
            html += '</div>';
            
            html += '<div class="field-setting">';
            html += '<label>HTML/CSS/JavaScript Code</label>';
            html += '<textarea id="field-html-content" rows="10" style="font-family:monospace;font-size:12px;">' + escapeHtml(field.html_content || '') + '</textarea>';
            html += '</div>';
            
            html += '<div class="field-setting">';
            html += '<button type="button" class="button" id="preview-html-code"><span class="dashicons dashicons-visibility" style="vertical-align:middle;"></span> Preview Code</button>';
            html += '</div>';
            
            html += '<div class="field-setting" id="html-preview-container" style="display:none;">';
            html += '<label>Live Preview</label>';
            html += '<div id="html-code-preview" style="background:#fff;border:1px solid #c3c4c7;border-radius:4px;padding:15px;min-height:100px;max-height:300px;overflow:auto;"></div>';
            html += '</div>';
            
            html += '<div class="field-setting" style="background:#f0f7fc;padding:10px;border-radius:4px;margin-top:10px;">';
            html += '<strong style="font-size:11px;">💡 Tips:</strong>';
            html += '<ul style="font-size:11px;margin:5px 0 0;color:#50575e;list-style:disc;padding-left:20px;">';
            html += '<li><strong>Shortcodes:</strong> You can use WordPress shortcodes like <code>[your_shortcode]</code> - they will render properly!</li>';
            html += '<li><strong>Calculator JS:</strong> To capture the estimate, add to your calculator:</li>';
            html += '</ul>';
            html += '<code style="font-size:10px;display:block;margin-top:5px;background:#fff;padding:8px;border-radius:3px;word-break:break-all;">';
            html += 'document.querySelector(\'.buildfuly-estimate-total\').value = totalAmount;<br>';
            html += 'document.querySelector(\'.buildfuly-estimate-breakdown\').value = breakdownText;';
            html += '</code>';
            html += '</div>';
            
            $settings.html(html);
            return;
        }
        
        // Divider field - special settings
        if (field.type === 'divider') {
            html += '<div class="field-setting">';
            html += '<label>Heading Text</label>';
            html += '<input type="text" id="field-label" value="' + escapeAttr(field.label) + '">';
            html += '</div>';
            
            html += '<div class="field-setting">';
            html += '<label>Style</label>';
            html += '<select id="field-divider-style">';
            html += '<option value="heading" ' + (field.divider_style === 'heading' ? 'selected' : '') + '>Heading with underline</option>';
            html += '<option value="line" ' + (field.divider_style === 'line' ? 'selected' : '') + '>Simple line</option>';
            html += '</select>';
            html += '</div>';
            
            $settings.html(html);
            return;
        }
        
        // Standard fields
        html += '<div class="field-setting">';
        html += '<label>Label</label>';
        html += '<input type="text" id="field-label" value="' + escapeAttr(field.label) + '">';
        html += '</div>';
        
        html += '<div class="field-setting">';
        html += '<label>Field Name (for data)</label>';
        html += '<input type="text" id="field-name" value="' + escapeAttr(field.name) + '">';
        html += '</div>';
        
        if (field.type !== 'checkbox' && field.type !== 'radio' && field.type !== 'select') {
            html += '<div class="field-setting">';
            html += '<label>Placeholder</label>';
            html += '<input type="text" id="field-placeholder" value="' + escapeAttr(field.placeholder) + '">';
            html += '</div>';
        }
        
        if (field.type === 'select' || field.type === 'radio') {
            html += '<div class="field-setting">';
            html += '<label>Options (one per line)</label>';
            html += '<textarea id="field-options" rows="4">' + escapeHtml(field.options || '') + '</textarea>';
            html += '</div>';
        }
        
        html += '<div class="field-setting">';
        html += '<label><input type="checkbox" id="field-required" ' + (field.required ? 'checked' : '') + '> Required field</label>';
        html += '</div>';
        
        $settings.html(html);
    }
    
    // Select field
    $(document).on('click', '.form-field-item', function(e) {
        if ($(e.target).closest('.field-actions').length) return;
        
        selectedFieldIndex = parseInt($(this).data('index'));
        renderFormPreview();
        renderFieldSettings();
    });
    
    // Move field up
    $(document).on('click', '.move-field-up', function(e) {
        e.stopPropagation();
        var index = $(this).closest('.form-field-item').data('index');
        if (index > 0) {
            var temp = formFields[index];
            formFields[index] = formFields[index - 1];
            formFields[index - 1] = temp;
            if (selectedFieldIndex === index) selectedFieldIndex = index - 1;
            else if (selectedFieldIndex === index - 1) selectedFieldIndex = index;
            renderFormPreview();
        }
    });
    
    // Move field down
    $(document).on('click', '.move-field-down', function(e) {
        e.stopPropagation();
        var index = $(this).closest('.form-field-item').data('index');
        if (index < formFields.length - 1) {
            var temp = formFields[index];
            formFields[index] = formFields[index + 1];
            formFields[index + 1] = temp;
            if (selectedFieldIndex === index) selectedFieldIndex = index + 1;
            else if (selectedFieldIndex === index + 1) selectedFieldIndex = index;
            renderFormPreview();
        }
    });
    
    // Delete field
    $(document).on('click', '.delete-field', function(e) {
        e.stopPropagation();
        var index = $(this).closest('.form-field-item').data('index');
        formFields.splice(index, 1);
        if (selectedFieldIndex === index) selectedFieldIndex = null;
        else if (selectedFieldIndex > index) selectedFieldIndex--;
        renderFormPreview();
        renderFieldSettings();
    });
    
    // Update field settings live
    $(document).on('input', '#field-label', function() {
        if (selectedFieldIndex !== null && formFields[selectedFieldIndex]) {
            formFields[selectedFieldIndex].label = $(this).val();
            renderFormPreview();
        }
    });
    
    $(document).on('input', '#field-name', function() {
        if (selectedFieldIndex !== null && formFields[selectedFieldIndex]) {
            formFields[selectedFieldIndex].name = $(this).val();
        }
    });
    
    $(document).on('input', '#field-placeholder', function() {
        if (selectedFieldIndex !== null && formFields[selectedFieldIndex]) {
            formFields[selectedFieldIndex].placeholder = $(this).val();
            renderFormPreview();
        }
    });
    
    $(document).on('input', '#field-options', function() {
        if (selectedFieldIndex !== null && formFields[selectedFieldIndex]) {
            formFields[selectedFieldIndex].options = $(this).val();
            renderFormPreview();
        }
    });
    
    $(document).on('change', '#field-required', function() {
        if (selectedFieldIndex !== null && formFields[selectedFieldIndex]) {
            formFields[selectedFieldIndex].required = $(this).is(':checked');
            renderFormPreview();
        }
    });
    
    // Update HTML content for custom code fields
    $(document).on('input', '#field-html-content', function() {
        if (selectedFieldIndex !== null && formFields[selectedFieldIndex]) {
            formFields[selectedFieldIndex].html_content = $(this).val();
            // Update live preview in form builder if visible
            updateHtmlLivePreview();
        }
    });
    
    // Preview HTML code button - with shortcode support
    $(document).on('click', '#preview-html-code', function() {
        var $container = $('#html-preview-container');
        var $preview = $('#html-code-preview');
        var $btn = $(this);
        
        if ($container.is(':visible')) {
            $container.hide();
            $btn.html('<span class="dashicons dashicons-visibility" style="vertical-align:middle;"></span> Preview Code');
        } else {
            var htmlContent = $('#field-html-content').val();
            
            // Check if content contains shortcodes
            var hasShortcodes = /\[[\w_-]+/.test(htmlContent);
            
            if (hasShortcodes) {
                // Use server-side rendering for shortcodes
                $preview.html('<div style="text-align:center;padding:30px;color:#646970;"><span class="spinner is-active" style="float:none;"></span> Rendering shortcodes...</div>');
                $container.show();
                $btn.html('<span class="dashicons dashicons-hidden" style="vertical-align:middle;"></span> Hide Preview');
                
                $.post(ajaxurl, {
                    action: 'buildfuly_render_shortcode_preview',
                    nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
                    html_content: htmlContent
                }, function(response) {
                    console.log('Shortcode preview response:', response);
                    if (response.success) {
                        // Create iframe with rendered content
                        var iframe = document.createElement('iframe');
                        iframe.style.width = '100%';
                        iframe.style.minHeight = '150px';
                        iframe.style.border = 'none';
                        iframe.style.background = '#fff';
                        
                        $preview.empty().append(iframe);
                        
                        var iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                        iframeDoc.open();
                        iframeDoc.write('<!DOCTYPE html><html><head><style>body{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;margin:0;padding:15px;text-align:center;display:flex;flex-direction:column;align-items:center;justify-content:center;}</style></head><body>' + response.data.html + '</body></html>');
                        iframeDoc.close();
                        
                        setTimeout(function() {
                            try {
                                var height = iframeDoc.body.scrollHeight;
                                iframe.style.height = Math.min(Math.max(height + 20, 100), 500) + 'px';
                            } catch(e) {}
                        }, 200);
                    } else {
                        $preview.html('<div style="color:#d63638;padding:15px;">Error: ' + (response.data || 'Unknown error') + '</div>');
                    }
                }).fail(function(xhr, status, error) {
                    console.error('AJAX error:', status, error);
                    $preview.html('<div style="color:#d63638;padding:15px;">AJAX Error: ' + error + '</div>');
                });
            } else {
                // Regular HTML preview with iframe
                var iframe = document.createElement('iframe');
                iframe.style.width = '100%';
                iframe.style.minHeight = '150px';
                iframe.style.border = 'none';
                iframe.style.background = '#fff';
                
                $preview.empty().append(iframe);
                
                var iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                iframeDoc.open();
                iframeDoc.write('<!DOCTYPE html><html><head><style>body{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;margin:0;padding:15px;text-align:center;display:flex;flex-direction:column;align-items:center;}</style></head><body>' + htmlContent + '</body></html>');
                iframeDoc.close();
                
                setTimeout(function() {
                    try {
                        var height = iframeDoc.body.scrollHeight;
                        iframe.style.height = Math.min(Math.max(height + 20, 100), 400) + 'px';
                    } catch(e) {}
                }, 100);
                
                $container.show();
                $btn.html('<span class="dashicons dashicons-hidden" style="vertical-align:middle;"></span> Hide Preview');
            }
        }
    });
    
    // Toggle preview in form builder - with shortcode support
    $(document).on('click', '.toggle-html-preview', function(e) {
        e.stopPropagation();
        var $container = $(this).closest('.html-preview-container');
        var $preview = $container.find('.html-live-preview');
        var $btn = $(this);
        var index = $(this).closest('.form-field-item').data('index');
        
        if ($preview.is(':visible')) {
            $preview.hide();
            $btn.text('Preview');
        } else {
            var field = formFields[index];
            if (field && field.html_content) {
                var htmlContent = field.html_content;
                var hasShortcodes = /\[[\w_-]+/.test(htmlContent);
                
                if (hasShortcodes) {
                    // Server-side render for shortcodes
                    $preview.html('<div style="text-align:center;padding:20px;color:#646970;"><span class="spinner is-active" style="float:none;"></span></div>');
                    $preview.show();
                    $btn.text('Hide');
                    
                    $.post(ajaxurl, {
                        action: 'buildfuly_render_shortcode_preview',
                        nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
                        html_content: htmlContent
                    }, function(response) {
                        if (response.success) {
                            var iframe = document.createElement('iframe');
                            iframe.style.width = '100%';
                            iframe.style.minHeight = '100px';
                            iframe.style.border = 'none';
                            iframe.style.background = '#fff';
                            iframe.style.borderRadius = '4px';
                            
                            $preview.empty().append(iframe);
                            
                            var iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                            iframeDoc.open();
                            iframeDoc.write('<!DOCTYPE html><html><head><style>body{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;margin:0;padding:15px;font-size:14px;text-align:center;}</style></head><body>' + response.data.html + '</body></html>');
                            iframeDoc.close();
                            
                            setTimeout(function() {
                                try {
                                    var height = iframeDoc.body.scrollHeight;
                                    iframe.style.height = Math.min(Math.max(height + 20, 80), 350) + 'px';
                                } catch(e) {}
                            }, 200);
                        }
                    });
                } else {
                    // Regular HTML preview
                    var iframe = document.createElement('iframe');
                    iframe.style.width = '100%';
                    iframe.style.minHeight = '100px';
                    iframe.style.border = 'none';
                    iframe.style.background = '#fff';
                    iframe.style.borderRadius = '4px';
                    
                    $preview.empty().append(iframe);
                    
                    var iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                    iframeDoc.open();
                    iframeDoc.write('<!DOCTYPE html><html><head><style>body{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;margin:0;padding:15px;font-size:14px;text-align:center;}</style></head><body>' + htmlContent + '</body></html>');
                    iframeDoc.close();
                    
                    setTimeout(function() {
                        try {
                            var height = iframeDoc.body.scrollHeight;
                            iframe.style.height = Math.min(Math.max(height + 20, 80), 300) + 'px';
                        } catch(e) {}
                    }, 100);
                    
                    $preview.show();
                    $btn.text('Hide');
                }
            } else {
                $preview.html('<p style="color:#646970;text-align:center;padding:20px;">No code added yet. Click to edit and add your HTML/Calculator.</p>');
                $preview.show();
                $btn.text('Hide');
            }
        }
    });
    
    function updateHtmlLivePreview() {
        var $preview = $('#html-code-preview');
        if ($preview.is(':visible')) {
            $('#preview-html-code').click().click(); // Toggle to refresh
        }
    }
    
    // Update divider style
    $(document).on('change', '#field-divider-style', function() {
        if (selectedFieldIndex !== null && formFields[selectedFieldIndex]) {
            formFields[selectedFieldIndex].divider_style = $(this).val();
            renderFormPreview();
        }
    });
    
    // Save form
    $('#save-form').on('click', function() {
        var formName = $('#form-name').val().trim();
        if (!formName) {
            alert('Please enter a form name');
            return;
        }
        
        if (formFields.length === 0) {
            alert('Please add at least one field to your form');
            return;
        }
        
        var settings = {
            button_text: $('#submit-button-text').val(),
            success_message: $('#success-message').val()
        };
        
        $.post(ajaxurl, {
            action: 'buildfuly_save_form',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            form_id: currentFormId,
            name: formName,
            fields: JSON.stringify(formFields),
            settings: JSON.stringify(settings)
        }, function(response) {
            if (response.success) {
                alert('Form saved! Shortcode: [buildfuly_form id="' + response.data.id + '"]');
                location.reload();
            } else {
                alert('Error saving form: ' + (response.data || 'Unknown error'));
            }
        });
    });
    
    // Preview form
    $(document).on('click', '.preview-form', function() {
        var formId = $(this).data('form-id');
        
        $.post(ajaxurl, {
            action: 'buildfuly_preview_form',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            form_id: formId
        }, function(response) {
            if (response.success) {
                $('#form-preview-content').html(response.data.html);
                $('#form-preview-modal').show();
            }
        });
    });
    
    // Delete form
    $(document).on('click', '.delete-form', function() {
        if (!confirm('Are you sure you want to delete this form?')) return;
        
        var formId = $(this).data('form-id');
        var $card = $(this).closest('.form-card');
        
        $.post(ajaxurl, {
            action: 'buildfuly_delete_form',
            nonce: '<?php echo wp_create_nonce('buildfuly_leads_nonce'); ?>',
            form_id: formId
        }, function(response) {
            if (response.success) {
                $card.fadeOut(function() { $(this).remove(); });
            }
        });
    });
    
    // Helper functions
    function escapeHtml(text) {
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function escapeAttr(text) {
        return (text || '').replace(/"/g, '&quot;').replace(/'/g, '&#39;');
    }
});
</script>
