<?php
/**
 * Image Gallery Admin Page
 * AI-powered image generation with Brave search, Gemini analysis, and DALL-E modification
 *
 * @link       https://build.fuly.io
 * @since      1.1.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/admin/partials
 */

// Prevent direct access
if ( ! defined( 'WPINC' ) ) {
	die;
}

// Get business settings for image generation context
$business_name = get_option( 'buildfuly_business_name', '' );
$business_description = get_option( 'buildfuly_business_description', '' );
$location = get_option( 'buildfuly_location', '' );
$logo_id = get_option( 'buildfuly_business_logo_id', '' );
$logo_url = $logo_id ? wp_get_attachment_url( $logo_id ) : '';

// Get theme colors
$primary_color = get_option( 'buildfuly_primary_color', '#667eea' );
$secondary_color = get_option( 'buildfuly_secondary_color', '#764ba2' );
$accent_color = get_option( 'buildfuly_accent_color', '#f59e0b' );

// Query Buildfuly-generated images from media library
$gallery_query = new WP_Query( array(
	'post_type'      => 'attachment',
	'post_status'    => 'inherit',
	'post_mime_type' => 'image',
	'posts_per_page' => 50,
	'meta_query'     => array(
		'relation' => 'OR',
		array(
			'key'     => '_buildfuly_generated',
			'value'   => '1',
			'compare' => '=',
		),
		array(
			'key'     => '_buildfuly_ai_generated',
			'value'   => '1',
			'compare' => '=',
		),
	),
	'orderby'        => 'date',
	'order'          => 'DESC',
) );
$generated_images = $gallery_query->posts;

// Auto-generate preference
$auto_generate = get_option( 'buildfuly_auto_generate_images', false );
?>

<div class="wrap buildfuly-admin-wrap">
	<div class="buildfuly-page-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 24px;">
		<div>
			<h1 style="display: flex; align-items: center; gap: 12px; margin: 0;">
				<span class="dashicons dashicons-format-image" style="font-size: 32px; width: 32px; height: 32px; color: #667eea;"></span>
				<?php esc_html_e( 'AI Image Gallery', 'buildfuly' ); ?>
			</h1>
			<p style="color: #6b7280; margin: 8px 0 0 0;">
				<?php esc_html_e( 'Generate unique, branded images using AI. Images are customized with your brand colors and can include your logo.', 'buildfuly' ); ?>
			</p>
		</div>
		<div style="display: flex; gap: 12px;">
			<button type="button" class="button button-primary" id="buildfuly-generate-image-btn">
				<span class="dashicons dashicons-plus-alt" style="vertical-align: middle;"></span>
				<?php esc_html_e( 'Generate New Image', 'buildfuly' ); ?>
			</button>
		</div>
	</div>

	<!-- Settings Card -->
	<div class="buildfuly-card" style="margin-bottom: 24px;">
		<h3 style="margin-top: 0;"><?php esc_html_e( '⚙️ Image Generation Settings', 'buildfuly' ); ?></h3>
		
		<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 24px;">
			<!-- Brand Context -->
			<div>
				<h4 style="margin: 0 0 12px 0; color: #374151;"><?php esc_html_e( 'Brand Context', 'buildfuly' ); ?></h4>
				<table class="form-table" style="margin: 0;">
					<tr>
						<th style="width: 120px; padding: 8px 0;">
							<label><?php esc_html_e( 'Business', 'buildfuly' ); ?></label>
						</th>
						<td style="padding: 8px 0;">
							<strong><?php echo esc_html( $business_name ?: __( 'Not set', 'buildfuly' ) ); ?></strong>
							<?php if ( $location ) : ?>
								<span style="color: #6b7280;"> - <?php echo esc_html( $location ); ?></span>
							<?php endif; ?>
						</td>
					</tr>
					<tr>
						<th style="padding: 8px 0;">
							<label><?php esc_html_e( 'Logo', 'buildfuly' ); ?></label>
						</th>
						<td style="padding: 8px 0;">
							<?php if ( $logo_url ) : ?>
								<img src="<?php echo esc_url( $logo_url ); ?>" style="max-height: 40px; border-radius: 4px;">
							<?php else : ?>
								<span style="color: #9ca3af;"><?php esc_html_e( 'No logo uploaded', 'buildfuly' ); ?></span>
								<a href="<?php echo esc_url( admin_url( 'admin.php?page=buildfuly-settings' ) ); ?>" style="margin-left: 8px;">
									<?php esc_html_e( 'Add in Settings', 'buildfuly' ); ?>
								</a>
							<?php endif; ?>
						</td>
					</tr>
				</table>
			</div>
			
			<!-- Brand Colors -->
			<div>
				<h4 style="margin: 0 0 12px 0; color: #374151;"><?php esc_html_e( 'Brand Colors (used in images)', 'buildfuly' ); ?></h4>
				<div style="display: flex; gap: 16px; align-items: center;">
					<div style="text-align: center;">
						<div style="width: 48px; height: 48px; border-radius: 8px; background: <?php echo esc_attr( $primary_color ); ?>; border: 2px solid #e5e7eb;"></div>
						<span style="font-size: 11px; color: #6b7280;"><?php esc_html_e( 'Primary', 'buildfuly' ); ?></span>
					</div>
					<div style="text-align: center;">
						<div style="width: 48px; height: 48px; border-radius: 8px; background: <?php echo esc_attr( $secondary_color ); ?>; border: 2px solid #e5e7eb;"></div>
						<span style="font-size: 11px; color: #6b7280;"><?php esc_html_e( 'Secondary', 'buildfuly' ); ?></span>
					</div>
					<div style="text-align: center;">
						<div style="width: 48px; height: 48px; border-radius: 8px; background: <?php echo esc_attr( $accent_color ); ?>; border: 2px solid #e5e7eb;"></div>
						<span style="font-size: 11px; color: #6b7280;"><?php esc_html_e( 'Accent', 'buildfuly' ); ?></span>
					</div>
				</div>
			</div>
		</div>
		
		<!-- Auto-generate Option -->
		<div style="margin-top: 20px; padding: 16px; background: #f0fdf4; border-radius: 8px; border: 1px solid #bbf7d0;">
			<label style="display: flex; align-items: center; gap: 12px; cursor: pointer;">
				<input type="checkbox" id="buildfuly-auto-generate-images" <?php checked( $auto_generate ); ?>>
				<div>
					<strong style="color: #15803d;"><?php esc_html_e( 'Auto-generate images for new pages and blogs', 'buildfuly' ); ?></strong>
					<p style="margin: 4px 0 0 0; font-size: 13px; color: #166534;">
						<?php esc_html_e( 'When enabled, AI will automatically generate relevant images when you create new pages or blog posts.', 'buildfuly' ); ?>
					</p>
				</div>
			</label>
		</div>
	</div>

	<!-- Note: The AI Image Generator modal is now provided by the reusable BuildfulyImageGenerator module -->

	<!-- Image Gallery Grid -->
	<div class="buildfuly-card">
		<h3 style="margin-top: 0; display: flex; align-items: center; justify-content: space-between;">
			<?php esc_html_e( '🖼️ Generated Images', 'buildfuly' ); ?>
			<span style="font-size: 14px; font-weight: normal; color: #6b7280;">
				<?php 
				/* translators: %d: Number of images */
				printf( esc_html__( '%d images', 'buildfuly' ), count( $generated_images ) ); 
				?>
			</span>
		</h3>
		
		<?php if ( empty( $generated_images ) ) : ?>
			<div style="text-align: center; padding: 60px 20px; color: #9ca3af;">
				<span class="dashicons dashicons-format-image" style="font-size: 64px; width: 64px; height: 64px; margin-bottom: 16px; opacity: 0.3;"></span>
				<h3 style="color: #6b7280; margin: 0 0 8px 0;"><?php esc_html_e( 'No images generated yet', 'buildfuly' ); ?></h3>
				<p style="margin: 0 0 20px 0;"><?php esc_html_e( 'Click "Generate New Image" to create your first AI-powered branded image.', 'buildfuly' ); ?></p>
				<button type="button" class="button button-primary" onclick="document.getElementById('buildfuly-generate-image-btn').click();">
					<?php esc_html_e( 'Generate Your First Image', 'buildfuly' ); ?>
				</button>
			</div>
		<?php else : ?>
			<div id="buildfuly-images-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)); gap: 20px;">
				<?php foreach ( $generated_images as $attachment ) : 
					$image_url = wp_get_attachment_url( $attachment->ID );
					$image_thumb = wp_get_attachment_image_src( $attachment->ID, 'medium' );
					$image_alt = get_post_meta( $attachment->ID, '_wp_attachment_image_alt', true );
					$is_ai = get_post_meta( $attachment->ID, '_buildfuly_ai_generated', true ) === '1';
				?>
					<div class="buildfuly-image-card" data-id="<?php echo esc_attr( $attachment->ID ); ?>" style="border: 1px solid #e5e7eb; border-radius: 12px; overflow: hidden; background: white;">
						<div style="position: relative; padding-top: 75%; background: #f3f4f6;">
							<img src="<?php echo esc_url( $image_thumb[0] ?? $image_url ); ?>" 
								 alt="<?php echo esc_attr( $image_alt ); ?>"
								 style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover;">
							<?php if ( $is_ai ) : ?>
								<span style="position: absolute; top: 8px; left: 8px; background: linear-gradient(135deg, #667eea, #764ba2); color: white; padding: 4px 8px; border-radius: 4px; font-size: 10px; font-weight: 600;">
									AI Generated
								</span>
							<?php endif; ?>
							<div class="image-overlay" style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.5); opacity: 0; transition: opacity 0.2s; display: flex; align-items: center; justify-content: center; gap: 8px;">
								<button type="button" class="button view-image-btn" data-url="<?php echo esc_url( $image_url ); ?>" title="<?php esc_attr_e( 'View Full Size', 'buildfuly' ); ?>">
									<span class="dashicons dashicons-visibility"></span>
								</button>
								<button type="button" class="button copy-url-btn" data-url="<?php echo esc_url( $image_url ); ?>" title="<?php esc_attr_e( 'Copy URL', 'buildfuly' ); ?>">
									<span class="dashicons dashicons-admin-links"></span>
								</button>
								<button type="button" class="button delete-image-btn" data-id="<?php echo esc_attr( $attachment->ID ); ?>" title="<?php esc_attr_e( 'Delete', 'buildfuly' ); ?>" style="color: #dc2626;">
									<span class="dashicons dashicons-trash"></span>
								</button>
							</div>
						</div>
						<div style="padding: 12px;">
							<p style="margin: 0; font-size: 13px; color: #374151; line-height: 1.4;">
								<?php echo esc_html( $attachment->post_title ?: __( 'AI Generated Image', 'buildfuly' ) ); ?>
							</p>
							<p style="margin: 8px 0 0 0; font-size: 11px; color: #9ca3af;">
								<?php echo esc_html( human_time_diff( strtotime( $attachment->post_date ), current_time( 'timestamp' ) ) . ' ' . __( 'ago', 'buildfuly' ) ); ?>
							</p>
						</div>
					</div>
				<?php endforeach; ?>
			</div>
		<?php endif; ?>
	</div>
</div>

<style>
.buildfuly-image-card:hover .image-overlay {
	opacity: 1 !important;
}

.gen-step {
	padding: 8px 0;
	color: #9ca3af;
	display: flex;
	align-items: center;
	gap: 8px;
}

.gen-step.active {
	color: #667eea;
	font-weight: 600;
}

.gen-step.complete {
	color: #10b981;
}

.gen-step .dashicons {
	font-size: 18px;
	width: 18px;
	height: 18px;
}

.buildfuly-spinner {
	width: 48px;
	height: 48px;
	border: 4px solid #e5e7eb;
	border-top-color: #667eea;
	border-radius: 50%;
	animation: spin 1s linear infinite;
}

@keyframes spin {
	to { transform: rotate(360deg); }
}

input[name="image-target"]:checked + span + span,
input[name="image-target"]:checked + span {
	color: #667eea;
}

label:has(input[name="image-target"]:checked) {
	border-color: #667eea !important;
	background: #f0f4ff;
}
</style>


<script>
jQuery(document).ready(function($) {
	// Use the shared BuildfulyImageGenerator module for generating images
	// This ensures the same code is used here and in the section editor
	$('#buildfuly-generate-image-btn').on('click', function() {
		if (typeof BuildfulyImageGenerator !== 'undefined') {
			BuildfulyImageGenerator.open({
				topic: '',
				onSelect: function(imageData) {
					// Image has been saved to media library, reload to show in gallery
					window.location.reload();
				}
			});
		} else {
			alert('<?php esc_html_e( 'Image generator module not loaded. Please refresh the page.', 'buildfuly' ); ?>');
		}
	});
	
	// Image card actions
	$('.view-image-btn').on('click', function() {
		window.open($(this).data('url'), '_blank');
	});
	
	$('.copy-url-btn').on('click', function() {
		navigator.clipboard.writeText($(this).data('url'));
		alert('<?php esc_html_e( 'URL copied to clipboard!', 'buildfuly' ); ?>');
	});
	
	$('.delete-image-btn').on('click', function() {
		if (!confirm('<?php esc_html_e( 'Delete this image from gallery?', 'buildfuly' ); ?>')) return;
		
		const $card = $(this).closest('.buildfuly-image-card');
		const attachmentId = $(this).data('id');
		
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_delete_gallery_image',
				nonce: buildfulyAdmin.nonce,
				attachment_id: attachmentId
			},
			success: function(response) {
				if (response.success) {
					$card.fadeOut(300, function() { $(this).remove(); });
				}
			}
		});
	});
	
	// Auto-generate preference
	$('#buildfuly-auto-generate-images').on('change', function() {
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_save_auto_generate_images',
				nonce: buildfulyAdmin.nonce,
				enabled: $(this).is(':checked') ? 1 : 0
			}
		});
	});
});
</script>

