/**
 * Buildfuly Shared Utilities
 * Common functions used across all Buildfuly modules
 * Follows DRY principle - Don't Repeat Yourself
 */

(function($) {
	'use strict';

	window.BuildfulyUtils = {
		/**
		 * Escape HTML entities to prevent XSS
		 * @param {string} text - Text to escape
		 * @returns {string} Escaped text
		 */
		escapeHtml: function(text) {
			if (!text) return '';
			const div = document.createElement('div');
			div.textContent = text;
			return div.innerHTML;
		},

		/**
		 * Unescape HTML entities (e.g., &lt; becomes <)
		 * IMPORTANT: This preserves HTML structure, only decodes entities
		 * @param {string} text - Text to unescape
		 * @returns {string} Unescaped text with HTML preserved
		 */
		unescapeHtml: function(text) {
			if (!text) return '';
			// Use textarea to decode HTML entities while preserving HTML structure
			const textarea = document.createElement('textarea');
			textarea.innerHTML = text;
			return textarea.value;
		},

		/**
		 * Show a toast notification message
		 * @param {string} message - Message to display
		 * @param {string} type - Message type: 'success', 'error', 'warning', 'info'
		 * @param {number} duration - Duration in ms (default: 4000)
		 */
		showMessage: function(message, type = 'info', duration = 4000) {
			const $container = $('#buildfuly-messages');
			if (!$container.length) return;

			const icons = {
				success: 'yes-alt',
				error: 'dismiss',
				warning: 'warning',
				info: 'info'
			};

			const $message = $(`
				<div class="buildfuly-message buildfuly-message-${type}" style="animation: slideInRight 0.3s ease;">
					<span class="dashicons dashicons-${icons[type] || 'info'}"></span>
					<span>${this.escapeHtml(message)}</span>
				</div>
			`);

			$container.append($message);

			setTimeout(() => {
				$message.fadeOut(300, function() {
					$(this).remove();
				});
			}, duration);
		},

		/**
		 * Show loading overlay
		 * @param {string} message - Loading message
		 */
		showLoading: function(message = 'Loading...') {
			$('#buildfuly-loading p').text(message);
			$('#buildfuly-loading').fadeIn();
		},

		/**
		 * Hide loading overlay
		 */
		hideLoading: function() {
			$('#buildfuly-loading').fadeOut();
		},

		/**
		 * Open WordPress media library
		 * @param {object} options - Options for media library
		 * @param {string} options.title - Modal title (default: 'Select Image')
		 * @param {string} options.buttonText - Button text (default: 'Use Image')
		 * @param {boolean} options.multiple - Allow multiple selection (default: false)
		 * @param {function} options.onSelect - Callback when image selected, receives attachment object
		 */
		openMediaLibrary: function(options = {}) {
			if (typeof wp === 'undefined' || !wp.media) {
				this.showMessage('Media library not available', 'error');
				return;
			}

			const defaults = {
				title: 'Select Image',
				buttonText: 'Use Image',
				multiple: false,
				onSelect: null
			};

			const settings = { ...defaults, ...options };

			const frame = wp.media({
				title: settings.title,
				button: { text: settings.buttonText },
				multiple: settings.multiple
			});

			frame.on('select', () => {
				const selection = frame.state().get('selection');
				if (settings.multiple) {
					const attachments = selection.map(att => att.toJSON());
					if (settings.onSelect) settings.onSelect(attachments);
				} else {
					const attachment = selection.first().toJSON();
					if (settings.onSelect) settings.onSelect(attachment);
				}
			});

			frame.open();
			return frame;
		},

		/**
		 * Get business context from settings
		 * @returns {object} Business context object
		 */
		getBusinessContext: function() {
			if (typeof buildfulyAdmin === 'undefined') {
				return { name: '', description: '', location: '' };
			}
			return {
				name: buildfulyAdmin.businessName || '',
				description: buildfulyAdmin.businessDescription || '',
				location: buildfulyAdmin.location || '',
				toString: function() {
					let str = this.name ? `${this.name}: ` : '';
					str += this.description || '';
					if (this.location) str += ` (Location: ${this.location})`;
					return str.trim();
				}
			};
		},

		/**
		 * Get API base URL
		 * @returns {string} API URL
		 */
		getApiUrl: function() {
			if (typeof buildfulyAdmin !== 'undefined' && buildfulyAdmin.apiUrl) {
				return buildfulyAdmin.apiUrl;
			}
			return 'https://build.fuly.io';
		},

		/**
		 * Make authenticated API request
		 * @param {string} endpoint - API endpoint (e.g., '/api/generate')
		 * @param {object} data - Request data
		 * @returns {Promise<object>} Response data
		 */
		apiRequest: async function(endpoint, data = {}) {
			const apiUrl = this.getApiUrl();
			const licenseKey = typeof buildfulyAdmin !== 'undefined' ? buildfulyAdmin.licenseKey : '';

			const response = await fetch(`${apiUrl}${endpoint}`, {
				method: 'POST',
				headers: {
					'Content-Type': 'application/json',
					'X-License-Key': licenseKey
				},
				body: JSON.stringify(data)
			});

			if (!response.ok) {
				const error = await response.json().catch(() => ({ message: 'Request failed' }));
				throw new Error(error.message || `HTTP ${response.status}`);
			}

			return response.json();
		},

		/**
		 * Generate unique ID
		 * @param {string} prefix - Optional prefix
		 * @returns {string} Unique ID
		 */
		generateId: function(prefix = '') {
			return prefix + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
		},

		/**
		 * Debounce function calls
		 * @param {function} func - Function to debounce
		 * @param {number} wait - Wait time in ms
		 * @returns {function} Debounced function
		 */
		debounce: function(func, wait = 300) {
			let timeout;
			return function executedFunction(...args) {
				const later = () => {
					clearTimeout(timeout);
					func(...args);
				};
				clearTimeout(timeout);
				timeout = setTimeout(later, wait);
			};
		},

		/**
		 * Debug logging (only in debug mode)
		 * @param {string} module - Module name
		 * @param {...any} args - Arguments to log
		 */
		debug: function(module, ...args) {
			if (typeof buildfulyAdmin !== 'undefined' && buildfulyAdmin.debug) {
				console.log(`[Buildfuly ${module}]`, ...args);
			}
		}
	};

})(jQuery);
