/**
 * Buildfuly Admin - Simplified Template System
 */

(function($) {
	'use strict';

	const BuildfulySimple = {
		currentTemplate: null,
		currentDevice: 'desktop',

		init: function() {
			this.bindEvents();
		},

		bindEvents: function() {
			// Tab switching
			$('.buildfuly-tab-btn').on('click', this.switchTab.bind(this));
			
			// Load template
			$('#buildfuly-load-template').on('click', this.loadTemplate.bind(this));
			
			// Device switching
			$('.buildfuly-device-btn').on('click', this.switchDevice.bind(this));
			
			// Color swatch clicking
			$('.buildfuly-color-swatch').on('click', this.changeColor.bind(this));
			
			// Regenerate button
			$('#buildfuly-regenerate').on('click', this.regenerateContent.bind(this));
			
			// Publish button
			$('#buildfuly-publish-content').on('click', this.publishContent.bind(this));
		},

		switchTab: function(e) {
			e.preventDefault();
			const tab = $(e.currentTarget).data('tab');
			
			// Update tab buttons
			$('.buildfuly-tab-btn').removeClass('active');
			$(e.currentTarget).addClass('active');
			
			// Update tab content
			$('.buildfuly-tab-content').removeClass('active');
			$('#buildfuly-tab-' + tab).addClass('active');
		},

		loadTemplate: function(e) {
			e.preventDefault();
			
			const templateType = $('#template-select').val();
			
			if (!templateType) {
				this.showMessage('Please select a template first', 'warning');
				return;
			}
			
			if (!window.BuildfulyTemplates || !window.BuildfulyTemplates[templateType]) {
				this.showMessage('Template not found', 'error');
				return;
			}
			
			this.currentTemplate = window.BuildfulyTemplates[templateType];
			this.displayTemplate();
			this.showMessage('Template loaded! You can now customize it.', 'success');
			
			// Show floating action buttons
			$('.buildfuly-floating-actions').fadeIn();
		},

		displayTemplate: function() {
			const preview = $('#buildfuly-preview');
			
			// Build the content HTML
			const html = `
				${this.currentTemplate.hero}
				${this.currentTemplate.features}
			`;
			
			preview.html(html);
		},

		regenerateContent: function(e) {
			e.preventDefault();
			
			if (!this.currentTemplate) {
				this.showMessage('Please load a template first', 'warning');
				return;
			}
			
			// Show loading
			$('#buildfuly-loading').fadeIn();
			
			// Simulate regeneration (in real app, this would call API)
			setTimeout(() => {
				this.displayTemplate();
				$('#buildfuly-loading').fadeOut();
				this.showMessage('Content regenerated!', 'success');
			}, 1000);
		},

		publishContent: function(e) {
			e.preventDefault();
			
			if (!this.currentTemplate) {
				this.showMessage('Please load a template first', 'warning');
				return;
			}
			
			const content = $('#buildfuly-preview').html();
			
			// Show loading
			$('#buildfuly-loading').fadeIn();
			
			// Save to WordPress (this would be a real API call)
			$.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'buildfuly_publish_content',
					nonce: $('#buildfuly-nonce').val(),
					content: content
				},
				success: (response) => {
					$('#buildfuly-loading').fadeOut();
					if (response.success) {
						this.showMessage('Content published to your website!', 'success');
					} else {
						this.showMessage('Failed to publish content', 'error');
					}
				},
				error: () => {
					$('#buildfuly-loading').fadeOut();
					this.showMessage('Error publishing content', 'error');
				}
			});
		},

		switchDevice: function(e) {
			e.preventDefault();
			
			const device = $(e.currentTarget).data('device');
			this.currentDevice = device;
			
			// Update button states
			$('.buildfuly-device-btn').removeClass('active');
			$(e.currentTarget).addClass('active');
			
			// Update preview
			const preview = $('#buildfuly-preview');
			preview.removeClass('device-desktop device-tablet device-mobile');
			preview.addClass('device-' + device);
		},

		changeColor: function(e) {
			const $swatch = $(e.currentTarget);
			const colorType = $swatch.data('color');
			const currentColor = $swatch.css('background-color');
			
			// In a real implementation, this would open a color picker
			this.showMessage(`Color picker for ${colorType} (coming soon!)`, 'info');
		},

		showMessage: function(message, type = 'info', duration = 3000) {
			const messagesContainer = $('#buildfuly-messages');
			
			if (!messagesContainer.length) {
				$('body').append('<div id="buildfuly-messages"></div>');
			}
			
			const messageHtml = `
				<div class="buildfuly-message buildfuly-message-${type}">
					${message}
				</div>
			`;
			
			const $message = $(messageHtml);
			$('#buildfuly-messages').append($message);
			
			setTimeout(() => {
				$message.fadeOut(300, function() {
					$(this).remove();
				});
			}, duration);
		},

		showLoading: function() {
			$('#buildfuly-loading').fadeIn();
		},

		hideLoading: function() {
			$('#buildfuly-loading').fadeOut();
		}
	};

	// Initialize when DOM is ready
	$(document).ready(function() {
		BuildfulySimple.init();
	});

})(jQuery);
