/**
 * Buildfuly SEO Metadata Management
 * Handles the Yoast-style SEO metadata UI
 * 
 * @since 1.0.4
 */

(function($) {
	'use strict';

	let currentPageId = null;

	/**
	 * Initialize SEO metadata handlers
	 */
	function init() {
		// Modal controls
		$('#buildfuly-open-seo-modal').on('click', openModal);
		$('#buildfuly-close-seo-modal, #buildfuly-cancel-seo').on('click', closeModal);
		$('.buildfuly-modal').on('click', function(e) {
			if ($(e.target).hasClass('buildfuly-modal')) {
				closeModal();
			}
		});

		// Character counters
		$('#buildfuly-seo-title').on('input', updateTitleCounter);
		$('#buildfuly-seo-description').on('input', updateDescCounter);

		// Live Google preview updates
		$('#buildfuly-seo-slug, #buildfuly-seo-title, #buildfuly-seo-description').on('input', updateGooglePreview);

		// Save button
		$('#buildfuly-save-seo').on('click', saveSeoMetadata);

		// Listen for page selection changes
		$(document).on('buildfuly:pageSelected', function(e, data) {
			console.log('Page selected event received:', data);
			if (data && typeof data === 'object') {
				currentPageId = data.pageId || data.wordpressId;
				loadSeoMetadata(data.pageId, data.wordpressId);
			} else {
				// Legacy support for just pageId
				currentPageId = data;
				loadSeoMetadata(data, null);
			}
		});

		// Listen for page publish success
		$(document).on('buildfuly:pagePublished', function(e, data) {
			console.log('Page published event received:', data);
			if (data && data.page_id) {
				currentPageId = data.page_id;
				// Load SEO metadata after publishing - use wordpress_id for database lookup
				loadSeoMetadata(null, data.page_id);
			}
		});
	}

	/**
	 * Open SEO metadata modal
	 */
	function openModal() {
		// Get the current page from the page builder
		let pageId = null;
		let wordpressId = null;
		let pageTitle = '';
		
		if (window.BuildfulyPageBuilder && window.BuildfulyPageBuilder.currentPage) {
			const page = window.BuildfulyPageBuilder.currentPage;
			pageId = page.id;
			wordpressId = page.wordpress_id;
			pageTitle = page.title || page.name || '';
			console.log('Opening SEO modal for page:', { 
				pageId, 
				wordpressId, 
				pageTitle, 
				hasSeo: !!page.seo,
				fullPage: page 
			});
		}
		
		// Try to get wordpressId from the page item's data attribute as fallback
		if (!wordpressId && pageId) {
			const $pageItem = $(`#${pageId}`);
			if ($pageItem.length) {
				const dataWpId = $pageItem.data('wordpress-id');
				if (dataWpId) {
					wordpressId = parseInt(dataWpId);
					console.log('Got wordpress_id from data attribute:', wordpressId);
				}
			}
		}
		
		// Fallback to URL or hidden field
		if (!pageId) {
			const urlParams = new URLSearchParams(window.location.search);
			pageId = urlParams.get('page_id') || currentPageId || $('#buildfuly-seo-page-id').val();
		}
		
		if (!pageId) {
			alert('Please select a page first');
			return;
		}
		
		// Update modal title with page name
		$('#buildfuly-seo-page-name').text(pageTitle ? '- ' + pageTitle : '');
		
		// Show/hide warning based on publish status
		if (!wordpressId) {
			console.log('No wordpress_id found - showing not published warning');
			$('#buildfuly-seo-not-published').show();
			$('#buildfuly-save-seo').prop('disabled', true).css('opacity', '0.5');
		} else {
			console.log('wordpress_id found:', wordpressId, '- enabling save button');
			$('#buildfuly-seo-not-published').hide();
			$('#buildfuly-save-seo').prop('disabled', false).css('opacity', '1');
		}
		
		currentPageId = pageId;
		loadSeoMetadata(pageId, wordpressId);
		$('#buildfuly-seo-modal').fadeIn(200);
	}

	/**
	 * Close SEO metadata modal
	 */
	function closeModal() {
		$('#buildfuly-seo-modal').fadeOut(200);
	}

	/**
	 * Update title character counter
	 */
	function updateTitleCounter() {
		const length = $(this).val().length;
		const $counter = $('#buildfuly-seo-title-count');
		
		$counter.text(length + '/60');
		
		if (length > 60) {
			$counter.removeClass('good warning').addClass('error');
		} else if (length > 50) {
			$counter.removeClass('good error').addClass('warning');
		} else if (length > 0) {
			$counter.removeClass('warning error').addClass('good');
		} else {
			$counter.removeClass('good warning error');
		}
	}

	/**
	 * Update description character counter
	 */
	function updateDescCounter() {
		const length = $(this).val().length;
		const $counter = $('#buildfuly-seo-desc-count');
		
		$counter.text(length + '/160');
		
		if (length > 160) {
			$counter.removeClass('good warning').addClass('error');
		} else if (length > 150) {
			$counter.removeClass('good error').addClass('warning');
		} else if (length > 0) {
			$counter.removeClass('warning error').addClass('good');
		} else {
			$counter.removeClass('good warning error');
		}
	}

	/**
	 * Update Google preview with current values
	 */
	function updateGooglePreview() {
		const slug = $('#buildfuly-seo-slug').val() || 'page-slug';
		const title = $('#buildfuly-seo-title').val() || 'Page Title • Brand';
		const description = $('#buildfuly-seo-description').val() || 'Meta description will appear here...';
		const domain = window.location.hostname;

		let displaySlug = slug === '/' ? '' : slug;
		$('#buildfuly-google-preview-url').text(domain + '/' + displaySlug);
		$('#buildfuly-google-preview-title').text(title);
		$('#buildfuly-google-preview-desc').text(description);
	}

	/**
	 * Update SEO status indicators (for future use)
	 */
	function updateSeoStatus(type) {
		// Placeholder for future SEO status indicators
		// Could show green/yellow/red status based on title length, description length, etc.
	}

	/**
	 * Load SEO metadata for a page
	 * @param {string} pageId - Local page ID (e.g., "page-1234567890")
	 * @param {number} wordpressId - WordPress post ID (optional, for database lookup)
	 */
	function loadSeoMetadata(pageId, wordpressId) {
		// Need at least one ID to proceed
		if (!pageId && !wordpressId) {
			console.log('No page ID or wordpress ID provided');
			return;
		}

		currentPageId = pageId || wordpressId;
		
		console.log('Loading SEO metadata for page:', { pageId, wordpressId });
		
		// Set the page ID in hidden field
		$('#buildfuly-seo-page-id').val(wordpressId || pageId);

		// First, check if SEO data is already in the page object
		if (window.BuildfulyPageBuilder && window.BuildfulyPageBuilder.currentPage) {
			const page = window.BuildfulyPageBuilder.currentPage;
			console.log('Current page object:', { id: page.id, wordpress_id: page.wordpress_id, hasSeo: !!page.seo, seo: page.seo });
			
			// Verify it's the same page (match by local ID or wordpress_id)
			if ((page.id === pageId || page.wordpress_id === wordpressId) && page.seo) {
				console.log('✓ Using SEO from current page object:', page.seo);
				populateSeoForm(page.seo);
				return;
			}
			
			// Get wordpressId from page if not provided
			if (!wordpressId && page.id === pageId && page.wordpress_id) {
				wordpressId = page.wordpress_id;
			}
		}

		// If page has SEO data in memory, use it
		if (window.BuildfulyPageBuilder && window.BuildfulyPageBuilder.pages) {
			const foundPage = window.BuildfulyPageBuilder.pages.find(p => p.id === pageId);
			if (foundPage) {
				console.log('Found page in array:', { id: foundPage.id, wordpress_id: foundPage.wordpress_id, hasSeo: !!foundPage.seo });
				if (foundPage.seo) {
					console.log('✓ Using SEO from found page:', foundPage.seo);
					populateSeoForm(foundPage.seo);
					return;
				}
				// Get wordpressId for database lookup
				if (!wordpressId && foundPage.wordpress_id) {
					wordpressId = foundPage.wordpress_id;
				}
			}
		}

		// If no wordpressId, this page hasn't been published yet - generate defaults
		if (!wordpressId) {
			console.log('Page not published yet, generating defaults');
			generateDefaultSeoData(pageId);
			return;
		}

		// Load from database using wordpress_id
		console.log('Attempting to load from database with wordpressId:', wordpressId);
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_get_seo_metadata',
				nonce: buildfulyAdmin.adminNonce,
				page_id: wordpressId
			},
			success: function(response) {
				console.log('SEO metadata response:', response);
				if (response.success && response.data && (response.data.slug || response.data.meta_title)) {
					console.log('✓ Using SEO from database:', response.data);
					populateSeoForm(response.data);
				} else {
					// For pages with no SEO in database, generate defaults
					console.log('No database SEO found, generating defaults');
					generateDefaultSeoData(pageId);
				}
			},
			error: function(xhr, status, error) {
				console.error('Error loading SEO metadata:', error);
				generateDefaultSeoData(pageId);
			}
		});
	}

	/**
	 * Generate default SEO data for unpublished pages
	 */
	function generateDefaultSeoData(pageId) {
		console.log('Generating default SEO for page:', pageId);
		console.log('BuildfulyPageBuilder available?', !!window.BuildfulyPageBuilder);
		
		// Try to find the page in the pages array
		let page = null;
		if (window.BuildfulyPageBuilder) {
			console.log('Current page:', window.BuildfulyPageBuilder.currentPage);
			console.log('Pages array:', window.BuildfulyPageBuilder.pages);
			
			if (window.BuildfulyPageBuilder.currentPage && window.BuildfulyPageBuilder.currentPage.id === pageId) {
				page = window.BuildfulyPageBuilder.currentPage;
				console.log('Using current page');
			} else if (window.BuildfulyPageBuilder.pages) {
				console.log('Searching pages array for ID:', pageId);
				page = window.BuildfulyPageBuilder.pages.find(p => {
					console.log('  Checking page:', p.id, 'vs', pageId);
					return p.id === pageId;
				});
				if (page) {
					console.log('Found page in array');
				}
			}
		}
		
		if (!page) {
			console.warn('Could not find page data for:', pageId);
			console.warn('Available BuildfulyPageBuilder:', window.BuildfulyPageBuilder);
			
			// Try to get from localStorage as fallback
			try {
				const savedPages = localStorage.getItem('buildfuly_pages');
				if (savedPages) {
					const pages = JSON.parse(savedPages);
					page = pages.find(p => p.id === pageId);
					if (page) {
						console.log('Found page in localStorage:', page);
					}
				}
			} catch (e) {
				console.error('Error reading from localStorage:', e);
			}
			
			if (!page) {
				clearSeoForm();
				return;
			}
		}
		
		console.log('Found page data:', { title: page.title, keywords: page.keywords, seo: page.seo });
		
		// If page already has SEO, use it
		if (page.seo) {
			console.log('Page already has SEO data, using it:', page.seo);
			populateSeoForm(page.seo);
			return;
		}
		
		const pageTitle = page.title || page.name || 'Untitled Page';
		const keywords = page.keywords || [];
		
		// Generate slug from title
		let slug = pageTitle.toLowerCase()
			.replace(/[^a-z0-9]+/g, '-')
			.replace(/^-+|-+$/g, '');
		
		// Check if homepage
		if (pageTitle.toLowerCase() === 'home') {
			slug = '/';
		}
		
		// Don't add brand to meta_title - WordPress/Yoast adds it automatically
		const metaTitle = pageTitle;
		
		// Generate meta description from keywords
		let metaDesc = '';
		const keywordList = Array.isArray(keywords) 
			? keywords.map(k => typeof k === 'string' ? k : (k.keyword || '')).filter(k => k)
			: [];
		
		if (keywordList.length > 0) {
			const topKeywords = keywordList.slice(0, 3);
			metaDesc = `Professional ${pageTitle.toLowerCase()} services. Expert in ${topKeywords.join(', ')}. Get started today!`;
		} else {
			metaDesc = `Learn more about ${pageTitle}. High-quality services and expert solutions.`;
		}
		
		// Ensure proper length
		if (metaDesc.length > 160) {
			metaDesc = metaDesc.substring(0, 157) + '...';
		}
		
		const seoData = {
			page_id: pageId,
			slug: slug,
			meta_title: metaTitle.substring(0, 60),
			meta_description: metaDesc,
			focus_keyphrase: keywordList[0] || pageTitle,
			keywords: keywordList.join(', '),
			page_url: '',
			domain: window.location.host
		};
		
		console.log('Generated default SEO data:', seoData);
		populateSeoForm(seoData);
	}

	/**
	 * Clear SEO form fields
	 */
	function clearSeoForm() {
		// Keep the page ID
		$('#buildfuly-seo-slug').val('');
		$('#buildfuly-seo-title').val('').trigger('input');
		$('#buildfuly-seo-description').val('').trigger('input');
		$('#buildfuly-seo-focus').val('');
		updateGooglePreview();
	}

	/**
	 * Populate SEO form with data
	 */
	function populateSeoForm(data) {
		console.log('populateSeoForm called with:', data);
		console.log('meta_description value:', data.meta_description);
		console.log('description value:', data.description);
		
		currentPageId = data.page_id;
		$('#buildfuly-seo-page-id').val(data.page_id);
		$('#buildfuly-seo-slug').val(data.slug || '');
		// Support both 'meta_title' and 'title' field names
		$('#buildfuly-seo-title').val(data.meta_title || data.title || '').trigger('input');
		// Support both 'meta_description' and 'description' field names
		$('#buildfuly-seo-description').val(data.meta_description || data.description || '').trigger('input');
		$('#buildfuly-seo-focus').val(data.focus_keyphrase || '');

		updateGooglePreview();
	}

	/**
	 * Save SEO metadata and sync to WordPress page
	 */
	function saveSeoMetadata() {
		const $btn = $(this);
		
		// Get WordPress ID from current page
		let wordpressId = null;
		if (window.BuildfulyPageBuilder && window.BuildfulyPageBuilder.currentPage) {
			wordpressId = window.BuildfulyPageBuilder.currentPage.wordpress_id;
			console.log('Saving SEO for wordpress_id:', wordpressId);
		}
		
		if (!wordpressId) {
			showMessage('error', 'Please publish the page first before editing SEO metadata');
			return;
		}

		$btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Saving...');

		const data = {
			action: 'buildfuly_update_seo_metadata',
			nonce: buildfulyAdmin.adminNonce,
			page_id: wordpressId,
			keywords: '', // You can add a keywords field if needed
			slug: $('#buildfuly-seo-slug').val(),
			meta_title: $('#buildfuly-seo-title').val(),
			meta_description: $('#buildfuly-seo-description').val(),
			focus_keyphrase: $('#buildfuly-seo-focus').val(),
			og_title: '',
			og_description: '',
			twitter_title: '',
			twitter_description: ''
		};
		
		console.log('Sending SEO update data:', data);

		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: data,
			success: function(response) {
				console.log('SEO update response:', response);
				if (response.success) {
					// Also update the local page object
					if (window.BuildfulyPageBuilder && window.BuildfulyPageBuilder.currentPage) {
						window.BuildfulyPageBuilder.currentPage.seo = {
							page_id: wordpressId,
							slug: data.slug,
							meta_title: data.meta_title,
							meta_description: data.meta_description,
							focus_keyphrase: data.focus_keyphrase
						};
						window.BuildfulyPageBuilder.currentPage.title = data.meta_title;
						window.BuildfulyPageBuilder.savePage();
						window.BuildfulyPageBuilder.renderPagesList();
					}
					
					showMessage('success', response.data.message || 'SEO metadata saved!');
					closeModal();
					// Trigger event for other components to react
					$(document).trigger('buildfuly:seoUpdated', [wordpressId, response.data]);
				} else {
					showMessage('error', response.data.message || 'Failed to save SEO metadata');
				}
			},
			error: function(xhr, status, error) {
				console.error('SEO update error:', error, xhr.responseText);
				showMessage('error', 'Failed to save SEO metadata. Please try again.');
			},
			complete: function() {
				$btn.prop('disabled', false).html('<span class="dashicons dashicons-saved"></span> Save SEO & Update Page');
			}
		});
	}

	/**
	 * Show message to user
	 */
	function showMessage(type, message) {
		// Use existing Buildfuly message system if available
		if (window.BuildfulyPageBuilder && window.BuildfulyPageBuilder.showMessage) {
			window.BuildfulyPageBuilder.showMessage(message, type);
		} else {
			// Fallback to alert
			alert(message);
		}
	}

	/**
	 * Add spinner CSS
	 */
	const style = document.createElement('style');
	style.textContent = `
		@keyframes spin {
			from { transform: rotate(0deg); }
			to { transform: rotate(360deg); }
		}
		.dashicons.spin {
			animation: spin 1s linear infinite;
		}
	`;
	document.head.appendChild(style);

	// Initialize when document is ready
	$(document).ready(init);

})(jQuery);
