/**
 * Buildfuly Content Sections - AI-Generated SEO-Optimized Content
 * Uses shared utilities from buildfuly-utils.js
 */

(function($) {
	'use strict';

	// Reference to shared utilities
	const Utils = window.BuildfulyUtils || {};

	const BuildfulySections = {
		sections: [],
		currentDevice: 'desktop',
		autoGenerateOnLoad: true,

		init: function() {
			this.bindEvents();
			this.loadSavedSections();
			
			// Auto-generate if:
			// 1. needsAutoBuild flag is set (provisioned site)
			// 2. OR autoGenerateOnLoad is true AND no sections exist
			const needsAutoBuild = window.buildfulyAdmin && window.buildfulyAdmin.needsAutoBuild;
			
			if (needsAutoBuild) {
				console.log('Buildfuly: Auto-building homepage for provisioned site...');
				this.autoGenerateHomepage();
				// Clear the flag via AJAX
				this.clearAutoBuildFlag();
			} else if (this.autoGenerateOnLoad && this.sections.length === 0) {
				this.autoGenerateHomepage();
			}
		},
		
		/**
		 * Clear the needsAutoBuild flag after generation starts
		 */
		clearAutoBuildFlag: function() {
			$.post(window.buildfulyAdmin.ajaxUrl, {
				action: 'buildfuly_update_option',
				nonce: window.buildfulyAdmin.ajaxNonce,
				option_name: 'buildfuly_needs_auto_build',
				option_value: ''
			});
		},

		bindEvents: function() {
			// Generate section
			$('#buildfuly-generate-section').on('click', this.generateSection.bind(this));
			
			// Device switching
			$('.buildfuly-device-btn').on('click', this.switchDevice.bind(this));
			
			// Clear all
			$('#buildfuly-clear-all').on('click', this.clearAll.bind(this));
			
			// Publish
			$('#buildfuly-publish-content').on('click', this.publishContent.bind(this));
			
			// Save on window unload
			$(window).on('beforeunload', this.saveSections.bind(this));
			
			// Auto-save every 30 seconds
			setInterval(this.saveSections.bind(this), 30000);
		},

	autoGenerateHomepage: function() {
		this.showMessage('Generating your homepage...', 'info', 5000);
		this.showLoading('Creating 6 content sections...');		// Get business info from settings
		const businessKeywords = this.getBusinessKeywords();
		
		// Get available components dynamically from BuildfulyPageBuilder
		const components = window.BuildfulyPageBuilder && window.BuildfulyPageBuilder.availableComponents;
		
		// Build component lists dynamically from available components
		let heroTypes = [];
		let testimonialTypes = [];
		let otherContentTypes = [];
		let ctaTypes = [];
		
		if (components) {
			// Get all hero components
			if (components.hero) {
				heroTypes = Object.keys(components.hero);
			}
			// Get all CTA components
			if (components.cta) {
				ctaTypes = Object.keys(components.cta);
			}
			// Get content components and categorize them
			if (components.content) {
				Object.keys(components.content).forEach(slug => {
					if (slug.includes('testimonial')) {
						testimonialTypes.push(slug);
					} else {
						otherContentTypes.push(slug);
					}
				});
			}
		}
		
		// Fallback if components not loaded yet
		if (heroTypes.length === 0) heroTypes = ['hero-centered', 'hero-split', 'hero-wide'];
		if (ctaTypes.length === 0) ctaTypes = ['cta-banner', 'cta-box'];
		if (otherContentTypes.length === 0) otherContentTypes = ['content-features', 'content-split', 'features-grid', 'how-it-works', 'faq'];
		
		// Pick random hero as first content component
		const selectedHero = heroTypes[Math.floor(Math.random() * heroTypes.length)];
		
		// Randomly decide whether to include a testimonial (50% chance)
		const includeTestimonial = Math.random() < 0.5;
		const selectedTestimonial = includeTestimonial 
			? testimonialTypes[Math.floor(Math.random() * testimonialTypes.length)]
			: null;
		
		// Shuffle other content types and pick 3-4 depending on testimonial
		const shuffledOther = otherContentTypes.sort(() => 0.5 - Math.random());
		const contentCount = includeTestimonial ? 3 : 4;
		const selectedOther = shuffledOther.slice(0, contentCount);
		
		// Pick random CTA type
		const selectedCta = ctaTypes[Math.floor(Math.random() * ctaTypes.length)];
		
		// Build final section list: hero + optional testimonial + 3-4 content + 1 cta (6 total)
		const sectionTypes = selectedTestimonial 
			? [selectedHero, selectedTestimonial, ...selectedOther, selectedCta]
			: [selectedHero, ...selectedOther, selectedCta];
		let generated = 0;
		
		// Hide placeholder and show floating actions
		$('.buildfuly-preview-placeholder').remove();
		$('.buildfuly-floating-actions').fadeIn();
		
		// Add skeleton frames for all sections
		const $container = $('#buildfuly-sections-container');
		$container.empty();
		sectionTypes.forEach((type, index) => {
			const skeletonId = 'skeleton-' + index;
			const $skeleton = $(`
				<div class="buildfuly-section buildfuly-skeleton" id="${skeletonId}" data-loading="true">
					<div class="buildfuly-skeleton-content">
						<div class="buildfuly-skeleton-header"></div>
						<div class="buildfuly-skeleton-text"></div>
						<div class="buildfuly-skeleton-text"></div>
						<div class="buildfuly-skeleton-text short"></div>
					</div>
				</div>
			`);
			$container.append($skeleton);
		});
		
		const generateNext = () => {
			if (generated < sectionTypes.length) {
				const skeletonId = 'skeleton-' + generated;
				this.generateSectionWithType(businessKeywords, sectionTypes[generated], (sectionData) => {
					// Replace skeleton with actual content
					const $skeleton = $(`#${skeletonId}`);
					const sectionId = 'section-' + Date.now();
					sectionData.id = sectionId;
					
					const $section = this.createSectionElement(sectionData);
					$skeleton.replaceWith($section);
					
					// Store section data
					this.sections.push(sectionData);
					
					generated++;
					setTimeout(generateNext, 800);
				});
			} else {
				this.hideLoading();
				this.saveSections();
				this.showMessage('Homepage generated! You can now edit or add more sections.', 'success');
			}
		};
		
		generateNext();
	},		getBusinessKeywords: function() {
			// Try to get from localStorage or use defaults
			const saved = localStorage.getItem('buildfuly_business_keywords');
			return saved || $('#business-keywords').val() || 'professional services, quality solutions, expert team';
		},

		generateSection: function(e) {
			e.preventDefault();
			
			const keywords = $('#business-keywords').val().trim();
			const sectionType = $('#section-type').val();
			
			if (!keywords) {
				this.showMessage('Please enter keywords for SEO optimization', 'warning');
				return;
			}
			
			// Save keywords for future use
			localStorage.setItem('buildfuly_business_keywords', keywords);
			
			this.generateSectionWithType(keywords, sectionType);
		},

	generateSectionWithType: function(keywords, sectionType, callback) {
		this.showLoading('Generating keyword-rich content...');
		
		// Call the WordPress REST API
		$.ajax({
			url: buildfulyAdmin.restUrl + '/content/generate-section',
			method: 'POST',
			beforeSend: function(xhr) {
				xhr.setRequestHeader('X-WP-Nonce', buildfulyAdmin.nonce);
			},
			data: JSON.stringify({
				keywords: keywords,
				section_type: sectionType
			}),
			contentType: 'application/json',
			success: (response) => {
				this.hideLoading();
				if (response.success && response.data) {
					if (callback) {
						// Pass section data to callback
						callback(response.data);
					} else {
						// No callback, add section directly
						this.addSection(response.data);
						this.showMessage('Section generated successfully!', 'success');
					}
				} else {
					this.showMessage(response.message || 'Failed to generate section', 'error');
				}
			},
			error: (xhr, status, error) => {
				this.hideLoading();
				console.error('Generation error:', error);
				this.showMessage('Error generating content. Please check your API settings.', 'error');
			}
		});
	},		addSection: function(sectionData) {
			// Hide placeholder if it exists
			$('.buildfuly-preview-placeholder').remove();
			
			// Show floating actions
			$('.buildfuly-floating-actions').fadeIn();
			
			// Create section HTML with controls
			const sectionId = 'section-' + Date.now();
			sectionData.id = sectionId;
			
			const $section = this.createSectionElement(sectionData);
			
			// Append to container
			$('#buildfuly-sections-container').append($section);
			
			// Store section data
			this.sections.push(sectionData);
			this.saveSections();
			
			// Scroll to new section
			$section[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
		},

		regenerateSection: function(sectionId) {
			const section = this.sections.find(s => s.id === sectionId);
			if (!section) return;
			
			this.showLoading('Regenerating section...');
			
			$.ajax({
				url: buildfulyAdmin.restUrl + '/content/generate-section',
				method: 'POST',
				beforeSend: function(xhr) {
					xhr.setRequestHeader('X-WP-Nonce', buildfulyAdmin.nonce);
				},
				data: JSON.stringify({
					keywords: section.keywords,
					section_type: section.type
				}),
				contentType: 'application/json',
				success: (response) => {
					this.hideLoading();
					if (response.success && response.data) {
						$(`#${sectionId} .buildfuly-section-content`).html(response.data.html);
						section.html = response.data.html;
						this.showMessage('Section regenerated!', 'success');
					}
				},
				error: () => {
					this.hideLoading();
					this.showMessage('Failed to regenerate section', 'error');
				}
			});
		},

		deleteSection: function(sectionId) {
			if (!confirm('Are you sure you want to delete this section?')) return;
			
			$(`#${sectionId}`).fadeOut(300, function() {
				$(this).remove();
			});
			
			this.sections = this.sections.filter(s => s.id !== sectionId);
			
			// Show placeholder if no sections left
			if (this.sections.length === 0) {
				$('#buildfuly-sections-container').html(`
					<div class="buildfuly-preview-placeholder">
						<div class="buildfuly-placeholder-icon">
							<span class="dashicons dashicons-welcome-write-blog"></span>
						</div>
						<h2>Generate Keyword-Rich Content Sections</h2>
						<p>Enter your keywords above and click "Generate Section" to create SEO-optimized content</p>
					</div>
				`);
				$('.buildfuly-floating-actions').fadeOut();
			}
		},

		moveSection: function(sectionId, direction) {
			const $section = $(`#${sectionId}`);
			if (direction === 'up') {
				$section.prev('.buildfuly-section').before($section);
			} else {
				$section.next('.buildfuly-section').after($section);
			}
		},

		openEditPanel: function(sectionId) {
			const section = this.sections.find(s => s.id === sectionId);
			if (!section) return;
			
			// Extract editable fields from HTML
			const $content = $('<div>').html(section.html);
			const fields = [];
			
			// Find all text elements with data-editable or common content selectors
			$content.find('h1, h2, h3, h4, h5, h6, p, span, a, button, .cta-heading, .cta-subheading, .hero-headline, .hero-subheadline, .feature-title, .feature-description, .testimonial-text, .testimonial-author').each(function(index) {
				const $el = $(this);
				const text = $el.text().trim();
				
				if (text && text.length > 0) {
					const tagName = $el.prop('tagName').toLowerCase();
					let fieldType = 'text';
					let label = tagName.toUpperCase();
					
					// Determine field type and label
					if (tagName.match(/^h[1-6]$/)) {
						label = 'Heading';
						fieldType = 'text';
					} else if (tagName === 'p') {
						label = 'Paragraph';
						fieldType = 'textarea';
					} else if ($el.hasClass('hero-headline') || $el.hasClass('cta-heading')) {
						label = 'Main Heading';
					} else if ($el.hasClass('hero-subheadline') || $el.hasClass('cta-subheading')) {
						label = 'Subheading';
						fieldType = 'textarea';
					} else if ($el.hasClass('feature-title') || $el.hasClass('testimonial-author')) {
						label = 'Title';
					} else if ($el.hasClass('feature-description') || $el.hasClass('testimonial-text')) {
						label = 'Description';
						fieldType = 'textarea';
					}
					
					fields.push({
						index: index,
						label: label,
						value: text,
						type: fieldType,
						selector: this.getElementPath($el, $content)
					});
				}
			}.bind(this));
			
			// Show edit panel
			this.showEditPanel(sectionId, fields);
		},

		getElementPath: function($element, $root) {
			const path = [];
			let $current = $element;
			
			while ($current.length && !$current.is($root)) {
				const tagName = $current.prop('tagName').toLowerCase();
				const className = $current.attr('class') || '';
				const index = $current.index();
				
				if (className) {
					path.unshift(`${tagName}.${className.split(' ')[0]}:eq(${index})`);
				} else {
					path.unshift(`${tagName}:eq(${index})`);
				}
				
				$current = $current.parent();
			}
			
			return path.join(' > ');
		},

		showEditPanel: function(sectionId, fields) {
			// Remove existing panel if any
			$('#buildfuly-edit-panel').remove();
			
			// Build form fields
			let fieldsHtml = '';
			fields.forEach((field, index) => {
				if (field.type === 'textarea') {
					fieldsHtml += `
						<div class="buildfuly-edit-field">
							<label>${field.label}</label>
							<textarea class="buildfuly-edit-input" data-index="${index}" rows="3">${field.value}</textarea>
						</div>
					`;
				} else {
					fieldsHtml += `
						<div class="buildfuly-edit-field">
							<label>${field.label}</label>
							<input type="text" class="buildfuly-edit-input" data-index="${index}" value="${field.value}">
						</div>
					`;
				}
			});
			
			const panel = $(`
				<div id="buildfuly-edit-panel" class="buildfuly-edit-panel">
					<div class="buildfuly-edit-panel-header">
						<h3>Edit Content</h3>
						<button class="buildfuly-edit-panel-close">
							<span class="dashicons dashicons-no-alt"></span>
						</button>
					</div>
					<div class="buildfuly-edit-panel-body">
						${fieldsHtml}
					</div>
					<div class="buildfuly-edit-panel-footer">
						<button class="buildfuly-btn buildfuly-btn-secondary" id="buildfuly-edit-cancel">Cancel</button>
						<button class="buildfuly-btn buildfuly-btn-primary" id="buildfuly-edit-save">Save Changes</button>
					</div>
				</div>
			`);
			
			$('body').append(panel);
			
			// Show panel with animation
			setTimeout(() => panel.addClass('active'), 10);
			
			// Bind events
			panel.find('.buildfuly-edit-panel-close, #buildfuly-edit-cancel').on('click', () => this.closeEditPanel());
			panel.find('#buildfuly-edit-save').on('click', () => this.saveEditedContent(sectionId, fields));
		},

		closeEditPanel: function() {
			const $panel = $('#buildfuly-edit-panel');
			$panel.removeClass('active');
			setTimeout(() => $panel.remove(), 300);
		},

		saveEditedContent: function(sectionId, fields) {
			const section = this.sections.find(s => s.id === sectionId);
			if (!section) return;
			
			// Get updated values
			const updates = [];
			$('#buildfuly-edit-panel .buildfuly-edit-input').each(function() {
				const index = parseInt($(this).data('index'));
				const newValue = $(this).val();
				updates[index] = newValue;
			});
			
			// Update HTML content
			const $content = $('<div>').html(section.html);
			let elementIndex = 0;
			
			$content.find('h1, h2, h3, h4, h5, h6, p, span, a, button, .cta-heading, .cta-subheading, .hero-headline, .hero-subheadline, .feature-title, .feature-description, .testimonial-text, .testimonial-author').each(function() {
				const $el = $(this);
				const text = $el.text().trim();
				
				if (text && text.length > 0) {
					if (typeof updates[elementIndex] !== 'undefined') {
						$el.text(updates[elementIndex]);
					}
					elementIndex++;
				}
			});
			
			// Update section data and DOM
			section.html = $content.html();
			$(`#${sectionId} .buildfuly-section-content`).html(section.html);
			
			// Save and close
			this.saveSections();
			this.closeEditPanel();
			this.showMessage('Content updated successfully!', 'success');
		},

		clearAll: function(e) {
			e.preventDefault();
			
			if (!confirm('Are you sure you want to clear all sections?')) return;
			
			$('#buildfuly-sections-container').html(`
				<div class="buildfuly-preview-placeholder">
					<div class="buildfuly-placeholder-icon">
						<span class="dashicons dashicons-welcome-write-blog"></span>
					</div>
					<h2>Generate Keyword-Rich Content Sections</h2>
					<p>Enter your keywords above and click "Generate Section" to create SEO-optimized content</p>
				</div>
			`);
			
			this.sections = [];
			this.saveSections();
			$('.buildfuly-floating-actions').fadeOut();
			this.showMessage('All sections cleared', 'info');
		},

		saveSections: function() {
			// Save to localStorage
			localStorage.setItem('buildfuly_sections', JSON.stringify(this.sections));
			
			// Also save to WordPress via AJAX
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_save_sections',
					nonce: $('#buildfuly-nonce').val(),
					sections: JSON.stringify(this.sections)
				},
				success: (response) => {
					console.log('Sections auto-saved');
				}
			});
		},

		loadSavedSections: function() {
			// Try to load from localStorage first
			const saved = localStorage.getItem('buildfuly_sections');
			if (saved) {
				try {
					this.sections = JSON.parse(saved);
					if (this.sections.length > 0) {
						this.autoGenerateOnLoad = false;
						this.renderSavedSections();
					}
				} catch (e) {
					console.error('Error loading saved sections:', e);
				}
			}
		},

		renderSavedSections: function() {
			$('.buildfuly-preview-placeholder').remove();
			$('.buildfuly-floating-actions').fadeIn();
			
			this.sections.forEach(section => {
				const $section = this.createSectionElement(section);
				$('#buildfuly-sections-container').append($section);
			});
		},

		createSectionElement: function(sectionData) {
			const $section = $(`
				<div class="buildfuly-section" id="${sectionData.id}" data-type="${sectionData.type}">
					<div class="buildfuly-section-controls">
						<button class="buildfuly-section-btn" data-action="edit" title="Edit Content">
							<span class="dashicons dashicons-edit"></span>
						</button>
						<button class="buildfuly-section-btn" data-action="regenerate" title="Regenerate">
							<span class="dashicons dashicons-update"></span>
						</button>
						<button class="buildfuly-section-btn" data-action="move-up" title="Move Up">
							<span class="dashicons dashicons-arrow-up-alt2"></span>
						</button>
						<button class="buildfuly-section-btn" data-action="move-down" title="Move Down">
							<span class="dashicons dashicons-arrow-down-alt2"></span>
						</button>
						<button class="buildfuly-section-btn" data-action="delete" title="Delete">
							<span class="dashicons dashicons-trash"></span>
						</button>
					</div>
					<div class="buildfuly-section-content" contenteditable="true">
						${sectionData.html}
					</div>
				</div>
			`);
			
			// Add event listeners
			$section.find('[data-action="delete"]').on('click', () => this.deleteSection(sectionData.id));
			$section.find('[data-action="regenerate"]').on('click', () => this.regenerateSection(sectionData.id));
			$section.find('[data-action="move-up"]').on('click', () => this.moveSection(sectionData.id, 'up'));
			$section.find('[data-action="move-down"]').on('click', () => this.moveSection(sectionData.id, 'down'));
			$section.find('[data-action="edit"]').on('click', () => this.openEditPanel(sectionData.id));
			
			// Save on content edit
			$section.find('.buildfuly-section-content').on('blur', () => {
				sectionData.html = $section.find('.buildfuly-section-content').html();
				this.saveSections();
			});
			
			return $section;
		},

		publishContent: function(e) {
			e.preventDefault();
			
			if (this.sections.length === 0) {
				this.showMessage('Please generate some content first', 'warning');
				return;
			}
			
			const allHtml = this.sections.map(s => s.html).join('\n\n');
			
			this.showLoading('Publishing to your website...');
			
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_publish_sections',
					nonce: $('#buildfuly-nonce').val(),
					sections: JSON.stringify(this.sections),
					html: allHtml
				},
				success: (response) => {
					this.hideLoading();
					if (response.success) {
						this.showMessage('Content published successfully!', 'success');
					} else {
						this.showMessage(response.data || 'Failed to publish', 'error');
					}
				},
				error: () => {
					this.hideLoading();
					this.showMessage('Error publishing content', 'error');
				}
			});
		},

		switchDevice: function(e) {
			e.preventDefault();
			
			const device = $(e.currentTarget).data('device');
			this.currentDevice = device;
			
			$('.buildfuly-device-btn').removeClass('active');
			$(e.currentTarget).addClass('active');
			
			const $container = $('#buildfuly-sections-container');
			$container.removeClass('device-desktop device-tablet device-mobile');
			$container.addClass('device-' + device);
		},

		/**
		 * Show loading overlay - delegates to Utils
		 */
		showLoading: function(message = 'Processing...') {
			Utils.showLoading(message);
		},

		/**
		 * Hide loading overlay - delegates to Utils
		 */
		hideLoading: function() {
			Utils.hideLoading();
		},

		/**
		 * Show message notification - delegates to Utils
		 */
		showMessage: function(message, type = 'info', duration = 3000) {
			Utils.showMessage(message, type, duration);
		}
	};

	// Initialize when DOM is ready
	$(document).ready(function() {
		BuildfulySections.init();
	});

})(jQuery);
