/**
 * Shared Page Suggestions functionality
 * Used by both onboarding and "Generate More Pages" feature
 */

var BuildfulyPageSuggestions = (function($) {
	'use strict';
	
	return {
		selectedPages: [],
		
		/**
		 * Generate page suggestions via AJAX
		 */
		generateSuggestions: function(businessData, count, existingPages, callbacks, loadingSelector) {
			const self = this;
			const requestData = {
				action: 'buildfuly_suggest_pages',
				nonce: buildfulyAdmin.nonce,
				description: businessData.description,
				business_name: businessData.name,
				address: businessData.address,
				count: count || 10,
				existing_pages: existingPages || []
			};
			
			console.log('Requesting page suggestions:', requestData);
			
			// Show loading animation
			if (loadingSelector) {
				self.showLoadingAnimation($(loadingSelector), businessData);
			}
			
			// Call API
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: requestData,
				success: function(response) {
					console.log('Page suggestions response:', response);
					
					let suggestedPages = [];
					if (response.success && response.data.pages) {
						suggestedPages = response.data.pages;
					} else {
						// Fallback
						suggestedPages = [
							{ name: 'About', keyword: 'our story, mission, and values' },
							{ name: 'Contact', keyword: 'get in touch with us' },
							{ name: 'Portfolio', keyword: 'showcase our best work and projects' },
							{ name: 'Pricing', keyword: 'pricing plans and packages' },
							{ name: 'Testimonials', keyword: 'customer reviews and success stories' },
							{ name: 'FAQ', keyword: 'frequently asked questions' },
							{ name: 'Blog', keyword: 'latest news and insights' }
						];
					}
					
					// Filter out existing pages if provided
					if (existingPages && existingPages.length > 0) {
						const existingLower = existingPages.map(p => p.toLowerCase());
						suggestedPages = suggestedPages.filter(p => 
							!existingLower.includes(p.name.toLowerCase())
						);
					}
					
					if (callbacks && callbacks.success) {
						callbacks.success(suggestedPages);
					}
				},
				error: function(xhr, status, error) {
					console.error('Page suggestions error:', status, error, xhr.responseText);
					
					// Fallback
					const suggestedPages = [
						{ name: 'About', keyword: 'our story, mission, and values' },
						{ name: 'Contact', keyword: 'get in touch with us' },
						{ name: 'Portfolio', keyword: 'showcase our best work' },
						{ name: 'Pricing', keyword: 'pricing plans and packages' },
						{ name: 'Testimonials', keyword: 'customer reviews and success stories' }
					];
					
					if (callbacks && callbacks.error) {
						callbacks.error(suggestedPages);
					} else if (callbacks && callbacks.success) {
						callbacks.success(suggestedPages);
					}
				}
			});
		},
		
		/**
		 * Show loading animation with real-time AI progress (like homepage preview)
		 */
		showLoadingAnimation: function($container, businessData) {
			const steps = [
				{ icon: '🔍', text: 'Analyzing your business...', delay: 200 },
				{ icon: '🧠', text: 'AI thinking about your industry...', delay: 1200 },
				{ icon: '🎯', text: 'Finding high-value pages...', delay: 2400 },
				{ icon: '📊', text: 'Checking search volumes...', delay: 3600 },
				{ icon: '✨', text: 'Building page suggestions...', delay: 4800 }
			];
			
			$container.html(`
				<div class="page-suggestions-loading">
					<div class="spinner"></div>
					<div id="ai-progress-steps" style="margin-top: 32px; max-width: 400px; margin-left: auto; margin-right: auto;">
						${steps.map((step, index) => `
							<div class="ai-step" data-step="${index}" style="opacity: 0; transform: translateY(10px); transition: all 0.4s ease; margin-bottom: 12px; padding: 12px 20px; background: rgba(255,255,255,0.9); border-radius: 8px; display: flex; align-items: center; gap: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.05);">
								<span style="font-size: 24px;">${step.icon}</span>
								<span style="font-size: 14px; font-weight: 500; color: #1e293b;">${step.text}</span>
							</div>
						`).join('')}
					</div>
				</div>
			`);
			
			// Animate steps appearing one by one
			steps.forEach((step, index) => {
				setTimeout(() => {
					const $step = $container.find(`[data-step="${index}"]`);
					$step.css({ opacity: '1', transform: 'translateY(0)' });
					
					// Fade out previous step
					if (index > 0) {
						const $prevStep = $container.find(`[data-step="${index - 1}"]`);
						$prevStep.css({ opacity: '0.4' });
					}
				}, step.delay);
			});
		},
		
		/**
		 * Render page checkboxes in a grid
		 */
		renderPageCheckboxes: function(suggestedPages, gridSelector, options) {
			const $grid = $(gridSelector);
			$grid.empty();
			this.selectedPages = [];
			
			const opts = $.extend({
				autoSelectHome: false,
				countSelector: null,
				submitSelector: null
			}, options);
			
			suggestedPages.forEach((pageObj) => {
				const pageName = typeof pageObj === 'string' ? pageObj : pageObj.name;
				const pageKeyword = typeof pageObj === 'string' ? `landing page for ${pageObj}` : pageObj.keyword;
				
				const pageData = { name: pageName, keyword: pageKeyword };
				const isHome = pageName === 'Home';
				const isAutoSelected = opts.autoSelectHome && isHome;
				
				const $item = $(`
					<div class="page-selection-item ${isAutoSelected ? 'default-page selected' : ''}" data-page='${JSON.stringify(pageData).replace(/'/g, "&apos;")}'>
						<input type="checkbox" ${isAutoSelected ? 'checked disabled' : ''} />
						<h4>${pageName}</h4>
						<p class="page-keyword">${pageKeyword}</p>
					</div>
				`);
				
				$grid.append($item);
				
				if (isAutoSelected) {
					this.selectedPages.push(pageData);
				}
			});
			
			this.updatePageCount(opts.countSelector, opts.submitSelector);
		},
		
		/**
		 * Setup event handlers for page selection
		 */
		setupEventHandlers: function(gridSelector, options) {
			const self = this;
			const opts = $.extend({
				countSelector: null,
				submitSelector: null,
				onSelectionChange: null
			}, options);
			
			// Checkbox change
			$(document).off('change.pagesuggestions', gridSelector + ' .page-selection-item input[type="checkbox"]');
			$(document).on('change.pagesuggestions', gridSelector + ' .page-selection-item input[type="checkbox"]', function() {
				const $item = $(this).closest('.page-selection-item');
				const pageData = JSON.parse($item.attr('data-page'));
				
				if ($(this).is(':checked')) {
					$item.addClass('selected');
					if (!self.selectedPages.some(p => p.name === pageData.name)) {
						self.selectedPages.push(pageData);
					}
				} else {
					$item.removeClass('selected');
					self.selectedPages = self.selectedPages.filter(p => p.name !== pageData.name);
				}
				
				self.updatePageCount(opts.countSelector, opts.submitSelector);
				
				if (opts.onSelectionChange) {
					opts.onSelectionChange(self.selectedPages);
				}
			});
			
			// Click on card to toggle checkbox
			$(document).off('click.pagesuggestions', gridSelector + ' .page-selection-item');
			$(document).on('click.pagesuggestions', gridSelector + ' .page-selection-item', function(e) {
				if ($(e.target).is('input') || $(this).hasClass('default-page')) return;
				
				const $checkbox = $(this).find('input[type="checkbox"]');
				if (!$checkbox.prop('disabled')) {
					$checkbox.prop('checked', !$checkbox.prop('checked')).trigger('change');
				}
			});
		},
		
		/**
		 * Update page count display with word cost
		 */
		updatePageCount: function(countSelector, submitSelector) {
			const count = this.selectedPages.length;
			const wordCost = count * 5; // 5 words per page
			
			if (countSelector) {
				// Update count text with word cost
				const $countContainer = $(countSelector).parent();
				const costText = wordCost > 0 ? ` • Cost: <strong>${wordCost} words</strong>` : '';
				
				// Check if cost element exists
				if ($countContainer.find('#buildfuly-suggest-cost, #buildfuly-suggest-cost-onboarding').length === 0) {
					// Just update the count
					$(countSelector).text(count);
				} else {
					// Update both count and cost
					$(countSelector).text(count);
					$('#buildfuly-suggest-cost, #buildfuly-suggest-cost-onboarding').text(wordCost + ' words');
				}
			}
			
			if (submitSelector) {
				$(submitSelector).prop('disabled', count === 0);
			}
		},
		
		/**
		 * Get selected pages
		 */
		getSelectedPages: function() {
			return this.selectedPages;
		},
		
		/**
		 * Clear selection
		 */
		clearSelection: function() {
			this.selectedPages = [];
		}
	};
	
})(jQuery);
