﻿/**
 * Buildfuly Page Builder - Multi-Page SEO Content Generator
 * Uses shared utilities from buildfuly-utils.js
 */

(function($) {
	'use strict';

	// Reference to shared utilities
	const Utils = window.BuildfulyUtils || {};

	const BuildfulyPageBuilder = {
		pages: [],
		currentPage: null,
		currentDevice: 'desktop',
		themeColors: null,
		usageStats: null, // Store current usage stats for pre-budget checks
		componentWordEstimates: null, // Store component word estimates from API

	/**
	 * Debug logging - delegates to Utils
	 */
	debug: function(...args) {
		Utils.debug('PageBuilder', ...args);
	},

	init: function() {
		this.debug('Initializing Page Builder...');
		this.loadThemeColors();
		this.bindEvents();
		
		// Hide sidebar expand button initially
		$('#buildfuly-sidebar-expand').hide();
		
		// Update color pickers with saved values
		this.updateColorPickers();

		// Apply CSS variables immediately on page load
		this.applyThemeColors();

		// Load usage stats and component word estimates
		this.loadUsageStats();
		this.loadComponentWordEstimates();
		
		// Load components from API
		this.loadComponentsFromAPI();
		
		// Load pages (async) - will call onPagesLoaded when done
		this.loadPages();
	},
	
	/**
	 * Called after pages are loaded (from WordPress or localStorage)
	 */
	onPagesLoaded: function() {
		// Show welcome screen or modal if no pages
		if (this.pages.length === 0) {
			this.showWelcomeScreen();
		} else {
			this.renderPagesList();
			
			// Check URL for page_id or open_page parameter to load the correct page
			const urlParams = new URLSearchParams(window.location.search);
			const urlPageId = urlParams.get('page_id') || urlParams.get('open_page');
			
			if (urlPageId) {
				// Try to find page by wordpress_id (WordPress post ID) or by buildfuly id
				const pageToLoad = this.pages.find(p => 
					p.wordpress_id == urlPageId || p.id === urlPageId
				);
				
				if (pageToLoad) {
					// Load the page from URL
					this.loadPage(pageToLoad.id);
				} else if (this.pages[0]) {
					// Fallback to first page
					this.loadPage(this.pages[0].id);
				}
			} else if (this.pages[0]) {
				// Fallback to first page
				this.loadPage(this.pages[0].id);
			}
		}
	},		bindEvents: function() {
		console.log('BuildfulyPageBuilder: Binding events...');
		
		// New page buttons
	$('#buildfuly-new-page, #buildfuly-get-started').on('click', (e) => this.showNewPageModal(e, true)); // true = brand new page
	$('#buildfuly-analyze-page').on('click', this.analyzePageDescription.bind(this));
	$('#buildfuly-back-to-description').on('click', this.backToDescription.bind(this));
	$('#buildfuly-create-page').on('click', this.createPage.bind(this));
	$('#buildfuly-cancel-page, .buildfuly-modal-close').on('click', this.hideNewPageModal.bind(this));
	
	// Generate More Pages modal
	$('#buildfuly-generate-more-pages').on('click', this.showSuggestPagesModal.bind(this));
	$('#buildfuly-suggest-close, #buildfuly-suggest-cancel').on('click', this.hideSuggestPagesModal.bind(this));
	$('#buildfuly-suggest-generate').on('click', this.generateSuggestedPages.bind(this));
	$('#buildfuly-suggest-add').on('click', this.addSelectedSuggestedPages.bind(this));
	
	// Suggested page selection
	$(document).on('change', '#suggested-pages-grid .page-selection-item input[type="checkbox"]', this.handleSuggestedPageSelection.bind(this));
	$(document).on('click', '#suggested-pages-grid .page-selection-item', this.handleSuggestedPageClick.bind(this));
	
	// Page count input - update display and slider background
	$(document).on('input', '#suggest-page-count', this.updateSuggestPageSlider.bind(this));
	
	// Layout customization controls
		$('#buildfuly-component-count').on('input', () => this.updateLayoutPreview(true));
		$('#buildfuly-no-hero, #buildfuly-no-testimonials').on('change', () => this.updateLayoutPreview(true));
		$('#buildfuly-randomize-layout').on('change', () => this.updateLayoutPreview(true));
		
		// Auto-generate images checkbox
		$('#buildfuly-auto-generate-images').on('change', function() {
			const isChecked = $(this).is(':checked');
			$('#buildfuly-auto-image-options').toggle(isChecked);
			$('#buildfuly-manual-images-section').toggle(!isChecked);
		});
		
		// Hero image upload
		$('#buildfuly-upload-page-images').on('click', this.uploadPageImages.bind(this));
		$(document).on('click', '.buildfuly-remove-page-image', this.removePageImage.bind(this));
		
		// Section image upload (single image for components)
		$('#buildfuly-upload-section-image').on('click', this.uploadSectionImage.bind(this));
		
		// Add section from checked keywords
		$(document).on('click', '#buildfuly-add-section-from-keywords', this.addSectionFromCheckedKeywords.bind(this));
		
		// Add section (legacy)
		$('#buildfuly-add-section').on('click', this.addSection.bind(this));
		
		// Device switching
		$('.buildfuly-device-btn').on('click', this.switchDevice.bind(this));		// Color pickers - use event delegation for dynamic elements
		$(document).on('change', '.buildfuly-color-picker', this.updateColor.bind(this));
		
		// Keywords collapse/expand toggle
		$('#buildfuly-keywords-toggle').on('click', this.toggleKeywords.bind(this));
		
		// Colors collapse/expand toggle
		$('#buildfuly-colors-toggle').on('click', this.toggleColors.bind(this));
		
		// Sidebar toggle
		$('#buildfuly-sidebar-toggle').on('click', this.toggleSidebar.bind(this));
		$('#buildfuly-sidebar-expand').on('click', this.toggleSidebar.bind(this));
		$('#buildfuly-sidebar-collapse-float').on('click', this.toggleSidebar.bind(this));
		
		// Regenerate colors - use event delegation AND direct binding
		$(document).on('click', '#buildfuly-regenerate-colors, #buildfuly-regenerate-colors-icon', function(e) {
			console.log('Regenerate colors clicked (delegated)');
			this.regenerateColors(e);
		}.bind(this));
		
		// Regenerate all sections button
		$(document).on('click', '#buildfuly-regenerate-all', this.regenerateAllSections.bind(this));
		
		// Batch AI images button
		$(document).on('click', '#buildfuly-batch-ai-images', this.batchGenerateAIImages.bind(this));
		
		// Also try direct binding after small delay
		setTimeout(() => {
			const $btn = $('#buildfuly-regenerate-colors, #buildfuly-regenerate-colors-icon');
			console.log('Regenerate button found:', $btn.length);
			if ($btn.length) {
				$btn.off('click').on('click', this.regenerateColors.bind(this));
				console.log('Direct click handler bound to regenerate button');
			}
		}, 500);			// Save and publish
			$('#buildfuly-save-page').on('click', this.savePage.bind(this));
			$('#buildfuly-publish-page').on('click', this.publishPage.bind(this));
			$('#buildfuly-preview-live').on('click', this.previewLivePage.bind(this));
			
			// Auto-save
			setInterval(this.autoSave.bind(this), 30000);
			
			console.log('BuildfulyPageBuilder: Events bound successfully');
		},

		/**
		 * Preview the live published page
		 */
		previewLivePage: function() {
			if (!this.currentPage) {
				this.showMessage('No page selected', 'warning');
				return;
			}
			
			if (!this.currentPage.wordpress_id) {
				this.showMessage('Please publish the page first to preview it.', 'warning');
				return;
			}
			
			// Get the page URL via AJAX using wordpress_id
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_get_page_url',
					nonce: $('#buildfuly-nonce').val(),
					wordpress_id: this.currentPage.wordpress_id
				},
				success: (response) => {
					if (response.success && response.data.url) {
						window.open(response.data.url, '_blank');
					} else {
						this.showMessage('Page not found. Please publish the page first.', 'error');
					}
				},
				error: () => {
					this.showMessage('Failed to get page URL.', 'error');
				}
			});
		},

		showWelcomeScreen: function() {
			$('.buildfuly-welcome-screen').show();
		},

		hideWelcomeScreen: function() {
			$('.buildfuly-welcome-screen').hide();
		},

		showEmptyPageState: function() {
			// Get keyword from either format (string or array)
			let keyword = this.currentPage.title;
			if (this.currentPage.keywords && Array.isArray(this.currentPage.keywords) && this.currentPage.keywords.length > 0) {
				keyword = this.currentPage.keywords.join(', ');
			} else if (this.currentPage.keyword) {
				keyword = this.currentPage.keyword;
			}
			
			// Display keyword in the empty state
			let keywordDisplay = '';
			if (this.currentPage.keywords && Array.isArray(this.currentPage.keywords) && this.currentPage.keywords.length > 0) {
				keywordDisplay = this.currentPage.keywords.join(', ');
			} else if (this.currentPage.keyword) {
				keywordDisplay = this.currentPage.keyword;
			}
			
			const $emptyState = $(`
				<div class="buildfuly-empty-page-state">
					<div class="empty-state-icon">
						<span class="dashicons dashicons-welcome-add-page"></span>
					</div>
					<h3>This page is empty</h3>
					<p>Generate AI-powered content for <strong>${this.currentPage.title}</strong></p>
					${keywordDisplay ? `<p class="empty-state-keyword"><em>"${keywordDisplay}"</em></p>` : ''}
					<button id="buildfuly-generate-empty-page" class="buildfuly-btn-gradient buildfuly-btn-large">
						<span class="dashicons dashicons-magic"></span>
						Generate Content
					</button>
				</div>
			`);
			
			$('#buildfuly-sections-container').append($emptyState);
			
		// Bind click handler
		$('#buildfuly-generate-empty-page').on('click', () => {
			// Pre-fill both fields with stored keyword or page title
			$('#page-type').val(keyword);
			$('#page-keywords').val(keyword);
			this.showNewPageModal({ preventDefault: () => {} });
		});
		},

showNewPageModal: function(e, isNewPage = false) {
	if (e && e.preventDefault) {
		e.preventDefault();
	}
	
	this.loadLayoutPreferences();
	$('#buildfuly-new-page-modal').fadeIn(200);
	
	setTimeout(() => {
		// If this is a brand new page (not regenerating existing), always start at step 1
		if (isNewPage) {
			$('#buildfuly-page-step-1').show();
			$('#buildfuly-page-step-2').hide();
			$('#buildfuly-create-page').hide();
			
			const pageTypeField = $('#page-type').val();
			const keywordsField = $('#page-keywords').val();
			
			if (!pageTypeField) $('#page-type').val('');
			if (!keywordsField) $('#page-keywords').val('');
			$('#page-title-input').val('');
			$('#buildfuly-page-images-ids').val('');
			$('#buildfuly-page-images-preview').empty();
			
			$('#page-type').focus();
			return;
		}
		
		// Otherwise, check if current page has keywords (for regenerating existing pages)
		let hasKeywords = false;
		
		if (this.currentPage) {
			if (this.currentPage.keywords && Array.isArray(this.currentPage.keywords) && this.currentPage.keywords.length > 0) {
				hasKeywords = true;
			} else if (this.currentPage.keyword && this.currentPage.keyword.trim() !== '') {
				hasKeywords = true;
			}
		}
		
		if (hasKeywords) {
			// Skip to step 2 - layout customization
			$('#buildfuly-page-step-1').hide();
			$('#buildfuly-page-step-2').show();
			$('#buildfuly-create-page').show();
			
			// Populate keywords field
			if (!$('#page-keywords').val() && this.currentPage) {
				const keywordValue = this.currentPage.keyword || 
					(this.currentPage.keywords ? this.currentPage.keywords.join(', ') : '');
				$('#page-keywords').val(keywordValue);
			}
			
			// Reset custom layout if randomize is enabled
			if ($('#buildfuly-randomize-layout').is(':checked')) {
				this.customLayout = null;
			}
			
			this.updateLayoutPreview(true);
		} else {
			// Show step 1 - analyze and generate keywords
			$('#buildfuly-page-step-1').show();
			$('#buildfuly-page-step-2').hide();
			$('#buildfuly-create-page').hide();
			
			const pageTypeField = $('#page-type').val();
			const keywordsField = $('#page-keywords').val();
			
			if (!pageTypeField) $('#page-type').val('');
			if (!keywordsField) $('#page-keywords').val('');
			$('#buildfuly-page-images-ids').val('');
			$('#buildfuly-page-images-preview').empty();
			
			$('#page-type').focus();
		}
	}, 50);
},	hideNewPageModal: function(e) {
		if (e) e.preventDefault();
		$('#buildfuly-new-page-modal').fadeOut(200);
	},
	
	showSuggestPagesModal: function(e) {
		if (e) e.preventDefault();
		
		// Reset form to defaults
		$('#suggest-page-count').val(5);
		$('#suggest-page-count-display').text(5);
		$('#suggest-location-relevant').prop('checked', true);
		$('#suggest-config-form').show();
		$('#suggested-pages-loading').hide();
		$('#suggested-pages-success').hide();
		this.updateSuggestPageSlider();
		
		// Show modal
		$('#buildfuly-suggest-pages-modal').fadeIn(200);
	},
	
	updateSuggestPageSlider: function() {
		const count = parseInt($('#suggest-page-count').val()) || 5;
		$('#suggest-page-count-display').text(count);
		
		// Update token cost estimate (approximately 5 tokens per page suggestion)
		const tokenCost = count * 5;
		$('#suggest-token-cost').text(`~${tokenCost}`);
		
		// Update slider background gradient
		const percentage = ((count - 1) / 19) * 100; // 1-20 range
		$('#suggest-page-count').css('background', 
			`linear-gradient(to right, #0071e3 0%, #0071e3 ${percentage}%, #e5e7eb ${percentage}%, #e5e7eb 100%)`
		);
	},
	
	// Extract city from address (same logic as onboarding)
	extractCityFromAddress: function(address) {
		if (!address || address.trim().length === 0) return '';
		
		// Common address formats: "123 Main St, New York, NY 10001"
		const parts = address.split(',').map(p => p.trim());
		
		if (parts.length >= 2) {
			// City is usually the second-to-last part (before state/zip)
			const cityPart = parts[parts.length - 2];
			const city = cityPart.replace(/\d+/g, '').trim();
			return city;
		}
		
		// Fallback: try to find a capitalized word that's not a number
		const words = address.split(/[\s,]+/);
		for (const word of words) {
			if (word.length > 2 && /^[A-Z][a-z]+$/.test(word) && !/^\d+$/.test(word)) {
				return word;
			}
		}
		
		return '';
	},
	
	generateSuggestedPages: function(e) {
		if (e) e.preventDefault();
		
		const count = parseInt($('#suggest-page-count').val()) || 5;
		
		// Get location from settings if relevant
		const locationRelevant = buildfulyAdmin.locationRelevant || false;
		let location = '';
		if (locationRelevant && buildfulyAdmin.businessLocation) {
			location = buildfulyAdmin.businessLocation;
		}
		
		// Get business data (WITHOUT business_name)
		const businessData = {
			description: buildfulyAdmin.businessDescription || '',
			location: location
		};
		
		// Get existing page titles to exclude
		const existingPages = this.pages.map(p => p.title.toLowerCase());
		
		console.log('Generating pages:', { count, location, businessData, existingPages });
		
		// Hide form, show loading
		$('#suggest-config-form').hide();
		$('#suggested-pages-loading').show();
		
		// Clear previous live pages
		$('#suggest-live-pages-grid').empty();
		$('#suggest-live-suggestions-header').hide();
		
		// Use shared animation helper with detailed context
		if (window.BuildfulyAnimations) {
			window.BuildfulyAnimations.animateGeneration({
				statusId: 'suggest-generation-status',
				substatusId: 'suggest-generation-substatus',
				liveContainerId: 'suggest-live-suggestions',
				context: {
					description: businessData.description,
					location: location,
					count: count,
					existingPages: existingPages
				}
			});
		}
		
		// Call API
		const self = this;
		$.ajax({
			url: ajaxurl,
			type: 'POST',
			data: {
				action: 'buildfuly_suggest_pages',
				nonce: buildfulyAdmin.nonce,
				description: businessData.description,
				location: businessData.location,
				count: count,
				existing_pages: existingPages
			},
			success: function(response) {
				console.log('API Response:', response);
				
				if (response.success && response.data && response.data.pages) {
					// Store suggested pages for later selection
					self.suggestedPages = response.data.pages;
					self.selectedSuggestedPages = [];
					
					// Show pages one by one with animation
					response.data.pages.forEach((pageData, index) => {
						setTimeout(() => {
							const pageName = typeof pageData === 'string' ? pageData : pageData.name;
							const keyword = typeof pageData === 'object' ? pageData.keyword : '';
							
							// Use shared animation helper
							if (window.BuildfulyAnimations) {
								window.BuildfulyAnimations.showLivePage({
									gridId: 'suggest-live-pages-grid',
									headerId: 'suggest-live-suggestions-header'
								}, pageName, keyword);
							}
						}, index * 200);
					});
					
					// Wait for animations to finish, then show selection interface
					const animationDelay = response.data.pages.length * 200 + 800;
					setTimeout(() => {
						$('#suggested-pages-loading').fadeOut(300, function() {
							self.renderSuggestedPagesSelection();
							$('#suggested-pages-selection').fadeIn(300);
							$('#buildfuly-suggest-generate').hide();
							$('#buildfuly-suggest-add').show();
						});
					}, animationDelay);
				} else {
					const errorMsg = (response.data && response.data.message) ? response.data.message : 'Failed to generate pages';
					console.error('Generation failed:', errorMsg);
					$('#suggested-pages-loading').hide();
					self.showMessage(errorMsg, 'error');
					setTimeout(() => {
						self.hideSuggestPagesModal();
					}, 2000);
				}
			},
			error: function(xhr, status, error) {
				console.error('Generation failed:', error);
				$('#suggested-pages-loading').hide();
				self.showMessage('Failed to generate pages. Please try again.', 'error');
				setTimeout(() => {
					self.hideSuggestPagesModal();
				}, 2000);
			}
		});
	},
	
	hideSuggestPagesModal: function(e) {
		if (e) e.preventDefault();
		$('#buildfuly-suggest-pages-modal').fadeOut(200, function() {
			// Reset modal state
			$('#suggest-config-form').show();
			$('#suggested-pages-loading').hide();
			$('#suggested-pages-selection').hide();
			$('#suggested-pages-success').hide();
			$('#buildfuly-suggest-generate').show();
			$('#buildfuly-suggest-add').hide();
		});
	},
	
	renderSuggestedPagesSelection: function() {
		const $grid = $('#suggested-pages-grid');
		$grid.empty();
		this.selectedSuggestedPages = [];
		
		this.suggestedPages.forEach((pageData) => {
			const pageName = typeof pageData === 'string' ? pageData : pageData.name;
			const pageKeyword = typeof pageData === 'string' ? `landing page for ${pageData}` : pageData.keyword;
			
			const $item = $(`
				<div class="page-selection-item" data-page='${JSON.stringify({ name: pageName, keyword: pageKeyword })}'>
					<input type="checkbox" />
					<h4>${pageName}</h4>
					<p class="page-keyword">${pageKeyword}</p>
				</div>
			`);
			
			$grid.append($item);
		});
		
		this.updateSuggestPageCount();
	},
	
	updateSuggestPageCount: function() {
		$('#suggest-page-count').text(this.selectedSuggestedPages.length);
	},
	
	handleSuggestedPageSelection: function(e) {
		const $item = $(e.target).closest('.page-selection-item');
		const pageData = JSON.parse($item.attr('data-page'));
		
		if ($(e.target).is(':checked')) {
			$item.addClass('selected');
			if (!this.selectedSuggestedPages.some(p => p.name === pageData.name)) {
				this.selectedSuggestedPages.push(pageData);
			}
		} else {
			$item.removeClass('selected');
			this.selectedSuggestedPages = this.selectedSuggestedPages.filter(p => p.name !== pageData.name);
		}
		
		this.updateSuggestPageCount();
	},
	
	handleSuggestedPageClick: function(e) {
		// Skip if clicking checkbox directly
		if ($(e.target).is('input')) return;
		
		const $checkbox = $(e.currentTarget).find('input[type="checkbox"]');
		$checkbox.prop('checked', !$checkbox.prop('checked')).trigger('change');
	},
	
	addSelectedSuggestedPages: function(e) {
		if (e) e.preventDefault();
		
		if (this.selectedSuggestedPages.length === 0) {
			this.showMessage('Please select at least one page to add', 'warning');
			return;
		}
		
		// Add selected pages
		this.selectedSuggestedPages.forEach(pageData => {
			const newPage = {
				id: 'page-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9),
				title: pageData.name,
				keyword: pageData.keyword,
				keywords: pageData.keyword.split(',').map(k => k.trim()),
				sections: [],
				colors: this.themeColors
			};
			
			this.pages.push(newPage);
			
			// Save each new page to WordPress database
			this.currentPage = newPage;
			this.savePageToDatabase();
		});
		
		// Update pages list
		this.renderPagesList();
		
		// Show success
		$('#suggested-pages-selection').hide();
		$('#suggested-pages-success').show();
		
		// Auto-close after 2 seconds
		setTimeout(() => {
			this.hideSuggestPagesModal();
			this.showMessage(`Added ${this.selectedSuggestedPages.length} page${this.selectedSuggestedPages.length > 1 ? 's' : ''}`, 'success');
		}, 2000);
	},

	/**
	 * Generate a proper page title from user's description
	 * e.g., "homepage for plumbing services" -> "Home"
	 *       "about us page for my company" -> "About Us"
	 *       "services we offer" -> "Services"
	 */
	generatePageTitle: function(description) {
		const desc = description.toLowerCase().trim();
		
		// Common page type mappings
		const pageTitleMappings = {
			'home': 'Home',
			'homepage': 'Home',
			'landing': 'Home',
			'main': 'Home',
			'about': 'About Us',
			'about us': 'About Us',
			'about me': 'About',
			'services': 'Services',
			'our services': 'Services',
			'what we do': 'Services',
			'contact': 'Contact',
			'contact us': 'Contact',
			'get in touch': 'Contact',
			'pricing': 'Pricing',
			'prices': 'Pricing',
			'plans': 'Pricing',
			'faq': 'FAQ',
			'faqs': 'FAQ',
			'questions': 'FAQ',
			'blog': 'Blog',
			'news': 'Blog',
			'articles': 'Blog',
			'portfolio': 'Portfolio',
			'work': 'Portfolio',
			'projects': 'Portfolio',
			'gallery': 'Gallery',
			'team': 'Our Team',
			'our team': 'Our Team',
			'testimonials': 'Testimonials',
			'reviews': 'Testimonials',
			'careers': 'Careers',
			'jobs': 'Careers',
			'privacy': 'Privacy Policy',
			'privacy policy': 'Privacy Policy',
			'terms': 'Terms of Service',
			'terms of service': 'Terms of Service',
			'legal': 'Legal'
		};
		
		// Check for exact or partial matches
		for (const [keyword, title] of Object.entries(pageTitleMappings)) {
			if (desc.includes(keyword)) {
				return title;
			}
		}
		
		// If no match, try to extract a sensible title
		// Remove common filler words and create title case
		const fillerWords = ['page', 'for', 'my', 'our', 'the', 'a', 'an', 'website', 'site', 'business', 'company'];
		let words = desc.split(/\s+/).filter(word => !fillerWords.includes(word));
		
		// Take first 2-3 meaningful words and title case them
		if (words.length > 0) {
			words = words.slice(0, 3);
			return words.map(word => word.charAt(0).toUpperCase() + word.slice(1)).join(' ');
		}
		
		// Fallback to title casing the original
		return description.split(' ').slice(0, 3).map(word => 
			word.charAt(0).toUpperCase() + word.slice(1).toLowerCase()
		).join(' ');
	},

	backToDescription: function(e) {
		e.preventDefault();
		$('#buildfuly-page-step-2').hide();
		$('#buildfuly-page-step-1').show();
		$('#buildfuly-create-page').hide(); // Hide generate button when going back
	},

	analyzePageDescription: function(e) {
		e.preventDefault();
		
		const pageType = $('#page-type').val().trim();
		if (!pageType) {
			this.showMessage('Please describe the page you want to build', 'warning');
			return;
		}

		const $btn = $('#buildfuly-analyze-page');
		const originalHtml = $btn.html();
		$btn.prop('disabled', true).html('<span class="spinner" style="width: 16px; height: 16px; border-width: 2px; margin-right: 8px;"></span> Analyzing...');

		$.ajax({
			url: buildfulyAdmin.restUrl + '/content/generate-keywords',
			method: 'POST',
			beforeSend: function(xhr) {
				xhr.setRequestHeader('X-WP-Nonce', buildfulyAdmin.nonce);
			},
			data: JSON.stringify({
				description: pageType,
				page_title: pageType,
				generate_seo: true  // Request SEO metadata generation
			}),
			contentType: 'application/json',
			success: (response) => {
				$btn.prop('disabled', false).html(originalHtml);
				
				if (response.success && response.data.keywords) {
					// Set generated keywords
					const keywords = Array.isArray(response.data.keywords) 
						? response.data.keywords.join(', ')
						: response.data.keywords;
					
					$('#page-keywords').val(keywords);
					
					// Generate a proper page title from the page type
					// Convert "homepage for plumbing services" to "Home" or "Services"
					const generatedTitle = this.generatePageTitle(pageType);
					$('#page-title-input').val(generatedTitle);
					
					// Store SEO data - if API didn't return it, generate defaults
					if (response.data.seo) {
						this.pendingSeoData = response.data.seo;
						console.log('Stored SEO from API:', this.pendingSeoData);
					} else {
						// Generate default SEO metadata
						const keywordList = Array.isArray(response.data.keywords) 
							? response.data.keywords 
							: response.data.keywords.split(',').map(k => k.trim());
						
						const slug = generatedTitle.toLowerCase() === 'home' 
							? '/' 
							: generatedTitle.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');
						
						// Don't add brand to meta_title - WordPress/Yoast adds it automatically
						const metaTitle = generatedTitle;
						
						// Create description from keywords
						const topKeywords = keywordList.slice(0, 3);
						const metaDescription = `Professional ${pageType.toLowerCase()} services. Expert in ${topKeywords.join(', ')}. Get started today!`;
						
						this.pendingSeoData = {
							slug: slug,
							meta_title: metaTitle.substring(0, 60),
							meta_description: metaDescription.substring(0, 160),
							focus_keyphrase: keywordList[0] || generatedTitle,
							keywords: keywords
						};
						console.log('Generated default SEO:', this.pendingSeoData);
					}
					
					// Move to step 2 with layout customization
					$('#buildfuly-page-step-1').hide();
					$('#buildfuly-page-step-2').show();
					$('#buildfuly-create-page').show();
					
					// Update layout preview
					this.updateLayoutPreview();
				} else {
					this.showMessage('Failed to generate keywords. Please try again.', 'error');
				}
			},
			error: (xhr, status, error) => {
				$btn.prop('disabled', false).html(originalHtml);
				
				// Even on error, try to proceed with manual keywords if entered
				const keywordsStr = $('#page-keywords').val().trim();
				if (keywordsStr) {
					const keywordList = keywordsStr.split(',').map(k => k.trim()).filter(k => k);
					
					// Generate default SEO from manual keywords
					const slug = pageType.toLowerCase() === 'home' 
						? '/' 
						: pageType.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');
					
					// Don't add brand to meta_title - WordPress/Yoast adds it automatically
					const metaTitle = pageType;
					
					const topKeywords = keywordList.slice(0, 3);
					const metaDescription = topKeywords.length > 0
						? `Professional ${pageType.toLowerCase()} services. Expert in ${topKeywords.join(', ')}. Get started today!`
						: `Learn more about ${pageType}. High-quality services and expert solutions.`;
					
					this.pendingSeoData = {
						slug: slug,
						meta_title: metaTitle.substring(0, 60),
						meta_description: metaDescription.substring(0, 160),
						focus_keyphrase: keywordList[0] || pageType,
						keywords: keywordsStr
					};
					console.log('Generated SEO from manual keywords:', this.pendingSeoData);
					
					this.showMessage('Analysis failed, but you can proceed with manual keywords', 'warning');
				} else {
					this.showMessage('Error analyzing page. Please try again or enter keywords manually.', 'error');
				}
			}
		});
	},

	loadLayoutPreferences: function() {
		// Load saved preferences from localStorage
		const prefs = localStorage.getItem('buildfuly_layout_prefs');
		if (prefs) {
			try {
				const parsed = JSON.parse(prefs);
				$('#buildfuly-component-count').val(parsed.componentCount || 6);
				$('#buildfuly-no-hero').prop('checked', parsed.noHero || false);
				$('#buildfuly-no-testimonials').prop('checked', parsed.noTestimonials || false);
				$('#buildfuly-component-count-label').text(parsed.componentCount || 6);
			} catch (e) {
				console.error('Failed to load layout preferences:', e);
			}
		}
	},

	saveLayoutPreferences: function() {
		const prefs = {
			componentCount: parseInt($('#buildfuly-component-count').val()),
			noHero: $('#buildfuly-no-hero').is(':checked'),
			noTestimonials: $('#buildfuly-no-testimonials').is(':checked')
		};
		localStorage.setItem('buildfuly_layout_prefs', JSON.stringify(prefs));
	},

	updateWordCount: function() {
		// Function disabled - word count removed from modal
	},

	updateLayoutPreview: function(forceRegenerate = false) {
		const count = parseInt($('#buildfuly-component-count').val());
		const noHero = $('#buildfuly-no-hero').is(':checked');
		const noTestimonials = $('#buildfuly-no-testimonials').is(':checked');
		const randomize = $('#buildfuly-randomize-layout').is(':checked');
		
		// Get page type to determine layout
		const pageType = $('#page-type').val().trim().toLowerCase();
		
		// Generate or use custom layout
		let layout;
		if (randomize && forceRegenerate) {
			// Only regenerate when explicitly forced (like clicking randomize button)
			layout = this.getCustomLayout(pageType, count, noHero, noTestimonials);
			this.customLayout = layout;
		} else if (!this.customLayout || this.customLayout.length === 0) {
			// Generate initial layout
			layout = this.getCustomLayout(pageType, count, noHero, noTestimonials);
			this.customLayout = layout;
		} else if (this.customLayout.length !== count) {
			// Slider changed - adjust component count
			if (this.customLayout.length < count) {
				// Need to add more components
				const toAdd = count - this.customLayout.length;
				const additionalLayout = this.getCustomLayout(pageType, toAdd, noHero, noTestimonials);
				// Add new components before the CTA (last item)
				const cta = this.customLayout.pop(); // Remove CTA temporarily
				this.customLayout = [...this.customLayout, ...additionalLayout];
				if (cta) this.customLayout.push(cta); // Add CTA back at end
			} else {
				// Need to remove components
				// Keep first (hero), last (CTA), remove from middle
				const toRemove = this.customLayout.length - count;
				if (toRemove > 0) {
					const first = this.customLayout[0];
					const last = this.customLayout[this.customLayout.length - 1];
					const middle = this.customLayout.slice(1, this.customLayout.length - 1);
					const keepMiddle = middle.slice(0, middle.length - toRemove);
					this.customLayout = [first, ...keepMiddle, last].filter(Boolean).slice(0, count);
				}
			}
			layout = this.customLayout;
		} else {
			// Use existing custom layout (allows manual edits even with randomize on)
			layout = this.customLayout;
		}
		
		// Calculate word estimate using cached word counts from API
		let totalWords = 0;
		layout.forEach(component => {
			// Use cached word count from API if available, otherwise use fallback
			if (this.componentWordCounts && this.componentWordCounts[component]) {
				totalWords += this.componentWordCounts[component];
			} else {
				// Fallback estimates
				totalWords += 250; // Default average
			}
		});
		
		$('#buildfuly-word-estimate').text(`~${totalWords.toLocaleString()}`);
		
		// Update count label and slider to match actual layout length
		$('#buildfuly-component-count-label').text(layout.length);
		$('#buildfuly-component-count').val(layout.length);
		
		// Save preferences
		this.saveLayoutPreferences();
		
		// Render layout preview with drag-and-drop
		const $preview = $('#buildfuly-layout-preview');
		$preview.empty();
		
		const componentNames = {
			'hero-split': 'Hero Section',
			'hero-wide': 'Hero Section',
			'hero': 'Hero Section',
			'random-hero': '🎲 Random Hero',
			'how-it-works': 'How It Works',
			'content-split': 'Content Feature',
			'features-grid': 'Features Grid',
			'feature-showcase': 'Feature Showcase',
			'feature-cards': 'Feature Cards',
			'benefits': 'Benefits',
			'stats': 'Stats',
			'testimonial': 'Testimonials',
			'testimonials-2': 'Testimonials',
			'faq': 'FAQ',
			'pricing': 'Pricing',
			'bento-grid': 'Bento Grid',
			'image-grid': 'Image Gallery',
			'process': 'Process Steps',
			'random-content': '🎲 Random Content',
			'cta-banner': 'Call to Action',
			'cta': 'Call to Action',
			'random-cta': '🎲 Random CTA'
		};
		
		// Check if randomize is enabled
		const isRandomizeEnabled = $('#buildfuly-randomize-layout').is(':checked');
		
		layout.forEach((component, index) => {
			let name = componentNames[component] || component;
			
			// Replace specific component names with "Random X" when randomize is enabled
			if (isRandomizeEnabled) {
				if (component.includes('hero') && !component.includes('random')) {
					name = '🎲 Random Hero';
				} else if (component.includes('cta') && !component.includes('random')) {
					name = '🎲 Random CTA';
				} else if (!component.includes('hero') && !component.includes('cta') && !component.includes('random')) {
					name = '🎲 Random Content';
				}
			}
			
			const icon = index === 0 ? 'star-filled' : (component.includes('cta') ? 'megaphone' : 'marker');
			
			const $item = $(`
				<div class="layout-preview-item" draggable="true" data-component="${component}" data-index="${index}" style="display: flex; align-items: center; gap: 8px; padding: 8px 12px; background: #f9fafb; border-radius: 4px; border-left: 3px solid ${index === 0 ? '#2563eb' : '#94a3b8'}; cursor: move;">
					<span class="dashicons dashicons-menu" style="font-size: 16px; color: #94a3b8;"></span>
					<span class="dashicons dashicons-${icon}" style="font-size: 16px; color: #64748b;"></span>
					<span style="font-size: 13px; color: #1e293b; flex: 1;">${name}</span>
					<button class="remove-component-btn" data-index="${index}" style="background: none; border: none; cursor: pointer; color: #94a3b8; padding: 4px;" title="Remove">
						<span class="dashicons dashicons-no-alt" style="font-size: 16px;"></span>
					</button>
				</div>
			`);
			
			$preview.append($item);
		});
		
		// Add component dropdown and randomize button (only once)
		if ($('#layout-component-select').length === 0) {
			const $controls = $('<div style="margin-top: 12px; display: flex; flex-direction: column; gap: 8px;"></div>');
			
			// Component dropdown
			const $select = $('<select id="layout-component-select" style="padding: 6px; border: 1px solid #e5e7eb; border-radius: 4px; font-size: 13px;"></select>');
			$select.append('<option value="">Add Component...</option>');
			
			// Populate from cached API components
			if (this.availableComponents) {
				if (this.availableComponents.hero) {
					const $heroGroup = $('<optgroup label="Hero Sections"></optgroup>');
					$heroGroup.append(`<option value="random-hero">🎲 Random Hero</option>`);
					for (const [slug, component] of Object.entries(this.availableComponents.hero)) {
						if (!slug.startsWith('cta')) {
							$heroGroup.append(`<option value="${slug}">${component.name || slug}</option>`);
						}
					}
					$select.append($heroGroup);
				}
				
				if (this.availableComponents.content) {
					const $contentGroup = $('<optgroup label="Content Sections"></optgroup>');
					$contentGroup.append(`<option value="random-content">🎲 Random Content</option>`);
					for (const [slug, component] of Object.entries(this.availableComponents.content)) {
						$contentGroup.append(`<option value="${slug}">${component.name || slug}</option>`);
					}
					$select.append($contentGroup);
				}
				
				if (this.availableComponents.cta) {
					const $ctaGroup = $('<optgroup label="Call to Actions"></optgroup>');
					$ctaGroup.append(`<option value="random-cta">🎲 Random CTA</option>`);
					for (const [slug, component] of Object.entries(this.availableComponents.cta)) {
						$ctaGroup.append(`<option value="${slug}">${component.name || slug}</option>`);
					}
					$select.append($ctaGroup);
				}
				
				if (this.availableComponents.form) {
					const $formGroup = $('<optgroup label="Forms"></optgroup>');
					for (const [slug, component] of Object.entries(this.availableComponents.form)) {
						$formGroup.append(`<option value="${slug}">${component.name || slug}</option>`);
					}
					$select.append($formGroup);
				}
			}
			
			// Randomize button
			const $randomizeBtn = $(`
				<button id="layout-randomize-btn" class="button button-secondary" style="width: 100%;">
					<span class="dashicons dashicons-update" style="margin-top: 4px;"></span> Randomize Layout
				</button>
			`);
			
			$controls.append($select);
			$controls.append($randomizeBtn);
			$preview.parent().append($controls);
		}
		
		// Bind drag and drop events
		this.bindLayoutDragDrop();
		
		// Bind remove buttons - allow deletion of all components including random
		$('.remove-component-btn').off('click').on('click', (e) => {
			const index = parseInt($(e.currentTarget).data('index'));
			this.customLayout.splice(index, 1);
			// Update slider to match new count
			$('#buildfuly-component-count').val(this.customLayout.length);
			this.updateLayoutPreview();
		});
		
		// Bind component dropdown
		$('#layout-component-select').off('change').on('change', (e) => {
			const component = $(e.target).val();
			
			if (component) {
				// Always allow adding components, regardless of randomize state
				this.customLayout.push(component);
				// Update slider to match new count
				$('#buildfuly-component-count').val(this.customLayout.length);
				this.updateLayoutPreview();
				$(e.target).val(''); // Reset dropdown
			}
		});
		
		// Bind randomize button
		$('#layout-randomize-btn').off('click').on('click', () => {
			this.updateLayoutPreview(true); // Force regenerate
		});
	},
	
	bindLayoutDragDrop: function() {
		const items = document.querySelectorAll('.layout-preview-item');
		let draggedItem = null;
		
		items.forEach(item => {
			item.addEventListener('dragstart', (e) => {
				draggedItem = item;
				item.style.opacity = '0.5';
			});
			
			item.addEventListener('dragend', (e) => {
				item.style.opacity = '1';
			});
			
			item.addEventListener('dragover', (e) => {
				e.preventDefault();
			});
			
			item.addEventListener('drop', (e) => {
				e.preventDefault();
				if (draggedItem !== item) {
					const draggedIndex = parseInt(draggedItem.dataset.index);
					const targetIndex = parseInt(item.dataset.index);
					
					// Reorder the layout
					const [removed] = this.customLayout.splice(draggedIndex, 1);
					this.customLayout.splice(targetIndex, 0, removed);
					
					// Re-render
					this.updateLayoutPreview();
				}
			});
		});
	},

	getCustomLayout: function(pageType, count, noHero, noTestimonials) {
		// Helper to normalize component type (remove variants)
		const getComponentType = (slug) => {
			// First strip numbers and common suffixes
			let normalized = slug.replace(/-\d+$/, '').replace(/-(split|wide|banner|showcase|cards|grid)$/, '');
			// Normalize plural forms (testimonials -> testimonial)
			normalized = normalized.replace(/s$/, '');
			return normalized;
		};
		
		// Start with base layout from getDefaultLayout
		let baseLayout = this.getDefaultLayout(pageType);
		
		// Remove hero if requested
		if (noHero) {
			baseLayout = baseLayout.filter(c => !c.includes('hero'));
		}
		
		// Remove testimonials if requested
		if (noTestimonials) {
			baseLayout = baseLayout.filter(c => !c.includes('testimonial'));
		}
		
		// Adjust to desired count
		if (baseLayout.length > count) {
			// Keep first (hero if exists), last (CTA), and most important middle components
			const first = baseLayout[0];
			const last = baseLayout[baseLayout.length - 1];
			const middle = baseLayout.slice(1, baseLayout.length - 1);
			
			// Calculate how many middle components we can keep
			const middleCount = count - 2; // Reserve for first and last
			const selectedMiddle = middle.slice(0, Math.max(0, middleCount));
			
			baseLayout = [first, ...selectedMiddle, last].filter(Boolean);
		} else if (baseLayout.length < count) {
			// Track used component types to prevent duplicates
			const usedTypes = baseLayout.map(c => getComponentType(c));
			
			// Get available content components that aren't already used
			const availableComponents = [];
			if (this.availableComponents && this.availableComponents.content) {
				Object.keys(this.availableComponents.content).forEach(slug => {
					const baseType = getComponentType(slug);
					if (!usedTypes.includes(baseType)) {
						availableComponents.push(slug);
					}
				});
			}
			
			// Shuffle for randomness
			const shuffled = availableComponents.sort(() => 0.5 - Math.random());
			
			// Add components to reach desired count
			const toAdd = count - baseLayout.length;
			for (let i = 0; i < toAdd && i < shuffled.length; i++) {
				const component = shuffled[i];
				const baseType = getComponentType(component);
				
				// Only add if not duplicate
				if (!usedTypes.includes(baseType)) {
					// Insert before CTA (last position)
					baseLayout.splice(baseLayout.length - 1, 0, component);
					usedTypes.push(baseType);
				}
			}
		}
		
		return baseLayout.slice(0, count);
	},	createPage: function(e) {
		e.preventDefault();
		
		if (!this.canGenerate()) {
			return;
		}
		
		const pageDescription = $('#page-type').val().trim(); // Description for content generation
		const pageTitle = $('#page-title-input').val().trim() || this.generatePageTitle(pageDescription); // Proper page title
		const keywordsStr = $('#page-keywords').val().trim();
		const pageImagesIds = $('#buildfuly-page-images-ids').val();
		
		// Auto-generate images options
		const autoGenerateImages = $('#buildfuly-auto-generate-images').is(':checked');
		const autoPickImages = $('#buildfuly-auto-pick-images').is(':checked');
		const includeLogoInImages = $('#buildfuly-include-logo-images').is(':checked');
		
		if (!pageDescription) {
			this.showMessage('Please describe the page you want to build', 'warning');
			return;
		}
		
		const keywordsArray = keywordsStr ? keywordsStr.split(',').map(k => k.trim()).filter(k => k) : [];
		
		this.pageImages = pageImagesIds ? pageImagesIds.split(',').map(id => parseInt(id)) : [];
		
		// Use custom layout if user has modified it, otherwise generate default
		let customLayout;
		if (this.customLayout && this.customLayout.length > 0) {
			customLayout = this.customLayout;
		} else {
			const count = parseInt($('#buildfuly-component-count').val()) || 6;
			const noHero = $('#buildfuly-no-hero').is(':checked');
			const noTestimonials = $('#buildfuly-no-testimonials').is(':checked');
			customLayout = this.getCustomLayout(pageDescription.toLowerCase(), count, noHero, noTestimonials);
		}
		
		const isGeneratingExistingPage = this.currentPage && 
			(!this.currentPage.sections || this.currentPage.sections.length === 0) &&
			(this.currentPage.name === pageTitle || this.currentPage.title === pageTitle);
		
		this.hideNewPageModal();
		
		// Reset customLayout for next time
		this.customLayout = null;
		
		// Store image generation options for the page generation flow
		this.imageGenerationOptions = autoGenerateImages ? {
			enabled: true,
			autoPick: autoPickImages,
			includeLogo: includeLogoInImages
		} : null;
		
		// Pass both pageTitle (for naming) and pageDescription (for content)
		this.generatePageWithLayout(pageTitle, keywordsArray, customLayout, isGeneratingExistingPage, pageDescription);
	},
	
	getDefaultLayout: function(pageType) {
		const type = pageType.toLowerCase();
		
		// Helper to normalize component type (remove variants like -2, -split, etc.)
		const getComponentType = (slug) => {
			// Remove numeric suffixes and variants
			let normalized = slug.replace(/-\d+$/, '').replace(/-(split|wide|banner|showcase|cards|grid)$/, '');
			// Normalize plural forms (testimonials -> testimonial)
			normalized = normalized.replace(/s$/, '');
			return normalized;
		};
		
		// Helper to get a random component from a type, avoiding duplicates
		const getRandomComponent = (componentType, usedTypes = []) => {
			if (this.availableComponents && this.availableComponents[componentType]) {
				const components = Object.keys(this.availableComponents[componentType])
					.filter(slug => {
						const baseType = getComponentType(slug);
						return !usedTypes.includes(baseType);
					});
				
				if (components.length === 0) return null;
				return components[Math.floor(Math.random() * components.length)];
			}
			return null;
		};
		
		// Get a hero component (not CTA)
		const getHero = () => {
			if (this.availableComponents && this.availableComponents.hero) {
				const heroComponents = Object.keys(this.availableComponents.hero)
					.filter(slug => !slug.startsWith('cta'));
				return heroComponents[Math.floor(Math.random() * heroComponents.length)] || null;
			}
			return null;
		};
		
		// AI-like logic to determine best sections using real component slugs
		const usedTypes = [];
		const hero = getHero();
		if (hero) usedTypes.push(getComponentType(hero));
		
		const cta = getRandomComponent('cta', usedTypes);
		if (cta) usedTypes.push(getComponentType(cta));
		
		// Get content components without duplicates
		const getContentComponents = (count) => {
			if (!this.availableComponents || !this.availableComponents.content) {
				return [];
			}
			const contentSlugs = Object.keys(this.availableComponents.content)
				.filter(slug => {
					const baseType = getComponentType(slug);
					return !usedTypes.includes(baseType);
				});
			
			const shuffled = contentSlugs.sort(() => 0.5 - Math.random());
			const selected = [];
			
			for (let i = 0; i < Math.min(count, shuffled.length); i++) {
				const component = shuffled[i];
				const baseType = getComponentType(component);
				
				// Double check not already used
				if (!usedTypes.includes(baseType)) {
					selected.push(component);
					usedTypes.push(baseType);
				}
			}
			
			return selected;
		};
		
		if (type.includes('home') || type.includes('landing')) {
			const content = getContentComponents(3);
			return [hero, ...content, cta].filter(Boolean);
		} else if (type.includes('about')) {
			const content = getContentComponents(3);
			return [hero, ...content, cta].filter(Boolean);
		} else if (type.includes('service') || type.includes('product')) {
			const content = getContentComponents(3);
			return [hero, ...content, cta].filter(Boolean);
		} else if (type.includes('contact')) {
			const content = getContentComponents(2);
			return [hero, ...content, cta].filter(Boolean);
		} else if (type.includes('pricing')) {
			const content = getContentComponents(3);
			return [hero, ...content, cta].filter(Boolean);
		} else if (type.includes('testimonial') || type.includes('review')) {
			const content = getContentComponents(2);
			return [hero, ...content, cta].filter(Boolean);
		} else {
			// Default varied layout
			const content = getContentComponents(3);
			return [hero, ...content, cta].filter(Boolean);
		}
	},

	resolveRandomComponents: function(sections) {
		// Helper to normalize component type (remove variants)
		const getComponentType = (slug) => {
			let normalized = slug.replace(/-\d+$/, '').replace(/-(split|wide|banner|showcase|cards|grid)$/, '');
			// Normalize plural forms (testimonials -> testimonial)
			normalized = normalized.replace(/s$/, '');
			return normalized;
		};
		
		// Track ALL component types (including non-random ones already in layout)
		const usedTypes = [];
		const resolvedSections = [];
		
		// First pass: track all non-random component types
		for (const sectionType of sections) {
			if (!sectionType.startsWith('random-')) {
				const baseType = getComponentType(sectionType);
				usedTypes.push(baseType);
			}
		}
		
		// Second pass: resolve random components avoiding all used types
		for (const sectionType of sections) {
			let actualComponent = sectionType;
			
			// Handle random-* types
			if (sectionType === 'random-hero') {
				// Select random hero (excluding CTA), avoiding duplicates
				if (this.availableComponents && this.availableComponents.hero) {
					const heroComponents = Object.keys(this.availableComponents.hero)
						.filter(slug => {
							if (slug.startsWith('cta')) return false;
							const baseType = getComponentType(slug);
							return !usedTypes.includes(baseType);
						});
					
					if (heroComponents.length > 0) {
						actualComponent = heroComponents[Math.floor(Math.random() * heroComponents.length)];
						console.log('🎲 Resolved random-hero to:', actualComponent);
						// Mark this type as used
						usedTypes.push(getComponentType(actualComponent));
					} else {
						console.warn('No unique hero components available, using fallback');
						actualComponent = 'hero-split';
					}
				}
			} else if (sectionType === 'random-content') {
				// Select random content component, avoiding duplicates
				if (this.availableComponents && this.availableComponents.content) {
					const contentComponents = Object.keys(this.availableComponents.content)
						.filter(slug => {
							const baseType = getComponentType(slug);
							return !usedTypes.includes(baseType);
						});
					
					if (contentComponents.length > 0) {
						actualComponent = contentComponents[Math.floor(Math.random() * contentComponents.length)];
						console.log('🎲 Resolved random-content to:', actualComponent);
						// Mark this type as used
						usedTypes.push(getComponentType(actualComponent));
					} else {
						console.warn('No unique content components available, using fallback');
						actualComponent = 'features-grid';
					}
				}
			} else if (sectionType === 'random-cta') {
				// Select random CTA, avoiding duplicates
				if (this.availableComponents && this.availableComponents.cta) {
					const ctaComponents = Object.keys(this.availableComponents.cta)
						.filter(slug => {
							const baseType = getComponentType(slug);
							return !usedTypes.includes(baseType);
						});
					
					if (ctaComponents.length > 0) {
						actualComponent = ctaComponents[Math.floor(Math.random() * ctaComponents.length)];
						console.log('🎲 Resolved random-cta to:', actualComponent);
						// Mark this type as used
						usedTypes.push(getComponentType(actualComponent));
					} else {
						console.warn('No unique CTA components available, using fallback');
						actualComponent = 'cta-banner';
					}
				}
			}
			
			resolvedSections.push(actualComponent);
		}
		
		console.log('Final resolved sections:', resolvedSections);
		console.log('Used component types:', usedTypes);
		
		return resolvedSections;
	},

generatePageWithLayout: function(pageTitle, keywords, sections, useExistingPage = false, pageDescription = null) {
	// pageTitle = The display name (e.g., "Home", "About Us")
	// pageDescription = The original description for content generation (e.g., "homepage for plumbing services")
	// If pageDescription not provided, use pageTitle
	const contentContext = pageDescription || pageTitle;
	
	// Resolve any random-* component types to actual components (avoiding duplicates)
	sections = this.resolveRandomComponents(sections);
	
	let page;
	let pageId;
	
	if (useExistingPage && this.currentPage) {
		// Use the existing page and update it
		page = this.currentPage;
		pageId = page.id;
		
		// Only update keywords and sections, preserve the original title/name
		page.keywords = keywords;
		page.sections = [];
		$('#buildfuly-sections-container').empty();
		
		// Keep the original title - don't overwrite with pageTitle
		// pageTitle parameter is just for layout generation, not for updating the title
	} else {
		// Create a new page
		pageId = 'page-' + Date.now();
		page = {
			id: pageId,
			name: pageTitle,  // Keep the actual page name from user input
			title: pageTitle, // Display title (proper name like "Home", "About Us")
			keywords: keywords,
			sections: [],
			colors: this.themeColors || this.generateThemeColors(),
			created: new Date().toISOString(),
			images: this.pageImages || [],
			wordpress_id: null, // WordPress page ID (set when published)
			seo: null // SEO metadata (will be populated from API)
		};
		
		this.currentPage = page;
		this.pages.push(page);
	}
	
	this.currentImageIndex = 0; // Track which image to use next		// Generate theme colors if first page
		if (!this.themeColors) {
			this.themeColors = page.colors;
			this.saveThemeColors();
			this.applyThemeColors();
		}
		
		this.showLoading(`Generating ${sections.length} SEO-optimized sections...`);
		
		// Add skeleton placeholders for all sections
		const $container = $('#buildfuly-sections-container');
		$container.empty();
		sections.forEach((type, index) => {
			const skeletonId = 'skeleton-' + index;
			const $skeleton = this.createSkeletonPlaceholder(skeletonId);
			$container.append($skeleton);
		});
		
	// Smart keyword distribution across sections
	// Hero uses first keyword, then rotate through others
	let keywordIndex = 0;
	
	// All keywords joined for AI to use naturally
	const allKeywords = Array.isArray(keywords) ? keywords.join(', ') : keywords;
	
	// Generate all sections in parallel for faster page creation
	const sectionPromises = sections.map((sectionType, index) => {
		return new Promise((resolve) => {
			const skeletonId = 'skeleton-' + index;
			
			// Small stagger to avoid overwhelming the server (100ms * index)
			setTimeout(() => {
				this.generateSection(allKeywords, sectionType, (sectionData) => {
					// Check if generation was stopped due to rate limit
					if (sectionData === 'rate_limit_stop') {
						resolve({ stopped: true, index: index });
						return;
					}
					
					// Replace skeleton with actual content
					const $skeleton = $(`#${skeletonId}`);
					
					if (sectionData && sectionData.html) {
						// Success: Add to page data with index for proper ordering
						const sectionId = 'section-' + Date.now() + '-' + index;
						sectionData.id = sectionId;
						sectionData.keyword = allKeywords;
						sectionData.order = index; // Store the intended order
						
						if ($skeleton.length) {
							const $section = this.createSectionElement(sectionData);
							$skeleton.replaceWith($section);
						} else {
							// Fallback: just add to page
							this.addSectionToPage(sectionData);
						}
						
						resolve({ success: true, data: sectionData, index: index });
					} else {
						// Failed: Remove skeleton and show warning
						console.warn(`BuildfulyAI: Skipping failed component: ${sectionType}`);
						if ($skeleton.length) {
							$skeleton.remove();
						}
						resolve({ success: false, index: index });
					}
				});
			}, index * 100); // Stagger by 100ms each
		});
	});
	
	// Wait for all sections to complete
	Promise.all(sectionPromises).then(async (results) => {
		// Check if any section was stopped due to rate limit
		const rateLimitStopped = results.some(r => r.stopped);
		if (rateLimitStopped) {
			this.hideLoading();
			return;
		}
		
		// Sort successful sections by their original index and add to page in order
		const successfulSections = results
			.filter(r => r.success && r.data)
			.sort((a, b) => a.index - b.index)
			.map(r => r.data);
		
		// Add sections to page in the correct order
		this.currentPage.sections = successfulSections;
		
		// Count successful sections
		const successCount = successfulSections.length;
		
		// Check if we should auto-generate images
		if (this.imageGenerationOptions && this.imageGenerationOptions.enabled) {
			console.log('[AI Images] Auto-generate enabled:', this.imageGenerationOptions);
			console.log('[AI Images] Page description:', contentContext);
			console.log('[AI Images] Sections count:', successfulSections.length);
			console.log('[AI Images] Sections:', successfulSections.map(s => ({ id: s.id, type: s.type })));
			
			this.showLoading('Generating AI images for your page...');
			
			try {
				await this.generateImagesForPage(contentContext, successfulSections, this.imageGenerationOptions);
			} catch (error) {
				console.error('Auto image generation error:', error);
				this.showMessage('Some images could not be generated', 'warning');
			}
			
			// Clear options
			this.imageGenerationOptions = null;
		}
		
		// All sections processed (success or failure)
		this.hideLoading();
		
		// Use pending SEO data from API if available, otherwise generate client-side
		if (this.pendingSeoData) {
			console.log('Using SEO from API:', this.pendingSeoData);
			page.seo = this.pendingSeoData;
			this.pendingSeoData = null; // Clear it
			
			this.savePage();
			this.renderPagesList();
			this.loadPage(pageId);
			
			const failedCount = sections.length - successCount;
			if (failedCount > 0) {
				this.showMessage(`Page "${pageTitle}" created with ${successCount} sections (${failedCount} skipped due to errors)`, 'success');
			} else {
				this.showMessage(`Page "${pageTitle}" created successfully with ${successCount} sections!`, 'success');
			}
		} else {
			// Fallback to client-side SEO generation
			this.generateSeoForPage(page, () => {
				this.savePage();
				this.renderPagesList();
				this.loadPage(pageId);
				
				const failedCount = sections.length - successCount;
				if (failedCount > 0) {
					this.showMessage(`Page "${pageTitle}" created with ${successCount} sections (${failedCount} skipped due to errors)`, 'success');
				} else {
					this.showMessage(`Page "${pageTitle}" created successfully with ${successCount} sections!`, 'success');
				}
			});
		}
	});
},

generateSection: function(keyword, sectionType, callback) {
		console.log('BuildfulyAI: Generating section:', sectionType, 'Keyword:', keyword);
		
		const requestData = {
			keywords: keyword || '',
			section_type: sectionType
		};
		
		// Intelligently distribute images across sections
		if (this.pageImages && this.pageImages.length > 0 && this.currentImageIndex < this.pageImages.length) {
			// Use images for visual sections: hero, features, benefits, testimonial
			const visualSections = ['hero', 'features', 'benefits', 'testimonial', 'about', 'feature-showcase', 'split-content', 'image-grid', 'feature-cards', 'bento-grid'];
			if (visualSections.includes(sectionType)) {
				requestData.image_id = this.pageImages[this.currentImageIndex];
				this.currentImageIndex++;
			}
		}
		
		this.generateWithCommonLogic({
			url: buildfulyAdmin.restUrl + '/content/generate-section',
			data: requestData,
			loadingMessage: null, // Suppress loading for batch operations
			successMessage: null, // Suppress success message for batch operations
			skipCommonTasks: true, // Handled by callback/batch operation
			onSuccess: (response) => {
				// Check if using template fallback
				if (response.data.html && response.data.html.includes('Launch Faster')) {
					console.warn('BuildfulyAI: ⚠️ Using template content. Check your OpenAI API key has billing enabled at https://platform.openai.com/account/billing');
				}
				
				if (callback) {
					// Pass section data to callback for skeleton replacement
					callback(response.data);
				} else {
					// No callback, add section directly (backward compatibility)
					this.addSectionToPage(response.data);
				}
			},
			onError: (xhr, status, error) => {
				console.error('BuildfulyAI: AJAX Error:', status, error);
				
				// Check if rate limit exceeded (429)
				if (xhr.status === 429) {
					this.hideLoading();
					// Stop generation completely - don't continue to next component
					if (callback) callback('rate_limit_stop');
					return;
				}
				
				// Check if this is a skippable component error
				if (xhr.responseJSON && xhr.responseJSON.skip) {
					console.warn('BuildfulyAI: Component failed (skipping):', xhr.responseJSON.message);
				}
				
				// Always call callback to continue with next section
				if (callback) callback(null);
			}
		});
	},

	addSectionToPage: function(sectionData, insertAfterSectionId = null) {
		// Safety check: don't add null sections (but allow empty html for custom-code)
		if (!sectionData) {
			console.warn('BuildfulyAI: Skipping null section data');
			return;
		}
		
		// For non-custom-code sections, require html content
		if (sectionData.type !== 'custom-code' && !sectionData.html) {
			console.warn('BuildfulyAI: Skipping section with no HTML');
			return;
		}
		
		const sectionId = 'section-' + Date.now();
		sectionData.id = sectionId;
		
		// Insert at specific position or append to end
		if (insertAfterSectionId) {
			const insertIndex = this.currentPage.sections.findIndex(s => s.id === insertAfterSectionId);
			if (insertIndex !== -1) {
				this.currentPage.sections.splice(insertIndex + 1, 0, sectionData);
			} else {
				this.currentPage.sections.push(sectionData);
			}
		} else if (insertAfterSectionId === '') {
			// Empty string means insert at top
			this.currentPage.sections.unshift(sectionData);
		} else {
			this.currentPage.sections.push(sectionData);
		}
		
		const $section = this.createSectionElement(sectionData);
		
		// Insert at correct position in DOM
		if (insertAfterSectionId) {
			const $afterSection = $(`#${insertAfterSectionId}`);
			if ($afterSection.length) {
				const $afterDivider = $afterSection.next('.buildfuly-section-divider');
				if ($afterDivider.length) {
					$afterDivider.after($section);
				} else {
					$afterSection.after($section);
				}
			} else {
				$('#buildfuly-sections-container').append($section);
			}
		} else if (insertAfterSectionId === '') {
			// Insert at top
			const $firstDivider = $('#buildfuly-sections-container .buildfuly-section-divider').first();
			if ($firstDivider.length) {
				$firstDivider.after($section);
			} else {
				$('#buildfuly-sections-container').prepend($section);
			}
		} else {
			$('#buildfuly-sections-container').append($section);
		}
		
		// Re-render dividers
		this.renderSectionDividers();
		
		// Update buttons in newly added section
		setTimeout(() => {
			this.updateButtonsFromSettings();
		}, 100);
	},
	
	/**
	 * Render dividers between sections with add buttons
	 */
	renderSectionDividers: function() {
		$('.buildfuly-section-divider').remove();
		
		const $container = $('#buildfuly-sections-container');
		const $sections = $container.find('.buildfuly-section');
		
		const self = this;
		
		// If no sections, add a divider at the top to allow adding first section
		if ($sections.length === 0) {
			$container.prepend(this.createSectionDivider(''));
			return;
		}
		
		// Add divider before first section (to insert at top)
		$sections.first().before(this.createSectionDivider(''));
		
		// Add divider after each section
		$sections.each(function() {
			$(this).after(self.createSectionDivider($(this).attr('id')));
		});
	},
	
	/**
	 * Create a section divider element with add button
	 */
	createSectionDivider: function(afterSectionId) {
		const self = this;
		const $divider = $(`
			<div class="buildfuly-section-divider" data-after-section="${afterSectionId}">
				<div class="buildfuly-section-divider-line"></div>
				<button class="buildfuly-section-add-btn" title="Add Section">
					<span class="dashicons dashicons-plus"></span>
				</button>
			</div>
		`);
		
		// Click adds section using the type from the toolbar dropdown
		$divider.find('.buildfuly-section-add-btn').on('click', function(e) {
			e.stopPropagation();
			const sectionType = $('#buildfuly-section-type').val() || 'random';
			self.generateAndAddSection(sectionType, afterSectionId);
		});
		
		return $divider;
	},
	
	/**
	 * Shared function to generate and add a section (used by toolbar and divider dropdown)
	 * @param {string} sectionType - Component type to generate
	 * @param {string|null} insertAfterSectionId - Section ID to insert after, '' for top, null for end
	 */
	generateAndAddSection: function(sectionType, insertAfterSectionId = null) {
		if (!this.currentPage) {
			this.showMessage('Please create or select a page first', 'warning');
			return;
		}
		
		// Handle custom code section
		if (sectionType === 'custom-code') {
			this.addSectionToPage({ type: 'custom-code', html: '' }, insertAfterSectionId);
			this.savePage();
			this.showMessage('Custom code section added! Click Edit to add your code.', 'success');
			return;
		}
		
		// Get keywords - prefer checked, fallback to first page keyword
		const checkedKeywords = [];
		$('.buildfuly-keyword-checkbox:checked').each(function() {
			checkedKeywords.push($(this).data('keyword'));
		});
		
		let keyword;
		if (checkedKeywords.length > 0) {
			keyword = checkedKeywords.join(', ');
		} else if (this.currentPage.keywords && this.currentPage.keywords.length > 0) {
			const kw = this.currentPage.keywords[0];
			keyword = typeof kw === 'string' ? kw : (kw.keyword || 'business');
		} else {
			keyword = this.currentPage.keyword || 'business';
		}
		
		// PRE-BUDGET CHECK
		if (this.usageStats && this.componentWordEstimates) {
			const estimatedWords = this.componentWordEstimates[sectionType] || this.componentWordEstimates['random'] || 200;
			const remainingWords = this.usageStats.remainingWords || 0;
			
			if (remainingWords < estimatedWords) {
				this.showMessage(
					`Insufficient credits. Need ~${estimatedWords} words but only ${remainingWords} remaining.`,
					'error'
				);
				return;
			}
		}
		
		const imageId = $('#buildfuly-section-image-id').val();
		
		// Add skeleton at position
		const skeletonId = 'skeleton-' + Date.now();
		const $skeleton = this.createSkeletonPlaceholder(skeletonId);
		
		if (insertAfterSectionId) {
			const $afterSection = $(`#${insertAfterSectionId}`);
			const $afterDivider = $afterSection.next('.buildfuly-section-divider');
			($afterDivider.length ? $afterDivider : $afterSection).after($skeleton);
		} else if (insertAfterSectionId === '') {
			$('#buildfuly-sections-container').prepend($skeleton);
		} else {
			$('#buildfuly-sections-container').append($skeleton);
		}
		
		this.showLoading(`Generating ${sectionType === 'random' ? 'section' : sectionType}...`);
		
		const self = this;
		
		$.ajax({
			url: buildfulyAdmin.restUrl + '/content/generate-section',
			method: 'POST',
			beforeSend: xhr => xhr.setRequestHeader('X-WP-Nonce', buildfulyAdmin.nonce),
			data: JSON.stringify({ keywords: keyword, section_type: sectionType, image_id: imageId }),
			contentType: 'application/json',
			success: function(response) {
				$(`#${skeletonId}`).remove();
				
				if (response.success && response.data) {
					response.data.keyword = keyword;
					self.addSectionToPage(response.data, insertAfterSectionId);
					self.savePage();
					self.updateWordCount();
					self.renderKeywordsPanel();
					self.initEncodedScripts();
					self.loadUsageStats();
					
					$('#buildfuly-section-image-id').val('');
					$('#buildfuly-section-image-indicator').hide();
					
					self.hideLoading();
					self.showMessage(response.data.from_cache 
						? '⚠️ Daily limit reached. Here\'s a sample component.' 
						: 'Section added!', 
						response.data.from_cache ? 'warning' : 'success'
					);
				} else {
					self.hideLoading();
					self.showMessage('Failed to generate section', 'error');
				}
			},
			error: function(xhr) {
				$(`#${skeletonId}`).remove();
				self.hideLoading();
				
				if (xhr.status === 429) {
					self.loadUsageStats();
					self.showMessage('Daily limit reached. Please upgrade or wait until tomorrow.', 'error');
				} else {
					self.showMessage(xhr.responseJSON?.message || 'Error generating section', 'error');
				}
			}
		});
	},
	
	addSection: function(e) {
		e.preventDefault();
		
		if (!this.currentPage) {
			this.showMessage('Please create or select a page first', 'warning');
			return;
		}
		
		const sectionType = $('#section-type').val();
		this.showLoading('Generating section...');
		
		// Find keyword with lowest density (SEO optimization)
		let lowestKeyword = null;
		let lowestDensity = Infinity;
		
		if (Array.isArray(this.currentPage.keywords) && this.currentPage.keywords.length > 0) {
			this.currentPage.keywords.forEach(kw => {
				const density = this.calculateKeywordDensity(kw);
				if (density < lowestDensity) {
					lowestDensity = density;
					lowestKeyword = kw;
				}
			});
		}
		
		const keyword = lowestKeyword || this.currentPage.keywords[0] || this.currentPage.keywords;
		
		this.generateSection(keyword, sectionType, () => {
			this.hideLoading();
			this.showMessage(`Section added with "${keyword}"!`, 'success');
			this.updateWordCount();
			this.renderKeywordsPanel();
			this.initEncodedScripts(); // Initialize carousel scripts
			this.savePage();
		});
	},
	
	// Add Custom Code Section directly (no modal) - uses shared function
	addCustomCodeSection: function(e) {
		if (e) e.preventDefault();
		this.generateAndAddSection('custom-code', null);
	},
	
	/**
	 * Create a skeleton placeholder for loading state
	 * Used by both page and component generation
	 */
	createSkeletonPlaceholder: function(skeletonId) {
		return $(`
			<div class="buildfuly-section buildfuly-skeleton" id="${skeletonId}" data-loading="true">
				<div class="buildfuly-skeleton-content">
					<div class="buildfuly-skeleton-header"></div>
					<div class="buildfuly-skeleton-text"></div>
					<div class="buildfuly-skeleton-text"></div>
					<div class="buildfuly-skeleton-text short"></div>
				</div>
			</div>
		`);
	},

	addSectionFromCheckedKeywords: function(e) {
		e.preventDefault();
		
		if (!this.currentPage) {
			this.showMessage('Please create or select a page first', 'warning');
			return;
		}
		
		// Get selected component type
		const sectionType = $('#buildfuly-section-type').val();
		
		if (!sectionType || sectionType === '') {
			this.showMessage('Please select a component type', 'warning');
			return;
		}
		
		// Validate keywords are checked (except for custom-code)
		if (sectionType !== 'custom-code') {
			const hasCheckedKeywords = $('.buildfuly-keyword-checkbox:checked').length > 0;
			if (!hasCheckedKeywords) {
				this.showMessage('Please check at least one keyword', 'warning');
				return;
			}
		}
		
		// Use shared generation function (null = append to end)
		this.generateAndAddSection(sectionType, null);
	},
		createSectionElement: function(sectionData) {
			// Backend PHP components use CSS classes that reference CSS variables
			// No need to apply inline colors - let CSS variables handle it
			let styledHtml = sectionData.html || sectionData.content || '';
			
			// CORRUPTION FIX: Detect and fix orphaned CSS (CSS without <style> tags)
			// This happens when style tags get corrupted but CSS content remains
			// Pattern: CSS rules like ".className {" at the start before any HTML tags
			const orphanedCssPattern = /^(\s*(?:\.[\w-]+\s*\{[^}]*\}[\s\S]*?)+)(<[a-zA-Z])/;
			const orphanedMatch = styledHtml.match(orphanedCssPattern);
			if (orphanedMatch) {
				console.warn('[CORRUPTION FIX] Detected orphaned CSS in section', sectionData.id, '- removing');
				// Remove the orphaned CSS (it's corrupted and we can't recover the proper styles)
				styledHtml = styledHtml.replace(orphanedCssPattern, '$2');
				// Also update the source data to prevent this from persisting
				sectionData.html = styledHtml;
			}
			
			// Custom code sections are NOT contenteditable - force users to use code editor
			const isCustomCode = sectionData.type === 'custom-code';
			const contentEditableAttr = isCustomCode ? 'false' : 'true';
			
			// Extract <style> tags and inject them into <head> to prevent contenteditable corruption
			// Store the original HTML with styles for publishing, but display without inline <style> tags
			let displayHtml = styledHtml;
			const styleRegex = /<style[^>]*>([\s\S]*?)<\/style>/gi;
			const extractedStyles = [];
			let styleMatch;
			
			while ((styleMatch = styleRegex.exec(styledHtml)) !== null) {
				extractedStyles.push(styleMatch[0]);
			}
			
			// Remove <style> tags from display HTML (they'll be in <head>)
			displayHtml = displayHtml.replace(styleRegex, '');
			
			// Inject extracted styles into document head with section ID for cleanup
			if (extractedStyles.length > 0) {
				// Remove any existing styles for this section first
				$(`style[data-section-id="${sectionData.id}"]`).remove();
				
				// Add new style tag to head
				const combinedStyles = extractedStyles.join('\n');
				$('head').append(`<style data-section-id="${sectionData.id}">${combinedStyles.replace(/<\/?style[^>]*>/gi, '')}</style>`);
			}
			
			// Show placeholder for empty custom code sections
			if (isCustomCode && !styledHtml.trim()) {
				displayHtml = `<div class="buildfuly-custom-code-placeholder" style="padding:60px 40px;background:linear-gradient(135deg,#f8fafc 0%,#e2e8f0 100%);border:2px dashed #94a3b8;border-radius:12px;text-align:center;">
					<span class="dashicons dashicons-editor-code" style="font-size:48px;width:48px;height:48px;color:#64748b;margin-bottom:16px;display:block;"></span>
					<p style="color:#64748b;font-size:16px;margin:0;">Click <strong>Edit Code</strong> to add your custom HTML, shortcode, or form</p>
				</div>`;
			}
			
			const $section = $(`
				<div class="buildfuly-section" id="${sectionData.id}" data-type="${sectionData.type}">
					<div class="buildfuly-section-controls">
						<button class="buildfuly-section-btn" data-action="edit" title="${isCustomCode ? 'Edit Code' : 'Edit Content'}">
							<span class="dashicons dashicons-${isCustomCode ? 'editor-code' : 'edit'}"></span>
						</button>
						<button class="buildfuly-section-btn" data-action="regenerate" title="Regenerate" ${isCustomCode ? 'style="display:none;"' : ''}>
							<span class="dashicons dashicons-update"></span>
						</button>
						<button class="buildfuly-section-btn" data-action="move-up" title="Move Up">
							<span class="dashicons dashicons-arrow-up-alt2"></span>
						</button>
						<button class="buildfuly-section-btn" data-action="move-down" title="Move Down">
							<span class="dashicons dashicons-arrow-down-alt2"></span>
						</button>
						<button class="buildfuly-section-btn" data-action="delete" title="Delete">
							<span class="dashicons dashicons-trash"></span>
						</button>
					</div>
					<div class="buildfuly-section-content" contenteditable="${contentEditableAttr}" ${isCustomCode ? 'data-custom-code="true"' : ''}>
						${displayHtml}
					</div>
				</div>
			`);
			
			// Don't apply inline colors - CSS variables handle everything
			
			// Bind events
			$section.find('[data-action="edit"]').on('click', () => this.openEditPanel(sectionData.id));
			$section.find('[data-action="delete"]').on('click', () => this.deleteSection(sectionData.id));
			$section.find('[data-action="regenerate"]').on('click', () => this.regenerateSection(sectionData.id));
			$section.find('[data-action="move-up"]').on('click', () => this.moveSection(sectionData.id, 'up'));
			$section.find('[data-action="move-down"]').on('click', () => this.moveSection(sectionData.id, 'down'));
			
			// Click anywhere on section to open edit panel
			$section.on('click', (e) => {
				// Don't trigger if clicking on a button or link inside the section
				// Don't trigger if currently dragging
				if (!$(e.target).closest('a, button, .buildfuly-section-controls').length && !this.isDragging) {
					this.openEditPanel(sectionData.id);
				}
			});
			
			// Add drag handles to text elements for reordering
			this.addDragHandlesToTextElements($section, sectionData.id);
			
			// Only bind blur/input handlers for non-custom-code sections
			if (sectionData.type !== 'custom-code') {
				$section.find('.buildfuly-section-content').on('blur', () => {
					// Get the edited HTML from contenteditable (this does NOT have <style> tags)
					let editedHtml = $section.find('.buildfuly-section-content').html();
					editedHtml = this.unescapeHtml(editedHtml);
					editedHtml = this.encodeScriptTags(editedHtml);
					
					// Clean up drag handles and wrappers before saving
					const $temp = $('<div>').html(editedHtml);
					$temp.find('.buildfuly-drag-handle').remove();
					$temp.find('.buildfuly-draggable-wrapper').each(function() {
						$(this).replaceWith($(this).contents());
					});
					editedHtml = $temp.html();
					
					// IMPORTANT: Preserve <style> tags from the original section.html
					// The contenteditable div doesn't have them (they're in <head>)
					// We need to get styles from the HEAD since sectionData.html may already be corrupted
					const headStyleEl = document.querySelector(`style[data-section-id="${sectionData.id}"]`);
					let preservedStyles = '';
					if (headStyleEl) {
						preservedStyles = `<style>${headStyleEl.textContent}</style>`;
					}
					
					// If we have styles from head, append them
					if (preservedStyles) {
						sectionData.html = editedHtml + '\n' + preservedStyles;
					} else {
						// Fallback: try to get from original sectionData.html
						const styleRegex = /<style[^>]*>[\s\S]*?<\/style>/gi;
						const originalStyles = sectionData.html?.match(styleRegex) || [];
						if (originalStyles.length > 0) {
							sectionData.html = editedHtml + '\n' + originalStyles.join('\n');
						} else {
							sectionData.html = editedHtml;
						}
					}
					
					this.autoSave();
				});
				
				// Update word count and keyword stats as user types
				$section.find('.buildfuly-section-content').on('input', () => {
					this.updateWordCount();
					this.renderKeywordsPanel();
				});
			} else {
				// For custom-code sections, click on content opens the code editor
				$section.find('.buildfuly-section-content').on('click', () => {
					this.openEditPanel(sectionData.id);
				});
			}
			
			return $section;
	},
	
	/**
	 * Update section display HTML - extracts <style> tags to <head> to prevent contenteditable corruption
	 * @param {string} sectionId - The section ID
	 * @param {string} html - The full HTML including <style> tags
	 */
	updateSectionDisplay: function(sectionId, html) {
		// Extract <style> tags and inject them into <head>
		const styleRegex = /<style[^>]*>([\s\S]*?)<\/style>/gi;
		const extractedStyles = [];
		let styleMatch;
		
		while ((styleMatch = styleRegex.exec(html)) !== null) {
			extractedStyles.push(styleMatch[0]);
		}
		
		// Remove <style> tags from display HTML
		let displayHtml = html.replace(styleRegex, '');
		
		// Update styles in head
		$(`style[data-section-id="${sectionId}"]`).remove();
		if (extractedStyles.length > 0) {
			const combinedStyles = extractedStyles.join('\n');
			const cleanStyles = combinedStyles.replace(/<\/?style[^>]*>/gi, '');
			$('head').append(`<style data-section-id="${sectionId}">${cleanStyles}</style>`);
		}
		
		// Update the display content (without <style> tags)
		$(`#${sectionId} .buildfuly-section-content`).html(displayHtml);
	},
	
	/**
	 * Add drag handles to text elements (headings and paragraphs) for reordering
	 */
	addDragHandlesToTextElements: function($section, sectionId) {
		const self = this;
		const $content = $section.find('.buildfuly-section-content');
		
		// Temporarily disable contenteditable during setup
		const wasEditable = $content.attr('contenteditable');
		
		// Find all direct headings and paragraphs (not nested in other elements)
		const $textElements = $content.find('h1, h2, h3, h4, h5, h6, p').filter(function() {
			// Only get top-level text elements
			return !$(this).closest('button, a.btn, li, .buildfuly-badge').length;
		});
		
		$textElements.each(function(index) {
			const $element = $(this);
			
			// Skip if already wrapped
			if ($element.parent().hasClass('buildfuly-draggable-wrapper')) return;
			
			// Wrap element in a draggable container
			$element.wrap('<div class="buildfuly-draggable-wrapper" style="position: relative; margin: 8px 0;"></div>');
			const $wrapper = $element.parent();
			
			// Add drag handle
			const $handle = $(`
				<div class="buildfuly-drag-handle" style="
					position: absolute;
					left: -40px;
					top: 0;
					width: 32px;
					height: 100%;
					display: flex;
					align-items: center;
					justify-content: center;
					cursor: grab;
					opacity: 0;
					transition: opacity 0.2s;
					background: rgba(0, 113, 227, 0.1);
					border-radius: 6px;
					color: #0071e3;
					font-size: 18px;
					font-weight: bold;
					user-select: none;
				" title="Drag to reorder">
					⋮⋮
				</div>
			`);
			
			$wrapper.prepend($handle);
			
			// Show handle on hover
			$wrapper.on('mouseenter', function() {
				$handle.css('opacity', '0.8');
			});
			
			$wrapper.on('mouseleave', function() {
				if (!self.isDragging) {
					$handle.css('opacity', '0');
				}
			});
			
			// Make wrapper draggable via the handle
			$handle.on('mousedown', function(e) {
				e.preventDefault();
				e.stopPropagation();
				
				self.isDragging = true;
				$content.attr('contenteditable', 'false'); // Disable contenteditable during drag
				
				const $dragging = $wrapper;
				const startY = e.pageY;
				const startTop = $dragging.position().top;
				
				// Clone for visual feedback
				const $ghost = $dragging.clone().css({
					position: 'absolute',
					left: $dragging.offset().left,
					top: $dragging.offset().top,
					width: $dragging.outerWidth(),
					opacity: 0.7,
					zIndex: 10000,
					pointerEvents: 'none',
					background: 'white',
					border: '2px solid #0071e3',
					borderRadius: '8px'
				});
				
				$('body').append($ghost);
				$dragging.css('opacity', '0.3');
				$handle.css('cursor', 'grabbing');
				
				// Get all draggable wrappers
				const $allWrappers = $content.find('.buildfuly-draggable-wrapper');
				
				$(document).on('mousemove.drag', function(e) {
					e.preventDefault();
					
					// Move ghost
					const deltaY = e.pageY - startY;
					$ghost.css('top', startTop + deltaY + $dragging.parent().offset().top);
					
					// Find drop target
					$allWrappers.each(function() {
						const $target = $(this);
						if ($target[0] === $dragging[0]) return;
						
						const rect = this.getBoundingClientRect();
						const midpoint = rect.top + (rect.height / 2);
						
						// Remove all indicators
						$target.css('border-top', '');
						$target.css('border-bottom', '');
						
						// Check if mouse is over this element
						if (e.pageY >= rect.top && e.pageY <= rect.bottom) {
							// Show indicator
							if (e.pageY < midpoint) {
								$target.css('border-top', '3px solid #0071e3');
							} else {
								$target.css('border-bottom', '3px solid #0071e3');
							}
						}
					});
				});
				
				$(document).on('mouseup.drag', function(e) {
					e.preventDefault();
					$(document).off('mousemove.drag mouseup.drag');
					
					// Find drop target
					let dropTarget = null;
					let insertBefore = true;
					
					$allWrappers.each(function() {
						const $target = $(this);
						if ($target[0] === $dragging[0]) return;
						
						const rect = this.getBoundingClientRect();
						const midpoint = rect.top + (rect.height / 2);
						
						if (e.pageY >= rect.top && e.pageY <= rect.bottom) {
							dropTarget = $target;
							insertBefore = (e.pageY < midpoint);
							return false; // break
						}
					});
					
					// Perform the move
					if (dropTarget) {
						if (insertBefore) {
							dropTarget.before($dragging);
						} else {
							dropTarget.after($dragging);
						}
						self.updateSectionFromContent(sectionId);
						self.showMessage('Text reordered', 'success', 1500);
					}
					
					// Cleanup
					$ghost.remove();
					$dragging.css('opacity', '1');
					$handle.css('cursor', 'grab');
					$allWrappers.css('border-top', '').css('border-bottom', '');
					
					// Re-enable contenteditable and reset dragging flag
					setTimeout(() => {
						$content.attr('contenteditable', wasEditable);
						self.isDragging = false;
						$handle.css('opacity', '0');
					}, 100);
				});
			});
		});
	},
	
	/**
	 * Update section data after content has been reordered
	 */
	updateSectionFromContent: function(sectionId) {
		const section = this.currentPage.sections.find(s => s.id === sectionId);
		if (!section) return;
		
		const $content = $(`#${sectionId} .buildfuly-section-content`);
		
		// Clone content and unwrap the draggable wrappers before saving
		const $clone = $content.clone();
		$clone.find('.buildfuly-drag-handle').remove(); // Remove drag handles
		$clone.find('.buildfuly-draggable-wrapper').each(function() {
			$(this).replaceWith($(this).contents()); // Unwrap, keeping inner content
		});
		
		let newHtml = $clone.html();
		
		// Preserve <style> tags from head
		const headStyleEl = document.querySelector(`style[data-section-id="${sectionId}"]`);
		if (headStyleEl) {
			newHtml += `\n<style>${headStyleEl.textContent}</style>`;
		}
		
		section.html = newHtml;
		this.autoSave();
	},
	
	/**
	 * Encode <script> tags to data-script attributes for WordPress compatibility
	 * This allows JavaScript to be saved and executed without WordPress stripping it
	 */
	encodeScriptTags: function(html) {
		// Match script tags and their content
		const scriptRegex = /<script[^>]*>([\s\S]*?)<\/script>/gi;
		let match;
		let processedHtml = html;
		
		while ((match = scriptRegex.exec(html)) !== null) {
			const fullMatch = match[0];
			const scriptContent = match[1].trim();
			
			if (scriptContent) {
				// Encode the script content
				const encodedJs = btoa(unescape(encodeURIComponent(scriptContent)));
				// Replace script tag with encoded data attribute div
				const replacement = `<div data-script="${encodedJs}" style="display:none;"></div>`;
				processedHtml = processedHtml.replace(fullMatch, replacement);
			} else {
				// Remove empty script tags
				processedHtml = processedHtml.replace(fullMatch, '');
			}
		}
		
		return processedHtml;
	},
	
	loadPage: function(pageId) {
		const page = this.pages.find(p => p.id === pageId);
		if (!page) return;
		
		console.log('Loading page:', pageId, 'wordpress_id:', page.wordpress_id);
		
		// Save current page images before switching
		if (this.currentPage && this.currentPage.id !== pageId) {
			const currentImages = $('#buildfuly-page-images-ids').val();
			this.currentPage.images = currentImages ? currentImages.split(',').map(id => parseInt(id)) : [];
			// Save to WordPress database instead of localStorage
			this.savePageToDatabase();
		}
		
		this.currentPage = page;
		
		// Initialize keywords array if not exists
		if (!page.keywords || !Array.isArray(page.keywords)) {
			// Parse from keyword string if available
			if (page.keyword) {
				page.keywords = page.keyword.split(',').map(k => k.trim()).filter(k => k);
			} else {
				page.keywords = [];
			}
		}
		
		// Load this page's images into the UI
		const pageImages = page.images || [];
		if (pageImages.length > 0) {
			// Load images into the field and preview
			$('#buildfuly-page-images-ids').val(pageImages.join(','));
			this.loadPageImagesPreview(pageImages);
		} else {
			// Clear images if this page has none
			$('#buildfuly-page-images-ids').val('');
			$('#buildfuly-page-images-preview').empty();
		}
		
		// Update UI
		$('.buildfuly-page-item').removeClass('active');
		$(`#${pageId}`).addClass('active');
		
		// Update URL to reflect selected page
		const url = new URL(window.location);
		url.searchParams.set('page_id', pageId);
		window.history.pushState({}, '', url);
		
		// Trigger event for SEO metadata UI - include both local and wordpress IDs
		$(document).trigger('buildfuly:pageSelected', [{
			pageId: pageId,
			wordpressId: page.wordpress_id || null
		}]);
		
		// Update page header
		$('#buildfuly-page-main-title').text(page.title);
		const sectionsCount = page.sections ? page.sections.length : 0;
		$('#buildfuly-section-count').text(sectionsCount + (sectionsCount === 1 ? ' section' : ' sections'));
		
		// Show UI elements
		$('#buildfuly-color-bar, #buildfuly-control-bar, #buildfuly-keywords-panel, #buildfuly-top-control-bar').show();
		$('.buildfuly-floating-actions').show();
		this.hideWelcomeScreen();
		
		// Render keywords panel
		this.renderKeywordsPanel();
		
		// Initialize keywords preview (collapsed by default)
		const keywords = page.keywords || [];
		const keywordText = keywords.map(k => typeof k === 'string' ? k : (k.keyword || '')).filter(k => k).join(', ');
		$('#buildfuly-keywords-preview').text(keywordText || 'No keywords').show();
		$('#buildfuly-keywords-list').hide();
		
		// Initialize color mini swatches (collapsed by default)
		if (page.colors) {
			this.updateColorMiniSwatches(page.colors);
		}
		$('#buildfuly-color-swatches-full').hide();
		$('#buildfuly-color-swatches-mini').show();
		$('#buildfuly-colors-arrow').removeClass('expanded');
		
		// Apply theme colors
		if (page.colors) {
			this.themeColors = page.colors;
			setTimeout(() => {
				this.applyThemeColors();
			}, 100);
		}
		
		// Render sections or show empty state
		$('#buildfuly-sections-container').empty();
		
		// Clear any old section styles from head
		$('style[data-section-id]').remove();
		
		if (page.sections && page.sections.length > 0) {
			page.sections.forEach((section, index) => {
				const $section = this.createSectionElement(section);
				$('#buildfuly-sections-container').append($section);
			});
			// Update word count and keywords after DOM is ready
			setTimeout(() => {
				this.updateWordCount();
				this.renderKeywordsPanel();
				this.initEncodedScripts(); // Initialize carousel scripts
				this.updateButtonsFromSettings(); // Update CTA buttons from settings
				this.renderSectionDividers(); // Add dividers between sections
			}, 100);
		} else {
			this.showEmptyPageState();
		}
		
		// Update publish button state based on whether page is already published
		this.updatePublishButtonState();
	},

	/**
	 * Update publish button to show "Update" if page is already published
	 */
	updatePublishButtonState: function() {
		const $publishBtn = $('#buildfuly-publish-page');
		const isPublished = this.currentPage && this.currentPage.wordpress_id && this.currentPage.wordpress_id > 0;
		
		if (isPublished) {
			$publishBtn.attr('title', 'Update Published Page')
				.addClass('is-published')
				.html('<span class="dashicons dashicons-update"></span>');
		} else {
			$publishBtn.attr('title', 'Publish Page')
				.removeClass('is-published')
				.html('<span class="dashicons dashicons-yes-alt"></span>');
		}
	},

	/**
	 * Update all CTA buttons in preview to match current settings
	 */
	updateButtonsFromSettings: function() {
		if (typeof buildfulyAdmin === 'undefined' || !buildfulyAdmin.ctaButtons) {
			return;
		}

		const buttons = buildfulyAdmin.ctaButtons;
		
		// Update all primary buttons with data-button-index="0" (linked to global settings)
		// Skip buttons with data-custom-button="true"
		const primarySelectors = [
			'.buildfuly-btn-primary[data-button-index="0"]',
			'.hero-btn-primary[data-button-index="0"]',
			'.cta-btn-primary[data-button-index="0"]',
			'.cta-box-btn[data-button-index="0"]'
		].join(', ');
		
		$('#buildfuly-sections-container').find(primarySelectors).each(function() {
			// Skip custom buttons
			if ($(this).attr('data-custom-button') === 'true') {
				return;
			}
			
			// Show/hide based on whether button exists and has text
			if (buttons[0] && buttons[0].text && buttons[0].text.trim() !== '') {
				$(this).text(buttons[0].text);
				if (buttons[0].url) $(this).attr('href', buttons[0].url);
				$(this).css('display', 'inline-flex');
			} else {
				// No primary button or empty text - hide it
				$(this).hide();
			}
		});
		
		// Update all secondary buttons with data-button-index="1"
		const secondarySelectors = [
			'.buildfuly-btn-secondary[data-button-index="1"]',
			'.hero-btn-secondary[data-button-index="1"]',
			'.cta-btn-secondary[data-button-index="1"]'
		].join(', ');
		
		$('#buildfuly-sections-container').find(secondarySelectors).each(function() {
			// Skip custom buttons
			if ($(this).attr('data-custom-button') === 'true') {
				return;
			}
			
			// Show/hide based on whether button exists and has text
			if (buttons[1] && buttons[1].text && buttons[1].text.trim() !== '') {
				$(this).text(buttons[1].text);
				if (buttons[1].url) $(this).attr('href', buttons[1].url);
				$(this).css('display', 'inline-flex');
			} else {
				// No secondary button configured - hide it
				$(this).hide();
			}
		});
		
		// Update review settings
		this.updateReviewsFromSettings();
	},

	/**
	 * Update all review elements in preview to match current settings
	 */
	updateReviewsFromSettings: function() {
		if (typeof buildfulyAdmin === 'undefined' || !buildfulyAdmin.reviewSettings) {
			return;
		}

		const reviews = buildfulyAdmin.reviewSettings;
		const platform = reviews.platform || 'Google';
		const url = reviews.url || '';
		
		// Update review platform text (e.g., "Google Reviews", "Yelp Reviews")
		$('#buildfuly-sections-container').find('.buildfuly-review-platform-text[data-review-linked="true"]').each(function() {
			const currentText = $(this).text();
			// Replace just the platform name, keep the " Reviews" suffix
			if (currentText.includes('Reviews')) {
				$(this).text(platform + ' Reviews');
			}
		});
		
		// Update review platform links/spans (e.g., "★ Google")
		$('#buildfuly-sections-container').find('.buildfuly-review-platform-link[data-review-linked="true"]').each(function() {
			const $el = $(this);
			const currentText = $el.text();
			
			// Update platform name (keep the star if present)
			if (currentText.includes('★')) {
				const newText = '★ ' + platform;
				if ($el.is('a')) {
					$el.text(newText);
					if (url) {
						$el.attr('href', url);
					}
				} else {
					// It's a span, check if we should convert to link
					if (url) {
						const styles = $el.attr('style') || '';
						const classes = $el.attr('class') || '';
						$el.replaceWith(
							$('<a>')
								.attr('href', url)
								.attr('target', '_blank')
								.attr('rel', 'noopener')
								.attr('class', classes)
								.attr('data-review-linked', 'true')
								.attr('style', styles + ' text-decoration: none;')
								.text(newText)
						);
					} else {
						$el.text(newText);
					}
				}
			} else {
				$el.text(platform);
			}
		});
		
		// Update standalone review platform elements
		$('#buildfuly-sections-container').find('.buildfuly-review-platform[data-review-linked="true"]').each(function() {
			const $el = $(this);
			
			if ($el.is('a')) {
				$el.text(platform);
				if (url) {
					$el.attr('href', url);
				}
			} else {
				// It's a span, check if we should convert to link
				if (url) {
					const styles = $el.attr('style') || '';
					const classes = $el.attr('class') || '';
					$el.replaceWith(
						$('<a>')
							.attr('href', url)
							.attr('target', '_blank')
							.attr('rel', 'noopener')
							.attr('class', classes)
							.attr('data-review-linked', 'true')
							.attr('style', styles)
							.text(platform)
					);
				} else {
					$el.text(platform);
				}
			}
		});
		
		// Update review badge and star containers
		$('#buildfuly-sections-container').find('.buildfuly-review-badge[data-review-linked="true"], .buildfuly-review-stars[data-review-linked="true"]').each(function() {
			const $el = $(this);
			
			if (url && !$el.is('a')) {
				// Convert div to link
				const styles = $el.attr('style') || '';
				const html = $el.html();
				$el.replaceWith(
					$('<a>')
						.attr('href', url)
						.attr('target', '_blank')
						.attr('rel', 'noopener noreferrer')
						.attr('class', $el.attr('class'))
						.attr('data-review-linked', 'true')
						.attr('style', styles + ' text-decoration: none; transition: opacity 0.2s;')
						.html(html)
						.on('mouseover', function() { $(this).css('opacity', '0.8'); })
						.on('mouseout', function() { $(this).css('opacity', '1'); })
				);
			} else if ($el.is('a')) {
				if (url) {
					$el.attr('href', url);
				}
			}
		});
	},

	renderKeywordsPanel: function() {
		if (!this.currentPage) return;
		
		// Save current checkbox states before re-rendering
		const checkedKeywords = [];
		const uncheckedKeywords = [];
		$('.buildfuly-keyword-checkbox').each(function() {
			if ($(this).is(':checked')) {
				checkedKeywords.push($(this).data('keyword'));
			} else {
				uncheckedKeywords.push($(this).data('keyword'));
			}
		});
		
		const $list = $('#buildfuly-keywords-list');
		$list.empty();
		
	const keywords = this.currentPage.keywords || [];
	
	keywords.forEach((keyword, index) => {
		const density = this.calculateKeywordDensity(keyword);
		// SEO Expert thresholds: <0.5% (too low), 0.5-2.5% (optimal), >2.5% (keyword stuffing)
		const densityClass = density < 0.5 ? 'low' : (density <= 2.5 ? 'good' : 'medium');
		// Better visual scaling: 1% = ~50% bar, 2% = 80% bar, 2.5% = 100% bar
		const densityPercent = Math.min((density / 2) * 100, 100);
		
		const exactOccurrences = this.countKeywordOccurrences(keyword);
		const similarOccurrences = this.countSimilarWords(keyword);
		const totalOccurrences = exactOccurrences + similarOccurrences;
		
		// Check all by default, unless previously unchecked
		const isChecked = !uncheckedKeywords.includes(keyword);
		
	const $item = $(`
		<div class="buildfuly-keyword-item" data-index="${index}">
			<input type="checkbox" class="buildfuly-keyword-checkbox" data-keyword="${keyword}" data-index="${index}" ${isChecked ? 'checked' : ''}>
			<div class="buildfuly-keyword-density">
				<div class="buildfuly-keyword-density-bar">
					<div class="buildfuly-keyword-density-fill ${densityClass}" style="width: ${densityPercent}%"></div>
					<div class="buildfuly-keyword-density-label">${density.toFixed(1)}%</div>
				</div>
			</div>
			<div class="buildfuly-keyword-content">
				<input type="text" class="buildfuly-keyword-input" value="${keyword}" data-index="${index}">
				<div class="buildfuly-keyword-stats">
					${exactOccurrences} exact${similarOccurrences > 0 ? ` • ${similarOccurrences} similar` : ''}
				</div>
			</div>
			<div class="buildfuly-keyword-actions">
				<button class="buildfuly-keyword-remove" data-index="${index}" title="Remove keyword">
					<span class="dashicons dashicons-no-alt"></span>
				</button>
			</div>
		</div>
	`);		$list.append($item);
	});		// Bind events
	$('.buildfuly-keyword-input').off('change').on('change', this.updateKeyword.bind(this));
	$('.buildfuly-keyword-remove').off('click').on('click', this.removeKeyword.bind(this));
	$('#buildfuly-add-keyword').off('click').on('click', this.addKeyword.bind(this));
	$('#buildfuly-regenerate-with-keywords').off('click').on('click', this.regenerateWithKeywords.bind(this));
},calculateKeywordDensity: function(keyword) {
	if (!this.currentPage || !this.currentPage.sections) return 0;
	
	const pageText = this.getPageText().toLowerCase();
	const keywordLower = keyword.toLowerCase();
	const totalWords = pageText.split(/\s+/).filter(w => w).length;
	
	if (totalWords === 0) return 0;
	
	// Count exact occurrences
	const exactOccurrences = this.countKeywordOccurrences(keyword);
	
	// Count similar word variations (like real SEO tools do)
	const similarOccurrences = this.countSimilarWords(keyword);
	
	// Include 60% weight for similar words (industry standard)
	const weightedOccurrences = exactOccurrences + (similarOccurrences * 0.6);
	
	// Calculate SEO keyword density: (total occurrences / total words) × 100
	// Standard formula used by Yoast, SEMrush, Ahrefs
	const density = (weightedOccurrences / totalWords) * 100;
	
	return density;
},	countKeywordOccurrences: function(keyword) {
		if (!this.currentPage || !this.currentPage.sections) return 0;
		
		const pageText = this.getPageText().toLowerCase();
		const keywordLower = keyword.toLowerCase();
		
		// Count exact phrase occurrences
		const regex = new RegExp('\\b' + keywordLower.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&') + '\\b', 'gi');
		const matches = pageText.match(regex);
		
		return matches ? matches.length : 0;
	},
	
	countSimilarWords: function(keyword) {
		if (!this.currentPage || !this.currentPage.sections) return 0;
		
		const pageText = this.getPageText().toLowerCase();
		const keywordLower = keyword.toLowerCase();
		
		// Generate similar word patterns
		const variations = this.generateWordVariations(keywordLower);
		
		let totalMatches = 0;
		variations.forEach(variation => {
			const regex = new RegExp('\\b' + variation.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&') + '\\b', 'gi');
			const matches = pageText.match(regex);
			if (matches) {
				totalMatches += matches.length;
			}
		});
		
		return totalMatches;
	},
	
	generateWordVariations: function(keyword) {
		const variations = [keyword];
		const words = keyword.split(/\s+/);
		
		// For each word in the keyword, add common variations
		words.forEach(word => {
			const lower = word.toLowerCase();
			
			// Add the word itself in lowercase
			variations.push(lower);
			
			// Remove trailing 's' for plurals
			if (lower.endsWith('s') && lower.length > 3) {
				variations.push(lower.slice(0, -1));
			}
			// Add 's' for singular
			if (!lower.endsWith('s')) {
				variations.push(lower + 's');
			}
			// Common word endings
			if (lower.endsWith('ing')) {
				variations.push(lower.slice(0, -3)); // build from building
				variations.push(lower.slice(0, -3) + 'er'); // builder from building
				variations.push(lower.slice(0, -3) + 'e'); // builte -> built correction
			}
			if (lower.endsWith('er')) {
				variations.push(lower.slice(0, -2)); // build from builder
				variations.push(lower.slice(0, -2) + 'ing'); // building from builder
			}
			if (lower.endsWith('ed')) {
				variations.push(lower.slice(0, -2)); // creat from created
				variations.push(lower.slice(0, -2) + 'e'); // create from created
				variations.push(lower.slice(0, -2) + 'ing'); // creating from created
			}
			if (lower.endsWith('ly')) {
				variations.push(lower.slice(0, -2)); // quick from quickly
			}
			if (lower.endsWith('tion')) {
				variations.push(lower.slice(0, -4)); // creat from creation
				variations.push(lower.slice(0, -4) + 'e'); // create from creation
			}
			// Common abbreviations
			if (lower === 'wordpress') {
				variations.push('wp');
			}
			if (lower === 'wp') {
				variations.push('wordpress');
			}
		});
		
		return [...new Set(variations)]; // Remove duplicates
	},

getPageText: function() {
	if (!this.currentPage) return '';
	
	let text = '';
	// Read from actual DOM to get current content (including edits)
	// Exclude aria-hidden elements (duplicate carousel cards, etc.)
	$('#buildfuly-sections-container .buildfuly-section-content').each(function() {
		const $section = $(this);
		// Clone the section to manipulate without affecting display
		const $clone = $section.clone();
		// Remove all aria-hidden elements
		$clone.find('[aria-hidden="true"]').remove();
		// Get text from visible elements only
		const sectionText = $clone.text();
		text += ' ' + sectionText;
	});
	
	return text;
},

updateWordCount: function() {
	// Word count is displayed in the page list sidebar via renderPagesList()
	// This function is kept for backward compatibility
	this.debug('updateWordCount() called - word count will be updated via renderPagesList()');
	const text = this.getPageText();
	this.debug('Page text length:', text.length);
	const wordCount = text.trim().split(/\s+/).filter(w => w).length;
	this.debug('Calculated word count:', wordCount);
},

	/**
	 * Initialize encoded scripts from data-script attributes
	 * Used for carousel and other interactive components in preview
	 */
	initEncodedScripts: function() {
		// First, reset all data-script-loaded attributes to allow re-initialization
		$('#buildfuly-sections-container [data-script-loaded]').removeAttr('data-script-loaded');
		
		const scriptElements = $('#buildfuly-sections-container [data-script]:not([data-script-loaded])');
		
		scriptElements.each(function() {
			const encoded = $(this).attr('data-script');
			if (!encoded || encoded.trim() === '') {
				return; // Skip empty attributes
			}
			
			try {
				// Decode base64 with UTF-8 support (reverse of btoa(unescape(encodeURIComponent())))
				const decoded = decodeURIComponent(escape(atob(encoded)));
				
				// Execute script safely
				try {
					const script = document.createElement('script');
					script.textContent = decoded;
					document.body.appendChild(script);
					// Mark as loaded to prevent duplicate execution
					$(this).attr('data-script-loaded', 'true');
				} catch (execError) {
					console.error('Failed to execute decoded script:', execError);
				}
			} catch (e) {
				console.error('Failed to decode script:', e);
				// Don't let script errors break the entire page
				$(this).attr('data-script-loaded', 'error');
			}
		});
	},

	updateKeyword: function(e) {
		const index = parseInt($(e.target).data('index'));
		const newKeyword = $(e.target).val().trim();
		
		if (newKeyword && this.currentPage.keywords[index] !== newKeyword) {
			this.currentPage.keywords[index] = newKeyword;
			this.savePage();
			this.renderKeywordsPanel();
		}
	},

	addKeyword: function() {
		const keyword = prompt('Enter a new SEO keyword:');
		if (keyword && keyword.trim()) {
			if (!this.currentPage.keywords) {
				this.currentPage.keywords = [];
			}
			this.currentPage.keywords.push(keyword.trim());
			this.savePage();
			this.renderKeywordsPanel();
		}
	},

	removeKeyword: function(e) {
		const index = parseInt($(e.currentTarget).data('index'));
		if (confirm('Remove this keyword?')) {
			this.currentPage.keywords.splice(index, 1);
			this.savePage();
			this.renderKeywordsPanel();
		}
	},

regenerateWithKeywords: function() {
	if (!this.currentPage || !this.currentPage.keywords || this.currentPage.keywords.length === 0) {
		this.showMessage('Please add at least one keyword first', 'warning');
		return;
	}
	
	if (!confirm('This will regenerate all sections on this page with the current keywords. Continue?')) {
		return;
	}
	
	// Clear current sections
	this.currentPage.sections = [];
	$('#buildfuly-sections-container').empty();
	
	// Regenerate with keywords using the default layout
	const layout = this.getDefaultLayout(this.currentPage.name || 'landing');
	this.generatePageWithLayout(this.currentPage.name, this.currentPage.keywords, layout);
},	generateSectionFromKeyword: function(e) {
	const keyword = $(e.currentTarget).data('keyword');
	const keywordIndex = $(e.currentTarget).data('index');
	
	this.debug('generateSectionFromKeyword() called for keyword:', keyword);
	
	if (!keyword || !this.currentPage) return;
	
	// Get the selected component type from the single dropdown
	const sectionType = $('#buildfuly-section-type').val() || 'random';
	
	this.debug('Section type:', sectionType);
	
	// Get selected image ID (if any)
	const imageId = $('#buildfuly-section-image-id').val();
	
	this.generateWithCommonLogic({
		url: buildfulyAdmin.restUrl + '/content/generate',
		data: {
			page_type: this.currentPage.title,
			keywords: keyword,
			component: sectionType,
			colors: this.themeColors,
			image_id: imageId
		},
		loadingMessage: `Generating ${sectionType === 'random' ? 'section' : sectionType} for "${keyword}"...`,
		successMessage: `Section created with keyword: ${keyword}`,
		onSuccess: (response) => {
			this.debug('Section generated successfully, adding to page');
			// Add section to current page
			const section = {
				id: 'section-' + Date.now(),
				type: sectionType,
				content: response.data.content || response.data,
				keyword: keyword
			};
			
			this.currentPage.sections.push(section);
			
			// Render the new section
			const $section = this.createSectionElement(section);
			$('#buildfuly-sections-container').append($section);
			
			this.debug('Section added to DOM, updating page list');
			
			// Update section count display
			this.renderPagesList();
			
			// Clear image after successful generation
			$('#buildfuly-section-image-id').val('');
			$('#buildfuly-section-image-indicator').hide();
			
			this.debug('generateSectionFromKeyword() completed');
		}
	});
	},

	renderPagesList: function() {
			const $list = $('#buildfuly-pages-list');
			$list.empty();
			
			// Get homepage ID from settings
			const homepageId = buildfulyAdmin.homepageId || 0;
			
			this.pages.forEach(page => {
				const sectionsCount = page.sections ? page.sections.length : 0;
				const isEmpty = sectionsCount === 0;
				const isHomepage = page.wordpress_id && page.wordpress_id === homepageId;
				const isPublished = page.wordpress_id && page.wordpress_id > 0;
				
				// Calculate word count
				let wordCount = 0;
				if (page.sections) {
					page.sections.forEach(section => {
						if (section.html) {
							const text = $('<div>').html(section.html).text();
							wordCount += text.trim().split(/\s+/).length;
						}
					});
				}
				
				// Get keyword preview (truncate to 20 chars)
				let keywordPreview = '';
				if (page.keywords && Array.isArray(page.keywords) && page.keywords.length > 0) {
					const firstKeyword = page.keywords[0];
					keywordPreview = firstKeyword.length > 20 
						? firstKeyword.substring(0, 20) + '...' 
						: firstKeyword;
				} else if (page.keyword) {
					keywordPreview = page.keyword.length > 20 
						? page.keyword.substring(0, 20) + '...' 
						: page.keyword;
				}
				
				// Home icon for homepage (can be set in Settings)
				const homeIcon = isHomepage ? '<span class="dashicons dashicons-admin-home buildfuly-home-icon" title="Homepage"></span>' : '';
				
				// Published badge
				const publishedBadge = isPublished ? '<span class="buildfuly-published-badge" title="Published to WordPress">✓</span>' : '';
				
				const $item = $(`
					<div class="buildfuly-page-item ${isEmpty ? 'empty-page' : ''} ${isHomepage ? 'is-homepage' : ''} ${isPublished ? 'is-published' : ''}" id="${page.id}" data-wordpress-id="${page.wordpress_id || ''}">
						${isEmpty ? '<button class="page-generate-btn-left" data-page-id="' + page.id + '"><span class="dashicons dashicons-welcome-write-blog"></span></button>' : homeIcon}
						<div class="buildfuly-page-item-content">
							<div class="buildfuly-page-item-title">
								<span class="buildfuly-page-title-text" data-page-id="${page.id}">${page.title}</span>
								${isHomepage ? ' <span class="buildfuly-home-badge">HOME</span>' : ''}${publishedBadge}
								<button class="buildfuly-edit-title-btn" data-page-id="${page.id}" title="Edit Title"><span class="dashicons dashicons-edit"></span></button>
							</div>
							${keywordPreview ? `<div class="buildfuly-page-item-keyword">${keywordPreview}</div>` : ''}
							<div class="buildfuly-page-item-meta">${sectionsCount} sections • ${wordCount} words${isPublished ? ' • <span style="color: #10b981;">Published</span>' : ''}</div>
						</div>
						<button class="page-delete-btn" data-page-id="${page.id}" title="Delete Page">
							<span class="dashicons dashicons-trash"></span>
						</button>
					</div>
				`);
				
				// Click on page item (not buttons)
				$item.on('click', (e) => {
					if (!$(e.target).closest('.page-generate-btn-left, .page-delete-btn, .buildfuly-edit-title-btn').length) {
						this.loadPage(page.id);
					}
				});
				
				// Click on Edit Title button
				$item.find('.buildfuly-edit-title-btn').on('click', (e) => {
					e.stopPropagation();
					const pageId = $(e.currentTarget).data('page-id');
					this.editPageTitle(pageId);
				});
				
			// Click on Generate button
			$item.find('.page-generate-btn-left').on('click', (e) => {
				e.stopPropagation();
				this.loadPage(page.id);
				// Small delay to ensure page is loaded
				setTimeout(() => {
					// Use page title/name correctly
					const pageTitle = page.title || page.name;
					
					// Convert keywords array to comma-separated string
					let keywordsStr = '';
					if (Array.isArray(page.keywords) && page.keywords.length > 0) {
						keywordsStr = page.keywords.join(', ');
					} else if (page.keyword) {
						// Fallback to singular 'keyword' if exists
						keywordsStr = page.keyword;
					}
					
					// Populate fields correctly: title vs keywords
					$('#page-type').val(pageTitle);
					$('#page-keywords').val(keywordsStr);
					this.showNewPageModal({ preventDefault: () => {} });
				}, 100);
			});
			
			// Click on Delete button
			$item.find('.page-delete-btn').on('click', (e) => {
				e.stopPropagation();
				const pageId = $(e.currentTarget).data('page-id');
				const pageToDelete = this.pages.find(p => p.id === pageId);
				
				if (confirm(`Delete "${pageToDelete.title}"? This cannot be undone.`)) {
					this.deletePage(pageId);
				}
			});
			
			$list.append($item);
			});
		},

	/**
	 * Edit page title inline
	 */
	editPageTitle: function(pageId) {
		const page = this.pages.find(p => p.id === pageId);
		if (!page) return;
		
		const $titleSpan = $(`.buildfuly-page-title-text[data-page-id="${pageId}"]`);
		const currentTitle = page.title;
		
		// Replace span with input
		const $input = $(`<input type="text" class="buildfuly-page-title-input" value="${currentTitle}" data-page-id="${pageId}" style="width: 100%; padding: 4px 8px; border: 1px solid #667eea; border-radius: 4px; font-size: 14px; font-weight: 600;">`);
		$titleSpan.replaceWith($input);
		$input.focus().select();
		
		// Save on blur or enter
		const saveTitle = () => {
			const newTitle = $input.val().trim();
			if (newTitle && newTitle !== currentTitle) {
				page.title = newTitle;
				page.name = newTitle;
				
				// Save to WordPress database
				const originalPage = this.currentPage;
				this.currentPage = page;
				this.savePageToDatabase();
				this.currentPage = originalPage;
				
				// Update current page if it's the one being edited
				if (this.currentPage && this.currentPage.id === pageId) {
					this.currentPage.title = newTitle;
					this.currentPage.name = newTitle;
				}
				
				this.showMessage('Page title updated', 'success');
			}
			
			// Re-render the pages list
			this.renderPagesList();
		};
		
		$input.on('blur', saveTitle);
		$input.on('keydown', (e) => {
			if (e.key === 'Enter') {
				e.preventDefault();
				saveTitle();
			} else if (e.key === 'Escape') {
				this.renderPagesList();
			}
		});
	},

	generateThemeColors: function() {
		// Generate a harmonious color palette (mix of light and dark themes)
		const colorSchemes = [
			// Professional Blue (Light)
			{ primary: '#2563eb', secondary: '#1e40af', accent: '#10b981', text: '#111827', 'text-light': '#6b7280', background: '#ffffff' },
			// Modern Purple (Light)
			{ primary: '#7c3aed', secondary: '#6d28d9', accent: '#ec4899', text: '#1f2937', 'text-light': '#6b7280', background: '#ffffff' },
			// Energetic Orange (Light)
			{ primary: '#f59e0b', secondary: '#d97706', accent: '#3b82f6', text: '#111827', 'text-light': '#6b7280', background: '#ffffff' },
			// Fresh Green (Light)
			{ primary: '#10b981', secondary: '#059669', accent: '#3b82f6', text: '#111827', 'text-light': '#6b7280', background: '#ffffff' },
			// Bold Red (Light)
			{ primary: '#ef4444', secondary: '#dc2626', accent: '#f59e0b', text: '#111827', 'text-light': '#6b7280', background: '#ffffff' },
			
			// Dark Theme: Electric Blue
			{ primary: '#3b82f6', secondary: '#60a5fa', accent: '#10b981', text: '#ffffff', 'text-light': '#d1d5db', background: '#000000' },
			// Dark Theme: Purple Neon
			{ primary: '#a855f7', secondary: '#c084fc', accent: '#ec4899', text: '#ffffff', 'text-light': '#d1d5db', background: '#000000' },
			// Dark Theme: Cyber Teal
			{ primary: '#14b8a6', secondary: '#2dd4bf', accent: '#8b5cf6', text: '#ffffff', 'text-light': '#d1d5db', background: '#000000' },
			// Dark Theme: Sunset Orange
			{ primary: '#f59e0b', secondary: '#fbbf24', accent: '#ec4899', text: '#ffffff', 'text-light': '#d1d5db', background: '#000000' },
			// Dark Theme: Emerald Green
			{ primary: '#10b981', secondary: '#34d399', accent: '#3b82f6', text: '#ffffff', 'text-light': '#d1d5db', background: '#000000' },
			// Dark Theme: Hot Pink
			{ primary: '#ec4899', secondary: '#f472b6', accent: '#a855f7', text: '#ffffff', 'text-light': '#d1d5db', background: '#000000' },
			// Dark Theme: Royal Purple
			{ primary: '#7c3aed', secondary: '#9333ea', accent: '#10b981', text: '#ffffff', 'text-light': '#d1d5db', background: '#000000' }
		];
		
		// Pick a random color scheme
		return colorSchemes[Math.floor(Math.random() * colorSchemes.length)];
	},	regenerateColors: function(e) {
		if (e) e.preventDefault();
		
		console.log('Regenerate colors clicked');
		
		if (!this.currentPage) {
			this.showMessage('Please create or select a page first', 'warning');
			return;
		}
		
		// Generate new colors
		const newColors = this.generateThemeColors();
		console.log('New colors generated:', newColors);
		
		// Normalize bg/background
		newColors.bg = newColors.background || newColors.bg || '#ffffff';
		newColors.background = newColors.bg;
		
		this.currentPage.colors = newColors;
		this.themeColors = newColors;
		
		// Update ALL pages with the new colors (not just current page)
		if (this.pages && this.pages.length > 0) {
			const self = this;
			this.pages.forEach(page => {
				page.colors = {...newColors};
				// Save each page to WordPress database
				const originalPage = self.currentPage;
				self.currentPage = page;
				self.savePageToDatabase();
				self.currentPage = originalPage;
			});
		}
		
		// Update mini swatches
		this.updateColorMiniSwatches(newColors);
		
		// Apply new colors - use applyThemeColors for consistency
		this.applyThemeColors();
		
		// Save to WordPress options
		this.saveThemeColors();
		
		this.savePage();
		this.showMessage('New color palette generated globally for all pages!', 'success', 4000);
	},		applyThemeColors: function() {
			if (!this.themeColors) return;
			
			console.log('Applying theme colors:', this.themeColors);
			
			// Update color picker values
			$('#color-primary').val(this.themeColors.primary || '#00bfa5');
			$('#color-secondary').val(this.themeColors.secondary || '#00897b');
			$('#color-accent').val(this.themeColors.accent || '#7c4dff');
			$('#color-text').val(this.themeColors.text || '#1e293b');
			$('#color-bg').val(this.themeColors.bg || this.themeColors.background || '#ffffff');
			
			// Update CSS variables instantly
			const root = document.documentElement;
			const primaryColor = this.themeColors.primary || '#00bfa5';
			root.style.setProperty('--buildfuly-color-primary', primaryColor);
			root.style.setProperty('--buildfuly-color-primary-light', this.hexToRgba(primaryColor, 0.1));
			root.style.setProperty('--buildfuly-color-primary-dark', this.darkenColor(primaryColor, 15));
			root.style.setProperty('--buildfuly-color-primary-text', this.getContrastColor(primaryColor));
			root.style.setProperty('--buildfuly-color-secondary', this.themeColors.secondary || '#00897b');
			root.style.setProperty('--buildfuly-color-accent', this.themeColors.accent || '#7c4dff');
			root.style.setProperty('--buildfuly-color-text', this.themeColors.text || '#1e293b');
			root.style.setProperty('--buildfuly-color-text-light', this.themeColors.text_light || this.themeColors['text-light'] || '#6c757d');
			root.style.setProperty('--buildfuly-color-background', this.themeColors.bg || this.themeColors.background || '#000000');
			root.style.setProperty('--buildfuly-color-star', this.themeColors.star || '#fbbf24');
		},

		applyColorsToHtml: function(html) {
			if (!this.themeColors) return html;
			
			// Wrap in div to manipulate
			const $temp = $('<div>').html(html);
			
			// Only apply to section wrapper divs, not globally
			$temp.find('[class^="buildfuly-"]').first().css({
				'background-color': this.themeColors.background
			});
			
			// Apply colors to buttons and CTAs within sections
			$temp.find('.buildfuly-cta, .buildfuly-btn, button').css({
				'background-color': this.themeColors.primary,
				'color': '#ffffff',
				'border': 'none',
				'padding': '12px 32px',
				'border-radius': '8px',
				'font-weight': '600',
				'cursor': 'pointer',
				'transition': 'all 0.3s'
			});
			
			// Apply accent color to links (not buttons)
			$temp.find('a:not(.buildfuly-cta):not(.buildfuly-btn)').css('color', this.themeColors.accent);
			
			// Apply secondary color to headings within sections
			$temp.find('h1, h2, h3').css({
				'color': this.themeColors.secondary,
				'margin-bottom': '16px'
			});
			
			// Apply text color to paragraphs within sections
			$temp.find('p, li').css('color', this.themeColors.text);
			
			return $temp.html();
		},

		updateColor: function(e) {
			const $input = $(e.target);
			const colorType = $input.attr('id').replace('color-', '');
			const colorValue = $input.val();
			
		console.log('Color updated:', colorType, colorValue);
		
		if (this.currentPage && this.themeColors) {
// Normalize 'bg' and 'background' to both exist
if (colorType === 'bg') {
this.currentPage.colors.bg = colorValue;
this.currentPage.colors.background = colorValue;
this.themeColors.bg = colorValue;
this.themeColors.background = colorValue;
} else {
this.currentPage.colors[colorType] = colorValue;
this.themeColors[colorType] = colorValue;
}
			
		// Update ALL pages with the new global color (not just current page)
		if (this.pages && this.pages.length > 0) {
			const self = this;
			this.pages.forEach(page => {
				if (!page.colors) {
					page.colors = {};
				}
				if (colorType === 'bg') {
					page.colors.bg = colorValue;
					page.colors.background = colorValue;
				} else {
					page.colors[colorType] = colorValue;
				}
			});
			// Save all pages to WordPress database (debounced to avoid too many requests)
			clearTimeout(this._colorSaveTimer);
			this._colorSaveTimer = setTimeout(() => {
				this.pages.forEach(page => {
					const originalPage = self.currentPage;
					self.currentPage = page;
					self.savePageToDatabase();
					self.currentPage = originalPage;
				});
			}, 1000);
		}
		
		// Save theme colors to WordPress options
		this.saveThemeColors();			// Update CSS variable instantly
			const root = document.documentElement;
			const varMap = {
				'primary': '--buildfuly-color-primary',
				'secondary': '--buildfuly-color-secondary',
				'accent': '--buildfuly-color-accent',
				'text': '--buildfuly-color-text',
				'text-light': '--buildfuly-color-text-light',
				'bg': '--buildfuly-color-background'
			};
			
			if (varMap[colorType]) {
				root.style.setProperty(varMap[colorType], colorValue);
				
				if (colorType === 'primary') {
					root.style.setProperty('--buildfuly-color-primary-light', this.hexToRgba(colorValue, 0.1));
					root.style.setProperty('--buildfuly-color-primary-dark', this.darkenColor(colorValue, 15));
					root.style.setProperty('--buildfuly-color-primary-text', this.getContrastColor(colorValue));
				}
			}
			
			// Update the mini color swatches at the top to reflect the new color
			this.updateColorMiniSwatches(this.currentPage.colors);
			
			this.showMessage('Color updated globally for all pages!', 'success', 2000);
			this.savePage();
		}
	},		deleteSection: function(sectionId) {
		if (!confirm('Delete this section?')) return;
		
		// Remove the section's styles from head
		$(`style[data-section-id="${sectionId}"]`).remove();
		
		$(`#${sectionId}`).fadeOut(300, function() {
			$(this).remove();
		});
		
		this.currentPage.sections = this.currentPage.sections.filter(s => s.id !== sectionId);
		this.updateWordCount();
		this.renderKeywordsPanel(); // Refresh keyword stats
		this.savePage();
	},

	regenerateSection: function(sectionId) {
		const section = this.currentPage.sections.find(s => s.id === sectionId);
		if (!section) return;
		
		// Use first keyword when regenerating
		const keyword = Array.isArray(this.currentPage.keywords) && this.currentPage.keywords.length > 0 
			? this.currentPage.keywords[0] 
			: this.currentPage.keywords;
		
		// Remove old section immediately
		$(`#${sectionId}`).remove();
		this.currentPage.sections = this.currentPage.sections.filter(s => s.id !== sectionId);
		
		// Generate new section using shared logic
		this.generateSection(keyword, section.type, () => {
			this.showMessage('Section regenerated!', 'success');
		});
	},regenerateAllSections: function(e, silent = false) {
	e.preventDefault();
	
	// Check if usage limit reached before making API call
	if (!this.canGenerate()) {
		return;
	}
	
	// Get current color values from color pickers
	// Get current color values from color pickers
	const primary = $('#color-primary').val();
	const secondary = $('#color-secondary').val();
	const accent = $('#color-accent').val();
	const text = $('#color-text').val();
	const textLight = $('#color-text-light').val();
	const bg = $('#color-bg').val();
	
	// Update CSS variables instantly without regenerating content
	const root = document.documentElement;
	root.style.setProperty('--buildfuly-color-primary', primary);
	root.style.setProperty('--buildfuly-color-primary-light', this.hexToRgba(primary, 0.1));
	root.style.setProperty('--buildfuly-color-primary-dark', this.darkenColor(primary, 15));
	root.style.setProperty('--buildfuly-color-primary-text', this.getContrastColor(primary));
	root.style.setProperty('--buildfuly-color-secondary', secondary);
	root.style.setProperty('--buildfuly-color-accent', accent);
	root.style.setProperty('--buildfuly-color-text', text);
	root.style.setProperty('--buildfuly-color-text-light', textLight);
	root.style.setProperty('--buildfuly-color-background', bg);
	// Save colors to WordPress options via AJAX
	$.ajax({
		url: buildfulyAdmin.ajaxUrl,
		method: 'POST',
		data: {
			action: 'buildfuly_save_colors',
			nonce: buildfulyAdmin.nonce,
			primary: primary,
			secondary: secondary,
			accent: accent,
			text: text,
			text_light: textLight,
			bg: bg
		},
		success: () => {
			if (!silent) this.showMessage('Colors updated instantly!', 'success');
		},
		error: () => {
			if (!silent) this.showMessage('Colors applied but save failed. Refresh page.', 'warning');
		}
	});
},

// Helper function to convert hex to rgba
hexToRgba: function(hex, alpha) {
	const r = parseInt(hex.slice(1, 3), 16);
	const g = parseInt(hex.slice(3, 5), 16);
	const b = parseInt(hex.slice(5, 7), 16);
	return `rgba(${r}, ${g}, ${b}, ${alpha})`;
},

// Helper function to darken color
darkenColor: function(hex, percent) {
	let r = parseInt(hex.slice(1, 3), 16);
	let g = parseInt(hex.slice(3, 5), 16);
	let b = parseInt(hex.slice(5, 7), 16);
	
	r = Math.max(0, Math.floor(r - (r * percent / 100)));
	g = Math.max(0, Math.floor(g - (g * percent / 100)));
	b = Math.max(0, Math.floor(b - (b * percent / 100)));
	
	return `#${r.toString(16).padStart(2, '0')}${g.toString(16).padStart(2, '0')}${b.toString(16).padStart(2, '0')}`;
},

// Helper function to get contrasting color (white or black) based on background
getContrastColor: function(hex) {
	// Convert hex to RGB
	const r = parseInt(hex.slice(1, 3), 16);
	const g = parseInt(hex.slice(3, 5), 16);
	const b = parseInt(hex.slice(5, 7), 16);
	
	// Calculate relative luminance (perceived brightness)
	const luminance = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
	
	// Return white for dark backgrounds, black for light backgrounds
	return luminance > 0.5 ? '#000000' : '#ffffff';
},

moveSection: function(sectionId, direction) {
	// Safety check - ensure we have a valid page and sections
	if (!this.currentPage || !this.currentPage.sections || !Array.isArray(this.currentPage.sections)) {
		console.error('moveSection: No valid page or sections');
		return;
	}
	
	const $section = $(`#${sectionId}`);
	if (!$section.length) {
		console.error('moveSection: Section element not found:', sectionId);
		return;
	}
	
	if (direction === 'up') {
		$section.prev('.buildfuly-section').before($section);
	} else {
		$section.next('.buildfuly-section').after($section);
	}
	
	// Update sections array to match new DOM order
	const newOrder = [];
	const self = this;
	$('#buildfuly-sections-container .buildfuly-section').each(function() {
		const id = $(this).attr('id');
		const section = self.currentPage.sections.find(s => s.id === id);
		if (section) {
			newOrder.push(section);
		}
	});
	
	// Only update if we found all sections
	if (newOrder.length === this.currentPage.sections.length) {
		this.currentPage.sections = newOrder;
		this.savePage();
	} else {
		console.error('moveSection: Section count mismatch, not saving. Expected:', this.currentPage.sections.length, 'Found:', newOrder.length);
	}
},

	/**
	 * Generate SEO metadata for a page using AI
	 */
	generateSeoForPage: function(page, callback) {
		if (!page || !page.title) {
			if (callback) callback();
			return;
		}
		
		// Extract keywords for SEO generation
		const keywords = Array.isArray(page.keywords) 
			? page.keywords.map(k => typeof k === 'string' ? k : (k.keyword || '')).filter(k => k).join(', ')
			: (page.keywords || '');
		
		// Generate slug from page title
		const slug = page.title.toLowerCase() === 'home' 
			? '/' 
			: page.title.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');
		
		// Extract text content from first section for description
		let metaDescription = '';
		if (page.sections && page.sections.length > 0) {
			const firstSection = page.sections[0];
			if (firstSection.html) {
				const tempDiv = document.createElement('div');
				tempDiv.innerHTML = firstSection.html;
				
				// Remove script and style tags before extracting text
				const scripts = tempDiv.getElementsByTagName('script');
				const styles = tempDiv.getElementsByTagName('style');
				while (scripts.length > 0) scripts[0].remove();
				while (styles.length > 0) styles[0].remove();
				
				const text = tempDiv.textContent || tempDiv.innerText || '';
				// Clean up whitespace: replace multiple spaces/newlines with single space
				const cleanText = text.replace(/\s+/g, ' ').trim();
				metaDescription = cleanText.substring(0, 155).trim();
				if (cleanText.length > 155) metaDescription += '...';
			}
		}
		
		// Fallback to keywords-based description if no content available
		if (!metaDescription) {
			if (keywords) {
				// Create engaging description from keywords
				const keywordList = keywords.split(',').map(k => k.trim()).slice(0, 3);
				metaDescription = `Professional ${page.title.toLowerCase()} services. Expert in ${keywordList.join(', ')}. Get started today!`;
				if (metaDescription.length > 160) {
					metaDescription = metaDescription.substring(0, 157) + '...';
				}
			} else {
				// Ultimate fallback - use page title
				metaDescription = `Learn more about ${page.title}. High-quality services and expert solutions.`;
			}
		}
		
		// Don't add brand to meta_title - WordPress/Yoast adds it automatically
		const metaTitle = page.title;
		
		// Store SEO in page object
		page.seo = {
			slug: slug,
			meta_title: metaTitle.substring(0, 60),
			meta_description: metaDescription.substring(0, 160),
			focus_keyphrase: keywords.split(',')[0]?.trim() || page.title,
			keywords: keywords
		};
		
		console.log('Generated SEO for page:', page.title, page.seo);
		
		if (callback) callback();
	},

	savePage: function() {
			if (!this.currentPage) {
				console.error('savePage: No current page');
				return;
			}
			
			// SAFETY: Never save a page with empty sections if it previously had sections
			// This prevents accidental data loss from bugs
			const pageInArray = this.pages.find(p => p.id === this.currentPage.id);
			if (pageInArray) {
				const currentSectionCount = this.currentPage.sections ? this.currentPage.sections.length : 0;
				const savedSectionCount = pageInArray.sections ? pageInArray.sections.length : 0;
				
				// If we're about to save 0 sections but the saved version has sections, something went wrong
				if (currentSectionCount === 0 && savedSectionCount > 0) {
					console.error('savePage: BLOCKED - Would delete', savedSectionCount, 'sections. This looks like a bug.');
					this.showMessage('Save blocked: sections would be lost. Please refresh.', 'error');
					return;
				}
			}
			
			// Sync current page colors to theme colors
			if (this.currentPage.colors) {
				this.themeColors = this.currentPage.colors;
			}
			
			// Save to localStorage immediately (backup/primary for now)
			try {
				localStorage.setItem('buildfuly_pages', JSON.stringify(this.pages));
				console.log('Saved to localStorage');
			} catch (e) {
				console.warn('localStorage save failed:', e.message);
			}
			
			// Also try to save to WordPress database (async, non-blocking)
			this.savePageToDatabase();
			
			// Save theme colors to WordPress options and localStorage
			this.saveThemeColors();
			
			// Update color pickers to reflect saved state
			this.updateColorPickers();
			
			// If page is already published (has wordpress_id), sync HTML to WordPress
			console.log('savePage - wordpress_id:', this.currentPage.wordpress_id);
			if (this.currentPage.wordpress_id) {
				console.log('Syncing to WordPress...');
				this.syncToWordPress();
			} else {
				console.log('Page not published yet, no wordpress_id');
			}
			
			this.showMessage('Page saved', 'info', 2000);
		},
		
		/**
		 * Save page builder data to WordPress database via AJAX
		 * This replaces localStorage as the primary storage mechanism
		 */
		savePageToDatabase: function(callback) {
			const self = this;
			
			if (!this.currentPage) {
				console.error('savePageToDatabase: No current page');
				if (callback) callback(false);
				return;
			}
			
			// Make sure current page is in pages array
			const pageIndex = this.pages.findIndex(p => p.id === this.currentPage.id);
			if (pageIndex >= 0) {
				this.pages[pageIndex] = this.currentPage;
			}
			
			// Clean page data before sending (remove base64 images)
			const cleanPageData = this.cleanPageDataForStorage(this.currentPage);
			const cleanSize = JSON.stringify(cleanPageData).length;
			
			console.log('Saving page data to WordPress database:', this.currentPage.id);
			console.log('Original size:', JSON.stringify(this.currentPage).length, 'bytes');
			console.log('Cleaned size:', cleanSize, 'bytes');
			
			// If still too large (over 1MB), don't save to database
			if (cleanSize > 1000000) {
				console.warn('Page data too large to save to database:', cleanSize, 'bytes');
				self.saveToLocalStorageBackup();
				if (callback) callback(false);
				return;
			}
			
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_save_page_data',
					nonce: buildfulyAdmin.adminNonce,
					page_data: JSON.stringify(cleanPageData)
				},
				success: function(response) {
					if (response.success) {
						console.log('Page data saved to WordPress:', response.data);
						// Also keep a backup in localStorage (will be cleaned periodically)
						self.saveToLocalStorageBackup();
						if (callback) callback(true);
					} else {
						console.error('Failed to save page data:', response.data?.message);
						// Fallback to localStorage if WordPress save fails
						self.saveToLocalStorageBackup();
						if (callback) callback(false);
					}
				},
				error: function(xhr, status, error) {
					console.error('AJAX error saving page data:', error);
					console.error('Response:', xhr.responseText);
					// Fallback to localStorage on error
					self.saveToLocalStorageBackup();
					if (callback) callback(false);
				}
			});
		},
		
		/**
		 * Clean page data before storage - remove base64 images, keep only URLs
		 */
		cleanPageDataForStorage: function(pageData) {
			const cleaned = JSON.parse(JSON.stringify(pageData)); // Deep clone
			
			if (cleaned.sections && Array.isArray(cleaned.sections)) {
				cleaned.sections.forEach(section => {
					// Remove base64 from HTML
					if (section.html) {
						section.html = section.html.replace(/src=["']data:image\/[^;]+;base64,[^"']+["']/gi, 'src=""');
					}
					
					// Remove base64 from image objects but keep URL
					if (section.image) {
						if (typeof section.image === 'object') {
							delete section.image.base64;
							delete section.image.dataUrl;
						}
					}
					
					// Clean nested data
					if (section.data && typeof section.data === 'object') {
						this.cleanNestedBase64(section.data);
					}
				});
			}
			
			return cleaned;
		},
		
		/**
		 * Recursively clean base64 from nested objects
		 */
		cleanNestedBase64: function(obj) {
			if (!obj || typeof obj !== 'object') return;
			
			for (let key in obj) {
				if (key === 'base64' || key === 'dataUrl' || key === 'imageData') {
					delete obj[key];
				} else if (typeof obj[key] === 'object') {
					this.cleanNestedBase64(obj[key]);
				} else if (typeof obj[key] === 'string' && obj[key].startsWith('data:image/')) {
					delete obj[key];
				}
			}
		},
		
		/**
		 * Save a minimal backup to localStorage (for crash recovery)
		 * Only stores essential data, not full page content
		 */
		saveToLocalStorageBackup: function() {
			try {
				// Only store minimal page metadata for recovery, not full content
				const minimalPages = this.pages.map(p => ({
					id: p.id,
					title: p.title,
					wordpress_id: p.wordpress_id,
					status: p.status,
					lastModified: p.lastModified || Date.now()
				}));
				
				localStorage.setItem('buildfuly_pages_index', JSON.stringify(minimalPages));
				console.log('Saved minimal backup to localStorage');
			} catch (e) {
				// localStorage might be full, that's okay - WordPress is primary storage now
				console.warn('Could not save localStorage backup:', e.message);
			}
		},
		
		// Sync page content to WordPress (for already published pages)
		syncToWordPress: function() {
			if (!this.currentPage || !this.currentPage.wordpress_id) return;
			
			const pageTitle = this.currentPage.title || this.currentPage.name || '';
			const styledHtml = this.generateStyledHtml();
			
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_sync_page',
					nonce: $('#buildfuly-nonce').val(),
					wordpress_id: this.currentPage.wordpress_id,
					page_title: pageTitle,
					page_content: styledHtml
				},
				success: (response) => {
					if (response.success) {
						console.log('✓ Page synced to WordPress');
					} else {
						console.error('Failed to sync:', response.data);
					}
				},
				error: (xhr, status, error) => {
					console.error('Sync error:', error);
				}
			});
		},
		
		deletePage: function(pageId) {
			// Find page index
			const pageIndex = this.pages.findIndex(p => p.id === pageId);
			if (pageIndex === -1) return;
			
			const deletedPage = this.pages[pageIndex];
			
			// Remove page from array
			this.pages.splice(pageIndex, 1);
			
			// Delete from WordPress database
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_delete_page_data',
					nonce: buildfulyAdmin.adminNonce,
					page_id: pageId,
					wordpress_id: deletedPage.wordpress_id || 0
				},
				success: function(response) {
					console.log('Page data deleted from WordPress:', response);
				},
				error: function(xhr, status, error) {
					console.error('Error deleting page data:', error);
				}
			});
			
			// If deleting current page, load another page
			if (this.currentPage && this.currentPage.id === pageId) {
				if (this.pages.length > 0) {
					// Load first available page
					this.loadPage(this.pages[0].id);
				} else {
					// No pages left, clear workspace
					this.currentPage = null;
					$('#buildfuly-workspace-content').html('<div style="text-align: center; padding: 60px; color: #9ca3af;"><p>No pages yet. Click "New Page" to get started.</p></div>');
				}
			}
			
			// Update pages list
			this.renderPagesList();
			this.showMessage(`Deleted "${deletedPage.title}"`, 'success', 3000);
		},

		setAsHomepage: function(wordpressId) {
			const self = this;
			
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_set_homepage',
					nonce: buildfulyAdmin.adminNonce,
					page_id: wordpressId
				},
				success: function(response) {
					if (response.success) {
						// Update local homepageId
						buildfulyAdmin.homepageId = wordpressId;
						
						// Re-render pages list to show home icon
						self.renderPagesList();
						self.showMessage('Homepage updated!', 'success', 3000);
					} else {
						self.showMessage(response.data.message || 'Failed to set homepage', 'error');
					}
				},
				error: function() {
					self.showMessage('Failed to set homepage. Please try again.', 'error');
				}
			});
		},

		// Debounce timer for autoSave
		_autoSaveTimer: null,
		_autoSavePending: false,

		autoSave: function() {
			if (!this.currentPage || this.pages.length === 0) {
				return;
			}
			
			// SAFETY: Check that we're not about to save empty sections
			const pageInArray = this.pages.find(p => p.id === this.currentPage.id);
			if (pageInArray) {
				const currentSectionCount = this.currentPage.sections ? this.currentPage.sections.length : 0;
				const savedSectionCount = pageInArray.sections ? pageInArray.sections.length : 0;
				
				if (currentSectionCount === 0 && savedSectionCount > 0) {
					console.error('autoSave: BLOCKED - Would delete', savedSectionCount, 'sections');
					return;
				}
			}
			
			// Debounce: Wait 2 seconds after last change before saving to database
			// This prevents excessive saves during rapid edits
			const self = this;
			
			if (this._autoSaveTimer) {
				clearTimeout(this._autoSaveTimer);
			}
			
			this._autoSavePending = true;
			
			this._autoSaveTimer = setTimeout(function() {
				if (self._autoSavePending && self.currentPage) {
					console.log('AutoSave: Saving to WordPress database...');
					self.savePageToDatabase();
					self._autoSavePending = false;
				}
			}, 2000); // 2 second debounce
			
			// Also save minimal backup to localStorage immediately (for crash recovery)
			this.saveToLocalStorageBackup();
		},

		loadPages: function() {
			const self = this;
			
			// Load pages from WordPress database (primary storage)
			this.loadPagesFromDatabase(function(success) {
				if (!success || self.pages.length === 0) {
					// Fallback: Try to load from localStorage (legacy/backup)
					self.loadPagesFromLocalStorage();
					
					// If we loaded from localStorage, migrate to WordPress
					if (self.pages.length > 0) {
						console.log('[MIGRATION] Found', self.pages.length, 'pages in localStorage, migrating to WordPress...');
						self.migratePagesToDatabase();
					}
				}
				
				// Notify that pages are loaded (for UI initialization)
				self.onPagesLoaded();
			});
		},
		
		/**
		 * Load pages from WordPress database via AJAX
		 */
		loadPagesFromDatabase: function(callback) {
			const self = this;
			
			console.log('Loading pages from WordPress database...');
			
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_load_all_pages',
					nonce: buildfulyAdmin.adminNonce
				},
				success: function(response) {
					if (response.success && response.data.pages) {
						self.pages = response.data.pages;
						console.log('Loaded', self.pages.length, 'pages from WordPress database');
						
						// Run migrations on loaded pages
						self.runPageMigrations();
						
						if (callback) callback(true);
					} else {
						console.log('No pages found in WordPress database');
						if (callback) callback(false);
					}
				},
				error: function(xhr, status, error) {
					console.error('Error loading pages from WordPress:', error);
					if (callback) callback(false);
				}
			});
		},
		
		/**
		 * Load pages from localStorage (legacy fallback)
		 */
		loadPagesFromLocalStorage: function() {
			const saved = localStorage.getItem('buildfuly_pages');
			if (saved) {
				try {
					const parsed = JSON.parse(saved);
					if (Array.isArray(parsed)) {
						this.pages = parsed;
						console.log('Loaded', this.pages.length, 'pages from localStorage (legacy)');
						this.runPageMigrations();
					}
				} catch (e) {
					console.error('Error loading pages from localStorage:', e);
				}
			}
		},
		
		/**
		 * Migrate pages from localStorage to WordPress database
		 */
		migratePagesToDatabase: function() {
			const self = this;
			let migrated = 0;
			
			this.pages.forEach(function(page) {
				$.ajax({
					url: buildfulyAdmin.ajaxUrl,
					type: 'POST',
					data: {
						action: 'buildfuly_save_page_data',
						nonce: buildfulyAdmin.adminNonce,
						page_data: JSON.stringify(page)
					},
					success: function(response) {
						migrated++;
						console.log('[MIGRATION] Migrated page:', page.id, '(' + migrated + '/' + self.pages.length + ')');
						
						// Clear localStorage after all pages migrated
						if (migrated === self.pages.length) {
							console.log('[MIGRATION] All pages migrated! Clearing localStorage...');
							try {
								localStorage.removeItem('buildfuly_pages');
							} catch (e) {
								console.warn('Could not clear localStorage:', e);
							}
						}
					}
				});
			});
		},
		
		/**
		 * Run data migrations on loaded pages
		 */
		runPageMigrations: function() {
			let needsSave = false;
			const self = this;
			
			this.pages.forEach(page => {
				if (page.sections && Array.isArray(page.sections)) {
					page.sections.forEach(section => {
						if (section.html) {
							// Fix 1: Escaped HTML entities in custom-code sections
							if (section.type === 'custom-code') {
								if (section.html.includes('&lt;') || section.html.includes('&gt;')) {
									console.log('[MIGRATION] Fixing escaped HTML in section:', section.id);
									section.html = this.unescapeHtml(section.html);
									section.html = this.encodeScriptTags(section.html);
									needsSave = true;
								}
							}
							
							// Fix 2: Orphaned CSS (CSS without <style> tags)
							const orphanedCssPattern = /^(\s*(?:\.[\w-]+\s*\{[^}]*\}[\s\S]*?)+)(<[a-zA-Z])/;
							if (orphanedCssPattern.test(section.html)) {
								console.log('[MIGRATION] Removing orphaned CSS in section:', section.id);
								section.html = section.html.replace(orphanedCssPattern, '$2');
								needsSave = true;
							}
							
							// Fix 3: CSS appearing after HTML but without <style> tags
							const trailingCssPattern = /(<\/[^>]+>)\s*(\.[a-zA-Z][\w-]*\s*\{[\s\S]*)/;
							if (trailingCssPattern.test(section.html) && !/<style/i.test(section.html)) {
								console.log('[MIGRATION] Removing trailing orphaned CSS in section:', section.id);
								section.html = section.html.replace(trailingCssPattern, '$1');
								needsSave = true;
							}
						}
					});
				}
			});
			
			if (needsSave) {
				console.log('[MIGRATION] Saving migrated pages to WordPress');
				// Save each migrated page
				this.pages.forEach(page => {
					self.currentPage = page;
					self.savePageToDatabase();
				});
			}
		},

		saveThemeColors: function() {
			localStorage.setItem('buildfuly_theme_colors', JSON.stringify(this.themeColors));
			
			// Also save to WordPress options
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_save_theme_colors',
					nonce: $('#buildfuly-nonce').val(),
					colors: JSON.stringify(this.themeColors)
				}
			});
		},

	loadThemeColors: function() {
		// First try to load from PHP (passed via wp_localize_script)
		if (typeof buildfulyAdmin !== 'undefined' && buildfulyAdmin.themeColors) {
			this.themeColors = buildfulyAdmin.themeColors;
			console.log('Loaded theme colors from PHP:', this.themeColors);
		} else {
			// Fallback to localStorage
			const saved = localStorage.getItem('buildfuly_theme_colors');
			if (saved) {
				try {
					this.themeColors = JSON.parse(saved);
					console.log('Loaded theme colors from localStorage:', this.themeColors);
				} catch (e) {
					console.error('Error loading theme colors:', e);
				}
			}
		}
		
		// Default colors if nothing found
		if (!this.themeColors) {
			this.themeColors = {
				primary: '#00bfa5',
				secondary: '#00897b',
				accent: '#7c4dff',
				text: '#1e293b',
				bg: '#ffffff'
			};
		}
	},

	updateColorPickers: function() {
		if (this.themeColors) {
			$('#color-primary').val(this.themeColors.primary || '#00bfa5');
			$('#color-secondary').val(this.themeColors.secondary || '#00897b');
			$('#color-accent').val(this.themeColors.accent || '#7c4dff');
			$('#color-text').val(this.themeColors.text || '#1e293b');
			$('#color-bg').val(this.themeColors.bg || this.themeColors.background || '#ffffff');
			$('#color-text-light').val(this.themeColors['text-light'] || '#64748b');
			
			// Also update the mini swatches to match
			this.updateColorMiniSwatches(this.themeColors);
		}
	},		publishPage: function(e) {
			e.preventDefault();
			
			if (!this.currentPage) {
				this.showMessage('No page selected', 'error');
				return;
			}
			
			// Get title from either title or name property
			const pageTitle = this.currentPage.title || this.currentPage.name || '';
			
			if (!pageTitle) {
				this.showMessage('Page title is required', 'error');
				return;
			}
			
			// Check if this is an update or new publish
			const isUpdate = this.currentPage.wordpress_id && this.currentPage.wordpress_id > 0;
			const loadingMessage = isUpdate ? 'Updating published page...' : 'Publishing page to WordPress...';
			this.showLoading(loadingMessage);
			
			// Generate styled HTML with inline CSS
			const styledHtml = this.generateStyledHtml();
			
		// Convert keywords array to comma-separated string for publishing
			let keywordsString = '';
			if (this.currentPage.keywords && Array.isArray(this.currentPage.keywords) && this.currentPage.keywords.length > 0) {
				keywordsString = this.currentPage.keywords.join(', ');
			} else if (this.currentPage.keyword) {
				keywordsString = this.currentPage.keyword;
			}
			// Fallback to page title if no keywords
			if (!keywordsString) {
				keywordsString = pageTitle;
			}
			
			// Store isUpdate flag for use in success callback
			const wasUpdate = isUpdate;
			
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_publish_page',
				nonce: $('#buildfuly-nonce').val(),
				page_title: pageTitle,
				page_content: styledHtml,
				page_keywords: keywordsString,
				page_colors: JSON.stringify(this.currentPage.colors),
				wordpress_id: this.currentPage.wordpress_id || 0
			},
			success: (response) => {
				this.hideLoading();
				if (response.success) {
					// Store WordPress ID in the page object
					if (response.data.wordpress_id) {
						this.currentPage.wordpress_id = response.data.wordpress_id;
						
						// Also update in pages array (in case currentPage isn't a reference)
						const pageIndex = this.pages.findIndex(p => p.id === this.currentPage.id);
						if (pageIndex !== -1) {
							this.pages[pageIndex].wordpress_id = response.data.wordpress_id;
						}
						
						console.log('✓ WordPress ID saved:', response.data.wordpress_id, 'for page:', this.currentPage.id);
					}
					
					// Store SEO data in the page object
					if (response.data.seo_data) {
						this.currentPage.seo = {
							page_id: response.data.wordpress_id,
							slug: response.data.seo_data.slug || '',
							meta_title: response.data.seo_data.title || this.currentPage.title,
							meta_description: response.data.seo_data.description || '',
							focus_keyphrase: response.data.seo_data.focus_keyphrase || '',
							word_count: response.data.seo_data.word_count || 0,
							keyword_density: response.data.seo_data.keyword_density || '0%'
						};
						
						// Also update in pages array
						const pageIndex = this.pages.findIndex(p => p.id === this.currentPage.id);
						if (pageIndex !== -1) {
							this.pages[pageIndex].seo = this.currentPage.seo;
						}
					}
					
					// Save page with SEO data
					this.savePage();
					
					// Re-render pages list to show publish status
					this.renderPagesList();
					
					// Update publish button to show "Update" state
					this.updatePublishButtonState();
					
					// Trigger event for SEO metadata UI
					$(document).trigger('buildfuly:pagePublished', response.data);
					
					// Different messages for publish vs update
					const successTitle = wasUpdate ? '✅ Page Updated!' : '✅ Page Published with SEO!';
					let message = '<strong>' + successTitle + '</strong><br><br>';
					message += '<a href="' + response.data.page_url + '" target="_blank" class="button button-primary">View Page</a><br><br>';
						
						// Show usage info
						if (response.data.usage) {
							message += '<div style="background: #f0f9ff; border-left: 4px solid #0ea5e9; padding: 12px; margin: 12px 0; border-radius: 4px;">';
							message += '📊 <strong>Usage:</strong> ' + response.data.usage.message;
							if (response.data.usage.remaining <= 2) {
								message += '<br><span style="color: #dc2626;">⚠️ Low usage remaining!</span>';
							}
							message += '</div>';
						}
						
						if (response.data.seo_data) {
							message += '<div style="text-align: left; margin-top: 12px; font-size: 13px; line-height: 1.8;">';
							message += '<strong>SEO Details:</strong><br>';
							message += '📄 <strong>Title:</strong> ' + (response.data.seo_data.title || 'Generated') + '<br>';
							message += '📝 <strong>Description:</strong> ' + (response.data.seo_data.description || 'Generated') + '<br>';
							message += '🔗 <strong>URL Slug:</strong> /' + response.data.seo_data.slug + '<br>';
							message += '📊 <strong>Word Count:</strong> ' + response.data.seo_data.word_count + ' words<br>';
							message += '🎯 <strong>Keyword Density:</strong> ' + response.data.seo_data.keyword_density + '<br>';
							message += '🔑 <strong>Focus Keyphrase:</strong> ' + response.data.seo_data.focus_keyphrase;
							message += '</div>';
						}
						
						this.showMessage(message, 'success', 10000);
					} else {
						this.showMessage('Failed to publish page: ' + (response.data.message || 'Unknown error'), 'error');
					}
				},
				error: () => {
					this.hideLoading();
					this.showMessage('Error publishing page', 'error');
				}
			});
		},

		generateStyledHtml: function() {
			// Concatenate section HTML with proper wrappers for custom code sections
			const sectionsHtml = this.currentPage.sections.map(section => {
				let html = section.html;
				
				// Clean up drag handles and wrappers before publishing
				const $temp = $('<div>').html(html);
				$temp.find('.buildfuly-drag-handle').remove(); // Remove drag handles
				$temp.find('.buildfuly-draggable-wrapper').each(function() {
					$(this).replaceWith($(this).contents()); // Unwrap, keeping inner content
				});
				html = $temp.html();
				
				// Wrap custom code sections in a centered container for proper layout
				// text-align: left prevents inherited centering from breaking form elements
				if (section.type === 'custom-code') {
					return '<div style="max-width: 1200px; margin: 0 auto; padding: 40px 20px; text-align: left;">' + html + '</div>';
				}
				return html;
			}).join('\n\n');
			
			return sectionsHtml;
		},

		switchDevice: function(e) {
			e.preventDefault();
			
			const device = $(e.currentTarget).data('device');
			this.currentDevice = device;
			
			$('.buildfuly-device-btn').removeClass('active');
			$(e.currentTarget).addClass('active');
			
			const $container = $('#buildfuly-sections-container');
			$container.removeClass('device-desktop device-tablet device-mobile');
			$container.addClass('device-' + device);
		},

		toggleKeywords: function(e) {
			e.preventDefault();
			const $list = $('#buildfuly-keywords-list');
			const $arrow = $('#buildfuly-keywords-arrow');
			const $preview = $('#buildfuly-keywords-preview');
			
			const isVisible = $list.is(':visible');
			
			if (isVisible) {
				// Collapsing
				$list.slideUp(200);
				$arrow.removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
				
				// Show keywords as text
				const keywords = this.currentPage.keywords || [];
				const keywordText = keywords.map(k => typeof k === 'string' ? k : (k.keyword || '')).filter(k => k).join(', ');
				$preview.text(keywordText || 'No keywords').show();
			} else {
				// Expanding
				$list.slideDown(200);
				$arrow.removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
				$preview.hide();
			}
		},

		toggleSidebar: function(e) {
			e.preventDefault();
			const $sidebar = $('#buildfuly-pages-sidebar');
			const $toggle = $('#buildfuly-sidebar-toggle .dashicons');
			const $expandBtn = $('#buildfuly-sidebar-expand');
			const $collapseBtn = $('#buildfuly-sidebar-collapse-float');
			
			$sidebar.toggleClass('collapsed');
			$toggle.toggleClass('dashicons-arrow-left-alt2 dashicons-arrow-right-alt2');
			
			// Show/hide floating buttons
			if ($sidebar.hasClass('collapsed')) {
				$expandBtn.fadeIn(200);
				$collapseBtn.fadeOut(200);
			} else {
				$expandBtn.fadeOut(200);
				$collapseBtn.fadeIn(200);
			}
		},

		toggleColors: function(e) {
			e.preventDefault();
			const $full = $('#buildfuly-color-swatches-full');
			const $mini = $('#buildfuly-color-swatches-mini');
			const $arrow = $('#buildfuly-colors-arrow');
			
			if ($full.is(':visible')) {
				// Collapsing
				$full.slideUp(200);
				$mini.slideDown(200);
				$arrow.removeClass('expanded');
			} else {
				// Expanding
				$mini.slideUp(200);
				$full.slideDown(200);
				$arrow.addClass('expanded');
			}
		},

		updateColorMiniSwatches: function(colors) {
			$('#buildfuly-color-swatches-mini .buildfuly-color-mini').each(function() {
				const colorType = $(this).data('color');
				let colorValue = '#000000';
				
				switch(colorType) {
					case 'primary':
						colorValue = colors.primary || '#00bfa5';
						break;
					case 'secondary':
						colorValue = colors.secondary || '#00897b';
						break;
					case 'text':
						colorValue = colors.text || '#1e293b';
						break;
					case 'text-light':
						colorValue = colors['text-light'] || '#64748b';
						break;
					case 'bg':
						colorValue = colors.background || '#000000';
						break;
				}
				
				$(this).css('background', colorValue);
			});
		},

		/**
		 * Show loading overlay - delegates to Utils
		 */
		showLoading: function(message = 'Processing...') {
			Utils.showLoading(message);
		},

		/**
		 * Hide loading overlay - delegates to Utils
		 */
		hideLoading: function() {
			Utils.hideLoading();
		},

	/**
	 * Show message notification - enhanced version with error handling
	 * Keeps custom error behavior but delegates basic messaging to Utils
	 */
	showMessage: function(message, type = 'info', duration = 3000) {
		// For error messages, require manual dismiss (custom behavior)
		if (type === 'error') {
			const $message = $(`
				<div class="buildfuly-message buildfuly-message-${type}" style="display: flex; align-items: center; justify-content: space-between;">
					<span style="flex: 1;">${Utils.escapeHtml(message)}</span>
					<button class="buildfuly-message-close" style="margin-left: 12px; background: none; border: none; color: inherit; cursor: pointer; font-size: 18px; line-height: 1; padding: 0 4px;">&times;</button>
				</div>
			`);
			
			$('#buildfuly-messages').append($message);
			
			$message.find('.buildfuly-message-close').on('click', function() {
				$message.fadeOut(300, function() {
					$(this).remove();
				});
			});
			return;
		}
		
		// For non-errors, delegate to Utils
		Utils.showMessage(message, type, duration);
	},

	uploadPageImages: function(e) {
			e.preventDefault();
			
			// Open WordPress Media Library
			const mediaUploader = wp.media({
				title: 'Select Page Images',
				button: {
					text: 'Add Images'
				},
				multiple: true,
				library: {
					type: 'image'
				}
			});
			
			mediaUploader.on('select', function() {
				const attachments = mediaUploader.state().get('selection').toJSON();
				const $preview = $('#buildfuly-page-images-preview');
				const currentIds = $('#buildfuly-page-images-ids').val();
				const idsArray = currentIds ? currentIds.split(',') : [];
				
				attachments.forEach(function(attachment) {
					// Limit to 7 images
					if (idsArray.length >= 7) {
						alert('Maximum 7 images allowed for optimal page design');
						return;
					}
					
					idsArray.push(attachment.id);
					
					const imageHtml = `
						<div class="buildfuly-page-image-item" data-id="${attachment.id}" style="position: relative; border-radius: 8px; overflow: hidden; aspect-ratio: 4/3; background: #f3f4f6;">
							<img src="${attachment.sizes.thumbnail ? attachment.sizes.thumbnail.url : attachment.url}" alt="" style="width: 100%; height: 100%; object-fit: cover;">
							<button type="button" class="buildfuly-remove-page-image" data-id="${attachment.id}" style="position: absolute; top: 4px; right: 4px; width: 24px; height: 24px; background: rgba(0,0,0,0.7); color: white; border: none; border-radius: 50%; cursor: pointer; font-size: 16px; line-height: 1;">&times;</button>
						</div>
					`;
					$preview.append(imageHtml);
				});
				
				$('#buildfuly-page-images-ids').val(idsArray.join(','));
			});
			
			mediaUploader.open();
		},

		removePageImage: function(e) {
			e.preventDefault();
			const imageId = $(e.currentTarget).data('id');
			const $item = $(e.currentTarget).closest('.buildfuly-page-image-item');
			
			// Remove from UI
			$item.fadeOut(200, function() {
				$(this).remove();
			});
			
			// Update hidden field
			const currentIds = $('#buildfuly-page-images-ids').val();
			const idsArray = currentIds.split(',').filter(id => id != imageId);
			$('#buildfuly-page-images-ids').val(idsArray.join(','));
		},

		loadPageImagesPreview: function(imageIds) {
			if (!imageIds || imageIds.length === 0) {
				$('#buildfuly-page-images-preview').empty();
				return;
			}
			
			const $preview = $('#buildfuly-page-images-preview');
			$preview.empty();
			
			// Fetch image URLs from WordPress via AJAX
			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'buildfuly_get_image_url',
					nonce: buildfulyAdmin.nonce,
					image_ids: imageIds
				},
				success: (response) => {
					if (response.success && response.data) {
						imageIds.forEach(imageId => {
							const imageData = response.data[imageId];
							if (imageData) {
								const imageHtml = `
									<div class="buildfuly-page-image-item" data-id="${imageId}" style="position: relative; border-radius: 8px; overflow: hidden; aspect-ratio: 4/3; background: #f3f4f6;">
										<img src="${imageData.thumbnail || imageData.url}" alt="" style="width: 100%; height: 100%; object-fit: cover;">
										<button type="button" class="buildfuly-remove-page-image" data-id="${imageId}" style="position: absolute; top: 4px; right: 4px; width: 24px; height: 24px; background: rgba(0,0,0,0.7); color: white; border: none; border-radius: 50%; cursor: pointer; font-size: 16px; line-height: 1;">&times;</button>
									</div>
								`;
								$preview.append(imageHtml);
							}
						});
					}
				},
				error: () => {
					console.error('Failed to load page images preview');
				}
			});
		},

	
	openEditPanel: function(sectionId) {
		if (!this.currentPage || !this.currentPage.sections) return;
		
		const section = this.currentPage.sections.find(s => s.id === sectionId);
		if (!section) return;

		// Special handling for custom-code sections
		if (section.type === 'custom-code') {
			this.showCustomCodeEditPanel(sectionId, section);
			return;
		}

		console.log('Section data:', section);
		console.log('AI Content:', section.ai_content);

		const fields = [];
		
		// Check if section has AI content (JSON structure)
		if (section.ai_content && typeof section.ai_content === 'object') {
			console.log('Using AI content JSON extraction');
			// Extract fields from AI content JSON
			this.extractFieldsFromJSON(section.ai_content, fields, '');
		} else {
			console.log('Using HTML fallback extraction');
			// Fallback: extract from HTML (less reliable)
			const $content = $('<div>').html(section.html);
			this.extractFieldsFromHTML($content, fields);
		}
		
		console.log('Extracted fields:', fields);

		this.showEditPanel(sectionId, fields);
	},
	
	extractFieldsFromJSON: function(obj, fields, prefix) {
		for (let key in obj) {
			if (!obj.hasOwnProperty(key)) continue;
			
			const value = obj[key];
			const fieldPath = prefix ? `${prefix}.${key}` : key;
			
			if (typeof value === 'string' && value.length > 0) {
				// Format label nicely
				let label = key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
				let fieldType = value.length > 60 || key.includes('description') || key.includes('answer') || key.includes('quote') ? 'textarea' : 'text';
				
				fields.push({
					path: fieldPath,
					label: label,
					value: value,
					type: fieldType
				});
			} else if (Array.isArray(value)) {
				// Handle arrays (testimonials, features, etc.)
				value.forEach((item, index) => {
					if (typeof item === 'object') {
						this.extractFieldsFromJSON(item, fields, `${fieldPath}[${index}]`);
					} else if (typeof item === 'string') {
						fields.push({
							path: `${fieldPath}[${index}]`,
							label: `${key} ${index + 1}`,
							value: item,
							type: 'text'
						});
					}
				});
			} else if (typeof value === 'object' && value !== null) {
				// Nested object
				this.extractFieldsFromJSON(value, fields, fieldPath);
			}
		}
	},
	
	extractFieldsFromHTML: function($content, fields) {
		// Fallback method for sections without AI content
		// Only extract from semantic content tags, skip buttons and navigation
		const seenTexts = new Set();
		
		$content.find('h1, h2, h3, h4, h5, h6, p').each(function() {
			const $el = $(this);
			
			// Skip if inside a button or link
			if ($el.closest('button, a.btn, a.button').length > 0) return;
			
			// Get text content
			let text = $el.text().trim();
			
			// Skip empty, very short, or duplicate text
			if (!text || text.length < 3 || seenTexts.has(text)) return;
			
			seenTexts.add(text);
			
			const tagName = $el.prop('tagName').toLowerCase();
			let label = tagName.toUpperCase();
			let fieldType = 'text';
			
			if (tagName.match(/^h[1-6]$/)) {
				label = 'Heading';
				fieldType = 'text';
			} else if (tagName === 'p') {
				label = 'Paragraph';
				fieldType = 'textarea';
			}
			
			fields.push({
				path: null,
				label: label,
				value: text,
				type: fieldType,
				htmlIndex: fields.length,
				tagName: tagName
			});
		});
	},

	showEditPanel: function(sectionId, fields) {
		$('#buildfuly-edit-panel').remove();
		
		// Get section info
		const section = this.currentPage.sections.find(s => s.id === sectionId);
		const sectionType = section ? (section.type || 'Unknown').replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase()) : 'Section';
		
		// Check if this component supports images (dynamic: any hero-, content-image, content-split, feature-)
		const supportsImage = section.type && (
			section.type.startsWith('hero-') || 
			section.type.startsWith('feature-') ||
			section.type === 'content-image' || 
			section.type === 'content-split'
		);
		
		// Extract current image from HTML if not in section metadata
		let currentImageUrl = '';
		let currentImageId = section.image_id || 0;
		
		if (supportsImage && section.html) {
			const $html = $('<div>').html(section.html);
			const $img = $html.find('img').first();
			if ($img.length) {
				currentImageUrl = $img.attr('src');
				// Also update section metadata if missing
				if (!section.image_id && section.ai_content && section.ai_content.image_id) {
					currentImageId = section.ai_content.image_id;
				}
			}
		}
		
		// Add icon for field type
		const getFieldIcon = (label) => {
			if (label.includes('Heading') || label.includes('H1') || label.includes('H2') || label.includes('H3')) return 'editor-textcolor';
			if (label.includes('Paragraph') || label.includes('Description')) return 'editor-alignleft';
			if (label.includes('Title')) return 'star-filled';
			return 'edit';
		};

		// Build image upload section if component supports it
		let imageUploadHtml = '';
		if (supportsImage) {
			imageUploadHtml = `
				<div class="buildfuly-edit-field">
					<label><span class="dashicons dashicons-format-image"></span>Component Image</label>
					<div style="display: flex; gap: 8px; align-items: center; flex-wrap: wrap;">
						<button type="button" class="buildfuly-btn buildfuly-btn-secondary" id="buildfuly-edit-upload-image" style="flex: 1; min-width: 100px;">
							<span class="dashicons dashicons-upload" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span>
							Choose
						</button>
						<button type="button" class="buildfuly-btn" id="buildfuly-edit-generate-ai-image" style="flex: 1; min-width: 100px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none; color: white;">
							<span class="dashicons dashicons-superhero-alt" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span>
							AI Generate
						</button>
						<button type="button" class="buildfuly-btn buildfuly-btn-secondary" id="buildfuly-edit-remove-image" style="background: #ef4444; border-color: #ef4444; color: white; flex: 0 0 auto;">
							<span class="dashicons dashicons-no" style="font-size: 16px; width: 16px; height: 16px;"></span>
						</button>
					</div>
					<input type="hidden" id="buildfuly-edit-image-id" value="${currentImageId}">
					<div id="buildfuly-edit-image-preview" style="margin-top: 12px; display: ${currentImageUrl ? 'block' : 'none'};">
						<img src="${currentImageUrl}" style="max-width: 100%; height: auto; border-radius: 8px; border: 2px solid #e5e7eb;">
					</div>
				</div>
			`;
		}
		
		// Build button controls section if component uses buttons (dynamic: any hero- or cta-)
		const usesButtons = section.type && (
			section.type.startsWith('hero-') || 
			section.type.startsWith('cta-')
		);
		let buttonControlsHtml = '';
		
		if (usesButtons) {
			const useCustomButtons = section.ai_content && section.ai_content.use_custom_buttons;
			const customButtons = section.ai_content && section.ai_content.custom_buttons || [];
			
			buttonControlsHtml = `
				<div class="buildfuly-edit-field" style="border-top: 1px solid #e5e7eb; padding-top: 20px; margin-top: 20px;">
					<label style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 12px;">
						<span><span class="dashicons dashicons-admin-links"></span>Button Settings</span>
						<label style="display: flex; align-items: center; gap: 8px; font-weight: normal; cursor: pointer;">
							<input type="checkbox" id="buildfuly-use-custom-buttons" ${useCustomButtons ? 'checked' : ''} style="margin: 0;">
							<span style="font-size: 13px;">Override global buttons</span>
						</label>
					</label>
					<div id="buildfuly-button-settings" style="display: ${useCustomButtons ? 'block' : 'none'};">
						<div style="margin-bottom: 12px;">
							<label style="font-size: 12px; color: #6b7280; margin-bottom: 4px; display: block;">Primary Button Text</label>
							<input type="text" id="buildfuly-button1-text" value="${customButtons[0]?.text || 'Start For Free'}" style="width: 100%; padding: 8px; border: 1px solid #e5e7eb; border-radius: 6px;">
						</div>
						<div style="margin-bottom: 12px;">
							<label style="font-size: 12px; color: #6b7280; margin-bottom: 4px; display: block;">Primary Button URL</label>
							<input type="text" id="buildfuly-button1-url" value="${customButtons[0]?.url || '#contact'}" style="width: 100%; padding: 8px; border: 1px solid #e5e7eb; border-radius: 6px;">
						</div>
						<div style="margin-bottom: 12px;">
							<label style="font-size: 12px; color: #6b7280; margin-bottom: 4px; display: block;">Secondary Button Text (optional)</label>
							<input type="text" id="buildfuly-button2-text" value="${customButtons[1]?.text || ''}" style="width: 100%; padding: 8px; border: 1px solid #e5e7eb; border-radius: 6px;">
						</div>
						<div>
							<label style="font-size: 12px; color: #6b7280; margin-bottom: 4px; display: block;">Secondary Button URL</label>
							<input type="text" id="buildfuly-button2-url" value="${customButtons[1]?.url || '#demo'}" style="width: 100%; padding: 8px; border: 1px solid #e5e7eb; border-radius: 6px;">
						</div>
					</div>
					<div id="buildfuly-global-buttons-info" style="display: ${useCustomButtons ? 'none' : 'block'}; padding: 12px; background: #f3f4f6; border-radius: 6px; font-size: 13px; color: #6b7280;">
						<span class="dashicons dashicons-info" style="float: left; margin-right: 8px;"></span>
						Using global button settings. Changes to global buttons will update all linked components automatically.
					</div>
				</div>
			`;
		}

		let fieldsHtml = '';
		if (fields.length === 0) {
			fieldsHtml = '<div class="buildfuly-edit-no-fields" style="text-align: center; padding: 40px 20px; color: #6b7280;"><span class="dashicons dashicons-info" style="font-size: 48px; width: 48px; height: 48px; opacity: 0.5;"></span><p style="margin-top: 16px; font-size: 14px;">No editable text fields found in this section.</p></div>';
		} else {
			fields.forEach((field, index) => {
				const icon = getFieldIcon(field.label);
				if (field.type === 'textarea') {
					fieldsHtml += `
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-${icon}"></span>${field.label}</label>
							<textarea class="buildfuly-edit-input" data-index="${index}" rows="4">${field.value}</textarea>
						</div>
					`;
				} else {
					fieldsHtml += `
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-${icon}"></span>${field.label}</label>
							<input type="text" class="buildfuly-edit-input" data-index="${index}" value="${field.value}">
						</div>
					`;
				}
			});
		}

		const panel = $(`
			<div id="buildfuly-edit-panel" class="buildfuly-edit-panel">
				<div class="buildfuly-edit-panel-header">
					<div class="buildfuly-edit-panel-header-content">
						<h3>Edit Component</h3>
						<p class="buildfuly-edit-panel-subtitle">
							<span class="dashicons dashicons-admin-settings"></span>
							${sectionType}
						</p>
					</div>
					<button class="buildfuly-edit-panel-close">
						<span class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				
				<!-- Tab Navigation -->
				<div class="buildfuly-edit-panel-tabs">
					<button class="buildfuly-edit-tab active" data-tab="content">
						<span class="dashicons dashicons-edit"></span>
						Content
					</button>
					<button class="buildfuly-edit-tab" data-tab="styling">
						<span class="dashicons dashicons-art"></span>
						Styling
					</button>
					<button class="buildfuly-edit-tab" data-tab="advanced">
						<span class="dashicons dashicons-admin-generic"></span>
						Advanced
					</button>
				</div>
				
				<div class="buildfuly-edit-panel-body">
					<!-- Content Tab -->
					<div class="buildfuly-edit-tab-content active" data-tab-content="content">
						<div class="buildfuly-edit-section-info">
							<div class="buildfuly-edit-section-info-title">Component Type</div>
							<div class="buildfuly-edit-section-info-type">${sectionType}</div>
						</div>
						${imageUploadHtml}
						${buttonControlsHtml}
						${fieldsHtml}
					</div>
					
					<!-- Styling Tab -->
					<div class="buildfuly-edit-tab-content" data-tab-content="styling" style="display: none;">
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-admin-appearance"></span>Background Color</label>
							<input type="color" id="buildfuly-edit-bg-color" class="buildfuly-color-input" value="${section.style?.backgroundColor || '#ffffff'}">
						</div>
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-editor-textcolor"></span>Text Color</label>
							<input type="color" id="buildfuly-edit-text-color" class="buildfuly-color-input" value="${section.style?.textColor || '#1f2937'}">
						</div>
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-move"></span>Padding (px)</label>
							<input type="range" id="buildfuly-edit-padding" min="0" max="200" value="${section.style?.padding || 100}" class="buildfuly-range-input">
							<span class="buildfuly-range-value">${section.style?.padding || 100}px</span>
						</div>
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-editor-justify"></span>Border Radius (px)</label>
							<input type="range" id="buildfuly-edit-radius" min="0" max="50" value="${section.style?.borderRadius || 0}" class="buildfuly-range-input">
							<span class="buildfuly-range-value">${section.style?.borderRadius || 0}px</span>
						</div>
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-visibility"></span>Shadow Intensity</label>
							<select id="buildfuly-edit-shadow" class="buildfuly-select-input">
								<option value="none" ${section.style?.shadow === 'none' ? 'selected' : ''}>None</option>
								<option value="sm" ${!section.style?.shadow || section.style?.shadow === 'sm' ? 'selected' : ''}>Small</option>
								<option value="md" ${section.style?.shadow === 'md' ? 'selected' : ''}>Medium</option>
								<option value="lg" ${section.style?.shadow === 'lg' ? 'selected' : ''}>Large</option>
								<option value="xl" ${section.style?.shadow === 'xl' ? 'selected' : ''}>Extra Large</option>
							</select>
						</div>
					</div>
					
					<!-- Advanced Tab -->
					<div class="buildfuly-edit-tab-content" data-tab-content="advanced" style="display: none;">
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-admin-links"></span>Custom CSS Class</label>
							<input type="text" id="buildfuly-edit-custom-class" class="buildfuly-edit-input" placeholder="my-custom-class" value="${section.customClass || ''}">
							<p class="description">Add custom CSS classes for styling</p>
						</div>
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-visibility"></span>Animation</label>
							<select id="buildfuly-edit-animation" class="buildfuly-select-input">
								<option value="none" ${!section.animation || section.animation === 'none' ? 'selected' : ''}>None</option>
								<option value="fade" ${section.animation === 'fade' ? 'selected' : ''}>Fade In</option>
								<option value="slide-up" ${section.animation === 'slide-up' ? 'selected' : ''}>Slide Up</option>
								<option value="slide-left" ${section.animation === 'slide-left' ? 'selected' : ''}>Slide Left</option>
								<option value="zoom" ${section.animation === 'zoom' ? 'selected' : ''}>Zoom In</option>
							</select>
						</div>
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-arrow-up-alt"></span>Spacing Above (px)</label>
							<input type="range" id="buildfuly-edit-margin-top" min="0" max="200" value="${section.style?.marginTop || 0}" class="buildfuly-range-input">
							<span class="buildfuly-range-value">${section.style?.marginTop || 0}px</span>
						</div>
						<div class="buildfuly-edit-field">
							<label><span class="dashicons dashicons-arrow-down-alt"></span>Spacing Below (px)</label>
							<input type="range" id="buildfuly-edit-margin-bottom" min="0" max="200" value="${section.style?.marginBottom || 0}" class="buildfuly-range-input">
							<span class="buildfuly-range-value">${section.style?.marginBottom || 0}px</span>
						</div>
						<div class="buildfuly-edit-field">
							<label style="display: flex; align-items: center; gap: 8px;">
								<input type="checkbox" id="buildfuly-edit-full-width" ${section.fullWidth ? 'checked' : ''}>
								<span>Full Width Container</span>
							</label>
						</div>
					</div>
				</div>
				<div class="buildfuly-edit-panel-footer">
					<button class="buildfuly-btn buildfuly-btn-secondary" id="buildfuly-edit-cancel">Cancel</button>
					<button class="buildfuly-btn buildfuly-btn-primary" id="buildfuly-edit-save">
						<span class="dashicons dashicons-saved" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span>
						Save Changes
					</button>
				</div>
			</div>
		`);

		$('body').append(panel);
		setTimeout(() => panel.addClass('active'), 10);

		// Bind events
		panel.find('.buildfuly-edit-panel-close, #buildfuly-edit-cancel').on('click', () => this.closeEditPanel());
		panel.find('#buildfuly-edit-save').on('click', () => this.saveEditedContent(sectionId, fields));
		panel.find('#buildfuly-edit-upload-image').on('click', () => this.uploadEditImage(sectionId));
		panel.find('#buildfuly-edit-generate-ai-image').on('click', () => this.generateAIImage(sectionId));
		panel.find('#buildfuly-edit-remove-image').on('click', () => this.removeEditImage(sectionId));
		
		// Tab switching
		panel.find('.buildfuly-edit-tab').on('click', function() {
			const tabName = $(this).data('tab');
			panel.find('.buildfuly-edit-tab').removeClass('active');
			panel.find('.buildfuly-edit-tab-content').hide();
			$(this).addClass('active');
			panel.find(`.buildfuly-edit-tab-content[data-tab-content="${tabName}"]`).show();
		});
		
		// Range input live updates
		panel.find('.buildfuly-range-input').on('input', function() {
			$(this).next('.buildfuly-range-value').text($(this).val() + 'px');
		});
		
		// Button settings toggle
		panel.find('#buildfuly-use-custom-buttons').on('change', function() {
			const useCustom = $(this).is(':checked');
			$('#buildfuly-button-settings').toggle(useCustom);
			$('#buildfuly-global-buttons-info').toggle(!useCustom);
		});
	},
	
	/**
	 * Show special edit panel for custom code sections
	 * Provides a code editor with instructions and form shortcode dropdown
	 */
	showCustomCodeEditPanel: function(sectionId, section) {
		$('#buildfuly-edit-panel').remove();
		
		// Decode any encoded scripts back to regular script tags for editing
		let editableHtml = section.html || '';
		
		// Convert data-script divs back to script tags for editing
		const scriptDivRegex = /<div data-script="([^"]+)"[^>]*><\/div>/gi;
		editableHtml = editableHtml.replace(scriptDivRegex, (match, encoded) => {
			try {
				const decoded = decodeURIComponent(escape(atob(encoded)));
				return `<script>\n${decoded}\n</script>`;
			} catch (e) {
				return match;
			}
		});
		
		// Remove HTML comments (instructions)
		editableHtml = editableHtml.replace(/<!--[\s\S]*?-->/g, '').trim();

		const panel = $(`
			<div id="buildfuly-edit-panel" class="buildfuly-edit-panel" style="width: 600px;">
				<div class="buildfuly-edit-panel-header">
					<div class="buildfuly-edit-panel-header-content">
						<h3>Edit Custom Code</h3>
						<p class="buildfuly-edit-panel-subtitle">
							<span class="dashicons dashicons-editor-code"></span>
							HTML, CSS, JavaScript or Shortcodes
						</p>
					</div>
					<button class="buildfuly-edit-panel-close">
						<span class="dashicons dashicons-no-alt"></span>
					</button>
				</div>
				<div class="buildfuly-edit-panel-body" style="padding: 0;">
					<div style="padding: 16px; background: #ecfdf5; border-bottom: 1px solid #a7f3d0;">
						<div style="font-weight: 600; color: #047857; margin-bottom: 10px;">
							<span class="dashicons dashicons-forms" style="font-size: 16px; width: 16px; height: 16px;"></span>
							Insert Form Shortcode
						</div>
						<div style="display: flex; gap: 10px; align-items: center;">
							<select id="buildfuly-form-shortcode-select" style="flex: 1; padding: 8px 12px; border: 1px solid #d1d5db; border-radius: 6px; font-size: 14px;">
								<option value="">-- Select a form --</option>
							</select>
							<button type="button" class="button" id="buildfuly-insert-form-shortcode" style="white-space: nowrap;">
								<span class="dashicons dashicons-plus-alt" style="font-size: 16px; width: 16px; height: 16px; vertical-align: middle;"></span>
								Insert
							</button>
						</div>
					</div>
					<div style="padding: 12px 16px; background: #f0f9ff; border-bottom: 1px solid #bae6fd;">
						<div style="font-size: 12px; color: #0c4a6e; line-height: 1.5;">
							<strong>Tips:</strong> Use <code>&lt;style&gt;</code> for CSS, plain HTML in the middle, <code>&lt;script&gt;</code> for JS (auto-encoded)
						</div>
					</div>
					<textarea id="buildfuly-custom-code-editor" style="
						width: 100%;
						height: 380px;
						font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
						font-size: 13px;
						line-height: 1.5;
						padding: 16px;
						border: none;
						resize: none;
						background: #1e293b;
						color: #e2e8f0;
						outline: none;
					" placeholder="Enter your HTML, CSS, JavaScript, or shortcode here..."></textarea>
					<div id="buildfuly-shortcode-preview-container" style="display:none; padding: 16px; background: #f8fafc; border-top: 1px solid #e2e8f0;">
						<div style="font-weight: 600; margin-bottom: 10px; color: #334155;">
							<span class="dashicons dashicons-visibility" style="font-size: 16px; width: 16px; height: 16px;"></span>
							Live Preview (Shortcodes Rendered)
						</div>
						<div id="buildfuly-shortcode-preview" style="background: #fff; border: 1px solid #e2e8f0; border-radius: 8px; padding: 20px; min-height: 100px; max-height: 400px; overflow: auto; text-align: center;"></div>
					</div>
				</div>
				<div class="buildfuly-edit-panel-footer">
					<button class="buildfuly-btn buildfuly-btn-secondary" id="buildfuly-edit-cancel">Cancel</button>
					<button class="buildfuly-btn buildfuly-btn-secondary" id="buildfuly-preview-shortcode" style="margin-left: 8px;">
						<span class="dashicons dashicons-visibility" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span>
						Preview
					</button>
					<button class="buildfuly-btn buildfuly-btn-primary" id="buildfuly-custom-code-save">
						<span class="dashicons dashicons-saved" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span>
						Save Code
					</button>
				</div>
			</div>
		`);

		$('body').append(panel);
		
		// Set textarea value using .val() - this properly handles HTML without escaping
		$('#buildfuly-custom-code-editor').val(editableHtml);
		
		setTimeout(() => panel.addClass('active'), 10);

		// Bind events
		panel.find('.buildfuly-edit-panel-close, #buildfuly-edit-cancel').on('click', () => this.closeEditPanel());
		panel.find('#buildfuly-custom-code-save').on('click', () => this.saveCustomCode(sectionId));
		panel.find('#buildfuly-preview-shortcode').on('click', () => this.previewShortcode());
		panel.find('#buildfuly-insert-form-shortcode').on('click', () => this.insertFormShortcode());
		
		// Load available forms into dropdown
		this.loadFormShortcodes();
	},
	
	/**
	 * Load form shortcodes from database
	 */
	loadFormShortcodes: function() {
		const $select = $('#buildfuly-form-shortcode-select');
		$select.html('<option value="">Loading forms...</option>');
		
		$.post(buildfulyAdmin.ajaxUrl, {
			action: 'buildfuly_get_forms_list',
			nonce: buildfulyAdmin.adminNonce
		}, (response) => {
			$select.html('<option value="">-- Select a form --</option>');
			
			if (response.success && response.data.forms && response.data.forms.length > 0) {
				response.data.forms.forEach(form => {
					$select.append(`<option value="[buildfuly_form id=&quot;${form.id}&quot;]">${form.name} (ID: ${form.id})</option>`);
				});
			} else {
				$select.append('<option value="" disabled>No forms found - create one in Leads & Forms</option>');
			}
		}).fail(() => {
			$select.html('<option value="">-- Select a form --</option>');
			$select.append('<option value="" disabled>Error loading forms</option>');
		});
	},
	
	/**
	 * Insert selected form shortcode into editor
	 */
	insertFormShortcode: function() {
		const $select = $('#buildfuly-form-shortcode-select');
		const shortcode = $select.val();
		
		if (!shortcode) {
			return;
		}
		
		const $editor = $('#buildfuly-custom-code-editor');
		const currentContent = $editor.val();
		
		// Insert shortcode (replace or append)
		if (currentContent.trim()) {
			$editor.val(currentContent + '\n\n' + shortcode);
		} else {
			$editor.val(shortcode);
		}
		
		// Reset dropdown
		$select.val('');
	},
	
	/**
	 * Preview shortcode content via AJAX
	 */
	previewShortcode: function() {
		const code = $('#buildfuly-custom-code-editor').val();
		const $container = $('#buildfuly-shortcode-preview-container');
		const $preview = $('#buildfuly-shortcode-preview');
		const $btn = $('#buildfuly-preview-shortcode');
		
		// Toggle visibility
		if ($container.is(':visible') && !$btn.hasClass('loading')) {
			$container.slideUp(200);
			$btn.html('<span class="dashicons dashicons-visibility" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span> Preview');
			return;
		}
		
		// Check if code has shortcodes
		const hasShortcodes = /\[[\w_-]+/.test(code);
		
		if (!hasShortcodes) {
			// No shortcodes - just render HTML directly in iframe
			$preview.html('');
			const iframe = document.createElement('iframe');
			iframe.style.width = '100%';
			iframe.style.minHeight = '150px';
			iframe.style.border = 'none';
			iframe.style.background = '#fff';
			
			$preview.append(iframe);
			
			const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
			iframeDoc.open();
			iframeDoc.write('<!DOCTYPE html><html><head><style>body{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;margin:0;padding:20px;}</style></head><body>' + code + '</body></html>');
			iframeDoc.close();
			
			setTimeout(() => {
				try {
					const height = iframeDoc.body.scrollHeight;
					iframe.style.height = Math.min(Math.max(height + 30, 100), 400) + 'px';
				} catch(e) {}
			}, 150);
			
			$container.slideDown(200);
			$btn.html('<span class="dashicons dashicons-hidden" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span> Hide Preview');
			return;
		}
		
		// Has shortcodes - use AJAX to render
		$btn.addClass('loading').html('<span class="spinner is-active" style="float:none; margin: 0 4px 0 0;"></span> Loading...');
		$preview.html('<div style="text-align:center; padding: 40px; color: #64748b;"><span class="spinner is-active" style="float:none;"></span><br>Rendering shortcodes...</div>');
		$container.slideDown(200);
		
		$.post(ajaxurl, {
			action: 'buildfuly_render_shortcode_preview',
			nonce: buildfulyAdmin.adminNonce,
			html_content: code
		}, (response) => {
			$btn.removeClass('loading');
			
			if (response.success) {
				// Create iframe with rendered content
				$preview.html('');
				const iframe = document.createElement('iframe');
				iframe.style.width = '100%';
				iframe.style.minHeight = '150px';
				iframe.style.border = 'none';
				iframe.style.background = '#fff';
				
				$preview.append(iframe);
				
				const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
				iframeDoc.open();
				iframeDoc.write('<!DOCTYPE html><html><head><style>body{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;margin:0;padding:20px;display:flex;flex-direction:column;align-items:center;}</style></head><body>' + response.data.html + '</body></html>');
				iframeDoc.close();
				
				setTimeout(() => {
					try {
						const height = iframeDoc.body.scrollHeight;
						iframe.style.height = Math.min(Math.max(height + 30, 100), 400) + 'px';
					} catch(e) {}
				}, 200);
				
				$btn.html('<span class="dashicons dashicons-hidden" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span> Hide Preview');
			} else {
				$preview.html('<div style="color: #dc2626; padding: 20px;">Error rendering preview: ' + (response.data || 'Unknown error') + '</div>');
				$btn.html('<span class="dashicons dashicons-visibility" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span> Preview');
			}
		}).fail(() => {
			$btn.removeClass('loading').html('<span class="dashicons dashicons-visibility" style="font-size: 16px; width: 16px; height: 16px; margin-right: 4px;"></span> Preview');
			$preview.html('<div style="color: #dc2626; padding: 20px;">Failed to load preview. Please try again.</div>');
		});
	},
	
	/**
	 * Save custom code section
	 */
	saveCustomCode: function(sectionId) {
		const section = this.currentPage.sections.find(s => s.id === sectionId);
		if (!section) return;
		
		let code = $('#buildfuly-custom-code-editor').val();
		
		// First, unescape any HTML entities (in case user pasted escaped code)
		code = this.unescapeHtml(code);
		
		// Encode script tags for WordPress compatibility
		code = this.encodeScriptTags(code);
		
		// Update section
		section.html = code;
		
		// Update the preview - show placeholder if empty
		const $content = $(`#${sectionId} .buildfuly-section-content`);
		
		if (!code.trim()) {
			// Show placeholder for empty section
			$content.html(`<div class="buildfuly-custom-code-placeholder" style="padding:60px 40px;background:linear-gradient(135deg,#f8fafc 0%,#e2e8f0 100%);border:2px dashed #94a3b8;border-radius:12px;text-align:center;">
				<span class="dashicons dashicons-editor-code" style="font-size:48px;width:48px;height:48px;color:#64748b;margin-bottom:16px;display:block;"></span>
				<p style="color:#64748b;font-size:16px;margin:0;">Click <strong>Edit Code</strong> to add your custom HTML, shortcode, or form</p>
			</div>`);
		} else {
			$content.html(code);
		}
		
		// Initialize scripts in preview - run after a small delay to ensure DOM update
		const self = this;
		setTimeout(function() {
			self.initEncodedScripts();
		}, 100);
		
		// Save and close
		this.savePage();
		this.closeEditPanel();
		this.showMessage('Custom code saved!', 'success');
	},
	
	/**
	 * Unescape HTML entities - delegates to Utils
	 */
	unescapeHtml: function(text) {
		return Utils.unescapeHtml(text);
	},
	
	/**
	 * Escape HTML entities - delegates to Utils
	 */
	escapeHtml: function(text) {
		return Utils.escapeHtml(text);
	},
	
	closeEditPanel: function() {
		const $panel = $('#buildfuly-edit-panel');
		$panel.removeClass('active');
		setTimeout(() => $panel.remove(), 300);
	},

	saveEditedContent: function(sectionId, fields) {
		if (!this.currentPage || !this.currentPage.sections) return;

		const section = this.currentPage.sections.find(s => s.id === sectionId);
		if (!section) return;

		// Get updated values from inputs
		const updates = {};
		$('#buildfuly-edit-panel .buildfuly-edit-input').each(function() {
			const index = parseInt($(this).data('index'));
			const newValue = $(this).val();
			updates[index] = newValue;
		});

		console.log('Saving updates:', updates);

		// Update AI content JSON if available
		if (section.ai_content && typeof section.ai_content === 'object') {
			console.log('Updating AI content JSON');
			fields.forEach((field, index) => {
				if (typeof updates[index] !== 'undefined' && field.path) {
					this.setValueByPath(section.ai_content, field.path, updates[index]);
				}
			});
			
			// Save button settings if present
			if ($('#buildfuly-use-custom-buttons').length) {
				const useCustomButtons = $('#buildfuly-use-custom-buttons').is(':checked');
				section.ai_content.use_custom_buttons = useCustomButtons;
				
				if (useCustomButtons) {
					section.ai_content.custom_buttons = [
						{
							text: $('#buildfuly-button1-text').val(),
							url: $('#buildfuly-button1-url').val()
						},
						{
							text: $('#buildfuly-button2-text').val(),
							url: $('#buildfuly-button2-url').val()
						}
					];
				} else {
					// Clear custom buttons when linked to global
					delete section.ai_content.custom_buttons;
				}
			}
		}
		
		// Always update HTML directly (no API call needed)
		console.log('Updating HTML directly');
		
		// Preserve <style> tags before jQuery manipulation (jQuery corrupts them)
		const styleTags = [];
		let workingHtml = section.html.replace(/<style[^>]*>[\s\S]*?<\/style>/gi, (match) => {
			const placeholder = `<!--STYLE_PLACEHOLDER_${styleTags.length}-->`;
			styleTags.push(match);
			return placeholder;
		});
		
		const $content = $('<div>').html(workingHtml);
		const seenTexts = new Set();
		
		// Build a map of original text to new text
		const textReplacements = {};
		fields.forEach((field, index) => {
			if (typeof updates[index] !== 'undefined' && updates[index] !== field.value) {
				// Normalize text for comparison (trim, collapse whitespace)
				const normalizedOriginal = field.value.replace(/\s+/g, ' ').trim();
				textReplacements[normalizedOriginal] = updates[index];
			}
		});
		
		console.log('Text replacements:', textReplacements);
		
		// Save styling options from Styling tab
		if (!section.style) section.style = {};
		section.style.backgroundColor = $('#buildfuly-edit-bg-color').val();
		section.style.textColor = $('#buildfuly-edit-text-color').val();
		section.style.padding = $('#buildfuly-edit-padding').val();
		section.style.borderRadius = $('#buildfuly-edit-radius').val();
		section.style.shadow = $('#buildfuly-edit-shadow').val();
		section.style.marginTop = $('#buildfuly-edit-margin-top').val();
		section.style.marginBottom = $('#buildfuly-edit-margin-bottom').val();
		
		// Save advanced options
		section.customClass = $('#buildfuly-edit-custom-class').val();
		section.animation = $('#buildfuly-edit-animation').val();
		section.fullWidth = $('#buildfuly-edit-full-width').is(':checked');
		
		// Apply styles to section HTML
		$content.find('section, .buildfuly-section, > div').first().css({
			'background-color': section.style.backgroundColor,
			'color': section.style.textColor,
			'padding-top': section.style.padding + 'px',
			'padding-bottom': section.style.padding + 'px',
			'border-radius': section.style.borderRadius + 'px',
			'margin-top': section.style.marginTop + 'px',
			'margin-bottom': section.style.marginBottom + 'px'
		});
		
		// Add shadow class
		if (section.style.shadow && section.style.shadow !== 'none') {
			$content.find('section, .buildfuly-section, > div').first().addClass(`shadow-${section.style.shadow}`);
		}
		
		// Add custom class
		if (section.customClass) {
			$content.find('section, .buildfuly-section, > div').first().addClass(section.customClass);
		}
		
		// Add animation attribute
		if (section.animation && section.animation !== 'none') {
			$content.find('section, .buildfuly-section, > div').first().attr('data-animation', section.animation);
		}
		
		// Find and replace text in matching elements
		$content.find('h1, h2, h3, h4, h5, h6, p, span, li, a').each(function() {
			const $el = $(this);
			
			// Skip if element has block-level children
			const hasBlockChildren = $el.children().filter(function() {
				const tag = this.tagName ? this.tagName.toLowerCase() : '';
				return !['span', 'strong', 'em', 'i', 'b', 'a'].includes(tag);
			}).length > 0;
			
			if (hasBlockChildren) return;
			
			const originalText = $el.text().replace(/\s+/g, ' ').trim();
			
			if (textReplacements.hasOwnProperty(originalText) && !seenTexts.has(originalText)) {
				console.log('Replacing:', originalText, '->', textReplacements[originalText]);
				$el.text(textReplacements[originalText]);
				seenTexts.add(originalText);
			}
		});
		
		// Update button text and URLs if custom buttons are enabled
		if ($('#buildfuly-use-custom-buttons').length && $('#buildfuly-use-custom-buttons').is(':checked')) {
			const $buttons = $content.find('.buildfuly-btn-primary, .buildfuly-btn-secondary');
			if ($buttons.length >= 1) {
				$buttons.eq(0).text($('#buildfuly-button1-text').val()).attr('href', $('#buildfuly-button1-url').val());
			}
			if ($buttons.length >= 2 && $('#buildfuly-button2-text').val()) {
				$buttons.eq(1).text($('#buildfuly-button2-text').val()).attr('href', $('#buildfuly-button2-url').val());
			}
		}
		
		// Get the modified HTML and restore <style> tags
		let resultHtml = $content.html();
		styleTags.forEach((styleTag, index) => {
			resultHtml = resultHtml.replace(`<!--STYLE_PLACEHOLDER_${index}-->`, styleTag);
		});
		
		section.html = this.encodeScriptTags(resultHtml);
		this.updateSectionDisplay(sectionId, section.html);
		
		console.log('Updated HTML:', section.html);
		
		this.savePage();
		this.closeEditPanel();
		this.showMessage('Content updated successfully!', 'success');
	},
	
	setValueByPath: function(obj, path, value) {
		// Set nested value using path like "features[0].title"
		const parts = path.split(/[\.\[\]]/).filter(p => p);
		let current = obj;
		
		for (let i = 0; i < parts.length - 1; i++) {
			if (!(parts[i] in current)) {
				current[parts[i]] = {};
			}
			current = current[parts[i]];
		}
		
		current[parts[parts.length - 1]] = value;
	},
	
	regenerateSectionHTML: function(sectionId, callback) {
		const section = this.currentPage.sections.find(s => s.id === sectionId);
		if (!section) return;
		
		// Get current image ID from edit panel if open
		const imageId = $('#buildfuly-edit-image-id').val();
		if (imageId) {
			section.image_id = parseInt(imageId);
		}
		
		// Call backend to regenerate HTML from AI content
		$.ajax({
			url: ajaxurl,
			method: 'POST',
			data: {
				action: 'buildfuly_regenerate_section_html',
				nonce: buildfulyAdmin.ajaxNonce,
				section_type: section.type,
				ai_content: section.ai_content,
				keywords: section.keywords,
				image_id: section.image_id || 0
			},
			success: (response) => {
				if (response.success && response.data.html) {
					section.html = response.data.html;
					this.updateSectionDisplay(sectionId, section.html);
					if (callback) callback();
				} else {
					this.showMessage('Failed to update content', 'error');
				}
			},
			error: () => {
				this.showMessage('Failed to update content', 'error');
			}
		});
	},
	
	uploadEditImage: function(sectionId) {
		// Open WordPress Media Library
		const mediaUploader = wp.media({
			title: 'Select Component Image',
			button: {
				text: 'Use this image'
			},
			multiple: false
		});

		mediaUploader.on('select', () => {
			const attachment = mediaUploader.state().get('selection').first().toJSON();
			this.applyImageToSection(sectionId, attachment.url, attachment.id);
			this.showMessage('Image updated successfully!', 'success');
		});

		mediaUploader.open();
	},
	
	/**
	 * Batch generate AI images for all sections that support images
	 */
	batchGenerateAIImages: async function() {
		if (!this.currentPage || !this.currentPage.sections) {
			this.showMessage('No page loaded', 'error');
			return;
		}
		
		// Find all sections that support images (dynamic: any hero-, feature-, content-image, content-split)
		const sectionsWithImages = this.currentPage.sections.filter(section => 
			section.type && (
				section.type.startsWith('hero-') || 
				section.type.startsWith('feature-') ||
				section.type === 'content-image' || 
				section.type === 'content-split'
			)
		);
		
		if (sectionsWithImages.length === 0) {
			this.showMessage('No sections support images. Add hero or content-image sections first.', 'warning');
			return;
		}
		
		// Confirm with user
		const confirmed = confirm(`Generate AI images for ${sectionsWithImages.length} section(s)?\n\nThis will:\n1. Search for relevant reference images\n2. Analyze them with AI\n3. Create unique modified versions\n\nThis may take 1-2 minutes per image.`);
		if (!confirmed) return;
		
		// Show loading overlay
		this.showLoading(`Generating AI images for ${sectionsWithImages.length} sections...`);
		
		let successCount = 0;
		let errorCount = 0;
		
		// Get page context for better prompts
		const pageTitle = this.currentPage.title || '';
		const businessName = buildfulyAdmin.businessName || '';
		const pageKeywords = this.currentPage.keywords || [];
		const keywordText = pageKeywords.map(k => typeof k === 'string' ? k : (k.keyword || '')).filter(k => k).join(', ');
		
		// Process each section sequentially to avoid overwhelming the API
		for (let i = 0; i < sectionsWithImages.length; i++) {
			const section = sectionsWithImages[i];
			
			// Update loading message
			$('#buildfuly-loading p').text(`Generating image ${i + 1} of ${sectionsWithImages.length}...`);
			
			// Build a smart prompt based on section type and content
			const prompt = this.buildImagePromptForSection(section, pageTitle, businessName, keywordText);
			
			console.log(`[Batch AI] Section ${i + 1}: "${prompt.substring(0, 100)}..."`);
			
			try {
				// Generate image via API (will use Brave search + Gemini modification)
				const response = await fetch(buildfulyAdmin.apiUrl + '/api/images/generate', {
					method: 'POST',
					headers: {
						'Content-Type': 'application/json',
						'Authorization': 'Bearer ' + buildfulyAdmin.licenseKey
					},
					body: JSON.stringify({
						prompt: prompt,
						businessName: businessName || null,
						logoUrl: buildfulyAdmin.logoUrl || null,
						colors: {
							primary: buildfulyAdmin.primaryColor || '#667eea',
							secondary: buildfulyAdmin.secondaryColor || '#764ba2'
						}
					})
				});
				
				const data = await response.json();
				
				if (data.success && data.imageUrl) {
					console.log(`[Batch AI] Success! Source: ${data.sourceUrl ? 'Brave search' : 'AI generated'}`);
					
					// Save to WordPress media library
					try {
						const saveResponse = await $.ajax({
							url: buildfulyAdmin.ajaxUrl,
							type: 'POST',
							data: {
								action: 'buildfuly_upload_image_from_url',
								nonce: $('#buildfuly-nonce').val(), // Use the correct nonce
								image_url: data.imageUrl,
								title: prompt.substring(0, 100),
								alt_text: data.description || prompt.substring(0, 100),
								is_ai_generated: 'true'
							}
						});
						
						// Only apply image if upload succeeded - don't use base64 fallback
						if (saveResponse.success && saveResponse.data?.url) {
							const imageUrl = saveResponse.data.url;
							const imageId = saveResponse.data.attachment_id || 0;
							
							// Apply WordPress URL to section (small file size)
							this.applyImageToSection(section.id, imageUrl, imageId);
							successCount++;
						} else {
							console.error('Failed to save image to Media Library:', saveResponse);
							errorCount++;
						}
					} catch (uploadError) {
						console.error('Error uploading image to Media Library:', uploadError);
						errorCount++;
					}
				} else {
					console.error('Image generation failed for section:', section.id, data.error);
					errorCount++;
				}
			} catch (error) {
				console.error('Error generating image for section:', section.id, error);
				errorCount++;
			}
			
			// Small delay between requests
			await new Promise(resolve => setTimeout(resolve, 1500));
		}
		
		// Hide loading
		this.hideLoading();
		
		// Show summary
		if (errorCount === 0) {
			this.showMessage(`Successfully generated ${successCount} AI images!`, 'success');
		} else {
			this.showMessage(`Generated ${successCount} images, ${errorCount} failed. Check console for details.`, 'warning');
		}
		
		// Save page
		this.savePage();
	},
	
	/**
	 * Build a smart prompt for image generation based on section type and content
	 * @param {Object} section - The section object
	 * @param {string} pageTitle - The page title
	 * @param {string} businessName - The business name
	 * @param {string} keywordText - Keywords for the page
	 * @returns {string} A descriptive prompt for image generation
	 */
	buildImagePromptForSection: function(section, pageTitle, businessName, keywordText) {
		const sectionType = section.type || '';
		let prompt = '';
		
		// Try to get content-based context first
		let contentContext = '';
		if (section.ai_content) {
			// Extract meaningful text from AI content
			const headline = section.ai_content.headline || section.ai_content.title || '';
			const subheadline = section.ai_content.subheadline || section.ai_content.subtitle || '';
			const description = section.ai_content.description || section.ai_content.content || '';
			
			if (headline) contentContext += headline + ' ';
			if (subheadline) contentContext += subheadline + ' ';
			if (description && description.length < 200) contentContext += description;
			
			contentContext = contentContext.trim();
		}
		
		// Build prompt based on section type with intelligent defaults
		if (sectionType.includes('hero')) {
			// Hero sections - company/brand imagery
			if (contentContext) {
				prompt = `Professional business hero image for "${contentContext.substring(0, 100)}"`;
			} else if (pageTitle) {
				prompt = `Professional hero image for ${businessName || 'a business'} - ${pageTitle}`;
			} else {
				prompt = `Professional modern business hero image for ${businessName || 'a company'}, team or office setting`;
			}
			prompt += ', high quality, professional photography style, vibrant colors';
			
		} else if (sectionType.includes('feature') || sectionType.includes('content')) {
			// Feature/Content sections - product/service imagery
			if (contentContext) {
				prompt = `Professional image showcasing "${contentContext.substring(0, 100)}"`;
			} else {
				prompt = `Professional product or service showcase image for ${businessName || 'a business'}`;
			}
			prompt += ', clean modern design, professional photography, well-lit';
			
		} else if (sectionType.includes('testimonial')) {
			// Testimonial sections - happy customers/people
			prompt = `Happy satisfied customer or business professional portrait, diverse person, friendly smile, professional setting`;
			prompt += ', natural lighting, genuine expression, business casual attire';
			
		} else if (sectionType.includes('team') || sectionType.includes('about')) {
			// Team/About sections - team/people imagery
			if (contentContext) {
				prompt = `Professional team or person image for "${contentContext.substring(0, 80)}"`;
			} else {
				prompt = `Professional business team or executive portrait for ${businessName || 'a company'}`;
			}
			prompt += ', corporate photography, friendly professional appearance, modern office';
			
		} else if (sectionType.includes('gallery') || sectionType.includes('portfolio')) {
			// Gallery/Portfolio - showcase work
			if (keywordText) {
				prompt = `High quality showcase image for ${keywordText}`;
			} else {
				prompt = `Professional portfolio showcase image for ${businessName || 'creative work'}`;
			}
			prompt += ', artistic composition, high resolution, visually striking';
			
		} else if (sectionType.includes('contact') || sectionType.includes('cta')) {
			// Contact/CTA - inviting professional imagery
			prompt = `Welcoming professional business image, modern office or meeting space`;
			prompt += ', inviting atmosphere, professional setting, warm lighting';
			
		} else if (sectionType.includes('service')) {
			// Services - what they offer
			if (contentContext) {
				prompt = `Professional service illustration for "${contentContext.substring(0, 80)}"`;
			} else {
				prompt = `Professional business service image for ${businessName || 'a company'}`;
			}
			prompt += ', clean professional style, modern business photography';
			
		} else {
			// Default fallback
			if (contentContext) {
				prompt = `Professional business image for "${contentContext.substring(0, 100)}"`;
			} else if (keywordText) {
				prompt = `Professional image related to ${keywordText}`;
			} else {
				prompt = `Professional business image for ${businessName || pageTitle || 'modern company'}`;
			}
			prompt += ', high quality, professional photography, clean modern style';
		}
		
		// Add business context if available
		if (businessName && !prompt.includes(businessName)) {
			prompt += `, for ${businessName}`;
		}
		
		return prompt;
	},
	
	/**
	 * Generate an AI image for the section using the reusable image generator module
	 */
	generateAIImage: function(sectionId) {
		const section = this.currentPage.sections.find(s => s.id === sectionId);
		if (!section) return;
		
		// Get context for the image topic suggestion
		let suggestedTopic = '';
		if (section.ai_content) {
			if (section.ai_content.heading) {
				suggestedTopic = section.ai_content.heading;
			} else if (section.ai_content.title) {
				suggestedTopic = section.ai_content.title;
			}
		}
		if (!suggestedTopic && this.currentPage) {
			suggestedTopic = this.currentPage.title || '';
		}
		
		// Use the reusable AI image generator module
		if (typeof BuildfulyImageGenerator !== 'undefined') {
			const self = this;
			
			BuildfulyImageGenerator.open({
				topic: suggestedTopic,
				onSelect: function(imageData) {
					if (!imageData || !imageData.url) return;
					
					// Use the same logic as uploadEditImage - just update the image src
					self.applyImageToSection(sectionId, imageData.url, imageData.attachmentId || 0);
					self.showMessage('AI image applied successfully!', 'success');
				}
			});
		} else {
			this.showMessage('AI Image Generator not loaded. Please refresh the page.', 'error');
		}
	},
	
	/**
	 * Apply an image URL to a section (used by both Choose Image and AI Generate)
	 * Uses regex-based replacement to preserve <style> tags that jQuery corrupts
	 */
	applyImageToSection: function(sectionId, imageUrl, imageId) {
		const section = this.currentPage.sections.find(s => s.id === sectionId);
		if (!section) return;
		
		// Update preview in edit panel
		$('#buildfuly-edit-image-id').val(imageId);
		$('#buildfuly-edit-image-preview img').attr('src', imageUrl);
		$('#buildfuly-edit-image-preview').show();
		
		let html = section.html;
		
		// Define all container patterns
		const editableImageRegex = /(<[^>]+data-editable=["']image["'][^>]*>)([\s\S]*?)(<\/[^>]+>)/i;
		const editableBgImageRegex = /(<[^>]+data-editable=["']background-image["'][^>]*>)([\s\S]*?)(<\/[^>]+>)/i;
		const heroImageRegex = /(<[^>]+class=["'][^"']*buildfuly-hero-image[^"']*["'][^>]*>)([\s\S]*?)(<\/[^>]+>)/i;
		const contentImageRegex = /(<[^>]+class=["'][^"']*content-image[^"']*["'][^>]*>)([\s\S]*?)(<\/[^>]+>)/i;
		
		const newImg = `<img src="${imageUrl}" alt="Component image" style="max-width: 100%; height: auto; border-radius: 12px; box-shadow: 0 20px 60px rgba(0,0,0,0.15);">`;
		
		// Check for background-image container FIRST (special handling)
		if (editableBgImageRegex.test(html)) {
			// For background-image containers, make them visible, add the image with an overlay
			html = html.replace(editableBgImageRegex, (match, openTag, content, closeTag) => {
				// Remove display:none and set display:block
				let updatedOpenTag = openTag.replace(/display:\s*[^;]+;?/gi, '');
				if (updatedOpenTag.includes('style="')) {
					updatedOpenTag = updatedOpenTag.replace(/style="([^"]*)"/, 'style="$1 display: block;"');
				} else {
					updatedOpenTag = updatedOpenTag.replace(/>$/, ' style="display: block;">');
				}
				const bgImg = `<img src="${imageUrl}" alt="Background image" style="width: 100%; height: 100%; object-fit: cover; position: absolute; top: 0; left: 0;">`;
				const overlay = `<div class="buildfuly-bg-overlay" style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; background: var(--buildfuly-color-background, rgba(255,255,255,0.85)); opacity: 0.85;"></div>`;
				return `${updatedOpenTag}${bgImg}${overlay}${closeTag}`;
			});
		} else {
			// Check if there's an existing <img> tag - use regex to preserve <style> tags
			const imgRegex = /<img\s+([^>]*?)src=["']([^"']+)["']([^>]*)>/i;
			
			if (imgRegex.test(html)) {
				// Replace the src attribute of the first img tag
				html = html.replace(imgRegex, (match, before, oldSrc, after) => {
					return `<img ${before}src="${imageUrl}"${after}>`;
				});
			} else if (editableImageRegex.test(html)) {
				html = html.replace(editableImageRegex, (match, openTag, content, closeTag) => {
					let updatedOpenTag = openTag;
					if (openTag.includes('style="')) {
						updatedOpenTag = openTag.replace(/style="([^"]*)"/, 'style="$1; flex: 1;"');
					} else {
						updatedOpenTag = openTag.replace(/>$/, ' style="flex: 1;">');
					}
					return `${updatedOpenTag}${newImg}${closeTag}`;
				});
			} else if (heroImageRegex.test(html)) {
				html = html.replace(heroImageRegex, (match, openTag, content, closeTag) => {
					let updatedOpenTag = openTag;
					if (openTag.includes('style="')) {
						updatedOpenTag = openTag.replace(/style="([^"]*)"/, 'style="$1; flex: 1;"');
					} else {
						updatedOpenTag = openTag.replace(/>$/, ' style="flex: 1;">');
					}
					return `${updatedOpenTag}${newImg}${closeTag}`;
				});
			} else if (contentImageRegex.test(html)) {
				html = html.replace(contentImageRegex, (match, openTag, content, closeTag) => {
					let updatedOpenTag = openTag;
					if (openTag.includes('style="')) {
						updatedOpenTag = openTag.replace(/style="([^"]*)"/, 'style="$1; flex: 1;"');
					} else {
						updatedOpenTag = openTag.replace(/>$/, ' style="flex: 1;">');
					}
					return `${updatedOpenTag}${newImg}${closeTag}`;
				});
			}
		}
		
		section.html = html;
		section.image_id = imageId;
		
		// Update AI content
		if (section.ai_content && typeof section.ai_content === 'object') {
			section.ai_content.image_url = imageUrl;
			section.ai_content.image_id = imageId;
		}
		
		// Update displayed section (extracts styles to head)
		this.updateSectionDisplay(sectionId, section.html);
		
		// Save
		this.savePage();
	},
	
	removeEditImage: function(sectionId) {
		const section = this.currentPage.sections.find(s => s.id === sectionId);
		if (!section) return;
		
		// Update preview in edit panel
		$('#buildfuly-edit-image-id').val(0);
		$('#buildfuly-edit-image-preview').hide();
		$('#buildfuly-edit-image-preview img').attr('src', '');
		
		let html = section.html;
		
		// Remove img tags using regex to preserve <style> tags
		// Handle different container types
		const editableImageRegex = /(<[^>]+data-editable=["']image["'][^>]*>)([\s\S]*?)(<\/[^>]+>)/gi;
		const editableBgImageRegex = /(<[^>]+data-editable=["']background-image["'][^>]*>)([\s\S]*?)(<\/[^>]+>)/gi;
		const heroImageRegex = /(<[^>]+class=["'][^"']*buildfuly-hero-image[^"']*["'][^>]*>)([\s\S]*?)(<\/[^>]+>)/gi;
		
		// Replace content of image containers with empty, and set flex: 0
		html = html.replace(editableImageRegex, (match, openTag, content, closeTag) => {
			let updatedOpenTag = openTag;
			if (openTag.includes('style="')) {
				updatedOpenTag = openTag.replace(/style="([^"]*)"/, (m, styles) => {
					// Remove any existing flex value and add flex: 0
					let newStyles = styles.replace(/flex:\s*[^;]+;?/g, '').trim();
					newStyles = newStyles ? newStyles + '; flex: 0;' : 'flex: 0;';
					return `style="${newStyles}"`;
				});
			} else {
				updatedOpenTag = openTag.replace(/>$/, ' style="flex: 0;">');
			}
			return `${updatedOpenTag}${closeTag}`;
		});
		
		// Handle background-image containers - hide them and clear content
		html = html.replace(editableBgImageRegex, (match, openTag, content, closeTag) => {
			let updatedOpenTag = openTag;
			// Set display: none to hide the background container
			if (openTag.includes('style="')) {
				updatedOpenTag = openTag.replace(/style="([^"]*)"/, (m, styles) => {
					// Remove display: block and add display: none
					let newStyles = styles.replace(/display:\s*[^;]+;?/gi, '').trim();
					newStyles = newStyles ? newStyles + '; display: none;' : 'display: none;';
					return `style="${newStyles}"`;
				});
			} else {
				updatedOpenTag = openTag.replace(/>$/, ' style="display: none;">');
			}
			// Return empty container (removes img and overlay)
			return `${updatedOpenTag}${closeTag}`;
		});
		
		html = html.replace(heroImageRegex, (match, openTag, content, closeTag) => {
			let updatedOpenTag = openTag;
			if (openTag.includes('style="')) {
				updatedOpenTag = openTag.replace(/style="([^"]*)"/, (m, styles) => {
					let newStyles = styles.replace(/flex:\s*[^;]+;?/g, '').trim();
					newStyles = newStyles ? newStyles + '; flex: 0;' : 'flex: 0;';
					return `style="${newStyles}"`;
				});
			} else {
				updatedOpenTag = openTag.replace(/>$/, ' style="flex: 0;">');
			}
			return `${updatedOpenTag}${closeTag}`;
		});
		
		section.html = html;
		section.image_id = 0;
		
		// Update AI content if available
		if (section.ai_content && typeof section.ai_content === 'object') {
			section.ai_content.image_url = '';
			section.ai_content.image_id = 0;
		}
		
		// Update the displayed section (extracts styles to head)
		this.updateSectionDisplay(sectionId, section.html);
		
		// Save the page
		this.savePage();
		this.showMessage('Image removed successfully!', 'success');
	},

	/**
	 * Generate AI images for all sections in a page during creation
	 * @param {string} pageDescription - Description of the page for prompt generation
	 * @param {Array} sections - Array of section data objects
	 * @param {Object} options - Generation options { autoPick, includeLogo }
	 */
	generateImagesForPage: async function(pageDescription, sections, options) {
		const self = this;
		
		console.log('[generateImagesForPage] Starting:', { pageDescription, sectionsCount: sections.length, options });
		
		// Determine which sections can have images
		const visualSectionTypes = ['hero', 'features', 'benefits', 'testimonial', 'about', 
			'feature-showcase', 'split-content', 'image-grid', 'feature-cards', 'bento-grid', 'team'];
		
		const sectionsWithImages = sections.filter(s => 
			visualSectionTypes.some(type => s.type?.includes(type) || s.id?.includes(type))
		);
		
		console.log('[generateImagesForPage] Visual sections found:', sectionsWithImages.length);
		console.log('[generateImagesForPage] Visual sections:', sectionsWithImages.map(s => ({ id: s.id, type: s.type })));
		
		if (sectionsWithImages.length === 0) {
			console.log('No visual sections found for image generation');
			return;
		}
		
		console.log(`Generating images for ${sectionsWithImages.length} sections`);
		
		// Get brand settings
		const businessName = window.buildfulyAdmin?.businessName || '';
		const logoUrl = window.buildfulyAdmin?.logoUrl || '';
		const primaryColor = window.buildfulyAdmin?.primaryColor || '#667eea';
		const secondaryColor = window.buildfulyAdmin?.secondaryColor || '#764ba2';
		
		try {
			// Step 1: Generate image search keywords for each section (simple, descriptive keywords)
			this.showLoading('Generating image search keywords...');
			
			const sectionTypes = sectionsWithImages.map(s => s.type || 'content');
			
			const promptsResponse = await fetch(buildfulyAdmin.apiUrl + '/api/images/generate-for-sections', {
				method: 'POST',
				headers: {
					'Content-Type': 'application/json',
					'Authorization': 'Bearer ' + buildfulyAdmin.licenseKey
				},
				body: JSON.stringify({
					pageDescription,
					sections: sectionTypes,
					businessName
				})
			});
			
			const promptsData = await promptsResponse.json();
			
			if (!promptsData.success || !promptsData.prompts || promptsData.prompts.length === 0) {
				console.error('Failed to generate image prompts:', promptsData);
				return;
			}
			
			console.log('Generated prompts:', promptsData.prompts);
			
			// Initialize images_generated array for the page
			if (!this.currentPage.images_generated) {
				this.currentPage.images_generated = [];
			}
			
			if (options.autoPick) {
				// Auto mode: generate all images in batch
				this.showLoading(`Generating ${promptsData.prompts.length} AI images...`);
				
				const batchResponse = await fetch(buildfulyAdmin.apiUrl + '/api/images/generate-batch', {
					method: 'POST',
					headers: {
						'Content-Type': 'application/json',
						'Authorization': 'Bearer ' + buildfulyAdmin.licenseKey
					},
					body: JSON.stringify({
						prompts: promptsData.prompts,
						businessName,
						colors: { primary: primaryColor, secondary: secondaryColor },
						logoUrl,
						includeLogo: options.includeLogo,
						autoPick: true
					})
				});
				
				const batchData = await batchResponse.json();
				
				if (batchData.success && batchData.images) {
					// Apply generated images to sections
					for (const imgResult of batchData.images) {
						if (imgResult.success && imgResult.imageUrl) {
							// Find the matching section
							const section = sectionsWithImages.find(s => 
								s.type === imgResult.sectionType || s.type?.includes(imgResult.sectionType)
							);
							
							if (section) {
								// Save to WordPress media library
								try {
									const saveResult = await $.ajax({
										url: buildfulyAdmin.ajaxUrl,
										type: 'POST',
										data: {
											action: 'buildfuly_upload_image_from_url',
											nonce: $('#buildfuly-nonce').val(), // Use the correct nonce
											image_url: imgResult.imageUrl,
											title: imgResult.searchQuery || pageDescription,
											alt_text: imgResult.description || imgResult.searchQuery,
											is_ai_generated: 'true'
										}
									});
									
									if (saveResult.success && saveResult.data) {
										// Apply image to section
										this.applyImageToSection(section.id, saveResult.data.url, saveResult.data.attachment_id);
										
										// Track in images_generated
										this.currentPage.images_generated.push({
											sectionId: section.id,
											sectionType: imgResult.sectionType,
											attachmentId: saveResult.data.attachment_id,
											url: saveResult.data.url,
											searchQuery: imgResult.searchQuery,
											description: imgResult.description,
											generatedAt: new Date().toISOString()
										});
									}
								} catch (saveError) {
									console.error('Failed to save image to media library:', saveError);
								}
							}
						}
					}
					
					this.showMessage(`Generated ${batchData.stats?.generated || 0} AI images!`, 'success');
				}
			} else {
				// Manual mode: open picker for each image
				// Store prompts for later use in picker
				this.pendingImagePrompts = promptsData.prompts;
				this.pendingSectionsForImages = sectionsWithImages;
				this.imageGenerationOptions = options;
				
				// Show first picker after page load
				this.showMessage('AI has prepared image suggestions. Click "Add AI Image" on any section to generate.', 'info');
			}
			
		} catch (error) {
			console.error('generateImagesForPage error:', error);
			this.showMessage('Failed to generate images: ' + error.message, 'error');
		}
	},

	uploadSectionImage: function(e) {
		e.preventDefault();
		
		// Open WordPress Media Library
		const mediaUploader = wp.media({
			title: 'Select Section Image',
			button: { text: 'Select Image' },
			multiple: false,
			library: { type: 'image' }
		});
		
		mediaUploader.on('select', function() {
			const attachment = mediaUploader.state().get('selection').first().toJSON();
			
			// Store image ID in hidden field
			$('#buildfuly-section-image-id').val(attachment.id);
			
			// Show green indicator
			$('#buildfuly-section-image-indicator').show();
		});
		
		mediaUploader.open();
	},

	/**
	 * Check if user can generate content (client-side pre-check)
	 * Returns true if OK to proceed, false if limit reached
	 */
	canGenerate: function() {
		// Check if usage limit reached (client-side pre-check to avoid wasting API calls)
		if (this.usageStats && this.usageStats.remaining <= 0) {
			const tier = this.usageStats.tier || 'starter';
			let message;
			if (tier === 'starter') {
				message = 'You have reached your limit of 3 total pages. Please upgrade to generate more pages!';
			} else {
				message = 'Daily usage limit reached. Please upgrade your plan or wait until tomorrow.';
			}
			this.showMessage(message, 'error');
			return false;
		}
		return true;
	},

	/**
	 * Wrapper for all generation requests - handles common logic
	 * @param {Object} config - Configuration object
	 * @param {string} config.url - API endpoint
	 * @param {Object} config.data - Request data
	 * @param {string} config.loadingMessage - Loading message to display (null to skip)
	 * @param {string} config.successMessage - Success message to display (null to skip)
	 * @param {boolean} config.skipCommonTasks - Skip word count, stats refresh, etc (for batch operations)
	 * @param {Function} config.onSuccess - Success callback (receives response)
	 * @param {Function} config.onError - Optional error callback
	 */
	generateWithCommonLogic: function(config) {
		// Pre-check usage limits
		if (!this.canGenerate()) {
			return;
		}

		// Show loading (unless explicitly null)
		if (config.loadingMessage !== null) {
			this.showLoading(config.loadingMessage || 'Generating content...');
		}

		// Make API request
		$.ajax({
			url: config.url,
			method: 'POST',
			beforeSend: function(xhr) {
				xhr.setRequestHeader('X-WP-Nonce', buildfulyAdmin.nonce);
			},
			data: JSON.stringify(config.data),
			contentType: 'application/json',
			dataType: 'json', // Expect JSON response
			success: (response) => {
				if (config.loadingMessage !== null) {
					this.hideLoading();
				}
				
				this.debug('GenerateWithCommonLogic SUCCESS:', {
					hasData: !!response.data,
					hasUsage: !!(response.data && response.data.usage),
					responseKeys: response.data ? Object.keys(response.data) : [],
					skipCommonTasks: config.skipCommonTasks,
					successMessage: config.successMessage
				});
				
				if (response.success && response.data) {
					// Update usage stats directly from API response (more efficient than separate call)
					if (response.data.usage) {
						this.debug('Updating usage from API response:', response.data.usage);
						this.updateUsageDisplay(response.data.usage);
					} else {
						this.debug('No usage data in response - will fetch separately');
					}
					
					// Execute success callback
					if (config.onSuccess) {
						this.debug('Calling onSuccess callback');
						config.onSuccess(response);
					}
					
					// Common post-generation tasks (unless explicitly skipped)
					if (!config.skipCommonTasks) {
						this.debug('Running common tasks: updateWordCount, renderKeywordsPanel, etc.');
						this.updateWordCount();
						this.debug('updateWordCount() completed');
						this.renderKeywordsPanel();
						this.renderPagesList(); // Update word count in sidebar
						
						// Only fetch usage stats if not already in response
						if (!response.data.usage) {
							this.debug('No usage in response, fetching separately');
							this.loadUsageStats(); // Refresh usage display
						}
						
						this.initEncodedScripts(); // Initialize any scripts
						this.savePage();
						this.debug('All common tasks completed');
					} else {
						this.debug('Skipping common tasks (skipCommonTasks: true)');
					}
					
					// Show success message (unless explicitly null)
					if (config.successMessage !== null && config.successMessage !== undefined) {
						this.showMessage(config.successMessage, 'success');
					}
				} else {
					this.debug('Generation failed:', response);
					this.showMessage(response.data?.message || 'Failed to generate content', 'error');
				}
			},
			error: (xhr, status, error) => {
				if (config.loadingMessage !== null) {
					this.hideLoading();
				}
				this.loadUsageStats(); // Refresh usage display
				
				console.error('AJAX Error Details:', {
					status: xhr.status,
					statusText: xhr.statusText,
					responseText: xhr.responseText ? xhr.responseText.substring(0, 500) : 'empty',
					error: error
				});
				
				// Check for 429 Rate Limit error
				if (xhr.status === 429) {
					const message = xhr.responseJSON?.message || 
						'Usage limit reached. Please upgrade your plan or wait until tomorrow.';
					this.showMessage(message, 'error');
				} else if (xhr.responseJSON?.skip) {
					// Component failed but can skip
					this.showMessage('Component failed to generate - skipping: ' + 
						(xhr.responseJSON.message || error), 'warning');
				} else {
					// Custom error handler or default
					if (config.onError) {
						config.onError(xhr, status, error);
					} else {
						const errorMsg = xhr.responseJSON?.message || error || 'Unknown error';
						this.showMessage('Error: ' + errorMsg, 'error');
					}
				}
			}
		});
	},

	loadUsageStats: function() {
		// Get license info from WordPress options
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'buildfuly_get_usage_stats',
				nonce: buildfulyAdmin.ajaxNonce
			},
			success: (response) => {
				if (response.success && response.data && response.data.usage) {
					const usage = response.data.usage;
					const remaining = usage.remaining || 0;
					const limit = usage.limit || 5;
					const remainingWords = Math.floor(remaining * 600);
					
					// Store usage stats for pre-budget checks
					this.usageStats = {
						remaining: remaining,
						limit: limit,
						remainingWords: remainingWords
					};
					
					$('#buildfuly-pages-remaining').text(remaining.toFixed(1));
					$('#buildfuly-words-count').text(remainingWords.toLocaleString());
					
					// Update color based on usage
					const $usageDisplay = $('#buildfuly-usage-display').parent();
					if (remaining <= 0) {
						$usageDisplay.css({'background': '#fee2e2', 'border-color': '#fca5a5'});
						$('#buildfuly-usage-display').css('color', '#991b1b');
					} else if (remaining < 1) {
						$usageDisplay.css({'background': '#fef3c7', 'border-color': '#fcd34d'});
						$('#buildfuly-usage-display').css('color', '#92400e');
					} else {
						// Reset to default blue colors
						$usageDisplay.css({'background': '#dbeafe', 'border-color': '#93c5fd'});
						$('#buildfuly-usage-display').css('color', '#1e40af');
					}
				} else if (!response.success && response.data && response.data.usage) {
					// Handle error response but still show the fallback usage data
					const usage = response.data.usage;
					const remaining = usage.remaining || 0;
					const limit = usage.limit || 5;
					const remainingWords = Math.floor(remaining * 600);
					
					$('#buildfuly-pages-remaining').text(remaining.toFixed(1));
					$('#buildfuly-words-count').text(remainingWords.toLocaleString());
					
					// Show error banner
					this.showMessage('⚠️ API Connection Error: ' + (response.data.message || 'Could not reach API server. Using cached data.'), 'error');
					
					// Show red since it's an error
					const $usageDisplay = $('#buildfuly-usage-display').parent();
					$usageDisplay.css({'background': '#fee2e2', 'border-color': '#fca5a5'});
					$('#buildfuly-usage-display').css('color', '#991b1b');
				} else {
					// Complete failure - show error banner
					this.showMessage('❌ API Error: Cannot load usage data. Please check that the API server is running on http://localhost:8000', 'error');
					$('#buildfuly-pages-remaining').text('?');
					$('#buildfuly-words-count').text('?');
				}
			},
			error: (xhr, status, error) => {
				// Network error - API server not reachable
				this.showMessage('❌ API Server Unreachable: Cannot connect to http://localhost:8000. Please start the API server.', 'error');
				$('#buildfuly-pages-remaining').text('?');
				$('#buildfuly-words-count').text('?');
			}
		});
	},

	updateUsageDisplay: function(usage) {
		if (!usage) {
			this.debug('updateUsageDisplay called with no usage data');
			return;
		}
		
		this.debug('updateUsageDisplay called with:', usage);
		
		const remaining = usage.remaining || 0;
		const remainingWords = usage.remaining_words || Math.floor(remaining * 600);
		
		this.debug('Updating display:', {
			remaining: remaining,
			remainingWords: remainingWords,
			formatted: remaining.toFixed(1) + ' pages, ' + remainingWords.toLocaleString() + ' words'
		});
		
		// Update cached stats for pre-checks
		this.usageStats = {
			remaining: remaining,
			limit: usage.limit || this.usageStats?.limit || 5,
			remainingWords: remainingWords
		};
		
		$('#buildfuly-pages-remaining').text(remaining.toFixed(1));
		$('#buildfuly-words-count').text(remainingWords.toLocaleString());
		
		this.debug('DOM updated - pages:', $('#buildfuly-pages-remaining').text(), 'words:', $('#buildfuly-words-count').text());
		
		// Update color based on usage
		const $usageDisplay = $('#buildfuly-usage-display').parent();
		if (remaining <= 0) {
			$usageDisplay.css({'background': '#fee2e2', 'border-color': '#fca5a5'});
			$('#buildfuly-usage-display').css('color', '#991b1b');
			$('#buildfuly-words-remaining').css('color', '#dc2626');
		} else if (remaining < 1) {
			$usageDisplay.css({'background': '#fef3c7', 'border-color': '#fcd34d'});
			$('#buildfuly-usage-display').css('color', '#92400e');
			$('#buildfuly-words-remaining').css('color', '#d97706');
		} else {
			// Reset to default blue colors
			$usageDisplay.css({'background': '#dbeafe', 'border-color': '#93c5fd'});
			$('#buildfuly-usage-display').css('color', '#1e40af');
			$('#buildfuly-words-remaining').css('color', '#3b82f6');
		}
	},

	loadComponentWordEstimates: function() {
		// Fetch component word estimates from API
		$.ajax({
			url: buildfulyAdmin.restUrl + '/components/word-estimates',
			method: 'GET',
			beforeSend: function(xhr) {
				xhr.setRequestHeader('X-WP-Nonce', buildfulyAdmin.nonce);
			},
			success: (response) => {
				if (response.success && response.data) {
					this.componentWordEstimates = response.data;
					console.log('BuildfulyPageBuilder: Loaded component word estimates:', this.componentWordEstimates);
				} else {
					console.warn('BuildfulyPageBuilder: Failed to load component word estimates, using defaults');
					this.componentWordEstimates = this.COMPONENT_WORD_ESTIMATES; // Fallback to hardcoded
				}
			},
			error: () => {
				console.warn('BuildfulyPageBuilder: Error loading component word estimates, using defaults');
				this.componentWordEstimates = this.COMPONENT_WORD_ESTIMATES; // Fallback to hardcoded
			}
		});
	},

	loadComponentsFromAPI: function() {
		console.log('BuildfulyPageBuilder: Loading components from API...');
		
		// Call WP AJAX endpoint that uses the API client
		$.ajax({
			url: buildfulyAdmin.ajaxUrl,
			method: 'POST',
			data: {
				action: 'buildfuly_get_components',
				nonce: buildfulyAdmin.adminNonce
			},
			success: (response) => {
				console.log('BuildfulyPageBuilder: Raw AJAX response:', response);
				console.log('BuildfulyPageBuilder: Response success:', response.success);
				console.log('BuildfulyPageBuilder: Response data:', response.data);
				
				if (response.success && response.data && response.data.components) {
					console.log('BuildfulyPageBuilder: Loaded components from API:', response.data.components);
					this.populateComponentDropdown(response.data.components);
				} else {
					console.error('BuildfulyPageBuilder: Failed to load components from API. Full response:', JSON.stringify(response));
				}
			},
			error: (xhr, status, error) => {
				console.error('BuildfulyPageBuilder: Error loading components from API:', error);
				console.error('BuildfulyPageBuilder: XHR:', xhr);
			}
		});
	},

	populateComponentDropdown: function(components) {
		console.log('BuildfulyPageBuilder: populateComponentDropdown called with:', components);
		console.log('BuildfulyPageBuilder: Components type:', typeof components);
		console.log('BuildfulyPageBuilder: Components keys:', Object.keys(components));
		
		// Store components for use in layout modal
		this.availableComponents = components;
		
		// Cache word counts from components
		this.componentWordCounts = {};
		for (const [type, typeComponents] of Object.entries(components)) {
			for (const [slug, component] of Object.entries(typeComponents)) {
				if (component.estimated_words) {
					this.componentWordCounts[slug] = component.estimated_words;
				}
			}
		}
		console.log('BuildfulyPageBuilder: Cached word counts:', this.componentWordCounts);
		
		const $select = $('#buildfuly-section-type');
		
		// Clear existing options except "Random Layout"
		$select.find('option:not(:first)').remove();
		$select.find('optgroup').remove();
		
		// Organize components by type
		const componentsByType = {
			hero: [],
			content: [],
			cta: [],
			form: []
		};
		
		// Group components by type
		for (const [type, typeComponents] of Object.entries(components)) {
			console.log('BuildfulyPageBuilder: Processing type:', type, 'with components:', typeComponents);
			
			if (componentsByType[type]) {
				for (const [slug, component] of Object.entries(typeComponents)) {
					console.log('BuildfulyPageBuilder: Adding component:', slug, component);
					componentsByType[type].push({
						slug: slug,
						name: component.name || component.slug || slug
					});
				}
			}
		}
		
		console.log('BuildfulyPageBuilder: Grouped components:', componentsByType);
		
		// Add Hero components (excluding CTAs)
		if (componentsByType.hero.length > 0) {
			const $heroGroup = $('<optgroup label="Hero Sections"></optgroup>');
			componentsByType.hero.forEach(component => {
				// Skip CTAs from hero group
				if (!component.slug.startsWith('cta')) {
					$heroGroup.append(`<option value="${component.slug}">${component.name}</option>`);
				}
			});
			$select.append($heroGroup);
		}
		
		// Add Content components
		if (componentsByType.content.length > 0) {
			const $contentGroup = $('<optgroup label="Content Sections"></optgroup>');
			componentsByType.content.forEach(component => {
				$contentGroup.append(`<option value="${component.slug}">${component.name}</option>`);
			});
			$select.append($contentGroup);
		}
		
		// Add CTA components
		if (componentsByType.cta.length > 0) {
			const $ctaGroup = $('<optgroup label="Call to Actions"></optgroup>');
			componentsByType.cta.forEach(component => {
				$ctaGroup.append(`<option value="${component.slug}">${component.name}</option>`);
			});
			$select.append($ctaGroup);
		}
		
		// Add Form components
		if (componentsByType.form.length > 0) {
			const $formGroup = $('<optgroup label="Forms"></optgroup>');
			componentsByType.form.forEach(component => {
				$formGroup.append(`<option value="${component.slug}">${component.name}</option>`);
			});
			$select.append($formGroup);
		}
		
		// Add Custom Code option (in its own group, excluded from random)
		const $customGroup = $('<optgroup label="Advanced"></optgroup>');
		$customGroup.append(`<option value="custom-code" data-no-random="true">✏️ Custom Code</option>`);
		$select.append($customGroup);
		
		console.log('BuildfulyPageBuilder: Component dropdown populated with', $select.find('option').length - 1, 'components');
	}
};
$(document).ready(function() {
	BuildfulyPageBuilder.init();
});

})(jQuery);




