/**
 * Buildfuly Image Generator Module
 * Reusable AI image generation functionality with reference image selection
 * 
 * Flow:
 * 1. User enters topic/prompt
 * 2. Search Brave for 6 reference images
 * 3. User picks one reference image
 * 4. Gemini modifies it to create unique version
 * 
 * Usage:
 *   BuildfulyImageGenerator.open({
 *     topic: 'optional default topic',
 *     onSelect: function(imageData) { 
 *       // imageData.url, imageData.attachmentId, imageData.description 
 *     }
 *   });
 * 
 * @package Buildfuly
 * @since 1.1.0
 */

(function($) {
	'use strict';

	window.BuildfulyImageGenerator = {
		
		modalId: 'buildfuly-ai-image-generator-modal',
		onSelectCallback: null,
		isInitialized: false,
		selectedReferenceUrl: null,
		currentPrompt: '',
		
		/**
		 * Initialize the modal HTML (only once)
		 */
		init: function() {
			if (this.isInitialized) return;
			
			// Get brand settings from buildfulyAdmin if available
			const businessName = window.buildfulyAdmin?.businessName || '';
			const logoUrl = window.buildfulyAdmin?.logoUrl || '';
			const primaryColor = window.buildfulyAdmin?.primaryColor || '#667eea';
			const secondaryColor = window.buildfulyAdmin?.secondaryColor || '#764ba2';
			
			const modalHtml = `
				<div id="${this.modalId}" class="buildfuly-modal" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.6); z-index: 100000; align-items: center; justify-content: center;">
					<div class="buildfuly-modal-content" style="background: white; border-radius: 12px; max-width: 700px; width: 95%; max-height: 90vh; overflow-y: auto; box-shadow: 0 25px 50px -12px rgba(0,0,0,0.25);">
						<div class="buildfuly-modal-header" style="padding: 20px 24px; border-bottom: 1px solid #e5e7eb; display: flex; justify-content: space-between; align-items: center;">
							<h2 style="margin: 0; font-size: 18px; display: flex; align-items: center; gap: 8px;">
								<span class="dashicons dashicons-art" style="color: ${primaryColor};"></span>
								Generate AI Image
							</h2>
							<button type="button" class="buildfuly-ai-image-close" style="background: none; border: none; font-size: 24px; cursor: pointer; color: #6b7280; line-height: 1;">&times;</button>
						</div>
						<div class="buildfuly-modal-body" style="padding: 24px;">
							
							<!-- Step 1: Topic Input -->
							<div id="ai-image-step-1">
								<div class="buildfuly-form-group">
									<label for="ai-image-topic" style="font-weight: 600; margin-bottom: 8px; display: block;">
										What should the image show?
									</label>
									<input type="text" id="ai-image-topic" class="buildfuly-control-input" 
										   placeholder="e.g., Professional team meeting, Happy customers, Modern office space..." 
										   style="width: 100%; padding: 12px; border: 1px solid #e5e7eb; border-radius: 8px; font-size: 14px;">
									<p style="margin: 8px 0 0 0; font-size: 13px; color: #6b7280;">
										We'll search for reference images, then you pick one for AI to modify.
									</p>
								</div>
								
								<div style="margin-top: 24px; display: flex; gap: 12px;">
									<button type="button" class="button" id="ai-image-cancel-btn" style="flex: 0 0 auto;">
										Cancel
									</button>
									<button type="button" class="button button-primary" id="ai-image-search-btn" style="flex: 1;">
										<span class="dashicons dashicons-search" style="vertical-align: middle; margin-right: 4px;"></span>
										Search Reference Images
									</button>
								</div>
							</div>
							
							<!-- Step 2: Searching -->
							<div id="ai-image-step-searching" style="display: none; text-align: center; padding: 40px 20px;">
								<div class="buildfuly-ai-spinner" style="width: 48px; height: 48px; border: 4px solid #e5e7eb; border-top-color: ${primaryColor}; border-radius: 50%; animation: buildfuly-spin 1s linear infinite; margin: 0 auto 20px;"></div>
								<h3 style="margin: 0 0 8px 0;">Searching for images...</h3>
								<p style="color: #6b7280; margin: 0;">Finding the best reference images</p>
							</div>
							
							<!-- Step 2: Pick Reference Image -->
							<div id="ai-image-step-2" style="display: none;">
								<div style="margin-bottom: 16px;">
									<h3 style="margin: 0 0 4px 0; font-size: 16px;">Pick a Reference Image</h3>
									<p style="margin: 0; color: #6b7280; font-size: 13px;">
										Select the image you want AI to use as inspiration. It will create a unique modified version.
									</p>
								</div>
								
								<div id="ai-image-results" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 12px; margin-bottom: 16px;">
									<!-- Reference images will be inserted here -->
								</div>
								
								<!-- Logo Option -->
								<div id="ai-image-logo-option" style="background: #f9fafb; border: 1px solid #e5e7eb; border-radius: 8px; padding: 12px; margin-bottom: 16px; ${logoUrl ? '' : 'display: none;'}">
									<label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
										<input type="checkbox" id="ai-image-include-logo" style="width: 18px; height: 18px; cursor: pointer;">
										<span style="flex: 1;">
											<strong style="display: block; font-size: 14px;">Include my logo</strong>
											<span style="font-size: 12px; color: #6b7280;">AI will try to place your logo on clothing/uniform or as a subtle brand element</span>
										</span>
										${logoUrl ? '<img src="' + logoUrl + '" style="width: 40px; height: 40px; object-fit: contain; border-radius: 4px; background: white; padding: 4px; border: 1px solid #e5e7eb;">' : ''}
									</label>
								</div>
								
								<div style="display: flex; gap: 12px;">
									<button type="button" class="button" id="ai-image-back-btn">
										<span class="dashicons dashicons-arrow-left-alt" style="vertical-align: middle;"></span>
										Back
									</button>
									<button type="button" class="button" id="ai-image-search-again-btn">
										<span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
										Search Again
									</button>
									<button type="button" class="button button-primary" id="ai-image-create-btn" style="flex: 1;" disabled>
										<span class="dashicons dashicons-art" style="vertical-align: middle; margin-right: 4px;"></span>
										Create from Selected
									</button>
								</div>
							</div>
							
							<!-- Step 3: Generation Progress -->
							<div id="ai-image-step-3" style="display: none; text-align: center; padding: 40px 20px;">
								<div class="buildfuly-ai-spinner" style="width: 48px; height: 48px; border: 4px solid #e5e7eb; border-top-color: ${primaryColor}; border-radius: 50%; animation: buildfuly-spin 1s linear infinite; margin: 0 auto 20px;"></div>
								<h3 id="ai-image-gen-status" style="margin: 0 0 8px 0;">Creating your image...</h3>
								<p style="color: #6b7280; margin: 0;" id="ai-image-gen-substatus">
									AI is modifying the reference to create something unique
								</p>
								<div id="ai-image-gen-steps" style="text-align: left; max-width: 280px; margin: 20px auto 0;">
									<div class="ai-gen-step" data-step="1" style="padding: 6px 0; color: #9ca3af; display: flex; align-items: center; gap: 8px;">
										<span class="dashicons dashicons-download"></span>
										Downloading reference...
									</div>
									<div class="ai-gen-step" data-step="2" style="padding: 6px 0; color: #9ca3af; display: flex; align-items: center; gap: 8px;">
										<span class="dashicons dashicons-visibility"></span>
										Analyzing with AI...
									</div>
									<div class="ai-gen-step" data-step="3" style="padding: 6px 0; color: #9ca3af; display: flex; align-items: center; gap: 8px;">
										<span class="dashicons dashicons-art"></span>
										Creating unique version...
									</div>
									<div class="ai-gen-step" data-step="4" style="padding: 6px 0; color: #9ca3af; display: flex; align-items: center; gap: 8px;">
										<span class="dashicons dashicons-cloud-upload"></span>
										Saving to media library...
									</div>
								</div>
							</div>
							
							<!-- Step 4: Result -->
							<div id="ai-image-step-4" style="display: none;">
								<div id="ai-image-preview" style="text-align: center; margin-bottom: 20px;">
									<!-- Generated image will appear here -->
								</div>
								<div id="ai-image-modifications" style="background: #f3f4f6; padding: 16px; border-radius: 8px; margin-bottom: 12px;">
									<h4 style="margin: 0 0 8px 0; font-size: 14px;">AI Modifications Applied:</h4>
									<ul id="ai-image-mod-list" style="margin: 0; padding-left: 20px; color: #374151; font-size: 13px;">
									</ul>
								</div>
								<!-- Collapsible Prompt & Reference Preview -->
								<div id="ai-image-prompt-section" style="margin-bottom: 20px;">
									<button type="button" id="ai-image-toggle-prompt" class="button" style="width: 100%; text-align: left; display: flex; align-items: center; justify-content: space-between;">
										<span><span class="dashicons dashicons-editor-code" style="vertical-align: middle;"></span> View AI Prompt & Reference Image</span>
										<span class="dashicons dashicons-arrow-down-alt2" style="transition: transform 0.2s;"></span>
									</button>
									<div id="ai-image-prompt-details" style="display: none; margin-top: 8px;">
										<!-- Reference Image Used -->
										<div id="ai-image-reference-used" style="margin-bottom: 12px; background: #f3f4f6; padding: 12px; border-radius: 8px;">
											<h5 style="margin: 0 0 8px 0; font-size: 13px; color: #374151; font-weight: 600;">
												<span class="dashicons dashicons-format-image" style="vertical-align: middle;"></span>
												Reference Image Used:
											</h5>
											<div id="ai-image-reference-thumbnail" style="text-align: center;">
												<!-- Reference image thumbnail will be inserted here -->
											</div>
										</div>
										<!-- AI Prompt -->
										<div style="background: #1e293b; color: #e2e8f0; padding: 12px; border-radius: 8px;">
											<h5 style="margin: 0 0 8px 0; font-size: 13px; color: #e2e8f0; font-weight: 600;">
												<span class="dashicons dashicons-editor-code" style="vertical-align: middle;"></span>
												AI Prompt:
											</h5>
											<div id="ai-image-prompt-content" style="font-family: monospace; font-size: 12px; white-space: pre-wrap; max-height: 200px; overflow-y: auto;">
											</div>
										</div>
									</div>
								</div>
								<div style="display: flex; gap: 12px;">
									<button type="button" class="button" id="ai-image-regenerate-btn">
										<span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
										Try Again
									</button>
									<button type="button" class="button button-primary" id="ai-image-use-btn" style="flex: 1;">
										<span class="dashicons dashicons-yes" style="vertical-align: middle;"></span>
										Use This Image
									</button>
								</div>
							</div>
						</div>
					</div>
				</div>
				<style>
					@keyframes buildfuly-spin {
						to { transform: rotate(360deg); }
					}
					.ai-gen-step.active {
						color: #667eea !important;
						font-weight: 600;
					}
					.ai-gen-step.complete {
						color: #10b981 !important;
					}
					#${this.modalId} .buildfuly-modal-content {
						animation: buildfuly-modal-in 0.2s ease-out;
					}
					@keyframes buildfuly-modal-in {
						from { opacity: 0; transform: scale(0.95); }
						to { opacity: 1; transform: scale(1); }
					}
					.ai-reference-image {
						cursor: pointer;
						border-radius: 8px;
						overflow: hidden;
						border: 3px solid transparent;
						transition: all 0.2s ease;
						aspect-ratio: 4/3;
						background: #f3f4f6;
					}
					.ai-reference-image:hover {
						border-color: #667eea;
						transform: scale(1.02);
					}
					.ai-reference-image.selected {
						border-color: #667eea;
						box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.3);
					}
					.ai-reference-image img {
						width: 100%;
						height: 100%;
						object-fit: cover;
					}
					.ai-reference-image .image-source {
						position: absolute;
						bottom: 0;
						left: 0;
						right: 0;
						background: linear-gradient(transparent, rgba(0,0,0,0.7));
						color: white;
						font-size: 10px;
						padding: 16px 8px 6px;
						white-space: nowrap;
						overflow: hidden;
						text-overflow: ellipsis;
					}
				</style>
			`;
			
			$('body').append(modalHtml);
			this.bindEvents();
			this.isInitialized = true;
		},
		
		/**
		 * Bind modal events
		 */
		bindEvents: function() {
			const self = this;
			const $modal = $(`#${this.modalId}`);
			
			// Close modal
			$modal.on('click', '.buildfuly-ai-image-close, #ai-image-cancel-btn', function() {
				self.close();
			});
			
			// Close on backdrop click
			$modal.on('click', function(e) {
				if (e.target === this) {
					self.close();
				}
			});
			
			// Search for images
			$modal.on('click', '#ai-image-search-btn, #ai-image-search-again-btn', function() {
				self.searchImages();
			});
			
			// Back to search
			$modal.on('click', '#ai-image-back-btn', function() {
				$('#ai-image-step-2').hide();
				$('#ai-image-step-1').show();
			});
			
			// Select reference image
			$modal.on('click', '.ai-reference-image', function() {
				$('.ai-reference-image').removeClass('selected');
				$(this).addClass('selected');
				self.selectedReferenceUrl = $(this).data('url');
				$('#ai-image-create-btn').prop('disabled', false);
			});
			
			// Create from selected
			$modal.on('click', '#ai-image-create-btn', function() {
				if (self.selectedReferenceUrl) {
					self.generateFromReference();
				}
			});
			
			// Regenerate (go back to pick step)
			$modal.on('click', '#ai-image-regenerate-btn', function() {
				$('#ai-image-step-4').hide();
				$('#ai-image-step-2').show();
			});
			
			// Use image
			$modal.on('click', '#ai-image-use-btn', function() {
				if (self.generatedImageData && self.onSelectCallback) {
					self.onSelectCallback(self.generatedImageData);
				}
				self.close();
			});
			
			// Enter key to search
			$modal.on('keypress', '#ai-image-topic', function(e) {
				if (e.which === 13) {
					$('#ai-image-search-btn').click();
				}
			});
			
			// Toggle prompt visibility
			$modal.on('click', '#ai-image-toggle-prompt', function() {
				const $content = $('#ai-image-prompt-details');
				const $arrow = $(this).find('.dashicons-arrow-down-alt2, .dashicons-arrow-up-alt2');
				
				if ($content.is(':visible')) {
					$content.slideUp(200);
					$arrow.removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
				} else {
					$content.slideDown(200);
					$arrow.removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
				}
			});
		},
		
		/**
		 * Open the modal
		 * @param {Object} options - { topic: string, onSelect: function }
		 */
		open: function(options) {
			this.init();
			
			options = options || {};
			this.onSelectCallback = options.onSelect || null;
			this.generatedImageData = null;
			this.selectedReferenceUrl = null;
			
			// Reset to step 1
			$('#ai-image-step-1').show();
			$('#ai-image-step-searching, #ai-image-step-2, #ai-image-step-3, #ai-image-step-4').hide();
			$('#ai-image-create-btn').prop('disabled', true);
			
			// Set default topic if provided
			$('#ai-image-topic').val(options.topic || '');
			
			// Show modal
			$(`#${this.modalId}`).css('display', 'flex');
			$('#ai-image-topic').focus();
		},
		
		/**
		 * Close the modal
		 */
		close: function() {
			$(`#${this.modalId}`).fadeOut(200);
		},
		
		/**
		 * Search for reference images
		 */
		searchImages: async function() {
			const topic = $('#ai-image-topic').val().trim();
			if (!topic) {
				alert('Please describe what the image should show.');
				return;
			}
			
			this.currentPrompt = topic;
			this.selectedReferenceUrl = null;
			$('#ai-image-create-btn').prop('disabled', true);
			
			// Show searching state
			$('#ai-image-step-1, #ai-image-step-2').hide();
			$('#ai-image-step-searching').show();
			
			try {
				const response = await fetch(buildfulyAdmin.apiUrl + '/api/images/search', {
					method: 'POST',
					headers: {
						'Content-Type': 'application/json',
						'Authorization': 'Bearer ' + buildfulyAdmin.licenseKey
					},
					body: JSON.stringify({
						query: topic,
						count: 6
					})
				});
				
				const data = await response.json();
				
				if (data.success && data.results && data.results.length > 0) {
					// Build image grid
					let html = '';
					data.results.forEach((img, i) => {
						html += `
							<div class="ai-reference-image" data-url="${img.url}" style="position: relative;">
								<img src="${img.thumbnail}" alt="${img.title}" loading="lazy" 
									 onerror="this.parentElement.style.display='none'">
								<div class="image-source">${img.source}</div>
							</div>
						`;
					});
					
					$('#ai-image-results').html(html);
					
					// Show pick step
					$('#ai-image-step-searching').hide();
					$('#ai-image-step-2').show();
					
				} else {
					throw new Error(data.error || 'No images found');
				}
				
			} catch (error) {
				console.error('Image search error:', error);
				alert('Failed to search images: ' + error.message);
				$('#ai-image-step-searching').hide();
				$('#ai-image-step-1').show();
			}
		},
		
		/**
		 * Generate from the selected reference image
		 */
		generateFromReference: async function() {
			if (!this.selectedReferenceUrl) {
				alert('Please select a reference image first.');
				return;
			}
			
			// Show progress
			$('#ai-image-step-2').hide();
			$('#ai-image-step-3').show();
			this.updateStep(1);
			
			try {
				// Get brand settings
				const businessName = window.buildfulyAdmin?.businessName || '';
				const logoUrl = window.buildfulyAdmin?.logoUrl || '';
				const primaryColor = window.buildfulyAdmin?.primaryColor || '#667eea';
				const secondaryColor = window.buildfulyAdmin?.secondaryColor || '#764ba2';
				
				// Get logo checkbox state
				const includeLogo = $('#ai-image-include-logo').is(':checked');
				
				this.updateStep(2);
				
				console.log('[ImageGenerator] Sending request with:', {
					referenceUrl: this.selectedReferenceUrl.substring(0, 50) + '...',
					prompt: this.currentPrompt,
					includeLogo: includeLogo,
					hasLogo: !!logoUrl
				});
				
				const response = await fetch(buildfulyAdmin.apiUrl + '/api/images/generate-from-reference', {
					method: 'POST',
					headers: {
						'Content-Type': 'application/json',
						'Authorization': 'Bearer ' + buildfulyAdmin.licenseKey
					},
					body: JSON.stringify({
						referenceUrl: this.selectedReferenceUrl,
						prompt: this.currentPrompt,
						businessName: businessName || null,
						logoUrl: logoUrl || null,
						includeLogo: includeLogo,
						colors: {
							primary: primaryColor,
							secondary: secondaryColor
						}
					})
				});
				
				this.updateStep(3);
				
				const data = await response.json();
				
				if (data.success) {
					this.updateStep(4);
					
					// Store the prompt used for display
					this.lastPromptUsed = data.promptUsed || null;
					
					// Save to WordPress media library via AJAX
					const saveResponse = await $.ajax({
						url: buildfulyAdmin.ajaxUrl,
						type: 'POST',
						data: {
							action: 'buildfuly_upload_image_from_url',
							nonce: $('#buildfuly-nonce').val(), // Use the correct nonce for this action
							image_url: data.imageUrl,
							title: this.currentPrompt,
							alt_text: data.description || this.currentPrompt,
							is_ai_generated: 'true'
						}
					});
					
					// Show result
					$('#ai-image-step-3').hide();
					$('#ai-image-step-4').show();
					
					$('#ai-image-preview').html(
						'<img src="' + data.imageUrl + '" style="max-width: 100%; max-height: 350px; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">'
					);
					
					const modifications = data.modifications || ['AI-generated unique image'];
					$('#ai-image-mod-list').html(
						modifications.map(m => '<li>' + m + '</li>').join('')
					);
					
					// Display the prompt used and reference image
					if (data.promptUsed) {
						$('#ai-image-prompt-content').text(data.promptUsed);
						
						// Show the reference image that was used
						if (this.selectedReferenceUrl) {
							$('#ai-image-reference-thumbnail').html(
								'<img src="' + this.selectedReferenceUrl + '" style="max-width: 100%; max-height: 180px; border-radius: 6px; border: 2px solid #d1d5db;">' +
								'<p style="margin: 8px 0 0 0; font-size: 11px; color: #6b7280; font-style: italic;">This image was used as reference for AI modifications</p>'
							);
						}
						
						$('#ai-image-prompt-section').show();
					} else {
						$('#ai-image-prompt-section').hide();
					}
					// Reset prompt toggle state
					$('#ai-image-prompt-details').hide();
					$('#ai-image-toggle-prompt .dashicons-arrow-up-alt2').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
					
					// Store for callback
					this.generatedImageData = {
						url: saveResponse.data?.url || data.imageUrl,
						attachmentId: saveResponse.data?.attachment_id,
						description: data.description || this.currentPrompt,
						modifications: modifications,
						promptUsed: data.promptUsed,
						referenceImageUrl: this.selectedReferenceUrl
					};
					
				} else {
					throw new Error(data.error || 'Generation failed');
				}
				
			} catch (error) {
				console.error('Image generation error:', error);
				alert('Failed to generate image: ' + error.message);
				$('#ai-image-step-3').hide();
				$('#ai-image-step-2').show();
			}
		},
		
		/**
		 * Update the progress step indicator
		 */
		updateStep: function(step) {
			$('.ai-gen-step').each(function() {
				const stepNum = parseInt($(this).data('step'));
				$(this).removeClass('active complete');
				if (stepNum < step) {
					$(this).addClass('complete');
				} else if (stepNum === step) {
					$(this).addClass('active');
				}
			});
		}
	};

})(jQuery);
