/**
 * Shared Button Management Logic (DRY principle)
 * Used by both Onboarding and Settings pages
 */

const BuildfulyButtonManager = {
    /**
     * Validate if a URL is valid (includes phone, email, hashtag anchors)
     */
    isValidUrl: function(url) {
        if (!url || url.trim() === '') return false;
        
        url = url.trim();
        
        // Check for tel: links
        if (url.startsWith('tel:')) return true;
        
        // Check for mailto: links
        if (url.startsWith('mailto:')) return true;
        
        // Check for hashtag anchors
        if (url.startsWith('#')) return true;
        
        // Check for valid HTTP/HTTPS URLs
        try {
            const urlObj = new URL(url);
            return urlObj.protocol === 'http:' || urlObj.protocol === 'https:';
        } catch (e) {
            // If it doesn't have protocol, check if it looks like a domain
            if (url.match(/^[a-zA-Z0-9][a-zA-Z0-9-]{1,61}[a-zA-Z0-9]\.[a-zA-Z]{2,}/)) {
                return true;
            }
            return false;
        }
    },

    /**
     * Format phone number for tel: link
     */
    formatPhoneLink: function(phone) {
        // Remove all non-digit characters
        const cleaned = phone.replace(/\D/g, '');
        return 'tel:+1' + cleaned; // Assuming US numbers, can be adjusted
    },

    /**
     * Create button HTML with live preview
     */
    createButtonRow: function(index, buttonData = {}) {
        const text = buttonData.text || '';
        const url = buttonData.url || '';
        const style = buttonData.style || 'primary';
        const type = buttonData.type || 'link';
        const phone = buttonData.phone || '';
        
        return `
            <div class="button-row" data-index="${index}">
                <div class="button-row-header">
                    <div class="button-row-controls">
                        <span class="drag-handle" title="Drag to reorder">
                            <span class="dashicons dashicons-menu"></span>
                        </span>
                        <h4 style="margin: 0; font-size: 15px; color: #1d1d1f;">Button ${index + 1}</h4>
                    </div>
                    <button type="button" class="remove-button-btn" data-index="${index}" title="Remove button">
                        <span class="dashicons dashicons-no-alt"></span>
                    </button>
                </div>
                
                <div class="button-row-content">
                    <div class="button-inputs">
                        <div class="input-group">
                            <label>Button Text</label>
                            <input type="text" 
                                   class="button-text-input" 
                                   placeholder="Get Started" 
                                   value="${text}"
                                   data-index="${index}">
                        </div>
                        
                        <div class="input-group">
                            <label>Button Action</label>
                            <select class="button-type-select" data-index="${index}">
                                <option value="link" ${type === 'link' ? 'selected' : ''}>Link to URL</option>
                                <option value="phone" ${type === 'phone' ? 'selected' : ''}>Phone Call</option>
                                <option value="anchor" ${type === 'anchor' ? 'selected' : ''}>Page Anchor (#section)</option>
                            </select>
                        </div>
                        
                        <div class="input-group button-url-group" style="display: ${type === 'link' ? 'block' : 'none'};">
                            <label>URL</label>
                            <input type="url" 
                                   class="button-url-input" 
                                   placeholder="https://example.com/contact" 
                                   value="${type === 'link' ? url : ''}"
                                   data-index="${index}">
                        </div>
                        
                        <div class="input-group button-phone-group" style="display: ${type === 'phone' ? 'block' : 'none'};">
                            <label>Phone Number</label>
                            <input type="tel" 
                                   class="button-phone-input" 
                                   placeholder="(555) 123-4567" 
                                   value="${type === 'phone' ? phone : ''}"
                                   data-index="${index}">
                        </div>
                        
                        <div class="input-group button-anchor-group" style="display: ${type === 'anchor' ? 'block' : 'none'};">
                            <label>Anchor (without #)</label>
                            <input type="text" 
                                   class="button-anchor-input" 
                                   placeholder="contact" 
                                   value="${type === 'anchor' ? url.replace('#', '') : ''}"
                                   data-index="${index}">
                        </div>
                        
                        <div class="input-group">
                            <label>Button Style</label>
                            <select class="button-style-select" data-index="${index}">
                                <option value="primary" ${style === 'primary' ? 'selected' : ''}>Primary (Blue)</option>
                                <option value="secondary" ${style === 'secondary' ? 'selected' : ''}>Secondary (White)</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="button-preview">
                        <label>Preview:</label>
                        <div class="preview-container">
                            <button type="button" class="preview-button" data-index="${index}">
                                ${text || 'Button Text'}
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
    },

    /**
     * Bind events for a button row
     */
    bindButtonEvents: function(container, onChange) {
        const $container = $(container);
        
        // Text input
        $container.on('input', '.button-text-input', function() {
            const index = $(this).data('index');
            const text = $(this).val();
            $(`.button-row[data-index="${index}"] .preview-button`).text(text || 'Button Text');
            if (onChange) onChange();
        });
        
        // Type selector
        $container.on('change', '.button-type-select', function() {
            const index = $(this).data('index');
            const type = $(this).val();
            const $row = $(`.button-row[data-index="${index}"]`);
            
            // Hide all input groups
            $row.find('.button-url-group, .button-phone-group, .button-anchor-group').hide();
            
            // Show relevant input group
            if (type === 'link') {
                $row.find('.button-url-group').show();
            } else if (type === 'phone') {
                $row.find('.button-phone-group').show();
            } else if (type === 'anchor') {
                $row.find('.button-anchor-group').show();
            }
            
            if (onChange) onChange();
        });
        
        // URL, phone, anchor inputs
        $container.on('input', '.button-url-input, .button-phone-input, .button-anchor-input', function() {
            if (onChange) onChange();
        });
        
        // Style selector - update preview
        $container.on('change', '.button-style-select', function() {
            const index = $(this).data('index');
            const style = $(this).val();
            const $preview = $(`.button-row[data-index="${index}"] .preview-button`);
            
            $preview.removeClass('preview-primary preview-secondary');
            $preview.addClass(`preview-${style}`);
            
            if (onChange) onChange();
        });
        
        // Remove button
        $container.on('click', '.remove-button-btn', function() {
            const index = $(this).data('index');
            $(`.button-row[data-index="${index}"]`).remove();
            if (onChange) onChange();
        });
        
        // Initialize preview styles
        $container.find('.button-row').each(function() {
            const index = $(this).data('index');
            const style = $(this).find('.button-style-select').val();
            $(`.button-row[data-index="${index}"] .preview-button`)
                .removeClass('preview-primary preview-secondary')
                .addClass(`preview-${style}`);
        });
    },

    /**
     * Collect button data from DOM
     */
    collectButtons: function(container) {
        const buttons = [];
        $(container).find('.button-row').each(function() {
            const $row = $(this);
            const index = $row.data('index');
            const text = $row.find('.button-text-input').val().trim();
            const type = $row.find('.button-type-select').val();
            const style = $row.find('.button-style-select').val();
            
            if (!text) return; // Skip empty buttons
            
            let url = '';
            let phone = '';
            
            if (type === 'link') {
                url = $row.find('.button-url-input').val().trim();
            } else if (type === 'phone') {
                phone = $row.find('.button-phone-input').val().trim();
                url = BuildfulyButtonManager.formatPhoneLink(phone);
            } else if (type === 'anchor') {
                const anchor = $row.find('.button-anchor-input').val().trim();
                url = anchor ? `#${anchor}` : '#';
            }
            
            buttons.push({ text, url, style, type, phone });
        });
        
        return buttons;
    },

    /**
     * Save buttons to WordPress options
     */
    saveButtons: function(buttons, callback) {
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'buildfuly_save_cta_buttons',
                buttons: JSON.stringify(buttons),
                nonce: buildfulyAdmin.ajaxNonce
            },
            success: function(response) {
                if (response.success) {
                    console.log('✅ Buttons saved successfully');
                    if (callback) callback(true);
                } else {
                    console.error('❌ Failed to save buttons:', response.data);
                    if (callback) callback(false, response.data);
                }
            },
            error: function(xhr, status, error) {
                console.error('❌ AJAX error saving buttons:', error);
                if (callback) callback(false, error);
            }
        });
    }
};
