/**
 * Buildfuly Blog Builder - AI Blog Post Generator
 * Reuses shared utilities from buildfuly-utils.js
 */

(function($) {
	'use strict';

	// Shorthand reference to shared utilities
	const Utils = window.BuildfulyUtils || {};

	const BuildfulyBlogBuilder = {
		blogs: [],
		currentBlog: null,
		currentTab: 'pages', // 'pages' or 'blogs'

	/**
	 * Debug logging
	 */
	debug: function(...args) {
		Utils.debug('Blog', ...args);
	},

	/**
	 * Initialize Blog Builder
	 */
	init: function() {
		this.debug('Initializing Blog Builder...');
		this.loadBlogs();
		this.bindEvents();
	},

	/**
	 * Bind all event handlers
	 */
	bindEvents: function() {
		// Tab switching
		$('.buildfuly-tab-switcher .buildfuly-tab-btn').on('click', (e) => this.switchTab(e));
		
		// New blog buttons
		$('#buildfuly-new-blog').on('click', (e) => this.showNewBlogModal(e));
		$('#buildfuly-analyze-blog').on('click', () => this.analyzeBlogTopic());
		$('#buildfuly-back-to-blog-topic').on('click', () => this.backToBlogTopic());
		$('#buildfuly-create-blog').on('click', () => this.createBlog());
		$('#buildfuly-cancel-blog, #buildfuly-blog-modal-close').on('click', () => this.hideNewBlogModal());
		
		// Generate blog ideas modal
		$('#buildfuly-generate-more-blogs').on('click', () => this.showSuggestBlogsModal());
		$('#buildfuly-suggest-blogs-close, #buildfuly-suggest-blogs-cancel').on('click', () => this.hideSuggestBlogsModal());
		$('#buildfuly-suggest-blogs-generate').on('click', () => this.generateSuggestedBlogs());
		$('#buildfuly-suggest-blogs-add').on('click', () => this.addSelectedBlogTopics());
		
		// Blog count slider
		$('#suggest-blog-count').on('input', () => this.updateBlogCountSlider());
		
		// Word count slider
		$('#buildfuly-blog-word-count').on('input', () => this.updateWordCountDisplay());
		
		// Featured image upload
		$('#buildfuly-upload-blog-image').on('click', () => this.uploadBlogImage());
		
		// Blog selection
		$(document).on('change', '#suggested-blogs-grid .page-selection-item input[type="checkbox"]', () => this.updateSelectedBlogsCount());
		$(document).on('click', '#suggested-blogs-grid .page-selection-item', (e) => this.handleBlogTopicClick(e));
	},

	/**
	 * Switch between Pages and Blogs tabs
	 */
	switchTab: function(e) {
		const $btn = $(e.currentTarget);
		const tab = $btn.data('tab');
		
		// Update button states
		$('.buildfuly-tab-switcher .buildfuly-tab-btn').removeClass('active');
		$btn.addClass('active');
		
		// Update button styles
		$('.buildfuly-tab-switcher .buildfuly-tab-btn').css({
			'background': 'transparent',
			'color': '#6b7280',
			'box-shadow': 'none'
		});
		$btn.css({
			'background': 'white',
			'color': '#374151',
			'box-shadow': '0 1px 2px rgba(0,0,0,0.05)'
		});
		
		// Show/hide content using active class
		if (tab === 'pages') {
			$('#buildfuly-tab-pages').addClass('active');
			$('#buildfuly-tab-blogs').removeClass('active');
			$('#buildfuly-pages-list').show();
			$('#buildfuly-blogs-list').hide();
			this.currentTab = 'pages';
		} else {
			$('#buildfuly-tab-pages').removeClass('active');
			$('#buildfuly-tab-blogs').addClass('active');
			$('#buildfuly-pages-list').hide();
			$('#buildfuly-blogs-list').show();
			this.currentTab = 'blogs';
			this.renderBlogsList();
		}
		
		this.debug('Switched to tab:', tab);
	},

	/**
	 * Load blogs from local storage
	 */
	loadBlogs: function() {
		const saved = localStorage.getItem('buildfuly_blogs');
		if (saved) {
			try {
				this.blogs = JSON.parse(saved);
			} catch (e) {
				this.blogs = [];
			}
		}
		this.debug('Loaded blogs:', this.blogs.length);
	},

	/**
	 * Save blogs to local storage
	 */
	saveBlogs: function() {
		localStorage.setItem('buildfuly_blogs', JSON.stringify(this.blogs));
	},

	/**
	 * Render the blogs list in sidebar
	 */
	renderBlogsList: function() {
		const $list = $('#buildfuly-blogs-list');
		$list.empty();
		
		if (this.blogs.length === 0) {
			$list.html(`
				<div style="text-align: center; padding: 40px 20px; color: #6b7280;">
					<span class="dashicons dashicons-edit" style="font-size: 48px; width: 48px; height: 48px; margin-bottom: 12px; opacity: 0.3;"></span>
					<p style="margin: 0;">No blog posts yet</p>
					<p style="margin: 8px 0 0 0; font-size: 13px;">Click "New Blog Post" to create one</p>
				</div>
			`);
			return;
		}
		
		this.blogs.forEach((blog, index) => {
			const isPublished = blog.wordpress_id && blog.published;
			const statusBadge = isPublished 
				? '<span class="buildfuly-published-badge" title="Published"><span class="dashicons dashicons-yes-alt"></span></span>'
				: '';
			
			const $item = $(`
				<div class="buildfuly-page-item ${blog.id === this.currentBlog?.id ? 'active' : ''} ${isPublished ? 'is-published' : ''}" data-blog-id="${blog.id}">
					<div class="buildfuly-page-icon">
						<span class="dashicons dashicons-edit"></span>
					</div>
					<div class="buildfuly-page-info" style="flex: 1; min-width: 0;">
						<div class="buildfuly-page-title" style="font-weight: 600; color: #1d1d1f; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
							${this.escapeHtml(blog.title || 'Untitled Blog')}
							${statusBadge}
						</div>
						<div class="buildfuly-page-meta" style="font-size: 12px; color: #6b7280; margin-top: 2px;">
							${blog.keywords ? blog.keywords.slice(0, 2).join(', ') : 'No keywords'}
							${isPublished ? '<span style="color: #10b981; margin-left: 8px;">Published</span>' : ''}
						</div>
					</div>
					<button class="buildfuly-page-delete" data-blog-id="${blog.id}" title="Delete" style="background: none; border: none; cursor: pointer; color: #ef4444; padding: 4px; opacity: 0; transition: opacity 0.2s;">
						<span class="dashicons dashicons-trash"></span>
					</button>
				</div>
			`);
			
			// Hover to show delete button
			$item.on('mouseenter', function() {
				$(this).find('.buildfuly-page-delete').css('opacity', '1');
			}).on('mouseleave', function() {
				$(this).find('.buildfuly-page-delete').css('opacity', '0');
			});
			
			// Click to load blog
			$item.on('click', (e) => {
				if (!$(e.target).closest('.buildfuly-page-delete').length) {
					this.loadBlog(blog.id);
				}
			});
			
			// Delete button
			$item.find('.buildfuly-page-delete').on('click', (e) => {
				e.stopPropagation();
				this.deleteBlog(blog.id);
			});
			
			$list.append($item);
		});
	},

	/**
	 * Show new blog modal
	 */
	showNewBlogModal: function(e) {
		if (e && e.preventDefault) e.preventDefault();
		
		// Reset modal
		$('#blog-topic').val('');
		$('#blog-title-input').val('');
		$('#blog-keywords').val('');
		$('#buildfuly-blog-word-count').val(1200);
		$('#buildfuly-blog-word-count-label').text('1200 words');
		$('#buildfuly-blog-reading-time').text('~5 min');
		$('#buildfuly-blog-image-id').val('');
		$('#buildfuly-blog-image-preview').empty();
		$('input[name="blog-type"][value="how-to"]').prop('checked', true);
		
		// Show step 1
		$('#buildfuly-blog-step-1').show();
		$('#buildfuly-blog-step-2').hide();
		$('#buildfuly-create-blog').hide();
		
		$('#buildfuly-new-blog-modal').fadeIn(200);
		$('#blog-topic').focus();
	},

	/**
	 * Hide new blog modal
	 */
	hideNewBlogModal: function() {
		$('#buildfuly-new-blog-modal').fadeOut(200);
	},

	/**
	 * Analyze blog topic and generate outline
	 */
	analyzeBlogTopic: async function() {
		const topic = $('#blog-topic').val().trim();
		
		if (!topic) {
			this.showMessage('Please enter a blog topic', 'error');
			return;
		}
		
		const $btn = $('#buildfuly-analyze-blog');
		const originalText = $btn.html();
		$btn.prop('disabled', true).html('<span class="spin dashicons dashicons-update"></span> Generating...');
		
		try {
			// Generate keywords and title from topic
			const keywords = this.generateBlogKeywords(topic);
			const title = this.generateBlogTitle(topic);
			
			// Populate step 2
			$('#blog-title-input').val(title);
			$('#blog-keywords').val(keywords.join(', '));
			
			// Show step 2
			$('#buildfuly-blog-step-1').hide();
			$('#buildfuly-blog-step-2').show();
			$('#buildfuly-create-blog').show();
			
		} catch (error) {
			console.error('Blog analysis error:', error);
			this.showMessage('Failed to analyze topic', 'error');
		} finally {
			$btn.prop('disabled', false).html(originalText);
		}
	},

	/**
	 * Generate keywords from blog topic
	 */
	generateBlogKeywords: function(topic) {
		// Simple keyword extraction
		const words = topic.toLowerCase().split(/\s+/);
		const keywords = [topic];
		
		// Add variations
		if (topic.toLowerCase().startsWith('how to')) {
			keywords.push(topic.replace(/^how to /i, ''));
			keywords.push(topic + ' guide');
		} else if (/^\d+/.test(topic)) {
			keywords.push(topic.replace(/^\d+\s*/, ''));
			keywords.push('best ' + topic.replace(/^\d+\s*/, ''));
		} else {
			keywords.push('how to ' + topic);
			keywords.push(topic + ' tips');
		}
		
		return keywords.slice(0, 4);
	},

	/**
	 * Generate blog title from topic
	 */
	generateBlogTitle: function(topic) {
		// Clean up and title case
		let title = topic.trim();
		
		// If it doesn't look like a title, format it
		if (!title.match(/^(how|what|why|when|where|\d+)/i)) {
			title = title.replace(/\b\w/g, l => l.toUpperCase());
		}
		
		return title;
	},

	/**
	 * Go back to blog topic step
	 */
	backToBlogTopic: function() {
		$('#buildfuly-blog-step-1').show();
		$('#buildfuly-blog-step-2').hide();
		$('#buildfuly-create-blog').hide();
	},

	/**
	 * Update word count display
	 */
	updateWordCountDisplay: function() {
		const wordCount = parseInt($('#buildfuly-blog-word-count').val());
		$('#buildfuly-blog-word-count-label').text(wordCount + ' words');
		
		// Calculate reading time (average 200 words per minute)
		const readingTime = Math.ceil(wordCount / 200);
		$('#buildfuly-blog-reading-time').text('~' + readingTime + ' min');
	},

	/**
	 * Upload featured image for blog - uses shared media library utility
	 */
	uploadBlogImage: function() {
		Utils.openMediaLibrary({
			title: 'Select Featured Image',
			buttonText: 'Use Image',
			onSelect: (attachment) => {
				$('#buildfuly-blog-image-id').val(attachment.id);
				
				const imageUrl = attachment.sizes?.medium?.url || attachment.url;
				$('#buildfuly-blog-image-preview').html(`
					<div style="position: relative; display: inline-block;">
						<img src="${imageUrl}" style="max-width: 200px; border-radius: 8px; border: 1px solid #e5e7eb;">
						<button type="button" class="buildfuly-remove-blog-image" style="position: absolute; top: -8px; right: -8px; background: #ef4444; color: white; border: none; border-radius: 50%; width: 24px; height: 24px; cursor: pointer; font-size: 16px; line-height: 1;">×</button>
					</div>
				`);
				
				$('.buildfuly-remove-blog-image').on('click', () => {
					$('#buildfuly-blog-image-id').val('');
					$('#buildfuly-blog-image-preview').empty();
				});
			}
		});
	},

	/**
	 * Create a new blog post
	 */
	createBlog: async function() {
		const title = $('#blog-title-input').val().trim();
		const keywords = $('#blog-keywords').val().split(',').map(k => k.trim()).filter(k => k);
		const wordCount = parseInt($('#buildfuly-blog-word-count').val());
		const blogType = $('input[name="blog-type"]:checked').val();
		const imageId = $('#buildfuly-blog-image-id').val();
		
		if (!title) {
			this.showMessage('Please enter a blog title', 'error');
			return;
		}
		
		const $btn = $('#buildfuly-create-blog');
		const originalText = $btn.html();
		$btn.prop('disabled', true).html('<span class="spin dashicons dashicons-update"></span> Generating...');
		
		// Show loading overlay
		$('#buildfuly-loading').show().find('p').text('Generating blog post content...');
		
		try {
			// Call API to generate blog content
			const response = await $.ajax({
				url: buildfulyAdmin.apiEndpoint + '/api/blogs/generate',
				method: 'POST',
				headers: {
					'Content-Type': 'application/json',
					'Authorization': 'Bearer ' + buildfulyAdmin.token
				},
				data: JSON.stringify({
					topic: title,
					keywords: keywords,
					word_count: wordCount,
					blog_type: blogType,
					business_context: this.getBusinessContext(),
					generate_html: true
				})
			});
			
			if (!response.success) {
				throw new Error(response.error || 'Failed to generate blog');
			}
			
			// Create blog object
			const blog = {
				id: 'blog_' + Date.now(),
				title: response.blog.title || title,
				topic: title,
				keywords: keywords,
				type: blogType,
				word_count: wordCount,
				content: response.blog,
				html: response.html,
				blocks: response.blocks,
				image_id: imageId,
				created: new Date().toISOString(),
				published: false,
				wordpress_id: null
			};
			
			// Add to blogs array
			this.blogs.unshift(blog);
			this.saveBlogs();
			
			// Hide modal and show blog
			this.hideNewBlogModal();
			this.loadBlog(blog.id);
			this.renderBlogsList();
			
			this.showMessage('Blog post generated successfully!', 'success');
			
		} catch (error) {
			console.error('Blog generation error:', error);
			this.showMessage(error.message || 'Failed to generate blog post', 'error');
		} finally {
			$btn.prop('disabled', false).html(originalText);
			$('#buildfuly-loading').hide();
		}
	},

	/**
	 * Load a blog post
	 */
	loadBlog: function(blogId) {
		const blog = this.blogs.find(b => b.id === blogId);
		if (!blog) return;
		
		this.currentBlog = blog;
		this.renderBlogsList();
		this.renderBlogPreview(blog);
		
		// Show workspace controls
		$('.buildfuly-floating-actions').show();
		$('#buildfuly-color-bar').show();
		$('.buildfuly-welcome-screen').hide();
	},

	/**
	 * Render blog preview in main area
	 */
	renderBlogPreview: function(blog) {
		const $container = $('#buildfuly-sections-container');
		
		if (!blog.html) {
			$container.html(`
				<div style="padding: 40px; text-align: center; color: #6b7280;">
					<p>Blog content not available</p>
				</div>
			`);
			return;
		}
		
		// Render blog HTML with styling
		$container.html(`
			<article class="buildfuly-blog-preview" style="max-width: 800px; margin: 0 auto; padding: 40px;">
				<header style="margin-bottom: 32px;">
					<h1 style="font-size: 36px; font-weight: 700; color: #1d1d1f; margin: 0 0 16px 0; line-height: 1.2;">
						${this.escapeHtml(blog.content.title || blog.title)}
					</h1>
					${blog.content.excerpt ? `<p style="font-size: 18px; color: #6b7280; margin: 0; line-height: 1.6;">${this.escapeHtml(blog.content.excerpt)}</p>` : ''}
				</header>
				<div class="blog-content" style="font-size: 17px; line-height: 1.8; color: #374151;">
					${blog.html}
				</div>
				${blog.content.tags && blog.content.tags.length > 0 ? `
					<footer style="margin-top: 32px; padding-top: 24px; border-top: 1px solid #e5e7eb;">
						<div style="display: flex; flex-wrap: wrap; gap: 8px;">
							${blog.content.tags.map(tag => `<span style="background: #f3f4f6; color: #6b7280; padding: 4px 12px; border-radius: 16px; font-size: 13px;">${this.escapeHtml(tag)}</span>`).join('')}
						</div>
					</footer>
				` : ''}
			</article>
		`);
	},

	/**
	 * Delete a blog post
	 */
	deleteBlog: function(blogId) {
		if (!confirm('Are you sure you want to delete this blog post?')) return;
		
		this.blogs = this.blogs.filter(b => b.id !== blogId);
		this.saveBlogs();
		
		if (this.currentBlog?.id === blogId) {
			this.currentBlog = null;
			$('#buildfuly-sections-container').html(`
				<div class="buildfuly-welcome-screen">
					<div class="buildfuly-welcome-icon">
						<span class="dashicons dashicons-edit"></span>
					</div>
					<h2>Create Your First Blog Post</h2>
					<p>Generate SEO-optimized blog content with AI</p>
				</div>
			`);
		}
		
		this.renderBlogsList();
		this.showMessage('Blog deleted', 'success');
	},

	/**
	 * Show suggest blogs modal
	 */
	showSuggestBlogsModal: function() {
		// Reset state
		$('#suggest-blogs-config-form').show();
		$('#suggested-blogs-loading').hide();
		$('#suggested-blogs-selection').hide();
		$('#suggested-blogs-success').hide();
		$('#buildfuly-suggest-blogs-generate').show();
		$('#buildfuly-suggest-blogs-add').hide();
		$('#suggest-blog-count').val(5);
		$('#suggest-blog-count-display').text('5');
		this.updateBlogCountSlider();
		
		$('#buildfuly-suggest-blogs-modal').fadeIn(200);
	},

	/**
	 * Hide suggest blogs modal
	 */
	hideSuggestBlogsModal: function() {
		$('#buildfuly-suggest-blogs-modal').fadeOut(200);
	},

	/**
	 * Update blog count slider
	 */
	updateBlogCountSlider: function() {
		const count = parseInt($('#suggest-blog-count').val());
		$('#suggest-blog-count-display').text(count);
		$('#suggest-blog-token-cost').text('~' + (count * 5));
		
		// Update slider background
		const percent = ((count - 1) / 19) * 100;
		$('#suggest-blog-count').css('background', `linear-gradient(to right, #10b981 0%, #10b981 ${percent}%, #e5e7eb ${percent}%, #e5e7eb 100%)`);
	},

	/**
	 * Generate suggested blog topics
	 */
	generateSuggestedBlogs: async function() {
		const count = parseInt($('#suggest-blog-count').val());
		
		// Show loading
		$('#suggest-blogs-config-form').hide();
		$('#suggested-blogs-loading').show();
		$('#suggest-blog-generation-status').text('Generating blog ideas...');
		
		try {
			const existingTitles = this.blogs.map(b => b.title);
			
			const response = await $.ajax({
				url: buildfulyAdmin.apiEndpoint + '/api/blogs/suggest',
				method: 'POST',
				headers: {
					'Content-Type': 'application/json',
					'Authorization': 'Bearer ' + buildfulyAdmin.token
				},
				data: JSON.stringify({
					description: buildfulyAdmin.businessDescription || '',
					business_name: buildfulyAdmin.businessName || '',
					location: buildfulyAdmin.location || '',
					count: count,
					existing_posts: existingTitles
				})
			});
			
			if (!response.success) {
				throw new Error(response.error || 'Failed to generate topics');
			}
			
			// Display topics
			this.displaySuggestedBlogs(response.topics);
			
		} catch (error) {
			console.error('Blog suggestion error:', error);
			$('#suggested-blogs-loading').hide();
			$('#suggest-blogs-config-form').show();
			this.showMessage(error.message || 'Failed to generate blog ideas', 'error');
		}
	},

	/**
	 * Display suggested blog topics
	 */
	displaySuggestedBlogs: function(topics) {
		$('#suggested-blogs-loading').hide();
		$('#suggested-blogs-selection').show();
		$('#buildfuly-suggest-blogs-generate').hide();
		$('#buildfuly-suggest-blogs-add').show();
		
		const $grid = $('#suggested-blogs-grid');
		$grid.empty();
		
		topics.forEach((topic, index) => {
			const keywords = topic.keywords ? topic.keywords.join(', ') : '';
			$grid.append(`
				<div class="page-selection-item" data-index="${index}">
					<input type="checkbox" checked data-topic='${JSON.stringify(topic).replace(/'/g, "&#39;")}'>
					<h4>${this.escapeHtml(topic.title)}</h4>
					<p class="page-keyword">${this.escapeHtml(keywords)}</p>
					${topic.type ? `<span style="display: inline-block; margin-top: 8px; padding: 2px 8px; background: #f3f4f6; border-radius: 4px; font-size: 11px; color: #6b7280;">${topic.type}</span>` : ''}
				</div>
			`);
		});
		
		this.updateSelectedBlogsCount();
	},

	/**
	 * Handle blog topic click
	 */
	handleBlogTopicClick: function(e) {
		if ($(e.target).is('input[type="checkbox"]')) return;
		
		const $item = $(e.currentTarget);
		const $checkbox = $item.find('input[type="checkbox"]');
		$checkbox.prop('checked', !$checkbox.prop('checked'));
		$item.toggleClass('selected', $checkbox.prop('checked'));
		this.updateSelectedBlogsCount();
	},

	/**
	 * Update selected blogs count
	 */
	updateSelectedBlogsCount: function() {
		const count = $('#suggested-blogs-grid input[type="checkbox"]:checked').length;
		$('#suggest-blogs-selected-count').text(count);
	},

	/**
	 * Add selected blog topics
	 */
	addSelectedBlogTopics: function() {
		const $checkboxes = $('#suggested-blogs-grid input[type="checkbox"]:checked');
		
		$checkboxes.each((i, checkbox) => {
			const topic = JSON.parse($(checkbox).attr('data-topic'));
			
			const blog = {
				id: 'blog_' + Date.now() + '_' + i,
				title: topic.title,
				topic: topic.topic || topic.title,
				keywords: topic.keywords || [],
				type: topic.type || 'guide',
				word_count: topic.estimated_words || 1200,
				content: null,
				html: null,
				created: new Date().toISOString(),
				published: false,
				wordpress_id: null,
				needsGeneration: true // Flag to indicate content needs to be generated
			};
			
			this.blogs.push(blog);
		});
		
		this.saveBlogs();
		
		// Show success
		$('#suggested-blogs-selection').hide();
		$('#suggested-blogs-success').show();
		$('#buildfuly-suggest-blogs-add').hide();
		
		// Refresh list
		this.renderBlogsList();
		
		// Auto close after delay
		setTimeout(() => {
			this.hideSuggestBlogsModal();
		}, 1500);
		
		this.showMessage(`${$checkboxes.length} blog topics added!`, 'success');
	},

	/**
	 * Get business context for AI - uses shared utility
	 */
	getBusinessContext: function() {
		return Utils.getBusinessContext().toString();
	},

	/**
	 * Show message notification - delegates to shared utility
	 */
	showMessage: function(message, type = 'success') {
		Utils.showMessage(message, type);
	},

	/**
	 * Escape HTML entities - delegates to shared utility
	 */
	escapeHtml: function(text) {
		return Utils.escapeHtml(text);
	}
};

	// Initialize when DOM is ready
	$(document).ready(function() {
		if ($('.buildfuly-admin-wrap').length) {
			BuildfulyBlogBuilder.init();
			
			// Make available globally
			window.BuildfulyBlogBuilder = BuildfulyBlogBuilder;
		}
	});

})(jQuery);
