/**
 * Shared Page Generation Animation Helper
 * Award-winning tower-building animation system
 */
window.BuildfulyAnimations = window.BuildfulyAnimations || {
	/**
	 * Animate page generation with tower-building visual
	 * @param {Object} config - Animation configuration
	 * @param {string} config.statusId - ID of main status text element
	 * @param {string} config.substatusId - ID of substatus text element
	 * @param {string} config.liveContainerId - ID for live page cards container (optional)
	 * @param {Object} config.context - Additional context data (optional)
	 */
	animateGeneration: function(config) {
		const $ = jQuery;
		const $status = $('#' + config.statusId);
		const $substatus = $('#' + config.substatusId);
		const context = config.context || {};
		
		// Create tower container if it doesn't exist
		if (!$('#buildfuly-tower').length) {
			$status.parent().prepend(`
				<div id="buildfuly-tower" style="
					display: flex;
					justify-content: center;
					align-items: flex-end;
					gap: 8px;
					height: 120px;
					margin-bottom: 32px;
				">
					<div class="tower-block" data-block="1" style="
						width: 48px;
						height: 0;
						background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
						border-radius: 8px 8px 0 0;
						transition: height 0.6s cubic-bezier(0.34, 1.56, 0.64, 1);
						box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
					"></div>
					<div class="tower-block" data-block="2" style="
						width: 48px;
						height: 0;
						background: linear-gradient(135deg, #0071e3 0%, #0077ed 100%);
						border-radius: 8px 8px 0 0;
						transition: height 0.6s cubic-bezier(0.34, 1.56, 0.64, 1);
						box-shadow: 0 4px 12px rgba(0, 113, 227, 0.3);
					"></div>
					<div class="tower-block" data-block="3" style="
						width: 48px;
						height: 0;
						background: linear-gradient(135deg, #10b981 0%, #059669 100%);
						border-radius: 8px 8px 0 0;
						transition: height 0.6s cubic-bezier(0.34, 1.56, 0.64, 1);
						box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
					"></div>
					<div class="tower-block" data-block="4" style="
						width: 48px;
						height: 0;
						background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
						border-radius: 8px 8px 0 0;
						transition: height 0.6s cubic-bezier(0.34, 1.56, 0.64, 1);
						box-shadow: 0 4px 12px rgba(245, 158, 11, 0.3);
					"></div>
				</div>
			`);
		} else {
			// Reset tower
			$('.tower-block').css('height', '0');
		}
		
		// Phase 1: Reading business info (800ms delay)
		setTimeout(() => {
			$('.tower-block[data-block="1"]').css('height', '60px');
			$status.html('📖 Reading your business information...');
			
			if (context.description) {
				const words = context.description.split(' ').length;
				$substatus.html(`Analyzing <strong>${words} words</strong> of context`);
			} else {
				$substatus.text('Understanding your business goals');
			}
		}, 800);
		
		// Phase 2: Smart filtering (2000ms delay)
		setTimeout(() => {
			$('.tower-block[data-block="2"]').css('height', '80px');
			$status.html('🎯 Preparing smart suggestions...');
			
			if (context.existingPages && context.existingPages.length > 0) {
				$substatus.html(`Avoiding <strong>${context.existingPages.length} existing page${context.existingPages.length > 1 ? 's' : ''}</strong> · Finding fresh ideas`);
			} else if (context.location) {
				$substatus.html(`Including <strong>${context.location}</strong> for local SEO optimization`);
			} else {
				$substatus.text('Identifying unique opportunities for your website');
			}
		}, 2000);
		
		// Phase 3: AI processing (3200ms delay)
		setTimeout(() => {
			$('.tower-block[data-block="3"]').css('height', '100px');
			$status.html('🤖 AI is crafting your pages...');
			
			if (context.count) {
				$substatus.html(`Generating <strong>${context.count} custom page${context.count > 1 ? 's' : ''}</strong> with SEO keywords`);
			} else {
				$substatus.text('Creating personalized page recommendations');
			}
		}, 3200);
		
		// Phase 4: Finalizing (4400ms delay)
		setTimeout(() => {
			$('.tower-block[data-block="4"]').css('height', '70px');
			$status.html('✨ Almost ready...');
			$substatus.html('Building your <strong>SEO-optimized</strong> page suggestions');
			
			// Show live suggestions container if configured
			if (config.liveContainerId) {
				setTimeout(() => {
					$('#' + config.liveContainerId).slideDown(300);
				}, 400);
			}
		}, 4400);
	},
	
	/**
	 * Show a live page card with animation
	 * @param {Object} config - Configuration
	 * @param {string} config.gridId - ID of grid element
	 * @param {string} config.headerId - ID of header element (optional)
	 * @param {string} pageName - Name of the page
	 * @param {string} keyword - Page keyword/description
	 */
	showLivePage: function(config, pageName, keyword) {
		const $ = jQuery;
		const $grid = $('#' + config.gridId);
		
		// Show header on first page
		if (config.headerId && $grid.children().length === 0) {
			$('#' + config.headerId).slideDown(200);
		}
		
		const pageCard = $(`
			<div class="live-page-card" style="
				background: linear-gradient(135deg, #ffffff 0%, #f9fafb 100%);
				padding: 14px 18px;
				border-radius: 14px;
				border: 1.5px solid #d2d2d7;
				opacity: 0;
				transform: translateY(-10px) scale(0.95);
				transition: all 0.4s cubic-bezier(0.34, 1.56, 0.64, 1);
				box-shadow: 0 2px 8px rgba(0,0,0,0.04);
			">
				<div style="
					font-size: 15px;
					font-weight: 600;
					color: #1d1d1f;
					margin-bottom: 6px;
					display: flex;
					align-items: center;
					gap: 8px;
				">
					<span style="
						display: inline-block;
						width: 6px;
						height: 6px;
						background: #0071e3;
						border-radius: 50%;
						animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
					"></span>
					${pageName}
				</div>
				${keyword ? `
					<div style="
						font-size: 12px;
						color: #86868b;
						line-height: 1.5;
						padding-left: 14px;
					">${keyword.split(',').slice(0, 2).join(', ')}</div>
				` : ''}
			</div>
		`);
		
		$grid.append(pageCard);
		
		// Animate in with bounce effect
		setTimeout(() => {
			pageCard.css({ 
				opacity: 1, 
				transform: 'translateY(0) scale(1)'
			});
		}, 50);
		
		// Add pulse animation if not already added
		if (!$('#buildfuly-pulse-animation').length) {
			$('head').append(`
				<style id="buildfuly-pulse-animation">
					@keyframes pulse {
						0%, 100% { opacity: 1; }
						50% { opacity: 0.3; }
					}
				</style>
			`);
		}
	}
};
