/**
 * Buildfuly Admin JavaScript
 * Handles content generation, styling, and API interactions
 */

(function($) {
	'use strict';

	const BuildfulyAdmin = {
		/**
		 * Initialize the admin interface
		 */
		init: function() {
			this.bindEvents();
			this.loadInitialData();
		},

		/**
		 * Bind event handlers
		 */
		bindEvents: function() {
			// Tabs
			$('.buildfuly-tab-button').on('click', this.switchTab.bind(this));

			// Theme Installation
			$('.buildfuly-install-theme').on('click', this.installTheme.bind(this));

			// Content Generation
			$('#buildfuly-generate-content').on('click', this.generateContent.bind(this));
			$('#buildfuly-clear-content').on('click', this.clearContent.bind(this));

			// Styling
			$('#buildfuly-generate-styles').on('click', this.generateStyles.bind(this));
			$('#buildfuly-apply-styles').on('click', this.applyStyles.bind(this));
			$('#buildfuly-reset-styles').on('click', this.resetStyles.bind(this));

			// Preview controls
			$('#buildfuly-preview-desktop, #buildfuly-preview-tablet, #buildfuly-preview-mobile').on('click', this.switchPreviewMode.bind(this));

			// Settings
			$('#buildfuly-export-data').on('click', this.exportData.bind(this));
			$('#buildfuly-import-data').on('click', this.importData.bind(this));
		},

		/**
		 * Load initial data
		 */
		loadInitialData: function() {
			// Load content if container exists
			if ($('#buildfuly-content-container').length) {
				this.loadContent();
			}

			// Load styles if in styling tab
			if ($('#buildfuly-tab-styling').length) {
				// Will load when tab is activated
			}

			// Check if we're on the settings page
			if ($('#buildfuly-history-container').length) {
				this.loadHistory();
			}
		},

		/**
		 * Switch between tabs
		 */
		switchTab: function(e) {
			e.preventDefault();
			const tab = $(e.currentTarget).data('tab');

			// Update button states
			$('.buildfuly-tab-btn').removeClass('active');
			$(e.currentTarget).addClass('active');

			// Update tab content
			$('.buildfuly-tab-content').removeClass('active');
			$('#buildfuly-tab-' + tab).addClass('active');

			// Load data for styling tab if needed
			if (tab === 'styling' && !this.stylesLoaded) {
				this.loadStyles();
				this.stylesLoaded = true;
			}
		},

		/**
		 * Install a theme
		 */
		installTheme: function(e) {
			e.preventDefault();
			const $button = $(e.currentTarget);
			const theme = $button.data('theme');
			const originalText = $button.html();

			if (!confirm('This will install and activate the ' + (theme === 'generatepress' ? 'GeneratePress' : 'Kadence') + ' theme. Continue?')) {
				return;
			}

			$button.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> Installing...');

			$.ajax({
				url: buildfulyAdmin.ajaxUrl,
				method: 'POST',
				data: {
					action: 'buildfuly_install_theme',
					theme: theme,
					nonce: buildfulyAdmin.nonce
				}
			}).done((response) => {
				if (response.success) {
					this.showMessage(response.data.message, 'success');
					// Reload the page after 2 seconds
					setTimeout(() => {
						location.reload();
					}, 2000);
				} else {
					this.showMessage(response.data.message || 'Failed to install theme', 'error');
					$button.prop('disabled', false).html(originalText);
				}
			}).fail((xhr) => {
				console.error('Theme installation error:', xhr);
				this.showMessage('An error occurred during theme installation. Check console for details.', 'error');
				$button.prop('disabled', false).html(originalText);
			});
		},
		/**
		 * Generate content
		 */
		generateContent: function(e) {
			e.preventDefault();

			const businessType = $('#business-type').val();
			const industry = $('#industry').val();

			this.showLoading();
			this.clearMessages();

			wp.apiFetch({
				path: '/buildfuly/v1/content/generate',
				method: 'POST',
				data: {
					business_type: businessType,
					industry: industry
				}
			}).then((response) => {
				this.hideLoading();
				if (response.success) {
					this.showMessage('success', response.message || 'Content generated successfully!');
					this.displayContent(response.data);
				} else {
					this.showMessage('error', response.message || 'Failed to generate content');
				}
			}).catch((error) => {
				this.hideLoading();
				this.showMessage('error', error.message || 'An error occurred');
			});
		},

		/**
		 * Load existing content
		 */
		loadContent: function() {
			wp.apiFetch({
				path: '/buildfuly/v1/content',
				method: 'GET'
			}).then((response) => {
				if (response.success && response.data) {
					this.displayContent(response.data);
				}
			}).catch((error) => {
				console.error('Error loading content:', error);
			});
		},

		/**
		 * Display content
		 */
		displayContent: function(content) {
			const container = $('#buildfuly-content-container');
			container.empty();

			if (Object.keys(content).length === 0) {
				container.html('<div class="buildfuly-card"><p style="text-align: center; color: var(--buildfuly-text-secondary);">No content generated yet. Click "Generate Content" to get started.</p></div>');
				return;
			}

			for (const [key, value] of Object.entries(content)) {
				const item = $(`
					<div class="buildfuly-content-item" data-key="${key}">
						<div class="buildfuly-content-label">${this.formatLabel(key)}</div>
						<div class="buildfuly-content-value" contenteditable="true">${value}</div>
						<div class="buildfuly-content-actions" style="margin-top: 12px; display: flex; gap: 8px;">
							<button class="button button-secondary button-small buildfuly-save-item">Save</button>
							<button class="button button-secondary button-small buildfuly-regenerate-item">Regenerate</button>
						</div>
					</div>
				`);

				// Bind save action
				item.find('.buildfuly-save-item').on('click', (e) => {
					e.preventDefault();
					const newValue = item.find('.buildfuly-content-value').text();
					this.updateContentItem(key, newValue);
				});

				// Bind regenerate action
				item.find('.buildfuly-regenerate-item').on('click', (e) => {
					e.preventDefault();
					this.regenerateComponent(key);
				});

				container.append(item);
			}
		},

		/**
		 * Update content item
		 */
		updateContentItem: function(key, value) {
			wp.apiFetch({
				path: `/buildfuly/v1/content/${key}`,
				method: 'PUT',
				data: { value: value }
			}).then((response) => {
				if (response.success) {
					this.showMessage('success', 'Content updated successfully', 2000);
				} else {
					this.showMessage('error', 'Failed to update content');
				}
			}).catch((error) => {
				this.showMessage('error', error.message || 'An error occurred');
			});
		},

		/**
		 * Regenerate component
		 */
		regenerateComponent: function(component) {
			const businessType = $('#business-type').val();

			wp.apiFetch({
				path: `/buildfuly/v1/content/component/${component}`,
				method: 'POST',
				data: { business_type: businessType }
			}).then((response) => {
				if (response.success) {
					this.showMessage('success', 'Component regenerated successfully', 2000);
					this.loadContent();
				} else {
					this.showMessage('error', 'Failed to regenerate component');
				}
			}).catch((error) => {
				this.showMessage('error', error.message || 'An error occurred');
			});
		},

		/**
		 * Clear all content
		 */
		clearContent: function(e) {
			e.preventDefault();

			if (!confirm('Are you sure you want to clear all generated content? This action cannot be undone.')) {
				return;
			}

			// Implementation would call API to clear content
			this.showMessage('info', 'Content cleared');
			$('#buildfuly-content-container').empty();
		},

		/**
		 * Generate styles
		 */
		generateStyles: function(e) {
			e.preventDefault();

			const businessType = $('#style-business-type').val();
			const component = $('#style-component').val();

			this.showMessage('info', 'Generating styles...', 0, '#buildfuly-styling-messages');

			wp.apiFetch({
				path: '/buildfuly/v1/styles/generate',
				method: 'POST',
				data: {
					business_type: businessType,
					component: component || undefined
				}
			}).then((response) => {
				if (response.success) {
					this.showMessage('success', response.message || 'Styles generated successfully', 3000, '#buildfuly-styling-messages');
					this.loadStyles();
					this.updatePreview();
				} else {
					this.showMessage('error', response.message || 'Failed to generate styles', 0, '#buildfuly-styling-messages');
				}
			}).catch((error) => {
				this.showMessage('error', error.message || 'An error occurred', 0, '#buildfuly-styling-messages');
			});
		},

		/**
		 * Load styles
		 */
		loadStyles: function() {
			wp.apiFetch({
				path: '/buildfuly/v1/styles',
				method: 'GET'
			}).then((response) => {
				if (response.success && response.data) {
					this.displayStyles(response.data);
					this.updatePreview();
				}
			}).catch((error) => {
				console.error('Error loading styles:', error);
			});
		},

		/**
		 * Display styles in editor
		 */
		displayStyles: function(styles) {
			// Colors
			const colorsContainer = $('#buildfuly-colors-editor');
			colorsContainer.empty();

			if (styles.colors && styles.colors.value) {
				for (const [key, value] of Object.entries(styles.colors.value)) {
					const colorGroup = $(`
						<div class="buildfuly-color-group">
							<div class="buildfuly-color-preview" style="background-color: ${value};" data-property="${key}"></div>
							<input type="text" class="buildfuly-input" value="${value}" data-property="${key}" placeholder="#000000" />
							<label>${this.formatLabel(key)}</label>
						</div>
					`);

					colorGroup.find('input').on('change', (e) => {
						const newValue = $(e.target).val();
						this.updateStyleProperty('colors', key, newValue);
					});

					colorsContainer.append(colorGroup);
				}
			}

			// Fonts
			const fontsContainer = $('#buildfuly-fonts-editor');
			fontsContainer.empty();

			if (styles.fonts && styles.fonts.value) {
				for (const [key, value] of Object.entries(styles.fonts.value)) {
					const fontGroup = $(`
						<div class="buildfuly-form-group" style="margin-bottom: 16px;">
							<label>${this.formatLabel(key)}</label>
							<select class="buildfuly-select" data-property="${key}">
								<option value="Inter" ${value === 'Inter' ? 'selected' : ''}>Inter</option>
								<option value="Roboto" ${value === 'Roboto' ? 'selected' : ''}>Roboto</option>
								<option value="Open Sans" ${value === 'Open Sans' ? 'selected' : ''}>Open Sans</option>
								<option value="Montserrat" ${value === 'Montserrat' ? 'selected' : ''}>Montserrat</option>
								<option value="Playfair Display" ${value === 'Playfair Display' ? 'selected' : ''}>Playfair Display</option>
								<option value="Source Sans Pro" ${value === 'Source Sans Pro' ? 'selected' : ''}>Source Sans Pro</option>
							</select>
						</div>
					`);

					fontGroup.find('select').on('change', (e) => {
						const newValue = $(e.target).val();
						this.updateStyleProperty('fonts', key, newValue);
					});

					fontsContainer.append(fontGroup);
				}
			}

			// Layout (simplified)
			const layoutContainer = $('#buildfuly-layout-editor');
			layoutContainer.empty();

			if (styles.layout && styles.layout.value) {
				for (const [key, value] of Object.entries(styles.layout.value)) {
					const layoutGroup = $(`
						<div class="buildfuly-form-group" style="margin-bottom: 16px;">
							<label>${this.formatLabel(key)}</label>
							<input type="text" class="buildfuly-input" value="${value}" data-property="${key}" />
						</div>
					`);

					layoutGroup.find('input').on('change', (e) => {
						const newValue = $(e.target).val();
						this.updateStyleProperty('layout', key, newValue);
					});

					layoutContainer.append(layoutGroup);
				}
			}
		},

		/**
		 * Update style property
		 */
		updateStyleProperty: function(component, property, value) {
			wp.apiFetch({
				path: '/buildfuly/v1/styles/property',
				method: 'PUT',
				data: {
					component: component,
					property: property,
					value: value
				}
			}).then((response) => {
				if (response.success) {
					this.updatePreview();
				}
			}).catch((error) => {
				console.error('Error updating style:', error);
			});
		},

		/**
		 * Update preview
		 */
		updatePreview: function() {
			const preview = $('#buildfuly-preview');
			// This would render a live preview of the homepage with current styles
			preview.html('<p style="text-align: center; padding: 40px; color: #86868b;">Preview will be rendered here</p>');
		},

		/**
		 * Apply styles
		 */
		applyStyles: function(e) {
			e.preventDefault();
			this.showMessage('success', 'Styles applied successfully!', 3000, '#buildfuly-styling-messages');
		},

		/**
		 * Reset styles
		 */
		resetStyles: function(e) {
			e.preventDefault();

			if (!confirm('Reset all styles to defaults?')) {
				return;
			}

			const businessType = $('#style-business-type').val();
			this.generateStyles(e);
		},

		/**
		 * Switch preview mode
		 */
		switchPreviewMode: function(e) {
			e.preventDefault();
			$('.buildfuly-preview-controls .button').removeClass('button-primary');
			$(e.currentTarget).addClass('button-primary');
		},

		/**
		 * Load theme info
		 */
		loadThemeInfo: function() {
			wp.apiFetch({
				path: '/buildfuly/v1/theme/info',
				method: 'GET'
			}).then((response) => {
				if (response.success) {
					const statusDiv = $('#buildfuly-theme-status');
					const theme = response.data.active_theme;
					const supported = response.data.supported;

					let html = `<p><strong>Active Theme:</strong> ${theme}</p>`;
					if (supported) {
						html += '<p style="color: white;">✓ Theme integration is supported and active</p>';
					} else {
						html += '<p style="color: rgba(255,255,255,0.8);">⚠ Theme integration not available. Styles will be applied via custom CSS.</p>';
					}

					statusDiv.html(html);
				}
			}).catch((error) => {
				console.error('Error loading theme info:', error);
			});
		},

		/**
		 * Load history
		 */
		loadHistory: function() {
			wp.apiFetch({
				path: '/buildfuly/v1/history?limit=20',
				method: 'GET'
			}).then((response) => {
				if (response.success && response.data) {
					this.displayHistory(response.data);
				}
			}).catch((error) => {
				console.error('Error loading history:', error);
			});
		},

		/**
		 * Display history
		 */
		displayHistory: function(history) {
			const container = $('#buildfuly-history-container');
			container.empty();

			if (history.length === 0) {
				container.html('<p style="text-align: center; color: var(--buildfuly-text-secondary); padding: 20px;">No history available</p>');
				return;
			}

			history.forEach(item => {
				const historyItem = $(`
					<div class="buildfuly-history-item">
						<div>
							<strong>${item.action_type}</strong><br>
							<small>${item.created_at}</small>
						</div>
					</div>
				`);
				container.append(historyItem);
			});
		},

		/**
		 * Export data
		 */
		exportData: function(e) {
			e.preventDefault();
			// Implementation for exporting data
			this.showMessage('info', 'Export functionality will be available soon');
		},

		/**
		 * Import data
		 */
		importData: function(e) {
			e.preventDefault();
			// Implementation for importing data
			this.showMessage('info', 'Import functionality will be available soon');
		},

		/**
		 * Show loading state
		 */
		showLoading: function() {
			$('#buildfuly-loading').fadeIn();
			$('#buildfuly-generate-content').prop('disabled', true);
		},

		/**
		 * Hide loading state
		 */
		hideLoading: function() {
			$('#buildfuly-loading').fadeOut();
			$('#buildfuly-generate-content').prop('disabled', false);
		},

		/**
		 * Show message
		 */
		showMessage: function(message, type = 'success', duration = 5000) {
			const messageEl = $(`
				<div class="buildfuly-message ${type}">
					${message}
				</div>
			`);

			$('#buildfuly-messages').append(messageEl);

			if (duration > 0) {
				setTimeout(() => {
					messageEl.fadeOut(() => messageEl.remove());
				}, duration);
			}
		},

		/**
		 * Clear messages
		 */
		clearMessages: function(container = '#buildfuly-messages') {
			$(container).empty();
		},

		/**
		 * Format label
		 */
		formatLabel: function(key) {
			return key
				.replace(/([A-Z])/g, ' $1')
				.replace(/[._-]/g, ' ')
				.replace(/^./, str => str.toUpperCase())
				.trim();
		}
	};

	// Initialize on document ready
	$(document).ready(function() {
		BuildfulyAdmin.init();
	});

})(jQuery);
