<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://build.fuly.io
 * @since      1.0.0
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for the admin area.
 *
 * @package    Buildfuly
 * @subpackage Buildfuly/admin
 * @author     Buildfuly Team
 */
class Buildfuly_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string    $plugin_name    The name of this plugin.
	 * @param    string    $version        The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		
		// Register row actions for WordPress Pages list
		add_filter( 'page_row_actions', array( $this, 'add_buildfuly_row_actions' ), 10, 2 );
		add_filter( 'post_row_actions', array( $this, 'add_buildfuly_row_actions' ), 10, 2 );
		
		// Add "Edit with Buildfuly" to admin bar
		add_action( 'admin_bar_menu', array( $this, 'add_buildfuly_admin_bar_link' ), 80 );
		
		// Register delete page handler
		add_action( 'admin_post_buildfuly_delete_page', array( $this, 'handle_delete_page' ) );
		
		// Show provisioned site notice in admin
		if ( self::is_provisioned_site() ) {
			add_action( 'admin_notices', array( $this, 'show_provisioned_site_notice' ) );
		}
		
		// Show "Build Your Site" notice if auto-build is pending
		if ( get_option( 'buildfuly_needs_auto_build', false ) ) {
			add_action( 'admin_notices', array( $this, 'show_build_site_notice' ) );
		}
	}

	/**
	 * Show notice prompting user to build their site
	 * Only shows when needs_auto_build flag is set (provisioned but not yet generated)
	 *
	 * @since    1.1.2
	 */
	public function show_build_site_notice() {
		$business_name = get_option( 'buildfuly_business_name', 'your site' );
		$builder_url = admin_url( 'admin.php?page=buildfuly' );
		?>
		<div class="notice notice-success is-dismissible" style="border-left-color: #10b981; padding: 15px;">
			<div style="display: flex; align-items: center; gap: 15px;">
				<span style="font-size: 32px;">🎉</span>
				<div style="flex: 1;">
					<strong style="font-size: 16px; display: block; margin-bottom: 5px;">
						<?php esc_html_e( 'Welcome! Your site is ready to be built.', 'buildfuly' ); ?>
					</strong>
					<p style="margin: 0; color: #666;">
						<?php 
						printf(
							/* translators: %s: business name */
							esc_html__( 'Click below to automatically generate your homepage for %s using AI.', 'buildfuly' ),
							'<strong>' . esc_html( $business_name ) . '</strong>'
						);
						?>
					</p>
				</div>
				<a href="<?php echo esc_url( $builder_url ); ?>" class="button button-primary button-hero" style="font-size: 14px;">
					<?php esc_html_e( '🚀 Build My Site Now', 'buildfuly' ); ?>
				</a>
			</div>
		</div>
		<?php
	}

	/**
	 * Check if this site was provisioned through fuly.io
	 * 
	 * Provisioned sites have pre-configured settings and API keys.
	 *
	 * @since    1.1.2
	 * @return   bool    True if site was provisioned through fuly.io
	 */
	public static function is_provisioned_site() {
		return (bool) get_option( 'buildfuly_is_provisioned', false );
	}

	/**
	 * Get provisioned site tier
	 *
	 * @since    1.1.2
	 * @return   string    Site tier (free, basic, pro)
	 */
	public static function get_provisioned_tier() {
		return get_option( 'buildfuly_tier', 'free' );
	}

	/**
	 * Get provisioned site limits based on tier
	 *
	 * @since    1.1.2
	 * @return   array    Tier limits
	 */
	public static function get_tier_limits() {
		$tier = self::get_provisioned_tier();
		$limits = array(
			'free' => array(
				'pages_total' => 3,
				'daily_requests' => 100,
				'has_branding' => true,
			),
			'basic' => array(
				'pages_total' => 10,
				'daily_requests' => 500,
				'has_branding' => false,
			),
			'pro' => array(
				'pages_total' => -1, // Unlimited
				'daily_requests' => -1, // Unlimited
				'has_branding' => false,
			),
		);
		return isset( $limits[ $tier ] ) ? $limits[ $tier ] : $limits['free'];
	}

	/**
	 * Show admin notice for provisioned sites
	 *
	 * @since    1.1.2
	 */
	public function show_provisioned_site_notice() {
		$tier = self::get_provisioned_tier();
		$limits = self::get_tier_limits();
		$provisioned_at = get_option( 'buildfuly_provisioned_at', '' );
		$dashboard_url = 'https://sites.fuly.io/dashboard?email=' . urlencode( get_option( 'admin_email', '' ) );
		
		// Only show on Buildfuly pages
		$screen = get_current_screen();
		if ( ! $screen || strpos( $screen->id, 'buildfuly' ) === false ) {
			return;
		}
		
		$page_limit_text = $limits['pages_total'] === -1 ? 'Unlimited' : $limits['pages_total'];
		?>
		<div class="notice notice-info buildfuly-provisioned-notice" style="border-left-color: #667eea; padding: 12px;">
			<div style="display: flex; align-items: center; gap: 12px;">
				<span style="font-size: 24px;">🚀</span>
				<div>
					<strong style="color: #667eea; text-transform: uppercase; font-size: 11px;">
						<?php echo esc_html( ucfirst( $tier ) ); ?> Plan Site
					</strong>
					<p style="margin: 4px 0 0 0;">
						<?php 
						printf(
							/* translators: %d: number of pages allowed */
							esc_html__( 'Your site is managed through Fuly.io. You can generate up to %s AI pages.', 'buildfuly' ),
							esc_html( $page_limit_text )
						);
						?>
						<a href="<?php echo esc_url( $dashboard_url ); ?>" target="_blank" style="margin-left: 8px;">
							<?php esc_html_e( 'Manage in Dashboard →', 'buildfuly' ); ?>
						</a>
					</p>
				</div>
				<?php if ( $tier === 'free' ) : ?>
				<a href="<?php echo esc_url( $dashboard_url ); ?>" class="button button-primary" style="margin-left: auto;">
					<?php esc_html_e( 'Upgrade Plan', 'buildfuly' ); ?>
				</a>
				<?php endif; ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Log debug messages only when WP_DEBUG is enabled.
	 *
	 * @since    1.0.0
	 * @param    string    $message    The message to log.
	 */
	private function debug_log( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			Buildfuly_Logger::debug( '[Buildfuly] ' . $message );
		}
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles( $hook ) {
		// Only load on our plugin pages
		if ( strpos( $hook, 'buildfuly' ) === false ) {
			return;
		}

		// Enqueue dashicons (WordPress icons)
		wp_enqueue_style( 'dashicons' );

		// Enqueue admin CSS
		wp_enqueue_style(
			$this->plugin_name,
			plugin_dir_url( __FILE__ ) . 'css/buildfuly-admin.css',
			array( 'dashicons' ),
			$this->version . '-' . time(),
			'all'
		);

		// Enqueue public CSS for preview (so preview matches published page)
	wp_enqueue_style(
		$this->plugin_name . '-public',
		plugin_dir_url( dirname( __FILE__ ) ) . 'public/css/buildfuly-public.css',
		array(),
		$this->version . '-' . time(),
		'all'
	);

	// Get current theme colors
	$primary = get_option( 'buildfuly_primary_color', '#00bfa5' );
	$secondary = get_option( 'buildfuly_secondary_color', '#00897b' );
	$accent = get_option( 'buildfuly_accent_color', '#7c4dff' );
	$text = get_option( 'buildfuly_text_color', '#1e293b' );
	$bg = get_option( 'buildfuly_bg_color', '#ffffff' );

	// Helper functions for color manipulation
	$hex_to_rgb = function( $hex ) {
		$hex = ltrim( $hex, '#' );
		if ( strlen( $hex ) === 3 ) {
			$hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
		}
		return array(
			hexdec( substr( $hex, 0, 2 ) ),
			hexdec( substr( $hex, 2, 2 ) ),
			hexdec( substr( $hex, 4, 2 ) )
		);
	};

	$darken_color = function( $hex, $percent ) use ( $hex_to_rgb ) {
		$rgb = $hex_to_rgb( $hex );
		$factor = 1 - ( $percent / 100 );
		return sprintf(
			'#%02x%02x%02x',
			max( 0, min( 255, round( $rgb[0] * $factor ) ) ),
			max( 0, min( 255, round( $rgb[1] * $factor ) ) ),
			max( 0, min( 255, round( $rgb[2] * $factor ) ) )
		);
	};

	$rgb = $hex_to_rgb( $primary );
	$primary_light = sprintf( 'rgba(%d, %d, %d, 0.1)', $rgb[0], $rgb[1], $rgb[2] );
	$primary_dark = $darken_color( $primary, 15 );

	// Add inline CSS with color variables and theme fonts
	$custom_css = "
		/* CSS Variables for instant color updates */
		:root {
			--buildfuly-color-primary: {$primary};
			--buildfuly-color-primary-light: {$primary_light};
			--buildfuly-color-primary-dark: {$primary_dark};
			--buildfuly-color-secondary: {$secondary};
			--buildfuly-color-accent: {$accent};
			--buildfuly-color-text: {$text};
			--buildfuly-color-text-light: #6c757d;
			--buildfuly-color-background: {$bg};
		}
		
		/* Try to inherit theme's font or use modern fallback */
		.buildfuly-sections-container,
		.buildfuly-section-content {
			font-family: var(--wp--preset--font-family--body, var(--wp--preset--font-family--base, -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Helvetica Neue', Arial, sans-serif)) !important;
		}
		
		/* Headings try to use theme heading font */
		.buildfuly-section-content h1,
		.buildfuly-section-content h2,
		.buildfuly-section-content h3,
		.buildfuly-section-content h4,
		.buildfuly-section-content h5,
		.buildfuly-section-content h6 {
			font-family: var(--wp--preset--font-family--heading, var(--wp--preset--font-family--body, var(--wp--preset--font-family--base, -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Helvetica Neue', Arial, sans-serif))) !important;
		}
	";
	wp_add_inline_style( $this->plugin_name, $custom_css );
}	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts( $hook ) {
		// Only load on our plugin pages
		if ( strpos( $hook, 'buildfuly' ) === false ) {
			return;
		}

		// Enqueue WordPress media library
		wp_enqueue_media();

		// Enqueue shared utilities (MUST be first - other scripts depend on it)
		wp_enqueue_script(
			$this->plugin_name . '-utils',
			plugin_dir_url( __FILE__ ) . 'js/buildfuly-utils.js',
			array( 'jquery' ),
			$this->version . '-' . time(),
			true
		);

		// Enqueue shared animation helpers
		wp_enqueue_script(
			$this->plugin_name . '-animations',
			plugin_dir_url( __FILE__ ) . 'js/buildfuly-animations.js',
			array( 'jquery', $this->plugin_name . '-utils' ),
			$this->version . '-' . time(),
			true
		);

		// Enqueue shared page suggestions script
		wp_enqueue_script(
			$this->plugin_name . '-page-suggestions',
			plugin_dir_url( __FILE__ ) . 'js/buildfuly-page-suggestions.js',
			array( 'jquery', $this->plugin_name . '-animations', $this->plugin_name . '-utils' ),
			$this->version . '-' . time(),
			true
		);

		// Enqueue our page builder script
		wp_enqueue_script(
			$this->plugin_name . '-page-builder',
			plugin_dir_url( __FILE__ ) . 'js/buildfuly-page-builder.js',
			array( 'jquery', $this->plugin_name . '-page-suggestions', $this->plugin_name . '-animations', $this->plugin_name . '-utils' ),
			$this->version . '-' . time(),
			true
		);

		// Enqueue SEO metadata handler
		wp_enqueue_script(
			$this->plugin_name . '-seo',
			plugin_dir_url( __FILE__ ) . 'js/buildfuly-seo.js',
			array( 'jquery', $this->plugin_name . '-page-builder' ),
			$this->version . '-' . time(),
			true
		);

		// Enqueue Blog Builder script
		wp_enqueue_script(
			$this->plugin_name . '-blog-builder',
			plugin_dir_url( __FILE__ ) . 'js/buildfuly-blog-builder.js',
			array( 'jquery', $this->plugin_name . '-page-builder', $this->plugin_name . '-utils' ),
			$this->version . '-' . time(),
			true
		);

		// Enqueue public JS for preview (so animations work in preview)
		wp_enqueue_script(
			$this->plugin_name . '-public',
			plugin_dir_url( dirname( __FILE__ ) ) . 'public/js/buildfuly-public.js',
			array( 'jquery' ),
			$this->version . '-' . time(),
			true
		);

		// Enqueue reusable AI image generator module
		wp_enqueue_script(
			$this->plugin_name . '-image-generator',
			plugin_dir_url( __FILE__ ) . 'js/buildfuly-image-generator.js',
			array( 'jquery', $this->plugin_name . '-page-builder' ),
			$this->version . '-' . time(),
			true
		);

		// Localize script with data
		wp_localize_script(
			$this->plugin_name . '-page-builder',
			'buildfulyAdmin',
			array(
				'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
				'restUrl'   => rest_url( 'buildfuly/v1' ),
				'apiUrl'    => defined( 'BUILDFULY_API_URL' ) ? BUILDFULY_API_URL : 'https://build.fuly.io',
				'licenseKey' => get_option( 'buildfuly_license_key', '' ),
				'nonce'     => wp_create_nonce( 'wp_rest' ),
				'ajaxNonce' => wp_create_nonce( 'buildfuly_generate_component' ),
				'adminNonce' => wp_create_nonce( 'buildfuly_admin_nonce' ),
				'pluginUrl' => BUILDFULY_PLUGIN_URL,
				'siteUrl'   => get_site_url(),
				'debug'     => defined( 'WP_DEBUG' ) && WP_DEBUG,
				'homepageId' => intval( get_option( 'buildfuly_homepage_id', 0 ) ),
				'themeColors' => array(
					'primary'   => get_option( 'buildfuly_primary_color', '#00bfa5' ),
					'secondary' => get_option( 'buildfuly_secondary_color', '#00897b' ),
					'accent'    => get_option( 'buildfuly_accent_color', '#7c4dff' ),
					'text'      => get_option( 'buildfuly_text_color', '#1e293b' ),
					'bg'        => get_option( 'buildfuly_bg_color', '#ffffff' ),
				),
				// Brand settings for AI image generator
				'primaryColor' => get_option( 'buildfuly_primary_color', '#667eea' ),
				'secondaryColor' => get_option( 'buildfuly_secondary_color', '#764ba2' ),
				'logoUrl' => $this->get_logo_url(),
				'ctaButtons' => json_decode( get_option( 'buildfuly_cta_buttons', '[]' ), true ),
				'businessName' => get_option( 'buildfuly_business_name', '' ),
				'businessDescription' => get_option( 'buildfuly_business_description', '' ),
				'businessLocation' => get_option( 'buildfuly_location', '' ),
				'location' => get_option( 'buildfuly_location', '' ),
				'locationRelevant' => (bool) get_option( 'buildfuly_location_relevant', false ),
				'reviewSettings' => array(
					'platform' => get_option( 'buildfuly_review_platform', 'Google' ),
					'url' => get_option( 'buildfuly_review_url', '' ),
				),
				// Provisioned site info (from fuly.io)
				'isProvisioned' => self::is_provisioned_site(),
				'provisionedTier' => self::get_provisioned_tier(),
				'tierLimits' => self::get_tier_limits(),
				'needsAutoBuild' => (bool) get_option( 'buildfuly_needs_auto_build', false ),
				'dashboardUrl' => self::is_provisioned_site() 
					? 'https://sites.fuly.io/dashboard?email=' . urlencode( get_option( 'admin_email', '' ) )
					: '',
			)
		);
	}
	
	/**
	 * Get the business logo URL
	 */
	private function get_logo_url() {
		$logo_id = get_option( 'buildfuly_business_logo_id', '' );
		return $logo_id ? wp_get_attachment_url( $logo_id ) : '';
	}

	/**
	 * Add admin menu items.
	 *
	 * @since    1.0.0
	 */
	public function add_admin_menu() {
		// Check permalink structure and show notice if not pretty
		add_action( 'admin_notices', array( $this, 'check_permalink_structure' ) );

		// Add main menu page
		add_menu_page(
			__( 'Buildfuly', 'buildfuly' ),
			__( 'Buildfuly', 'buildfuly' ),
			'manage_options',
			'buildfuly',
			array( $this, 'display_admin_page' ),
			'dashicons-layout',
			58
		);

		// Add Content & Styling submenu (unified)
		add_submenu_page(
			'buildfuly',
			__( 'Content & Styling', 'buildfuly' ),
			__( 'Content & Styling', 'buildfuly' ),
			'manage_options',
			'buildfuly',
			array( $this, 'display_admin_page' )
		);

		// Add Settings submenu
		add_submenu_page(
			'buildfuly',
			__( 'Settings', 'buildfuly' ),
			__( 'Settings', 'buildfuly' ),
			'manage_options',
			'buildfuly-settings',
			array( $this, 'display_settings_page' )
		);

		// Add Leads & Forms submenu
		add_submenu_page(
			'buildfuly',
			__( 'Leads & Forms', 'buildfuly' ),
			__( 'Leads & Forms', 'buildfuly' ),
			'manage_options',
			'buildfuly-leads',
			array( $this, 'display_leads_page' )
		);

		// Add SEO Settings submenu
		add_submenu_page(
			'buildfuly',
			__( 'SEO', 'buildfuly' ),
			__( '🔍 SEO', 'buildfuly' ),
			'manage_options',
			'buildfuly-seo',
			array( $this, 'display_seo_page' )
		);

		// Add Image Gallery submenu
		add_submenu_page(
			'buildfuly',
			__( 'Image Gallery', 'buildfuly' ),
			__( '🖼️ Images', 'buildfuly' ),
			'manage_options',
			'buildfuly-images',
			array( $this, 'display_images_page' )
		);

		// Add Logo Generator submenu
		add_submenu_page(
			'buildfuly',
			__( 'Logo Generator', 'buildfuly' ),
			__( '✨ Logo Generator', 'buildfuly' ),
			'manage_options',
			'buildfuly-logo',
			array( $this, 'display_logo_page' )
		);

		// Add Speed Optimizer submenu
		add_submenu_page(
			'buildfuly',
			__( 'Speed Optimizer', 'buildfuly' ),
			__( '⚡ Speed', 'buildfuly' ),
			'manage_options',
			'buildfuly-speed',
			array( $this, 'display_speed_page' )
		);

		// Add external link to pricing page
		add_submenu_page(
			'buildfuly',
			__( 'Upgrade', 'buildfuly' ),
			'<span style="color: #00a32a;">⭐ Upgrade</span>',
			'manage_options',
			'https://build.fuly.io/pricing',
			''
		);
		
		// Add JavaScript to make the external link work
		add_action( 'admin_footer', array( $this, 'pricing_link_script' ) );
	}
	
	/**
	 * Add JavaScript to make pricing link external.
	 *
	 * @since    1.0.5
	 */
	public function pricing_link_script() {
		?>
		<script type="text/javascript">
		jQuery(document).ready(function($) {
			$('a[href="https://build.fuly.io/pricing"]').attr('target', '_blank').attr('href', 'https://build.fuly.io/pricing');
		});
		</script>
		<?php
	}

	/**
	 * Check if pretty permalinks are enabled.
	 *
	 * @since    1.0.4
	 */
	public function check_permalink_structure() {
		$permalink_structure = get_option( 'permalink_structure' );
		
		// If no permalink structure or using default (empty), show notice
		if ( empty( $permalink_structure ) ) {
			$screen = get_current_screen();
			if ( $screen && strpos( $screen->id, 'buildfuly' ) !== false ) {
				echo '<div class="notice notice-warning is-dismissible">';
				echo '<p><strong>⚠️ Pretty Permalinks Not Enabled</strong></p>';
				echo '<p>Your pages will use URLs like <code>?page_id=123</code> instead of SEO-friendly slugs.</p>';
				echo '<p><a href="' . esc_url( admin_url( 'options-permalink.php' ) ) . '" class="button button-primary">Enable Pretty Permalinks</a> ';
				echo '<span style="margin-left: 10px;">Recommended: <strong>Post name</strong> (e.g., <code>/sample-page/</code>)</span></p>';
				echo '</div>';
			}
		}
	}

/**
 * Display the main admin page (Content & Styling).
 *
 * @since    1.0.0
 */
public function display_admin_page() {
	// Check if onboarding is completed
	$onboarding_completed = get_option( 'buildfuly_onboarding_completed', '0' );
	$business_name = get_option( 'buildfuly_business_name', '' );
	
	// Show onboarding if not completed OR no business name set
	if ( $onboarding_completed !== '1' || empty( $business_name ) ) {
		require_once plugin_dir_path( __FILE__ ) . 'partials/buildfuly-onboarding.php';
		return;
	}
	
	require_once plugin_dir_path( __FILE__ ) . 'partials/buildfuly-admin-display.php';
}	/**
	 * Display the settings page.
	 *
	 * @since    1.0.0
	 */
	public function display_settings_page() {
		require_once plugin_dir_path( __FILE__ ) . 'partials/buildfuly-settings-display.php';
	}

	/**
	 * Display the leads page.
	 *
	 * @since    1.0.6
	 */
	public function display_leads_page() {
		require_once plugin_dir_path( __FILE__ ) . 'partials/buildfuly-leads-display.php';
	}

	/**
	 * Display the SEO settings page.
	 *
	 * @since    1.0.7
	 */
	public function display_seo_page() {
		require_once plugin_dir_path( __FILE__ ) . 'partials/buildfuly-seo-settings.php';
	}

	/**
	 * Display the Image Gallery page.
	 *
	 * @since    1.1.0
	 */
	public function display_images_page() {
		require_once plugin_dir_path( __FILE__ ) . 'partials/buildfuly-images-gallery.php';
	}

	/**
	 * Display the Logo Generator admin page.
	 *
	 * @since    1.1.0
	 */
	public function display_logo_page() {
		require_once plugin_dir_path( __FILE__ ) . 'partials/buildfuly-logo-generator.php';
	}

	/**
	 * Display the Speed Optimizer admin page.
	 *
	 * @since    1.1.1
	 */
	public function display_speed_page() {
		require_once plugin_dir_path( __FILE__ ) . 'partials/buildfuly-speed-optimizer-display.php';
	}

	/**
	 * Register plugin settings.
	 *
	 * @since    1.0.1
	 */
	public function register_settings() {
		// Homepage settings
		register_setting( 'buildfuly_business_settings', 'buildfuly_homepage_id', array(
			'default' => 0,
			'sanitize_callback' => array( $this, 'sanitize_homepage_id' )
		) );
		
		// Business information settings
		register_setting( 'buildfuly_business_settings', 'buildfuly_business_name', array(
			'sanitize_callback' => array( $this, 'sanitize_and_update_business_name' )
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_business_description', array(
			'sanitize_callback' => 'sanitize_textarea_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_business_logo_id', array(
			'sanitize_callback' => 'absint'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_phone', array(
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_address', array(
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_city', array(
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_location_relevant', array(
			'default' => false,
			'sanitize_callback' => function( $value ) {
				return $value ? '1' : '';
			}
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_location', array(
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_state', array(
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_zip', array(
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_review_platform', array(
			'default' => 'yelp',
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_review_url', array(
			'default' => '',
			'sanitize_callback' => 'esc_url_raw'
		) );
		// Backward compatibility for old review_link option
		register_setting( 'buildfuly_business_settings', 'buildfuly_review_link', array(
			'default' => '',
			'sanitize_callback' => 'esc_url_raw'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_year_founded', array(
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_mission_statement', array(
			'sanitize_callback' => 'sanitize_textarea_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_services', array(
			'sanitize_callback' => 'sanitize_textarea_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_service_area', array(
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_hours', array(
			'sanitize_callback' => 'sanitize_textarea_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_certifications', array(
			'sanitize_callback' => 'sanitize_textarea_field'
		) );
		
		// CTA Buttons settings (JSON array of buttons)
		register_setting( 'buildfuly_business_settings', 'buildfuly_cta_buttons', array(
			'default' => json_encode( array(
				array( 'text' => 'Get Started', 'url' => '#contact', 'style' => 'primary' )
			) ),
			'sanitize_callback' => function( $value ) {
				// Decode JSON, validate, re-encode
				$buttons = json_decode( $value, true );
				if ( ! is_array( $buttons ) ) {
					return json_encode( array() );
				}
				// Sanitize each button
				$sanitized = array();
				foreach ( $buttons as $btn ) {
					if ( isset( $btn['text'] ) && isset( $btn['url'] ) ) {
						$url = $btn['url'];
						
						// Allow tel:, mailto:, and # anchors in addition to standard URLs
						if ( strpos( $url, 'tel:' ) === 0 || 
						     strpos( $url, 'mailto:' ) === 0 || 
						     strpos( $url, '#' ) === 0 ) {
							// These are valid, just sanitize the text
							$url = sanitize_text_field( $url );
						} else {
							// Standard URL validation
							$url = esc_url_raw( $url );
						}
						
						$sanitized[] = array(
							'text' => sanitize_text_field( $btn['text'] ),
							'url' => $url,
							'style' => isset( $btn['style'] ) ? sanitize_text_field( $btn['style'] ) : 'primary'
						);
					}
				}
				return json_encode( $sanitized );
			}
		) );
		
		// Contact form embed code
		register_setting( 'buildfuly_business_settings', 'buildfuly_contact_form_embed', array(
			'sanitize_callback' => 'wp_kses_post'
		) );
		
		// Header/Footer settings
		register_setting( 'buildfuly_business_settings', 'buildfuly_takeover_layout', array(
			'default' => '1',
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_enable_header', array(
			'default' => '1',
			'sanitize_callback' => 'sanitize_text_field'
		) );
		register_setting( 'buildfuly_business_settings', 'buildfuly_enable_footer', array(
			'default' => '1',
			'sanitize_callback' => 'sanitize_text_field'
		) );
	
		register_setting( 'buildfuly_business_settings', 'buildfuly_ai_model', array(
			'default' => 'gpt-4o',
			'sanitize_callback' => function( $value ) {
				return empty( $value ) ? 'gpt-4o' : sanitize_text_field( $value );
			}
		) );
		
		// Image uploads
		register_setting( 'buildfuly_business_settings', 'buildfuly_business_images', array(
			'sanitize_callback' => array( $this, 'sanitize_image_uploads' )
		) );

		// Speed optimizer settings
		register_setting( 'buildfuly_speed_settings', 'buildfuly_image_quality', array(
			'default' => 82,
			'sanitize_callback' => 'intval'
		) );
		register_setting( 'buildfuly_speed_settings', 'buildfuly_webp_quality', array(
			'default' => 80,
			'sanitize_callback' => 'intval'
		) );
		register_setting( 'buildfuly_speed_settings', 'buildfuly_enable_webp', array(
			'default' => true,
			'sanitize_callback' => function( $value ) { return (bool) $value; }
		) );
		register_setting( 'buildfuly_speed_settings', 'buildfuly_enable_lazy_load', array(
			'default' => true,
			'sanitize_callback' => function( $value ) { return (bool) $value; }
		) );
		register_setting( 'buildfuly_speed_settings', 'buildfuly_defer_js', array(
			'default' => true,
			'sanitize_callback' => function( $value ) { return (bool) $value; }
		) );
		register_setting( 'buildfuly_speed_settings', 'buildfuly_auto_compress', array(
			'default' => true,
			'sanitize_callback' => function( $value ) { return (bool) $value; }
		) );
		register_setting( 'buildfuly_speed_settings', 'buildfuly_optimize_css', array(
			'default' => true,
			'sanitize_callback' => function( $value ) { return (bool) $value; }
		) );
		register_setting( 'buildfuly_speed_settings', 'buildfuly_remove_query_strings', array(
			'default' => true,
			'sanitize_callback' => function( $value ) { return (bool) $value; }
		) );
		register_setting( 'buildfuly_speed_settings', 'buildfuly_enable_resource_hints', array(
			'default' => true,
			'sanitize_callback' => function( $value ) { return (bool) $value; }
		) );
	}

	/**
	 * Sanitize and set homepage ID.
	 * Also updates WordPress front page settings.
	 *
	 * @since    1.1.0
	 * @param    int    $page_id    The page ID to set as homepage.
	 * @return   int                Sanitized page ID.
	 */
	public function sanitize_homepage_id( $page_id ) {
		$page_id = absint( $page_id );
		
		if ( $page_id > 0 ) {
			// Verify it's a valid Buildfuly page
			$is_buildfuly = get_post_meta( $page_id, 'buildfuly_generated', true );
			
			if ( $is_buildfuly && get_post_status( $page_id ) === 'publish' ) {
				// Set as WordPress front page
				update_option( 'page_on_front', $page_id );
				update_option( 'show_on_front', 'page' );
				
				// Mark this page as homepage type
				update_post_meta( $page_id, 'buildfuly_is_homepage', '1' );
				
				// Clear homepage flag from other pages
				global $wpdb;
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query( $wpdb->prepare(
					"DELETE FROM {$wpdb->postmeta} WHERE meta_key = 'buildfuly_is_homepage' AND post_id != %d",
					$page_id
				) );
			}
		} else {
			// If homepage is being cleared, reset to default blog posts
			update_option( 'show_on_front', 'posts' );
			delete_option( 'page_on_front' );
			
			// Clear all homepage flags
			global $wpdb;
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->delete( $wpdb->postmeta, array( 'meta_key' => 'buildfuly_is_homepage' ) );
		}
		
		return $page_id;
	}
	
	/**
	 * Sanitize uploaded images array.
	 *
	 * @since    1.0.2
	 */
	public function sanitize_image_uploads( $images ) {
		if ( ! is_array( $images ) ) {
			return array();
		}
		
		return array_map( 'absint', $images );
	}

	/**
	 * Check if a compatible theme is installed and active.
	 *
	 * @since    1.0.0
	 * @return   array    Array with 'active' boolean and 'theme' name
	 */
	public function check_theme_compatibility() {
		$current_theme = wp_get_theme();
		$theme_name = $current_theme->get( 'Name' );
		$theme_template = $current_theme->get_template();

		$compatible_themes = array( 'GeneratePress', 'Kadence' );
		
		return array(
			'active' => in_array( $theme_name, $compatible_themes ) || in_array( $theme_template, array( 'generatepress', 'kadence' ) ),
			'theme'  => $theme_name,
			'template' => $theme_template,
		);
	}

	/**
	 * AJAX handler to install and activate a theme.
	 *
	 * @since    1.0.0
	 */
	public function ajax_install_theme() {
		// Verify nonce
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce should not be sanitized
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly-admin' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
		}

		if ( ! current_user_can( 'install_themes' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to install themes.', 'buildfuly' ) ) );
		}

		$theme_slug = isset( $_POST['theme'] ) ? sanitize_text_field( wp_unslash( $_POST['theme'] ) ) : '';
		
		if ( ! in_array( $theme_slug, array( 'generatepress', 'kadence' ) ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid theme selection.', 'buildfuly' ) ) );
		}

		// Include necessary WordPress files
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/misc.php';
		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/theme.php';

		// Create upgrader skin that suppresses output
		$skin = new WP_Ajax_Upgrader_Skin();
		$upgrader = new Theme_Upgrader( $skin );
		
		// Get theme info from WordPress.org API
		$api = themes_api( 'theme_information', array(
			'slug' => $theme_slug,
			'fields' => array( 'sections' => false, 'tags' => false )
		) );

		if ( is_wp_error( $api ) ) {
			wp_send_json_error( array( 'message' => $api->get_error_message() ) );
		}

		// Install the theme
		$result = $upgrader->install( $api->download_link );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		if ( $result === false ) {
			wp_send_json_error( array( 'message' => __( 'Theme installation failed.', 'buildfuly' ) ) );
		}

		// Activate the theme
	switch_theme( $theme_slug );

	wp_send_json_success( array(
		/* translators: %s: Theme name (e.g., GeneratePress, Kadence) */
		'message' => sprintf( __( '%s has been installed and activated successfully!', 'buildfuly' ), ucfirst( $theme_slug ) )
	) );
}	/**
	 * AJAX handler to save theme colors to WordPress options.
	 *
	 * @since    1.0.0
	 */
	public function ajax_save_theme_colors() {
		// Verify nonce
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce should not be sanitized
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly-admin' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
		}

		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to save colors.', 'buildfuly' ) ) );
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Complex JSON array sanitized below
		$colors = isset( $_POST['colors'] ) ? json_decode( stripslashes( wp_unslash( $_POST['colors'] ) ), true ) : array();
		
		if ( empty( $colors ) ) {
			wp_send_json_error( array( 'message' => __( 'No colors provided.', 'buildfuly' ) ) );
		}

		// Save each color to individual WordPress options
		if ( isset( $colors['primary'] ) ) {
			update_option( 'buildfuly_primary_color', sanitize_hex_color( $colors['primary'] ) );
		}
		if ( isset( $colors['secondary'] ) ) {
			update_option( 'buildfuly_secondary_color', sanitize_hex_color( $colors['secondary'] ) );
		}
		if ( isset( $colors['accent'] ) ) {
			update_option( 'buildfuly_accent_color', sanitize_hex_color( $colors['accent'] ) );
		}
		if ( isset( $colors['text'] ) ) {
			update_option( 'buildfuly_text_color', sanitize_hex_color( $colors['text'] ) );
		}
		if ( isset( $colors['bg'] ) ) {
			update_option( 'buildfuly_bg_color', sanitize_hex_color( $colors['bg'] ) );
		}

		// Also save as single object for backwards compatibility
		update_option( 'buildfuly_theme_colors', $colors );

		wp_send_json_success( array(
			'message' => __( 'Theme colors saved successfully!', 'buildfuly' )
		) );
	}

	/**
	 * AJAX handler to save colors instantly (without regenerating content).
	 *
	 * @since    1.0.4
	 */
	public function ajax_save_colors() {
		// Verify nonce
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce should not be sanitized
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'wp_rest' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to save colors.', 'buildfuly' ) ) );
		}

		// Save each color to WordPress options
		if ( isset( $_POST['primary'] ) ) {
			update_option( 'buildfuly_primary_color', sanitize_hex_color( wp_unslash( $_POST['primary'] ) ) );
		}
		if ( isset( $_POST['secondary'] ) ) {
			update_option( 'buildfuly_secondary_color', sanitize_hex_color( wp_unslash( $_POST['secondary'] ) ) );
		}
		if ( isset( $_POST['accent'] ) ) {
			update_option( 'buildfuly_accent_color', sanitize_hex_color( wp_unslash( $_POST['accent'] ) ) );
		}
		if ( isset( $_POST['text'] ) ) {
			update_option( 'buildfuly_text_color', sanitize_hex_color( wp_unslash( $_POST['text'] ) ) );
		}
		if ( isset( $_POST['text_light'] ) ) {
			update_option( 'buildfuly_text_light_color', sanitize_hex_color( wp_unslash( $_POST['text_light'] ) ) );
		}
		if ( isset( $_POST['bg'] ) ) {
			update_option( 'buildfuly_bg_color', sanitize_hex_color( wp_unslash( $_POST['bg'] ) ) );
		}		wp_send_json_success( array(
			'message' => __( 'Colors updated!', 'buildfuly' )
		) );
	}

	/**
	 * AJAX handler to publish a page to WordPress.
	 *
	 * @since    1.0.0
	 */
	public function ajax_publish_page() {
		// Verify nonce
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce should not be sanitized
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly-admin' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
		}

		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to publish pages.', 'buildfuly' ) ) );
		}

		$page_title = isset( $_POST['page_title'] ) ? sanitize_text_field( wp_unslash( $_POST['page_title'] ) ) : '';
		// Don't use wp_kses_post as it applies wpautop - just stripslashes for our pre-formatted HTML
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- HTML content, sanitized with wp_kses below
		$page_content = isset( $_POST['page_content'] ) ? stripslashes( wp_unslash( $_POST['page_content'] ) ) : '';
		$page_keywords = isset( $_POST['page_keywords'] ) ? sanitize_text_field( wp_unslash( $_POST['page_keywords'] ) ) : '';
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- JSON array, decoded and sanitized below
		$page_colors = isset( $_POST['page_colors'] ) ? json_decode( stripslashes( wp_unslash( $_POST['page_colors'] ) ), true ) : array();
		$wordpress_id = isset( $_POST['wordpress_id'] ) ? intval( $_POST['wordpress_id'] ) : 0;

		if ( empty( $page_title ) || empty( $page_content ) ) {
			wp_send_json_error( array( 'message' => __( 'Page title and content are required.', 'buildfuly' ) ) );
		}

		// Allow style tags, script tags, and inline styles for our generated content
		add_filter( 'wp_kses_allowed_html', function( $allowed, $context ) {
			if ( $context === 'post' ) {
				$allowed['style'] = array();
				$allowed['script'] = array(
					'type' => true,
					'src' => true,
					'async' => true,
					'defer' => true,
				);
				// Allow style attribute on all elements
				foreach ( $allowed as $tag => $attrs ) {
					$allowed[$tag]['style'] = true;
				}
			}
			return $allowed;
		}, 10, 2 );

		// Use wp_kses without wpautop by temporarily removing the filter
		$has_wpautop = has_filter( 'content_save_pre', 'wpautop' );
		if ( $has_wpautop ) {
			remove_filter( 'content_save_pre', 'wpautop' );
		}
		
		$page_content = wp_kses( $page_content, wp_kses_allowed_html( 'post' ) );
		
		// Restore wpautop filter if it was there
		if ( $has_wpautop ) {
			add_filter( 'content_save_pre', 'wpautop' );
		}

		// Generate comprehensive SEO metadata
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-seo.php';
		$seo = new Buildfuly_SEO();
		
		// Extract primary keyword and validate
		$primary_keyword = trim( explode( ',', $page_keywords )[0] );
		if ( empty( $primary_keyword ) ) {
			// Use page title as fallback if no keywords provided
			$primary_keyword = $page_title;
		}
		
		// Check if SEO metadata already exists for this page
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_seo_pages';
		$existing_seo = null;
		
		if ( $wordpress_id && get_post( $wordpress_id ) ) {
			$existing_seo = $wpdb->get_row( $wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- SEO metadata read during page publish, table name is prefixed and safe
				"SELECT * FROM $table_name WHERE page_id = %d",
				$wordpress_id
			), ARRAY_A );
		}
		
		// If existing SEO metadata exists, use it; otherwise generate new
		if ( $existing_seo && ! empty( $existing_seo['meta_title'] ) ) {
			// Use existing SEO metadata
			$seo_metadata = array(
				'page_title'        => $existing_seo['meta_title'],
				'meta_description'  => $existing_seo['meta_description'],
				'focus_keyphrase'   => ! empty( $existing_seo['focus_keyphrase'] ) ? $existing_seo['focus_keyphrase'] : trim( $primary_keyword ),
				'slug'              => $existing_seo['slug'],
				'keyword_density'   => '', // Will be recalculated
				'word_count'        => 0,  // Will be recalculated
			);
			
		// Recalculate word count and keyword density with new content
		// Exclude content within seo-ignore comments
		$content_for_seo = preg_replace('/<!--\s*seo-ignore-start\s*-->.*?<!--\s*seo-ignore-end\s*-->/s', '', $page_content);
		$text_content = wp_strip_all_tags( $content_for_seo );
		$words = str_word_count( $text_content );
		$seo_metadata['word_count'] = $words;			// Use existing focus keyphrase or fall back to primary keyword
			$focus_keyphrase = ! empty( $seo_metadata['focus_keyphrase'] ) ? $seo_metadata['focus_keyphrase'] : trim( $primary_keyword );
			if ( ! empty( $focus_keyphrase ) ) {
				$keyword_count = substr_count( strtolower( $text_content ), strtolower( $focus_keyphrase ) );
				$seo_metadata['keyword_density'] = $words > 0 ? round( ( $keyword_count / $words ) * 100, 2 ) : 0;
			} else {
				$seo_metadata['keyword_density'] = 0;
			}
			
			$page_title = $existing_seo['meta_title']; // Use existing SEO title
			
			Buildfuly_Logger::debug( 'BuildfulyAI: Using existing SEO metadata for page ID ' . $wordpress_id );
		} else {
			// Generate new SEO metadata only if none exists
			$seo_metadata = $seo->generate_seo_metadata( trim( $primary_keyword ), $page_content );
			
			// Use AI-generated page title if available
			if ( ! empty( $seo_metadata['page_title'] ) ) {
				$page_title = $seo_metadata['page_title'];
			}
			
			Buildfuly_Logger::debug( 'BuildfulyAI: Generated new SEO metadata - Title: ' . $seo_metadata['page_title'] );
		}

		// Store the raw content temporarily
		$raw_content = $page_content;
		
		// Add filter to prevent content modification during insert
		add_filter( 'wp_insert_post_data', function( $data ) use ( $raw_content ) {
			// Only modify if this is our Buildfuly content
			if ( $data['post_content'] === $raw_content || strpos( $data['post_content'], 'buildfuly-' ) !== false ) {
				// Restore the exact original content without any WordPress processing
				$data['post_content'] = $raw_content;
			}
			return $data;
		}, 99 );
		
		// Disable all content filters before inserting
		remove_filter( 'content_save_pre', 'wp_filter_post_kses' );
		remove_filter( 'content_save_pre', 'wp_targeted_link_rel' );
		remove_filter( 'content_save_pre', 'balanceTags', 50 );
		kses_remove_filters();
		
		// Determine page type based on title/keywords
		$page_type = 'other';
		$title_lower = strtolower( $page_title );
		$keywords_lower = strtolower( $page_keywords );
		
		// Check for homepage
		if ( $title_lower === 'home' || stripos( $primary_keyword, 'home' ) !== false ) {
			$page_type = 'home';
		}
		// Check for contact/about pages
		elseif ( strpos( $title_lower, 'contact' ) !== false || strpos( $title_lower, 'get in touch' ) !== false ) {
			$page_type = 'contact';
		}
		elseif ( strpos( $title_lower, 'about' ) !== false || strpos( $title_lower, 'our story' ) !== false ) {
			$page_type = 'about';
		}
		// Check for location pages
		elseif ( strpos( $keywords_lower, 'near me' ) !== false || strpos( $keywords_lower, 'location' ) !== false ) {
			$page_type = 'location';
		}
		// Default to service page for all other generated pages
		else {
			$page_type = 'service';
		}
		
		// Allow override from POST data
		if ( isset( $_POST['page_type'] ) && ! empty( $_POST['page_type'] ) ) {
			$page_type = sanitize_text_field( wp_unslash( $_POST['page_type'] ) );
		}

		// Create or update the page
		$page_data = array(
			'post_title'   => $page_title,
			'post_content' => $page_content,
			'post_status'  => 'publish',
			'post_type'    => 'page',
			'meta_input'   => array(
				'buildfuly_keywords' => $page_keywords,
				'buildfuly_colors' => $page_colors,
				'buildfuly_generated' => true,
				'buildfuly_page_type' => $page_type,
			)
		);

		// If wordpress_id exists, update the existing page instead of creating new
		if ( $wordpress_id && get_post( $wordpress_id ) ) {
			$page_data['ID'] = $wordpress_id;
			$page_id = wp_update_post( $page_data );
		} else {
			$page_id = wp_insert_post( $page_data );
		}
		
		// Re-enable filters after insertion
		kses_init_filters();
		add_filter( 'content_save_pre', 'wp_filter_post_kses' );
		add_filter( 'content_save_pre', 'wp_targeted_link_rel' );
		add_filter( 'content_save_pre', 'balanceTags', 50 );

		if ( is_wp_error( $page_id ) ) {
			wp_send_json_error( array( 'message' => $page_id->get_error_message() ) );
		}

		// Update slug after page creation (WordPress needs the page to exist first)
		// Handle homepage slug specially
		$slug = isset( $seo_metadata['slug'] ) && ! empty( $seo_metadata['slug'] ) ? $seo_metadata['slug'] : sanitize_title( $page_title );
		$is_homepage = ( $page_title === 'Home' || strtolower( $page_title ) === 'home' || stripos( $primary_keyword, 'home' ) !== false );
		
		if ( $is_homepage ) {
			$slug = '/';
			// For homepage, set it as the front page and Buildfuly homepage
			update_option( 'page_on_front', $page_id );
			update_option( 'show_on_front', 'page' );
			update_option( 'buildfuly_homepage_id', $page_id );
			update_post_meta( $page_id, 'buildfuly_is_homepage', '1' );
		} else {
			wp_update_post( array(
				'ID' => $page_id,
				'post_name' => $slug,
			) );
		}

		// Apply comprehensive SEO metadata
		$seo->apply_seo_to_page( $page_id, $seo_metadata );

		// Extract FAQ items from page content for FAQ schema
		$this->extract_and_store_faq_items( $page_id, $page_content );
		
		// Set first image in content as featured image (for OG/social previews)
		$this->set_first_image_as_featured( $page_id, $page_content );

		// Store SEO metadata in database for editing
		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_seo_pages';
		
		// Ensure we have a focus keyphrase - use primary keyword as fallback
		$focus_keyphrase = ! empty( $seo_metadata['focus_keyphrase'] ) ? $seo_metadata['focus_keyphrase'] : trim( $primary_keyword );
		
	// Debug: Log what we're about to save
	Buildfuly_Logger::debug( 'BuildfulyAI: Saving to database - meta_title: ' . ( ! empty( $seo_metadata['page_title'] ) ? $seo_metadata['page_title'] : $page_title ) );
	Buildfuly_Logger::debug( 'BuildfulyAI: Saving to database - focus_keyphrase: ' . $focus_keyphrase );
	Buildfuly_Logger::debug_data( 'BuildfulyAI: Saving to database - full seo_metadata', $seo_metadata );		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SEO metadata insert/update during page publish
		$wpdb->replace(
			$table_name,
			array(
				'page_id'              => $page_id,
				'keywords'             => $page_keywords,
				'slug'                 => $slug,
				'meta_title'           => ! empty( $seo_metadata['page_title'] ) ? $seo_metadata['page_title'] : $page_title,
				'meta_description'     => ! empty( $seo_metadata['meta_description'] ) ? $seo_metadata['meta_description'] : '',
				'focus_keyphrase'      => $focus_keyphrase,
				'og_title'             => '', // Can be customized later
				'og_description'       => '', // Can be customized later
				'twitter_title'        => '', // Can be customized later
				'twitter_description'  => '', // Can be customized later
			),
			array( '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s' )
		);

		// Flush rewrite rules to ensure clean URLs work
		flush_rewrite_rules( false );

		// Prepare SEO data for response - ensure all fields have values
		$response_seo_data = array(
			'slug' => $slug,
			'title' => ! empty( $seo_metadata['page_title'] ) ? $seo_metadata['page_title'] : $page_title,
			'description' => ! empty( $seo_metadata['meta_description'] ) ? $seo_metadata['meta_description'] : '',
			'word_count' => isset( $seo_metadata['word_count'] ) ? $seo_metadata['word_count'] : 0,
			'keyword_density' => ( isset( $seo_metadata['keyword_density'] ) ? $seo_metadata['keyword_density'] : 0 ) . '%',
			'focus_keyphrase' => $focus_keyphrase,
		);
		
		// Log for debugging
		Buildfuly_Logger::debug_data( 'BuildfulyAI: Publishing SEO data', $response_seo_data );

		wp_send_json_success( array(
			'message' => __( 'Page published successfully with SEO optimization!', 'buildfuly' ),
			'page_id' => $page_id,
			'wordpress_id' => $page_id,
			'page_url' => get_permalink( $page_id ),
			'seo_data' => $response_seo_data,
		) );
	}

	/**
	 * AJAX handler to get SEO metadata for a page.
	 *
	 * @since    1.0.4
	 */
	public function ajax_get_seo_metadata() {
		check_ajax_referer( 'buildfuly_admin_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
		}

		$page_id = isset( $_POST['page_id'] ) ? intval( $_POST['page_id'] ) : 0;

		if ( ! $page_id ) {
			wp_send_json_error( array( 'message' => __( 'Page ID is required.', 'buildfuly' ) ) );
		}

	global $wpdb;
	$table_name = $wpdb->prefix . 'buildfuly_seo_pages';
	
	$seo_data = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- SEO metadata fetch for AJAX request, table name is prefixed and safe
		$wpdb->prepare( "SELECT * FROM $table_name WHERE page_id = %d", $page_id ),
		ARRAY_A
	);		if ( ! $seo_data ) {
			// Return default structure based on page title for unpublished pages
			$page_post = get_post( $page_id );
			$page_title = $page_post ? $page_post->post_title : '';
			
			// Generate default slug from title
			$default_slug = $page_title ? sanitize_title( $page_title ) : '';
			
			// Check if this is homepage
			$is_homepage = ( strtolower( $page_title ) === 'home' );
			if ( $is_homepage ) {
				$default_slug = '/';
			}
			
			wp_send_json_success( array(
				'page_id' => $page_id,
				'slug' => $default_slug,
				'meta_title' => $page_title,
				'meta_description' => '',
				'focus_keyphrase' => '',
				'keywords' => '',
				'og_title' => '',
			'og_description' => '',
			'twitter_title' => '',
			'twitter_description' => '',
			'page_url' => get_permalink( $page_id ) ?: '',
			'domain' => wp_parse_url( home_url(), PHP_URL_HOST ),
		) );
	}

	// Get page URL for preview
	$page_url = get_permalink( $page_id );
	$seo_data['page_url'] = $page_url;
	$seo_data['domain'] = wp_parse_url( home_url(), PHP_URL_HOST );		wp_send_json_success( $seo_data );
	}

	/**
	 * AJAX handler to update SEO metadata and sync to WordPress page.
	 *
	 * @since    1.0.4
	 */
	public function ajax_update_seo_metadata() {
		check_ajax_referer( 'buildfuly_admin_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
		}

		$page_id = isset( $_POST['page_id'] ) ? intval( $_POST['page_id'] ) : 0;
		$keywords = isset( $_POST['keywords'] ) ? sanitize_text_field( wp_unslash( $_POST['keywords'] ) ) : '';
		$slug = isset( $_POST['slug'] ) ? sanitize_title( wp_unslash( $_POST['slug'] ) ) : '';
		$meta_title = isset( $_POST['meta_title'] ) ? sanitize_text_field( wp_unslash( $_POST['meta_title'] ) ) : '';
		$meta_description = isset( $_POST['meta_description'] ) ? sanitize_textarea_field( wp_unslash( $_POST['meta_description'] ) ) : '';
		$focus_keyphrase = isset( $_POST['focus_keyphrase'] ) ? sanitize_text_field( wp_unslash( $_POST['focus_keyphrase'] ) ) : '';
		$og_title = isset( $_POST['og_title'] ) ? sanitize_text_field( wp_unslash( $_POST['og_title'] ) ) : '';
		$og_description = isset( $_POST['og_description'] ) ? sanitize_textarea_field( wp_unslash( $_POST['og_description'] ) ) : '';
		$twitter_title = isset( $_POST['twitter_title'] ) ? sanitize_text_field( wp_unslash( $_POST['twitter_title'] ) ) : '';
		$twitter_description = isset( $_POST['twitter_description'] ) ? sanitize_textarea_field( wp_unslash( $_POST['twitter_description'] ) ) : '';

		if ( ! $page_id ) {
			wp_send_json_error( array( 'message' => __( 'Page ID is required.', 'buildfuly' ) ) );
		}

		// Handle homepage slug
		if ( $slug === '/' || $slug === 'home' || empty( $slug ) ) {
			$slug = '/';
			$wp_slug = ''; // Empty slug for homepage in WordPress
		} else {
			$wp_slug = $slug;
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'buildfuly_seo_pages';
		
		// Use REPLACE to insert or update
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SEO metadata update via AJAX
		$result = $wpdb->replace(
			$table_name,
			array(
				'page_id'              => $page_id,
				'keywords'             => $keywords,
				'slug'                 => $slug,
				'meta_title'           => $meta_title,
				'meta_description'     => $meta_description,
				'focus_keyphrase'      => $focus_keyphrase,
				'og_title'             => $og_title,
				'og_description'       => $og_description,
				'twitter_title'        => $twitter_title,
				'twitter_description'  => $twitter_description,
			),
			array( '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s' )
		);

		if ( $result === false ) {
			wp_send_json_error( array( 'message' => __( 'Failed to save SEO metadata in database.', 'buildfuly' ) ) );
		}

		// Sync to WordPress page - update title and slug
		$update_data = array(
			'ID'         => $page_id,
			'post_title' => $meta_title, // Update page title to SEO title
		);
		
		// Only update slug if it's not homepage
		if ( $slug !== '/' ) {
			$update_data['post_name'] = $wp_slug;
		}
		
		wp_update_post( $update_data );

		// Update meta fields
		update_post_meta( $page_id, 'buildfuly_keywords', $keywords );
		update_post_meta( $page_id, '_yoast_wpseo_title', $meta_title );
		update_post_meta( $page_id, '_yoast_wpseo_metadesc', $meta_description );
		update_post_meta( $page_id, '_yoast_wpseo_focuskw', $focus_keyphrase );
		
		// Open Graph
		if ( ! empty( $og_title ) ) {
			update_post_meta( $page_id, '_yoast_wpseo_opengraph-title', $og_title );
		}
		if ( ! empty( $og_description ) ) {
			update_post_meta( $page_id, '_yoast_wpseo_opengraph-description', $og_description );
		}
		
		// Twitter
		if ( ! empty( $twitter_title ) ) {
			update_post_meta( $page_id, '_yoast_wpseo_twitter-title', $twitter_title );
		}
		if ( ! empty( $twitter_description ) ) {
			update_post_meta( $page_id, '_yoast_wpseo_twitter-description', $twitter_description );
		}

		// Flush rewrite rules if slug changed
		flush_rewrite_rules( false );

		wp_send_json_success( array(
			'message' => __( 'SEO metadata updated and synced to page successfully!', 'buildfuly' ),
			'page_url' => get_permalink( $page_id ),
		) );
	}

	/**
	 * AJAX handler to sync page content to WordPress (for already published pages).
	 *
	 * @since    1.0.5
	 */
	public function ajax_sync_page() {
		// Verify nonce
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce should not be sanitized
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly-admin' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
		}

		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to edit pages.', 'buildfuly' ) ) );
		}

		$wordpress_id = isset( $_POST['wordpress_id'] ) ? intval( $_POST['wordpress_id'] ) : 0;
		$page_title = isset( $_POST['page_title'] ) ? sanitize_text_field( wp_unslash( $_POST['page_title'] ) ) : '';
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- HTML content
		$page_content = isset( $_POST['page_content'] ) ? stripslashes( wp_unslash( $_POST['page_content'] ) ) : '';

		if ( ! $wordpress_id ) {
			wp_send_json_error( array( 'message' => __( 'WordPress page ID is required.', 'buildfuly' ) ) );
		}

		// Check if the page exists
		$existing_page = get_post( $wordpress_id );
		if ( ! $existing_page || $existing_page->post_type !== 'page' ) {
			wp_send_json_error( array( 'message' => __( 'Page not found in WordPress.', 'buildfuly' ) ) );
		}

		// Update the page
		$update_data = array(
			'ID'           => $wordpress_id,
			'post_content' => $page_content,
		);

		// Only update title if provided
		if ( ! empty( $page_title ) ) {
			$update_data['post_title'] = $page_title;
		}

		$result = wp_update_post( $update_data, true );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success( array(
			'message' => __( 'Page synced to WordPress successfully!', 'buildfuly' ),
			'page_url' => get_permalink( $wordpress_id ),
		) );
	}

	/**
	 * AJAX handler to get a page URL by ID.
	 *
	 * @since    1.0.7
	 */
	public function ajax_get_page_url() {
		// Verify nonce
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce should not be sanitized
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly-admin' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
		}

		// Accept either page_id or wordpress_id parameter
		$page_id = isset( $_POST['page_id'] ) ? intval( $_POST['page_id'] ) : 0;
		if ( ! $page_id ) {
			$page_id = isset( $_POST['wordpress_id'] ) ? intval( $_POST['wordpress_id'] ) : 0;
		}

		if ( ! $page_id ) {
			wp_send_json_error( array( 'message' => __( 'Page ID is required.', 'buildfuly' ) ) );
		}

		$page = get_post( $page_id );
		if ( ! $page || $page->post_type !== 'page' ) {
			wp_send_json_error( array( 'message' => __( 'Page not found.', 'buildfuly' ) ) );
		}

		wp_send_json_success( array(
			'url' => get_permalink( $page_id ),
		) );
	}

	/**
	 * AJAX handler to generate component code with AI.
	 *
	 * @since    1.0.4
	 */
	public function ajax_generate_component() {
		check_ajax_referer( 'buildfuly_generate_component', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
		}

		$component_name = isset( $_POST['component_name'] ) ? sanitize_text_field( wp_unslash( $_POST['component_name'] ) ) : '';
		$description = isset( $_POST['component_description'] ) ? sanitize_textarea_field( wp_unslash( $_POST['component_description'] ) ) : '';
		$category = isset( $_POST['component_category'] ) ? sanitize_text_field( wp_unslash( $_POST['component_category'] ) ) : 'content';
		$ai_content = isset( $_POST['ai_content_support'] ) && $_POST['ai_content_support'] === 'true';
		$use_colors = isset( $_POST['use_theme_colors'] ) && $_POST['use_theme_colors'] === 'true';
		$use_buttons = isset( $_POST['use_cta_buttons'] ) && $_POST['use_cta_buttons'] === 'true';
		$design_style = isset( $_POST['design_style'] ) ? sanitize_text_field( wp_unslash( $_POST['design_style'] ) ) : 'modern';

		if ( empty( $component_name ) || empty( $description ) ) {
			wp_send_json_error( array( 'message' => __( 'Component name and description are required.', 'buildfuly' ) ) );
		}

		// Validate component name format
		if ( ! preg_match( '/^[a-z0-9\-]+$/', $component_name ) ) {
			wp_send_json_error( array( 'message' => __( 'Component name must be lowercase letters, numbers, and hyphens only.', 'buildfuly' ) ) );
		}

		// Build AI prompt
		$prompt = $this->build_component_generation_prompt( $component_name, $description, $category, $ai_content, $use_colors, $use_buttons, $design_style );

		// ALL generation goes through API - no local OpenAI
		Buildfuly_Logger::debug( 'BuildfulyAI: Custom component generation not available - use API components instead' );
		
		wp_send_json_error( array( 
			'message' => __( 'Custom component generation is not available. Use the page builder to generate pages with components from the API.', 'buildfuly' ) 
		) );
	}

	/**
	 * Build the AI prompt for component generation.
	 *
	 * @since    1.0.4
	 */
	private function build_component_generation_prompt( $name, $description, $category, $ai_content, $use_colors, $use_buttons, $design_style ) {
		$prompt = "Generate a WordPress/Buildfuly component PHP file.\n\n";
		$prompt .= "COMPONENT DETAILS:\n";
		$prompt .= "- Name: {$name}\n";
		$prompt .= "- Category: {$category}\n";
		$prompt .= "- Description: {$description}\n";
		$prompt .= "- Design Style: {$design_style}\n\n";

		$prompt .= "BUILDFULY PARADIGMS (CRITICAL - MUST FOLLOW):\n\n";

		$prompt .= "1. FILE STRUCTURE:\n";
		$prompt .= "```php\n";
		$prompt .= "<?php\n";
		$prompt .= "/**\n * Component Name - Description\n */\n\n";
		if ( $use_colors ) {
			$prompt .= "\$colors = buildfuly_get_theme_colors();\n";
		}
		$prompt .= "\$primary = \$keywords[0];\n\n";

		if ( $ai_content ) {
			$prompt .= "// AI content support\n";
			$prompt .= "\$heading = 'Default Heading';\n";
			$prompt .= "if ( isset( \$ai_content ) && is_array( \$ai_content ) ) {\n";
			$prompt .= "    if ( isset( \$ai_content['heading'] ) ) \$heading = \$ai_content['heading'];\n";
			$prompt .= "    // Add more fields as needed\n";
			$prompt .= "}\n\n";
		}

		if ( $use_buttons ) {
			$prompt .= "// Get CTA buttons from settings\n";
			$prompt .= "\$cta_buttons = json_decode( get_option( 'buildfuly_cta_buttons', '[]' ), true );\n";
			$prompt .= "if ( ! empty( \$cta_buttons ) && is_array( \$cta_buttons ) ) {\n";
			$prompt .= "    \$button = \$cta_buttons[0];\n";
			$prompt .= "    \$button_text = isset( \$button['text'] ) ? \$button['text'] : 'Get Started';\n";
			$prompt .= "    \$button_url = isset( \$button['url'] ) ? \$button['url'] : '#contact';\n";
			$prompt .= "} else {\n";
			$prompt .= "    \$button_text = 'Get Started';\n";
			$prompt .= "    \$button_url = '#contact';\n";
			$prompt .= "}\n\n";
		}

		$prompt .= "return sprintf(\n";
		$prompt .= "    '<section>HTML HERE</section>',\n";
		$prompt .= "    // sprintf arguments\n";
		$prompt .= ");\n";
		$prompt .= "```\n\n";

		$prompt .= "2. STYLING RULES:\n";
		$prompt .= "- ALL CSS inline: style=\"...\"\n";
		$prompt .= "- Responsive: padding: 80-140px (vertical) 40px (horizontal)\n";
		if ( $use_colors ) {
			$prompt .= "- Colors: \$colors['primary'] and \$colors['secondary']\n";
			$prompt .= "- Gradients: linear-gradient(135deg, %s 0%%, %s 100%%) where %s are colors\n";
		}
		$prompt .= "- Shadows: box-shadow: 0 20px 60px rgba(0,0,0,0.3)\n";
		$prompt .= "- Typography: headings 48-72px, body 18-24px, line-height 1.4-1.6\n";
		$prompt .= "- Spacing: margin-bottom: 24-56px between elements\n";
		$prompt .= "- Borders: border-radius: 12-20px for modern look\n\n";

		$prompt .= "3. DESIGN STYLE - {$design_style}:\n";
		if ( $design_style === 'modern' ) {
			$prompt .= "- Use gradients, large shadows, rounded corners (16-24px)\n";
			$prompt .= "- Overlapping elements with z-index\n";
			$prompt .= "- Bright, vibrant colors with opacity variations\n";
		} elseif ( $design_style === 'minimal' ) {
			$prompt .= "- Lots of whitespace, subtle shadows (0 4px 20px rgba(0,0,0,0.08))\n";
			$prompt .= "- Simple borders, smaller border-radius (8px)\n";
			$prompt .= "- Clean typography, generous padding\n";
		} elseif ( $design_style === 'bold' ) {
			$prompt .= "- Extra large typography (headings 72-96px)\n";
			$prompt .= "- High contrast, dramatic shadows\n";
			$prompt .= "- Strong geometric shapes\n";
		} elseif ( $design_style === 'elegant' ) {
			$prompt .= "- Subtle effects, refined shadows\n";
			$prompt .= "- Sophisticated color palette with gradients\n";
			$prompt .= "- Balanced spacing, graceful animations\n";
		} elseif ( $design_style === 'playful' ) {
			$prompt .= "- Bright colors, fun shapes\n";
			$prompt .= "- Rounded elements (border-radius: 24px+)\n";
			$prompt .= "- Creative layouts, asymmetric design\n";
		}
		$prompt .= "\n";

		$prompt .= "4. SECURITY (CRITICAL):\n";
		$prompt .= "- esc_html() for text content\n";
		$prompt .= "- esc_attr() for HTML attributes\n";
		$prompt .= "- esc_url() for URLs\n\n";

		$prompt .= "5. SPRINTF USAGE:\n";
		$prompt .= "- Use sprintf() to return HTML\n";
		$prompt .= "- %s for variables in order\n";
		$prompt .= "- %% to escape % signs in CSS\n\n";

		$prompt .= "OUTPUT REQUIREMENTS:\n";
		$prompt .= "- Return ONLY the PHP code, no explanations\n";
		$prompt .= "- Start with <?php\n";
		$prompt .= "- Include proper comments\n";
		$prompt .= "- Follow the exact structure shown above\n";
		$prompt .= "- Make it production-ready, fully functional\n";
		$prompt .= "- Implement the description creatively: {$description}\n\n";

		$prompt .= "Generate the complete component file now:";

		return $prompt;
	}


	/**
	 * Generate registry code snippet.
	 *
	 * @since    1.0.4
	 */
	private function generate_registry_code( $component_name, $category ) {
		$code = "// Add to component-registry.php\n\n";
		$code .= "'{$component_name}' => array(\n";
		$code .= "    'name' => '" . ucwords( str_replace( '-', ' ', $component_name ) ) . "',\n";
		$code .= "    'category' => '{$category}',\n";
		$code .= "    'file' => '{$component_name}.php',\n";
		$code .= "    'ai_support' => true,\n";
		$code .= "    'ai_structure' => array(\n";
		$code .= "        'heading' => 'string',\n";
		$code .= "        'subheading' => 'string',\n";
		$code .= "        // Add more fields as needed\n";
		$code .= "    ),\n";
		$code .= "),";

		return $code;
	}

	/**
	 * AJAX handler to save component file.
	 *
	 * @since    1.0.4
	 */
	public function ajax_save_component() {
		check_ajax_referer( 'buildfuly_save_component', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$filename = isset( $_POST['filename'] ) ? sanitize_file_name( wp_unslash( $_POST['filename'] ) ) : '';
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Code content is validated and sanitized per field in save_custom_css_file
	$code = isset( $_POST['code'] ) ? $_POST['code'] : '';

	if ( empty( $filename ) || empty( $code ) ) {
		wp_send_json_error( array( 'message' => __( 'Filename and code are required.', 'buildfuly' ) ) );
	}		// Validate filename
		if ( ! preg_match( '/^[a-z0-9\-]+\.php$/', $filename ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid filename format.', 'buildfuly' ) ) );
		}

		// Validate code starts with <?php
		if ( strpos( trim( $code ), '<?php' ) !== 0 ) {
			wp_send_json_error( array( 'message' => __( 'Code must start with <?php', 'buildfuly' ) ) );
		}

		$components_dir = plugin_dir_path( dirname( __FILE__ ) ) . 'includes/components/';
		$filepath = $components_dir . $filename;

		// Check if file already exists
		if ( file_exists( $filepath ) ) {
			wp_send_json_error( array( 'message' => __( 'File already exists. Please choose a different name or delete the existing file first.', 'buildfuly' ) ) );
		}

		// Save file
		$result = file_put_contents( $filepath, $code );

		if ( $result === false ) {
			wp_send_json_error( array( 'message' => __( 'Failed to write file. Check directory permissions.', 'buildfuly' ) ) );
		}

		wp_send_json_success( array(
			'message' => __( 'Component saved successfully!', 'buildfuly' ),
			'filepath' => $filepath
		) );
	}

	/**
	 * AJAX handler to regenerate section HTML from AI content.
	 *
	 * @since    1.0.6
	 */
	public function ajax_regenerate_section_html() {
		check_ajax_referer( 'buildfuly_generate_component', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
		}


	$section_type = isset( $_POST['section_type'] ) ? sanitize_text_field( wp_unslash( $_POST['section_type'] ) ) : '';
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- AI content is complex HTML structure, sanitized per field when rendered
	$ai_content = isset( $_POST['ai_content'] ) ? $_POST['ai_content'] : null;
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Keywords array is sanitized per item when processed
	$keywords = isset( $_POST['keywords'] ) ? $_POST['keywords'] : '';
	$image_id = isset( $_POST['image_id'] ) ? intval( $_POST['image_id'] ) : 0;

	if ( empty( $section_type ) || empty( $ai_content ) ) {
		wp_send_json_error( array( 'message' => __( 'Section type and AI content are required.', 'buildfuly' ) ) );
	}		// Convert keywords string to array if needed
		if ( is_string( $keywords ) ) {
			$keyword_array = array_map( 'trim', explode( ',', $keywords ) );
		} else {
			$keyword_array = array( 'business' );
		}

		// Generate HTML from the AI content
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-content-generator.php';
		$generator = new Buildfuly_Content_Generator( $this->plugin_name, $this->version );
		
		$html = $generator->generate_section_html( $section_type, $keyword_array, $image_id, $ai_content );

if ( $html === false || empty( $html ) ) {
    wp_send_json_error( array( 
        /* translators: %s: Component name (e.g., hero, testimonials, pricing) */
        'message' => sprintf( 
            /* translators: %s: Component name that failed to generate */
            __( 'Failed to generate component "%s". Please check API connection and license status.', 'buildfuly' ),
            $section_type
        )
    ) );
}

wp_send_json_success( array(
	'html' => $html,
	'type' => $section_type
) );
}	/**
	 * AJAX handler to get image URL by ID.
	 *
	 * @since    1.0.6
	 */
	public function ajax_get_image_url() {
		check_ajax_referer( 'buildfuly_generate_component', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
		}

		$image_id = isset( $_POST['image_id'] ) ? intval( $_POST['image_id'] ) : 0;

		if ( $image_id <= 0 ) {
			wp_send_json_error( array( 'message' => __( 'Invalid image ID.', 'buildfuly' ) ) );
		}

		$image_url = wp_get_attachment_image_url( $image_id, 'large' );

		if ( ! $image_url ) {
			wp_send_json_error( array( 'message' => __( 'Image not found.', 'buildfuly' ) ) );
		}

		wp_send_json_success( array(
			'url' => $image_url
		) );
	}

	/**
	 * AJAX handler to suggest pages based on business description.
	 *
	 * @since    1.0.5
	 */
	public function ajax_suggest_pages() {
		// Verify nonce
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce should not be sanitized
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'wp_rest' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
		}

	$description = isset( $_POST['description'] ) ? sanitize_textarea_field( wp_unslash( $_POST['description'] ) ) : '';
	$count = isset( $_POST['count'] ) ? intval( $_POST['count'] ) : 10;
	$business_name = isset( $_POST['business_name'] ) ? sanitize_text_field( wp_unslash( $_POST['business_name'] ) ) : '';
	$address = isset( $_POST['address'] ) ? sanitize_text_field( wp_unslash( $_POST['address'] ) ) : '';
	$location = isset( $_POST['location'] ) ? sanitize_text_field( wp_unslash( $_POST['location'] ) ) : '';
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Array of page suggestions, sanitized per item when processed
	$existing_pages = isset( $_POST['existing_pages'] ) && is_array( $_POST['existing_pages'] ) ? $_POST['existing_pages'] : array();

	if ( empty( $description ) ) {
		wp_send_json_error( array( 'message' => __( 'Description is required.', 'buildfuly' ) ) );
	}

	// Try to use API first for page suggestions
	$api_url = defined( 'BUILDFULY_API_URL' ) ? trailingslashit( BUILDFULY_API_URL ) : 'https://build.fuly.io/';
	$api_url .= 'api/pages/suggest';
	
	$request_body = array(
		'description' => $description,
		'count' => $count
	);
	
	// Add optional fields if provided
	if ( ! empty( $business_name ) ) {
		$request_body['business_name'] = $business_name;
	}
	if ( ! empty( $location ) ) {
		$request_body['location'] = $location;
	}
	if ( ! empty( $existing_pages ) ) {
		$request_body['existing_pages'] = $existing_pages;
	}
	
	Buildfuly_Logger::debug( 'Buildfuly: Requesting page suggestions with data: ' . wp_json_encode( $request_body ) );
	
	// Get JWT token for authenticated request (token tracking)
	$api_token = get_option( 'buildfuly_api_token' );
	$headers = array(
		'Content-Type' => 'application/json',
	);
	
	// Add JWT if available (for token tracking)
	if ( ! empty( $api_token ) ) {
		$headers['Authorization'] = 'Bearer ' . $api_token;
		Buildfuly_Logger::debug( 'Buildfuly: Sending authenticated request with JWT for token tracking' );
	} else {
		Buildfuly_Logger::debug( 'Buildfuly: No active license - sending unauthenticated request (no token tracking)' );
	}
	
	$response = wp_remote_post( $api_url, array(
		'headers' => $headers,
		'body' => wp_json_encode( $request_body ),
		'timeout' => 30,
		'sslverify' => true,
	) );
	
	// If API call succeeded, use the result
	if ( ! is_wp_error( $response ) ) {
		$status_code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );
		
		Buildfuly_Logger::debug( 'Buildfuly: API response status: ' . $status_code );
		Buildfuly_Logger::debug( 'Buildfuly: API response body: ' . substr( $body, 0, 500 ) );
		Buildfuly_Logger::debug( 'Buildfuly: API URL was: ' . $api_url );
		
		if ( $status_code === 200 && ! empty( $data['success'] ) && ! empty( $data['pages'] ) ) {
			Buildfuly_Logger::debug( 'Buildfuly: Successfully received ' . count( $data['pages'] ) . ' pages from API' );
			wp_send_json_success( array( 'pages' => $data['pages'] ) );
			return;
		} else {
			Buildfuly_Logger::debug( 'Buildfuly: API returned error for page suggestions (code: ' . $status_code . ')' );
			if ( isset( $data['error'] ) ) {
				Buildfuly_Logger::debug( 'Buildfuly: API error message: ' . $data['error'] );
			}
			// Send error details to frontend for debugging
			$fallback_pages = array(
				array( 
					'name' => 'Debug Info', 
					'keyword' => 'API Status: ' . $status_code . ' | URL: ' . $api_url . ' | Error: ' . (isset($data['error']) ? $data['error'] : 'No pages returned')
				),
			);
		}
	} else {
		Buildfuly_Logger::debug( 'Buildfuly: API request failed: ' . $response->get_error_message() );
		// Send error to frontend
		$fallback_pages = array(
			array( 
				'name' => 'API Error', 
				'keyword' => 'WordPress HTTP Error: ' . $response->get_error_message() 
			),
		);
	}

	// If we got here, something failed - add regular fallback pages
	if ( ! isset( $fallback_pages ) ) {
		$fallback_pages = array();
	}
	
	$fallback_pages = array_merge( $fallback_pages, array(
		array( 'name' => 'Home', 'keyword' => 'main landing page showcasing ' . $description ),
		array( 'name' => 'About', 'keyword' => 'our story, mission, and values' ),
		array( 'name' => 'Contact', 'keyword' => 'get in touch with us' ),
		array( 'name' => 'Portfolio', 'keyword' => 'showcase our best work and projects' ),
		array( 'name' => 'Pricing', 'keyword' => 'pricing plans and packages' ),
		array( 'name' => 'Testimonials', 'keyword' => 'customer reviews and success stories' ),
		array( 'name' => 'FAQ', 'keyword' => 'frequently asked questions' ),
		array( 'name' => 'Blog', 'keyword' => 'latest news and insights' ),
	) );
	
	Buildfuly_Logger::debug( 'Buildfuly: Using fallback pages' );
	wp_send_json_success( array( 'pages' => array_slice( $fallback_pages, 0, $count ) ) );
}

/**
 * AJAX handler to complete onboarding.
 *
 * @since    1.0.5
 */
public function ajax_complete_onboarding() {
	// Verify nonce
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce should not be sanitized
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'wp_rest' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Business data array is sanitized per field below
	$business_data = isset( $_POST['business_data'] ) ? $_POST['business_data'] : array();
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Pages array is sanitized per item when processed
	$pages = isset( $_POST['pages'] ) ? $_POST['pages'] : array();

	// Save business settings
	if ( isset( $business_data['name'] ) ) {
		update_option( 'buildfuly_business_name', sanitize_text_field( $business_data['name'] ) );
		
		// Update site title to business name
		update_option( 'blogname', sanitize_text_field( $business_data['name'] ) );
	}
	if ( isset( $business_data['description'] ) ) {
		update_option( 'buildfuly_business_description', sanitize_textarea_field( $business_data['description'] ) );
	}
	if ( isset( $business_data['phone'] ) ) {
		update_option( 'buildfuly_phone', sanitize_text_field( $business_data['phone'] ) );
	}
	if ( isset( $business_data['address'] ) ) {
		update_option( 'buildfuly_address', sanitize_text_field( $business_data['address'] ) );
	}
	if ( isset( $business_data['location'] ) ) {
		update_option( 'buildfuly_location', sanitize_text_field( $business_data['location'] ) );
	}
	if ( isset( $business_data['location_relevant'] ) ) {
		update_option( 'buildfuly_location_relevant', (bool) $business_data['location_relevant'] );
	}
	if ( isset( $business_data['services'] ) ) {
		update_option( 'buildfuly_services', sanitize_text_field( $business_data['services'] ) );
	}

	// Save CTA buttons
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Buttons array is sanitized per field below
	$buttons = isset( $_POST['buttons'] ) ? $_POST['buttons'] : array();
	if ( ! empty( $buttons ) && is_array( $buttons ) ) {
		$sanitized_buttons = array();
		foreach ( $buttons as $button ) {
			if ( isset( $button['text'] ) && isset( $button['url'] ) ) {
				$url = $button['url'];
				
				// Allow tel:, mailto:, and # anchors in addition to standard URLs
				if ( strpos( $url, 'tel:' ) === 0 || 
				     strpos( $url, 'mailto:' ) === 0 || 
				     strpos( $url, '#' ) === 0 ) {
					$url = sanitize_text_field( $url );
				} else {
					$url = esc_url_raw( $url );
				}
				
				$sanitized_buttons[] = array(
					'text' => sanitize_text_field( $button['text'] ),
					'url' => $url,
					'style' => isset( $button['style'] ) ? sanitize_text_field( $button['style'] ) : 'primary'
				);
			}
		}
		update_option( 'buildfuly_cta_buttons', json_encode( $sanitized_buttons ) );
	}

	// Save initial pages list (will be created as empty pages with keywords)
	$sanitized_pages = array();
	if ( ! empty( $pages ) && is_array( $pages ) ) {
		foreach ( $pages as $page ) {
			if ( is_array( $page ) ) {
				$sanitized_pages[] = array(
					'name' => sanitize_text_field( $page['name'] ),
					'keyword' => isset( $page['keyword'] ) ? sanitize_textarea_field( $page['keyword'] ) : ''
				);
			} else {
				// Fallback for string format
				$sanitized_pages[] = array(
					'name' => sanitize_text_field( $page ),
					'keyword' => ''
				);
			}
		}
	}
	update_option( 'buildfuly_initial_pages', json_encode( $sanitized_pages ) );

	// Mark onboarding as complete
	update_option( 'buildfuly_onboarding_completed', '1' );

	wp_send_json_success( array( 
		'message' => __( 'Onboarding completed!', 'buildfuly' ),
		'redirect' => admin_url( 'admin.php?page=buildfuly' ),
		'pages' => $sanitized_pages // Send pages with keywords to be created in localStorage
	) );
}

/**
 * AJAX handler to reset onboarding
 */
public function ajax_reset_onboarding() {
	check_ajax_referer( 'wp_rest', 'nonce' );
	
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Unauthorized' );
		return;
	}
	
	// Clear all onboarding-related options
	delete_option( 'buildfuly_onboarding_completed' );
	delete_option( 'buildfuly_business_name' );
	delete_option( 'buildfuly_business_description' );
	delete_option( 'buildfuly_phone' );
	delete_option( 'buildfuly_address' );
	delete_option( 'buildfuly_services' );
	delete_option( 'buildfuly_cta_buttons' );
	delete_option( 'buildfuly_initial_pages' );
	
	wp_send_json_success( array( 
		'message' => __( 'Onboarding reset successfully!', 'buildfuly' )
	) );
}

/**
 * Regenerate all published Buildfuly pages with updated settings
 * Useful when review platform, business name, or other settings change
 */
public function ajax_regenerate_pages() {
	check_ajax_referer( 'wp_rest', 'nonce' );
	
	if ( ! current_user_can( 'edit_pages' ) ) {
		wp_send_json_error( 'Unauthorized' );
		return;
	}
	
	// Get all published pages with buildfuly_page_data meta
	$pages = get_posts( array(
		'post_type' => 'page',
		'post_status' => 'publish',
		'posts_per_page' => -1,
		'meta_query' => array(
			array(
				'key' => 'buildfuly_page_data',
				'compare' => 'EXISTS'
			)
		)
	) );
	
	$regenerated = 0;
	$content_generator = new Buildfuly_Content_Generator();
	
	foreach ( $pages as $page ) {
		// Get the page data
		$page_data = get_post_meta( $page->ID, 'buildfuly_page_data', true );
		
		if ( empty( $page_data ) ) {
			continue;
		}
		
		$page_data = json_decode( $page_data, true );
		
		if ( ! isset( $page_data['sections'] ) || ! is_array( $page_data['sections'] ) ) {
			continue;
		}
		
		// Regenerate each section with current settings
		$new_html = '';
		
		// Use API for ALL component generation - no local fallback
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-api-client.php';
		$api_client = new Buildfuly_API_Client();
		
		foreach ( $page_data['sections'] as $section ) {
			if ( ! isset( $section['type'] ) ) {
				continue;
			}
			
			// Get keywords for this section
			$keywords = isset( $section['keywords'] ) ? $section['keywords'] : array();
			if ( empty( $keywords ) && isset( $page_data['keywords'] ) ) {
				$keywords = $page_data['keywords'];
			}
			
			// Generate section via API ONLY
			$response = $api_client->generate_component(
				$section['type'],
				$keywords,
				isset( $section['image_id'] ) ? $section['image_id'] : null
			);
			
			if ( is_wp_error( $response ) ) {
				Buildfuly_Logger::debug( 'BuildfulyAI: Failed to generate component via API: ' . $response->get_error_message() );
				$section_html = '<div class="buildfuly-error">⚠️ API Error: ' . esc_html( $response->get_error_message() ) . '</div>';
			} else {
				$section_html = isset( $response['html'] ) ? $response['html'] : '<div class="buildfuly-error">⚠️ No content returned from API</div>';
			}
			
			$new_html .= $section_html;
		}
		
		// Update the page content
		wp_update_post( array(
			'ID' => $page->ID,
			'post_content' => $new_html
		) );
		
		$regenerated++;
	}
	

wp_send_json_success( array(
	/* translators: %d: Number of pages regenerated */
	'message' => sprintf( 
		/* translators: %d: Number of pages that were regenerated */
		__( 'Successfully regenerated %d page(s) with updated settings.', 'buildfuly' ),
		$regenerated
	),
	'pages_updated' => $regenerated
) );
}

/**
 * AJAX handler to get usage statistics
 */
public function ajax_get_usage_stats() {
	check_ajax_referer( 'buildfuly_generate_component', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-api-client.php';
	$api_client = new Buildfuly_API_Client();
	
	$usage_stats = $api_client->get_usage_stats();
	
	if ( is_wp_error( $usage_stats ) ) {
		$error_message = $usage_stats->get_error_message();
		$error_code = $usage_stats->get_error_code();
		$error_data = $usage_stats->get_error_data();
		
		Buildfuly_Logger::debug( 'BuildfulyAI: Usage stats error - Code: ' . $error_code . ', Message: ' . $error_message );
		Buildfuly_Logger::debug_data( 'BuildfulyAI: Usage stats error data', $error_data );
		
		wp_send_json_error( array( 
			'message' => $error_message . ' (Error: ' . $error_code . ')',
			'error_code' => $error_code,
			'error_data' => $error_data,
			'usage' => array(
				'remaining' => 0,
				'limit' => 5
			)
		) );
		return;
	}

	wp_send_json_success( array(
		'usage' => $usage_stats
	) );
}

/**
 * AJAX handler to get available components from API
 */
public function ajax_get_components() {
	// Use general admin nonce instead of specific component nonce
	check_ajax_referer( 'buildfuly_admin_nonce', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-api-client.php';
	$api_client = new Buildfuly_API_Client();
	
	$components = $api_client->get_components();
	
	if ( is_wp_error( $components ) ) {
		Buildfuly_Logger::debug( 'BuildfulyAI: Failed to load components from API - ' . $components->get_error_message() );
		wp_send_json_error( array( 
			'message' => $components->get_error_message()
		) );
		return;
	}

	wp_send_json_success( array(
		'components' => $components
	) );
}

/**
 * AJAX handler for license activation
 */
public function ajax_activate_license() {
	check_ajax_referer( 'buildfuly_admin_nonce', 'nonce' );
	
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
		return;
	}
	
	$license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';
	
	if ( empty( $license_key ) ) {
		wp_send_json_error( 'Please enter a license key.' );
		return;
	}
	
	require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-api-client.php';
	require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-license.php';
	
	$license_manager = new Buildfuly_License();
	$result = $license_manager->activate( $license_key );
	
	if ( is_wp_error( $result ) ) {
		wp_send_json_error( $result->get_error_message() );
		return;
	}
	
	wp_send_json_success( 'License activated successfully!' );
}

/**
 * AJAX handler for license deactivation
 */
public function ajax_deactivate_license() {
	check_ajax_referer( 'buildfuly_admin_nonce', 'nonce' );
	
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'Permission denied.' );
		return;
	}
	
	require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-api-client.php';
	require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-buildfuly-license.php';
	
	$license_manager = new Buildfuly_License();
	$result = $license_manager->deactivate();
	
	if ( is_wp_error( $result ) ) {
		wp_send_json_error( $result->get_error_message() );
		return;
	}
	
	wp_send_json_success( 'License deactivated successfully.' );
}

/**
 * AJAX handler for updating WordPress options
 * Used during onboarding to save settings without full form submission
 */
public function ajax_update_option() {
	check_ajax_referer( 'buildfuly_admin_nonce', 'nonce' );
	
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => 'Permission denied.' ) );
		return;
	}
	
	$option_name = isset( $_POST['option_name'] ) ? sanitize_text_field( wp_unslash( $_POST['option_name'] ) ) : '';
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Option value can be any type, sanitized based on option_name below
	$option_value = isset( $_POST['option_value'] ) ? $_POST['option_value'] : '';
	
	Buildfuly_Logger::debug( 'AJAX Update Option - Name: ' . $option_name );
	Buildfuly_Logger::debug( 'AJAX Update Option - Value type: ' . gettype( $option_value ) );
	Buildfuly_Logger::debug_data( 'AJAX Update Option - Value', $option_value );
	
	if ( empty( $option_name ) ) {
		wp_send_json_error( array( 'message' => 'Option name is required.' ) );
		return;
	}
	
	// Only allow updating specific Buildfuly options
	$allowed_options = array(
		'buildfuly_cta_buttons',
		'buildfuly_business_name',
		'buildfuly_business_description',
		'buildfuly_city',
		'buildfuly_location_relevant'
	);
	
	if ( ! in_array( $option_name, $allowed_options, true ) ) {
		wp_send_json_error( array( 'message' => 'Invalid option name.' ) );
		return;
	}
	
	// Apply the same sanitization as the register_setting callback
	if ( $option_name === 'buildfuly_cta_buttons' ) {
		// Handle both array (from AJAX with no stringify) and JSON string
		if ( is_array( $option_value ) ) {
			$buttons = $option_value;
		} else {
			$buttons = json_decode( $option_value, true );
		}
		
		if ( ! is_array( $buttons ) ) {
			wp_send_json_error( array( 
				'message' => 'Invalid button data.',
				'received' => $option_value,
				'type' => gettype( $option_value )
			) );
			return;
		}
		
		$sanitized = array();
		foreach ( $buttons as $btn ) {
			if ( isset( $btn['text'] ) && isset( $btn['url'] ) ) {
				$url = $btn['url'];
				
				// Allow tel:, mailto:, and # anchors in addition to standard URLs
				if ( strpos( $url, 'tel:' ) === 0 || 
				     strpos( $url, 'mailto:' ) === 0 || 
				     strpos( $url, '#' ) === 0 ) {
					$url = sanitize_text_field( $url );
				} else {
					$url = esc_url_raw( $url );
				}
				
				$sanitized[] = array(
					'text' => sanitize_text_field( $btn['text'] ),
					'url' => $url,
					'style' => isset( $btn['style'] ) ? sanitize_text_field( $btn['style'] ) : 'primary'
				);
			}
		}
		$option_value = json_encode( $sanitized );
		
		Buildfuly_Logger::debug( 'Saving buttons to database: ' . $option_value );
	} else {
		$option_value = sanitize_text_field( $option_value );
	}
	
	// Update the option
	$updated = update_option( $option_name, $option_value );
	
	Buildfuly_Logger::debug( 'Update result for ' . $option_name . ': ' . ( $updated ? 'true' : 'false' ) );
	Buildfuly_Logger::debug( 'Current value in DB: ' . get_option( $option_name, 'NOT SET' ) );
	
	if ( $updated ) {
		wp_send_json_success( array( 
			'message' => 'Option updated successfully.',
			'saved_value' => $option_value,
			'db_value' => get_option( $option_name )
		) );
	} else {
		wp_send_json_success( array( 
			'message' => 'Option value unchanged.',
			'current_value' => get_option( $option_name )
		) );
	}
}

/**
 * AJAX handler for setting a page as the homepage
 *
 * @since 1.1.0
 */
public function ajax_set_homepage() {
	check_ajax_referer( 'buildfuly_admin_nonce', 'nonce' );
	
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => 'Permission denied.' ) );
		return;
	}
	
	$page_id = isset( $_POST['page_id'] ) ? absint( $_POST['page_id'] ) : 0;
	
	if ( ! $page_id ) {
		wp_send_json_error( array( 'message' => 'Page ID is required.' ) );
		return;
	}
	
	// Verify it's a valid published page
	$page = get_post( $page_id );
	if ( ! $page || $page->post_type !== 'page' || $page->post_status !== 'publish' ) {
		wp_send_json_error( array( 'message' => 'Invalid page. Please publish the page first.' ) );
		return;
	}
	
	// Verify it's a Buildfuly-generated page
	$is_buildfuly = get_post_meta( $page_id, 'buildfuly_generated', true );
	if ( ! $is_buildfuly ) {
		wp_send_json_error( array( 'message' => 'This page was not generated by Buildfuly.' ) );
		return;
	}
	
	// Set as WordPress front page
	update_option( 'page_on_front', $page_id );
	update_option( 'show_on_front', 'page' );
	
	// Update Buildfuly homepage option
	update_option( 'buildfuly_homepage_id', $page_id );
	
	// Mark this page as homepage
	update_post_meta( $page_id, 'buildfuly_is_homepage', '1' );
	
	// Clear homepage flag from other pages
	global $wpdb;
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->query( $wpdb->prepare(
		"DELETE FROM {$wpdb->postmeta} WHERE meta_key = 'buildfuly_is_homepage' AND post_id != %d",
		$page_id
	) );
	
	// Update SEO slug to '/' for homepage
	$table_name = $wpdb->prefix . 'buildfuly_seo_pages';
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->update(
		$table_name,
		array( 'slug' => '/' ),
		array( 'page_id' => $page_id ),
		array( '%s' ),
		array( '%d' )
	);
	
	// Flush rewrite rules
	flush_rewrite_rules( false );
	
	wp_send_json_success( array(
		'message' => 'Homepage updated successfully!',
		'page_id' => $page_id,
		'page_title' => $page->post_title,
		'page_url' => home_url( '/' )
	) );
}

/**
 * AJAX handler to sync WordPress page IDs for Buildfuly pages.
 * Returns a map of page titles to their WordPress IDs.
 *
 * @since 1.0.5
 */
public function ajax_sync_page_ids() {
	check_ajax_referer( 'buildfuly_admin_nonce', 'nonce' );

	if ( ! current_user_can( 'edit_posts' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	// Get all Buildfuly-generated pages
	$pages = get_posts( array(
		'post_type'      => 'page',
		'post_status'    => 'publish',
		'posts_per_page' => -1,
		'meta_key'       => 'buildfuly_generated',
		'meta_value'     => '1',
	) );

	$page_map = array();
	foreach ( $pages as $page ) {
		// Map by title (lowercase for matching)
		$page_map[ strtolower( $page->post_title ) ] = array(
			'wordpress_id' => $page->ID,
			'title'        => $page->post_title,
			'slug'         => $page->post_name,
			'url'          => get_permalink( $page->ID ),
		);
	}

	wp_send_json_success( array(
		'pages' => $page_map,
		'count' => count( $page_map ),
	) );
}

/**
 * AJAX handler to upload an image from a URL to the WordPress media library.
 *
 * @since    1.0.8
 */
public function ajax_upload_image_from_url() {
	error_log( 'ajax_upload_image_from_url called' );
	error_log( 'POST keys: ' . print_r( array_keys( $_POST ), true ) );
	
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly-admin' ) ) {
		error_log( 'Nonce check failed' );
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}
	
	error_log( 'Nonce verified OK' );

	if ( ! current_user_can( 'upload_files' ) ) {
		error_log( 'User cannot upload files' );
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$image_url = isset( $_POST['image_url'] ) ? wp_unslash( $_POST['image_url'] ) : '';
	$title = isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '';
	$alt_text = isset( $_POST['alt_text'] ) ? sanitize_text_field( wp_unslash( $_POST['alt_text'] ) ) : '';
	$is_ai_generated = isset( $_POST['is_ai_generated'] ) && $_POST['is_ai_generated'] === 'true';
	
	error_log( 'Image URL length: ' . strlen( $image_url ) );
	error_log( 'Image URL prefix: ' . substr( $image_url, 0, 50 ) );
	
	// Auto-generate alt text from title if not provided (important for SEO)
	if ( empty( $alt_text ) && ! empty( $title ) ) {
		$alt_text = $title;
	}

	if ( empty( $image_url ) ) {
		error_log( 'Image URL is empty' );
		wp_send_json_error( array( 'message' => __( 'Image URL is required.', 'buildfuly' ) ) );
	}

	require_once ABSPATH . 'wp-admin/includes/file.php';
	require_once ABSPATH . 'wp-admin/includes/media.php';
	require_once ABSPATH . 'wp-admin/includes/image.php';

	// Check if it's a base64 data URL
	if ( strpos( $image_url, 'data:image/' ) === 0 ) {
		// Parse data URL: data:image/png;base64,xxxxx
		$matches = array();
		if ( ! preg_match( '/^data:image\/(\w+);base64,(.+)$/', $image_url, $matches ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid image data URL format.', 'buildfuly' ) ) );
		}
		
		$extension = $matches[1];
		$base64_data = $matches[2];
		$image_data = base64_decode( $base64_data );
		
		if ( $image_data === false ) {
			wp_send_json_error( array( 'message' => __( 'Failed to decode base64 image.', 'buildfuly' ) ) );
		}
		
		// Create temp file
		$tmp_file = wp_tempnam( 'buildfuly_' );
		file_put_contents( $tmp_file, $image_data );
		
		$file_array = array(
			'name'     => sanitize_file_name( ( $title ? $title : 'buildfuly-image' ) . '-' . time() . '.' . $extension ),
			'tmp_name' => $tmp_file,
		);
	} else {
		// Regular URL - download it
		$image_url = esc_url_raw( $image_url );
		$tmp_file = download_url( $image_url );

		if ( is_wp_error( $tmp_file ) ) {
			wp_send_json_error( array( 'message' => __( 'Failed to download image: ', 'buildfuly' ) . $tmp_file->get_error_message() ) );
		}

		$file_array = array(
			'name'     => sanitize_file_name( ( $title ? $title : 'buildfuly-image' ) . '-' . time() . '.png' ),
			'tmp_name' => $tmp_file,
		);
	}

	// Upload to media library
	$attachment_id = media_handle_sideload( $file_array, 0, $title );

	// Clean up temp file
	if ( file_exists( $tmp_file ) ) {
		wp_delete_file( $tmp_file );
	}

	if ( is_wp_error( $attachment_id ) ) {
		wp_send_json_error( array( 'message' => __( 'Failed to upload image: ', 'buildfuly' ) . $attachment_id->get_error_message() ) );
	}

	// Set alt text
	if ( $alt_text ) {
		update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt_text );
	}

	// Mark as Buildfuly generated
	update_post_meta( $attachment_id, '_buildfuly_generated', '1' );
	if ( $is_ai_generated ) {
		update_post_meta( $attachment_id, '_buildfuly_ai_generated', '1' );
	}

	// Get attachment info
	$attachment_url = wp_get_attachment_url( $attachment_id );
	$attachment_thumb = wp_get_attachment_image_src( $attachment_id, 'thumbnail' );

	wp_send_json_success( array(
		'attachment_id' => $attachment_id,
		'url'           => $attachment_url,
		'thumbnail'     => $attachment_thumb ? $attachment_thumb[0] : $attachment_url,
		'title'         => get_the_title( $attachment_id ),
		'alt'           => $alt_text,
	) );
}

/**
 * AJAX handler to get gallery images (Buildfuly generated).
 *
 * @since    1.0.8
 */
public function ajax_get_gallery_images() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly-admin' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	$page = isset( $_POST['page'] ) ? intval( $_POST['page'] ) : 1;
	$per_page = isset( $_POST['per_page'] ) ? intval( $_POST['per_page'] ) : 20;
	$filter = isset( $_POST['filter'] ) ? sanitize_text_field( wp_unslash( $_POST['filter'] ) ) : 'all';

	$meta_query = array(
		'relation' => 'OR',
		array(
			'key'     => '_buildfuly_generated',
			'value'   => '1',
			'compare' => '=',
		),
		array(
			'key'     => '_buildfuly_ai_generated',
			'value'   => '1',
			'compare' => '=',
		),
	);

	// Filter by type
	if ( $filter === 'ai' ) {
		$meta_query = array(
			array(
				'key'     => '_buildfuly_ai_generated',
				'value'   => '1',
				'compare' => '=',
			),
		);
	} elseif ( $filter === 'uploaded' ) {
		$meta_query = array(
			array(
				'key'     => '_buildfuly_generated',
				'value'   => '1',
				'compare' => '=',
			),
			array(
				'key'     => '_buildfuly_ai_generated',
				'compare' => 'NOT EXISTS',
			),
		);
	}

	$query_args = array(
		'post_type'      => 'attachment',
		'post_status'    => 'inherit',
		'post_mime_type' => 'image',
		'posts_per_page' => $per_page,
		'paged'          => $page,
		'meta_query'     => $meta_query,
		'orderby'        => 'date',
		'order'          => 'DESC',
	);

	$query = new WP_Query( $query_args );
	$images = array();

	foreach ( $query->posts as $attachment ) {
		$images[] = array(
			'id'           => $attachment->ID,
			'url'          => wp_get_attachment_url( $attachment->ID ),
			'thumbnail'    => wp_get_attachment_image_src( $attachment->ID, 'thumbnail' )[0],
			'medium'       => wp_get_attachment_image_src( $attachment->ID, 'medium' )[0],
			'title'        => $attachment->post_title,
			'alt'          => get_post_meta( $attachment->ID, '_wp_attachment_image_alt', true ),
			'date'         => $attachment->post_date,
			'ai_generated' => get_post_meta( $attachment->ID, '_buildfuly_ai_generated', true ) === '1',
		);
	}

	wp_send_json_success( array(
		'images'      => $images,
		'total'       => $query->found_posts,
		'pages'       => $query->max_num_pages,
		'current_page' => $page,
	) );
}

/**
 * AJAX handler to delete a gallery image.
 *
 * @since    1.0.8
 */
public function ajax_delete_gallery_image() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly-admin' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'delete_posts' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$attachment_id = isset( $_POST['attachment_id'] ) ? intval( $_POST['attachment_id'] ) : 0;

	if ( ! $attachment_id ) {
		wp_send_json_error( array( 'message' => __( 'Attachment ID is required.', 'buildfuly' ) ) );
	}

	// Verify it's a Buildfuly image
	$is_buildfuly = get_post_meta( $attachment_id, '_buildfuly_generated', true ) === '1' ||
					get_post_meta( $attachment_id, '_buildfuly_ai_generated', true ) === '1';

	if ( ! $is_buildfuly ) {
		wp_send_json_error( array( 'message' => __( 'This image was not generated by Buildfuly.', 'buildfuly' ) ) );
	}

	// Delete the attachment
	$result = wp_delete_attachment( $attachment_id, true );

	if ( $result === false ) {
		wp_send_json_error( array( 'message' => __( 'Failed to delete image.', 'buildfuly' ) ) );
	}

	wp_send_json_success( array(
		'message' => __( 'Image deleted successfully.', 'buildfuly' ),
	) );
}

/**
 * AJAX handler to save auto-generate images preference.
 *
 * @since    1.0.8
 */
public function ajax_save_auto_generate_images() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly-admin' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$enabled = isset( $_POST['enabled'] ) && $_POST['enabled'] === '1';
	update_option( 'buildfuly_auto_generate_images', $enabled );

	wp_send_json_success( array(
		'message' => __( 'Preference saved.', 'buildfuly' ),
		'enabled' => $enabled,
	) );
}

/**
 * AJAX handler to save business logo.
 *
 * @since    1.1.0
 */
public function ajax_save_business_logo() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly_admin_nonce' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$logo_id = isset( $_POST['logo_id'] ) ? intval( $_POST['logo_id'] ) : 0;

	if ( ! $logo_id ) {
		wp_send_json_error( array( 'message' => __( 'Logo ID is required.', 'buildfuly' ) ) );
	}

	// Verify attachment exists
	$attachment = get_post( $logo_id );
	if ( ! $attachment || $attachment->post_type !== 'attachment' ) {
		wp_send_json_error( array( 'message' => __( 'Invalid attachment.', 'buildfuly' ) ) );
	}

	// Save logo ID
	update_option( 'buildfuly_business_logo_id', $logo_id );
	
	// Also set as custom logo for the theme (if theme supports it)
	if ( current_theme_supports( 'custom-logo' ) ) {
		set_theme_mod( 'custom_logo', $logo_id );
	}

	$logo_url = wp_get_attachment_url( $logo_id );

	wp_send_json_success( array(
		'message'  => __( 'Logo saved successfully.', 'buildfuly' ),
		'logo_id'  => $logo_id,
		'logo_url' => $logo_url,
	) );
}

/**
 * AJAX handler to save favicon as site icon.
 *
 * @since    1.1.0
 */
public function ajax_save_favicon() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly_admin_nonce' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$favicon_id = isset( $_POST['favicon_id'] ) ? intval( $_POST['favicon_id'] ) : 0;

	if ( ! $favicon_id ) {
		wp_send_json_error( array( 'message' => __( 'Favicon ID is required.', 'buildfuly' ) ) );
	}

	// Verify attachment exists
	$attachment = get_post( $favicon_id );
	if ( ! $attachment || $attachment->post_type !== 'attachment' ) {
		wp_send_json_error( array( 'message' => __( 'Invalid attachment.', 'buildfuly' ) ) );
	}

	// Save as WordPress site icon
	update_option( 'site_icon', $favicon_id );
	
	// Also save in Buildfuly options
	update_option( 'buildfuly_favicon_id', $favicon_id );

	$favicon_url = wp_get_attachment_url( $favicon_id );

	wp_send_json_success( array(
		'message'     => __( 'Favicon saved successfully and set as site icon.', 'buildfuly' ),
		'favicon_id'  => $favicon_id,
		'favicon_url' => $favicon_url,
	) );
}

/**
 * AJAX handler to create favicon from logo
 * Fetches logo from WordPress media library and converts to base64
 */
public function ajax_create_favicon_from_logo() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly_admin_nonce' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$logo_url = isset( $_POST['logo_url'] ) ? wp_unslash( $_POST['logo_url'] ) : '';
	$business_name = isset( $_POST['business_name'] ) ? sanitize_text_field( wp_unslash( $_POST['business_name'] ) ) : '';
	$business_type = isset( $_POST['business_type'] ) ? sanitize_text_field( wp_unslash( $_POST['business_type'] ) ) : '';

	if ( empty( $logo_url ) ) {
		wp_send_json_error( array( 'message' => __( 'Logo URL is required.', 'buildfuly' ) ) );
	}

	// Convert logo URL to base64 (handles localhost URLs)
	$image_data = null;
	$mime_type = 'image/png';

	if ( strpos( $logo_url, 'data:' ) === 0 ) {
		// Already base64
		$image_data = $logo_url;
	} else {
		// Get attachment ID from URL
		$attachment_id = attachment_url_to_postid( $logo_url );
		
		if ( $attachment_id ) {
			// Get file path from attachment ID
			$file_path = get_attached_file( $attachment_id );
			
			if ( $file_path && file_exists( $file_path ) ) {
				$image_content = file_get_contents( $file_path );
				$mime_type = get_post_mime_type( $attachment_id );
				
				if ( ! $mime_type ) {
					$mime_type = 'image/png';
				}
				
				$image_data = 'data:' . $mime_type . ';base64,' . base64_encode( $image_content );
			} else {
				wp_send_json_error( array( 'message' => __( 'Logo file not found on server.', 'buildfuly' ) ) );
			}
		} else {
			// Fallback: try to fetch via HTTP (for external URLs)
			$response = wp_remote_get( $logo_url, array( 'timeout' => 30 ) );
			
			if ( is_wp_error( $response ) ) {
				wp_send_json_error( array( 'message' => __( 'Failed to fetch logo: ', 'buildfuly' ) . $response->get_error_message() ) );
			}

			$image_content = wp_remote_retrieve_body( $response );
			$content_type = wp_remote_retrieve_header( $response, 'content-type' );
			
			if ( $content_type ) {
				$mime_type = $content_type;
			}

			$image_data = 'data:' . $mime_type . ';base64,' . base64_encode( $image_content );
		}
	}

	// Call Azure API
	$api_url = defined( 'BUILDFULY_API_URL' ) ? BUILDFULY_API_URL : 'https://build.fuly.io';
	$license_key = get_option( 'buildfuly_license_key', '' );

	$api_response = wp_remote_post( $api_url . '/api/logo/create-favicon', array(
		'timeout' => 60,
		'headers' => array(
			'Content-Type' => 'application/json',
			'Authorization' => 'Bearer ' . $license_key,
		),
		'body' => wp_json_encode( array(
			'imageUrl' => $image_data,
			'businessName' => $business_name,
			'businessType' => $business_type,
			'size' => 512,
		) ),
	) );

	if ( is_wp_error( $api_response ) ) {
		wp_send_json_error( array( 'message' => __( 'API request failed: ', 'buildfuly' ) . $api_response->get_error_message() ) );
	}

	$result = json_decode( wp_remote_retrieve_body( $api_response ), true );

	if ( ! $result || ! $result['success'] ) {
		$error_msg = isset( $result['error'] ) ? $result['error'] : __( 'Unknown error', 'buildfuly' );
		wp_send_json_error( array( 'message' => __( 'Failed to create favicon: ', 'buildfuly' ) . $error_msg ) );
	}

	wp_send_json_success( array(
		'faviconUrl' => $result['faviconUrl'],
		'size' => isset( $result['size'] ) ? $result['size'] : 512,
	) );
}

/**
 * AJAX handler to proxy an image and return as base64.
 * Used to bypass CORS restrictions for client-side canvas operations.
 *
 * @since    1.2.0
 */
public function ajax_proxy_image() {
	error_log( 'ajax_proxy_image called' );
	
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly_admin_nonce' ) ) {
		error_log( 'Proxy image: nonce failed' );
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'upload_files' ) ) {
		error_log( 'Proxy image: permission denied' );
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$image_url = isset( $_POST['image_url'] ) ? wp_unslash( $_POST['image_url'] ) : '';
	error_log( 'Proxy image URL: ' . substr( $image_url, 0, 100 ) );

	if ( empty( $image_url ) ) {
		wp_send_json_error( array( 'message' => __( 'Image URL is required.', 'buildfuly' ) ) );
	}

	// If already base64, return as-is
	if ( strpos( $image_url, 'data:' ) === 0 ) {
		error_log( 'Proxy: already base64' );
		wp_send_json_success( array( 'base64' => $image_url ) );
	}

	// Try to get from WordPress media library first
	$attachment_id = attachment_url_to_postid( $image_url );
	
	if ( $attachment_id ) {
		error_log( 'Proxy: found local attachment ID: ' . $attachment_id );
		$file_path = get_attached_file( $attachment_id );
		
		if ( $file_path && file_exists( $file_path ) ) {
			$image_content = file_get_contents( $file_path );
			$mime_type = get_post_mime_type( $attachment_id ) ?: 'image/png';
			$base64 = 'data:' . $mime_type . ';base64,' . base64_encode( $image_content );
			error_log( 'Proxy: returning local file, length: ' . strlen( $base64 ) );
			wp_send_json_success( array( 'base64' => $base64 ) );
		}
	}

	// Fetch from remote URL
	error_log( 'Proxy: fetching remote URL' );
	$response = wp_remote_get( $image_url, array( 'timeout' => 30 ) );

	if ( is_wp_error( $response ) ) {
		error_log( 'Proxy: remote fetch failed: ' . $response->get_error_message() );
		wp_send_json_error( array( 'message' => __( 'Failed to fetch image: ', 'buildfuly' ) . $response->get_error_message() ) );
	}

	$image_content = wp_remote_retrieve_body( $response );
	$content_type = wp_remote_retrieve_header( $response, 'content-type' );
	$mime_type = $content_type ?: 'image/png';
	
	// Clean mime type (remove charset if present)
	if ( strpos( $mime_type, ';' ) !== false ) {
		$mime_type = trim( explode( ';', $mime_type )[0] );
	}

	$base64 = 'data:' . $mime_type . ';base64,' . base64_encode( $image_content );
	error_log( 'Proxy: returning remote file, length: ' . strlen( $base64 ) );

	wp_send_json_success( array( 'base64' => $base64 ) );
}

/**
 * AJAX handler to save page builder data to WordPress database.
 * Stores page data as post meta (_buildfuly_page_data) instead of localStorage.
 *
 * @since    1.2.0
 */
public function ajax_save_page_data() {
	// Debug logging
	error_log( 'ajax_save_page_data called' );
	error_log( 'POST data: ' . print_r( array_keys( $_POST ), true ) );
	
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly_admin_nonce' ) ) {
		error_log( 'Nonce verification failed' );
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'edit_pages' ) ) {
		error_log( 'User lacks edit_pages capability' );
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$page_data = isset( $_POST['page_data'] ) ? $_POST['page_data'] : null;
	
	if ( ! $page_data ) {
		error_log( 'No page_data in POST' );
		wp_send_json_error( array( 'message' => __( 'Page data is required.', 'buildfuly' ) ) );
	}

	// If page_data is a string, decode it
	if ( is_string( $page_data ) ) {
		$page_data = json_decode( stripslashes( $page_data ), true );
		if ( json_last_error() !== JSON_ERROR_NONE ) {
			error_log( 'JSON decode error: ' . json_last_error_msg() );
			wp_send_json_error( array( 'message' => __( 'Invalid JSON in page data.', 'buildfuly' ) ) );
		}
	}

	if ( ! $page_data || ! isset( $page_data['id'] ) ) {
		error_log( 'Page data missing or no ID field' );
		wp_send_json_error( array( 'message' => __( 'Invalid page data.', 'buildfuly' ) ) );
	}

	$page_id = sanitize_text_field( $page_data['id'] );
	$wordpress_id = isset( $page_data['wordpress_id'] ) ? intval( $page_data['wordpress_id'] ) : 0;

	// Clean the page data before saving (remove base64 images, keep only URLs)
	$clean_page_data = $this->clean_page_data_for_storage( $page_data );

	// If page has a WordPress ID, save to that post's meta
	if ( $wordpress_id && get_post( $wordpress_id ) ) {
		update_post_meta( $wordpress_id, '_buildfuly_page_data', $clean_page_data );
		update_post_meta( $wordpress_id, '_buildfuly_page_id', $page_id );
		
		wp_send_json_success( array(
			'message' => __( 'Page data saved to WordPress.', 'buildfuly' ),
			'page_id' => $page_id,
			'wordpress_id' => $wordpress_id,
			'storage' => 'post_meta'
		) );
	} else {
		// Page not yet published - save to options table with unique key
		$option_key = 'buildfuly_draft_page_' . $page_id;
		update_option( $option_key, $clean_page_data, false ); // false = don't autoload
		
		wp_send_json_success( array(
			'message' => __( 'Draft page data saved.', 'buildfuly' ),
			'page_id' => $page_id,
			'storage' => 'option'
		) );
	}
}

/**
 * AJAX handler to load page builder data from WordPress database.
 *
 * @since    1.2.0
 */
public function ajax_load_page_data() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly_admin_nonce' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'edit_pages' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$page_id = isset( $_POST['page_id'] ) ? sanitize_text_field( $_POST['page_id'] ) : '';
	$wordpress_id = isset( $_POST['wordpress_id'] ) ? intval( $_POST['wordpress_id'] ) : 0;

	if ( ! $page_id && ! $wordpress_id ) {
		wp_send_json_error( array( 'message' => __( 'Page ID or WordPress ID is required.', 'buildfuly' ) ) );
	}

	$page_data = null;

	// Try to load from post meta first (if published)
	if ( $wordpress_id && get_post( $wordpress_id ) ) {
		$page_data = get_post_meta( $wordpress_id, '_buildfuly_page_data', true );
	}

	// If not found and we have a page_id, try options table (draft)
	if ( ! $page_data && $page_id ) {
		$option_key = 'buildfuly_draft_page_' . $page_id;
		$page_data = get_option( $option_key, null );
	}

	if ( $page_data ) {
		wp_send_json_success( array(
			'page_data' => $page_data
		) );
	} else {
		wp_send_json_error( array( 'message' => __( 'Page data not found.', 'buildfuly' ) ) );
	}
}

/**
 * AJAX handler to load all page builder pages from WordPress database.
 *
 * @since    1.2.0
 */
public function ajax_load_all_pages() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly_admin_nonce' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'edit_pages' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$all_pages = array();

	// Get all published Buildfuly pages (pages with _buildfuly_page_data meta)
	$published_pages = get_posts( array(
		'post_type'      => 'page',
		'posts_per_page' => -1,
		'post_status'    => array( 'publish', 'draft', 'pending', 'private' ),
		'meta_key'       => '_buildfuly_page_data',
		'meta_compare'   => 'EXISTS',
	) );

	foreach ( $published_pages as $post ) {
		$page_data = get_post_meta( $post->ID, '_buildfuly_page_data', true );
		if ( $page_data ) {
			// Ensure wordpress_id is set correctly
			$page_data['wordpress_id'] = $post->ID;
			$page_data['status'] = $post->post_status;
			$all_pages[] = $page_data;
		}
	}

	// Get all draft pages from options table
	global $wpdb;
	$draft_options = $wpdb->get_results(
		"SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE 'buildfuly_draft_page_%'"
	);

	foreach ( $draft_options as $option ) {
		$page_data = maybe_unserialize( $option->option_value );
		if ( $page_data && isset( $page_data['id'] ) ) {
			// Check if this draft has been published (has wordpress_id in our published pages)
			$already_published = false;
			foreach ( $all_pages as $published ) {
				if ( isset( $published['id'] ) && $published['id'] === $page_data['id'] ) {
					$already_published = true;
					break;
				}
			}
			
			// Only add draft if not already in published pages
			if ( ! $already_published ) {
				$page_data['status'] = 'draft';
				$all_pages[] = $page_data;
			}
		}
	}

	wp_send_json_success( array(
		'pages' => $all_pages,
		'count' => count( $all_pages )
	) );
}

/**
 * AJAX handler to delete page builder data from WordPress database.
 *
 * @since    1.2.0
 */
public function ajax_delete_page_data() {
	// Verify nonce
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), 'buildfuly_admin_nonce' ) ) {
		wp_send_json_error( array( 'message' => __( 'Security check failed.', 'buildfuly' ) ) );
	}

	if ( ! current_user_can( 'delete_pages' ) ) {
		wp_send_json_error( array( 'message' => __( 'Permission denied.', 'buildfuly' ) ) );
	}

	$page_id = isset( $_POST['page_id'] ) ? sanitize_text_field( $_POST['page_id'] ) : '';
	$wordpress_id = isset( $_POST['wordpress_id'] ) ? intval( $_POST['wordpress_id'] ) : 0;

	if ( ! $page_id ) {
		wp_send_json_error( array( 'message' => __( 'Page ID is required.', 'buildfuly' ) ) );
	}

	// Delete from options table (draft)
	$option_key = 'buildfuly_draft_page_' . $page_id;
	delete_option( $option_key );

	// If has WordPress ID, delete the post meta (but not the post itself - that's handled separately)
	if ( $wordpress_id && get_post( $wordpress_id ) ) {
		delete_post_meta( $wordpress_id, '_buildfuly_page_data' );
		delete_post_meta( $wordpress_id, '_buildfuly_page_id' );
	}

	wp_send_json_success( array(
		'message' => __( 'Page data deleted.', 'buildfuly' ),
		'page_id' => $page_id
	) );
}

/**
 * Clean page data before storage by removing base64 images and keeping only URLs.
 *
 * @since    1.2.0
 * @param    array    $page_data    The page data to clean.
 * @return   array                  Cleaned page data.
 */
private function clean_page_data_for_storage( $page_data ) {
	// Remove any base64 image data from sections
	if ( isset( $page_data['sections'] ) && is_array( $page_data['sections'] ) ) {
		foreach ( $page_data['sections'] as &$section ) {
			// Clean HTML content - remove base64 images
			if ( isset( $section['html'] ) ) {
				$section['html'] = preg_replace(
					'/src=["\']data:image\/[^;]+;base64,[^"\']+["\']/i',
					'src=""',
					$section['html']
				);
			}
			
			// If section has image data with base64, remove it but keep URL
			if ( isset( $section['image'] ) && is_array( $section['image'] ) ) {
				if ( isset( $section['image']['base64'] ) ) {
					unset( $section['image']['base64'] );
				}
			}
			
			// Clean any nested data structures
			if ( isset( $section['data'] ) && is_array( $section['data'] ) ) {
				$section['data'] = $this->clean_nested_base64( $section['data'] );
			}
		}
	}

	// Remove any temporary/transient data
	unset( $page_data['_temp'] );
	unset( $page_data['_editing'] );

	return $page_data;
}

/**
 * Recursively clean base64 data from nested arrays.
 *
 * @since    1.2.0
 * @param    array    $data    The data to clean.
 * @return   array             Cleaned data.
 */
private function clean_nested_base64( $data ) {
	if ( ! is_array( $data ) ) {
		// Check if it's a base64 string
		if ( is_string( $data ) && strpos( $data, 'data:image/' ) === 0 ) {
			return ''; // Remove base64 image strings
		}
		return $data;
	}

	foreach ( $data as $key => &$value ) {
		if ( $key === 'base64' || $key === 'imageData' ) {
			unset( $data[$key] );
		} elseif ( is_array( $value ) ) {
			$value = $this->clean_nested_base64( $value );
		} elseif ( is_string( $value ) && strpos( $value, 'data:image/' ) === 0 ) {
			$data[$key] = ''; // Remove base64 image strings
		}
	}

	return $data;
}

/**
 * Add "Edit with Buildfuly" and "Delete" links to WordPress Pages list
 *
 * @param array   $actions An array of row action links
 * @param WP_Post $post    The post object
 * @return array Modified actions array
 */
public function add_buildfuly_row_actions( $actions, $post ) {
	// Debug: log that function is called
	error_log( 'add_buildfuly_row_actions called for post ' . $post->ID . ' type: ' . $post->post_type );
	
	// Only add for pages
	if ( $post->post_type !== 'page' ) {
		error_log( 'Skipping - not a page' );
		return $actions;
	}
	
	// Check if user has permission to edit pages
	if ( ! current_user_can( 'edit_pages' ) ) {
		error_log( 'Skipping - user cannot edit pages' );
		return $actions;
	}
	
	error_log( 'Adding Buildfuly edit link' );
	
	// Add "Edit with Buildfuly" link
	$buildfuly_url = admin_url( 'admin.php?page=buildfuly&open_page=' . $post->ID );
	$actions['buildfuly_edit'] = sprintf(
		'<a href="%s" style="color: #0071e3; font-weight: 600;">%s</a>',
		esc_url( $buildfuly_url ),
		__( 'Edit with Buildfuly', 'buildfuly' )
	);
	
	// Move "Edit with Buildfuly" to be second (after "Edit")
	if ( isset( $actions['edit'] ) ) {
		$edit = $actions['edit'];
		$buildfuly = $actions['buildfuly_edit'];
		unset( $actions['edit'], $actions['buildfuly_edit'] );
		$actions = array_merge(
			array( 'edit' => $edit, 'buildfuly_edit' => $buildfuly ),
			$actions
		);
	}
	
	error_log( 'Actions: ' . print_r( array_keys( $actions ), true ) );
	
	return $actions;
}

/**
 * Auto-update site title to match business name from SEO settings
 * Runs once when business name is set
 */
public function auto_update_site_title() {
	// Check if we've already done this
	$already_updated = get_option( 'buildfuly_site_title_updated', false );
	if ( $already_updated ) {
		return;
	}
	
	// Get business name from SEO settings
	$business_name = get_option( 'buildfuly_business_name', '' );
	
	if ( ! empty( $business_name ) ) {
		// Update site title to business name
		update_option( 'blogname', $business_name );
		
		// Update tagline to business type if available
		$business_type = get_option( 'buildfuly_business_type', '' );
		if ( ! empty( $business_type ) ) {
			update_option( 'blogdescription', $business_type );
		}
		
		// Mark as updated so we don't do it again
		update_option( 'buildfuly_site_title_updated', true );
	}
}

/**
 * Sanitize business name and update site title
 *
 * @param string $value The business name value
 * @return string Sanitized business name
 */
public function sanitize_and_update_business_name( $value ) {
	$sanitized = sanitize_text_field( $value );
	
	// Update site title to match business name
	if ( ! empty( $sanitized ) ) {
		update_option( 'blogname', $sanitized );
	}
	
	return $sanitized;
}

/**
 * Add "Edit with Buildfuly" link to WordPress admin bar
 * Shows when viewing a page on the front-end
 *
 * @param WP_Admin_Bar $wp_admin_bar The admin bar object
 */
public function add_buildfuly_admin_bar_link( $wp_admin_bar ) {
	// Only show on pages (not posts or other content types)
	if ( ! is_page() ) {
		return;
	}
	
	// Only show for users who can edit pages
	if ( ! current_user_can( 'edit_pages' ) ) {
		return;
	}
	
	// Get current page ID
	$page_id = get_the_ID();
	if ( ! $page_id ) {
		return;
	}
	
	// Build Buildfuly URL
	$buildfuly_url = admin_url( 'admin.php?page=buildfuly&open_page=' . $page_id );
	
	// Add the main "Edit with Buildfuly" link to admin bar
	$wp_admin_bar->add_node( array(
		'id'    => 'edit-with-buildfuly',
		'title' => '<span class="ab-icon dashicons dashicons-admin-page" style="margin-top: 2px;"></span><span class="ab-label">Edit with Buildfuly</span>',
		'href'  => $buildfuly_url,
		'meta'  => array(
			'title' => __( 'Edit this page with Buildfuly', 'buildfuly' ),
		),
	) );
	
	// Add "Delete Page" as a submenu item
	$delete_url = wp_nonce_url(
		admin_url( 'admin-post.php?action=buildfuly_delete_page&page_id=' . $page_id ),
		'buildfuly_delete_page_' . $page_id
	);
	
	$wp_admin_bar->add_node( array(
		'parent' => 'edit-with-buildfuly',
		'id'     => 'delete-buildfuly-page',
		'title'  => '<span class="ab-icon dashicons dashicons-trash" style="color: #dc3232; margin-top: 2px;"></span><span class="ab-label" style="color: #dc3232;">Delete Page</span>',
		'href'   => $delete_url,
		'meta'   => array(
			'title'   => __( 'Delete this page', 'buildfuly' ),
			'onclick' => 'return confirm("Are you sure you want to delete this page? This cannot be undone.");',
		),
	) );
}

/**
 * Handle page deletion from admin bar
 * Deletes both WordPress page and Buildfuly page data
 */
public function handle_delete_page() {
	// Check if page_id is provided
	if ( ! isset( $_GET['page_id'] ) ) {
		wp_die( __( 'No page ID provided.', 'buildfuly' ) );
	}
	
	$page_id = intval( $_GET['page_id'] );
	
	// Verify nonce
	if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'buildfuly_delete_page_' . $page_id ) ) {
		wp_die( __( 'Security check failed.', 'buildfuly' ) );
	}
	
	// Check permissions
	if ( ! current_user_can( 'delete_pages' ) ) {
		wp_die( __( 'You do not have permission to delete pages.', 'buildfuly' ) );
	}
	
	// Get page data to find the Buildfuly page ID
	$page_data = get_post_meta( $page_id, '_buildfuly_page_data', true );
	$buildfuly_page_id = isset( $page_data['id'] ) ? $page_data['id'] : null;
	
	// Delete the WordPress page (move to trash)
	$deleted = wp_trash_post( $page_id );
	
	if ( ! $deleted ) {
		wp_die( __( 'Failed to delete page.', 'buildfuly' ) );
	}
	
	// Delete Buildfuly page data (meta)
	delete_post_meta( $page_id, '_buildfuly_page_data' );
	
	// Delete draft data if exists
	if ( $buildfuly_page_id ) {
		delete_option( 'buildfuly_draft_page_' . $buildfuly_page_id );
	}
	
	// Redirect to pages list with success message
	wp_redirect( admin_url( 'edit.php?post_type=page&deleted=1' ) );
	exit;
}

/**
 * Extract FAQ items from page content and store as post meta for FAQ schema.
 *
 * @since    1.1.2
 * @param    int       $page_id       WordPress page ID
 * @param    string    $page_content  Page HTML content
 */
private function extract_and_store_faq_items( $page_id, $page_content ) {
	$faq_items = array();
	
	// Look for FAQ sections - match the Buildfuly FAQ component structure
	// Pattern: buildfuly-faq-question followed by buildfuly-faq-answer
	if ( preg_match_all( '/<h3[^>]*class="[^"]*buildfuly-heading[^"]*"[^>]*>([^<]+)<\/h3>.*?<p[^>]*class="[^"]*buildfuly-text[^"]*"[^>]*>([^<]+)<\/p>/is', $page_content, $matches, PREG_SET_ORDER ) ) {
		foreach ( $matches as $match ) {
			$question = trim( wp_strip_all_tags( $match[1] ) );
			$answer = trim( wp_strip_all_tags( $match[2] ) );
			
			// Only include if it looks like a FAQ (has question mark or starts with question word)
			if ( ! empty( $question ) && ! empty( $answer ) ) {
				if ( strpos( $question, '?' ) !== false || preg_match( '/^(what|how|why|when|where|who|is|can|do|does|will|should)/i', $question ) ) {
					$faq_items[] = array(
						'question' => $question,
						'answer' => $answer,
					);
				}
			}
		}
	}
	
	// Also try to find accordion-style FAQs
	if ( empty( $faq_items ) && preg_match_all( '/<button[^>]*class="[^"]*buildfuly-faq-question[^"]*"[^>]*>.*?<h3[^>]*>([^<]+)<\/h3>.*?<\/button>.*?<div[^>]*class="[^"]*buildfuly-faq-answer[^"]*"[^>]*>.*?<p[^>]*>([^<]+)<\/p>/is', $page_content, $matches, PREG_SET_ORDER ) ) {
		foreach ( $matches as $match ) {
			$question = trim( wp_strip_all_tags( $match[1] ) );
			$answer = trim( wp_strip_all_tags( $match[2] ) );
			
			if ( ! empty( $question ) && ! empty( $answer ) ) {
				$faq_items[] = array(
					'question' => $question,
					'answer' => $answer,
				);
			}
		}
	}
	
	// Store FAQ items as post meta if we found any
	if ( ! empty( $faq_items ) ) {
		update_post_meta( $page_id, '_buildfuly_faq_items', $faq_items );
		Buildfuly_Logger::debug( 'BuildfulyAI: Stored ' . count( $faq_items ) . ' FAQ items for page ' . $page_id );
	}
}

/**
 * Set the first image in page content as the featured image (thumbnail).
 * This is used for OG image, social sharing, and page previews.
 *
 * @since    1.1.2
 * @param    int       $page_id       WordPress page ID
 * @param    string    $page_content  Page HTML content
 */
private function set_first_image_as_featured( $page_id, $page_content ) {
	// Skip if page already has a featured image
	if ( has_post_thumbnail( $page_id ) ) {
		return;
	}
	
	// Find the first image in the content
	// Look for WordPress attachment URLs or any img src
	$attachment_id = 0;
	
	// First try to find images with WordPress attachment class (most reliable)
	if ( preg_match( '/wp-image-(\d+)/', $page_content, $matches ) ) {
		$attachment_id = intval( $matches[1] );
	}
	
	// If no wp-image class found, try to match image URLs to attachments
	if ( ! $attachment_id && preg_match( '/<img[^>]+src=["\']([^"\']+)["\']/', $page_content, $matches ) ) {
		$image_url = $matches[1];
		
		// Try to get attachment ID from URL
		$attachment_id = attachment_url_to_postid( $image_url );
		
		// If not found, try without size suffix (-300x200, -1024x768, etc.)
		if ( ! $attachment_id ) {
			$image_url_base = preg_replace( '/-\d+x\d+(\.\w+)$/', '$1', $image_url );
			$attachment_id = attachment_url_to_postid( $image_url_base );
		}
	}
	
	// Set as featured image if we found a valid attachment
	if ( $attachment_id && get_post( $attachment_id ) ) {
		set_post_thumbnail( $page_id, $attachment_id );
		Buildfuly_Logger::debug( 'BuildfulyAI: Set featured image for page ' . $page_id . ' to attachment ' . $attachment_id );
	}
}
}










